########################################################################
#
# PyNX - Python tools for Nano-structures Crystallography
#   (c) 2008-2015 : Univ. Joseph Fourier (Grenoble 1), CEA/INAC/SP2M
#   (c) 2013-2014 : Fondation Nanosciences, Grenoble
#   (c) 2016-present : ESRF-European Synchrotron Radiation Facility
#       authors:
#         Vincent Favre-Nicolin, favre@esrf.fr
#         Ondrej Mandula
#
# Example of the ptychograpic reconstruction of experimental data (GaAs nanowire with insertion of InAs QDs).
# Data taken in Bragg geometry for (113) reflection.
# This example uses OpenCL for faster calculations

import os
import timeit
from pylab import figure, imshow, grid
import pyopencl as cl
from pynx.ptycho.old import clptycho
import numpy as np
from pynx.utils.math import smaller_primes
from pynx.utils.phase import minimize_grad_phase
from pynx.utils.plot_utils import insertColorwheel, complex2rgbalin

thedir = os.path.dirname(__file__)

# Reading experimental data
data = load(os.path.join(thedir, 'data/data_113.npz'))
ampl = sqrt(data['iobs'])  # the data are intensities
nx = ampl.shape[-1]
dsize = smaller_primes(nx)
if nx != dsize:
    # Need to adapt size for OpenCL FFT
    d = (nx - dsize) // 2
    ampl = ampl[:, d:-d, d:-d]

# Reading the positions of the scans
posx, posy = data['posx'], data['posy']
# Initial object (random) and probe (Gaussian approximation to the X-ray beam)
obj0 = data['obj0']
probe0 = data['probe0']

obj0 = (np.random.uniform(0, 1, obj0.shape) * exp(1j * np.random.uniform(-np.pi, np.pi, obj0.shape))).astype(
    np.complex64)

pixel_size_object = 0.011e-6
lambdaz = 7.1e-9

gpu = None  # Use 'Tesla' or 'GTX' or 'Iris' etc... to choose a GPU
if gpu is None:
    ctx = cl.create_some_context()  # interactive choice of GPU & OpenCL device
    gpu = ctx.devices[0]

doplot = True
t0 = timeit.default_timer()
p = clptycho.Ptycho2D(iobs=ampl ** 2, positions=(posy, posx), probe=probe0.copy(), obj=obj0.copy(), opencl_device=gpu,
                      lambdaz=lambdaz, pixel_size_object=pixel_size_object)

if False:
    # Alternating projections, updating object only
    p.run_alternating_projection(40, update_object=True, update_probe=False, verbose=10, doplot=doplot)
    # Alternating projections, updating object and probe
    p.run_alternating_projection(100, update_object=True, update_probe=True, verbose=10, doplot=doplot)
else:
    # Difference map, updating object only
    p.run_difference_map(40, update_object=True, update_probe=False, verbose=10, doplot=doplot)
    # Difference map, updating object and probe
    p.run_difference_map(100, update_object=True, update_probe=True, verbose=10, doplot=doplot)

# Maximum likelihood conjugate gradient optimization (using Poisson noise), updating object and probe
p.run_ml_poisson(40, update_object=True, update_probe=True, verbose=10, doplot=doplot)

# Maximum likelihood with regularisation constant:
# p.run_ml_poisson(100, update_object=True, update_probe=True, verbose=10, reg_fac=1e-4, doplot=doplot)
print("Elapsed time (total): %5.1fs" % (timeit.default_timer() - t0))

# visualisation:
om = minimize_grad_phase(p.obj, mask=~p.scan_area_obj)  # phase flattening
figure()
imshow(complex2rgbalin(om[0] * p.scan_area_obj))
grid(b=True)
insertColorwheel()
