########################################################################
#
# Example of the ptychograpic reconstruction of experimental data (GaAs nanowire with insertion of InAs QDs). Data taken in Bragg geometry for (004) reflection.
# (c) Univ. Grenoble Alpes
# (c) Fondation Nanosciences
# (c) CEA Grenoble / INAC
# Authors: Vincent Favre-Nicolin, Ondrej Mandula
#
########################################################################

from pylab import *
from pynx.ptycho import simulation, shape
from pynx.ptycho.old import clptycho
from pynx.utils.phase import minimize_grad_phase
from pynx.utils.plot_utils import insertColorwheel, complex2rgbalin
from pynx.utils.math import smaller_primes
import pyopencl as cl

thedir = os.path.dirname(__file__)

data = load(os.path.join(thedir, 'data/data_intensity_004.npy'))
ampl = sqrt(data)  # the data are intensities
nx = ampl.shape[-1]
dsize = smaller_primes(nx)
if nx != dsize:
    # Need to adapt size for OpenCL FFT
    d = (nx - dsize) // 2
    ampl = ampl[:, d:-d, d:-d]

# Reading the positions of the scans
posx, posy = load(os.path.join(thedir, 'data/pos_004.npy'))
# Intializing object
ny, nx = shape.calc_obj_shape(posy, posx, ampl.shape[1:])  # Size of the reconstructed object (obj)
obj_init_info = {'type': 'random', 'shape': (ny, nx)}
# Initializing probe
probe_init_info = {'type': 'gauss', 'shape': ampl.shape[1:], 'sigma_pix': (10, 40), 'rotation': 0.}

init = simulation.Simulation(obj_info=obj_init_info, probe_info=probe_init_info)

lambdaz = 7.1e-9
pixel_size_object = 11e-9
doplot = True

gpu = None  # Use 'Tesla' or 'GTX' or 'Iris' etc... to choose a GPU
if gpu is None:
    ctx = cl.create_some_context()  # interactive choice of GPU & OpenCL device
    gpu = ctx.devices[0]

##################

init.make_obj()
init.make_probe()

p = clptycho.Ptycho2D(iobs=ampl ** 2, positions=(posy, posx), probe=init.probe.values.copy(),
                      obj=init.obj.values.copy(), opencl_device=gpu,
                      lambdaz=lambdaz, pixel_size_object=pixel_size_object)

# Alternating projections, updating object only
p.run_alternating_projection(40, update_object=True, update_probe=False, verbose=10, doplot=doplot)
# Alternating projections, updating object and probe
p.run_alternating_projection(40, update_object=True, update_probe=True, verbose=10, doplot=doplot)
# Maximum likelihood conjugate gradient optimization (using Poisson noise), updating object and probe
p.run_ml_poisson(40, update_object=True, update_probe=True, verbose=10, doplot=doplot)

# Maximum likelihood with regularisation constant:
p.run_ml_poisson(100, update_object=True, update_probe=True, verbose=10, reg_fac=1e-310, doplot=doplot)

# visualisation:
om = minimize_grad_phase(p.obj, mask=~p.scan_area_obj)  # phase flattening
figure()
imshow(complex2rgbalin(om[0] * p.scan_area_obj))
grid(b=True)
insertColorwheel()
