__device__ float LLKPoisson(const float obs, const float calc)
{
  if(obs<=0.1) return calc; // observed intensity is zero

  return calc - obs + obs * log(obs / calc);
}

__device__ float LLKGaussian(const float obs, const float calc)
{
  const float tmp = obs - calc;
  return tmp * tmp / (obs + 1);
}

__device__ float LLKEuclidian(const float obs, const float calc)
{
  const float tmp = sqrtf(obs) - sqrt(calc);
  return 4 * tmp * tmp;
}

/** Reduction kernel function :
* compute the log-likelihood given observed data following Poisson, Gaussian and Euclidian statistics, and
* a stack of complex data corresponding to the calculated modes.
* Masked pixels are ignored.
* Reference:  [New Journal of Physics 14 (2012) 063004, doi:10.1088/1367-2630/14/6/063004]
* \param i: the point in the 3D observed intensity array for which the llk is calaculated
* \param iobs: the observed in tensity array, shape=(stack_size, ny, nx)
* \param psi: the calculated complex amplitude, shape=(nb_obj, nb_probe, stack_size, ny, nx)
* \param mask: the mask (0=good pixel, >0 masked pixel) of shape (ny, nx)
* \return: a float4 vector with (poisson llk, gaussian llk, euclidian llk, icalc)
*/
__device__  my_float4 LLKAll(const int i, float *iobs, pycuda::complex<float> *psi)
{
  const float obs = iobs[i];

  if(obs < 0) return 0;

  const float calc = dot(psi[i],psi[i]);

  return my_float4(LLKPoisson(obs, calc), LLKGaussian(obs, calc), LLKEuclidian(obs, calc), calc);
}

/** Reduction kernel function :
* compute the log-likelihood given observed data following Poisson, Gaussian and Euclidian statistics, and
* a stack of complex data corresponding to the calculated modes.
* Masked pixels are ignored.
* Reference:  [New Journal of Physics 14 (2012) 063004, doi:10.1088/1367-2630/14/6/063004]
* \param i: the point in the 3D observed intensity array for which the llk is calaculated
* \param iobs: the observed in tensity array, shape=(stack_size, ny, nx)
* \param psi: the calculated complex amplitude, shape=(nb_obj, nb_probe, stack_size, ny, nx)
* \param mask: the mask (0=good pixel, >0 masked pixel) of shape (ny, nx)
* \return: a float4 vector with (poisson llk, gaussian llk, euclidian llk, icalc)
*/
__device__  my_float4 LLKAllIcalc(const int i, float *iobs, float *icalc)
{
  const float obs = iobs[i];

  if(obs < 0) return 0;

  const float calc = fmaxf(icalc[i], 1e-8f);

  return my_float4(LLKPoisson(obs, calc), LLKGaussian(obs, calc), LLKEuclidian(obs, calc), calc);
}
