#!/bin/bash

# This is a complete installation script for a python virtual environment with PyNX.
# This has been tested on debian 8 and macOS computers.
# It assumes you already have installed :
# - python>=3.4 (>=3.5 recommended)
# - git, cmake, compilers (Xcode with developer tools on macOS)
# - opencl headers and drivers (native on macOS, libraries needed on linux)
# - cuda development tools and drivers (optional)

echo $1
if [ -z $2 ];
then
  echo "No directory or python executable given for installation !";
  echo "Usage: mkenv.sh DIRNAME PYTHON_EXE PYNX_VERSION"
  echo "   with: DIRNAME the name of the new directory to create the python virtual environement, e.g. pynx-env"
  echo "         PYTHON_EXE the name of the python executable, e.g. python3.4"
  echo "         PYNX_VERSION (optional) the git tag for the pynx version to be installed"
  echo "example: mkenv.sh pynx-env python3.5"
  echo "example: mkenv.sh pynx-3.6.2-py36 python3.6 v3.6.2"
  exit
fi


# If we are at ESRF, activate proxy
if [[ $(hostname -f) == *.esrf.fr ]];
then
  echo "exporting ESRF PROXY environment parameters"
  export http_proxy=http://proxy.esrf.fr:3128
  export https_proxy=https://proxy.esrf.fr:3128
  export no_proxy='localhost,127.0.0.1,localaddress,.esrf.fr,.esrf.eu '
fi

echo
echo "#############################################################################################"
echo " Creating & the destination directory"
echo "#############################################################################################"
echo


if [ -d "$PWD/$1" ]; then
    echo "ERROR: directory $PWD/$1 already exists !"
    echo " Please remove the target directory first."
    exit
fi

if mkdir -p $1;
then
  echo "Installing in: " $1
else
  echo "Cannot install in: " $1
  echo "Exiting"
  exit
fi

cd $1
export BASEDIR=$PWD
echo $BASEDIR

echo
echo "#############################################################################################"
echo " Creating the python virtual environment"
echo "#############################################################################################"
echo
# Create the python virtual environment, without system packages
cd $BASEDIR
if [[ "$OSTYPE" == "darwin"* ]]; then
  # See https://matplotlib.org/faq/osx_framework.html
  $2 -m venv ./
else
  virtualenv -p $2 ./
fi

source bin/activate

echo
echo "#############################################################################################"
echo " Installing python packages"
echo "#############################################################################################"
echo
# install requirements
pip install --upgrade pip
pip install setuptools wheel --upgrade
# lxml and pillow are only necessary to avoid fabio warning
pip install numpy cython scipy --upgrade
pip install matplotlib ipython notebook scikit-image h5py hdf5plugin silx pillow lxml fabio ipywidgets ipympl --upgrade

echo
echo "#############################################################################################"
echo " Installing pyopencl packages"
echo "#############################################################################################"
echo

if pip install mako pyopencl --upgrade ; then
    has_pyopencl=1
else
    echo "pyopencl installation failed - OpenCL WILL NOT BE AVAILABLE !" ;
    echo "  OpenCL is needed for most applications - Check your opencl headers and your internet access if pip failed" ;
    echo "  You can proceed if you do not have a GPU and wish only to make tests using the CPU (much slower)" ;
    echo ;
    read -p  "Press Ctrl-C to abort, or RETURN to continue" ;
    has_pyopencl=0
fi

mkdir -p $BASEDIR/dev

echo
echo "#############################################################################################"
echo " Installing clfft"
echo "#############################################################################################"
echo
if [ $has_pyopencl -eq 1 ]; then

    # Install clfft, also in the virtualenv (recent versions allow prime numbers up to 13)
    cd $BASEDIR/dev
    git clone https://github.com/clMathLibraries/clFFT.git
    cd clFFT
    mkdir build
    cd build
    cmake -DCMAKE_INSTALL_PREFIX:PATH=$BASEDIR/ ../src
    make all install

    # Make sure LD_LIBRARY_PATH and LIBRARY_PATH is included when activating the python virtual environment, so that
    # the clfft library can be found.
    # You get bonus points if you understand the whole awk script.
    if  [[ "$OSTYPE" == "linux"* ]]; then
      mv $BASEDIR/bin/activate $BASEDIR/bin/activate.ORIG
      awk '{sub("unset VIRTUAL_ENV","if [ -n \"$_OLD_LD_LIBRARY_PATH\" ] ; then\n        LD_LIBRARY_PATH=\"$_OLD_LD_LIBRARY_PATH\"\n      export LD_LIBRARY_PATH\n      unset _OLD_LD_LIBRARY_PATH\n        LIBRARY_PATH=\"$_OLD_LIBRARY_PATH\"\n      export LIBRARY_PATH\n      unset _OLD_LIBRARY_PATH\n    fi\n    unset VIRTUAL_ENV")};1' $BASEDIR/bin/activate.ORIG > $BASEDIR/bin/activate_tmp
      awk '{sub("export VIRTUAL_ENV","export VIRTUAL_ENV\n\n_OLD_LD_LIBRARY_PATH=\"$LD_LIBRARY_PATH\"\nLD_LIBRARY_PATH=\"$VIRTUAL_ENV/lib:$VIRTUAL_ENV/lib64:$LD_LIBRARY_PATH\"\nexport LD_LIBRARY_PATH\n \nOLD_LIBRARY_PATH=\"$LIBRARY_PATH\"\nLIBRARY_PATH=\"$VIRTUAL_ENV/lib:$VIRTUAL_ENV/lib64:$LIBRARY_PATH\"\nexport LIBRARY_PATH\n")};1' $BASEDIR/bin/activate_tmp > $BASEDIR/bin/activate
      #rm -f $BASEDIR/bin/activate_tmp
    elif [[ "$OSTYPE" == "darwin"* ]]; then
      mv $BASEDIR/bin/activate $BASEDIR/bin/activate.ORIG
      # As of 06/2018, it seems lib64 is not used/created any more, create it for compatibility
      mkdir -p $BASEDIR/lib64
      awk '{sub("unset VIRTUAL_ENV","if [ -n \"$_OLD_DYLD_LIBRARY_PATH\" ] ; then\n        DYLD_LIBRARY_PATH=\"$_OLD_DYLD_LIBRARY_PATH\"\n      export DYLD_LIBRARY_PATH\n      unset _OLD_DYLD_LIBRARY_PATH\n    fi\n    unset VIRTUAL_ENV")};1' $BASEDIR/bin/activate.ORIG > $BASEDIR/bin/activate_tmp
      awk '{sub("export VIRTUAL_ENV","export VIRTUAL_ENV\n\n_OLD_DYLD_LIBRARY_PATH=\"$DYLD_LIBRARY_PATH\"\nDYLD_LIBRARY_PATH=\"$VIRTUAL_ENV/lib:$VIRTUAL_ENV/lib64:$DYLD_LIBRARY_PATH\"\nexport DYLD_LIBRARY_PATH\n")};1' $BASEDIR/bin/activate_tmp > $BASEDIR/bin/activate
      rm -f $BASEDIR/bin/activate_tmp
    fi

    # Activate again with the new environment variable
    deactivate
    cd $BASEDIR
    source bin/activate
else
     echo "pyopencl not available - not installing clfft"
fi

echo
echo "#############################################################################################"
echo " Installing gpyfft"
echo "#############################################################################################"
echo
if [ $has_pyopencl -eq 1 ]; then
    ## Install gpyfft - more tricky ! May require hand modifications in setup.py..
    cd $BASEDIR/dev
    git clone https://github.com/geggo/gpyfft.git
    cd gpyfft
    ## Get working version, gpyfft too often has incompatible changes
    git checkout 2c07fa8e7674757
    mv setup.py setup.py.ORIG
    awk '{sub("/Users/gregor/Devel/clFFT/src/library","'"$BASEDIR"'" "/lib/")};1' setup.py.ORIG > tmp1.py
    awk '{sub("/Users/gregor/Devel/","'"$BASEDIR"'" "/dev/")};1' tmp1.py > tmp2.py
    awk '{sub("/home/gregor/devel/","'"$BASEDIR"'" "/dev/")};1' tmp2.py > tmp1.py
    awk '{sub("/usr/local/lib64","'"$BASEDIR"'" "/lib/")};1' tmp1.py > tmp2.py
    awk '{sub("os.path.join\\(CLFFT_DIR, \47src\47, \47include\47\\)", "os.path.join(CLFFT_DIR, \"src\", \"include\"), \"'"$BASEDIR"'" "/include/\"")};1' tmp1.py > setup.py

    rm -f tmp*.py
    python setup.py install
else
     echo "pyopencl not available - not installing gpyfft"
fi

echo
echo "#############################################################################################"
echo " Installing PyNX"
echo "#############################################################################################"
echo
cd $BASEDIR/dev
echo "Select method to download PyNX: git (if you have a https://gitlab.esrf.fr account) or ftp (no login required):"
select yn in "ftp" "git"; do
    case $yn in
        ftp ) curl -O http://ftp.esrf.fr/pub/scisoft/PyNX/pynx-latest.tar.bz2 ; tar -xjf pynx-latest.tar.bz2 ; break;;
        git ) git clone https://gitlab.esrf.fr/favre/PyNX.git pynx ; break;;
    esac
done

cd pynx

if [ -z $3 ];
    then echo "No tag given - using git pynx master head"
else
    git checkout tags/$3
fi

python setup.py install


echo
echo "#############################################################################################"
echo "Installing pyCUDA (optional)"
echo "#############################################################################################"
echo
# To also have CUDA support - use wheels if missing cuda.h for compilation
if [[ $($(command -v nvcc --version)) ]] ;
then
     pip install pycuda scikit-cuda --upgrade || echo "###\nCould not install pycuda - CUDA probably missing ?\n###\n"
else
  if [[ $(hostname -f) == debian8-devel.esrf.fr ]];
  then
     # Use precompiled wheels on debian8-devel because of missing missing cuda.h
     pip install /data/id01/inhouse/favre/pycuda-scikit-wheels/*.whl
  else
     echo "CUDA compiler (nvcc) no detected - not installing CUDA"
  fi
fi

cd $BASEDIR
if [[ $(hostname -f) == debian8-devel.esrf.fr ]];
then
    # For debian8-devel, install SIP from source, pyqt5 and pyopengl
    echo
    echo "#############################################################################################"
    echo "Installing SIP from source and pyqt5, pyopengl"
    echo "#############################################################################################"
    echo
    # SIP must be installed from source, and as it installs a header in ..env/include/python3.4m , that include dir cannot be a symlink to the original python include dir
    cd $BASEDIR/include
    mv python3.4m python3.4m-lnk
    mkdir python3.4m
    cp -Rf python3.4m-lnk/* python3.4m/
    rm -Rf python3.4m-lnk
    cd $BASEDIR/dev
    wget https://downloads.sourceforge.net/project/pyqt/sip/sip-4.19.3/sip-4.19.3.tar.gz
    tar -xzf sip-4.19.3.tar.gz
    cd sip-4.19.3/
    python configure.py
    make install

    pip install pyqt5 --no-deps --upgrade
    pip install pyopengl --upgrade
fi

echo
echo "#############################################################################################"
echo "Finished installation !"
echo "#############################################################################################"
echo
echo "To use the PyNX environment, use source $BASEDIR/bin/activate"
echo
if [[ $(hostname -f) != debian8-devel.esrf.fr ]];
then
    echo "If you wish to also use the SILX viewer (recommended), you should also install pyqt5 and pyopengl:"
    echo " source $BASEDIR/bin/activate ; pip install pyqt5 pyopengl"
fi
