// VOLUME BROWSER - VUES ORTHOGONALES

// paquetages ImageJ
import ij.*;
import ij.gui.*;
import ij.process.*;
import ij.plugin.*;
import ij.io.*;

import java.io.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

public class File_Extractor implements PlugIn {
		public void run(String arg) {
			ToolBar barre=new ToolBar();
			barre.setVisible(true);
		}
}

//___________________________________________________________________________________________
// classe utilitaire pour l'extraction des donnees dans les fichiers .info

class RawExtractor {

	private String filePath;
	private String fileName;
	
	private int width;
	private int height;
	private int length;
	private char fileType;	// a=8bits;	b=32bits Little-Endian;	c=32bits Big-Endian
	
	// CONSTRUCTEUR
	public RawExtractor(String filePath, String fileName) {
		this.filePath=filePath;
		this.fileName=fileName;
		fileType='z';
	}
	
	// methodes d'acces en lecture aux membres
	public int getWidth()		{return width;}
	public int getHeight()		{return height;}
	public int getLength()		{return length;}
	public char getFileType()	{return fileType;}

	// retourne une chaine ne conservant que les chiffres de la chaine d'origine
	// accepte un signe '-' devant les chiffres
	// utilisee pour supprimer espaces, tabulations, sauts de lignes et tous caracteres parasites
	// retourne une chaine vide si pas de chiffre trouve. Tester le resultat avec snum.equals("")==true
	private String extractNumeric(String s) {
		String snum=new String();

		// parcours de la chaine de caractere pour le filtrage des caracteres indesirables	
		for(int i=0;i<s.length();i++) {
			// si le caractere est un chiffre OU si un signe '-' est trouve avant un chiffre
			if ((s.charAt(i)>='0' && s.charAt(i)<='9')||(s.charAt(i)=='-' && snum.length()==0))
				snum=snum+s.charAt(i);
		}
		// si au moins 1 caractere et ne commence pas par '-' OU si au moins 2 caracteres et commence par '-'
		if ((snum.length()>0 && snum.charAt(0)!='-')||(snum.length()>1 && snum.charAt(0)=='-'))
			return snum;
		else
			return "";
	}	
	
	// retourne une chaine ne conservant que "YES" de la chaine d'origine
	// retourne une chaine vide si n'a pas trouve "YES": snum.equals("")==true
	private boolean extractYesNo(String s) {
		String snum=new String();

		// parcours de la chaine de caractere pour le filtrage des caracteres indesirables	
		for(int i=0;i<s.length();i++) {
			// si le caractere est une lettre de l'alphabet majuscule ou minuscule
			if ((s.charAt(i)>='A' && s.charAt(i)<='Z') || (s.charAt(i)>='a' && s.charAt(i)<='z')) {
				snum=snum+s.charAt(i);
			}
		}
		// si la chaine vaut "YES"
		if (snum.equalsIgnoreCase("YES"))
			return true;
		else
			return false;
	}
	
	// lit les 5 derniers caracteres de la chaine passee en parametre
	// retourne True si cette chaine=".info"
	// retourne False sinon
	private boolean isInfoFile(String s) {
		String stringEnd;
		stringEnd=fileName.substring(fileName.length()-5,fileName.length());
		if (stringEnd.equalsIgnoreCase(".info"))
			return true;
		else
			return false;
	}
	
	// lit les 4 derniers caracteres de la chaine passee en parametre
	// retourne True si cette chaine=".raw"
	// retourne False sinon
	private boolean isRawFile(String s) {
		String stringEnd;
		stringEnd=fileName.substring(fileName.length()-4,fileName.length());
		if (stringEnd.equalsIgnoreCase(".raw"))
			return true;
		else
			return false;
	}	
	
	// lecture des parametres dans le fichier info accompagnant le fichier passe en argument
	// le fichier passe en argument peut-etre le .raw ou le .raw.info
	// le fichier info est recherche dans le repertoire du fichier passe en argument
	// retourne true si tous les parametres ont etes determines avec succes
	// retourne false si au moins un n'est pas conforme
	public boolean setParameters() throws IOException {
		
		String paramX=new String();
		String paramY=new String();
		String paramZ=new String();
		
		boolean ret=false;			// valeur de retour
		boolean error=false;		// true si tentative de lecture (fseek) en dehors du fichier
				
		String path=new String();	// chemin complet du fichier a ouvrir
		String chaine=new String();	// contenu du fichier de parametres
		
		int p=0;					// position a lire dans le fichier (fseek(p))
		char n;						// caractere lu (a la position p)

		// si ni .raw ni .info
		if (!isRawFile(fileName) && !isInfoFile(fileName)) {
			IJ.error("Only .raw ou .info");
			return false;
		}		
		// ajout de ".info" au ".raw" s'il n'y en a pas;
		// pour permettre de selectionner indifferemment le ".raw" ou le ".raw.info" a l'ouverture

		// si se termine par .raw, ajout de .info au nom de fichier avant de creer l'acces
		if (isRawFile(fileName)) {
			fileName=fileName+".info";
		}

		// si se termine par .info, creation d'un acces sans ajouter ".info"
		if (isInfoFile(fileName)) {
			path=filePath+fileName;
			
			// creation d'access direct au fichier binaire
			RandomAccessFile f=new RandomAccessFile(path,"r");
			long size=f.length();

			// copie du contenu du fichier dans une chaine de caracteres
			// si tentative de lecture en dehors du fichier, error=true
			while((!error)&&(p<size)) {
				if(p<size) {
					f.seek(p);
					n=(char)f.read();
					chaine=chaine+n;
					p++;
				}
				else error=true;
			}
			f.close();
			// si aucune tentative de lecture en dehors du fichier n'a ete faite
			if (!error) {
				int posX=chaine.indexOf("NUM_X =");
				int posY=chaine.indexOf("NUM_Y =");
				int posZ=chaine.indexOf("NUM_Z =");
				int posZend=0;
				// recherche si fichier 8 bits
				int posRecad=chaine.indexOf("RECAD =");
				// recherche si fichier 32 bits-High
				int posLowByte=chaine.indexOf("BYTEORDER = LOWBYTEFIRST");
				// ni 8 bits, ni 32 bits-High
				int posHighByte=chaine.indexOf("BYTEORDER = HIGHBYTEFIRST");
				// recherce si fichier 32 bits-Low
				if (posRecad==-1 && posHighByte==-1 && posLowByte==-1) {
					posZend=chaine.length();
					fileType='z';
				}
				// si 8 bits
				else if (posRecad!=-1) {
					posZend=posRecad;
					fileType='a';
				}
				// si 32 bits-Low
				else if (posLowByte!=-1) {
					posZend=posLowByte;
					fileType='b';
				}
				// si 32 bits-High
				else if (posHighByte!=-1) {
					posZend=posHighByte;
					fileType='c';
				}
				
				// si NUM_X NUM_Y et NUM_Z existent
				if (posX!=-1 && posY!=-1 && posZ!=-1) {
					paramX=chaine.substring(posX+7,posY);
					paramY=chaine.substring(posY+7,posZ);
					paramZ=chaine.substring(posZ+7,posZend);

					paramX=extractNumeric(paramX);
					paramY=extractNumeric(paramY);
					paramZ=extractNumeric(paramZ);
				}
				else ret=false;
				// si au moins une des String contenant un parametre est vide
				if (paramX.equals("")||paramY.equals("")||paramZ.equals(""))
					ret=false;
				// la methode a lue tous les parametres indispensables
				else
					ret=true;
			}
			if (ret) {
				// affectations aux variables membres
				width=Integer.parseInt(paramX);
				height=Integer.parseInt(paramY);
				length=Integer.parseInt(paramZ);
			}
		}
		return (ret);
	}
	
	// extraction d'une zone rectangulaire de slice directement dans le fichier 8bits
	// renvoie l'image generee si tout s'est bien deroule
	// retourne null si les parametres sont incoherents (entree de methode ou tentative de lecture en dehors du fichier)
	public ImagePlus subSlice8(int x, int y, int wZone, int hZone, int z, int wImage, int hImage) throws IOException {
	
		ImagePlus imp;
		int start=z*wImage*hImage;

		// creation d'une nouvelle image
		imp=NewImage.createByteImage ("x="+x+",y="+y+",z="+z+",width="+wZone+",height="+hZone,wZone,hZone,1,NewImage.FILL_BLACK);
		// recuperation du processor de la nouvelle image
		ImageProcessor ip=imp.getProcessor();
		// le tableau de pixels reference l'image
		// toute ecriture dans le tableau pixels modifie l'image ipNew
		byte[] pixels=(byte[])ip.getPixels();
		RandomAccessFile f;
		f=new RandomAccessFile(filePath+fileName,"r");
		long size=f.length();	// taille du fichier
		
		int p=0;				// compteur d'iteration
		long position;			// position reelle dans le fichier
		byte n;					// valeur lue dans fichier

		// deplacement en lignes
		for(int dy=0;dy<hZone;dy++) {
			// positionnement sur le premier point de la iy eme ligne de la zone
			// deplacement en colonnes
			int depLine=(y+dy)*wImage;
			int depPoint=start+(depLine)+x;
			for(int dx=0;dx<wZone;dx++) {
				// deplacement du point sur la ligne de la zone
				position=depPoint+dx;
				// la position calculee n'est pas comprise entre le debut et la fin du fichier
				if((position>=0)&&(position<size)) {
					f.seek(position);
					n=(byte)f.read();
					pixels[p]=n;
					p++;
				}
				else
					return null;
			}
		}
		f.close();
		return imp;
	}
		
	// extraction d'une zone rectangulaire de slice directement dans le fichier 32 bits
	// swap	si true = effectue une inversion des bits
	// renvoie l'image generee si tout s'est bien deroule
	// retourne null si les parametres sont incoherents (entree de methode ou tentative de lecture en dehors du fichier)
	public ImagePlus subSlice32(int x, int y, int wZone, int hZone, int z, int wImage, int hImage, boolean swap) throws IOException {
	
		int start=(z*wImage*hImage)+x;

		// creation d'une nouvelle image
		ImagePlus imp=NewImage.createFloatImage ("x="+x+",y="+y+",z="+z+",width="+wZone+",height="+hZone,wZone,hZone,1,NewImage.FILL_BLACK);
		// recuperation du processor de la nouvelle image
		ImageProcessor ip=imp.getProcessor();
		// le tableau de pixels reference l'image
		// toute ecriture dans le tableau pixels modifie l'image ipNew
		float[] pixels=(float[])ip.getPixels();
		RandomAccessFile f;
		f=new RandomAccessFile(filePath+fileName,"r");
		long size=f.length();	// taille du fichier
		
		int p=0;				// compteur d'iteration
		long position;			// position reelle dans le fichier
		float n;				// valeur lue dans le fichier
		if (swap==false) {
			// deplacement en lignes
			for(int dy=0;dy<hZone;dy++) {
				// positionnement sur le premier point de la iy eme ligne de la zone
				// deplacement en colonnes
				int depLine=(y+dy)*wImage;
				int depPoint=start+depLine;
				for(int dx=0;dx<wZone;dx++) {
					// deplacement du point sur la ligne de la zone
					position=(depPoint+dx)*4;
					// la position calculee n'est pas comprise entre le debut et la fin du fichier
					if((position>=0)&&(position<size)) {
						f.seek(position);
						n=(float)f.readFloat();
						pixels[p]=n;
						p++;
					}
					else
						return null;
				}
			}
		}
		// if swap = true
		else {
			// deplacement en lignes
			for(int dy=0;dy<hZone;dy++) {
				// positionnement sur le premier point de la iy eme ligne de la zone
				// deplacement en colonnes
				int depLine=(y+dy)*wImage;
				int depPoint=start+depLine;
				for(int dx=0;dx<wZone;dx++) {
					// deplacement du point sur la ligne de la zone
					position=(depPoint+dx)*4;
					// la position calculee n'est pas comprise entre le debut et la fin du fichier
					if((position>=0)&&(position<size)) {
						f.seek(position);
				        int i = f.read();
        				int j = f.read();
        				int k = f.read();
        				int l = f.read();
        				if((i | j | k | l) < 0) throw new EOFException();
            			int tmpint = (l << 24) + (k << 16) + (j << 8) + i;
            			n = Float.intBitsToFloat(tmpint);
						pixels[p]=n;
						p++;  
					}
					else
						return null;
				}
			}				
		}
		f.close();
		return imp;
	}

	// extraction d'un sous volume dans un fichier Stack
	// renvoie l'image generee si tout s'est bien deroule (bon deroulement des appels a subSlice)
	// renvoie null si subSlice a renvoye un null (mauvais parametres passes a subSlice lors d'un appel)
	public ImagePlus subStack8(int x, int y, int wZone, int hZone, int z1, int z2, int wImage, int hImage)  {

		ImagePlus imp;
		ImageStack impS;
		
		ImagePlus sliceExtracted;
		
		imp=NewImage.createByteImage (null,wZone,hZone,0,NewImage.FILL_BLACK);
		impS=imp.createEmptyStack();
		try {
			int z=z1;
			while (z<=z2) {
				sliceExtracted=subSlice8(x,y,wZone,hZone,z,wImage,hImage);
				if (sliceExtracted!=null)
					impS.addSlice("orig."+z,sliceExtracted.getProcessor());
				else return (null);
				z++;
			}
			imp.setStack("x="+x+",y="+y+",z="+z1+" to "+z2+",width="+wZone+",height="+hZone,impS);
		}
		catch(IOException ex) {
			return (null);
		}
		return(imp);
	}
	
	// extraction d'un sous volume dans un fichier Stack
	// renvoie l'image generee si tout s'est bien deroule (bon deroulement des appels a subSlice)
	// renvoie null si subSlice a renvoye un null (mauvais parametres passes a subSlice lors d'un appel)
	public ImagePlus subStack32(int x, int y, int wZone, int hZone, int z1, int z2, int wImage, int hImage, boolean swap)  {

		ImagePlus imp;
		ImageStack impS;
		
		ImagePlus sliceExtracted;
		
		imp=NewImage.createByteImage (null,wZone,hZone,0,NewImage.FILL_BLACK);
		impS=imp.createEmptyStack();
		try {
			int z=z1;
			while (z<=z2) {
				sliceExtracted=subSlice32(x,y,wZone,hZone,z,wImage,hImage,swap);
				if (sliceExtracted!=null)
					impS.addSlice("orig."+z,sliceExtracted.getProcessor());
				else return (null);
				z++;
			}
			imp.setStack("x="+x+",y="+y+",z="+z1+" to "+z2+",width="+wZone+",height="+hZone,impS);
		}
		catch(IOException ex) {
			return (null);
		}
		return(imp);
	}
}

// pour la conservation des valeurs des zones de texte de boite DialogExtractStack
class InfoStack {
	public int x;
	public int y;
	public int width;
	public int height;
	public int z1;
	public int z2;	
	public int wImage;
	public int hImage;
	public char fileType;
}

//___________________________________________________________________________________________
// barre d'outil implementant tous les boutons
// la fermeture de la fenetre de la barre d'outil ferme toutes les vues

class ToolBar extends JFrame implements ActionListener {

	private JButton buttonExtractStack;	// Lecture d'un sous-volume dans un fichier ferme
	private JButton buttonSave;			// Sauvegarde de l'image

	private String s=File.separator;
	
	private ImageIcon icBar;

	private InfoStack infoSt;	// Objet contenant les valeurs des champs de la boite de dialogue stack
	private RawExtractor ext;

	File currentPath;			// chemin courant d'Image J		
	String absPath;				// chemin absolu des icones reconstruit
	

	// CONSTRUCTEUR
	public ToolBar() {
		try {
			currentPath=new File(".");
			absPath=currentPath.getCanonicalPath()+File.separator+"plugins"+File.separator+"icons"+File.separator;
			icBar=new ImageIcon(absPath+"logoesrf.gif");
		}
		catch(IOException e){IJ.error("Exception:\n"+e);}
	
		setIconImage(icBar.getImage());
		setTitle("File Extractor");

		Rectangle rAppli=new Rectangle();
		rAppli=IJ.getInstance().getBounds();
		setBounds(rAppli.x,rAppli.y+rAppli.height,rAppli.width/2,69);
		setResizable(false);
		IJ.getInstance().setResizable(true);

		Container contenu=getContentPane();
		GridBagLayout g=new GridBagLayout();
		contenu.setLayout(g);
		GridBagConstraints c=new GridBagConstraints();
		// pas d'ajustement de taille des composants
		c.fill=c.NONE;
		
		c.gridheight=	1;
		c.gridwidth=	1;		
		c.weightx=		40;
		c.weighty=		30;
		
		// ecouteur de fermeture de la barre d'outil
		ToolBarListener ear=new ToolBarListener();
 		addWindowListener(ear);
	
		// bouton d'extraction de volumes
		c.gridx=		1;
		c.gridy=		0;
		//buttonExtractStack=new JButton("extract stack");
		ImageIcon icst=new ImageIcon(absPath+"extractstack.gif");
		buttonExtractStack=new JButton(icst);
		contenu.add(buttonExtractStack,c);
		buttonExtractStack.addActionListener(this);
		buttonExtractStack.setToolTipText("Extract Stack");
		//buttonExtractStack.setBorder(BorderFactory.createRaisedBevelBorder());

		// bouton de sauvegarde
		c.gridx=		2;
		c.gridy=		0;
		//buttonSave=new JButton("save .info");
		ImageIcon icsav=new ImageIcon(absPath+"saveinfo.gif");
		buttonSave=new JButton(icsav);
		contenu.add(buttonSave,c);
		buttonSave.addActionListener(this);
		buttonSave.setToolTipText("Save .info parameters");
		//buttonSave.setBorder(BorderFactory.createRaisedBevelBorder());
		buttonSave.setFont(new Font("dialog.bold",Font.PLAIN,10));
	}

	class DialogExtractStack extends JDialog implements ActionListener {
		
		private boolean readSuccess;		// True en cas de succes de la lecture des parametres
											// False sinon
		private String readDir;				// si ReadSuccess=True, contient le directory
		private String readFile;			// sir ReadSuccess=True, contient le file
		private JButton buttonAutoParam;
		private JLabel labelparam;
		private JLabel labelx;
		public JTextField txtx;	
		private JLabel labely;
		public JTextField txty;	
		private JLabel labelw;
		private JTextField txtw;	
		private JLabel labelh;
		private JTextField txth;	
		private JLabel labelz1;
		public JTextField txtz1;	
		private JLabel labelz2;
		public JTextField txtz2;	
		private JLabel labelwImage;
		private JTextField txtwImage;	
		private JLabel labelhImage;
		private JTextField txthImage;	

		private ButtonGroup groupTypeFile;
		private JRadioButton b8bits;
		private JRadioButton b32bitsLittle;
		private JRadioButton b32bitsBig;

		private JButton buttonExtract;
		
		private Dimension reso;				// taille systeme de l'ecran
		private int frameWi;				// largeur de la frame a construire
		private int frameHe;				// hauteur de la frame a construire
	
		private ImageIcon icReadInfo=new ImageIcon(absPath+"readinfo.gif");
		private ImageIcon icExtract=new ImageIcon(absPath+"extract.gif");
	
		public DialogExtractStack(JFrame proprio) {
			
			super(proprio,"extract Stack",true);
			readSuccess=false;
			readDir=new String();
			readFile=new String();
	
			frameWi=200;	// largeur de la frame a construire
			frameHe=400;	// hauteur de la frame a construire
	
			reso=Toolkit.getDefaultToolkit().getScreenSize();
			setBounds((reso.width-frameWi)/2,(reso.height-frameHe)/2,frameWi,frameHe);
			setResizable(false);
			Container contenu=getContentPane();
			GridBagLayout g=new GridBagLayout();
			contenu.setLayout(g);
			GridBagConstraints c=new GridBagConstraints();
			// pas d'ajustement de taille des composants
			c.fill=c.NONE;
			
			c.gridheight=	1;
	
			// bouton Auto Param
			c.gridx=		0;
			c.gridy=		0;
			c.gridwidth=	2;
			c.weightx=		100;
			c.weighty=		20;
			buttonAutoParam=new JButton(icReadInfo);
			contenu.add(buttonAutoParam,c);
			buttonAutoParam.addActionListener(this);
			buttonAutoParam.setToolTipText("Read parameters from .info file");
			//buttonAutoParam.setBorder(BorderFactory.createRaisedBevelBorder());
	
			c.weightx=		50;
			c.weighty=		10;
	
			// label param lus
			c.gridx=		0;
			c.gridy=		1;
			labelparam=new JLabel("");
			labelparam.setFont(new Font("dialog.bold",Font.PLAIN,10));
			contenu.add(labelparam,c);
			
			
			c.gridwidth=	1;
			
			// label x
			c.gridx=		0;
			c.gridy=		2;
			labelx=new JLabel("x");
			labelx.setToolTipText("Absciss of top left point (from 0 to wImage-1)");
			contenu.add(labelx,c);
			
			// zone texte x
			c.gridx=		1;
			c.gridy=		2;
			txtx=new JTextField(4);
			txtx.setToolTipText("Absciss of top left point (from 0 to wImage-1)");
			contenu.add(txtx,c);
			
			// label y
			c.gridx=		0;
			c.gridy=		3;
			labely=new JLabel("y");
			labely.setToolTipText("Ordinate of top left point (from 0 to hImage-1)");
			contenu.add(labely,c);
			
			// zone texte y 
			c.gridx=		1;
			c.gridy=		3;
			txty=new JTextField(4);
			txty.setToolTipText("Ordinate of top left point (from 0 to hImage-1)");
			contenu.add(txty,c);
			
			// label width
			c.gridx=		0;
			c.gridy=		4;
			labelw=new JLabel("width");
			labelw.setToolTipText("Width of zone to extract (number of pixels>0)");
			contenu.add(labelw,c);
			
			// zone texte width
			c.gridx=		1;
			c.gridy=		4;
			txtw=new JTextField(4);
			txtw.setToolTipText("Width of zone to extract (number of pixels>0)");
			contenu.add(txtw,c);
			
			// label height
			c.gridx=		0;
			c.gridy=		5;
			labelh=new JLabel("height");
			labelh.setToolTipText("Height of zone to extract (number of pixels>0)");
			contenu.add(labelh,c);
			
			// zone texte height 
			c.gridx=		1;
			c.gridy=		5;
			txth=new JTextField(4);
			txth.setToolTipText("Height of zone to extract (number of pixels>0)");
			contenu.add(txth,c);
			
			// label z1 
			c.gridx=		0;
			c.gridy=		6;
			labelz1=new JLabel("z1");
			labelz1.setToolTipText("First Slice to extract (from 0 to number of slices-1)");
			contenu.add(labelz1,c);
			
			// zone texte z1 
			c.gridx=		1;
			c.gridy=		6;
			txtz1=new JTextField(4);
			txtz1.setToolTipText("First Slice to extract (from 0 to number of slices-1)");
			contenu.add(txtz1,c);
	
			// label z2 
			c.gridx=		0;
			c.gridy=		7;
			labelz2=new JLabel("z2");
			labelz2.setToolTipText("Last Slice to extract (from 0 to number of slices-1)");
			contenu.add(labelz2,c);
			
			// zone texte z2 
			c.gridx=		1;
			c.gridy=		7;
			txtz2=new JTextField(4);
			txtz2.setToolTipText("Last Slice to extract (from 0 to number of slices-1)");
			contenu.add(txtz2,c);
			
			// label wImage 
			c.gridx=		0;
			c.gridy=		8;
			labelwImage=new JLabel("wImage");
			labelwImage.setToolTipText("Width of source image (number of pixels>0)");
			contenu.add(labelwImage,c);
			
			// zone texte wImage 
			c.gridx=		1;
			c.gridy=		8;
			txtwImage=new JTextField(4);
			txtwImage.setToolTipText("Width of source image (number of pixels>0)");
			contenu.add(txtwImage,c);
			
			// label hImage 
			c.gridx=		0;
			c.gridy=		9;
			labelhImage=new JLabel("hImage");
			labelhImage.setToolTipText("Height of source image (number of pixels>0)");
			contenu.add(labelhImage,c);
			
			// zone texte hImage 
			c.gridx=		1;
			c.gridy=		9;
			txthImage=new JTextField(4);
			txthImage.setToolTipText("Height of source image (number of pixels>0)");
			contenu.add(txthImage,c);

			c.gridwidth=	2;
			
			// radio buttons 8bits, 32bits Little-Endian, 32bits Big-Endian
			groupTypeFile=new ButtonGroup();
			
			c.gridx=		0;
			c.gridy=		10;
			b8bits=new JRadioButton("8 bits",true);
			groupTypeFile.add(b8bits);
			b8bits.setToolTipText("8 bits");
			contenu.add(b8bits,c);
			
			c.gridx=		0;
			c.gridy=		11;
			b32bitsLittle=new JRadioButton("32 bits Little-Endian");
			b32bitsLittle.setToolTipText("32 bits, Little-Endian Byte Order");
			groupTypeFile.add(b32bitsLittle);
			contenu.add(b32bitsLittle,c);
			
			c.gridx=		0;
			c.gridy=		12;
			b32bitsBig=new JRadioButton("32 bits Big-Endian");
			b32bitsBig.setToolTipText("32 bits, Big-Endian Byte Order");
			groupTypeFile.add(b32bitsBig);
			contenu.add(b32bitsBig,c);
	
			// bouton Extract
			c.gridx=		0;
			c.gridy=		13;
			c.gridwidth=	2;
			c.gridheight=	1;
			c.weightx=		100;
			c.weighty=		20;
			buttonExtract=new JButton(icExtract);
			contenu.add(buttonExtract,c);
			buttonExtract.addActionListener(this);
			buttonExtract.setToolTipText("Process to extraction");
			//buttonExtract.setBorder(BorderFactory.createRaisedBevelBorder());
		}
		
		public void actionPerformed(ActionEvent e) {
			
			if (e.getSource()==buttonAutoParam) {
				// boite de dialogue d'ouverture de fichiers ImageJ
				// deuxieme argument=path; si non null: pas d'ouverture de fenetre
				OpenDialog fileToProcess=new OpenDialog("Read info parameters","");
				String dir=fileToProcess.getDirectory();
				String fil=fileToProcess.getFileName();
				ext=new RawExtractor(dir,fil);
				boolean success=false;
	
				try { success=ext.setParameters(); }
				// exception sur fichier de parametre
				catch (IOException exParam) { IJ.error("File problem.\nException:\n"+exParam); }
				// tous les parametres lus avec succes
				if (success) {
					readSuccess=true;

					// parametres lus dans le fichier
					infoSt.wImage=ext.getWidth();
					txtwImage.setText(""+infoSt.wImage);

					infoSt.hImage=ext.getHeight();
					txthImage.setText(""+infoSt.hImage);

					infoSt.z2=ext.getLength()-1;
					txtz2.setText(""+infoSt.z2);

					infoSt.fileType=ext.getFileType();
					if (infoSt.fileType=='a')
						b8bits.setSelected(true);
					else if (infoSt.fileType=='b')
						b32bitsLittle.setSelected(true);
					else if (infoSt.fileType=='c')
						b32bitsBig.setSelected(true);

					// parametres par defaut
					infoSt.width=infoSt.wImage;
					txtw.setText(""+infoSt.width);

					infoSt.height=infoSt.hImage;
					txth.setText(""+infoSt.height);

					labelparam.setText("reading successfully");
					txtz2.requestFocus();
					readDir=dir;
					readFile=fil;
					String end;
					end=readFile.substring(readFile.length()-5,readFile.length());
					if (end.equalsIgnoreCase(".info"))
						readFile=readFile.substring(0,readFile.length()-5);
				}
				// au moins un parametre indetermine
				// un echec de determination des parametres remet les champs a blancs
				// et les valeurs correspondant a 0
				else {
					readSuccess=false;
					infoSt.wImage=0;
					infoSt.hImage=0;
					txtwImage.setText("");
					txthImage.setText("");
					txtw.setText("");
					txth.setText("");
					labelparam.setText("not found");
					txtx.requestFocus();
					IJ.error("Parameters not found");
				}
			}
			
			String dir;
			String fil;
			
			if (e.getSource()==buttonExtract) {
	
				// verification des donnees
				// pour eviter plantage de parseInt
				try {
					infoSt.x=Integer.parseInt(txtx.getText());
					infoSt.y=Integer.parseInt(txty.getText());
					infoSt.width=Integer.parseInt(txtw.getText());
					infoSt.height=Integer.parseInt(txth.getText());
					infoSt.z1=Integer.parseInt(txtz1.getText());
					infoSt.z2=Integer.parseInt(txtz2.getText());
					infoSt.wImage=Integer.parseInt(txtwImage.getText());
					infoSt.hImage=Integer.parseInt(txthImage.getText());
				}
				catch (NumberFormatException exn) {
					IJ.error ("Bad parameters");
					return;
				}
	
				// si pas d'exception,
				// verification de la coherence des parametres:
	
				// x et y pas dans l'image
				// ne verifie pas que z1 et z2>zmax de l'image
				if (infoSt.x<0||infoSt.x>=infoSt.wImage||infoSt.y<0||infoSt.y>=infoSt.hImage||infoSt.z1<0||infoSt.z2<0) {
					IJ.error("Incoherent parameters: X, Y, Z1 or Z2 out of image");
					return;
				}
				// hauteur ou largeur de la zone a extraire plus grande que celle de l'image
				if (infoSt.width>infoSt.wImage||infoSt.height>infoSt.hImage) {
					IJ.error("Incoherent parameters: height or width of zone to long");
					return;
				}
				// zone deborde l'image
				//  x a droite     y en bas
				if ((infoSt.x+infoSt.width)>(infoSt.wImage)||(infoSt.y+infoSt.height)>(infoSt.hImage)) {
					IJ.error("Incoherent parameters: zone to extract out of image");
					return;
				}
				if (infoSt.z2<infoSt.z1) {
					IJ.error("Incoherent parameters: Z2 must be larger ou equal than Z1");
					return;
				}
	
				if (readSuccess==false) {
					// boite de dialogue d'ouverture de fichiers ImageJ
					// deuxieme argument=path; si non null: pas d'ouverture de fenetre
					OpenDialog fileToProcess=new OpenDialog("extract Stack from...","");
					dir=fileToProcess.getDirectory();
					fil=fileToProcess.getFileName();
				}
				else {
					dir=readDir;
					fil=readFile;
				}
	
				// Extraction et affichage de la couche
				// dir et fil null si bouton Cancel ou Fermeture de la fenetre

				if (!dir.equals(null) && !fil.equals(null)) {
					
					ext=new RawExtractor(dir,fil);
					ImagePlus impNew;
					impNew=null;

					if (b8bits.isSelected()) {
						impNew=ext.subStack8(infoSt.x,infoSt.y,infoSt.width,infoSt.height,infoSt.z1,infoSt.z2,infoSt.wImage,infoSt.hImage);
					}
					
					else if (b32bitsLittle.isSelected()) {
						impNew=ext.subStack32(infoSt.x,infoSt.y,infoSt.width,infoSt.height,infoSt.z1,infoSt.z2,infoSt.wImage,infoSt.hImage,true);
					}
					else if (b32bitsBig.isSelected()) {
						impNew=ext.subStack32(infoSt.x,infoSt.y,infoSt.width,infoSt.height,infoSt.z1,infoSt.z2,infoSt.wImage,infoSt.hImage,false);
					}

					if (impNew==null)
						IJ.error("Bad parameters");
					else {
						dispose();
						impNew.show();
						impNew.updateAndDraw();
					}
				}
			}
		}
	}

	//_______________________________________________
	// gestion des evenements clics des boutons
	
	public void actionPerformed(ActionEvent e) {
		Object b=e.getSource();
		
		// ----- Extract stack -----
		
		if (b==buttonExtractStack) {
				infoSt=new InfoStack();
				JFrame st=new JFrame();
				ImageIcon icStack=new ImageIcon(absPath+"pie.gif");
				st.setIconImage(icStack.getImage());
				DialogExtractStack dialStack=new DialogExtractStack(st);
				// parametres par defaut
				infoSt.x=0;
				infoSt.y=0;
				infoSt.z1=0;
				infoSt.z2=0;
				dialStack.txtx.setText(""+infoSt.x);
				dialStack.txty.setText(""+infoSt.y);
				dialStack.txtz1.setText(""+infoSt.z1);
				dialStack.txtz2.setText(""+infoSt.z2);
				dialStack.setVisible(true);
		}

		// ----- Save with .info -----
		
		String fileName=new String();
		
		if (b==buttonSave) {

		// recup du nom de l'image
		String name=""+WindowManager.getCurrentImage().getWindow();
		SaveDialog fileToSave=new SaveDialog("Save file",""+name,".raw.info");

		String stringEnd;
		fileName=fileToSave.getFileName();
		stringEnd=fileName.substring(fileName.length()-5,fileName.length());

		if (!stringEnd.equalsIgnoreCase(".info")) {
			fileName=fileName+".info";
		}
			try {
				FileWriter f=new FileWriter(""+fileToSave.getDirectory()+fileName);
				PrintWriter sortie=new PrintWriter (f);	// .raw.info
				String numX=new String();
				String numY=new String();
				String numZ=new String();
				
 				if (infoSt!=null) {
 					int z=infoSt.z2-infoSt.z1+1;
					numX=""+infoSt.width;
					numY=""+infoSt.height;
					numZ=""+(infoSt.z2-infoSt.z1+1);
				}
				sortie.println("NUM_X =      "+numX);
				sortie.println("NUM_Y =      "+numY);
				sortie.println("NUM_Z =      "+numZ);
				if (infoSt.fileType=='a')
					sortie.println("RECAD =\tYES");
				if (infoSt.fileType=='b')
					sortie.println("BYTEORDER = LOWBYTEFIRST");
				if (infoSt.fileType=='c')
					sortie.println("BYTEORDER = HIGHBYTEFIRST");
				sortie.close();
				
				IJ.showMessage("File saved as "+fileName);
				
				if (infoSt!=null) infoSt=null;
			}
			catch (IOException ex) {IJ.error("Exception:\n"+ex);}
		}
	}
	
	//_______________________________________________
	// ecouteur de fermeture de la fenetre contenant la barre d'outil
	// declenche la fermeture des vues
	
	public class ToolBarListener extends WindowAdapter {
		public void windowClosing(WindowEvent e) {
			//IJ.error("actions de fermeture ici");
		}
	}
}