#
#  xmltv.py - Python interface to XMLTV format, based on XMLTV.pm
#
#  Copyright (C) 2001 James Oakley
#
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Lesser General Public
#  License as published by the Free Software Foundation; either
#  version 2.1 of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public
#  License along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

import unittest
from StringIO import StringIO
from logilab.common.testlib import TestCase

from narval.extensions.xmltv import *
from narval.actions.Entertainment import *

XML_DATA = """<?xml version="1.0" encoding="iso-8859-1"?>
    <!DOCTYPE tv SYSTEM "xmltv.dtd">
    <tv date="20030811003608 -0300" source-info-url="http://www.funktronics.ca/python-xmltv" source-info-name="Funktronics" generator-info-name="python-xmltv" generator-info-url="http://www.funktronics.ca/python-xmltv">
      <channel id="C10eltv.zap2it.com">
        <display-name>Channel 10 ELTV</display-name>
        <url>http://www.eastlink.ca/</url>
      </channel>
      <channel id="C11cbht.zap2it.com">
        <display-name lang="en">Channel 11 CBHT</display-name>
        <icon src="http://tvlistings2.zap2it.com/tms_network_logos/cbc.gif"/>
      </channel>
      <programme start="20030702000000 ADT" channel="C23robtv.zap2it.com" stop="20030702003000 ADT">
        <title>This Week in Business</title>
        <category>Biz</category>
        <category>Fin</category>
        <date>2003</date>
        <audio>
          <stereo>stereo</stereo>
        </audio>
      </programme>
      <programme start="20030702000000 ADT" channel="C36wuhf.zap2it.com" stop="20030702003000 ADT">
        <title>Seinfeld</title>
        <sub-title>The Engagement</sub-title>
        <desc>In an effort to grow up, George proposes marriage to former girlfriend Susan.</desc>
        <category>Comedy</category>
        <country>USA</country>
        <language>English</language>
        <orig-language>English</orig-language>
        <premiere lang="en">Not really. Just testing</premiere>
        <last-chance>Hah!</last-chance>
        <credits>
          <actor>Jerry Seinfeld</actor>
          <producer>Larry David</producer>
        </credits>
        <date>1995</date>
        <length units="minutes">22</length>
        <episode-num system="xmltv_ns">7 . 1 . 1/1</episode-num>
        <video>
          <colour>1</colour>
          <present>1</present>
          <aspect>4:3</aspect>
        </video>
        <audio>
          <stereo>stereo</stereo>
        </audio>
        <previously-shown start="19950921103000 ADT" channel="C12whdh.zap2it.com"/>
        <new/>
        <subtitles type="teletext">
          <language>English</language>
        </subtitles>
        <rating system="VCHIP">
          <value>PG</value>
          <icon src="http://some.ratings/PGicon.png" width="64" height="64"/>
        </rating>
        <star-rating>
          <value>4/5</value>
          <icon src="http://some.star/icon.png" width="32" height="32"/>
        </star-rating>
      </programme>
    </tv>
    """

HEADER = {
          'date':"20030811003608 -0300",
          'source-info-url':"http://www.funktronics.ca/python-xmltv",
          'source-info-name':"Funktronics",
          'generator-info-name':"python-xmltv",
          'generator-info-url':"http://www.funktronics.ca/python-xmltv"}


PROGRAMMES = [{'audio': [{'stereo': u'stereo'}],
               'category': [(u'Biz', u''), (u'Fin', u'')],
               'channel': u'C23robtv.zap2it.com',
               'date': [u'2003'],
               'start': u'20030702000000 ADT',
               'stop': u'20030702003000 ADT',
               'title': [(u'This Week in Business', u'')]},
              {'audio': [{'stereo': u'stereo'}],
               'category': [(u'Comedy', u'')],
               'channel': u'C36wuhf.zap2it.com',
               'country': [(u'USA', u'')],
               'credits': [{'producer': [u'Larry David'], 'actor': [u'Jerry Seinfeld']}],
               'date': [u'1995'],
               'desc': [(u'In an effort to grow up, George proposes marriage to former girlfriend Susan.',
                         u'')],
               'episode-num': [(u'7 . 1 . 1/1', u'xmltv_ns')],
               'language': [(u'English', u'')],
               'last-chance': [(u'Hah!', u'')],
               'length': [{'units': u'minutes', 'length': 22}],
               'new': [1],
               'orig-language': [(u'English', u'')],
               'premiere': [(u'Not really. Just testing', u'en')],
               'previously-shown': [{'channel': u'C12whdh.zap2it.com',
                                     'start': u'19950921103000 ADT'}],
               'rating': [{'icon': [{'height': u'64',
                                     'src': u'http://some.ratings/PGicon.png',
                                     'width': u'64'}],
                           'system': u'VCHIP',
                           'value': u'PG'}],
               'star-rating': [{'icon': [{'height': u'32',
                                                  'src': u'http://some.star/icon.png',
                                          'width': u'32'}],
                                'value': u'4/5'}],
               'start': u'20030702000000 ADT',
               'stop': u'20030702003000 ADT',
               'sub-title': [(u'The Engagement', u'')],
               'subtitles': [{'type': u'teletext', 'language': (u'English', u'')}],
                       'title': [(u'Seinfeld', u'')],
               'video': [{'colour': 1, 'aspect': u'4:3', 'present': 1}]}]

CHANNELS = [{'display-name': [(u'Channel 10 ELTV', u'')],
             'id': u'C10eltv.zap2it.com',
             'url': [u'http://www.eastlink.ca/']},
            {'display-name': [(u'Channel 11 CBHT', u'en')],
             'icon': [{'src': u'http://tvlistings2.zap2it.com/tms_network_logos/cbc.gif'}],
             'id': u'C11cbht.zap2it.com'}]

class XmltvTest(TestCase):

    def test_read_all(self):
        xmldata = StringIO(XML_DATA)
        output = read_data(xmldata)
        self.assertDictEquals(output, HEADER)
        
    def test_read_programmes(self):
        xmldata = StringIO(XML_DATA)
        output = read_programmes(xmldata)
        self.assertEquals(len(output), len(PROGRAMMES))
        for out, known in zip(output, PROGRAMMES):
            self.assertDictEquals(out, known)

    def test_read_channels(self):
        xmldata = StringIO(XML_DATA)
        output = read_channels(xmldata)
        self.assertEquals(len(output), len(CHANNELS))
        for out, known in zip(output, CHANNELS):
            self.assertDictEquals(out, known)

    def test_write(self):
        """ Test the writer """
        string_file = StringIO('')
        w = Writer(string_file, encoding="iso-8859-1",
                   date="20030811003608 -0300",
                   source_info_url="http://www.funktronics.ca/python-xmltv",
                   source_info_name="Funktronics",
                   generator_info_name="python-xmltv",
                   generator_info_url="http://www.funktronics.ca/python-xmltv")
        for c in CHANNELS:
            w.write_channel(c)
        for p in PROGRAMMES:
            w.write_programme(p)
        w.end()
        string_file.seek(0)
        for written, original in zip(string_file.read().splitlines(),
                                    XML_DATA.splitlines()):
            self.assertEquals(written.strip(), original.strip())

if __name__ == '__main__':
    unittest.main()
