import unittest
from narval.extensions.xml_template import *

class GeneratorTC(unittest.TestCase):

    def setUp(self):
        self.root = XMLTemplateNode('rdf:RDF', {})
        self.root.prefixes = {'rdf': 'http://www.w3.org/1999/02/22-rdf-syntax-ns#',
                         'rdfs': 'http://www.w3.org/2000/01/rdf-schema#',
                         'foaf': 'http://xmlns.com/foaf/0.1/',
                         }
        templ = XMLTemplateNode('foaf:Person', {'rdf:about': '%s'})
        templ.set_cardinality('*')
        templ.append( XMLTemplateNode('foaf:isDescribedIn', {'rdf:resource': '%s'}))
        templ.append( XMLTemplateNode('foaf:name', {}, content='%s'))
        child_person = XMLTemplateNode('foaf:Person', {})
        child_person.append( XMLTemplateNode('rdfs:seeAlso', {'rdf:resource' :'%s'}))
        knows = XMLTemplateNode('foaf:knows', {})
        knows.append(child_person)
        knows.set_cardinality('*')
        templ.append( knows )
        templ.append( XMLTemplateNode('foaf:interest', {'rdf:resource': '%s'}))
        templ[-1].set_cardinality('*')
        self.root.append(templ)

    def test_generation(self):
        gen = Generator(self.root)
        self.assertEquals(gen.next_question(), 'add a new foaf:Person ?')
        gen.push_answer('yes')
        self.assertEquals(gen.next_question(), 'rdf:about for foaf:Person :')
        gen.push_answer('http://www.logilab.fr/~toto')
        self.assertEquals(gen.next_question(), 'rdf:resource for foaf:Person/foaf:isDescribedIn :')
        gen.push_answer('http://www.logilab.fr/~toto')
        self.assertEquals(gen.next_question(), 'foaf:name for foaf:Person :')
        gen.push_answer('Toto')
        self.assertEquals(gen.next_question(), 'add a new foaf:knows to foaf:Person ?')
        gen.push_answer('oui')
        self.assertEquals(gen.next_question(), 'rdf:resource for foaf:Person/foaf:knows/foaf:Person/rdfs:seeAlso :')
        gen.push_answer('http://www.logilab.fr/~tutu')
        self.assertEquals(gen.next_question(), 'add a new foaf:knows to foaf:Person ?')
        gen.push_answer('y')
        self.assertEquals(gen.next_question(), 'rdf:resource for foaf:Person/foaf:knows/foaf:Person/rdfs:seeAlso :')
        gen.push_answer('http://www.logilab.fr/~titi')
        self.assertEquals(gen.next_question(), 'add a new foaf:knows to foaf:Person ?')
        gen.push_answer('no')
        self.assertEquals(gen.next_question(), 'add a new foaf:interest to foaf:Person ?')
        gen.push_answer('o')
        self.assertEquals(gen.next_question(), 'rdf:resource for foaf:Person/foaf:interest :')
        gen.push_answer('http://www.logilab.fr/projects/narval')
        self.assertEquals(gen.next_question(), 'add a new foaf:interest to foaf:Person ?')
        gen.push_answer('n')
        self.assertEquals(gen.next_question(), 'add a new foaf:Person ?')
        gen.push_answer('n')
        self.assertRaises(NoMoreQuestion, gen.next_question)
        for i, (l1, l2) in enumerate(zip('''<?xml version="1.0" encoding="UTF-8"?>
<rdf:RDF xmlns:foaf="http://xmlns.com/foaf/0.1/" xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#" xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#">
  <foaf:Person rdf:about="http://www.logilab.fr/~toto">
    <foaf:isDescribedIn rdf:resource="http://www.logilab.fr/~toto"/>
    <foaf:name>Toto</foaf:name>
    <foaf:knows>
      <foaf:Person>
        <rdfs:seeAlso rdf:resource="http://www.logilab.fr/~tutu"/>
      </foaf:Person>
    </foaf:knows>
    <foaf:knows>
      <foaf:Person>
        <rdfs:seeAlso rdf:resource="http://www.logilab.fr/~titi"/>
      </foaf:Person>
    </foaf:knows>
    <foaf:interest rdf:resource="http://www.logilab.fr/projects/narval"/>
  </foaf:Person>
</rdf:RDF>'''.splitlines(), gen.as_xml().splitlines())):
            self.assertEquals(l1, l2, '%r != %r on index %s' % (l1, l2, i))



class ReaderTC(unittest.TestCase):

    def setUp(self):
        self.template = XMLTemplateReader().from_string('''
<rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
         xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#"
         xmlns:foaf="http://xmlns.com/foaf/0.1/">
<foaf:Person rdf:about="%s">
  <foaf:isDescribedIn rdf:resource="%s"/>
  <foaf:name>%s</foaf:name>
  <foaf:mbox rdf:resource="%s"/>
  <foaf:nick>%s</foaf:nick>
  <foaf:homepage rdf:resource="%s" />
  
  <foaf:interest rdf:resource="%s"/>
  <foaf:interest rdf:resource="%s"/>

  <foaf:currentProject rdf:resource="%s"/>
  <foaf:currentProject rdf:resource="%s"/>

  <foaf:knows rdf:resource="%s"/>
  <foaf:knows rdf:resource="%s"/>
</foaf:Person>
<foaf:Person/>
</rdf:RDF>
''')

    def test_base(self):
        template = self.template
        self.assertEquals(template.qname, 'rdf:RDF')
        self.assertEquals(len(template), 1)
        self.assertEquals(template.prefixes,
                          {'rdf': 'http://www.w3.org/1999/02/22-rdf-syntax-ns#',
                           'rdfs': 'http://www.w3.org/2000/01/rdf-schema#',
                           'foaf': 'http://xmlns.com/foaf/0.1/'})
        self.assertEquals(template.attrs, {})
        self.assertEquals(template.content, '')
        person_template = template[0]
        self.assertEquals(person_template.qname, 'foaf:Person')
        self.assertEquals(len(person_template), 8, list(person_template))
        self.assertEquals(person_template.attrs, {'rdf:about': '%s'})
        self.assertEquals(person_template.content, '')
        self.assertEquals(person_template[0].qname, 'foaf:isDescribedIn')
        self.assertEquals(person_template[0].content, '')
        self.assertEquals(person_template[1].qname, 'foaf:name')
        self.assertEquals(person_template[1].content, '%s')
        self.assertEquals(person_template[2].qname, 'foaf:mbox')
        self.assertEquals(person_template[3].qname, 'foaf:nick')
        self.assertEquals(person_template[4].qname, 'foaf:homepage')
        self.assertEquals(person_template[5].qname, 'foaf:interest')
        self.assertEquals(person_template[6].qname, 'foaf:currentProject')
        self.assertEquals(person_template[7].qname, 'foaf:knows')
        
    def test_cardinality(self):
        template = self.template
        self.assertEquals(template.cardinality, '')
        person_template = template[0]
        self.assertEquals(person_template.cardinality, '*')
        self.assertEquals(person_template[0].cardinality, '')
        self.assertEquals(person_template[1].cardinality, '')
        self.assertEquals(person_template[2].cardinality, '')
        self.assertEquals(person_template[3].cardinality, '')
        self.assertEquals(person_template[4].cardinality, '')
        self.assertEquals(person_template[5].cardinality, '*')
        self.assertEquals(person_template[6].cardinality, '*')
        self.assertEquals(person_template[7].cardinality, '*')
        
if __name__ == '__main__':
    unittest.main()
