# Copyright (c) 2004 DoCoMo Euro-Labs GmbH (Munich, Germany).
# Copyright (c) 2001-2004 LOGILAB S.A. (Paris, FRANCE).
#
# http://www.docomolab-euro.com/ -- mailto:tarlano@docomolab-euro.com
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""narval interpreter's configuration unit tests

:version: $Revision:$   
:author: Logilab

:copyright:
  2001-2004 LOGILAB S.A. (Paris, FRANCE)
  
  2004 DoCoMo Euro-Labs GmbH (Munich, Germany)

:contact:
  http://www.logilab.fr/ -- mailto:contact@logilab.fr
  
  http://www.docomolab-euro.com/ -- mailto:tarlano@docomolab-euro.com
"""

__revision__ = "$Id: config.py 20 2004-04-15 14:43:51Z syt $"
__docformat__ = "restructuredtext en"

import unittest
import os
import sys
from narval import config
from narval.config import normjoin

class ConfiguratorTest(unittest.TestCase):
    def setUp(self):
        self.confile = '/tmp/narval.conf' 
        f = open(self.confile, 'w')
        f.write('[Directories]\n')
        f.write('HOME: None\n')
        f.write('SHARE: /usr/share/narval\n')
        f.close()
        self.configurator = config.Configurator()
        os.environ['NARVAL_HOME']='/tmp/foo'
        if 'NARVAL_SHARE' in os.environ:
            del os.environ['NARVAL_SHARE']

    def tearDown(self):
        os.unlink(self.confile)
        del os.environ['NARVAL_HOME']

    def testReadFileConfig(self):
        self.configurator._parse_file(self.confile)
        config = self.configurator.get_configuration()
        self.assertEquals(None, config.get_home())
        self.assertEquals('/usr/share/narval', config.get_share())

    def testReadEnvironmentConfig(self):
        self.configurator._parse_environ()
        config = self.configurator.get_configuration()
        self.assertEquals('/tmp/foo', config.get_home())
        self.assertEquals(None, config.get_share())
        if sys.platform != 'win32':
            self.assertEquals(os.path.expanduser('~'), config.get_user_home())

    def testReadRegistryConfig(self):
        self.configurator._parse_registry()
        config = self.configurator.get_configuration()
        if sys.platform == 'win32':
            self.assertEquals('/tmp/foo', config.get_home())
            self.assertEquals(None, config.get_share())
        else:
            self.assertEquals(None, config.get_share())
            self.assertEquals(None, config.get_home())

    def testCommandLineConfig(self):
        self.configurator.set_home_from_command_line('/tmp/toto')
        self.configurator.set_share_from_command_line('/tmp/titi')
        config = self.configurator.get_configuration()
        self.assertEquals('/tmp/toto', config.get_home())
        self.assertEquals('/tmp/titi', config.get_share())

    def testCommandLineHasHigherPriority(self):
        self.configurator.set_home_from_command_line('/tmp/toto')
        self.configurator.set_share_from_command_line('/tmp/titi')
        self.configurator._parse_registry()
        self.configurator._parse_environ()
        self.configurator._parse_file(self.confile)
        config = self.configurator.get_configuration()
        self.assertEquals('/tmp/toto', config.get_home())
        self.assertEquals('/tmp/titi', config.get_share())
        

    def testGuessConfigFileLocation(self):
        head, tail = os.path.split(os.path.abspath(__file__))
        userconf = normjoin(head, '..',  'conf', 'narval.conf')
        #userconf = os.path.expanduser('~/.narval.conf')
        candidates = self.configurator._guess_candidate_config_files()
        self.assertEquals(candidates, ['/etc/narval/narval.conf', userconf])
        config.__file__ = '/usr/lib/python2.3/site-packages/narval/config.py'
        candidates = self.configurator._guess_candidate_config_files()
        self.assertEquals(candidates, ['/etc/narval/narval.conf'])
        #config.__file__ = '/home/alf/lib/python/narval/config.py'
        #candidates = self.configurator._guess_candidate_config_files()
        #self.assertEquals(candidates, ['/etc/narval/narval.conf', os.path.expanduser('~/etc/narval.conf'), userconf])
            
if __name__ == '__main__':
    unittest.main()

