# Copyright (c) 2004-2005 DoCoMo Euro-Labs GmbH (Munich, Germany).
# Copyright (c) 2001-2005 LOGILAB S.A. (Paris, FRANCE).
#
# http://www.docomolab-euro.com/ -- mailto:tarlano@docomolab-euro.com
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""utilities for serializing / deserializing of narval elements


:version: $Revision:$ 
:author: Logilab 

:copyright:
  2001-2005 LOGILAB S.A. (Paris, FRANCE)
  
  2004-2005 DoCoMo Euro-Labs GmbH (Munich, Germany)
  
:contact:
  http://www.logilab.fr/ -- mailto:contact@logilab.fr
  
  http://www.docomolab-euro.com/ -- mailto:tarlano@docomolab-euro.com
"""

__revision__ = '$Id: misc.py,v 1.1 2001/12/06 10:45:17 syt Exp $'
__docformat__ = 'restructuredtext en'

from mx.DateTime import strptime

def attr_string_value(string, attr=None):
    """transform an xml string to a python string: nothing has to be
    done since the xml parser has already transformed back entities (isn't it ?)
    """
    return string

def attr_string_rev_value(string):
    """transform a python string to an xml attribute string: escape some
    special chars such as <, >, ', \"
    """
    return (string.replace('"', '&quot;').replace("'", '&apos;')
            .replace('<', '&lt;').replace('>', '&gt;'))


def yn_value(string, attr=None):
    """map 'yes' / 'no' to their equivalent boolean values

    :type string: str
    :param string: the attribute value, 'yes' or 'no'

    :type attr: str or None
    :param attr: the name of the attribute, used if necessary in error message

    :raise ValueError: if the given value is not 'yes' or 'no'

    :rtype: bool
    :return: the boolean value of the attribute
    """
    if string == 'yes':
        return True
    elif string == 'no':
        return False
    if not attr:
        raise ValueError('got %r, expected "yes" or "no"' % string)
    raise ValueError('got %r for %r, expected "yes" or "no"' % (string, attr))

def yn_rev_value(boolean):
    """map boolean values to their equivalent yes / no

    :type boolean: bool
    :param boolean: the boolean value

    :rtype: str
    :return: 'yes' or 'no' according to the boolean value
    """
    if boolean:
        return 'yes'
    return 'no'

def date_value(string, attr=None):
    """get a DateTime object from an ISO date string
    
    :type string: str
    :param string: the ISO date string

    :rtype: `mx.DateTime.DateTime`
    :return: the mx date object
    """
    try:
        return strptime(string, '%Y-%m-%d')
    except:
        raise ValueError('got %r for %r, expected %%Y/%%m/%%d' % (string, attr))
        
def date_rev_value(date):
    """get an ISO date string from a DateTime object
    
    :type date: `mx.DateTime.DateTime`
    :param date: the mx date object

    :rtype: str
    :return: the equivalent ISO date string
    """
    return date.strftime('%Y-%m-%d')

def date_time_value(string):
    """get a DateTime object from an ISO date/time string
    
    :type string: str
    :param string: the ISO date/time string

    :rtype: `mx.DateTime.DateTime`
    :param date: the mx date object
    """
    return strptime(string, '%Y-%m-%d %H:%M')

def date_time_rev_value(datetime):
    """get an ISO date/time string from a DateTime object
    
    :type date: `mx.DateTime.DateTime`
    :param date: the mx date object

    :rtype: str
    :return: the equivalent ISO date/time string
    """
    return datetime.strftime('%Y-%m-%d %H:%M')

def float_rev_value(float_value):
    """serialize a float value using 2 number after the comma

    :type float_value: float
    :param date: the float number

    :rtype: str
    :return: the equivalent string
    """
    return '%.2f' % float_value


C_STEP = 1
C_PLAN = 2
C_PARENT = 3
C_MEM = 4

_CONTEXT = {'step': C_STEP, 'plan': C_PLAN,
           'parent_plan': C_PARENT, 'memory': C_MEM}
_REV_CONTEXT = {C_STEP: 'step',
               C_PLAN: 'plan',
               C_PARENT: 'parent_plan',
               C_MEM: 'memory'}

def context_value(string):
    """get the context number from a string

    :type string: str
    :param string: one of "step", "plan", "parent_plan", "memory"
    
    :rtype: int
    :return: the context number
    """
    return _CONTEXT[string]

def context_rev_value(context_num):
    """get the context name from its number

    :type context_num: int
    :param context_num: the context number
    
    :rtype: str
    :return: the context name
    """
    return _REV_CONTEXT[context_num]
