# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.

# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""tests for the astng builder module

Copyright (c) 2003-2005 LOGILAB S.A. (Paris, FRANCE).
http://www.logilab.fr/ -- mailto:contact@logilab.fr
"""

__revision__ = "$Id: unittest_astng_builder.py,v 1.1 2005/04/25 14:47:04 syt Exp $"

import unittest
from os.path import join, abspath

from logilab.common.astng import builder, astng, ASTNGManager, Module

import data
from data import module as test_module


class BuilderTC(unittest.TestCase):
        
    def setUp(self):
        self.builder = builder.ASTNGBuilder(ASTNGManager())
        
    def test_border_cases(self):
        """check that a file with no trailing new line is parseable"""
        self.builder.file_build('data/noendingnewline.py', 'data.noendingnewline')
        self.assertRaises(astng.ASTNGBuildingException,
                          self.builder.file_build, 'data/inexistant.py', 'whatever')
        
    def test_inspect_build(self):
        """test astng tree build from a living object"""
        import __builtin__
        builtin_astng = self.builder.inspect_build(__builtin__)
        fclass = builtin_astng.locals['file']
        self.assert_('name' in fclass.locals)
        self.assert_('mode' in fclass.locals)
        self.assert_('read' in fclass.locals)
        self.assert_(isinstance(fclass.locals['read'], astng.Function))
        # check builtin function has argnames == None
        dclass = builtin_astng.locals['dict']
        self.assertEquals(dclass.locals['has_key'].argnames, None)
        # 
        import time
        time_astng = self.builder.module_build(time)
        self.assert_(time_astng)
        #
        unittest_astng = self.builder.inspect_build(unittest)
        

    def test_package_name(self):
        """test base properties and method of a astng module"""
        datap = self.builder.file_build('data/__init__.py', 'data')
        self.assertEquals(datap.name, 'data')
        self.assertEquals(datap.package, 1)
        datap = self.builder.file_build('data/__init__.py', 'data.__init__')
        self.assertEquals(datap.name, 'data')
        self.assertEquals(datap.package, 1)
    
class BuilderFileBuildTC(unittest.TestCase):

    def setUp(self):
        abuilder = builder.ASTNGBuilder(ASTNGManager())
        self.module = abuilder.file_build('data/module.py', 'data.module')

    def test_module_base_props(self):
        """test base properties and method of a astng module"""
        module = self.module
        self.assertEquals(module.name, 'data.module')
        self.assertEquals(module.doc, "test module for astng\n")
        self.assertEquals(module.source_line(), 0)
        self.assertEquals(module.parent, None)
        self.assertEquals(module.get_frame(), module)
        self.assertEquals(module.root(), module)
        self.assertEquals(module.file, join(abspath(data.__path__[0]), 'module.py'))
        self.assertEquals(module.pure_python, 1)
        self.assertEquals(module.package, 0)
        self.assert_(not module.is_statement())
        self.assertEquals(module.get_statement(), module)
        self.assertEquals(module.node.get_statement(), module)
        
    def test_module_locals(self):
        """test the 'locals' dictionary of a astng module"""
        module = self.module
        _locals = module.locals
        self.assertEquals(len(_locals), 13)
        self.assert_(_locals is module.globals)
        keys = _locals.keys()
        keys.sort()
        self.assertEquals(keys, ['MY_DICT', 'YO', 'YOUPI',
                                 '__revision__',
                                 'clean', 'cvrtr', 'debuild', 'global_access',
                                 'modutils', 'nested_args', 'os', 'redirect',
                                 'spawn'])

    def test_function_base_props(self):
        """test base properties and method of a astng function"""
        module = self.module
        function = module.locals['global_access']
        self.assertEquals(function.name, 'global_access')
        self.assertEquals(function.doc, 'function test')
        self.assertEquals(function.source_line(), 14)
        self.assert_(function.parent)
        self.assertEquals(function.get_frame(), function)
        self.assertEquals(function.parent.get_frame(), module)
        self.assertEquals(function.root(), module)
        self.assertEquals(function.argnames, ['key', 'val'])
        self.assertEquals(function.is_method(), 0)

    def test_function_locals(self):
        """test the 'locals' dictionary of a astng function"""
        _locals = self.module.locals['global_access'].locals
        self.assertEquals(len(_locals), 4)
        keys = _locals.keys()
        keys.sort()
        self.assertEquals(keys, ['i', 'key', 'local', 'val'])
        
    def test_class_base_props(self):
        """test base properties and method of a astng class"""
        module = self.module
        klass = module.locals['YO']
        self.assertEquals(klass.name, 'YO')
        self.assertEquals(klass.doc, 'hehe')
        self.assertEquals(klass.source_line(), 27)
        self.assert_(klass.parent)
        self.assertEquals(klass.get_frame(), klass)
        self.assertEquals(klass.parent.get_frame(), module)
        self.assertEquals(klass.root(), module)
        self.assertEquals(klass.basenames, [])

    def test_class_locals(self):
        """test the 'locals' dictionary of a astng class"""
        module = self.module
        klass1 = module.locals['YO']
        klass2 = module.locals['YOUPI']
        locals1 = klass1.locals
        locals2 = klass2.locals
        keys = locals2.keys()
        keys.sort()
        self.assertEquals(locals1.keys(), ['a', '__init__'])
        self.assertEquals(keys, ['__init__', 'class_attr', 'class_method',
                                'method', 'static_method'])

    def test_class_instance_attrs(self):
        module = self.module
        klass1 = module.locals['YO']
        klass2 = module.locals['YOUPI']
        self.assertEquals(klass1.instance_attrs.keys(), ['yo'])
        self.assertEquals(klass2.instance_attrs.keys(), ['member'])

    def test_class_basenames(self):
        module = self.module
        klass1 = module.locals['YO']
        klass2 = module.locals['YOUPI']
        self.assertEquals(klass1.basenames, [])
        self.assertEquals(klass2.basenames, ['YO'])
        
    def test_method_base_props(self):
        """test base properties and method of a astng method"""
        klass2 = self.module.locals['YOUPI']
        # "normal" method
        method = klass2.locals['method']
        self.assertEquals(method.name, 'method')
        self.assertEquals(method.argnames, ['self'])
        self.assertEquals(method.doc, 'method test')
        self.assertEquals(method.source_line(), 47)
        self.assertEquals(method.is_method(), 1)
        self.assertEquals(method.is_static_method(), 0)
        self.assertEquals(method.is_class_method(), 0)
        # class method
        method = klass2.locals['class_method']
        self.assertEquals(method.argnames, ['cls'])
        self.assertEquals(method.is_method(), 1)
        self.assertEquals(method.is_static_method(), 0)
        self.assertEquals(method.is_class_method(), 1)
        # static method
        method = klass2.locals['static_method']
        self.assertEquals(method.argnames, [])
        self.assertEquals(method.is_method(), 1)
        self.assertEquals(method.is_static_method(), 1)
        self.assertEquals(method.is_class_method(), 0)

    def test_method_locals(self):
        """test the 'locals' dictionary of a astng method"""
        klass2 = self.module.locals['YOUPI']
        method = klass2.locals['method']
        _locals = method.locals
        self.assertEquals(len(_locals), 6)
        keys = _locals.keys()
        keys.sort()
        self.assertEquals(keys, ['MY_DICT', 'a', 'autre', 'b', 'local', 'self'])


class BuilderModuleBuildTC(BuilderFileBuildTC):

    def setUp(self):
        abuilder = builder.ASTNGBuilder(ASTNGManager())
        self.module = abuilder.module_build(test_module)

    def test_object_attr(self):
        self.assertEquals(self.module.object, test_module)
        function = self.module.locals['global_access']
        self.assertEquals(function.object, test_module.global_access)
        klass = self.module.locals['YO']
        self.assertEquals(klass.object, test_module.YO)

__all__ = ('BuilderModuleBuildTC', 'BuilderFileBuildTC', 'BuilderTC')

if __name__ == '__main__':
    unittest.main()
