# Copyright (c) 2002 Detlev Offenbach <detlev@die-offenbachs.de>

import unittest
import string
import sys
import traceback
import time
import re

from qt import *

from UnittestForm import UnittestForm
from Icons import IconProvider

class UnittestDialog(UnittestForm):
    def __init__(self,prog = None,fromEric = 0,parent = None,name = None,modal = 0,fl = 0):
        UnittestForm.__init__(self,parent,name,modal,fl)
        
        self.setCaption(self.trUtf8("Unittest"))
        if not fromEric:
            self.closeButton.setText(self.trUtf8("&Exit"))
        
        self.fileHistory = QStringList()
        self.running = 0
        self.savedModulelist = None
        self.savedSysPath = sys.path
        self.fromEric = fromEric
        if prog:
            self.insertProg(prog)

    def colorizeProgressbar(self, color):
        """
        private methode to set the color of the progressbar
        """
        pal = self.progressProgressBar.palette()
        pal.setColor(QColorGroup.Highlight, color)
        self.progressProgressBar.setPalette(pal)
        
    def insertProg(self, prog):
        """
        public slot to insert the filename prog into the 
        testsuiteComboBox object
        """
        # prepend the selected file to the testsuite combobox
        self.fileHistory.remove(prog)
        self.fileHistory.prepend(prog)
        self.testsuiteComboBox.clear()
        self.testsuiteComboBox.insertStringList(self.fileHistory)
        
    def handleFileDialog(self):
        """
        private slot to open a file dialog
        """
        prog = QFileDialog.getOpenFileName(None, 
                    self.trUtf8("Python Files (*.py)"), self)
        
        if prog.isNull():
            return
        
        self.insertProg(prog)
        
    def handleStartTest(self):
        """
        private slot to start the test
        """
        if self.running:
            return
        
        prog = self.testsuiteComboBox.currentText()
        if not str(prog):
            QMessageBox.critical(self, 
                    self.trUtf8("Unittest"), 
                    self.trUtf8("You must enter a test suite file."))
            return
        
        # prepend the selected file to the testsuite combobox
        self.fileHistory.remove(prog)
        self.fileHistory.prepend(prog)
        self.testsuiteComboBox.clear()
        self.testsuiteComboBox.insertStringList(self.fileHistory)
        self.sbLabel.setText(self.trUtf8("Preparing Testsuite"))
        qApp.processEvents()
        
        # build the testname from the filename without extension and ".suite"
        fi = QFileInfo(prog)
        testName = str(fi.baseName(1)) + ".suite"
        sys.path = [str(fi.dirPath(1))] + self.savedSysPath
        
        # clean up list of imported modules to force a reimport upon running the test
        if self.savedModulelist:
            for modname in sys.modules.keys():
                if not self.savedModulelist.has_key(modname):
                    # delete it
                    del(sys.modules[modname])
        self.savedModulelist = sys.modules.copy()
        
        # now try to generate the testsuite
        try:
            test = unittest.defaultTestLoader.loadTestsFromName(testName)
        except:
            exc_type, exc_value, exc_tb = sys.exc_info()
            QMessageBox.critical(self, 
                    self.trUtf8("Unittest"),
                    self.trUtf8("Unable to run test <b>%1</b>.<br>%2<br>%3")
                        .arg(testName)
                        .arg(str(exc_type))
                        .arg(str(exc_value)))
            return
            
        self.testResult = QtTestResult(self)
        self.totalTests = test.countTestCases()
        self.running = 1
        self.handleRunning()
        startTime = time.time()
        test.run(self.testResult)
        stopTime = time.time()
        self.timeTaken = float(stopTime - startTime)
        self.running = 0
        self.handleStopped()
        
    def handleStopTest(self):
        """
        private slot to stop the test
        """
        if self.testResult:
            self.testResult.stop()
            
    def handleRunning(self):
        """
        private method to set the GUI in running mode
        """
        # reset counters and error infos
        self.runCount = 0
        self.failCount = 0
        self.errorCount = 0
        self.remainingCount = self.totalTests
        self.errorInfo = []

        # reset the GUI
        self.progressCounterRunCount.setText(str(self.runCount))
        self.progressCounterFailureCount.setText(str(self.failCount))
        self.progressCounterErrorCount.setText(str(self.errorCount))
        self.progressCounterRemCount.setText(str(self.remainingCount))
        self.errorsListBox.clear()
        self.progressProgressBar.setTotalSteps(self.totalTests)
        self.colorizeProgressbar(QColor("green"))
        self.progressProgressBar.reset()
        self.stopButton.setEnabled(1)
        self.startButton.setEnabled(0)
        qApp.processEvents()
        
    def handleStopped(self):
        """
        private method to set the GUI in stopped mode
        """
        self.startButton.setEnabled(1)
        self.stopButton.setEnabled(0)
        if self.runCount == 1:
            self.sbLabel.setText(self.trUtf8("Ran %1 test in %2s")
                .arg(self.runCount)
                .arg("%.3f" % self.timeTaken))
        else:
            self.sbLabel.setText(self.trUtf8("Ran %1 tests in %2s")
                .arg(self.runCount)
                .arg("%.3f" % self.timeTaken))

    def handleTestFailed(self, test, err):
        """
        public method called if a test fails
        """
        self.failCount = self.failCount + 1
        self.progressCounterFailureCount.setText(str(self.failCount))
        self.errorsListBox.insertItem(self.trUtf8("Failure: %1").arg(str(test)))
        self.errorInfo.append((test, err))
        
    def handleTestErrored(self, test, err):
        """
        public method called if a test errors
        """
        self.errorCount = self.errorCount + 1
        self.progressCounterErrorCount.setText(str(self.errorCount))
        self.errorsListBox.insertItem(self.trUtf8("Error: %1").arg(str(test)))
        self.errorInfo.append((test, err))
        
    def handleTestStarted(self, test):
        """
        public method called if a test is about to be run
        """
        self.sbLabel.setText(str(test))
        qApp.processEvents()
        
    def handleTestFinished(self, test):
        """
        public method called if a test has finished.
        It is also called if it has already failed or errored.
        """
        # update the counters
        self.remainingCount = self.remainingCount - 1
        self.runCount = self.runCount + 1
        self.progressCounterRunCount.setText(str(self.runCount))
        self.progressCounterRemCount.setText(str(self.remainingCount))
        
        # update the progressbar
        if self.errorCount:
            self.colorizeProgressbar(QColor("red"))
        elif self.failCount:
            self.colorizeProgressbar(QColor("orange"))
        self.progressProgressBar.setProgress(self.runCount)
        
    def handleListboxDoubleClick(self, lbitem):
        """
        private slot called by doubleclicking an errorlist entry.
        It will popup a dialog showing the stacktrace.
        If called from eric, an additional button is displayed
        to show the python source in an eric source viewer (in
        erics main window.
        """
        self.errListIndex = self.errorsListBox.index(lbitem)
        text = lbitem.text()

        # get the error info
        test, err = self.errorInfo[self.errListIndex]
        tracebackLines = apply(traceback.format_exception, err + (10,))
        tracebackText = string.join(tracebackLines, "")

        # now build the dialog
        self.dlg = QDialog()
        self.dlg.setCaption(text)
        self.dlg.resize(450,250)
        self.dlg.layout = QVBoxLayout(self.dlg,6,6)

        self.dlg.testLabel = QLabel(str(test), self.dlg)
        self.dlg.layout.addWidget(self.dlg.testLabel)

        self.dlg.traceback = QTextEdit(tracebackText, None, self.dlg)
        self.dlg.traceback.setTextFormat(Qt.PlainText)
        self.dlg.traceback.setReadOnly(1)
        self.dlg.layout.addWidget(self.dlg.traceback)

        buttonLayout = QHBoxLayout(None,0,6)
        spacer = QSpacerItem(0,0,QSizePolicy.Expanding,QSizePolicy.Minimum)
        buttonLayout.addItem(spacer)
        self.dlg.closeButton = QPushButton(self.trUtf8("&Close"), self.dlg)
        self.dlg.closeButton.setDefault(1)
        self.dlg.connect(self.dlg.closeButton, SIGNAL("clicked()"),
                        self.dlg, SLOT("accept()"))
        buttonLayout.addWidget(self.dlg.closeButton)
        # one more button if called from eric
        if self.fromEric:
            self.dlg.showButton = QPushButton(self.trUtf8("&Show Source"), 
                            self.dlg)
            self.dlg.connect(self.dlg.showButton, SIGNAL("clicked()"),
                            self.handleShowSource)
            buttonLayout.addWidget(self.dlg.showButton)
        spacer = QSpacerItem(0,0,QSizePolicy.Expanding,QSizePolicy.Minimum)
        buttonLayout.addItem(spacer)
        self.dlg.layout.addLayout(buttonLayout)

        # and now fire it up
        self.dlg.show()
        self.dlg.exec_loop()
        
    def handleShowSource(self):
        """
        private slot to show the source of a traceback in an
        eric source viewer.
        """
        if not self.fromEric:
            return
            
        # get the error info
        test, err = self.errorInfo[self.errListIndex]
        tracebackLines = apply(traceback.format_exception, err + (10,))
        fmatch = re.search(r'File "(.*?)", line (\d*?),.*', tracebackLines[1])
        if fmatch:
            fn, ln = fmatch.group(1, 2)
            self.emit(PYSIGNAL('unittestFile'),(fn,int(ln),1))

class QtTestResult(unittest.TestResult):
    """
    A TestResult derivative to work with a graphical GUI
    """
    def __init__(self, parent):
        unittest.TestResult.__init__(self)
        self.parent = parent
        
    def addFailure(self, test, err):
        unittest.TestResult.addFailure(self, test, err)
        self.parent.handleTestFailed(test, err)
        
    def addError(self, test, err):
        unittest.TestResult.addError(self, test, err)
        self.parent.handleTestErrored(test, err)
        
    def startTest(self, test):
        unittest.TestResult.startTest(self, test)
        self.parent.handleTestStarted(test)

    def stopTest(self, test):
        unittest.TestResult.stopTest(self, test)
        self.parent.handleTestFinished(test)
