"""
    Drawable1D.py
    1D Graphics displaying widget
"""
import qt
import qwt 
import Numeric 

from PyDVT import __version__,__date__,__author__
from Binding import Pen,Brush

SIZE_SIMBOL=qt.QSize(10,10)

def GetSymbolCode(symbol):
    SymbolCode={ 'circle':      qwt.QwtSymbol.Ellipse ,      \
                 'square':      qwt.QwtSymbol.Rect  ,      \
                 'diamond':     qwt.QwtSymbol.Diamond ,      \
                 'plus':        qwt.QwtSymbol.Cross  ,      \
                 'cross':       qwt.QwtSymbol.XCross  ,      \
                 'star':        qwt.QwtSymbol.LTriangle ,      \
                 'triangle':    qwt.QwtSymbol.Triangle         \
    }
    if symbol in SymbolCode.keys(): return  SymbolCode[symbol]
    else:                           return  SymbolCode['circle']




class GraphicalItem:

    def __init__(self,view,pen=Pen((0,0,0),1,"solid"),brush=None,yscale=0):
        self.view=view
        self.visible=0
        if pen is not None:self.pen=qt.QPen(pen.GetColor(), pen.GetWidth() ,pen.GetStyle())
        else:self.pen=qt.QPen(qt.QColor(0,0,0), 1 ,qt.QPen.SolidLine)        
        if brush is None: self.brush = qt.QBrush(qt.QBrush.NoBrush)
        else: self.brush = qt.QBrush(brush.GetColor(), brush.GetStyle() )         
        self.bounding_rect=None
        self.X0,self.Y0,self.X1,self.Y1=None,None,None,None
        self.yscale=yscale
        

    def erase(self,p=None):
        if self.visible:
            self.visible=0
            self._paint(p)
            
    def show(self,p=None):
         if self.visible==0:
             self.visible=1
             self._paint(p)

    def SetCoords(self,X0=None,Y0=None,X1=None,Y1=None):
        if (self.X0,self.Y0,self.X1,self.Y1)==(X0,Y0,X1,Y1): return        
        self.erase()
        (self.X0,self.Y0,self.X1,self.Y1)=(X0,Y0,X1,Y1)        
        self.show()

    def GetCoords(self):
        if self.X1 is not None and self.Y1 is not None: return (self.X0,self.Y0,self.X1,self.Y1)
        else: return (self.X0,self.Y0)
             

    def Update(self,pen=None,brush=None,textOptsDic=None):
        self.erase()
        if pen is not None: self.pen=qt.QPen(pen.GetColor(), pen.GetWidth() ,pen.GetStyle())
        if brush is not None: self.brush=qt.QBrush(brush.GetColor(), brush.GetStyle() )     
        if textOptsDic is not None:
            if textOptsDic.has_key('orient'): self.orient=textOptsDic['orient']
            if textOptsDic.has_key('align'): self.align=textOptsDic['align']
            if textOptsDic.has_key('text'): self.text=textOptsDic['text']
            if textOptsDic.has_key('style'):self.style=textOptsDic['style']
        self.show()

    def _paint (self,p=None):
        if p is None: p=qt.QPainter(self.view.qwt.canvas())
        if self.X0 is not None: tx0=self.view.qwt.transform(qwt.QwtPlot.xBottom, self.X0)
        else: tx0=None
        if self.X1 is not None: tx1=self.view.qwt.transform(qwt.QwtPlot.xBottom, self.X1)
        else: tx1=None
        if self.Y0 is not None:
            if self.yscale==1: ty0=self.view.qwt.transform(qwt.QwtPlot.yRight, self.Y0)
            else: ty0=self.view.qwt.transform(qwt.QwtPlot.yLeft, self.Y0)
        else: ty0=None
        if self.Y1 is not None:
            if self.yscale==1: ty1=self.view.qwt.transform(qwt.QwtPlot.yRight, self.Y1)
            else: ty1=self.view.qwt.transform(qwt.QwtPlot.yLeft, self.Y1)
        else: ty1=None
        self.BoundingRect=(tx0,ty0,tx1,ty1)
        p.setPen(self.pen)
        p.setBrush(self.brush)
        p.setRasterOp(qt.Qt.NotXorROP)
        self.paint(p)


    #VIRTUALS
    def paint(self,p):
        pass
    

class RectangleItem(GraphicalItem):        
    def paint(self,p):        
        x,y,w,h=self.BoundingRect[0],self.BoundingRect[1],self.BoundingRect[2]-self.BoundingRect[0],self.BoundingRect[3]-self.BoundingRect[1]
        p.drawRect( x,y,w,h)

                            
class LineItem(GraphicalItem):        
    def paint(self,p):        
        p.drawLine( self.BoundingRect[0],self.BoundingRect[1],self.BoundingRect[2],self.BoundingRect[3])


class SymbolItem(GraphicalItem):        
    def __init__(self,view,symbol="circle",pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0"),yscale=0):
        GraphicalItem.__init__(self,view,pen,brush,yscale)
        self.symbol=symbol

    def paint(self,p):
        #TODO: Just plotting circle        
        p.drawEllipse(qt.QRect ( qt.QPoint(self.BoundingRect[0]-SIZE_SIMBOL.width()/2+1,self.BoundingRect[1]-SIZE_SIMBOL.height()/2+1), SIZE_SIMBOL ) )


class TextItem(GraphicalItem):        
    def __init__(self,view,text, align="left",orient=0.,pen=Pen((0,0,0),1,"solid"),brush=None,style="normal",yscale=0):
        GraphicalItem.__init__(self,view,pen,brush,yscale)
        self.text=text
        self.align=align
        self.orient=orient
        self.style=style

    def paint(self,p):
        #TODO: FIXME
        p.drawText ( self.BoundingRect[0], self.BoundingRect[1],qt.QString(self.text),-1)
        #p.drawText ( self.BoundingRect[0], self.BoundingRect[1],40,20,qt.Qt.AlignHCenter or qt.Qt.AlignVCenter,qt.QString(self.text)) 


class Plot(qwt.QwtPlot):
    def drawCanvas(self,p):
        qwt.QwtPlot.drawCanvas(self,p)
        for obj in self.parent().GraphicObjects:
            obj.visible=0
            obj.show(p)




class Drawable_1D(qt.QGrid):
    """
    1D Graphics displaying widget interface definition
    
    Application Level Interface:
    ===========================
    This class has some methods that can be called at application level. They refer to adding
    and changing graphical elements over the graphic displaying. They are called through
    View.GetDrawable() object:    
        PutLine
        PutRectangle
        PutSymbol
        PutText
        SetObjectCoords
        GetObjectCoords
        EraseObject
        UpdateObject
        
    This class generates the following callbacks to parent (a View object):
        _RightButtonPress
        _ButtonPress
        _DoubleClick
        _ButtonRelease
        _Motion
        _ButtonPressMotion
        _KeyPress
    """
    def __init__(self,parent,zoom_mode="ON",scroll_mode="ON"):
        """
        Parameters:
            parent: parent widget (a View object)
            zoom_mode: "ON" or "OFF" "
            scroll_mode: "ON", "OFF" or "AUTO"
        """
        #qwt.QwtPlot.__init__(self,parent)

        #self.back_color = qt.QColor()
        #self.back_color.setRgb( 0xFF, 0xFF, 0xFF )
        if scroll_mode!="OFF":
            qt.QGrid.__init__(self,2,parent,None)
            self.qwt=  Plot(self)
            self.VScroll=qt.QScrollBar(qt.Qt.Vertical,self)
            self.HScroll=qt.QScrollBar(qt.Qt.Horizontal,self)
            if scroll_mode=="AUTO":
                self.VScroll.hide()
                self.HScroll.hide()
                self.qwt.setSizePolicy(qt.QSizePolicy(qt.QSizePolicy.Expanding,qt.QSizePolicy.Expanding))
                self.VScroll.setSizePolicy(qt.QSizePolicy(qt.QSizePolicy.Fixed,qt.QSizePolicy.Expanding))
                self.HScroll.setSizePolicy(qt.QSizePolicy(qt.QSizePolicy.Expanding,qt.QSizePolicy.Fixed))
        else:
            qt.QGrid.__init__(self,1,parent,None)
            self.qwt=  Plot(self)
        
        self.parent=parent
        self.ScrollMode=scroll_mode
        self.Gridlines=0
        self.init=1
        self.functions={}

        self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot=0,1,0,1,None,None
        self.xmin, self.xmax, self.ymin, self.ymax,self.y2min, self.y2max=self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot
        self.ZoomedX=0
        self.ZoomedY=0

        self.LogX=0
        self.LogY=0
        self.PolarEnv=0
         
        if self.ScrollMode!="OFF":
            self.HScroll.setMinValue(0)
            self.HScroll.setMaxValue(0)
            self.HScroll.setEnabled(0)
            self.VScroll.setMinValue(0)
            self.VScroll.setMaxValue(0)
            self.VScroll.setEnabled(0)

        self.style="Line"

        self.TitleLabel=self.XLabel=self.Y1Label=self.Y2Label=""
        self.GraphicObjects=[]        

        qt.QObject.connect(self.qwt,qt.SIGNAL('plotMouseMoved(const QMouseEvent&)'),self.plotMouseMoved)
        qt.QObject.connect(self.qwt,qt.SIGNAL('plotMousePressed(const QMouseEvent&)'),self.plotMousePressed)
        qt.QObject.connect(self.qwt,qt.SIGNAL('plotMouseReleased(const QMouseEvent&)'),self.plotMouseReleased)


        self.qwt.setAutoLegend(0)
        self.qwt.enableLegend(0)
        self.qwt.enableGridX(0)
        self.qwt.enableGridY(0)
        self.qwt.canvas().setMouseTracking(1)
        self.setFocusPolicy(qt.QWidget.StrongFocus)
        self.setFocus()
        

    
    def Show(self):
        """
        Displays widget
        """
        self.show()


    def GetHeight(self):
        """
        Returns height of the widget
        """
        return self.height()


    def GetWidth(self):
        """
        Returns width of the widget
        """
        return self.width()


    def ToggleGridlines(self):
        """
        Toggles gridline state (initial state: off)
        """
        if self.Gridlines == 0:
            self.qwt.enableGridX(1)
            self.qwt.enableGridY(1)
            self.Gridlines=1
        else:
            self.qwt.enableGridX(0)
            self.qwt.enableGridY(0)
            self.Gridlines=0
        self.qwt.replot()


    def SetLabels(self,title_label=None,x_label=None,y_label=None,y2_label=None):
        """
        Parameters:
            title_label: Label on the top of drawing
            x_label: Label for x axis
            y_label: Label for y axis
            y2_label: Label for second (right-side) y axis
        """
        if title_label is not None: self.qwt.setTitle(title_label)
        if x_label is not None: self.qwt.setAxisTitle(qwt.QwtPlot.xBottom, x_label)
        if y_label is not None: self.qwt.setAxisTitle(qwt.QwtPlot.yLeft, y_label)
        if y2_label is not None: self.qwt.setAxisTitle(qwt.QwtPlot.yRight, y2_label)

            
    def SetEnv(self,xmin,xmax,ymin,ymax,y2min=None,y2max=None,reset_zoom=0):
        """
        Parameters:
            xmin,xmax: lower,upper values of x axis
            ymin,ymax: lower,upper values of y axis
            y2min,y2max: lower,upper values of y2 axis
            reset_zoom: if non-zero, resets zoom state.
        """
        #self.qwt.setAxisAutoScale(qwt.QwtPlot.xBottom)
        #self.qwt.setAxisAutoScale(qwt.QwtPlot.yLeft)
        #self.qwt.setAxisAutoScale(qwt.QwtPlot.yRight)
        if reset_zoom or (self.xmintot, self.xmaxtot,self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot)!=(xmin,xmax,ymin,ymax,y2min,y2max):
            self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot=xmin,xmax,ymin,ymax,y2min,y2max        
            self.qwt.setAxisScale(qwt.QwtPlot.xBottom,xmin,xmax)        
            self.qwt.setAxisScale(qwt.QwtPlot.yLeft,ymin,ymax)        
            if y2min is not None and y2max is not None:
                self.qwt.enableAxis(qwt.QwtPlot.yRight)
                self.qwt.setAxisScale(qwt.QwtPlot.yRight,y2min,y2max)            
            else:
                self.qwt.enableAxis(qwt.QwtPlot.yRight,0)
            self.xmin, self.xmax, self.ymin, self.ymax,self.y2min, self.y2max=self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot
        


    def SetLogX(self,on,minvalue=1):
        """
        Parameters:
            on: if non-zero sets x axis logarithmic
                if zero, sets x axis linear
            minvalue: if on is non-zero sets the displaying value
                      for all values lass than or equals to zero
        """
        if on: self.qwt.setAxisOptions(qwt.QwtPlot.xBottom, qwt.QwtAutoScale.Logarithmic)
        else: self.qwt.setAxisOptions(qwt.QwtPlot.xBottom, qwt.QwtAutoScale.None)

        
    def SetLogY(self,on,minvalue=0.000001):
        """
        Sets drawing limits
        Parameters:
            on: if non-zero sets y axis logarithmic
                if zero, sets y axis linear
            minvalue: if on is non-zero sets the displaying value
                      for all values lass than or equals to zero
        """
        if on:
            self.qwt.setAxisOptions(qwt.QwtPlot.yLeft, qwt.QwtAutoScale.Logarithmic)
            self.qwt.setAxisOptions(qwt.QwtPlot.yRight, qwt.QwtAutoScale.Logarithmic)
        else:
            self.qwt.setAxisOptions(qwt.QwtPlot.yLeft, qwt.QwtAutoScale.None)
            self.qwt.setAxisOptions(qwt.QwtPlot.yRight, qwt.QwtAutoScale.None)

        
    def SetPolarEnv(self,value):
        """
        Parameters:
            value: if non-zero enables polar environment 
                   if zero, disables
        """
        #TODO
        pass


    def EnableLegend(self,enable,position="bottom"):
        """
        Parameters:
            enable: if non-zero adds legend. If zero, removes.
            position: if enable is non-zero  sets the position of the legend: "top", "left",
                      "right" or "bottom"
        """
        self.qwt.setAutoLegend(enable)
        if enable:
            if position == "top": self.qwt.setLegendPos(qwt.Qwt.Top)
            elif position == "left": self.qwt.setLegendPos(qwt.Qwt.Left)
            elif position == "right": self.qwt.setLegendPos(qwt.Qwt.Right)
            else : self.qwt.setLegendPos(qwt.Qwt.Bottom)
        

    def RemoveGraphicObjects(self):
        """
        Removes all graphical objects on the plotting
        (Added with the Put... methods)
        """
        self.GraphicObjects=[]

       
    def RemoveFunction(self,name=None):
        """
        Parameters:
            name: name of the function to be removed from the drawable.
                  if none, all are removed
        """
        if name is None:
            self.qwt.removeCurves()
            self.functions={}
        elif name in self.functions.keys():
            self.qwt.removeCurves(self.functions[name]["curve"])
            del self.functions[name]

            
    def SetFunction(self,name,function_pars):
        """
        Parameters:
            name: name of the function to be included
            function_pars: dictionary with the function properties (see GraphView)
                           In addiction:
                            "valid_range": tuple, defines a validity interval
                                           (the functions is only drawn in this interval)
        """
        #TODO: consider "valid_range" (scans)
        name=function_pars["name"]
        pen=function_pars["pen"]
        if name not in self.functions.keys(): 
            self.functions[name]={"curve":self.qwt.insertCurve(function_pars["name"]),"symbol":function_pars["symbol"],"yscale":function_pars["yscale"],"pen":pen}
        func=self.functions[name]["curve"]
        self.qwt.setCurvePen(func, qt.QPen(pen.GetColor(), pen.GetWidth() ,pen.GetStyle()))
        self._SetFunctionStyle(self.functions[name],self.style)
        self.qwt.setCurveData(func, function_pars["xdata"], function_pars["data"])
        if function_pars["yscale"] == 1: self.qwt.setCurveYAxis(func,qwt.QwtPlot.yRight)


    def SetFunctionItem(self,name,item,val):
        """
        Sets "item" of the function named "name" to the value "val"
        """        
        if name in self.functions.keys():
            self.functions[name][item]=val
            func=self.functions[name]["curve"]
            if item == "pen": self.qwt.setCurvePen(func, qt.QPen(val.GetColor(), val.GetWidth() ,val.GetStyle()))
            elif item == "yscale":
                if val == 1: self.qwt.setCurveYAxis(func,qwt.QwtPlot.yRight)
                else: self.qwt.setCurveYAxis(func,qwt.QwtPlot.yLeft)
            elif item == "symbol": self._SetFunctionStyle(self.functions[name],self.style)
                    

    def Redraw(self):
        """
        Draws itself        
        """
        self.qwt.replot()

    def IncrementalUpdate(self,function_name,range,restart=0):
        """
        Draws just the interval defined bu
        Parameters:
            function_name: name of the function
            range: tuple,(min,max) values for updating
            restart: if non-zero, erases previously displayed functions values
                     of function_name
        """
        pass


    def Save(self,filename,format):        
        """
        Parameters:
            filename: name of the output file.
            format: format to be saved ("PS","PNG","BMP" or "JPG")
        """
        #TODO: FINISH
        pix=qt.QPixmap(1200,800)#self.qwt.width(),self.qwt.height())
        pix.fill( qt.Qt.gray )
        opts=qwt.QwtPlotPrintFilter()        
        self.qwt.printPlot(pix,opts)

        if format=="PS" or format=="JPG":
            import Image,os
            tmpfile="__tmpfile.bmp"
            pix.save(tmpfile,"BMP")
            pilim=Image.open(tmpfile)
            if format=="PS":
                #pilim.save(filename,"EPS")
                #TODO: CORRECT THIS
                import Image,MyPSDraw
                WIDTH,HEIGHT=595,842
                xmin,ymin=30,140
                xmax,ymax=WIDTH-xmin,HEIGHT-ymin                
                
                f=open(filename,'w+')
                fontSize=14
                ps=MyPSDraw.MyPSDraw(f)
                ps.begin_document('test ps')
                box=(xmin,ymin,xmax,ymax)
                #title="Test postscript printing with PSDraw"
                #ps.setfont("Helvetica-Bold",fontSize)
                #ps.text((600/2-(0.5*fontSize*len(title))/2,(ymax*72)+10),title) # with stuff to center it
                ps.image(box,pilim,{'fit':0,'keepAspect':1})
                #ps.setink(5)
                ps.rectangle(box)
                #comments="isn't it beautiful to see these comments printed ???"
                #ps.setfont("Helvetica",fontSize) # need to do it each time text is added
                #ps.text((600/2-(0.44*fontSize*len(comments))/2,(ymin*72)-10),comments) # with stuff to center it
                ps.end_document()
                f.close()
                
            else:                
                pilim.save(filename,"JPEG")
            os.remove(tmpfile)
            return 1
        return pix.save(filename,format)
        
    
    def SetStyle(self,style):
        """
        Parameters:
            style: "Line", "Points", "PointsLine" or "Bars"
        """
        self.style=style
        for name in self.functions.keys(): self._SetFunctionStyle(self.functions[name],style)
        self.qwt.replot()


    def _SetFunctionStyle(self,function,style):
        func=function["curve"]
        symbol=function["symbol"]
        pen=self.qwt.curve(func).pen()        
        if style=="Line":
            self.qwt.setCurveStyle(func,qwt.QwtCurve.Lines)
            self.qwt.setCurveSymbol(func,qwt.QwtSymbol(qwt.QwtSymbol.None,qt.QBrush(),pen,SIZE_SIMBOL))
        elif style=="Points":
            self.qwt.setCurveStyle(func,qwt.QwtCurve.NoCurve)
            self.qwt.setCurveSymbol(func,qwt.QwtSymbol(GetSymbolCode(symbol),qt.QBrush(),pen,SIZE_SIMBOL))
        elif style=="PointsLine":
            self.qwt.setCurveStyle(func,qwt.QwtCurve.Lines)
            self.qwt.setCurveSymbol(func,qwt.QwtSymbol(GetSymbolCode(symbol),qt.QBrush(),pen,SIZE_SIMBOL))
        elif style=="Bars":
            if function["yscale"]==1: self.qwt.setCurveBaseline(func,self.y2mintot) 
            else: self.qwt.setCurveBaseline(func,self.ymintot)                                                                                                         
            self.qwt.setCurveStyle(func,qwt.QwtCurve.Sticks)
            self.qwt.setCurveSymbol(func,qwt.QwtSymbol(qwt.QwtSymbol.None,qt.QBrush(),pen,SIZE_SIMBOL))


    def SetZoom (self,(x0,y0),(x1,y1)):        
        """
        Parameters:
            (x0,y0): upper-left corner
            (x1,y1): upper-left corner 
        """
        self.xmin=min(x1,x0)
        self.xmax=max(x1,x0)
        self.ymin=min(y1,y0)
        self.ymax=max(y1,y0)
        self.qwt.setAxisScale(qwt.QwtPlot.yLeft,self.ymin, self.ymax)
        if self.y2mintot is not None and self.y2maxtot is not None:            
            self.y2min=(float (y0)-self.ymintot) * (float(self.y2maxtot)-self.y2mintot) /  (float(self.ymaxtot)-self.ymintot) +self.y2mintot
            self.y2max=(float (y1)-self.ymintot) * (float(self.y2maxtot)-self.y2mintot) /  (float(self.ymaxtot)-self.ymintot) +self.y2mintot
            self.qwt.setAxisScale(qwt.QwtPlot.yRight,self.y2min,self.y2max)
        else:
            self.y2min=self.y2max=None                    
        self.qwt.setAxisScale(qwt.QwtPlot.xBottom,self.xmin, self.xmax)

        if self.ScrollMode!="OFF":
            if self.ZoomedX:
                self.disconnect(self.HScroll,qt.SIGNAL("valueChanged(int)"),self.hscroll)
            if self.ZoomedY:
                self.disconnect(self.VScroll,qt.SIGNAL("valueChanged(int)"),self.vscroll)
            if self.xmin!=self.xmintot or self.xmax!=self.xmaxtot:
                self.HScroll.setMinValue(0)
                self.HScroll.setMaxValue(999)        
                self.HScroll.setValue((float(self.xmin)-self.xmintot)*999./((float(self.xmaxtot)-(self.xmax-self.xmin))-self.xmintot))
                self.HScroll.setEnabled(1)
                self.connect(self.HScroll,qt.SIGNAL("valueChanged(int)"),self.hscroll)
                if self.ScrollMode=="AUTO":
                    self.HScroll.show()                
            if self.ymin!=self.ymintot or self.ymax!=self.ymaxtot:
                self.VScroll.setMinValue(0)
                self.VScroll.setMaxValue(999)        
                self.VScroll.setValue(999-((float(self.ymin)-self.ymintot)*999./((float(self.ymaxtot)-(self.ymax-self.ymin))-self.ymintot)))
                self.VScroll.setEnabled(1)
                self.connect(self.VScroll,qt.SIGNAL("valueChanged(int)"),self.vscroll)
                if self.ScrollMode=="AUTO":
                    self.VScroll.show()                

        if self.xmin!=self.xmintot or self.xmax!=self.xmaxtot:self.ZoomedX=1
        if self.ymin!=self.ymintot or self.ymax!=self.ymaxtot:self.ZoomedY=1
        
        self.qwt.replot()


    def ResetZoom(self,update=1):
        """
        Parameters:
            update: if non-zero triggers a Redraw
        """
        self.qwt.setAxisScale(qwt.QwtPlot.xBottom,self.xmintot, self.xmaxtot)        
        self.qwt.setAxisScale(qwt.QwtPlot.yLeft,self.ymintot, self.ymaxtot)        
        if self.y2mintot is not None and self.y2maxtot is not None:
            self.qwt.setAxisScale(qwt.QwtPlot.yRight,self.y2mintot, self.y2maxtot)
        self.xmin, self.xmax, self.ymin, self.ymax,self.y2min, self.y2max=self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot

        if self.ScrollMode!="OFF":           
            if self.ZoomedX:
                self.disconnect(self.HScroll,qt.SIGNAL("valueChanged(int)"),self.hscroll)
                self.HScroll.setEnabled(0)       
                self.HScroll.setMinValue(0)
                self.HScroll.setMaxValue(0)
                if self.ScrollMode=="AUTO":
                    self.HScroll.hide()
                self.ZoomedX=0
            if self.ZoomedY:
                self.disconnect(self.VScroll,qt.SIGNAL("valueChanged(int)"),self.vscroll)
                self.VScroll.setEnabled(0)       
                self.VScroll.setMinValue(0)
                self.VScroll.setMaxValue(0)
                if self.ScrollMode=="AUTO":
                    self.VScroll.hide()
                self.ZoomedY=0
        self.qwt.replot()

		
    def PutLine(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid"),yscale=0):
        """
        Parameters:
            X0,Y0: First point (data coords = coordinates of the displayed axis)
            X1,Y1: Second point (data coords = coordinates of the displayed axis)
            pen: Pen object
            yscale: if equals to 1 it's referenced to second y axis
        Returns a reference to the object
        """
        obj=LineItem(self,pen,None,yscale)        
        self.GraphicObjects.append(obj)
        self.SetObjectCoords(obj,X0,Y0,X1,Y1)
        return obj

    
    def PutRectangle(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0"),yscale=0):
        """
        Parameters:
            X0,Y0: upper-left corner (data coords = coordinates of the displayed axis)
            X1,Y1: lower-right corner (data coords = coordinates of the displayed axis)
            pen: Pen object to the contour
            brush: Brush object for the filling
            yscale: if equals to 1 it's referenced to second y axis
        Returns a reference to the object
        """
        obj=RectangleItem(self,pen,brush,yscale)        
        self.GraphicObjects.append(obj)
        self.SetObjectCoords(obj,X0,Y0,X1,Y1)
        return obj
        

    def PutSymbol(self,X0,Y0,symbol="circle",pen=Pen((0,0,0),1,"solid"),yscale=0):
        """
        Parameters:
            X0,Y0: Location (data coords = coordinates of the displayed axis)
            symbol: 'circle', square', 'diamond', 'plus', 'cross', 'star' or 'triangle'        
            pen: Pen object
            yscale: if equals to 1 it's referenced to second y axis
        Returns a reference to the object
        """
        obj=SymbolItem(self,symbol,pen,None,yscale)         
        self.GraphicObjects.append(obj)
        self.SetObjectCoords(obj,X0,Y0,None,None)
        return obj


    def PutText(self,X0,Y0,text, align="center",orient=0.,pen=Pen((0,0,0),1,"solid"),style="normal",yscale=0,brush=Brush((255,255,255),"fill_100")):
        """
        Parameters:
            X0,Y0: Location (data coords = coordinates of the displayed axis)
            align: 'center', 'left' or 'right'
            orient: angle (in degrees) of the text
            pen: Pen object for the text
            style: 'normal', 'bold' oe 'italic'
            yscale: if equals to 1 it's referenced to second y axis
            brush: Brush object for the background
        Returns a reference to the object
        """
        obj=TextItem(self,text,align,orient,pen,brush,style,yscale)        
        self.GraphicObjects.append(obj)
        self.SetObjectCoords(obj,X0,Y0,None,None)
        return obj


    def SetObjectCoords(self,obj,*args):
        """
        Changes the coordinates of an graphical object
        Parameters:
            obj: reference to the object
            *args: X0,Y0,X1,Y1,... in data coords = coordinates of the displayed axis
        """
        obj.SetCoords(*args)


    def GetObjectCoords(self,obj):
        """
        Parameters:
            obj: reference to the object
        Returns the coordinates of the graphical object in data coords = coordinates
        of the displayed axis
        """
        return obj.GetCoords()

    
    def EraseObject(self,obj):
        """
        Erases graphical object
        Parameters:
            obj: reference to the object
        """
        obj.erase ()
        self.GraphicObjects.remove(obj)
        


    def UpdateObject(self,obj,pen=None,brush=None,textOptsDic=None):
        """
        Changes properties of an object
        Parameters:
            pen: new Pen object
            brush: new Brush object
            textOptsDic: dictionary with new text properties
                         keys: 'orient', 'align', 'text', 'style'
                         values as in PutText method
        """
        if obj in self.GraphicObjects:
            obj.Update(pen,brush,textOptsDic)
            

    def GetVisibleXAxis(self):
        """
        Returns the visible x axis (lower and higher values)
        """
        return (self.xmin,self.xmax)


    def GetVisibleYAxis(self):
        """
        Returns the visible y axis (lower and higher values)
        """
        return (self.ymin,self.ymax)


    def GetVisibleY2Axis(self):
        """
        Returns the visible y2 axis (lower and higher values)
        """
        return (self.y2min,self.y2max)
    

    def GetFunctionScale(self,name):
        """
        Returns which y axis the function named "name" is referenced to
        """
        try:
            return self.functions[name]["yscale"]
        except:
            return None

    
    def ImageCoord2ViewCoord(self, coord):
        """
        Tranforms image coordinates (referenced to original image) into 
        view coordinates (as generated by the events)
        """
        return coord        


    def ViewCoord2ImageCoord(self, coord):
        """
        Tranforms view coordinates (as generated by the events) into
        image coordinates ( referenced to original image )
        """
        return coord        


    def DataCoord2ViewCoord(self, coord):
        """
        Tranforms data coordinates (coordinates of the displayed axis) into 
        view coordinates (as generated by the events)
        """
        pass        


    def ViewCoord2DataCoord(self, coord):
        """
        Tranforms view coordinates (as generated by the events) into 
        data coordinates (coordinates of the displayed axis)
        """
        data_coord =(self.qwt.invTransform(qwt.QwtPlot.xBottom, coord[0]), \
               self.qwt.invTransform(qwt.QwtPlot.yLeft, coord[1]))        
        return (1,data_coord)


    def Destroy(self):
        """
        Cleanup
        """
        self.functions={}
        self.qwt=  None
        self.parent=None
        self.GraphicObjects=[]   
        self.VScroll=None
        self.HScroll=None


    def GetViewport(self):
        """
        CALLED BY Binding MODULE
        """
        return self.qwt.canvas()


    def plotMouseMoved(self, e):
        if e.state()==0:
           self.parent._Motion(self.ViewCoord2ImageCoord((e.pos().x(),e.pos().y())))  
        elif e.state()==1:
           self.parent._ButtonPressMotion(self.ViewCoord2ImageCoord((e.pos().x(),e.pos().y())))

    def plotMousePressed(self, e):
        import time 
        now=time.time()

        if e.button() == qt.Qt.RightButton:
            if self.parent is not None: self.parent._RightButtonPress(e.globalPos ())
        elif e.button() == qt.Qt.LeftButton:
            self.parent._ButtonPress(self.ViewCoord2ImageCoord((e.pos().x(),e.pos().y())))

        #NOT NICE< BUT THERE'S NO SIGNAL TO DOUBLE CLICK        
        if hasattr(self,"last_click") and (now-self.last_click < 0.3):
            if self.parent is not None: self.parent._DoubleClick((e.pos().x(),e.pos().y()))
            del(self.last_click)
        else: self.last_click=time.time()
            
        

    def plotMouseReleased(self, e):
        if e.button() == qt.Qt.LeftButton:
            self.parent._ButtonRelease(self.ViewCoord2ImageCoord((e.pos().x(),e.pos().y())))


    def hscroll( self, value ):
        window=self.xmax-self.xmin
        self.xmin=(float(value)*((float(self.xmaxtot)-(window))-self.xmintot)/999.)+self.xmintot
        self.xmax=self.xmin+window
        self.qwt.setAxisScale(qwt.QwtPlot.xBottom,self.xmin, self.xmax)
        self.qwt.replot()

    def vscroll( self, value ):
        window=self.ymax-self.ymin
        value=999-value
        self.ymin=(float(value)*((float(self.ymaxtot)-(window))-self.ymintot)/999.)+self.ymintot
        self.ymax=self.ymin+window
        self.qwt.setAxisScale(qwt.QwtPlot.yLeft,self.ymin, self.ymax)
        if self.y2mintot is not None and self.y2maxtot is not None:            
            self.y2min=(float (self.ymin)-self.ymintot) * (float(self.y2maxtot)-self.y2mintot) /  (float(self.ymaxtot)-self.ymintot) +self.y2mintot
            self.y2max=(float (self.ymax)-self.ymintot) * (float(self.y2maxtot)-self.y2mintot) /  (float(self.ymaxtot)-self.ymintot) +self.y2mintot
            self.qwt.setAxisScale(qwt.QwtPlot.yRight,self.y2min,self.y2max)
        self.qwt.replot()


    def keyPressEvent(self,e):
       ret=str(e.text())
       if ret=="":
            ret= e.key()
       if self.parent is not None: self.parent._KeyPress(ret)
       e.ignore()
    