"""
    GraphView.py
    View derived class for 1d plotting
"""    

from PyDVT import __version__,__date__,__author__


from View import *
from Data import Data
from Filter import Filter
from DataSelection import OrthoLineSelection


import GraphViewSelect
import Command
import Numeric
import math
from BinarySearch import BinarySearch


class GraphFilter(Filter):
    """
    View's standard filter.
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0,yscale=0,xdata=None,pen=Pen((0,0,0),2,"solid"),symbol="none"):
        """
        Parameters:
            name: Optional  name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
            yscale: Optional. If ysacale==1 the graphic is referenced to right-sided y scale.
            xdata: Optional NumPy array defining x coordinates
                   (can be received by source  as well)
            pen: Optional Pen for drawing (it can be set afterwards with GraphView's SetPen)
            symbols: Optional string meaning the simbol for point representation.
                     It can be: 'circle','square','diamond','plus','cross','star','triangle'
                                'none' = no symbols
                     (it can be set afterwards with GraphView's SetSymbol)
        """
        Filter.__init__(self,source,synchronized,buffer_input)
        self.yscale=yscale
        if xdata is not None and type(xdata) != Numeric.arraytype: self.xdata=Numeric.array(xdata)
        else:self.xdata=xdata
        self.pen=pen
        self.symbol=symbol
        if name==None: self.name=str(self)
        else: self.name=name

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            "name": string: name of function or None
            "data": NumPy array
            "xdata" 1d NumPy array 
            "yscale": if 1, right-scale referenced
            "pen"  Pen object      
            "symbol"  plot symbol
        """
        sel=self.GetInput()
        if "data" not in sel.keys(): return {"name":self.name}
        if sel["data"] is None:  return {"name":self.name}
        array=sel["data"].flat

        if self.xdata is not None: ax= self.xdata
        elif "xdata" in sel.keys(): ax=sel["xdata"]
        else: ax=Numeric.arrayrange(array.shape[0])
        
        ret={"name":self.name,"data":array,"xdata":ax,"yscale":self.yscale,"pen":self.pen,"symbol":self.symbol}
        if "delimeters" in  sel.keys(): ret["delimeters"]=sel["delimeters"]
        return ret
        


class PolarFilter(GraphFilter):
    """
    View's standard filter in polar coordinates.
    Assumes receiving polar coords teta in "xdata" and r in "data"
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0,pen=Pen((0,0,0),2,"solid"),symbol="none"):
        """
        Parameters:
            name: Optional  name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
            pen: Optional Pen for drawing
            symbols: Optional string meaning the simbol for point representation.
                     It can be: 'circle','square','diamond','plus','cross','star','triangle'
                                'none' = no symbols
        """
        GraphFilter.__init__(self,name,source,synchronized,buffer_input,0,None,pen,symbol)

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys: same as GraphFilter
        """
        sel=GraphFilter.GetOutput(self)
        if "data" not in sel.keys():  return {"name":self.name}
        if "xdata" not in sel.keys(): return {"name":self.name}
        ar=Numeric.sin(sel["xdata"])*sel["data"]
        at=Numeric.cos(sel["xdata"])*sel["data"]
        sel["data"]=ar
        sel["xdata"]=at
        return sel
        
        

class Graph(GraphFilter):
    """
    View's simplified filter for direct NumPy interface.
    (creates and hides it's own data object)
    """
    def __init__(self,data=None,yscale=0,xdata=None,pen=Pen((0,0,0),2,"solid"),symbol="none",name=None):
        """
        Parameters:
            data: NumPy 1d array to be displayed
            yscale: Optional. If yscale==1 the graphic is referenced to right-sided y scale.
            xdata: Optional NumPy array defining x coordinates
                   (can be received by source  as well)
            pen: Optional Pen for drawing
            symbols: Optional string meaning the simbol for point representation.
                     It can be: 'circle','square','diamond','plus','cross','star','triangle'
                                'none' = no symbols
        """
        self.data=Data()
        if data is not None: self.data.AppendPage(array=data)
        sel=OrthoLineSelection(self.data,[0,],["ALL",],0)
        GraphFilter.__init__(self,name,sel,1,0,yscale,xdata,pen,symbol)

    def Destroy(self,source=None):
        """
        If filter gets out of scope, this should be called to destroy
        internally created data object.
        """
        if self.Alive==0: return
        self.Alive=0
        self.data.Destroy()


class PolarGraph(Graph):
    """
    View's simplified filter for direct NumPy interface in polar coordinates.
    (creates and hides it's own data object)
    """
    def __init__(self,rdata,tdata,pen=Pen((0,0,0),2,"solid"),symbol="none",name=None):
        """
        Parameters:
            rdata: NumPy 1d array - R coords
            tdata: NumPy 1d array - Teta coords
            pen: Optional Pen for drawing
            symbols: Optional string meaning the simbol for point representation.
                     It can be: 'circle','square','diamond','plus','cross','star','triangle'
                                'none' = no symbols
        """
        data=Numeric.sin(tdata)*rdata
        xdata=Numeric.cos(tdata)*rdata
        Graph.__init__(self,data,0,xdata,pen,symbol,name)
            



class HistFilter(GraphFilter):
    """
    View's standard filter for histogram calculation
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0,yscale=0,pen=Pen((0,0,0),2,"solid"),symbol="none"):
        """
        Parameters:
            name: Optional  name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
            yscale: Optional. If ysacale==1 the graphic is referenced to right-sided y scale.
            pen: Optional Pen for drawing
            symbols: Optional string meaning the simbol for point representation.
                     It can be: 'circle','square','diamond','plus','cross','star','triangle'
                                'none' = no symbols
        """
        
        GraphFilter.__init__(self,name,source,synchronized,buffer_input,yscale,None,pen,symbol)
        self.div=100

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys: same as GraphFilter
        """

        sel=GraphFilter.GetOutput(self)
        
        if sel is None: return {"name":self.name}
        try:
            import sys
            sort=Numeric.sort(sel["data"])
            if sort[-1]==sort[0]: end=sort[0]+0.000001
            else: end=sort[-1]
            step=float(end-sort[0])/self.div
            lev=Numeric.arange(sort[0],end,step)
            n = Numeric.searchsorted(sort, lev)
            n = Numeric.concatenate([n, [len(sel["data"])]])
            hist=n[1:]-n[:-1]
        except:
            return {"name":self.name}

        sel["data"]=hist
        sel["xdata"]=lev
        return sel        

    def SetDivisions(self,div):
        """
        Sets number of divisions for histogram calculation.
        Default is 100.
        """        
        self.div=div
        

class Hist(HistFilter):
    """
    View's simplified filter for calculate histograms, with direct NumPy interface 
    (creates and hides it's own data object)
    """
    def __init__(self,data=None,yscale=0,pen=Pen((0,0,0),2,"solid"),symbol="none",name=None):
        """
        Parameters:
            data: NumPy 1d array to be displayed
            yscale: Optional. If yscale==1 the graphic is referenced to right-sided y scale.
            pen: Optional Pen for drawing
            symbols: Optional string meaning the simbol for point representation.
                     It can be: 'circle','square','diamond','plus','cross','star','triangle'
                                'none' = no symbols
        """
        self.data=Data()
        if data is not None: self.data.AppendPage(array=data)
        sel=OrthoLineSelection(self.data,[0,],["ALL",],0)
        HistFilter.__init__(self,name,sel,1,0,yscale,pen,symbol)

    def Destroy(self,source=None):
        """
        If filter gets out of scope, this should be called to destroy
        internally created data object.
        """
        if self.Alive==0: return
        self.Alive=0
        self.data.Destroy()



class GraphView(View):
    """
    View derived class for graphical plotting of 1d arrays.
    GraphView expects the following keys from it's source (GetOutput
    method):
    "data": 1d NumPy array to be drawn (y values)
    "xdata":1d NumPy array with the x values. If None, take's [0 .. x dimention of "data"]
            Have to be ordered(ascending or descending), but in Polar mode.
    "name": Optional. Not used for the moment.
    "yscale": Optional. If equals to one, plotting is referenced to right-side scale.    
    "pen":  Optional. Pen for drawing current source.
    "symbol":  Optional. Symbol for drawing current source.
    "nopts": Optional. If set, defines the number of valid points in data
    "delimeters": Optional. List of tuples defining (start,end) slices of xdata.
                  When in style "Line" or "PointsLine", the extremes of these
                  slices defined by (start,end) are not connected by lines.
               


    Interface:
    ===========================
    View interface
    SetZoom
    ResetZoom
    ZoomTo
    ClearZoomSelect
    SetZoomStyle
    SetLabels
    SetXScaleLinear
    SetXScaleLog
    SetYScaleLinear
    SetYScaleLog
    GetPositionValues
    GetClosestFunction
    SetCursorType
    SetPolarEnv
    EnableLegend
    GetPolarCoords
    SetPen
    SetSymbol
    SetFunctionYScale
    SetStyle
    SetXAxis
    SetYAxis
    SetY2Axis
    GetXAxis
    GetYAxis
    GetY2Axis
    
    Overridables:    
    ===========================
    View overridables
    CreateMenu
    EventPosition 
    EventDoubleClick 


    Some of Drawable's methods are callable at application level (they're accessed
    through View.GetDrawable() (see GUIBinding.Drawable1D):    
    """
    def __init__(self, parent=None, pars={}, **kw):
        """
        See View.__init__
        Parameters:
          parent: Parent window
          pars:   Dictionary with View initialization options
                  New options defined by this class (in addiction to base class options):
                   "AddStyleSelect": Adds graph style selection in popup menu
                                     (line, points, line+points and bars)
                                     Default: 0
                   "AddCursorSelect": Adds cursor selection in popup menu
                                     (crosshairs, vertica and none)
                                     Default: 0
                   "AddStatus": Adds status bar to the view to display mouse position
                                or function values.
                                Default: 0
                   "AutoHideStatus": if "AddCursorSelect" is selected, hides status
                                     bar if "None" cursor is sele1cted.
                                     Default: 0
                   "AddLog": Adds log scale option in popup menu
                             Default: 1
          kw:     keywords to Container initializatregisterion                
        """
        View.__init__(self, parent, pars,**kw)
        if "AddStyleSelect" in pars.keys(): self.AddStyleSelect=pars["AddStyleSelect"]
        else: self.AddStyleSelect=0
        if "AddCursorSelect"  in pars.keys(): self.AddCursorSelect=pars["AddStyleSelect"]
        else: self.AddCursorSelect=0
        if "AddStatus"  in pars.keys(): self.AddStatus=pars["AddStatus"]
        else: self.AddStatus=0
        if "AutoHideStatus"  in pars.keys(): self.AutoHideStatus=pars["AutoHideStatus"]
        else: self.AutoHideStatus=0        
        if "AddLog"  in pars.keys(): self.AddLog=pars["AddLog"]
        else: self.AddLog=1
        
        self.CreateMenu()
        self.Functions={}
        self.XAxis=self.BackXAxis=self.YAxis=self.BackYAxis=self.Y2Axis=self.BackY2Axis=None
        self.LogX=self.LogY=0
        self.PolarEnv=0
        self.minxval,self.maxxval,self.minyval,self.maxyval,self.miny2val,self.maxy2val=0,1,0,1,0,1

        if hasattr(self,"ZoomBrush")==0: self.ZoomBrush=Brush((240,240,240),"fill_100")
        if hasattr(self,"ZoomPen")==0:   self.ZoomPen=Pen((0,0,0),1,"solid")      

        if self.AddStatus:
          self.label = Label(self)        
          self.label.Show()
        else: self.label=None
        self.SetCursorType("None")
        self.SetPointer("cross")


    def SetZoomStyle(self,pen=Pen((0,0,0),1,"solid"),brush=Brush((240,240,240),"fill_100")):
        """
        Changes pen and brush of the zoom GraphViewSelect object.
        """
        self.ZoomBrush=brush
        self.ZoomPen=pen
        
        
    def CreateMenu(self):
        """
        Can be overwritten by derived classes to create a
        different popup menu
        """
        if self.MenuPopup is not None:self.AddMenuSeparator()
        if self.ZoomMode=="ON":
           self.ZoomMenuItem=self.AddMenuPopupItem("Zoom",self.ZoomTo)
           self.ResetZoomMenuItem=self.AddMenuPopupItem("Reset Zoom",self.ResetZoom)
           self.AddMenuSeparator()
        self.AddMenuPopupItem("Toggle Gridlines",self.Drawable.ToggleGridlines)
        if self.AddLog:
           self.logxitem=self.AddMenuPopupItem("Toggle Log X",self._ToggleLogX)
           self.logyitem=self.AddMenuPopupItem("Toggle Log Y",self._ToggleLogY)
        if self.AddCursorSelect:
          self.cursor_popup_menu=Menu(self.MenuPopup)          
          self.cursor_popup_menu.AddCommand("Vertical",Command.Command(self.SetCursorType, cursortype="Vertical"),"radiobutton")
          self.cursor_popup_menu.AddCommand("Crosshairs",Command.Command(self.SetCursorType, cursortype="Crosshairs"),"radiobutton")
          self.cursor_popup_menu.AddCommand("None",Command.Command(self.SetCursorType, cursortype="None"),"radiobutton")
          self.cursor_popup_menu.SetCheckedRadio("None")
          self.AddMenuPopupCascade("Cursor",self.cursor_popup_menu)
        if self.AddStyleSelect:
          self.style_popup_menu=Menu(self.MenuPopup)          
          self.style_popup_menu.AddCommand("Line",Command.Command(self.Drawable.SetStyle, style="Line"),"radiobutton")
          self.style_popup_menu.AddCommand("Points",Command.Command(self.Drawable.SetStyle, style="Points"),"radiobutton")
          self.style_popup_menu.AddCommand("PointsLine",Command.Command(self.Drawable.SetStyle, style="PointsLine"),"radiobutton")
          self.style_popup_menu.AddCommand("Bars",Command.Command(self.Drawable.SetStyle, style="Bars"),"radiobutton")
          self.style_popup_menu.SetCheckedRadio("Line")
          self.AddMenuPopupCascade("Style",self.style_popup_menu)


    def SetLabels(self,title_label=None,x_label=None,y_label=None,y2_label=None):
        """
        Sets drawing labels
        Parameters:
          title_label: Title of the drawing
          x_label: Label of x axis
          y_label: Label of y axis
          y2_label: Label of second (right-side) y axis
        """
        self.Drawable.SetLabels(title_label,x_label,y_label,y2_label)


    def _ToggleLogX(self):
        self.LogX = not self.LogX        
        if self.LogX: self.SetXScaleLog()
        else:         self.SetXScaleLinear()      
        self.Update()


    def _ToggleLogY(self):
        self.LogY = not self.LogY        
        if self.LogY: self.SetYScaleLog()
        else:         self.SetYScaleLinear()      
        self.Update()

    def SetXScaleLinear(self):
        """
        Sets linear scale for x axis (default)
        """
        self.Drawable.SetLogX(0)

    def SetXScaleLog(self,minval=1):
        """
        Sets log scale for x axis
        Parameters:
           minvalue: all values less than or equals to 0 are set to it
        """
        self.Drawable.SetLogX(1,minval)


    def SetYScaleLinear(self):
        """
        Sets linear scale for y axis (default)
        """
        self.Drawable.SetLogY(0)

    def SetYScaleLog(self,minval=0.000001):
        """
        Sets log scale for y axis
        Parameters:
           minvalue: all values less than or equals to 0 are set to it
        """
        self.Drawable.SetLogY(1,minval)
            

    def _EventZoomSelection(self,source):
        rect=source.GetSelection()["BoundingRect"]
        source.Destroy()
        self.ZoomSelect=None        
        self.Drawable.SetZoom(rect[0],rect[1])


    def ZoomTo(self):
        """
        Creates GraphViewSelect object to get a zoom by the user through mouse interface.
        """
        self.ZoomSelect=GraphViewSelect.GraphViewSelectRect(self.Source[0],self._EventZoomSelection)
        self.ZoomSelect.SetBrush(self.ZoomBrush)
        self.ZoomSelect.SetPen(self.ZoomPen)
        self.ZoomSelect.ConnectView(self)


    def ClearZoomSelect(self):
        """
        Called to destroy zoom ViewSelect object.
        Called in the zoom selection event, but applications can assure
        the destruction by calling this.
        """
        if (self.ZoomSelect is not None):
          self.ZoomSelect.Destroy()
          self.ZoomSelect=None


    def SetZoom(self,(x0,y0),(x1,y1)):
        """
        Sets zoom to given plotting coordinates.
        Parameters:
            (x0,y0): Upper-left coords
            (x1,y1): Lower-right coords
        """
        self.Drawable.SetZoom((x0,y0),(x1,y1))


    def ResetZoom(self):
        """
        Resets zoom.
        """        
        self.Drawable.ResetZoom()


    def LockPosition(self,value):
        """
        Controls repositioning of view after data change.
        Parameters:
        Value: If zero, view stays at the same position and zoom state in the
               case of a DataChange event. Otherwise, views returns tu initial
               position. By default FlagLockPosition is set to zero.
        """
        self.FlagLockPosition=value
        if value==1:
            self.XAxis=(self.minxval,self.maxxval)
            self.YAxis=(self.minyval,self.maxyval)
            self.Y2Axis=(self.miny2val,self.maxy2val)
        else:
            self.XAxis=self.BackXAxis
            self.YAxis=self.BackYAxis
            self.Y2Axis=self.BackY2Axis
        


    def GetPositionValues(self,position):
        """
        For a given plotting position=(x0,y0), returns a list of  tuples (name,(x,y))
        where:
            - name is the name of a source function
            - x is the source function x value closest x to x0
            - y is the value of function at x
        """        
        ret=[]
        coords=[position[0],position[1]]
        
        for function in self.Functions.values():
            if "data" in function.keys():
                try:
                    #pos=Numeric.searchsorted(function["xdata"],Numeric.array([coords[0],]))[0]
                    if "nopts" in function.keys():
                        pos=BinarySearch(function["xdata"],coords[0],function["nopts"])
                        if function["nopts"] and pos>function["nopts"]-1:pos=function["nopts"]-1
                    else:
                        pos=BinarySearch(function["xdata"],coords[0])
                    pos=max(pos-1,0)            
                    ind=function["xdata"][pos]
                    val=function["data"][pos]
                    
                    if (pos+1)<function["xdata"].shape[0]:
                        nextind=function["xdata"][pos+1]
                        if abs(nextind-coords[0])<abs(ind-coords[0]):
                            ind=nextind
                            val=function["data"][pos+1]
                    ret.append((function["name"],(ind,val)))
                except:
                    pass
                
        return ret

        
    def GetClosestFunction(self,coord):
        """
        Returns, if any, the closest source function to the coord=(x,y)
        """        
        min=0.1
        ret=None
        vals = self.GetPositionValues(coord)
        (xmin,xmax)=self.Drawable.GetVisibleXAxis()
        (y1min,y1max)=self.Drawable.GetVisibleYAxis()
        (y2min,y2max)=self.Drawable.GetVisibleY2Axis()
        for (name,pos) in vals:
            cursor=[coord[0],coord[1]]
            function=[pos[0],pos[1]]
            
            if self.Drawable.GetFunctionScale (name)==1 and y2min is not None and y2max is not None:
                ymin,ymax=y2min,y2max
                cursor[1]= ((y2max-y2min)*(coord[1]-y1min)/(y1max-y1min))+y2min
            else:
                ymin,ymax=y1min,y1max
            if function[0]<xmin or function[0]>xmax or function[1]<ymin or function[1]>ymax: continue
            posnorm=((float(function[0])-xmin)/(float(xmax)-xmin),(float(function[1])-ymin)/(float(ymax)-ymin))                
            coordnorm=((float(cursor[0])-xmin)/(float(xmax)-xmin),(float(cursor[1])-ymin)/(float(ymax)-ymin))    
            dist= pow(pow(coordnorm[0]-posnorm[0],2.)+pow(coordnorm[1]-posnorm[1],2.),.5)
            if dist< min:
                min=dist
                ret=name
        return ret
        

    
    def SetCursorType(self,cursortype):
        """
        Sets the cursor
        Parameters: 
            cursortype: "None" (default), "Crosshairs or "Vertical""
                        All types generate EventPosition events
        """
        if cursortype=="None":
            cursor=GraphViewSelect.GraphViewSelectMousePosition(None,self.EventPosition)
        elif cursortype=="Vertical":
            cursor=GraphViewSelect.GraphViewSelectVerticalCursor(None,self.EventPosition)
        elif cursortype=="Crosshairs":
            cursor=GraphViewSelect.GraphViewSelectCrosshairs(None,self.EventPosition)
        else:
            raise "GraphView: Bad cursor type"                   
        self.SetCursor(cursor)
        if self.AddStatus and self.AutoHideStatus:
            if cursortype=="None": self.label.Hide()
            else: self.label.Show()
        if hasattr(self,"cursor_popup_menu"):self.cursor_popup_menu.SetCheckedRadio(cursortype)
        

        

    def SetPolarEnv(self,value,rmax=None):
        """
        Sets/resets polar environment.
        Parameters:
           value: if non-zero sets polar environment on.
           rmax: if value non-zero, indicates the maximum radius for the plotting.
                 If None, set to the biggest value displayed
        """
        self.PolarEnv=value
        self.RMax=rmax
        self.Drawable.SetPolarEnv(value)
        if value:
            if hasattr(self,"logxitem"): self.DisableMenuItem(self.logxitem)
            if hasattr(self,"logyitem"): self.DisableMenuItem(self.logyitem)
        else:
            if hasattr(self,"logxitem"): self.EnableMenuItem(self.logxitem)
            if hasattr(self,"logyitem"): self.EnableMenuItem(self.logyitem)


    def EnableLegend(self,enable,position="bottom"):
        """
        Parameters:
            enable: if non-zero adds legend. If zero, removes.
            position: if enable is non-zero  sets the position of the legend: "top", "left",
                      "right" or "bottom"
        """
        self.Drawable.EnableLegend(enable,position)


    def GetPolarCoords(self,coords):
        """
        ViewSelect and mouse events are generated always in orthogonal coords,
        regardless polar environment is enabled or not. In this case this 
        method can be called to convert the coords.
        Parameters:
          coords: orthogonal coords
        Returns: polar coords
        """
        r=math.sqrt(math.pow(coords[0],2)+math.pow(coords[1],2))
        if coords[0]: t=math.atan2(coords[1],coords[0])
        else: return (0,0)
        return (r,t)
        

    def SetPen(self,name,pen):
        """
        Sets pen for a source plotting
        Parameters:
          name: name of the source function
          pen: Pen object
        """
        for sel in self.Source:
            if (hasattr(sel,"name") and sel.name==name) or name==str(sel):
                sel.pen=pen                
                self.Drawable.SetFunctionItem(name,"pen",pen)
                self.Update()
            

    def SetFunctionYScale(self,name,yscale):
        """
        Sets y axis the function is related to
        Parameters:
          name: name of the source function
          yscale: 0 or 1
        """
        for sel in self.Source:
            if (hasattr(sel,"name") and sel.name==name) or name==str(sel):
                sel.yscale=yscale                
                self.Drawable.SetFunctionItem(name,"yscale",yscale)
                self.Update()


    def SetSymbol(self,name,symbol):
        """
        Sets symbol for a source plotting
        Parameters:
          name: name of the source function
          pen: Pen object
        """
        for sel in self.Source:
            if (hasattr(sel,"name") and sel.name==name) or name==str(sel):
                sel.symbol=symbol
                self.Drawable.SetFunctionItem(name,"symbol",symbol)
                self.Update()
        

    def SetStyle(self,style):
        """
        Sets style for plotting
        Parameters:
          style: "Line", "Points", "PointsLine" or "Bars"
        """
        self.Drawable.SetStyle(style)
        if hasattr(self,"style_popup_menu"):self.style_popup_menu.SetCheckedRadio(style)

    def SetXAxis(self,x_axis=None):
        """
        Fixes the range of x axis.
        Parameters:
          x_Axis: tuple (min_value,maxvalue)
                  If x_Axis==None (default) range definition is automatic.
        """
        self.BackXAxis=x_axis
        self.XAxis=x_axis

    def SetYAxis(self,y_axis=None):
        """
        Fixes the range of y axis.
        Parameters:
          y_Axis: tuple (min_value,maxvalue)
                  If y_Axis==None (default) range definition is automatic.
        """
        self.BackYAxis=y_axis
        self.YAxis=y_axis

    def SetY2Axis(self,y2_axis=None):
        """
        Fixes the range of y2 axis.
        Parameters:
          y2_Axis: tuple (min_value,maxvalue)
                  If y2_Axis==None (default) range definition is automatic.
        """
        self.BackY2Axis=y2_axis
        self.Y2Axis=y2_axis


    def GetXAxis(self):
        """
        Returns X Axis = (min,max)
        """        
        return (self.minxval,self.maxxval)

    def GetYAxis(self):
        """
        Returns Y Axis = (min,max)
        """
        return (self.minyval,self.maxyval)

    def GetY2Axis(self):
        """
        Returns Y2 Axis = (min,max)
        """
        return (self.miny2val,self.maxy2val)
    
    def _DrawAxis(self,sels):
        if sels != {}:
            if self.PolarEnv:
                if self.RMax is not None:                    
                    minr,maxr=-self.RMax,self.RMax
                else:
                    minr,maxr=0.,0.
                    for sel in sels.values():
                        if "xdata" in sel.keys()and "data" in sel.keys():
                                if len(sel["xdata"])==len(sel["data"])>0:
                                    r=sel["data"]*sel["data"]+sel["xdata"]*sel["xdata"]                                    
                                    fmax=math.sqrt(max(r))
                                    if fmax>maxr:minr,maxr=-fmax,fmax
                self.minxval,self.maxxval,self.minyval,self.maxyval=minr,maxr,minr,maxr
                self.Drawable.SetEnv(minr,maxr,minr,maxr,reset_zoom=not self.FlagLockPosition)
                return
            
            if self.XAxis is None:
                minarr=[]
                maxarr=[]
                for sel in sels.values():
                    if "xdata" in sel.keys() and len(sel["xdata"]):
                        minarr.append(min(sel["xdata"])-0.000001)
                        maxarr.append(max(sel["xdata"])+0.000001)
                if len (minarr) and len(maxarr):
                    (self.minxval,self.maxxval)=(min(minarr),max(maxarr))
            else:
                (self.minxval,self.maxxval)=self.XAxis
                            
            if self.YAxis is None:
                minarr=[]
                maxarr=[]
                for sel in sels.values():
                    if "data" in sel.keys()  and len(sel["data"]):
                        if sel["yscale"] == 0:
                            minarr.append(min(sel["data"])-0.000001)
                            maxarr.append(max(sel["data"])+0.000001)
                
                if len (minarr) and len(maxarr):
                    (self.minyval,self.maxyval)=(min(minarr),max(maxarr))
            else:
                (self.minyval,self.maxyval)=self.YAxis

            drawY2=0
            for sel in sels.values():
                if "yscale" in sel.keys():
                    if sel["yscale"]==1:
                        drawY2=1
                        break
                
            if drawY2:
                if self.Y2Axis is None:
                    minarr=[]
                    maxarr=[]
                    for sel in sels.values():
                        if "data" in sel.keys()  and len(sel["data"]):
                            if sel["yscale"]==1:
                                minarr.append(min(sel["data"])-0.000001)
                                maxarr.append(max(sel["data"])+0.000001)
                    if len (minarr) and len(maxarr):
                        (self.miny2val,self.maxy2val)=(min(minarr),max(maxarr))
                else:
                    (self.miny2val,self.maxy2val)=self.Y2Axis                
                self.Drawable.SetEnv(self.minxval,self.maxxval,self.minyval,self.maxyval,self.miny2val,self.maxy2val,reset_zoom=not self.FlagLockPosition)            
            else:
                self.miny2val=self.maxy2val=None
                self.Drawable.SetEnv(self.minxval,self.maxxval,self.minyval,self.maxyval,reset_zoom=not self.FlagLockPosition)


#################################
#            EVENTS
#################################

    def EventPosition(self,source):
        """
        Overridable: Cursor callback event
                     GraphView threats it if "AddStaus" option set in __init__.
                     In this case, if this function is overriden ,derived
                     classes should call GraphView.EventPosition
        Parameters:
            source: GraphViewSelect object thar generated the event
        """
        if self.AddStatus:
            sel=source.GetSelection()
            coord=sel["Position"].DataCoord            
            type=source.GetType()
            if type=="Crosshairs" or type=="MousePosition":
              if self.PolarEnv:
                  self.label.SetText("r:%07.6f   t:%07.6f" % self.GetPolarCoords(coord))
              else:
                  self.label.SetText("x:%07.6f   y:%07.6f" % coord)
            elif type=="VerticalCursor":
              statusString=""
              for (name,(x,y)) in self.GetPositionValues(coord):
                  if self.PolarEnv: 
                      statusString=statusString + "r:%07.6f t:%07.6f    " % self.GetPolarCoords((x,y))
                  else:
                      statusString=statusString + "x:%07.6f y:%07.6f    " % (x,y)
              self.label.SetText(statusString)
            else:
              self.label.SetText("")
              

    def EventDoubleClick(self,pos):
        """
        Overridable: Double click callback event
        Parameters:
            pos: position of the event
        """
        
        pass 


#################################
#            VIRTUALS
#################################

    def CreateDrawable(self):
        """
        Virtual: Implements creation of  a Drawable object (which implements the
        Drawable interface)
        """        
        self.Drawable= Drawable_1D(self,self.ZoomMode,self.ScrollMode)
        self.Drawable.Show()


    def GetPosition(self, event):
        """
        Virtual: See View.GetPosition
        """
        ret=Position()
        if event==(-1,-1):  return None
        ret.ViewCoord=event
        (ret.Inside,ret.DataCoord)=self.Drawable.ViewCoord2DataCoord(ret.ViewCoord)
        return ret        

            
    def DataChanged(self,source=None):
        """
        Virtual: See View.DataChanged
        """
        if self.Source != ():
            if source is None:
                self.Drawable.RemoveFunction()
                self.Functions={}
                for i in self.Source:
                    sel=i.GetOutput()
                    if "name" not in sel.keys() or sel["name"]==None: sel["name"] = str(i)
                    if "nopts" in sel.keys():
                        if sel["nopts"]:
                            sel["xdata"] = sel["xdata"][0:sel["nopts"]]
                            sel["data"] = sel["data"][0:sel["nopts"]]
                        else:
                            del sel["data"]
                            del sel["xdata"] 
                    self.Functions[sel["name"]]=sel
            else:
                sel=source.GetOutput()
                if "name" not in sel.keys() or sel["name"]==None: sel["name"] = str(source)
                if sel["name"]  not in self.Functions.keys(): return
                if "nopts" in sel.keys(): 
                    if sel["nopts"]:
                        sel["xdata"] = sel["xdata"][0:sel["nopts"]]
                        sel["data"] = sel["data"][0:sel["nopts"]]
                    else:
                        del sel["data"]
                        del sel["xdata"] 
                self.Functions[sel["name"]]=sel
            self._DrawAxis(self.Functions)
            for i in self.Functions.keys():
                if "data" in self.Functions[i].keys():
                    self.Drawable.SetFunction(i,self.Functions[i])
                
        else:
            self.Drawable.RemoveFunction()
            self.Functions={}
            self.Drawable.RemoveGraphicObjects()
        self.Update()
        ### This event should be generated inside ViewSelect's Update???
        if self.Cursor is not None and "Position" in self.Cursor.Selection.keys(): self.EventPosition(self.Cursor)


    def Redraw(self):   
        """
        Virtual: See View.Redraw
        """
        self.Drawable.Redraw()
        

    def DataCoord2ViewCoord(self,coord):
        """
        Virtual: See View.DataCoord2ImageCoord
        """
        return self.Drawable.DataCoord2ViewCoord(coord)


    def ViewCoord2DataCoord(self,coord):
        """
        Virtual: See View.ViewCoord2DataCoord
        """
        return self.Drawable.ViewCoord2DataCoord(coord)
        
    def GetSaveFormats(self):
        """
        Returns tuple with supported save formats
        """
        return ("PNG","BMP","PS","JPG")






