###########################################################
#
# CVS Id:       $Id: util.py,v 1.2 2002/06/20 13:29:49 svensson Exp $
#
# File:         util.py
#
# Package:      scisoft/python/scidxp
#
# Author:       Olof Svensson (May 2002)
#
# Description:  This module contains various utility
#               classes for the scidxp package.
#
# Change log, most recent first:
#
# Update 19/06/2002 O Svensson (svensson@esrf.fr)
#                   Committed file to CVS repository
#
#

from scidxp.node import *
import Numeric, copy, string


class DefaultArray(Array):

  def __init__(self, data, unit="", dataType=None, extends=None):
    self._data = Numeric.array((data))
    if extends is None:
      self._extnds = Numeric.array((self._data.shape))
      self._extnds = self._extnds[::-1].tolist()
    else:
      self._extnds = extends
    if len(self._extnds) > 1:
      for extend in self._extnds:
        if extend == 1:
          del self._extnds[extend]
    self._rank = len(self._extnds)
    self._dataCount = 1
    for i in self._extnds:
      self._dataCount = self._dataCount*i
    self._dataType = dataType
    if dataType is None:
      self._dataType = self.numpyToDataType(self._data.typecode())
      if self._dataType == 1:
        self._data = Numeric.array(map(ord, data))
    self._unit = unit
    self._arrayChangeSupport = ArrayChangeSupport(self)

  def __repr__(self):
    if self._dataType == 1:
      repr = string.join(map(chr,self._data.tolist()),"")
    else:
      if self._dataCount == 1:
        repr = str(self._data.tolist()[0])
      else:
        repr = str(self._data.tolist())
    return repr

  def getRank(self):
    return self._rank

  def getExtends(self):
    return self._extnds

  def getExtend(self, dim):
    return self._extnds[dim]

  def getDataCount(self):
    return self._dataCount

  def getDataType(self):
    return self._dataType

  def getUnit(self):
    return self._unit

  def getData(self, new_type = 0):
    if (new_type == 0) or (new_type == self._dataType):
      #if (self._dataType == 1):
      #  data = string.join(map(chr,self._data.tolist()),"")
      #else:
      data = self._data
    else:
      data = self._data.astype(self.dataTypeToNumpy(new_type))
      #if (self.new_type == 1):
      #  data = string.join(map(self._data.tolist()),"")
    return data

  def getDataSlab(self, first, last, new_type = 0):
    if (len(first) != self._rank) or (len(last) != self._rank):
      raise "Index errorfor getDataSlab"
    slabslice = []
    slice_indeces = Numeric.array(range(self._rank))[::-1].tolist()
    for i in slice_indeces:
      slabslice.append(slice(first[i],last[i]))
    if (new_type == 0) or (new_type == self._dataType):
      data = self._data[slabslice]
    else:
      data = self._data[slabslice].astype(self.dataTypeToNumpy(new_type))
    return data

  def dataTypeToNumpy(self, dataType):
    if (dataType) == Array.CHAR:
      numpyType = "c"
    elif (dataType) == Array.INT8:
      numpyType = "1"
    elif (dataType) == Array.INT16:
      numpyType = "s"
    elif (dataType) == Array.INT32:
      numpyType = "i"
    elif (dataType) == Array.INT64:
      numpyType = "l"
    elif (dataType) == Array.FLOAT32:
      numpyType = "f"
    elif (dataType) == Array.FLOAT64:
      numpyType = "d"
    else:
      raise "DefaultDataType.dataTypeToNumpy: Not supported data type: "+dataType
    return numpyType

  def numpyToDataType(self, numpyType):
    if (numpyType) == "c":
      dataType = Array.CHAR
    elif (numpyType) == "l":
      dataType = Array.INT8
    elif (numpyType) == "s":
      dataType = Array.INT16
    elif (numpyType) =="i":
      dataType = Array.INT32
    elif (numpyType) =="O":
      dataType = Array.INT32
    elif (numpyType) =="l":
      dataType = Array.INT64
    elif (numpyType) == "f":
      dataType = Array.FLOAT32
    elif (numpyType) == "d":
      dataType = Array.FLOAT64
    else:
      #print "DEBUG: ", self._data, self._data.shape
      raise "DefaultDataType.numpyToDataType: Not supported data type: "+numpyType
    return dataType

  def addArrayEventListener(self, arrayEventListener):
    self._arraySupport.addArrayEventListener(arrayEventListener)

  def removeArrayEventListener(self, arrayEventListener):
    self._arraySupport.removeArrayEventListener(arrayEventListener)

  def fireArrayEvent(self, arrayEvent):
    self._arraySupport.fireArrayEvent(arrayEvent)



class DefaultAttribute(DefaultArray):

  def __init__(self, name, data, unit=""):
    DefaultArray.__init__(self, data, unit)
    self._name = name

  def getName(self):
    return self._name

  #def __repr__(self):
  #  return self._name


class DefaultScale(DefaultArray):

  def __init__(self, name, data, unit=""):
    DefaultArray.__init__(self, data, unit)
    self._name = name

  def getName(self):
    return self._name


class IndexScale(DefaultScale):

  def __init__(self, name, length, unit=""):
    DefaultScale._init_(self, name, Numeric.arrayrange(length), unit)


class LinearScale(DefaultScale):
  
  def __init__(self, name, a, b, length, unit=""):
    DefaultScale._init_(self, name, Numeric.arrayrange(a, b+(b*1.0-a)/(length-1), (b*1.0-a)/(length-1)), unit)


class NodeChangeException:
  pass

class DefaultNode:

  def __init__(self, name):
    self._mutable             = 1
    self._disposed            = 0
    self._name                = name
    self._parent              = None
    self._attributes          = {}
    self._nodeChangeSupport   = NodeChangeSupport(self)

  def getName(self):
    self.checkDisposed()
    return self._name

  def setName(self, name):
    self.checkDisposed()
    self.checkMutable()
    self._name = name

  def getParentNode(self):
    self.checkDisposed()
    return self._parent

  def setParentNode(self, parent):
    self.checkDisposed()
    self.checkMutable()
    self._parent = parent

  def getAttributes(self):
    #print "DEBUG: in scidxp_util getAttributes", [self]
    self.checkDisposed()
    #list = {}
    #for attribute in self._attributes.values():
    #  list[attribute.getName()]=attribute
    return self._attributes

  def setAttribute(self, name, array):
    #print "DEBUG: in scidxp_util setAttribute", [self], name, array
    self.checkDisposed()
    self.checkMutable()
    if name is None:
      raise "Illegal argument excpetion in DefaultNode.setAttribute: name is None"
    self._attributes[name]=array

  def getAttribute(self, name):
    self.checkDisposed()
    return self._attributes[name]

  def addNodeEventListener(self, nodeListener):
    self.checkDisposed()
    self._nodeChangeSupport.addNodeEventListener(nodeListener)

  def removeNodeEventListener(self, nodeListener):
    self.checkDisposed()
    self._nodeChangeSupport.removeNodeEventListener(nodeListener)

  def fireNodeEvent(self, nodeEvent):
    self.checkDisposed()
    self._nodeChangeSupport.fireNodeEvent(nodeEvent)

  def isGroupNode(self):
    self.checkDisposed()
    return 0

  def isDataSetNode(self):
    self.checkDisposed()
    return 0

  def isMutable(self):
    self.checkDisposed()
    return self._mutable

  def dispose(self):
    self.checkDisposed()
    self.checkMutable()
    if self._parent is None:
      self._parent.removeChild(self)
    self._parent = None
    self._name = None
    self._attibutes = {}
    self._disposed = 1
    self._nodeChangeSupport.nodeDisposed()
    self._nodeChangeSupprt = None

  def checkMutable(self):
    if not self.isMutable():
      raise NodeChangeException

  def checkDisposed(self):
    if self._disposed:
      raise "Node disposed"

  def toString(self):
    return self._name

  #def __repr__(self):
  #  return self._name


class DefaultGroupNode(DefaultNode):

  def __init__(self, name):
    DefaultNode.__init__(self, name)
    self._nodes = {}

  def insertChild(self, node):
    #print "DEBUG: scidxp_util insertChild: ", [node]
    self.checkDisposed()
    self.checkMutable()
    if node is None:
      raise "Node cannot be None for insertChild in DefaultGroupNode"
    if node.getParentNode() is not None:
      node.getParentNode().removeChild(node)
      nodeEvent = NodeEvent(self, NodeEvent.STRUCTURE_CHANGED, None)
      node.getParentNode().fireNodeEvent(nodeEvent)
    node.setParentNode(self)
    self._nodes[node.getName()] = node
    nodeEvent = NodeEvent(self, NodeEvent.STRUCTURE_CHANGED, None)
    self.fireNodeEvent(nodeEvent)

  def removeChild(self, node):
    self.checkDisposed()
    self.checkMutable()
    if node is None:
      raise "Node cannot be None for removeChild in DefaultGroupNode"
    if self._nodes == {}:
      return
    del self._nodes[node.getName()]
    #nodeEvent = NodeEvent(self, NodeEvent.STRUCTURE_CHANGED, None)
    #self.fireNodeEvent(nodeEvent)

  def getChild(self, name):
    self.checkDisposed()
    if name in self._nodes.keys():
      return self._nodes[name]
    else:
      return None

  def getChildren(self):
    return self._nodes

  def isGroupNode(self):
    self.checkDisposed()
    return 1


class DefaultDataSetNode(DefaultNode):

  def __init__(self, name, array, scales=None):
    DefaultNode.__init__(self, name)
    self._array = array
    self._scales = scales

  def setData(self, array, scales=None):
    self.checkDisposed()
    self.checkMutable()
    if array is None:    
      raise "Array cannot be None for setData in DefaultDataSetNode"
    self._array = array
    self._scales = scales
    #self._nodeChangeSupport.dataChanged()

  def getData(self):
    self.checkDisposed()
    return self._array

  def isDataSetNode(self):
    self.checkDisposed()
    return 1



    


