"""
    ScanGraphView.py
    GraphView derived class for scan plotting
"""    

from PyDVT import __version__,__date__,__author__


from GraphView import *
from BinarySearch import BinarySearch



class ScanGraphFilter(GraphFilter):
    """
    Filter disigned to interface SPS scan data (SPSData) to ScanGraphView.
    For a shared array arr in Spec, it should exist an shared array arr_ENV 
    with the scan information fields, at least:
      "nopts": number of valid points of the scan      
    The fields
    of arr_ENV are read by SPSData in "envdict" entry of page info.
    
    """

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            See GraphFilter.GetOutput
            In addiction:
                "nopts": integer, number of valid points of the scan
                "envdict": dictionary with all the scan environment parameters (SPS)
        """
        
        sel=GraphFilter.GetOutput(self)
        if "data" not in sel.keys(): return sel

        data=self.GetData()
        datasel=self.GetDataSelection()
        page=datasel.GetInfo()["index_list"][0]
        page_info=data.GetPageInfo(page)
        
        if "envdict" in page_info.keys():
            sel["envdict"]=page_info["envdict"]
            if "nopts" in page_info["envdict"].keys(): sel["nopts"]=int(page_info["envdict"]["nopts"])
        return sel


class ScanGraphView(GraphView):
    def __init__(self,*args,**keys):
        """
        Constructor
        See GraphView.__init__
        """
        GraphView.__init__(self,*args,**keys)
        self.position=-1
        self.AutoRescale=0

    def SetAutoRescale(self,value):
        """
        If value non-zero, axis are automatically rescaled during scan if ploting gets out of the visible axis
        """
        self.AutoRescale=value
    
    def DataChanged(self,source=None):
        """
        Constructor
        See GraphView.DataChanged
        """
        if source==None:            
            GraphView.DataChanged(self,source)
            self.position=-1                
        else:
            sel=source.GetOutput()
            
            #Process scan _ENV options here
            if "envdict" in sel.keys():
                envdict=sel["envdict"]
                update=0
                if "xmin" and "xmax" in envdict.keys():
                    newaxis=(float(envdict["xmin"]),float(envdict["xmax"]))
                    if newaxis != self.XAxis:
                        self.SetXAxis(newaxis)
                        update=1
                if update:
                    GraphView.DataChanged(self,source)
                    return
                
            if "name" not in sel.keys() or sel["name"]==None: sel["name"] = str(source)
            self.Functions[sel["name"]]=sel            

            if "data" not in sel.keys():
                self.Drawable.RemoveFunction(sel["name"])
                self.Redraw()
            else:
                if "nopts" in sel.keys(): pos=sel["nopts"]
                else:
                    self.position=-1
                    pos=len(sel["data"])
                if self.position==-1: self.position=pos
                self.Drawable.SetFunction(sel["name"],sel)
                if (pos <= self.position):
                    restart=1
                    self.position=0
                else: restart=0

                if self.AutoRescale:
                    rescale=0
                    try:
                        if "yscale" in sel.keys() and sel["yscale"]==1: yaxis=self.GetY2Axis()
                        else:yaxis=self.GetYAxis()
                        newymin,newymax=yaxis[0],yaxis[1]
                        yarr=Numeric.take(sel["data"],range(self.position,pos))
                        ymin,ymax=min(min(yarr),yaxis[0]),max(max(yarr),yaxis[1])
                        ysize=ymax-ymin
                        if ymin<yaxis[0]: newymin= ymin-ysize/2
                        if ymax > yaxis[1]: newymax= ymax+ysize/2
                        if newymin!=yaxis[0] or newymax!=yaxis[1]:
                            if "yscale" in sel.keys() and sel["yscale"]==1: self.SetY2Axis(newaxis)
                            else: self.SetYAxis((newymin,newymax))
                            rescale=1
                    except: pass                       
                    try:
                        xaxis=self.GetXAxis()
                        newxmin,newxmax=xaxis[0],xaxis[1]
                        xarr=Numeric.take(sel["xdata"],range(self.position,pos))
                        xmin,xmax=min(min(xarr),xaxis[0]),max(max(xarr),xaxis[1])
                        xsize=xmax-xmin
                        if xmin<xaxis[0]: newxmin= xmin-xsize/2
                        if xmax > xaxis[1]: newxmax= xmax+xsize/2
                        if newxmin!=xaxis[0] or newxmax!=xaxis[1]:
                            self.SetXAxis((newxmin,newxmax))
                            rescale=1
                    except: pass                       
                    if rescale:
                        GraphView.DataChanged(self,source)
                        return
                self.Drawable.IncrementalUpdate(sel["name"],(self.position,pos),restart)
                self.position=pos
                return
                
            #I'm calling Redraw instead of Update in order not to generate the
            #ImageChange event: this makes the Update of zoom select to be called,
            #and in it the temporary draw is erased.
            self.Redraw()

    