"""
    Drawable2D.py
    2D Image displaying widget
        
"""

from PyDVT import __version__,__date__,__author__

import qt
import qtcanvas

from Binding import Pen,Brush




imageRTTI = 984376
class ImageItem (qtcanvas.QCanvasRectangle):

    def __init__(self,view):
        qtcanvas.QCanvasRectangle.__init__(self,view.Canvas)
        self.View=view
        

    def setImage(self,pixmap):
        self.pixmap=pixmap
        if (self.width()!=pixmap.width())or(self.height()!=pixmap.height()):
            self.setSize( pixmap.width(), pixmap.height() )
        
    def rtti (self):
        return imageRTTI
    
    
    def drawShape (self,p):
        if  self.View.RedrawingObjectFlag:            
            p.drawPixmap( self.View.orx, self.View.ory, self.pixmap, self.View.orx , self.View.ory , self.View.orw , self.View.orh )        
        elif self.View.DrawingFlag:
            p.drawPixmap( self.View.rx, self.View.ry, self.pixmap, self.View.rx , self.View.ry , self.View.rw , self.View.rh )
        else:
            p.scale(self.View.ZoomFactor,self.View.ZoomFactor)        
            p.drawPixmap( int(self.x()), int(self.y()), self.pixmap )
            

    
class GraphicalItem:

    def __init__(self,view,pen=Pen((0,0,0),1,"solid"),brush=None):
        self.View=view
        self.Visible=0
        self.bounding_rect=[0,0,0,0]
        self.Z=1
        self.RasterOp=None  #not used so far
        self.Pen = qt.QPen(pen.GetColor(), pen.GetWidth() ,pen.GetStyle())
        if brush==None: self.Brush = qt.QBrush(qt.QBrush.NoBrush)
        else: self.Brush = qt.QBrush(brush.GetColor(), brush.GetStyle() )         
        

    def hide(self):
        self.Visible=0
        self.View.RedrawCanvasRegion(self.bounding_rect[0],self.bounding_rect[1],self.bounding_rect[2],self.bounding_rect[3])
            
    def show(self):
        self.Visible=1
        self.View.RedrawCanvasRegion(self.bounding_rect[0],self.bounding_rect[1],self.bounding_rect[2],self.bounding_rect[3])            

    #VIRTUALS
    def draw(self,p):
        pass

    def SetCoords(self,*args):
        pass


class RectangleItem(GraphicalItem):
    def __init__(self,view,pen,brush):
        self.X0=self.Y0=self.W=self.H=0
        GraphicalItem.__init__(self,view,pen,brush)
        
    def draw(self,p):
        if self.Visible:
            p.setPen(self.Pen)
            p.setBrush(self.Brush)
            p.setRasterOp(qt.Qt.NotEraseROP)
            if self.RasterOp != None: p.setRasterOp(self.RasterOp)
            #TODO: WHY IN HELL I HAVE TO DO THIS???
            if qt.qVersion()>="3.0.0":
                p.drawRect( self.X0,self.Y0,self.W+1,self.H+1 )
            else:
                p.drawRect( self.X0,self.Y0,self.W,self.H)
                            
    def SetCoords(self,X0,Y0,X1,Y1):
        if (self.X0,self.Y0,self.W,self.H)==(X0,Y0,X1-X0,Y1-Y0): return
        self.hide()
        self.bounding_rect[0]=min(X0,X1)
        self.bounding_rect[1]=min(Y0,Y1)
        self.bounding_rect[2]=max(X0,X1)
        self.bounding_rect[3]=max(Y0,Y1)
        (self.X0,self.Y0,self.W,self.H)=(X0,Y0,X1-X0,Y1-Y0)
        self.show()



class LineItem(GraphicalItem):
    def __init__(self,view,pen):
        self.X0=self.Y0=self.X1=self.Y1=0
        GraphicalItem.__init__(self,view,pen)
        
    def draw(self,p):
        if self.Visible:
            p.setPen(self.Pen)
            p.setBrush(self.Brush)
            if self.RasterOp != None: p.setRasterOp(self.RasterOp)
            p.drawLine( self.X0,self.Y0,self.X1,self.Y1 )
                
    def SetCoords(self,X0,Y0,X1,Y1):
        if (self.X0,self.Y0,self.X1,self.Y1)==(X0,Y0,X1,Y1): return
        self.hide()
        self.bounding_rect[0]=min(X0,X1)
        self.bounding_rect[1]=min(Y0,Y1)
        self.bounding_rect[2]=max(X0,X1)
        self.bounding_rect[3]=max(Y0,Y1)
        (self.X0,self.Y0,self.X1,self.Y1)=(X0,Y0,X1,Y1)
        self.show()



class Drawable_2D(qtcanvas.QCanvasView):
   def __init__(self,parent,zoom_mode="ON",scroll_mode="ON"):
       self.Canvas=qtcanvas.QCanvas(parent)
       self.ZoomMode=zoom_mode
       self.ScrollMode=scroll_mode
       self.Canvas.setDoubleBuffering(1) 
       qtcanvas.QCanvasView.__init__(self,self.Canvas,parent)       
       self.viewport().setMouseTracking(1)
       self.Canvas.setBackgroundColor(qt.QColor("gainsboro"))
       self.ZoomFactor=1.0
       self.xpos=self.ypos=0
       self.ImageItem=None
       self.parent=parent
       self.DrawingFlag=0
       self.RedrawingObjectFlag=0
       self.ContentsWidth=0
       self.ContentsHeight=0
       self.ContentsWidthAnt,self.ContentsHeightAnt=self.ContentsWidth,self.ContentsHeight
       self.GraphicObjects=[]
       
       if self.ScrollMode=="ON":
           self.setVScrollBarMode(qt.QScrollView.AlwaysOn)
           self.setHScrollBarMode(qt.QScrollView.AlwaysOn)
       elif self.ScrollMode=="OFF":
           self.setVScrollBarMode(qt.QScrollView.AlwaysOff)
           self.setHScrollBarMode(qt.QScrollView.AlwaysOff)
       else:
           self.setVScrollBarMode(qt.QScrollView.Auto)
           self.setHScrollBarMode(qt.QScrollView.Auto)
       self.setFocusPolicy(qt.QWidget.StrongFocus)
       self.setFocus()


   def drawContents(self, painter, cx=None, cy=None, cw=None, ch=None):      
      if cx == None:   # none is for wrong overridden function
         return qt.QFrame.drawContents(self, painter)
      else:
        self.DrawingFlag=1
        painter.scale(self.ZoomFactor,self.ZoomFactor)
        if (cx >= self.ContentsWidth) or (cy >= self.ContentsHeight): return        
        (self.rx,self.ry,self.rw,self.rh)=(float(cx)/self.ZoomFactor, float(cy)/self.ZoomFactor, float(cw+self.ZoomFactor)/self.ZoomFactor +1, float(ch+self.ZoomFactor)/self.ZoomFactor +1)
        qtcanvas.QCanvasView.drawContents(self, painter, self.rx,self.ry,self.rw,self.rh)
        for obj in self.GraphicObjects:
                obj.draw(painter)
        self.DrawingFlag=0

   def _RectSuperpose(self,rect1,rect2):
       xcoin=(((rect1[0] <= rect2[0])and(rect1[2] >= rect2[0])) or((rect2[2] >= rect1[0])and(rect2[2] <= rect1[2])) )
       ycoin=(((rect1[1] <= rect2[1])and(rect1[3] >= rect2[1])) or((rect2[3] >= rect1[1])and(rect2[3] <= rect1[3])) )
       return xcoin and ycoin
       
   def setScrollTracking(Tracking):
       self.horizontalScrollBar().setTracking(Tracking)
       self.verticalScrollBar().setTracking(Tracking)


   def ViewCoord2ImageCoord(self,view_coord):
        if (self.ImageItem==None):
            return (-1,-1)
        else:
            xval=int(float(view_coord[0])/self.ZoomFactor)
            yval=int(float(view_coord[1])/self.ZoomFactor)
            xval = min (xval,int(self.Canvas.width())-1)
            xval = max (xval,0)
            yval = min (yval,int(self.Canvas.height())-1)
            yval = max (yval,0)
            return (xval,yval)

        
   def ImageCoord2ViewCoord(self,image_coord):
        if (self.ImageItem==None) or (image_coord[0]>=self.Canvas.width()) or (image_coord[1]>=self.Canvas.height()):
            return (-1,-1)
        else:
            return (int(float(image_coord[0])*self.ZoomFactor),int(float(image_coord[1])*self.ZoomFactor))

   def resizeEvent(self,e):
        qtcanvas.QCanvasView.resizeEvent(self,e)
        if  self.ZoomMode=="FIT_TO_SCREEN":
            if (self.Canvas.width()==0) or (self.Canvas.height()==0): return
            self.ZoomFactor=min(float(self.viewport().width())/float(self.Canvas.width()),(float(self.viewport().height())/float(self.Canvas.height())))
            self._DrawContents()
        

   def contentsMousePressEvent(self, e):
    if e.button() == qt.Qt.RightButton:
        if self.parent is not None: self.parent._RightButtonPress(e.globalPos ())

    elif e.button() == qt.Qt.LeftButton:
        if self.parent is not None: self.parent._ButtonPress(self.ViewCoord2ImageCoord((e.x(),e.y())))
               
   def contentsMouseDoubleClickEvent(self,e):       
        if self.parent is not None: self.parent._DoubleClick(self.ViewCoord2ImageCoord((e.x(),e.y())))


   def contentsMouseReleaseEvent(self,e):
       if e.button() == qt.Qt.LeftButton:
           if self.parent is not None: self.parent._ButtonRelease(self.ViewCoord2ImageCoord((e.x(),e.y())))
           

   def contentsMouseMoveEvent(self,e):       
       if e.state()==0:
           if self.parent is not None: self.parent._Motion(self.ViewCoord2ImageCoord((e.x(),e.y())))
       elif e.state()==1:
           if self.parent is not None: self.parent._ButtonPressMotion(self.ViewCoord2ImageCoord((e.x(),e.y())))

   def keyPressEvent(self,e):
       ret=str(e.text())
       if ret=="":
            ret= e.key()
       if self.parent is not None: self.parent._KeyPress(ret)
       e.ignore()

   def Save(self,filename,format="BMP"):           
       if format=="PS" or format=="JPG":
            import Image,os
            tmpfile="__tmpfile.bmp"
            self.ImageItem.pixmap.save(tmpfile,"BMP")
            pilim=Image.open(tmpfile)
            if format=="PS":
                #pilim.save(filename,"EPS")
                #TODO: CORRECT THIS
                import Image,MyPSDraw
                WIDTH,HEIGHT=595,842
                xmin,ymin=30,140
                xmax,ymax=WIDTH-xmin,HEIGHT-ymin                
                
                f=open(filename,'w+')
                fontSize=14
                ps=MyPSDraw.MyPSDraw(f)
                ps.begin_document('test ps')
                box=(xmin,ymin,xmax,ymax)
                #title="Test postscript printing with PSDraw"
                #ps.setfont("Helvetica-Bold",fontSize)
                #ps.text((600/2-(0.5*fontSize*len(title))/2,(ymax*72)+10),title) # with stuff to center it
                ps.image(box,pilim,{'fit':0,'keepAspect':1})
                #ps.setink(5)
                ps.rectangle(box)
                #comments="isn't it beautiful to see these comments printed ???"
                #ps.setfont("Helvetica",fontSize) # need to do it each time text is added
                #ps.text((600/2-(0.44*fontSize*len(comments))/2,(ymin*72)-10),comments) # with stuff to center it
                ps.end_document()
                f.close()
                
            else:                
                pilim.save(filename,"JPEG")
            os.remove(tmpfile)
            return 1
       return self.ImageItem.pixmap.save(filename,format)
       


   def DrawImage(self,image_string,image_size,depth,colormap_table,reset_zoom=0):
        if reset_zoom:
            self.ZoomFactor=1.0
            self.xpos=self.ypos=0
        else:
            (xcanvaspos,ycanvaspos)=self.GetWindowPosition()
            (self.xpos,self.ypos)=(int(self.ZoomFactor*xcanvaspos),int(self.ZoomFactor*ycanvaspos))
        
        pixmap=qt.QPixmap()
        if depth==8:
            Img = qt.QImage( image_string,image_size[0],image_size[1],8,colormap_table,256,qt.QImage.IgnoreEndian)
            #if hasattr(self,"alloc_context"): qt.QColor.destroyAllocContext( self.alloc_context )        
            #self.alloc_context = qt.QColor.enterAllocContext()            
            pixmap.convertFromImage(Img)         
            #print pixmap.convertToImage().numColors()
            #qt.QColor.leaveAllocContext()
        else:
            Img = qt.QImage( image_string,image_size[0],image_size[1],32,None,0,qt.QImage.IgnoreEndian)
            pixmap.convertFromImage(Img)
                    
        if self.ImageItem is None:
            self.ImageItem=ImageItem(self)
            self.ImageItem.show()
        self.ImageItem.setImage(pixmap)
        
        if (self.Canvas.width()!=Img.width())or(self.Canvas.height()!=Img.height()):
            self.Canvas.resize(Img.width(),Img.height())
        self._DrawContents()
        

   def _DrawContents(self):
        if self.ZoomMode=="FIT_TO_SCREEN":
            self.ZoomFactor=min(float(self.viewport().width())/float(self.Canvas.width()),(float(self.viewport().height())/float(self.Canvas.height())))

        self.viewport().setUpdatesEnabled(0)
        self.ContentsWidth=int(float(self.Canvas.width())*self.ZoomFactor)
        self.ContentsHeight=int(float(self.Canvas.height())*self.ZoomFactor )
        if (self.contentsWidth()!= self.ContentsWidth)or (self.contentsHeight()!= self.ContentsHeight):
            self.resizeContents ( self.ContentsWidth,self.ContentsHeight )
        self.setContentsPos(self.xpos,self.ypos)
        self.viewport().setUpdatesEnabled(1)
        if  ((self.ContentsWidth<self.viewport().width()) or (self.ContentsHeight<self.viewport().height()))and((self.ContentsHeightAnt>self.ContentsHeight) or (self.ContentsWidthAnt>self.ContentsWidth) or (self.ZoomMode=="FIT_TO_SCREEN")):
            self.viewport().update()
        else:
            self.viewport().repaint(0,0,self.ContentsWidth,self.ContentsHeight,0)
        (self.ContentsWidthAnt,self.ContentsHeightAnt)=(self.ContentsWidth,self.ContentsHeight)
        
        
   def EraseImage(self):
        if self.ImageItem!=None:
            self.ImageItem=None
            self.Canvas.resize(0,0)
            self.resizeContents ( 0,0)
            self.ContentsWidth=0
            self.ContentsHeight=0
            self.viewport().update()

   def SetZoom(self, val=None, rect=None, com=None):
        """
        Sets zoom state 
        Parameters: (only one must be set)
        val:  a float, sets directly a value to the zoom factor
        rect: a tuple in the format (x0,y0,x1,y1), that defines the coordinates of
              the upper-left and the lower-right points of a zoomed area, in
              data selection coordinates.
        com:  can be "fit", "normal", "in" and "out"
        """        
        if val != None:
          (xcanvaspos,ycanvaspos)=self.GetWindowPosition()
          self.ZoomFactor=val
          (self.xpos,self.ypos)=(int(self.ZoomFactor*xcanvaspos),int(self.ZoomFactor*ycanvaspos))
        elif rect != None:
          DataWidth = max(rect[2] - rect[0] , 1)
          DataHeight = max(rect[3] - rect[1] , 1)
          if DataWidth==0: return
          if DataHeight==0: return
          self.ZoomFactor=min((float(self.width())/float(DataWidth)),float(self.height())/float(DataHeight))
          self.xpos=rect[0]*self.ZoomFactor
          self.ypos=rect[1]*self.ZoomFactor
          
        elif com != None:
          if com == "fit":                        
            if (self.Canvas.width()==0) or (self.Canvas.height()==0): return
            self.ZoomFactor=min(float(self.viewport().width())/float(self.Canvas.width()),(float(self.viewport().height())/float(self.Canvas.height())))
            self.xpos=self.ypos=0
          elif com == "normal":            
            self.ZoomFactor=1.0
            self.xpos=self.ypos=0
          elif com == "in":
            self.SetZoom(val=self.ZoomFactor*2.0)
            return
          elif com == "out":
            self.SetZoom(val=self.ZoomFactor*0.5)
            return
        if self.ZoomFactor < 0.001:
          self.ZoomFactor = 0.001
          
        self._DrawContents()


   def MoveTo(self,position):
        self.horizontalScrollBar().setValue(int(position[0]))
        self.horizontalScrollBar().setValue(int(position[1]))
        


   def PutLine(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid")):
        obj=LineItem(self,pen)
        self.GraphicObjects.append(obj)
        self.SetObjectCoords(obj,X0,Y0,X1,Y1)        
        return obj
    
   def  PutRectangle(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0")):
        obj=RectangleItem(self,pen,brush)
        self.GraphicObjects.append(obj)
        self.SetObjectCoords(obj,X0,Y0,X1,Y1)
        return obj

   def SetObjectCoords(self,obj,*args):
       obj.SetCoords(*args)

    
   def EraseObject(self,obj):
        obj.hide ()
        self.GraphicObjects.remove(obj)

   def RedrawCanvasRegion (self,X,Y,X2,Y2):
       #TODO: I have to understand what I'm doing
       (PosX,PosY)=self.GetWindowPosition()
       X2=min (X2,PosX + int (float(self.visibleWidth())/self.ZoomFactor))+2
       Y2=min (Y2,PosY+  int (float(self.visibleHeight())/self.ZoomFactor))+2
       X,Y=max (X,PosX), max (Y,PosY)
       if (X>X2) or (Y>Y2): return       
       W,H=max (X2-X,1), max (Y2-Y,1)

       self.RedrawingObjectFlag=1
       
       (self.rx,self.ry,self.rw,self.rh)=(int(self.ZoomFactor*X),int(self.ZoomFactor*Y),int(self.ZoomFactor*W),int(self.ZoomFactor*H))       
       if self.ZoomFactor<1:self.rw,self.rh=max (self.rw,1)+1 , max (self.rh,1)+1
       (self.orx,self.ory,self.orw,self.orh)=(float(self.rx)/self.ZoomFactor, float(self.ry)/self.ZoomFactor, float(self.rw)/self.ZoomFactor , float(self.rh)/self.ZoomFactor )
       self.orx,self.ory,self.orw,self.orh=int(self.orx),int(self.ory),int(self.orw)+2,int(self.orh)+2
       if self.ZoomFactor>1:self.orw,self.orh = self.orw+2,self.orh+2
       self.repaintContents ( self.rx,self.ry,self.rw,self.rh, 0 )
       
       self.RedrawingObjectFlag=0
       
   def SetPalette(self,palette):
        pass
    
    
   def SetScrollRegion(self,scroll_region):
        pass
            
   def Show(self):
       self.show()
    
   def Destroy(self):
       self.parent=None
       self.ImageItem=None
       self.Canvas=None
       self.GraphicObjects=[]
       

   def GetWindowPosition(self):
       return (int (float(self.contentsX())/self.ZoomFactor),int (float(self.contentsY())/self.ZoomFactor))

   def GetWindowHeight(self):
        return self.height()

   def GetWindowWidth(self):
        return self.width()

   def GetZoomFactor(self):
       return self.ZoomFactor

   def GetHeight(self):
        return self.Canvas.height()

   def GetWidth(self):
        return self.Canvas.width()

   def GetViewport(self):
        return self.viewport()
