static char RcsId[] = "$Header.";

#include <SpecFile.h>
#include <SpecFileP.h>

#define SF_INDEX_SUFFIX      ".sfI"
#define SF_SIGNATURE         "2ruru Sf0.0"
#define SF_SIGNATURE_MAXSIZE 20
#define STATIC_BUFSIZE 60


/*********************************************************************
 *   Function:		SpecFile *SfOpen( name, error )
 *
 *   Description:	Opens connection to Spec data file. 
 *			Creates index list in memory.
 *
 *   Parameters:
 *		Input :	
 *			(1) Filename 	
 *		Output:	   
 *			(2) error number
 *   Returns:
 *			SpecFile pointer.
 *			NULL if not successful.
 *			( If an error has occured during reading file lines
 *			  ( SF_ERR_FILE_READ or SF_ERR_MEMORY_ALLOC )
 *			  a non NULL SpecFile pointer will be returned
 *			  and the partial data normally can be accessed.)
 *   Possible errors:
 *			SF_ERR_FILE_OPEN
 *			SF_ERR_FILE_READ
 *			SF_ERR_MEMORY_ALLOC
 *			
 *********************************************************************/
/*
 *   $Log:
 *
 *********************************************************************/

SpecFile *
SfOpen( name, error ) 

char *name;
int  *error;
{
     SpecFile   *sf = NULL;
     FILE	*file, *idxfile;
     char       *idxname;
     char        signature[SF_SIGNATURE_MAXSIZE] = {0};
     long        sflen, oldsflen;

     static char buffer[STATIC_BUFSIZE];

     if ( sf_debug ) 
        fprintf( stderr, "Entered SfOpen()\n");

     /*
      * Open file
      */

     if ( (file = fopen(name, "r")) == NULL ) {
          *error = SF_ERR_FILE_OPEN;
          return( (SpecFile *)NULL );
     }
     sflen = getflength(file, name);
 
     if (sf_index) {
        if (strlen(name) * sizeof(SF_INDEX_SUFFIX) > STATIC_BUFSIZE) {
           idxname = (char *) malloc (strlen(name) * sizeof(SF_INDEX_SUFFIX) * sizeof(char) + 1);
        } else {
           idxname = buffer;
        }
        strcat(strcpy(idxname, name), SF_INDEX_SUFFIX);
        if (idxfile = fopen(idxname, "r")) {
           if (fread(signature, SF_SIGNATURE_MAXSIZE, 1, idxfile) &&
              !strncmp(signature, SF_SIGNATURE, SF_SIGNATURE_MAXSIZE) &&
              fread(&oldsflen, sizeof(oldsflen), 1, idxfile)) {
              if (oldsflen == (sflen = getflength(file, name))) {
                 sf = sfReadIndex(idxfile);
                 sf->fd = file;
                 sf->sflen=sflen;
                 sf->sfname=name;
              } 
           }
           fclose(idxfile);
        }
     }

     if (!sf) {
        if (sf = sfReadFile(file, error)) {
           if (sf->no_scans && sf_index)
              sfWriteIndex(idxname, sf, sflen);
           sf->sflen=sflen;
           sf->sfname=name;
        }
     }

     if (idxname != buffer) free(idxname);
     return(sf);
}

/***********************************************************************/
int
sfUpdateList(sf,error)
SpecFile *sf;
int	 *error;
{
     char       *buf;
     char        buf2[BUFFER_SIZE];
     char        ctrl=0;
     static SpecScan   scan;
 
     if ( sf_debug )
        fprintf(stderr, "Entering sfUpdateList\n");

     if (sf==NULL) { 
        *error =SF_ERR_LIST_NOT_FOUND; /* there must be a list when updating */ 
        return( -1 ); 
     }
     /* 
      * Tells analyzeline that it's the beginning of a update
      * Unlinks the last element because maybe the last scan is not finished
      */
     analyzeLine( sf, &ctrl, error );


     /*
      * Alloc memory for line buffer.
      */
     if ( (buf=(char *)malloc( BUFFER_SIZE )) == (char *)NULL ) {
	  free( sf );
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( -1 );
     }
     /*
      * Read line by line and get interesting information.
      */
     do {
	  if ( fgets(buf, BUFFER_SIZE, sf->fd) != (char *)NULL ) {
	       if ( ferror(sf->fd) ) {
		    free( buf );
		    *error = SF_ERR_FILE_READ;
		    return( -1 );
	       } 
	       if ( strlen(buf) == BUFFER_SIZE -1 ) {
                 if ( buf[ strlen(buf) -1 ] != '\n') {
		    do {
		       if ( fgets(buf2, sizeof(buf2), sf->fd) != (char *)NULL ) {
			   if ( ferror(sf->fd) ) {
				free( buf );
				*error = SF_ERR_FILE_READ;
				return( -1 );
			   }		    
			   buf=(char *)realloc(buf,strlen(buf)+strlen(buf2)+1);
			   if ( buf == (char *)NULL ) {
				*error = SF_ERR_MEMORY_ALLOC;  
				return( -1 );
			   }
			   strcat( buf, buf2 ) ;
		       }
		    }while( strlen(buf2) == sizeof(buf2) -1 );       
                 }
	       }
	       if ( analyzeLine( sf, buf, error ) ) {
		    free( buf );
		    return( -1 );
	       }
         }
     } while ( !feof( sf->fd ) ); 
     free( buf );      
     /*
      * Get the last scan
      */
     analyzeLine( sf, (char *)NULL, error );

     return( 0 );
}
/*********************************************************************
 *   Function:		SpecFile *SfUpdate( sf, error )
 *
 *   Description:	Updates connection to Spec data file . 
 *			Appends to index list in memory.
 *
 *   Parameters:
 *		Input :	
 *			(1) sf (pointer to the index list in memory) 
 *                      (2) Filename	
 *		Output:	   
 *			(2) error number
 *   Returns:
 *                      ( 0 ) => Ok.
 *			(-1 ) => no list in memory.
 *                  
 *   Possible errors:
 *                      SF_ERR_LIST_NOT_FOUND
 *			SF_ERR_FILE_OPEN
 *			SF_ERR_MEMORY_ALLOC
 *			SF_ERR_FILE_READ
 *			
 *********************************************************************/
/*
 *   $Log:
 *         Added by D.F.C Jan/98      
 *         Modified       May/98
 *
 *********************************************************************/

int 
SfUpdate ( sf, error ) 

SpecFile *sf;
int  *error;
{
     char       *dammit, *idxname;
     char        signature[SF_SIGNATURE_MAXSIZE] = {0};
     long        sflen, oldsflen;

     static char buffer[STATIC_BUFSIZE];

     if ( sf_debug ) 
        fprintf( stderr, "Entered SfUpdate()\n");

     sflen = getflength(sf->fd, dammit);
     oldsflen=sf->sflen;
     if ( sf_debug ) 
        fprintf( stderr, "Length=%d\n",sflen);
     if (oldsflen != (sflen = getflength(sf->fd, dammit))) {
        if ( sf_debug ) 
           fprintf( stderr, "Length=%ld,old=%ld\n",sflen,sf->sflen);
        fseek(sf->fd,oldsflen,SEEK_SET); /* Goes to the last scan indx*/

        sf->sflen=sflen;

        if (sfUpdateList(sf,sflen,error)) 
           return( -1 );
        else if (sf_index) {
           /* Updates index file. Since it is not possible to overwrite 
              a simple value in the middle of the file, and we need to 
              change the value for the length of the scan file (sflen),
              the whole index file is written */
           if (strlen(sf->sfname) * sizeof(SF_INDEX_SUFFIX) > STATIC_BUFSIZE) {
              idxname=(char *)malloc(strlen(sf->sfname) 
                      * sizeof(SF_INDEX_SUFFIX) * sizeof(char) + 1);
           } else {
              idxname = buffer;
           }
           strcat(strcpy(idxname, sf->sfname), SF_INDEX_SUFFIX);
           if (sf_debug)  fprintf(stderr, "Updating index file\n");
           sfWriteIndex(idxname,sf,sflen); 
           if (idxname != buffer) free(idxname);
        }
     }else {
        if ( sf_debug ) 
           fprintf(stderr,"Scan File \"%s\" is up to date\n",sf->sfname);
     }

     return(0);

}

/******************************************************************/

void sfWriteIndex(name, sf, sflen)
char *name;
SpecFile *sf;
long  sflen;
{
     ObjectList  *ptr;
     FILE        *fdi;

     if ( sf_debug )
        fprintf(stderr, "Writing indexes\n");

     if ((fdi = fopen(name,"w")) == NULL) {
          if ( sf_debug )
               fprintf(stderr, "Cannot open index file for writing\n");
     } else {
          if ( sf_debug )
               fprintf(stderr, "Writing index file \n");

          fwrite(SF_SIGNATURE, SF_SIGNATURE_MAXSIZE, 1, fdi);
          fwrite(&sflen, sizeof(sflen), 1, fdi);
          for (ptr=sf->list.first;ptr;ptr=ptr->next)
               fwrite((SpecScan *)(ptr->contents),sizeof(SpecScan),1,fdi);
          fclose(fdi);
     }
}
/*******************************************************************/
SpecFile *
sfReadIndex(fdi)
FILE  *fdi;
{
     SpecFile         *sf;
     long              loop;
     int               i;


     static SpecScan   scan;

     loop = 1;
     i    = 0;

     if ( sf_debug )
        fprintf(stderr, "Reading index file\n");

     /*
      * Alloc memory for SpecFile struct.
      */
     if( (sf = (SpecFile *) malloc(sizeof(SpecFile))) == (SpecFile *)NULL ) {
/*
          *error = SF_ERR_MEMORY_ALLOC;
*/
          return(NULL);
     }

     /*
      * Initialize SpecFile structure.
      */
     sf->fd             = fdi;
     sf->list.first     = (ObjectList *)NULL;
     sf->list.last      = (ObjectList *)NULL;
     sf->no_scans       =  0;
     sf->current        = (ObjectList *)NULL;
     sf->data           = (double    **)NULL;
     sf->data_info      = (long       *)NULL;
     sf->labels         = (char      **)NULL;
     sf->no_labels      = -1;
     sf->motor_pos      = (double     *)NULL;
     sf->no_motor_pos   = -1;
     sf->motor_names    = (char      **)NULL;
     sf->no_motor_names = -1;
     sf->header_offset  = -1;
     sf->epoch          = -1;
    /*
     * Read until EOF
     */
    while (loop) {
             if (fread(&scan,sizeof(SpecScan),1,fdi)==0) {
                 loop = 0;
                 break;
             }
             if ( addToList( &(sf->list), (void *)&scan, (long)sizeof(SpecScan)) ) {
/*
                  *error = SF_ERR_MEMORY_ALLOC;
*/
                  return( sf );
             }
             i++;

    }
    sf->no_scans = i;

    return(sf);
}
/***********************************************************************/
SpecFile *
sfReadFile(file, name, error)
     
FILE	*file;
char    *name;
int	*error;
{
     SpecFile	*sf;
     char       *buf;
     char        buf2[BUFFER_SIZE];
     
     /*
      * Alloc memory for SpecFile struct.
      */
     if((sf = (SpecFile *) malloc(sizeof(SpecFile))) == (SpecFile *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( sf );
     }
     /*
      * Alloc memory for line buffer.
      */
     if ( (buf=(char *)malloc( BUFFER_SIZE )) == (char *)NULL ) {
	  free( sf );
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( (SpecFile *)NULL );
     }
     /*
      * Initialize SpecFile structure.
      */
     sf->fd	       	= file;
     sf->sfname	       	= name;
     sf->list.first    	= (ObjectList *)NULL;
     sf->list.last     	= (ObjectList *)NULL;
     sf->no_scans      	=  0;
     sf->current       	= (ObjectList *)NULL;
     sf->data		= (double    **)NULL;
     sf->data_info     	= (long	      *)NULL;
     sf->labels	       	= (char	     **)NULL;
     sf->no_labels    	= -1;
     sf->motor_pos     	= (double     *)NULL;
     sf->no_motor_pos   = -1;
     sf->motor_names	= (char	     **)NULL;
     sf->no_motor_names	= -1;
     sf->header_offset	= -1;
     sf->epoch	       	= -1;
     /*
      * Read line by line and get interesting information.
      */
     while ( !feof( file ) ) {
	  if ( fgets(buf, BUFFER_SIZE, file) != (char *)NULL ) {
	       if ( ferror(file) ) {
		    free( buf );
		    if ( !feof(file) ) return( sf );
		    *error = SF_ERR_FILE_READ;
		    return( sf );
	       } 
	       if ( strlen(buf) == BUFFER_SIZE -1 ) {
                 if ( buf[ strlen(buf) -1 ] != '\n') {
		    do {
		       if ( fgets(buf2, sizeof(buf2), file) != (char *)NULL ) {
			   if ( ferror(file) ) {
				free( buf );
				if ( !feof(file) ) return( sf );
				*error = SF_ERR_FILE_READ;
				return( sf );
			   }		    
			   buf=(char *)realloc(buf,strlen(buf)+strlen(buf2)+1);
			   if ( buf == (char *)NULL ) {
				*error = SF_ERR_MEMORY_ALLOC;  
				return( sf );
			   }
			   strcat( buf, buf2 ) ;
		      }
		  } while( strlen(buf2) == sizeof(buf2) -1 );       
                }
	       }
	       if ( analyzeLine( sf, buf, error ) ) {
		    free( buf );
		    return( sf );
	       }
         }
     }
     free( buf );      
     /*
      * Get the last scan
      */
     analyzeLine( sf, (char *)NULL, error );
     
     return( sf );
}


/*********************************************************************
 *   Function:		int SfClose( sf )
 *
 *   Description:	Closes a file previously opened with SfOpen()
 *			and frees all memory .
 *   Parameters:
 *		Input:	
 *			File pointer   
 *   Returns:
 *			0 :  close successful
 *		       -1 :  errors occured
 *
 *********************************************************************/
int
SfClose( sf )

SpecFile *sf;
{
     register ObjectList  *ptr;
     register ObjectList  *prevptr;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfClose()\n");

     for( ptr=sf->list.last ; ptr ; ptr=prevptr ) {
 	  free( (SpecScan *)ptr->contents ); 
          prevptr = ptr->prev;
	  free( (ObjectList *)ptr );
     }     

     if( fclose(sf->fd) ) {
	  return( -1 ) ;
     }
     if (sf->no_scans) {
        if (sf->data_info != NULL) {
           freeArrNZ( (void ***)&(sf->data),	      sf->data_info[ROW] );
           free     (  sf->data_info	 );
        }
        if (sf->no_labels) {
           freeArrNZ( (void ***)&(sf->labels),      sf->no_labels	 );
           free     (  sf->motor_pos 	 );
           freeArrNZ( (void ***)&(sf->motor_names), sf->no_motor_names );
        }
     }
     free     (  sf );
     sf = (SpecFile *)NULL;

     return   (   0 );
}

/*********************************************************************
 *   Function:		int analyzeLine( sf, line, error )
 *
 *   Description:	analyzes data from line and acts on the scan list
 *			if necessary.
 *			This is the function that controls flow information
 *			into the list.
 *   Parameters:
 *		Input :	(1) SpecFile pointer 
 *			(2) Line        
 *
 *		Output: (3) error number
 *   Returns:
 *                 Linked list of SpecScan structures,
 *		   other SpecFile data:
 *
 *		   SpecFile.no_scans	  => total no.of scans
 *			    current	  =>  NULL
 *			    data,
 *			    data_info,	
 *			    labels,
 *			    motor_pos,
 *			    motor_names	  => not read ( set to NULL ).
 *			    no_labels,
 *			    no_motor_pos,
 *			    no_motor_names=> -1
 *			    header_offset => -1 ( last read #F header )
 *			    epoch	  => -1 ( epoch in the last header )
 *   Return value:
 *			 0 : OK
 *			-1 : error
 *   Possible errors:
 *			SF_ERR_FILE_READ
 *			SF_ERR_MEMORY_ALLOC    
 *			
 *********************************************************************/
int 
analyzeLine( sf, line, error )
     
SpecFile  *sf;
char      *line;
int	  *error;
{
     char		c;
     char	       *therest;
     register  char    *strptr;
     static    SpecScan	scan, *scandavid;
     static    state	= WAITING_FILE;
     static    long	index			=  0;
     static    long     order			=  1;
     static    long	offset			= -1;
     static    long	data_offset		= -1;
     static    long	last_data_offset	= -1;
     static    long	last_header		= -1;
     static    long	data_lines		=  0;
     static    long	header_lines_before	=  0;
     static    long 	header_lines_after	=  0;
    

     if ( sf_debug ) 
       fprintf( stderr, "Entered analyzeLine()\n");

     if (line == NULL) {  /* last line */
	  /* 
	   * Write last scan.
	   */
	  if (state == READING_SCAN ||
	      state == READING_DATA) {
	       scan.data_offset		= data_offset;
	       scan.last_data_offset	= last_data_offset;
	       scan.data_lines		= data_lines;
	       scan.header_lines_before	= header_lines_before;
	       scan.header_lines_after	= header_lines_after;
	       if ( addToList( &(sf->list), (void *)&scan, (long)sizeof(SpecScan)) ) {
		    *error = SF_ERR_MEMORY_ALLOC;
		    return( -1 );
	       }
	  }
	  /* 
	   * And clear for next file.
	   */
	  state	      = WAITING_FILE;
	  index			=  0;
	  order			=  1;
	  offset		= -1;
	  data_offset		= -1;
	  last_data_offset	= -1;
	  last_header		= -1;
	  data_lines		=  0;
	  header_lines_before	=  0;
	  header_lines_after	=  0;
	  
	  return( 0 );
     }

     if (*line==0) {
        /* begining of an update */

 	  state       		= READING_DATA;
	  index			= 
		((SpecScan *)(sf->list.last->contents))->index;
	  order			= 
		((SpecScan *)(sf->list.last->contents))->order;
	  offset		= 
		((SpecScan *)(sf->list.last->contents))->offset;
	  data_offset		= 
		((SpecScan *)(sf->list.last->contents))->data_offset;
	  last_data_offset	= 
		((SpecScan *)(sf->list.last->contents))->last_data_offset;
	  last_header		= 
		((SpecScan *)(sf->list.last->contents))->last_header;
	  data_lines		= 
		((SpecScan *)(sf->list.last->contents))->data_lines;
                    

	  header_lines_before	= 
		((SpecScan *)(sf->list.last->contents))->header_lines_before;
	  header_lines_after	= 
		((SpecScan *)(sf->list.last->contents))->header_lines_after;

          unlinkFromList( &(sf->list), (ObjectList *)(sf->list.last)); 


     } else if (*line=='#') {
	  /*
	   * Header line.
	   */
	  c = *(line+1);
	  
	  switch (c) {
	       
	     case SF_FILE_NAME: 
	       /*
		* New file header.
		*/
	       if (state == READING_SCAN ||
		   state == READING_DATA) {
		    scan.data_offset		= data_offset;
		    scan.last_data_offset	= last_data_offset;
		    scan.data_lines		= data_lines;
		    scan.header_lines_before	= header_lines_before;
		    scan.header_lines_after	= header_lines_after;
		    if (addToList( &(sf->list),(void *)&scan,(long)sizeof(SpecScan)) ) {
			 *error = SF_ERR_MEMORY_ALLOC;
			 return( -1 );
		    }
	       }
	       state		= READING_HEADER;
	       last_header      = ftell(sf->fd) - strlen(line) * sizeof(char);
	       if ( last_header == -1L ) {
		    *error = SF_ERR_FILE_READ;
		    return( -1 );
	       }

	       break;
	       
	     case SF_SCAN_NUM:
	       /* 
		* New scan.
		*/
	       if (state == READING_SCAN ||
		   state == READING_DATA) {
		    scan.data_offset		= data_offset;
		    scan.last_data_offset	= last_data_offset;
		    scan.data_lines		= data_lines;
		    scan.header_lines_before	= header_lines_before;
		    scan.header_lines_after	= header_lines_after;
		    if (addToList( &(sf->list),(void *)&scan,(long)sizeof(SpecScan)) ) {
			 *error = SF_ERR_MEMORY_ALLOC;
			 return( -1 );
		    }
	       }
	       state        = READING_SCAN;
	       scan.index   = ++( sf->no_scans );
	       therest      = line+2;
	       scan.scan_no = atol( therest );
	       scan.order   = setScanOrder( &(sf->list), scan.scan_no);
	       scan.offset  = ftell(sf->fd) - strlen(line) * sizeof(char);
	       if ( scan.offset == -1L ) {
		    *error = SF_ERR_FILE_READ;
		    return( -1 );
	       }	       
	       data_offset		= -1;
	       last_data_offset		= -1;
	       scan.last_header		= last_header;
	       data_lines		=  0;
	       header_lines_before	=  1;
	       header_lines_after	=  0;
	       break;
	       
	     default:
	       if (state == READING_DATA) {
		    header_lines_after += 1;
	       }
	       if (state == READING_SCAN) {
		    header_lines_before += 1;
	       }		      
	       break;
	  }
     } 
     else {
	  /* 
	   * Dataline or empty.
	   */
	  strptr = line;
	  for ( c=*strptr ; c!='\0' && c!= '\n' ; c=*(strptr++) ) {
	       if ( c != ' ' && c != '\t') {
		    /* This line has data */
		    if (data_lines == 0) {

			 state = READING_DATA;

			 data_offset = ftell(sf->fd)-strlen(line)*sizeof(char);
			 if ( data_offset == -1L) {
			      *error = SF_ERR_FILE_READ;
			      return( -1 );
			 }	       
		    }
		    data_lines++;
		    last_data_offset = ftell(sf->fd)-strlen(line)*sizeof(char);
		    if ( last_data_offset == -1L) {
			 *error = SF_ERR_FILE_READ;
			 return( -1 );
		    }	       	   
		    break;
	       }
	  }
     }
     return( 0 );
}

/*********************************************************************
 *   Function:		int setScanOrder( list, number )
 *			( Internal function used only during opening of
 *							  the SpecFile )
 *   Description:	Gets scan order.
 *
 *   Parameters:
 *		Input:	(1) list pointer   
 *			(2) scan number
 *   Returns:
 *			Scan order
 *
 *********************************************************************/
int
setScanOrder( list, number )

ListHeader *list;
long	    number;
{
     register ObjectList       *ptr;
     long			order=1;

     if ( sf_debug ) 
       fprintf( stderr, "Entered setScanOrder()\n");

     for ( ptr=list->first; ptr ; ptr=ptr->next ) {
	  if ( ((SpecScan *)ptr->contents)->scan_no == number ) {
	       order++;
	  }
     }
     return( order );
}
  
/*********************************************************************
 *   Function:		long *SfList( sf, error )
 *
 *   Description:	Creates an array with all scan numbers.
 *
 *   Parameters:
 *		Input :	SpecFile pointer
 *   Returns:
 *			Array with scan numbers.
 *			NULL if errors occured.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *			
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long *
SfList( sf, error )
     
SpecFile	*sf;
int		*error;
{
     register	ObjectList	*ptr;
     long			*scan_list;
     long			 i = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfList()\n");

     scan_list = (long *)malloc( sizeof(long) * (sf->no_scans) );
     if ( scan_list == (long *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return( scan_list );
     }
     
     for ( ptr=sf->list.first ; ptr ; ptr=ptr->next ) {
	  scan_list[i] = ( ((SpecScan *)(ptr->contents))->scan_no );  
	  i++;
     }    
     return( scan_list );
}

long
SfIndexes( sf, number, idxlist )

SpecFile       *sf;
long            number;
long          **idxlist;
{
     ObjectList     *ptr;
     long            i;
     long           *indexes;
     long           *arr;

     i = 0;
     indexes =  (long *)malloc(sf->no_scans * sizeof(long));

     if ( sf_debug )
       fprintf( stderr, "Entered SfIndexes()\n");

     for (ptr = sf->list.first; ptr; ptr=ptr->next ) {
           if ( number == ((SpecScan *)(ptr->contents))->scan_no) {
              indexes[i] = ((SpecScan *)(ptr->contents))->index;
              i++;
           }
     }
         
     if (i == 0) 
        arr = (long *) NULL;
     else {
        arr = (long *)malloc(sizeof(long) * i);
        memcpy(arr,indexes,sizeof(long) * i);
     }

     *idxlist = arr;
     free(indexes);
     return( i );
}

/*********************************************************************
 *   Function:		long SfIndex( sf, number, order )
 *
 *   Description:	Gets scan index from scan number and order.
 *
 *   Parameters:
 *		Input :	(1) Scan number
 *			(2) Scan order 
 *   Returns:
 *			Index number. 
 *			(-1) if not found.
 *			
 *********************************************************************/
long
SfIndex( sf, number, order )

SpecFile       *sf;
long		number;
long		order;
{
     ObjectList		*ptr;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfIndex()\n");

     ptr = findScanByNo( &(sf->list), number, order );
     if ( ptr != (ObjectList *)NULL ) 
        return( ((SpecScan *)(ptr->contents))->index ); 
     
     return( -1 );
}

/*********************************************************************
 *   Function:		int findIndex( scan, number )
 *
 *   Description:	Compares if number == scan index .
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) number 
 *   Returns:
 *			0 : not found
 *			1 : found
 *
 *********************************************************************/
int
findIndex( scan, number )

SpecScan	*scan;
void		*number;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered findIndex()\n");

     return( scan->index == *(long *)number );
}

/*********************************************************************
 *   Function:		int findNoAndOr( scan, number )
 *			      ( Number 
 *				     Order )
 *
 *   Description:	Compares if number1 = scan number and
 *				    number2 = scan order 
 *   Parameters:
 *		Input:	(1) SpecScan pointer
 *			(2) number[1] 
 *   Returns:
 *			0 : not found
 *			1 : found
 *
 *********************************************************************/
int
findNoAndOr( scan, number )

SpecScan	*scan;
void		*number;
{

     long *n = (long *)number;

     if ( sf_debug ) 
       fprintf( stderr, "Entered findNoAndOr()\n");

     return( ( scan->scan_no == *n++ )&&
	     ( scan->order   == *n ));
}

/*********************************************************************
 *   Function:		ObjectList *findScanByIndex( list, index )
 *
 *   Description:	Looks for a scan . 
 *
 *   Parameters:
 *		Input:	(1) List pointer
 *			(2) scan index 
 *   Returns:
 *			ObjectList pointer if found ,      
 *			NULL if not.
 *
 *********************************************************************/
ObjectList *
findScanByIndex( list, index )

ListHeader  *list;
long         index;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered findScanByIndex()\n");

     return findInList( list, findIndex, (void *)&index );
}

/*********************************************************************
 *   Function:		ObjectList findScanByNo( list, scan_no, order )
 *
 *   Description:	Looks for a scan . 
 *
 *   Parameters:
 *		Input:	(1) List pointer
 *			(2) scan number
 *			(3) scan order 
 *   Returns:
 *			ObjectList pointer if found ,      
 *			NULL if not.
 *
 *********************************************************************/
ObjectList *
findScanByNo( list, scan_no, order )

ListHeader     *list;
long		scan_no;
long		order;
{
     long	 value[2];

     if ( sf_debug ) 
       fprintf( stderr, "Entered findScanByNo()\n");

     value[0] = scan_no;
     value[1] = order;

     return( findInList( list, findNoAndOr, (void *)value) );
}

/*********************************************************************
 *   Function:		void deleteScan( sf, index )
 *
 *   Description:	Removes one Scan from the list,
 *			updates SpecFile data.
 *   Parameters:
 *		Input :	(1) SpecFile pointer
 *			(2) Pointer to the element
 *
 *********************************************************************/
void
deleteScan( sf, index )

SpecFile	*sf;
long		 index;

{
     ObjectList		*ptr;
	

     if ( sf_debug ) 
       fprintf( stderr, "Entered deleteScan()\n");

     if ( (ptr = findScanByIndex( &(sf->list),index)) != (ObjectList *)NULL ) {
	  if ( (ptr == sf->current) && (ptr->prev != (ObjectList *)NULL) ) {
	       setScanCurrent( sf, ptr->prev );
	  } else {
	       setScanCurrent( sf, ptr->next );
	  }
	  sf->no_scans--;
	  unlinkFromList( &(sf->list), ptr );
     }
}

/*********************************************************************
 *   Function:		long mulstrtod( str, arr, error )
 *
 *   Description:	Converts string to data array.( double array )
 *
 *   Parameters:
 *		Input :	(1) String
 *
 *		Output:   
 *			(2) Data array
 *			(3) error number
 *   Returns:
 *			Number of values.
 *			( -1 ) in case of errors. 
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long 
mulstrtod( str, arr, error )
     
char		*str;
double	       **arr;
int		*error;
{
     int	 count,q,i=0;
     double	*ret;
     double	*tmpret;
     char	*str2;

     
     if ( sf_debug ) 
       fprintf( stderr, "Entered mulstrtod()\n");

     *arr = (double *)NULL;
     str2 = str;
     
     tmpret = (double *)malloc( sizeof(double) * (int)((strlen(str))/2 +1 ));
     if ( tmpret == (double *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return( -1 );
     }
     while( (q = sscanf(str2, "%lf%n", &(tmpret[i]), &count)) > 0 ) {
	  i++;
	  str2 += count;
     }
     str2++;
     
     if ( !i ) {
	  free( tmpret );
	  return( i );
     }
     ret = (double *)malloc( sizeof(double) * i );
     if ( ret == (double *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  free( tmpret );
	  return( -1 );
     }
     memcpy(ret, tmpret, i * sizeof(double) );
     
     free( tmpret );
     *arr = ret;
     return( i );
}

/*********************************************************************
 *   Function:		char *readFileLine( fd, error )
 *
 *   Description:	Reads one line from a file.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *
 *		Output:	(2) error number
 *   Returns:
 *			Pointer to the line,
 *			NULL in case of errors.
 *
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *			SF_ERR_FILE_READ
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
readFileLine( fd, error )

FILE    *fd;
int     *error;
{
     char	       *buf ;
     char		buf2[BUFFER_SIZE];


     if ( sf_debug ) 
       fprintf( stderr, "Entered readFileLine()\n");

     clearerr( fd );
     /*
      * Alloc memory for line buffer.
      */
     if ( (buf=(char *)malloc( BUFFER_SIZE )) == (char *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return( buf );
     }
     /*
      * Read line 
      */	 
     if ( fgets( buf, BUFFER_SIZE, fd ) != (char *)NULL ) { 
	  if ( ferror(fd) ) {
	       free( buf );
	       if ( !feof( fd ) ) return( (char *)NULL );
	       *error = SF_ERR_FILE_READ;
	       return( (char *)NULL );
	  }
	  if ( strlen(buf) == BUFFER_SIZE -1 ) {
               if ( buf[ strlen(buf) -1 ] != '\n') {
	       do {
		    if ( fgets (buf2, sizeof(buf2), fd) != (char *)NULL ) { 
			 if ( ferror(fd) ) {
			      free( buf );
			      if ( !feof( fd ) ) return( (char *)NULL );
			      *error = SF_ERR_FILE_READ;
			      return( (char *)NULL );
			 }
			 buf=(char *)realloc( buf,strlen(buf)+strlen(buf2)+1 );
			 if ( buf == (char *)NULL ) {
			      *error = SF_ERR_MEMORY_ALLOC;
			      return( (char *)NULL );
			 }
			 strcat( buf, buf2 ) ;
		    } 
	       } while( strlen(buf2) == sizeof(buf2) -1 );       
	  }
        }
     }
     if ( buf[ strlen(buf)-1 ] == '\n' ) buf[ strlen(buf)-1 ] = '\0';
     return( buf );
}

/*********************************************************************
 *   Function:		char *findLine( fd, string, end, error )
 *
 *   Description:	Gets a line, which beginning == 'string'.
 *
 *   Parameters:
 *		Input :	(1) File pointer( where to begin ...)  
 *			(2) string
 *			(3) end     ( ... where to stop the search )
 *		Output:
 *			(4) error number
 *   Returns:
 *			Pointer to the line .
 *			NULL in case of errors or if not found.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => readFileLine()
 *			SF_ERR_FILE_READ
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
findLine( fd, string, end, error )

FILE	*fd;
char	*string;
long	 end;
int	*error;
{
     char	       *str ;
     long		position;
     int		found;
     unsigned		i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered findLine()\n");

     /*
      * Read line from the file
      */
     if ( (position = ftell( fd )) == -1L ) {
	  *error = SF_ERR_FILE_READ;
	  return( (char *)NULL );
     }
     while ( !feof( fd ) && position < end ) {
	  if ( (str = readFileLine( fd, error )) == (char *)NULL ) {
	       return( (char *)NULL );
	  }
	  /*
	   * Compare
	   */
	  found = 1;
	  for ( i=0 ; i < strlen(string) && string[i]!='\n' ; i++ ) {
	       if ( str[i] != string[i] ) {
		    found = 0;
		    break;
	       }
	  }
	  if ( found ) {
	       return( str );
	  } else {
	       if ( (position = ftell( fd )) == -1L ) {
		    *error = SF_ERR_FILE_READ;
		    free( str );
		    return( (char *)NULL );
	       }
	  }
	  free( str );
     }
     return( (char *)NULL );
}

/*********************************************************************
 *   Function:		char *readHeader( fd, sf_char, end, error )
 *
 *   Description:	Gets one '#sf_char' line.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) sf_character
 *			(3) end ( where to stop the search )
 *		Output:
 *			(4) error number
 *   Returns:
 *			Pointer to the line ,
 *			NULL in case of errors.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *			SF_ERR_FILE_READ	| => findLine()
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
readHeader( fd, sf_char, end, error )

FILE	*fd;
char	 sf_char;
long	 end;
int	*error;
{
     char	       *line;
     char	       *str ;
     char	       *c;
     char	       string[]="# \0";
     long		i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered readHeader()\n");

     string[1] = sf_char;
     if ( (str = findLine( fd, string, end, error ))==(char *)NULL ) 
        return( str );

     /*
      * Delete 'string' from line.
      */
     c = str;
     for ( i=strlen(string) ; i ; i-- ) c++; 
     /*
      * Delete blanks
      */
     for ( ; *c==' ' || *c=='\t' ; c++ );
     line = (char *)malloc( sizeof(char) * (strlen( c ) + 1) ); 
     if ( line == (char *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  free( str );
	  return( line );
     }
     strcpy( line, c );
     free( str );
     return( line );    
}

/*********************************************************************
 *   Function:		long SfNoColumns( sf, index, error )
 *
 *   Description:	Gets number of columns in a scan
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Number of scan columns.(From #N line !)
 *			( -1 ) if errors occured.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => readHeader()
 *			SF_ERR_LINE_NOT_FOUND	
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *
 *********************************************************************/
long
SfNoColumns( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     char		*line;
     long		 end;
     long		 col = -1;
     int		 err =  0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfNoColumns()\n");

     /*
      * Find scan
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( col );
     }

     setScanCurrent( sf, list );
     scan = list->contents;

     end  = ((SpecScan *)list->contents)->data_offset;

     if ( end < 1 ) {
	  if ( (ObjectList *)list->next == (ObjectList *)NULL ) {

	       if ( fseek( sf->fd, 0, SEEK_END ) ) {
		    *error = SF_ERR_FILE_READ;
		    return( col );		    
	       }
	       if ( (end = ftell( sf-> fd ))==-1L ) {
		    *error = SF_ERR_FILE_READ;
		    return( col );		    
	       }	       
	       
	  } else {
	       if ( ((SpecScan *)list->next->contents)->last_header !=
						  scan->last_header ) {
		    end = ((SpecScan *)list->next->contents)->last_header;
	       } else {
		    end = ((SpecScan *)list->next->contents)->offset;
	       }
	  }
     }
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, scan->offset, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( col );
     }

     line = readHeader( sf->fd, SF_COLUMNS, end, &err );
     if ( line == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LINE_NOT_FOUND;
	  else *error = err;
	  return( col );
     }
     col = atol( line );
     free(  line );
     return( col );
}

/*********************************************************************
 *   Function:		long SfEpoch( sf, index, error )
 *
 *   Description:	Gets epoch from the last file header.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Epoch.(From #E line !)
 *			( -1 ) if errors occured.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => readHeader()
 *			SF_ERR_LINE_NOT_FOUND	
 *			SF_ERR_FILE_READ
 *			SF_ERR_HEADER_NOT_FOUND
 *			SF_ERR_SCAN_NOT_FOUND
 *
 *********************************************************************/
long 
SfEpoch( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     char		*line;
     long		 start;
     long		 end;
     long		 epoch = -1;
     int		 err   =  0;

     if ( sf_debug ) 
       fprintf( stderr, "Entered SfEpoch()\n");

     /*
      * Find scan , set to current
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( -1 );
     }

     scan = list->contents;
     setScanCurrent( sf, list );

     if (sf->epoch > -1) return( sf->epoch );

     if ( (start = scan->last_header) < 0 ) {
	  *error = SF_ERR_HEADER_NOT_FOUND ;
	  return( -1 );
     }
     end = findFirstInFile( list );
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, start, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( -1 );
     }

     line = readHeader( sf->fd,  SF_EPOCH, end, &err );
     if ( line == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LINE_NOT_FOUND;
	  else *error = err;
	  return( -1 );
     }
     epoch =  atol( line );
     free(  line );
     sf->epoch = epoch;
     return( epoch );
}

/*********************************************************************
 *   Function:		char *SfDate( sf, index, error )
 *
 *   Description:	Gets date from scan header
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Date.(From #D line !),
 *			NULL => errors.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => readHeader()
 *			SF_ERR_LINE_NOT_FOUND	
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
SfDate( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     char		*line;
     long		 end;
     int		 err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfDate()\n");

     /*
      * Find scan , set to current
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( (char *)NULL );
     }
     scan = list->contents;
     setScanCurrent( sf, list );

     end  = ((SpecScan *)list->contents)->data_offset;
     if ( end < 1 ) {
	  if ( (ObjectList *)list->next == NULL ) {
	       if ( fseek( sf->fd, 0, SEEK_END ) ) {
		    *error = SF_ERR_FILE_READ;
		    return( (char *)NULL );
	       }
	       if ( (end = ftell( sf-> fd ))==-1L ) {
		    *error = SF_ERR_FILE_READ;
		    return( (char *)NULL );
	       }	       
	  } else {
	       if ( ((SpecScan *)list->next->contents)->last_header !=
						  scan->last_header ) {
		    end = ((SpecScan *)list->next->contents)->last_header;
	       } else {
		    end = ((SpecScan *)list->next->contents)->offset;
	       }
	  }
     }
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, scan->offset, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( (char *)NULL );
     }

     line = readHeader( sf->fd, SF_DATE, end, &err );
     if ( line == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LINE_NOT_FOUND;
	  else *error = err;
     }
     return( line );
}
   
/*********************************************************************
 *   Function:		char *SfCommand( sf, index, error )
 *
 *   Description:	Reads '#S' line ( without #S and scan number ).
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			String pointer,
 *			NULL => errors.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *			SF_ERR_LINE_NOT_FOUND	 
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
SfCommand( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     char		*line;
     char		*ret_line;
     char		*c;
     long		 end;
     int		 err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfCommand()\n");

     /*
      * Find scan
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( (char *)NULL );
     }
     setScanCurrent( sf, list );
     scan = list->contents;
     end  = ((SpecScan *)list->contents)->data_offset;
     if ( end < 1 ) {
	  if ( (ObjectList *)list->next == NULL ) {
	       if ( fseek( sf->fd, 0, SEEK_END ) ) {
		    *error = SF_ERR_FILE_READ;
		    return( (char *)NULL );
	       }
	       if ( (end = ftell( sf-> fd ))==-1L ) {
		    *error = SF_ERR_FILE_READ;
		    return( (char *)NULL );
	       }	       
	  } else {
	       if ( ((SpecScan *)list->next->contents)->last_header !=
						  scan->last_header ) {
		    end = ((SpecScan *)list->next->contents)->last_header;
	       } else {
		    end = ((SpecScan *)list->next->contents)->offset;
	       }
	  }
     }
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, scan->offset, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( (char *)NULL );
     }
	
     line = readHeader( sf->fd, SF_SCAN_NUM, end, &err );
     if ( line == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LINE_NOT_FOUND;
	  else *error = err;
	  return( (char *)NULL );
     }
     /*
      * Delete scan number and blanks after it .
      */
     c = line;
     c += (int) log10((double) atoi( line )) + 1;

     while ( *c == ' ' || *c == '\t' ) {
	  c++;
     }
     /*
      * Return the rest .
      */
     ret_line = (char *)malloc(sizeof(char) * (strlen(c)+1) );  
     if ( ret_line == (char *)NULL ) {
	  *error =  SF_ERR_MEMORY_ALLOC;
	  free( line );
	  return( ret_line );
     }
     strcpy( ret_line, c );     
     free(  line );
     return( ret_line );
}

/*********************************************************************
 *   Function:	long SfFileHeader( sf, index, string, lines, error )
 *
 *   Description:	Gets all lines from the last file header
 *			which begin with 'string'.
 *			If string = "\0" or NULL all header lines
 *			will be returned.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Scan index
 *			(3) String
 *		Output:
 *			(4) Array of found lines (strings).
 *			(5) error number
 *   Returns:
 *			Number of read lines , 
 *			( -1 ) in case of errors.
 *   Possible errors:
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *			SF_ERR_HEADER_NOT_FOUND	
 *			SF_ERR_MEMORY_ALLOC	| => readHeaderLines()  
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfFileHeader( sf, index, string, lines, error )

SpecFile       *sf;
long		index;
char	       *string;
char	     ***lines;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     long		 end;
     long		 i=0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfFileHeader()\n");

     *lines = (char **)NULL;

     /*
      * Find scan
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( -1 );
     }
     setScanCurrent( sf, list );
     scan = list->contents;

     if ( scan->last_header < 0 ) {
	  *error = SF_ERR_HEADER_NOT_FOUND;
	  return( -1 );
     }
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, scan->last_header, SEEK_SET ) ) {
	  *error = SF_ERR_FILE_READ;
	  return( -1 );
     }

     end = findFirstInFile( list );

     i = readHeaderLines( sf->fd, string, end, lines, error );
     
     return( i );
}
     
/*********************************************************************
 *   Function:	long SfHeader( sf, index, string, lines, error )
 *
 *   Description:	Gets all lines from scan, which begin with 'string'.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *			(3) String
 *		Output:
 *			(4) Array of strings
 *			(5) error number
 *   Returns:
 *			Number of read lines,  
 *			( -1 ) => errors.
 *   Possible errors:
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *			SF_ERR_MEMORY_ALLOC	| => readHeaderLines()     
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfHeader( sf, index, string, lines, error )

SpecFile       *sf;
long		index;
char	       *string;
char	     ***lines;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     long		 end;
     long		 i=0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfHeader()\n");

     *lines = (char **)NULL;

     /*
      * Find scan
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( -1 );
     }
     setScanCurrent( sf, list );
     scan = list->contents;

     if ( (ObjectList *)list->next == NULL ) {
	  if ( fseek( sf->fd, 0, SEEK_END ) ) {
	       *error = SF_ERR_FILE_READ;
	       return( -1 );
	  }
	  if ( (end = ftell( sf->fd ))==-1L ) {
	       *error = SF_ERR_FILE_READ;
	       return( -1 );
	  }	       
     } else {
	  if ( ((SpecScan *)list->next->contents)->last_header !=
					     scan->last_header ) {
	       end = ((SpecScan *)list->next->contents)->last_header;
	  } else {
	       end = ((SpecScan *)list->next->contents)->offset;
	  }
     }
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, scan->offset, SEEK_SET ) ) {
	  *error = SF_ERR_FILE_READ;
	  return( -1 );
     }
     i = readHeaderLines( sf->fd, string, end, lines, error );
     
     return( i );
}
     
/*********************************************************************
 *   Function:		long SfNoHeaderBefore( sf, index, error )
 *
 *   Description:	Gets number of scan header lines before data.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Scan index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Number of scan header lines before data , 
 *			( -1 ) => errors.
 *   Possible errors:
 *			SF_ERR_SCAN_NOT_FOUND
 *
 *********************************************************************/
long
SfNoHeaderBefore( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList	       *list;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfNoHeaderBefore()\n");

     list = findScanByIndex( &(sf->list), index );
     if ( list != (ObjectList *)NULL ) {
	  setScanCurrent( sf, list );
	  return( ((SpecScan *)list->contents)->header_lines_before );
     }
     *error = SF_ERR_SCAN_NOT_FOUND;
     return( -1 );
}  

/*********************************************************************
 *   Function:	long readHeaderLines( fd, string, end, ret, error )
 *
 *   Description:	Gets all lines which begin with 'string'
 *			from file.
 *
 *   Parameters:
 *		Input :	(1) File pointer ( beginning of the search )  
 *			(2) String
 *			(3) end mark ( where to stop the search )
 *		Output:
 *			(4) Found lines
 *			(5) error number
 *   Returns:
 *			Number of found lines. 
 *			(-1) => error.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *			SF_ERR_FILE_READ	| => findLine()
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
readHeaderLines( fd, string, end, ret, error )

FILE	       *fd;
char	       *string;
long		end;
char	     ***ret;
int	       *error;
{
     char     **lines;
     char      *one_line;
     char      *string2;
     long	i = 0;
     long       stlen;


     if ( sf_debug ) 
       fprintf( stderr, "Entered readHeaderLines()\n");

     *ret = (char **)NULL;

     /*
      * Allocate memory for an array of strings
      */	 
     if ( (lines = (char **)malloc( sizeof(char *) )) == (char **)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return ( -1 );
     }     
     /*
      * Add '#' to 'string'.
      */	 
     if (string != (char *)NULL) {
         stlen = strlen(string);
     } else {
         stlen = 0;
     }
     string2 = (char *)malloc( (stlen+2) * sizeof(char) );
     if ( string2 == (char *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return ( -1 );
     }     
     if ( string2 == (char *)NULL ) return ( -1 );
     string2[0]='#';
     string2[1]='\0';
     if (string != (char *) NULL) strcat( string2, string );
     /*
      * Read line and add it to the array 
      */
     while ((one_line=findLine( fd, string2, end, error)) != (char *)NULL) {
	  lines = (char **)realloc( lines, (i+1) * sizeof(char *) );
	  if ( lines == (char **)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       free( string2 );
	       free( one_line ) ;
	       return ( -1 );
	  }
	  lines[i] = (char *)malloc( sizeof(char) * (strlen( one_line ) + 1) );
	  if ( lines[i] == (char *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       free( string2 );
	       free( one_line ) ;
	       freeArr( &lines, i );
	       return ( -1 );
	  }
	  strcpy( lines[i], one_line );
	  free( one_line );
	  i++;
     }
     free( string2 );

     if ( i > 0 ) *ret=lines;
     else free( lines );
     
     return( i );
     
}

/*********************************************************************
 *   Function:		long SfNoDataLines( sf, index, error )
 *
 *   Description:	Gets number of data lines in a scan
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Number of data lines , 
 *			( -1 ) => errors.
 *   Possible errors:
 *			SF_ERR_SCAN_NOT_FOUND
 *
 *********************************************************************/
long
SfNoDataLines( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList	       *list;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfNoDataLines()\n");

     list = findScanByIndex( &(sf->list), index );
     if ( list != (ObjectList *)NULL ) {
	  setScanCurrent( sf, list );
	  return( ((SpecScan *)list->contents)->data_lines );
     }
     *error = SF_ERR_SCAN_NOT_FOUND;
     return( -1 );
}  

/*********************************************************************
 *   Function:		int checkSameFile( sf, list )
 *
 *   Description:	Checks if the current scan file header and 
 *			the new scan file header are the same.
 *   Parameters:
 *		Input :	(1) SpecFile pointer   
 *			(2) New scan 
 *   Returns:
 *		1 - the same
 *		0 - not the same
 *
 *********************************************************************/
int
checkSameFile( sf, list )

SpecFile       	*sf;
ObjectList	*list;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered checkSameFile()\n");

     if (sf->current) {
     return ( ((SpecScan *)sf->current->contents)->last_header ==
	      ((SpecScan *)list->contents)->last_header  ); 
     } else return(1);
}

/*********************************************************************
 *   Function:		int checkSameScan( sf, index )
 *
 *   Description:	Checks if the current scan and 
 *			the new scan are the same.
 *   Parameters:
 *		Input :	(1) SpecFile pointer   
 *			(2) New scan index 
 *   Returns:
 *		1 - the same
 *		0 - not the same
 *
 *********************************************************************/
int
checkSameScan( sf, index )

SpecFile       	*sf;
long		 index;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered checkSameScan()\n");


     return ( ((SpecScan *)sf->current->contents)->index == index ); 
}

/*********************************************************************
 *   Function:		void setScanCurrent( sf, list )
 *
 *   Description:	Sets 'list' to current scan. 
 *			Updates SpecFile structure.
 *   Parameters:
 *		Input :	(1) SpecFile pointer   
 *			(2) New scan 
 *
 *********************************************************************/
void
setScanCurrent( sf, list )

SpecFile       	*sf;
ObjectList	*list;
{

     if ( sf_debug ) 
       fprintf( stderr, "Entered setScanCurrent()\n");

     if ( sf->current != list ) {
          if (sf->data_info != NULL) {
	    freeArrNZ( (void ***)&(sf->data), sf->data_info[ROW] );
	    free( sf->data_info );
	    sf->data_info	= (long *)NULL;
          }
	  
          if (sf->no_labels != -1) {
	     freeArrNZ( (void ***)&(sf->labels), sf->no_labels );
	     sf->no_labels	   = -1;
          }

          if (sf->no_motor_pos != -1) {
	     free( sf->motor_pos );
	     sf->motor_pos = (double *)NULL;
	     sf->no_motor_pos = -1;
          }

	  
	  if ( !checkSameFile( sf, list ) ) {

               if (sf->no_motor_names != -1)
	          freeArrNZ( (void ***)&(sf->motor_names), sf->no_motor_names );
	       sf->no_motor_names	= -1;
	       sf->epoch		= -1;
	       sf->header_offset = ((SpecScan *)list->contents)->last_header;
	  }
	  sf->current = list;
     }
}


/*********************************************************************
 *   Function:		double **cpyDblArr( data, info, error)
 *
 *   Description:	Copies an array of doubles.
 *
 *   Parameters:
 *		Input :	(1) Data array pointer   
 *			(2) Data info  pointer
 *		Output:
 *			(3) error number
 *   Returns:
 *			copy of the array of doubles ,
 *			NULL => errors.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
double **
cpyDblArr( data, info, error)

double	**data;
long	 *info;
int	 *error;
{
     int	i;
     double   **copy;
     

     if ( sf_debug ) 
       fprintf( stderr, "Entered cpyDblArr()\n");

     copy = (double **)malloc( sizeof(double *) * info[ROW] );
     if ( copy == (double **)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( copy );
     }
     for( i=0 ; i < info[ROW] ; i++ ) {
	  copy[i] = (double *)malloc( sizeof(double) * info[COL] );
	  if ( copy[i] == (double *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC; 
	       freeArr( &copy, i );
	       return( (double **)NULL );
	  }
	  memcpy( copy[i], data[i], sizeof(double) * info[COL] );
     }
     return( copy );
}	

/*********************************************************************
 *   Function:		int SfData(sf, index, data, data_info, error)
 *
 *   Description:	Gets data.
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) Data array
 *			(4) Data info : [0] => no_lines
 *					[1] => no_columns
 *					[2] = ( 0 ) => regular	
 *					      ( 1 ) => not regular ! 
 *			(5) error number
 *   Returns:
 *			(  0 ) => OK
 *		        ( -1 ) => errors occured
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *			SF_ERR_LINE_NOT_FOUND
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
int 
SfData( sf, index, data, data_info, error )
 
SpecFile       *sf;
long		index;
double	     ***data;
long	      **data_info;
int	       *error;
{
     ObjectList	       *list;
     SpecScan	       *scan;
     char	       *str;
     long		regular  = 0;
     long		no_lines = 0;
     long      		no_columns;
     long		last_columns = 0;
     long		i;
     long		position;
     double	       *one_line;
     double	      **new_data;
     int		err = 0;
 

     if ( sf_debug ) 
       fprintf( stderr, "Entered SfData()\n");

     *data = (double **)NULL;
     *data_info = (long *)malloc( sizeof(long) * D_INFO );
     if ( *data_info == (long *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( -1 );
     }
     /*
      * Check if data already read and copy.
      */
     if ( checkSameScan( sf, index ) && sf->data!=(double **)NULL &&
					sf->data_info!=(long *)NULL) {
	  
	  new_data = cpyDblArr( sf->data, sf->data_info, &err );
	  if ( err ) {
	       free( *data_info );
	       *error = err;
	       return( -1 );
	  }
	  for ( i=0 ; i<D_INFO ; i++ ) (*data_info)[i] = sf->data_info[i] ;
	  *data = new_data;
	  return( 0 );
     }
     /*
      * Find scan.
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  free( *data_info );
	  return( -1 );
     }
     setScanCurrent( sf, list );
     scan = list->contents;
     /*
      * Set file pointer position. 
      */
     if ( fseek( sf->fd, scan->data_offset, SEEK_SET ) ) {
	  if ( scan->data_lines < 1 ) *error = SF_ERR_LINE_NOT_FOUND;
	  else			      *error = SF_ERR_FILE_READ;
	  free( *data_info );
	  return( -1 );
     }
     /*
      * Alloc memory for data array.
      */
     new_data = (double **)malloc( sizeof(double *) * (scan->data_lines) );
     if ( new_data==(double **)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  free( *data_info );
	  return( -1 );
     }
     /*
      * Read all data lines and ...
      */
     while ( no_lines < scan->data_lines ) {
	  if( (str = readFileLine( sf->fd, error )) != (char *)NULL ) {
	       /*
		* ... covert into double.
		*/
	       no_columns = mulstrtod( str, &one_line, error );
	       if ( no_columns < 0 ) {
		    free( *data_info );
		    freeArr( &new_data, no_lines );
		    free( str );
		    return( -1 );
	       }

	       if ( no_columns ) {
		    if ( no_lines == 0 ) last_columns = no_columns;
		    else { 
			 if ( no_columns != last_columns ) {
			      regular = 1;
			      last_columns = (no_columns > last_columns) ? 
			      no_columns:last_columns;
			 }
		    }

		    /*
		     * If scan was aborted and current line is not regular and 
		     * if the next line == last one and if it contents '#' 
		     * (probably #C ) than do not read it.
		     */
		    if ( regular ) {
			 
			 if ( (position = ftell( sf->fd )) == -1L ) {
			      *error = SF_ERR_FILE_READ;
			      return( -1 );
			 }	       
			 if ( position == scan->last_data_offset ) { 
			      free( str );
			      if ( (str=readFileLine(sf->fd,error))!=(char *)NULL) {
				   
				   if ( (strchr(str,(int)'#')) != (char *)NULL ) {
					
					free( one_line );
					free( str );
					break;
					
				   } else {
					if ( fseek( sf->fd, position, SEEK_SET ) ) {
					     *error = SF_ERR_FILE_READ;
					     return( -1 );
					}
				   }
			      }
			 }
		    }
		    /*
		     * Alloc memory for one data line more ...
		     */
		    new_data[no_lines]=(double *)
					malloc(sizeof(double) * no_columns );
		    if ( new_data[no_lines]==(double *)NULL ) {
			 *error = SF_ERR_MEMORY_ALLOC; 
			 free( *data_info );
			 freeArr( &new_data, no_lines );
			 free( str );
			 return( -1 );
		    }
		    /*
		     * copy new data line to array.
		     */
		    memcpy( new_data[no_lines], one_line, sizeof(double) * no_columns );

		    free( one_line );
		    no_lines++;
	       } 
	       free( str );
	  } else {
	       free( *data_info );
	       freeArr( &new_data, no_lines );
	       return( -1 );
	  }
     }
     (*data_info)[ROW]	= no_lines;
     (*data_info)[COL]	= last_columns; 
     (*data_info)[REG]	= regular;
     /*
      * Make a copy for SpecFile structure 
      */	 
     sf->data = cpyDblArr( new_data, *data_info, &err) ;
     if ( err ) {
	  free( *data_info );
	  freeArr( &new_data, no_lines );
	  *error = err;
	  return( -1 );
     }
     sf->data_info = (long *)malloc( sizeof(long) * D_INFO );
     if ( sf->data_info == (long *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  freeArr( &new_data, no_lines );
	  free( *data_info );
	  return( -1 );
     } 
     for ( i=0 ; i<D_INFO ; i++ )  sf->data_info[i] = (*data_info)[i] ;

     *data = new_data;
     return( 0 );
}

/*********************************************************************
 *   Function:		long SfDataAsString(sf, index, data, error)
 *
 *   Description:	Gets all scan data lines.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) Data lines array
 *			(4) error number
 *   Returns:
 *			Number of read lines.
 *		        ( -1 ) => errors occured
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *			SF_ERR_LINE_NOT_FOUND
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfDataAsString( sf, index, data, error )
 
SpecFile       *sf;
long		index;
char	     ***data;
int	       *error;
{
     ObjectList	       *list;
     SpecScan	       *scan;
     char	       *str;
     long		no_lines = 0;
     char	      **new_data;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfDataAsString()\n");
     
     *data = (char **)NULL;
     /*
      * Find scan.
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( -1 );
     }
     setScanCurrent( sf, list );
     scan = list->contents;
     /*
      * Set file pointer position. 
      */
     if ( fseek( sf->fd, scan->data_offset, SEEK_SET ) ) {
	  if ( scan->data_lines < 1 ) *error = SF_ERR_LINE_NOT_FOUND;
	  else			      *error = SF_ERR_FILE_READ;
	  return( -1 );
     }
     /*
      * Alloc memory for data array.
      */
     new_data = (char **)malloc( sizeof(char *) * scan->data_lines );
     if ( new_data==(char **)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( -1 );
     }
     /*
      * Read all data lines.
      */
     while ( no_lines < scan->data_lines ) {
	  if( (str = readFileLine( sf->fd, error )) != (char *)NULL ) {
	       /*
		* Alloc memory for one data line more ...
		*/
 	       new_data[no_lines]=(char *)
				  malloc(sizeof(char) * (strlen(str)+1));
	       if ( new_data[no_lines]==(char *)NULL ) {
		    *error = SF_ERR_MEMORY_ALLOC; 
		    freeArr( &new_data, no_lines );
		    free( str );
		    return( -1 );
	       }
	       /*
		* copy new data line to array.
		*/
	       memcpy( new_data[no_lines], str, sizeof(char)*(strlen(str)+1) );
	       free( str );
	       no_lines++;
	  } else {
	       freeArr( &new_data, no_lines );
	       return( -1 );
	  }
     }
     *data = new_data;
     return( no_lines );
}

/*********************************************************************
 *   Function:		long SfDataLine( sf, index, line, data, error )
 *
 *   Description:	Gets single data line from a scan.
 *
 *   Parameters:
 *		Input:	(1) SpecFile pointer   
 *			(2) Scan index
 *			(3) Line number
 *		Output:	(4) Data line (array of doubles)
 *			(5) Error number
 *   Returns:
 *			number of data values in the line , 
 *			( -1 ) if  errors occured .
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *			SF_ERR_LINE_NOT_FOUND
 *			SF_ERR_FILE_READ	| => SfData()
 *			SF_ERR_SCAN_NOT_FOUND	|
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfDataLine( sf, index, line, data_line, error )
     
SpecFile       *sf;
long		index;
long		line;
double	      **data_line;
int	       *error;
{
     double    *one_line;
     double   **data;
     long      *data_info;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfDataLine()\n");

     *data_line = (double *)NULL;
     /*
      * Check if data already read.
      */
     if ( !checkSameScan( sf, index ) || sf->data==(double **)NULL || 
					 sf->data_info==(long *)NULL ) {

	  if ( SfData(sf, index, &data, &data_info, error)) return( -1 );
	  freeArrNZ( (void ***)&data, data_info[ROW] );
	  free( data_info );
     }
     /*
      *  Check if line exists.
      */
     if ( abs(line) > sf->data_info[ROW] || !line ) {
	  *error = SF_ERR_LINE_NOT_FOUND;
	  return( -1 );
     }
     if ( line < 0 ) {
	  line = sf->data_info[ROW] + line + 1 ;
     }
     /*
      *  Copy one data line.
      */
     one_line = (double *)malloc( sizeof(double) * sf->data_info[COL] );
     if ( one_line == (double *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( -1 );
     }
     memcpy(one_line, sf->data[line-1], sizeof(double) * sf->data_info[COL]);
     
     *data_line = one_line;
     return( sf->data_info[COL] );
}

/*********************************************************************
 *   Function:		long SfDataCol( sf, index, col, data_col, error )
 *
 *   Description:	Gets single data column from a scan.
 *
 *   Parameters:
 *		Input:	(1) SpecFile pointer   
 *			(2) Scan index
 *			(3) Column number
 *		Output:	(4) Data column
 *			(5) Error number
 *   Returns:
 *			number of data values in the column , 
 *			( -1 ) if  errors occured .
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *			SF_ERR_COL_NOT_FOUND
 *			SF_ERR_FILE_READ	| => SfData()
 *			SF_ERR_SCAN_NOT_FOUND	|
 *			SF_ERR_LINE_NOT_FOUND	|
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfDataCol( sf, index, col, data_col, error )
     
SpecFile       *sf;
long		index;
long		col;
double	      **data_col;
int	       *error;
{
     double    *one_col;
     double   **data;
     long      *data_info;
     long	i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfDataCol()\n");

     *data_col = (double *)NULL;
     /*
      * Check if data already read.
      */
     if ( !checkSameScan( sf, index ) || sf->data==(double **)NULL || 
					 sf->data_info==(long *)NULL ) {

	  if ( SfData(sf, index, &data, &data_info, error)) return( -1 );
	  freeArrNZ( (void ***)&data, data_info[ROW] );
	  free( data_info );
     }
     /*
      *  Check if column exists.
      */
     if ( abs(col) > sf->data_info[COL] || !col ) {
	  *error = SF_ERR_COL_NOT_FOUND;
	  return( -1 );
     }
     if ( col < 0 ) {
	  col = sf->data_info[COL] + col + 1 ;
     }
     /*
      *  Copy one data column.
      */
     one_col = (double *)malloc( sizeof(double) * sf->data_info[ROW] );
     if ( one_col == (double *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( -1 );
     }
     for ( i=0 ; i < sf->data_info[ROW] ; i++ )
       one_col[i] = sf->data[i][col-1];
     
     *data_col = one_col;
     return( sf->data_info[ROW] );
}

/*********************************************************************
 *   Function:    long  SfDataColByName( sf, index, label, data_col, error )
 *
 *   Description:	Gets single data column from a scan.
 *
 *   Parameters:
 *		Input:	(1) SpecFile pointer   
 *			(2) Scan index
 *			(3) Column label
 *		Output:	(4) Data column
 *			(5) Error number
 *   Returns:
 *			number of data values in the column , 
 *			( -1 ) if  errors occured .
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	||
 *			SF_ERR_FILE_READ	|| => SfDataCol()
 *			SF_ERR_LINE_NOT_FOUND	||
 *			SF_ERR_LINE_EMPTY	|  => SfAllLabels()
 *			SF_ERR_SCAN_NOT_FOUND	
 *			SF_ERR_COL_NOT_FOUND
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfDataColByName( sf, index, label, data_col, error )
     
SpecFile       *sf;
long		index;
char	       *label;
double	      **data_col;
int	       *error;
{
     ObjectList	 *scan;
     long	  i,j;
     int	  found = 0;
     long	  no_labels;
     char	**labels;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfDataColByName()\n");

     *data_col = (double *)NULL;

     scan = findScanByIndex( &(sf->list), index );
     if ( scan == (ObjectList *)NULL ) {
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( -1 );
     }
     /*
      * Check if labels already read.
      */
     if ( !checkSameScan( sf, scan ) || sf->labels==(char **)NULL ||
					sf->no_labels < 1 ) {

	  no_labels = SfAllLabels( sf, index, &labels, error );
	  if ( !no_labels ) return( -1 );
	  freeArrNZ( (void ***)&labels, no_labels );
     }
     /*
      *  Get index of the column.
      */
     for ( i=0 ; i < sf->no_labels && !found ; i++ ) { 
	  found = 1;
	  for ( j=0 ; label[j]!='\n' && label[j]!='\0' ; j++ ) {
	       if ( label[j] != sf->labels[i][j] ) {
		    found = 0;
		    break;
	       }
	  }
     }

     if ( !found ) {
	  *error = SF_ERR_COL_NOT_FOUND;
	  return( -1 );
     }
     /*
      *  Get searched position.
      */     
     return( SfDataCol( sf, index, i, data_col, error ) );
}

/*********************************************************************
 *   Function:		char **cpyStrArr( arr, lines, error)
 *
 *   Description:	Copies a string array.
 *
 *   Parameters:
 *		Input :	(1) String array pointer   
 *			(2) No. of lines
 *		Output:
 *			(3) Error number
 *   Returns:
 *			copy of string array ,
 *			NULL => errors.
 *
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char **
cpyStrArr( arr, lines, error )

char	**arr;
long	  lines;
int	 *error;
{
     int	i;
     char     **copy;
     

     if ( sf_debug ) 
       fprintf( stderr, "Entered cpyStrArr()\n");

     copy = (char **)malloc( sizeof(char *) * lines );
     if ( copy == (char **)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( copy );
     }
     for( i=0 ; i < lines ; i++ ) {
	  copy[i] = (char *)malloc( sizeof(char) * ( 1 + strlen(arr[i]) ) );
	  if ( copy[i] == (char *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC; 
	       freeArr( &copy, i );
	       return( (char **)NULL );
	  }
	  memcpy( copy[i], arr[i], sizeof(char) * ( 1 + strlen(arr[i]) ) );
     }
     return( copy );
}	

/*********************************************************************
 *   Function:		freePtr( ptr );
 *			       
 *   Description:	Frees memory pointed to by 'ptr'.
 *
 *   Parameters:
 *		Input :	(1) Pointer   
 *
 *********************************************************************/
void
freePtr( ptr )

void	*ptr;
{
     free( ptr );
}

/*********************************************************************
 *   Function:		freeArrNZ( ptr, lines );
 *			       
 *   Description:	Frees an array if 'lines' > zero.
 *
 *   Parameters:
 *		Input :	(1) Array pointer   
 *			(2) No. of lines
 *
 *********************************************************************/
void
freeArrNZ( ptr, lines )

void   ***ptr;
long	  lines;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered freeArrNZ()\n");
     

     if ( *ptr != (void **)NULL  &&  lines > 0 ) {
	  for ( ; lines ; lines-- ) {
	       free( (*ptr)[lines-1] );
	  }
	  free( *ptr );
	  *ptr = ( void **)NULL ;
     }
}

/*********************************************************************
 *   Function:		freeArr( ptr, lines );
 *
 *   Description:	Frees an array.
 *			 'prt' will be always freed !!!
 *
 *   Parameters:
 *		Input :	(1) Array pointer   
 *			(2) No. of lines
 *
 *********************************************************************/
void
freeArr( ptr, lines )

void   ***ptr;
long	  lines;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered freeArr()\n");

     if ( *ptr != (void **)NULL ) {
	  if ( lines > 0 ) {
	       for ( ; lines ; lines-- ) {
		    free( (*ptr)[lines-1] );
	       }
	  }
	  free( *ptr );
	  *ptr = ( void **)NULL ;
     }
}

/*********************************************************************
 *   Function:		long lines2words( lines, no_lines, words, error )
 *
 *   Description:	Makes an array of words from an array of lines
 *			( words are separated by TWO blanks )
 *   Parameters:
 *		Input :	(1) Array of lines
 *			(2) Number of lines
 *		Output:	(3) Array of words
 *			(4) Error number 
 *   Returns:
 *			Number of words,
 *			( -1 ) => errors. 			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
lines2words( lines, no_lines, words, error )

char	      **lines;
long		no_lines;
char	     ***words;
int	       *error;
{
     char      *tmp;
     char     **word;
     char      *c;
     long	i;
     long	j;
     long	no_words=0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered lines2words()\n");

     *words = (char **)NULL;

     if ( no_lines < 1 ) return( -1 );
     /*
      * Alloc memory for array pointer.
      */
     word = (char **)malloc( sizeof(char *) );
     if (  word == (char **)NULL ) {
	  *error =  SF_ERR_MEMORY_ALLOC;
	  return( -1 );
     }
     /*
      * Create an array of words 
      */	 
     for ( i=0 ; i < no_lines ; i++ ) {

	  c = lines[i];

	  /*
	   * !!!! Delete first word in the line and blanks after it !!!!! 
	   */	 
	  while ( *c!=' ' && *c!='\t' && *c!='\n' && *c!='\0' ) c++;
	  while ( *c == ' ' || *c == '\t' ) c++; 

	  tmp = (char *)malloc( sizeof(char) * (strlen( c )+1) );
	  if ( tmp == (char *)NULL ) {
	       freeArr( &word, no_words );
	       *error =  SF_ERR_MEMORY_ALLOC;
	       return( -1 );
	  }
          while ( *c!='\n' &&  *c!='\0' ) {
	       /*
		* Copy word. 
		*/	 
	       for (j=0 ; *c!='\t' && *c!='\n' && *c!='\0' ; j++) {
		    if ( *c==' ' ) {
			 if ( *(++c)==' ' ) {
			      break;
			 } else c--;
		    }
     		    tmp[j] = *c;
		    c++;
	       }
	       tmp[j]='\0';
	       /*
		* Delete blanks. 
		*/	 
	       while ( *c == ' ' || *c == '\t' ) c++; 
	       /*
		* Add word to the array. 
		*/	 
	       word = (char **)realloc( word, sizeof(char *) * (no_words+1) );
	       if ( word == (char **)NULL ) {
		    *error = SF_ERR_MEMORY_ALLOC;
		    return( -1 );
	       }
	       word[no_words] = (char *)malloc( (j+1)*sizeof(char) );
	       if ( word[no_words] == (char *)NULL ) {
		    freeArr( &word, no_words );
		    *error =  SF_ERR_MEMORY_ALLOC;
		    return( -1 );
	       }
	       strcpy( word[no_words], tmp );
               no_words++;
	  }
	  free( tmp );
     }

     if ( no_words < 1 ) {
	  free( word );
	  word = (char **)NULL;
     }
     *words = word;
     return( no_words );
}

/*********************************************************************
 *   Function:		long SfAllLabels( sf, index, labels, error )
 *
 *   Description:	Reads all labels in #L lines
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *		Output:	(3) Labels  
 *			(4) Error number 
 *   Returns:
 *			Number of labels
 *			( -1 ) if error.		
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	||=> cpyStrArr(),lines2words()
 *			SF_ERR_SCAN_NOT_FOUND	| => SfHeader()
 *			SF_ERR_FILE_READ	|
 *			SF_ERR_LINE_EMPTY
 *			SF_ERR_LINE_NOT_FOUND
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfAllLabels( sf, index, labels, error )

SpecFile       *sf;
long		index;
char	     ***labels;
int	       *error;
{
     char     **lines;
     long	no_lines  = 0;
     long	no_labels = 0;
     char       string[] = " \0";


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfAllLabels()\n");

     *labels = (char **)NULL;
     /*
      * Check if labels already read.
      */
     if ( checkSameScan( sf, index )  && sf->labels != (char **)NULL  && 
					 sf->no_labels > 0 ) {
	  /*
	   * Make a copy.
	   */
	  *labels = cpyStrArr( sf->labels, sf->no_labels, error );
	  if ( *labels == (char **)NULL ) return ( -1 );

	  return( sf->no_labels );
     }
     /*
      * Read every '#SF_LABEL' line 
      */	
     string[0]= SF_LABEL;
     no_lines = SfHeader( sf, index, string, &lines, error );
     if ( no_lines < 1 ) {
	  if ( !no_lines ) *error = SF_ERR_LINE_NOT_FOUND; 
	  return( -1 );
     }
     /*
      * Transform into words
      */
     no_labels = lines2words( lines, no_lines, labels, error );
     freeArrNZ( (void ***)&lines, no_lines );
     if ( no_labels < 1 ) {
	  if ( !no_labels) *error = SF_ERR_LINE_EMPTY;
	  return( -1 );
     }
     /*
      * Make a copy for SpecFile structure 
      */	 
     sf->labels = cpyStrArr( *labels, no_labels, error );
     if ( sf->labels == (char **)NULL ) {
	  freeArr( labels, no_labels );
	  return( -1 );
     }
     sf->no_labels = no_labels;
     return( no_labels );
}

/*********************************************************************
 *   Function:		char *getStrFromArr( array, size, number, error )
 *
 *   Description:	Reads one string from array of strings.
 *
 *   Parameters:
 *		Input :	(1) Array pointer
 *			(2) Total number of elements in the array
 *			(3) Number of wanted element
 *		Output:	(4) Error number 
 *   Returns:
 *			Pointer to the copy of the searched string,
 *			NULL if errors occured .
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
getStrFromArr( array, size, number, error )
     
char	      **array;
long		size;
long		number;
int	       *error;
{
     char	 *str ;


     if ( sf_debug ) 
       fprintf( stderr, "Entered getStrFromArr()\n");

     /*
      *  Check if element exists.
      */
     if ( abs(number) > size || !number ) return( (char *)NULL );

     if ( number < 0 )  number += size + 1 ;
     /*
      *  Copy .
      */
     str = (char *)malloc( sizeof(char) * (1+strlen(array[number-1])) );
     if ( str == (char *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC; 
	  return( (char *)NULL );
     }
     memcpy(str, array[number-1], sizeof(char) * (1+strlen(array[number-1])) );
     return( str );
}

/*********************************************************************
 *   Function:		char *SfLabel( sf, index, column, error )
 *
 *   Description:	Reads one label.
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *			(3) Column number
 *		Output:	(4) Error number 
 *   Returns:
 *			Pointer to the label ,
 *			or NULL if errors occured.			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => getStrFromArr()
 *			SF_ERR_LABEL_NOT_FOUND
 *			SF_ERR_LINE_EMPTY	|
 *			SF_ERR_LINE_NOT_FOUND	|
 *			SF_ERR_SCAN_NOT_FOUND	| => SfAllLabels()
 *			SF_ERR_FILE_READ	|
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
SfLabel( sf, index, column, error )
     
SpecFile       *sf;
long		index;
long		column;
int	       *error;
{
     char	**labels;
     char	 *label ;
     long	  no_labels;
     int	  err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfLabel()\n");

     /*
      * Check if labels already read.
      */
     if ( !checkSameScan( sf, index ) || sf->labels==(char **)NULL || 
					 sf->no_labels < 1 ) {

	  no_labels = SfAllLabels( sf, index, &labels, error );
	  if ( no_labels < 1) return( (char *)NULL );
	  freeArrNZ( (void ***)&labels, no_labels );
     }
     /*
      *  Get searched element from array.
      */
     label = getStrFromArr( sf->labels, sf->no_labels, column, &err );
     if ( label == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LABEL_NOT_FOUND; 
	  else *error = err;
     }
     return( label );
}

/*********************************************************************
 *   Function:		long SfAllMotors( sf, index, names, error )
 *
 *   Description:	Reads all motor names in #O lines (in file header)
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *		Output:	(3) Names  
 *			(4) Error number 
 *   Returns:
 *			Number of found names
 *			( -1 ) if errors.		
 *   Possible errors:
 *			SF_ERR_SCAN_NOT_FOUND	  
 *			SF_ERR_LINE_NOT_FOUND	
 *			SF_ERR_LINE_EMPTY	
 *			SF_ERR_MEMORY_ALLOC    || => cpyStrArr(),lines2words()
 *			SF_ERR_FILE_READ	|
 *			SF_ERR_HEADER_NOT_FOUND	| => SfFileHeader()
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfAllMotors( sf, index, names, error )

SpecFile       *sf;
long		index;
char	     ***names;
int	       *error;
{
     ObjectList	*list;
     char      **lines;
     long	 no_lines = 0;
     long	 no_names = 0;
     char        string[] = " \0";


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfAllMotors()\n");

     *names = ( char **)NULL;

     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) {
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( -1 );
     }
     /*
      * Check if already read.
      */
     if ( checkSameFile( sf, list ) &&  sf->motor_names != (char **)NULL && 
					sf->no_motor_names > 0) {
	  /*
	   * Make a copy.
	   */
	  setScanCurrent( sf, list );
	  *names = cpyStrArr( sf->motor_names, sf->no_motor_names, error );
	  if ( *names == (char **)NULL ) return( -1 );

	  return( sf->no_motor_names );
     }
     /*
      * Read every '#SF_MOTOR_NAMES' line 
      */	
     string[0]= SF_MOTOR_NAMES;
     no_lines = SfFileHeader( sf, index, string, &lines, error );
     if ( no_lines < 1 ) {
	  if ( !no_lines ) *error = SF_ERR_LINE_NOT_FOUND; 
	  return( -1 );
     }
     /*
      * Transform into words
      */
     no_names = lines2words( lines, no_lines, names, error );
     freeArrNZ( (void ***)&lines, no_lines );
     if ( no_names < 1 ) {
	  if ( !no_names ) *error = SF_ERR_LINE_EMPTY;
	  return( -1 );
     }
     /*
      * Make a copy for SpecFile structure 
      */	 
     sf->motor_names = cpyStrArr( *names, no_names, error );
     if ( sf->motor_names == (char **)NULL ) {
	  freeArr( names, no_names );
	  return( -1 );
     }
     sf->no_motor_names = no_names;
     return( no_names );
}

/*********************************************************************
 *   Function:		char *SfMotor( sf, index, number, error )
 *
 *   Description:	Reads name of the motor specified by number.
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *			(3) Motor number
 *		Output:	(4) Error number 
 *   Returns:
 *			Pointer to the name , 
 *			NULL if errors occured or not found.	
 *   Possible errors:
 *			SF_ERR_SCAN_NOT_FOUND	
 *			SF_ERR_MOTOR_NOT_FOUND
 *			SF_ERR_HEADER_NOT_FOUND	|
 *			SF_ERR_LINE_EMPTY	|
 *			SF_ERR_LINE_NOT_FOUND	| => SfAllMotors()
 *			SF_ERR_FILE_READ	|
 *			SF_ERR_MEMORY_ALLOC    || => getStrFromArr()
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
char *
SfMotor( sf, index, number, error )
     
SpecFile       *sf;
long		index;
long		number;
int	       *error;
{
     ObjectList	 *scan;
     char	**names;
     char	 *name ;
     long	  no_names;
     int	  err = 0;

     if ( sf_debug ) 
       fprintf( stderr, "Entered SfMotor()\n");

     scan = findScanByIndex( &(sf->list), index );

     if ( scan == (ObjectList *)NULL ) {
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( (char *)NULL );
     }
     /*
      * Check if names already read.
      */
     if ( !checkSameFile( sf, scan ) || sf->motor_names==(char **)NULL || 
					sf->no_motor_names < 1 ) {

	  no_names = SfAllMotors( sf, index, &names, error );
	  if ( no_names < 1 ) return( (char *)NULL );
	  freeArrNZ( (void ***)&names, no_names );
     }
     /*
      *  Get searched element from array.
      */
     setScanCurrent( sf, scan );
     name = getStrFromArr(sf->motor_names, sf->no_motor_names, number, &err );
     if ( name == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_MOTOR_NOT_FOUND; 
	  else *error = err;
     }
     return( name );
}

/*********************************************************************
 *   Function:		long SfAllMotorPos( sf, index, pos, error )
 *
 *   Description:	Reads motor positions ( #P lines ).
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *		Output:	(3) Motor positions
 *			(4) Error number 
 *   Returns:
 *			Number of positions
 *			( -1 ) if error.		
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	
 *			SF_ERR_LINE_EMPTY
 *			SF_ERR_LINE_NOT_FOUND
 *			SF_ERR_SCAN_NOT_FOUND	| => SfHeader()
 *			SF_ERR_FILE_READ	|
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfAllMotorPos( sf, index, pos, error )

SpecFile       *sf;
long		index;
double	      **pos;
int	       *error;
{
     char     **lines;
     char      *c;
     char      *tmp;
     char       string[] = " \0";
     long	i;
     long	no_lines = 0;
     long	no_positions = 0;
     double    *positions ;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfAllMotorPos()\n");

     *pos = (double *)NULL;
     /*
      * Check if motor_pos already read.
      */
     if ( checkSameScan( sf, index ) && sf->motor_pos != (double *)NULL && 
					sf->no_motor_pos > 0 ) {
	  /*
	   * Make a copy.
	   */
	  positions = (double *)malloc( sizeof(double) * sf->no_motor_pos );
	  if ( positions == (double *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       return( -1 );
	  }
	  memcpy( positions, sf->motor_pos, sizeof(double) * sf->no_motor_pos);
	  *pos = positions;
	  return( sf->no_motor_pos );
     }
     /*
      * Read every '#SF_MOTOR_POSITIONS' line 
      */	
     string[0]= SF_MOTOR_POSITIONS;
     no_lines = SfHeader( sf, index, string, &lines, error );
     if ( no_lines < 1 ) {
	  if ( !no_lines ) *error = SF_ERR_LINE_NOT_FOUND; 
	  return( -1 );
     }

     /*
      * Make one line from many.
      */
     tmp = (char *)malloc( sizeof(char) );
    *tmp = '\0';

     for ( i=0 ; i < no_lines ; i++ ) {
	  c = lines[i];
	  /*
	   * Delete first word in the line and blanks after it . 
	   */	 
	  while ( *c!=' ' && *c!='\t' && *c!='\n' && *c!='\0' ) c++;

	  tmp = (char *)realloc( tmp, sizeof(char)*(strlen(c)+strlen(tmp)+1) );
	  tmp = strcat( tmp, c  );
     }

     /*
      * Transform into an array of doubles.
      */
     no_positions = mulstrtod( tmp, &positions, error );
     free( tmp );
     freeArrNZ( (void ***)&lines, no_lines );
     if ( no_positions < 1 ) {
	  *error = SF_ERR_LINE_EMPTY;
	  return( -1 ); 
     }

     /*
      * Make a copy for SpecFile structure 
      */	 
     sf->motor_pos = malloc( sizeof(double) * no_positions );
     if ( sf->motor_pos == (double *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  freeArr( &positions, no_positions );
	  return( -1 );
     }
     memcpy( sf->motor_pos, positions, sizeof(double) * no_positions );

     sf->no_motor_pos = no_positions;
     *pos = positions;

     return( no_positions );
}

/*********************************************************************
 *   Function:		double SfMotorPos( sf, index, number, error )
 *
 *   Description:	Reads one motor position.
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *			(3) Motor number
 *		Output:	(4) Error number 
 *   Returns:
 *			Motor position,
 *			( +HUGE_VAL ) in case of errors.
 *   Possible errors:
 *			SF_ERR_POSITION_NOT_FOUND
 *			SF_ERR_MEMORY_ALLOC	|
 *			SF_ERR_LINE_EMPTY	|
 *			SF_ERR_LINE_NOT_FOUND	| => SfAllMotorPos()
 *			SF_ERR_SCAN_NOT_FOUND	| 
 *			SF_ERR_FILE_READ	|
 *
 *********************************************************************/
double
SfMotorPos( sf, index, number, error )
     
SpecFile       *sf;
long		index;
long		number;
int	       *error;
{
     double	 *pos;
     long	  no_positions;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfMotorPos()\n");

     /*
      * Check if positions already read.
      */
     if ( !checkSameScan( sf, index ) || sf->motor_pos==(double *)NULL ||
					 sf->no_motor_pos < 1 ) {
          printf("New scan\n");

	  no_positions = SfAllMotorPos( sf, index, &pos, error );
	  if ( no_positions < 1 ) return( +HUGE_VAL );
	  free( pos );
     }
     /*
      *  Check if element exists.
      */
     if ( abs(number) > sf->no_motor_pos || !number ) {
	  *error = SF_ERR_POSITION_NOT_FOUND;
	  return( +HUGE_VAL );
     }
     if ( number < 0 )  number += sf->no_motor_pos + 1 ;
     /*
      *  Get searched element from array.
      */     
     return( sf->motor_pos[number-1] );
}

/*********************************************************************
 *   Function:		double SfMotorPosByName( sf,index,name,error )
 *
 *   Description:	Reads one motor position.
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *			(3) Motor name
 *		Output:	(4) Error number 
 *   Returns:
 *			Motor position.			
 *			( +HUGE_VAL ) in case of errors.
 *   Possible errors:
 *			SF_ERR_POSITION_NOT_FOUND
 *			SF_ERR_SCAN_NOT_FOUND	
 *			SF_ERR_LINE_EMPTY	| => SfAllMotorPos()
 *			SF_ERR_MEMORY_ALLOC	|
 *			SF_ERR_LINE_NOT_FOUND	| => SfAllMotors()
 *			SF_ERR_FILE_READ	|
 *
 *********************************************************************/
double
SfMotorPosByName( sf, index, name, error )
     
SpecFile       *sf;
long		index;
char	       *name;
int	       *error;
{
     ObjectList	 *scan;
     long	  i,j;
     int	  found = 0;
     long	  no_names;
     char	**names;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfMotorPosByName()\n");

     scan = findScanByIndex( &(sf->list), index );

     if ( scan == (ObjectList *)NULL ) {
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( +HUGE_VAL );
     }
     /*
      * Check if motor names already read.
      */
     if ( !checkSameFile( sf, scan ) || sf->motor_names==(char **)NULL ||
					sf->no_motor_names < 1 ) {

	  no_names = SfAllMotors( sf, index, &names, error );
	  if ( no_names < 1 ) return( +HUGE_VAL );
	  freeArrNZ( (void ***)&names, no_names );
     }
     /*
      *  Get index of the element.
      */
     setScanCurrent( sf, scan );
     for ( i=0 ; i < sf->no_motor_names && !found ; i++ ) { 
	  found = 1;
	  for ( j=0 ; name[j]!='\n' && name[j]!='\0' ; j++ ) {
	       if ( name[j] != sf->motor_names[i][j] ) {
		    found = 0;
		    break;
	       }
	  }
     }

     if ( !found ) {
	  *error = SF_ERR_POSITION_NOT_FOUND;
	  return( +HUGE_VAL );
     }
     /*
      *  Get searched position.
      */     
     printf("Getting position for motor %d in scan %d\n",i,index);
     return( SfMotorPos( sf, index, i, error ) );
}

/*********************************************************************
 *   Function:		char SfFileDate( sf, index, error )
 *
 *   Description:	Gets date from the last file header
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Date.(From #D line !)
 *			NULL => errors.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   | => readHeader()
 *			SF_ERR_LINE_NOT_FOUND 
 *			SF_ERR_LINE_EMPTY
 *			SF_ERR_FILE_READ
 *			SF_ERR_HEADER_NOT_FOUND
 *			SF_ERR_SCAN_NOT_FOUND
 *
 *********************************************************************/
char * 
SfFileDate( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     char		*date;
     long		 start;
     long		 end;
     int		 err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfFileDate()\n");

     /*
      * Find scan , set to current
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( (char *)NULL  );
     }

     scan = list->contents;
     setScanCurrent( sf, list );

     if ( (start = scan->last_header) < 0 ) {
	  *error = SF_ERR_HEADER_NOT_FOUND ;
	  return( (char *)NULL );
     }
     end = findFirstInFile( list );
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, start, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( (char *)NULL );
     }
     /*
      * Read line containing date.
      */	 
     date = readHeader( sf->fd, SF_DATE, end, &err );
     if ( date == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LINE_NOT_FOUND;
	  else *error = err;
	  return( (char *)NULL );
     }
     if ( *date=='\n' ) {
	  *error = SF_ERR_LINE_EMPTY;
	  free( date );
	  return( (char *)NULL );
     }
     return( date );
}

/*********************************************************************
 *   Function:		long findFirstInFile( current )
 *
 *   Description:	Looks for the first scan after the file header. 
 *			( or first scan in the file if no headers before )
 *   Parameters:
 *			(1) Poiter to current object.
 *   Returns:
 *			Scan offset.
 *
 *********************************************************************/
long 
findFirstInFile( current )

ObjectList	*current;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered findFirstInFile()\n");

     while ( current->prev != (ObjectList *)NULL &&
	     ((SpecScan *)current->contents)->last_header ==
	     ((SpecScan *)current->prev->contents)->last_header ) {
	  current = current->prev;
     }
     return( ((SpecScan *)current->contents)->offset );
}

/*********************************************************************
 *   Function:		char *findWordInLine( line, word, error )
 *
 *   Description:	Looks for 'word' in given line and returns a 
 *		       	copy of the rest of the line after the found word .
 *
 *   Parameters:
 *		Input :	(1) Line pointer   
 *			(2) Word pointer
 *		Output:
 *			(3) error number
 *   Returns:
 *			Rest of the line after word.
 *			NULL => not found.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   
 *
 *********************************************************************/
char *
findWordInLine( line, word, error )

char	*line;
char	*word;
int	*error;
{
     char	*ret;
     int	 found = 0;
 

     if ( sf_debug ) 
       fprintf( stderr, "Entered findWordInLine()\n");

     /*
      * Find word. 
      */	 
/*     for ( ; *line!='\0' && *line!='\n' && !found ; line++ ) {
	  for ( i=0 ; word[i]!='\0' && word[i]!='\n' ; i++ ) {
	       if ( *line != word[i] ) {
		    found = 0;
		    break;
	       }
	       line++;
	       found = 1;
	  }
     }
     if ( !found ) {
	  *error = SF_ERR_WORD_NOT_FOUND;
	  return( (char *)NULL );
     }
*/

     line = strstr( line, word ); 
     if ( line == (char *)NULL ) {
	  return( line );
     }
     line += strlen( word );

     /*
      * Delete blanks. 
      */	 
     while ( *line == ' ' || *line == '\t' ) line++; 

     /*
      * Copy the rest. 
      */	 
     ret = (char *)malloc( sizeof(char) * ( 1 + strlen( line )) );
     if ( ret == (char *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return( ret );
     }
     memcpy( ret, line, sizeof(char) * ( 1 + strlen( line )) );

     return( ret );
}

/*********************************************************************
 *   Function:		long readFirstFileC( sf, index, error )
 *
 *   Description:	Gets first comment line from last file header.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *		        #C line.
 *			NULL if errors or not found.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC   | => readHeader()
 *			SF_ERR_LINE_NOT_FOUND  
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *			SF_ERR_HEADER_NOT_FOUND
 *
 *********************************************************************/
char * 
readFirstFileC( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     char		*line;
     long		 start;
     long		 end;
     int		 err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered readFirstFileC()\n");

     /*
      * Find scan , set to current
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( (char *)NULL  );
     }

     scan = list->contents;
     setScanCurrent( sf, list );

     if ( (start = scan->last_header) < 0 ) {
	  *error = SF_ERR_HEADER_NOT_FOUND ;
	  return( (char *)NULL );
     }
     end = findFirstInFile( list );
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, start, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( (char *)NULL );
     }
     /*
      * Read 'SF_COMMENT' line.
      */	 

     line = readHeader( sf->fd, SF_COMMENT, end, &err );
     if ( line == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LINE_NOT_FOUND;
	  else *error = err;
     }

     return( line );
}

/*********************************************************************
 *   Function:		char *SfUser( sf, index, error )
 *
 *   Description:	Gets spec user information from the last file header
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			User.(From 1st #C line !)
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	||=>  findWordInLine()
 *			SF_ERR_LINE_NOT_FOUND	| 
 *			SF_ERR_FILE_READ	|
 *			SF_ERR_SCAN_NOT_FOUND	| =>  getFirstFileC()
 *			SF_ERR_HEADER_NOT_FOUND	|
 *			SF_ERR_USER_NOT_FOUND	
 *
 *********************************************************************/
char * 
SfUser( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     char		*line;
     char		 word[] = "User =";
     char		*user;
     int		 err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfUser()\n");

     /*
      * Read first comment line in file header.
      */	 
     line = readFirstFileC( sf, index, error ); 
     if ( line == (char *)NULL ) return( (char *)NULL );
     /*
      * Find user.
      */	 
     user = findWordInLine( line, word, &err );
     free( line );

     if ( user == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_USER_NOT_FOUND;
	  else *error = err;
     }
     return( user );
}

/*********************************************************************
 *   Function:		long SfTitle( sf, index, error )
 *
 *   Description:	Gets spec title information from the last file header
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Title.(From 1st #C line !)
 *			NULL => errors.
 *   Possible errors:
 *			SF_ERR_LINE_EMPTY	
 *			SF_ERR_MEMORY_ALLOC	
 *			SF_ERR_LINE_NOT_FOUND	| 
 *			SF_ERR_FILE_READ	|
 *			SF_ERR_SCAN_NOT_FOUND	| =>  getFirstFileC()
 *			SF_ERR_HEADER_NOT_FOUND	|
 *
 *********************************************************************/
char * 
SfTitle( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     char		*line;
     char		*c;
     char		*title;
     long		 i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfTitle()\n");

     /*
      * Read first comment line in file header.
      */	 
     line = readFirstFileC( sf, index, error ); 
     if ( line == (char *)NULL ) return( (char *)NULL );

     /*
      * Get title.( first word )
      */	
     c = line;
     for ( i=0 ; *c!='\t' && *c!='\n' && *c!='\0' ; i++ ) {
	  if ( *c==' ' ) {
	       if ( *(++c)==' ' ) {
		    break;
	       } else c--;
	  }
	  c++;
     }
     if ( i==0 ) {
	  *error = SF_ERR_LINE_EMPTY;
	  return( (char *)NULL );
     }

     title = (char *)malloc( sizeof(char) * ( i+1 ) );
     if ( title == (char *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return( title );
     }
     memcpy( title, line, sizeof(char) * i  );
     title[i] = '\0';
     free( line );

     return( title );
}

/*********************************************************************
 *   Function:		double *SfHKL( sf, index, error )
 *
 *   Description:	Reads '#Q' line.
 *
 *   Parameters:
 *		Input :	(1) File pointer   
 *			(2) Index
 *		Output:
 *			(3) error number
 *   Returns:
 *			Poiter to a 3x1 dbl. array( HKL[0]=HKL[H]=H_value,
 *						    HKL[1]=HKL[K]=K_value,
 *						    HKL[2]=HKL[L]=L_value.
 *			NULL => errors.
 *   Possible errors:
 *			SF_ERR_LINE_EMPTY   
 *			SF_ERR_FILE_READ
 *			SF_ERR_SCAN_NOT_FOUND
 *			SF_ERR_LINE_NOT_FOUND	 
 *			SF_ERR_MEMORY_ALLOC	| => mulstrtod()
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
double *
SfHKL( sf, index, error )

SpecFile       *sf;
long		index;
int	       *error;
{
     ObjectList		*list;
     SpecScan		*scan;
     char		*line;
     double		*HKL;
     long		 end;
     long		 i;
     int		 err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfHKL()\n");

     /*
      * Find scan
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) { 
	  *error = SF_ERR_SCAN_NOT_FOUND;
	  return( (double *)NULL );
     }
     setScanCurrent( sf, list );
     scan = list->contents;
     end  = ((SpecScan *)list->contents)->data_offset;
     if ( end < 1 ) {
	  if ( (ObjectList *)list->next == NULL ) {
	       if ( fseek( sf->fd, 0, SEEK_END ) ) {
		    *error = SF_ERR_FILE_READ;
		    return( (double *)NULL );
	       }
	       if ( (end = ftell( sf-> fd ))==-1L ) {
		    *error = SF_ERR_FILE_READ;
		    return( (double *)NULL );
	       }	       
	  } else {
	       if ( ((SpecScan *)list->next->contents)->last_header !=
						  scan->last_header ) {
		    end = ((SpecScan *)list->next->contents)->last_header;
	       } else {
		    end = ((SpecScan *)list->next->contents)->offset;
	       }
	  }
     }
     /*
      * Set file pointer position
      */	 
     if ( fseek( sf->fd, scan->offset, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( (double *)NULL );
     }
	
     line = readHeader( sf->fd, SF_RECIP_SPACE, end, &err );
     if ( line == (char *)NULL ) {
	  if ( !err ) *error = SF_ERR_LINE_NOT_FOUND;
	  else *error = err;
	  return( (double *)NULL );
     }

     /*
      * Convert into double .
      */
     i = mulstrtod( line, &HKL, error );
     free( line );

     if ( i < 0 ) return( (double *)NULL );
     if ( i != 3 ) {
	  *error = SF_ERR_LINE_EMPTY;
	  free( HKL );
	  return( (double *)NULL );
     }
     return( HKL );
}

/*********************************************************************
 *   Function:		long SfGeometry( sf, index, lines, error )
 *
 *   Description:	Reads all #G lines.
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Scan index
 *		Output:	(3) all #G lines  
 *			(4) Error number 
 *   Returns:
 *			Number of lines
 *			( -1 ) if error.		
 *   Possible errors:
 *			SF_ERR_SCAN_NOT_FOUND	| => SfHeader()
 *			SF_ERR_MEMORY_ALLOC	|   
 *			SF_ERR_FILE_READ	|
 *
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long
SfGeometry( sf, index, lines, error )

SpecFile       *sf;
long		index;
char	     ***lines;
int	       *error;
{
     long	no_lines  = 0;
     char       string[] = " \0";


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfGeometry()\n");

     *lines = (char **)NULL;

     /*
      * Read every '#SF_GEOMETRY' line 
      */	
     string[0]= SF_GEOMETRY;
     no_lines = SfHeader( sf, index, string, lines, error );
     if ( !no_lines ) {
	  *error = SF_ERR_LINE_NOT_FOUND; 
	  return( -1 );
     }
     return( no_lines );
}

/*********************************************************************
 *   Function:		int checkAborted( sf, ptr, error )
 *
 *   Description:	Checks if scan was aborted or not .
 *
 *   Parameters:
 *		Input :	(1) SpecScan pointer
 *			(2) Pointer to the scan
 *		Output:	(3) Error number 
 *   Returns:
 *		(-1 )	: error
 *	        ( 0 )	: not aborted
 *	        ( 1 )	: aborted
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => readHeader()   
 *			SF_ERR_FILE_READ	
 *
 *********************************************************************/
int 
checkAborted( sf, ptr, error )

SpecFile	*sf;
ObjectList	*ptr;
int		*error;

{
     SpecScan	*scan;
     long	 start;
     long	 end;
     char	*comment;
     int	 err = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered checkAborted()\n");

     clearerr( sf->fd );
     scan = ptr->contents;

     if ( (ObjectList *)ptr->next == NULL ) {
	  if ( fseek( sf->fd, 0, SEEK_END ) ) {
	       *error = SF_ERR_FILE_READ;
	       return( -1 );
	  }
	  if ( (end = ftell( sf-> fd ))==-1L ) {
	       *error = SF_ERR_FILE_READ;
	       return( -1 );
	  }	       
     } else {
	  if ( ((SpecScan *)ptr->next->contents)->last_header !=
						scan->last_header ) {
	       end = ((SpecScan *)ptr->next->contents)->last_header;
	  } else {
	       end = ((SpecScan *)ptr->next->contents)->offset;
	  }
     }
     /*
      * Set file pointer position
      */	 
     if ( (start = scan->last_data_offset) < 1 ) return( 1 );
     if ( fseek( sf->fd, start, SEEK_SET )) {
	  *error = SF_ERR_FILE_READ;
	  return( -1 );
     }

     /*
      * Read line containing 1st comment after data.
      */	 
     comment = readHeader( sf->fd, SF_COMMENT, end, &err );
     if ( comment == (char *)NULL ) {
	  if ( !err ) return( 0 );
	  else *error = err;
	  return( -1 );
     }
     
     if ( !strstr( comment, "aborted" )) {
	  free( comment );
	  return( 0 );
     }

     free( comment );
     return( 1 );
}

/*********************************************************************
 *   Function:		long SfCondList( sf, cond, scan_list, error )
 *
 *   Description:	Creates an array with all scan numbers.
 *
 *   Parameters:
 *		Input :	(1) SpecFile pointer
 *			(2) Condition :	0 => not aborted scans ( NOT_ABORTED )
 *				       -1 =>     aborted scans ( ABORTED )
 *				       nn => more than 'nn' data lines
 *		Output: (3) Scan list
 *			(4) error code	
 *   Returns:
 *			Number of found scans.
 *			( -1 ) if errors occured.
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *			
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long 
SfCondList( sf, cond, scan_list, error )
     
SpecFile	*sf;
long		 cond;
long	       **scan_list;
int		*error;
{
     register	ObjectList	*ptr;
     long			*list;
     long			 i = 0;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfCondList()\n");

     *scan_list = (long *)NULL;

     list = (long *)malloc( sizeof(long) * (sf->no_scans) );
     if ( list == (long *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return( -1 );
     }

     /*
      * Aborted scans .
      */
     if ( cond < 0 ) {
	  for ( ptr=sf->list.first ; ptr ; ptr=ptr->next ) {

	       if ( checkAborted( sf, ptr, error ) < 0 ) {
		    free( list );
		    return( -1 );
	       }
	       if ( !checkAborted( sf, ptr, error ) ) continue;
	       
	       list[i] = ( ((SpecScan *)(ptr->contents))->scan_no );  
	       i++;
	  }   
     }

     /*
      * Not aborted scans .
      */
     if ( cond == 0 ) {
	  for ( ptr=sf->list.first ; ptr ; ptr=ptr->next ) {

	       if ( checkAborted( sf, ptr, error ) < 0 ) {
		    free( list );
		    return( -1 );
	       }
	       if ( checkAborted( sf, ptr, error ) ) continue;
	       
	       list[i] = ( ((SpecScan *)(ptr->contents))->scan_no );  
	       i++;
	  }   
     }

     /*
      * Scans with more than 'x' data lines .
      */
     if ( cond > 0 ) {
	  for ( ptr=sf->list.first ; ptr ; ptr=ptr->next ) {
	       
	       if ( ((SpecScan *)ptr->contents)->data_lines <= cond ) continue;
	       
	       list[i] = ( ((SpecScan *)(ptr->contents))->scan_no );  
	       i++;
	  }   
     }     

     *scan_list = (long *)realloc( list, sizeof(long) * i );
     if ( *scan_list == (long *)NULL ) {
	  *error = SF_ERR_MEMORY_ALLOC;
	  return( -1 );
     }

     return( i );
}

/*********************************************************************
 *   Function:		long SfScanNo( sf )
 *
 *   Description:	Gets number of scans.
 *
 *   Parameters:
 *		Input :(1) SpecFile pointer
 *   Returns:
 *		Number of scans.
 *
 *********************************************************************/
long 
SfScanNo( sf )

SpecFile *sf;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered SfScanNo()\n");

     return( sf->no_scans );
}

/*********************************************************************
 *   Function:		int SfNumberOrder( sf, index, number, order )
 *
 *   Description:	Gets scan number and order from index.
 *
 *   Parameters:
 *		Input :
 *			(1) SpecFile pointer
 *			(2) Scan index
 *		Output:	
 *			(3) Scan number
 *			(4) Scan order
 *   Returns:
 *		( -1 ) => not found
 *		(  0 ) => found		
 *
 *********************************************************************/
int
SfNumberOrder( sf, index, number, order )

SpecFile       *sf;
long		index;
long	       *number;
long	       *order;
{
     register ObjectList	*list;

     if ( sf_debug ) 
       fprintf( stderr, "Entered SfNumberOrder()\n");

     *number = -1;
     *order  = -1;

     /*
      * Find scan .
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) return( -1 );
     
     *number = ((SpecScan *)list->contents)->scan_no;
     *order  = ((SpecScan *)list->contents)->order;

     return( 0 );
}

/*********************************************************************
 *   Function:		long SfNumber( sf, index )
 *
 *   Description:	Gets scan number from index.
 *
 *   Parameters:
 *		Input :	(1) SpecFile pointer
 *			(2) Scan index
 *   Returns:
 *		Scan number.
 *		( -1 ) => not found
 *
 *********************************************************************/
long
SfNumber( sf, index  )

SpecFile       *sf;
long		index;
{
     register ObjectList	*list;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfNumber()\n");

     /*
      * Find scan .
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) return( -1 );
     
     return( ((SpecScan *)list->contents)->scan_no );
}

/*********************************************************************
 *   Function:		long SfOrder( sf, index )
 *
 *   Description:	Gets scan order from index.
 *
 *   Parameters:
 *		Input :	(1) SpecFile pointer
 *			(2) Scan index
 *   Returns:
 *		Scan order.
 *		( -1 ) => not found
 *
 *********************************************************************/
long
SfOrder( sf, index  )

SpecFile       *sf;
long		index;
{
     register ObjectList	*list;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfOrder()\n");

     /*
      * Find scan .
      */
     list = findScanByIndex( &(sf->list), index );
     if ( list == (ObjectList *)NULL ) return( -1 );
     
     return( ((SpecScan *)list->contents)->order );
}

/*********************************************************************
 **********************	   WRITE FUNCTIONS    ************************
 *********************************************************************/


/*********************************************************************
 *   Function:		SpecFileOut *SfoInit( sf, error )
 *
 *   Description:	Initializes a SpecFileOut structure:
 *				- pointer to SpecFile
 *				- list of scans to be copied
 *				- size of this list
 *				- last written file header
 *   Parameters:
 *		Input :	(1) SpecFile pointer
 *			
 *		Output:
 *			(2) error number
 *   Returns:
 *			Pointer to the initialized SpecFileOut structure.
 *			NULL in case of an error.			
 *
 *   Possible errors:
 *			SF_ERR_MEMOREY_ALLOC
 *			
 *   Remark:	This function MUST be the FIRST called before
 *		any other WRITE function is called !
 *
 *********************************************************************/
SpecFileOut *
SfoInit( sf, error )

SpecFile	*sf;
int		*error;
{
     SpecFileOut	*sfo;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoInit()\n");

     /*
      * Alloc memory 
      */
     sfo = (SpecFileOut *)malloc( sizeof(SpecFileOut) );
	  if ( sfo == (SpecFileOut *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       return( (SpecFileOut *)NULL );
	  }

     /*
      * Initialize 
      */
     sfo->sf		= sf;
     sfo->list		= (long *)NULL;
     sfo->list_size    	=  0;
     sfo->last_header  	= -1;     

     return( sfo );
}

/*********************************************************************
 *   Function:		long  SfoGetList( sfo, list, error )
 *
 *   Description:	 Makes a copy of the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer 
 *			
 *		Output: (2) Copy of the output list of spec scan indices.
 *			(3) error code
 *   Returns:
 *			Number of scan indices in the output list ,
 *			(  0 ) => list empty( (long *)NULL ) ), no errors
 *			( -1 ) in case of an error.
 *
 *   Possible errors:
 *			SF_ERR_MEMOREY_ALLOC
 *			
 *   Remark:  The memory allocated should be freed by the application
 *
 *********************************************************************/
long 
SfoGetList( sfo, list, error )

SpecFileOut	*sfo;
long	       **list;
int		*error;
{
     long	i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoGetList()\n");

     *list = (long *)NULL;

     if ( sfo->list_size > 0 ) {
	  *list = (long *)malloc( sfo->list_size * sizeof(long) );
	  if ( *list == (long *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       return( -1 );
	  }
	  for ( i=0 ; i < sfo->list_size ; i++ ) {
	       (*list)[i] = sfo->list[i];
	  }    
     } else *list = (long *)NULL;   

     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		long SfoSelectOne( sfo, index, error )
 *
 *   Description:	Adds one scan index to the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer
 *			(2) Scan index
 *		Output:
 *			(3) error code
 *   Returns:
 *			( -1 ) => error
 *			Number of scan indices in the SpecFileOut list.
 *		      
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *
 *********************************************************************/
long
SfoSelectOne( sfo, index, error )

SpecFileOut	*sfo;
long		 index;
int		*error;
{
     long	i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoSelectOne()\n");

     /*
      * Check if index exists or if it's out of range.
      */
     if ( index > sfo->sf->no_scans || index < 1 ) {
	  return( sfo->list_size );
     }

     /*
      * Alloc memory for the new index and add it to the list.
      */
     if ( sfo->list == (long *)NULL ) {
	  sfo->list = (long *)malloc( sizeof(long) );
	  if ( sfo->list == (long *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       return( -1 );
	  }
	  sfo->list_size = 1;
     } else {
	  /*
	   * Is the new index already in list ?
	   */
	  for ( i=0 ; i<sfo->list_size ; i++ ) 
	    if ( index == sfo->list[i] ) return( sfo->list_size );
	  sfo->list = realloc( sfo->list, ++(sfo->list_size) * sizeof(long) );
	  if ( sfo->list == (long *)NULL ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       sfo->list_size = 0;
	       return( -1 );
	  }
     }     
     sfo->list[sfo->list_size-1] = index;

     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		long SfoSelect( sfo, list, error )
 *
 *   Description:	Adds several scan indices to the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer
 *			(2) List scan indices (!The last element 
 *						MUST be a '0' !)
 *		Output:
 *			(3) error code
 *   Returns:
 *			( -1 ) => error
 *			Number of scan indices in the SpecFileOut list.
 *			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => SfoSelectOne()
 *
 *********************************************************************/
long
SfoSelect( sfo, list, error )

SpecFileOut	*sfo;
long		*list;
int		*error;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoSelect()\n");

     for ( ; *list != 0 ; list++ ) {
	  if ( SfoSelectOne( sfo, *list , error ) < 0 ) return( -1 );
     }
     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		long SfoSelectRange( sfo, begin, end, error )
 *
 *   Description:	Adds scan indices between 'begin' and 'end'
 *			to the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer
 *			(2) First ...
 *			(3) Last index to be added
 *		Output:
 *			(4) error code
 *   Returns:
 *			( -1 ) => error
 *			Number of scan indices in the SpecFileOut list.
 *			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => SfoSelectOne()
 *
 *********************************************************************/
long
SfoSelectRange( sfo, begin, end, error )

SpecFileOut	*sfo;
long		 begin;
long		 end;
int		*error;
{
     long	i;

     
     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoSelectRange()\n");

     if ( begin > end ) {
	  i=begin;
	  begin = end;
	  end = i;
     }
     if ( begin < 1 || end > sfo->sf->no_scans ) {
	  return( sfo->list_size );
     }
     for ( i=begin ; i<=end ; i++ ) {
	  if ( SfoSelectOne( sfo, i , error ) < 0 ) return( -1 );
     }
     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		long SfoSelectAll( sfo, error )
 *
 *   Description:	Writes all scan indices in the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	(1) SpecFileOutput pointer 
 *		Output:	(2) error number
 *   Returns:
 *			( -1 ) => error
 *			Number of scan indices in the SpecFileOut list.
 *			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *
 *********************************************************************/
long
SfoSelectAll( sfo, error )

SpecFileOut	*sfo;
int		*error;
{
     long	i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoSelectAll()\n");

     if ( sfo->sf->no_scans > 0 ) {
	  for ( i=1 ; i<=sfo->sf->no_scans ; i++ ) {
	       if ( SfoSelectOne( sfo, i , error ) < 0 ) return( -1 );
	  }    
     }    
     return( sfo->list_size );
}
	      
/*********************************************************************
 *   Function:		long SfoRemoveOne( sfo, index, error )
 *
 *   Description:	Removes one scan index from the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer
 *			(2) Scan index to be removed
 *		Output:
 *			(3) error code
 *   Returns:
 *			Number of scans left ,
 *			(  0 ) => list empty( (long *)NULL ) ), no errors
 *			( -1 ) => error.
 *			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC
 *
 *********************************************************************/
long
SfoRemoveOne( sfo, index, error )

SpecFileOut	*sfo;
long		 index;
int		*error;
{
     long	i;
     int	found = 0;
     

     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoRemoveOne()\n");

     /*
      * Look for scan index and delete.
      */
     for ( i=0 ; i < (sfo->list_size - found) ; i++ ) {
	  if ( sfo->list[i] == index ) found = 1;
	  if ( found ) sfo->list[i]=sfo->list[i+1];
     }

     /*
      * Free unused memory
      */
     if ( found ) {
	  (sfo->list_size)--;
	  sfo->list = realloc( sfo->list, sfo->list_size * sizeof(long) );
	  if ( sfo->list == (long *)NULL && sfo->list_size != 0 ) {
	       *error = SF_ERR_MEMORY_ALLOC;
	       return( -1 );
	  }
     }
     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		long SfoRemove( sfo, list, error )
 *
 *   Description:	Removes several scans indices from the 
 *						SpecFileOut list.
 *
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer
 *			(2) List of scan indices to be removed
 *			    ( !!! The last element MUST be a '0' !!! )
 *		Output:
 *			(3) error code
 *   Returns:
 *			Number of scan indices left , 
 *			(  0 ) => list empty( (long *)NULL ) ), no errors
 *			( -1 ) => error.
 *			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => SfoRemoveOne()
 *
 *********************************************************************/
long
SfoRemove( sfo, list, error )

SpecFileOut	*sfo;
long		*list;
int		*error;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoRemove()\n");

     for ( ; *list != 0 ; list++ ) {
	  if ( SfoRemoveOne( sfo, *list , error ) < 0 ) return( -1 );
     }
     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		long SfoRemoveRange( sfo, begin, end, error )
 *
 *   Description:	Removes scans indices from 'begin' to 'end'
 *					from the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	
 *			(1) SpecFileOut pointer
 *			(2) First ...
 *			(3) Last index to be removed
 *		Output:
 *			(4) error code
 *   Returns:
 *			Number of scan indices left , 
 *			(  0 ) => list empty( (long *)NULL ) ), no errors
 *			( -1 ) => error.
 *			
 *   Possible errors:
 *			SF_ERR_MEMORY_ALLOC	| => SfoRemoveOne()
 *
 *********************************************************************/
long
SfoRemoveRange( sfo, begin, end, error )

SpecFileOut	*sfo;
long		 begin;
long		 end;
int		*error;
{
     long	i;

     
     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoRemoveRange()\n");

     if ( begin > end ) {
	  i=begin;
	  begin = end;
	  end = i;
     }
     if ( begin < 1 || end > sfo->sf->no_scans ) {
	  return( sfo->list_size );
     }
     for ( i=begin ; i <= end ; i++ ) {
	  if ( SfoRemoveOne( sfo, i, error ) < 0 ) return( -1 );
     }
     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		long SfoRemoveAll( sfo, error )
 *
 *   Description:	Removes all scans indices
 *					from the SpecFileOut list.
 *
 *   Parameters:
 *		Input :	
 *			(1) SpecFileOut pointer
 *		Output:
 *			(2) error code
 *   Returns:
 *			( 0 ) => OK 
 *
 *********************************************************************/
long
SfoRemoveAll( sfo, error )

SpecFileOut	*sfo;
int		*error;
{
     free( sfo->list );
     sfo->list		= (long *)NULL;
     sfo->list_size    	=  0;
     sfo->last_header  	= -1;     
     return( 0 );
}

/*********************************************************************
 *   Function:		int cpyBlock( input, output, start, end, error )
 *
 *   Description:	Makes a copy of file block between 'start' and
 *			'end'.
 *   Parameters:
 *		Input :	(1) Input file pointer
 *			(2) Output file pointer
 *			(3) Start
 *			(4) End of block
 *		Output:
 *			(5) error code
 *   Returns:
 *			( 0 ) : Operation successful
 *			(-1 ) : Errors occured 
 *   Possible errors:
 *			SF_ERR_FILE_READ
 *			SF_ERR_FILE_WRITE
 *
 *********************************************************************/
int
cpyBlock( input, output, start, end, error )

FILE		*input;
FILE		*output;
long		 start;
long		 end;
int		*error;
{
     long	position;
     char	 c;


     if ( sf_debug ) 
       fprintf( stderr, "Entered cpyBlock()\n");

     /*
      * Set file pointer position.
      */	 
     if ( fseek( input, start, SEEK_SET ) ) {
	  *error = SF_ERR_FILE_READ;
	  return( -1 );
     }
     
     if ( (position = ftell( input )) == -1L ) {
	  *error = SF_ERR_FILE_READ;
	  return( -1 );
     }	       

     /*
      * Copy
      */	      
     while ( position < end ) {  
	  c = getc( input );
	  if ( ferror( input ) ) {
	       *error = SF_ERR_FILE_READ;
	       return( -1 );
	  }
	  fputc( c, output );
	  if ( ferror( input ) ) {
	       *error = SF_ERR_FILE_WRITE;
	       return( -1 );
	  }
	  if ( (position = ftell( input )) == -1L ) {
	       *error = SF_ERR_FILE_READ;
	       return( -1 );
	  }	       
     }
     return( 0 );
}

/*********************************************************************
 *   Function:		int SfoWrite( sfo, name, error )
 *
 *   Description:	Writes (appends) SpecScans specified in the sfo->list
 *			in the file 'name'. Related file headers are copied
 *			too.
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer
 *			(2) Output file name
 *		Output:
 *			(3) error number
 *   Returns:
 *			Number of written scans,
 *			(-1 ) => Errors occured 
 *   Possible errors:
 *			SF_ERR_FILE_WRITE	| => cpyBlock()
 *			SF_ERR_FILE_READ	
 *			SF_ERR_FILE_OPEN
 *			SF_ERR_FILE_CLOSE
 *
 *********************************************************************/
long
SfoWrite( sfo, name, error )

SpecFileOut	*sfo;
char		*name;
int		*error;
{
     FILE	*input;
     FILE	*output;
     ObjectList	*ptr;
     SpecScan	*scan;
     long	 end;
     long	 i;


     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoWrite()\n");

     if ( sfo == (SpecFileOut *)NULL || sfo->list_size<1 ) return( 0 );

     clearerr( sfo->sf->fd );

     /* 
      * Open file
      */
     if ( (output = fopen(name, "a+")) == NULL ) {
	  *error = SF_ERR_FILE_OPEN;
	  return( -1 );
     }

     input = sfo->sf->fd;

     for ( i=0 ; i < sfo->list_size ; i++ ) {
	  /*
	   * Find scan
	   */
	  ptr = findScanByIndex( &(sfo->sf->list), sfo->list[i] );
	  if ( ptr == (ObjectList *)NULL ) continue;
	  scan = ptr->contents;
	  /*
	   * File header already written ?
	   */
	  if ( scan->last_header!=sfo->last_header && scan->last_header!= -1) {
	       if ( cpyBlock( input, output, scan->last_header, 
			      findFirstInFile( ptr ), error ) )
		 return( -1 );
	       sfo->last_header = scan->last_header;
	  }	  

	  /*
	   * Set end mark.
	   */
	  if ( (ObjectList *)ptr->next == NULL ) {
	       if ( fseek( input, 0, SEEK_END ) ) {
		    *error = SF_ERR_FILE_READ;
		    return( -1 );
	       }
	       if ( (end = ftell( input ))==-1L ) {
		    *error = SF_ERR_FILE_READ;
		    return( -1 );
	       }	       
	  } else {
	       if ( ((SpecScan *)ptr->next->contents)->last_header !=
							scan->last_header ) {
		    end = ((SpecScan *)ptr->next->contents)->last_header;
	       } else {
		    end = ((SpecScan *)ptr->next->contents)->offset;
	       }
	  }

	  /*
	   * Write scan.
	   */
	  if ( cpyBlock( input, output, scan->offset, end, error ) )
	    return( -1 );
     }
     if ( fclose( output ) ) {
	  *error = SF_ERR_FILE_CLOSE;
	  return( -1 );
     }
     return( sfo->list_size );
}

/*********************************************************************
 *   Function:		int SfoClose( sfo )
 *
 *   Description:	Frees all memory used by 
 *			SpecFileOut structure. 
 *   Parameters:
 *		Input :	(1) SpecFileOut pointer
 *
 *   Remark:	This function should be called after all
 *					 writing operations.
 *
 *********************************************************************/
void
SfoClose( sfo )

SpecFileOut	*sfo;
{
     if ( sf_debug ) 
       fprintf( stderr, "Entered SfoClose()\n");

     /*
      * Free memory. 
      */
     free( sfo->list );
     free( sfo );
}     

/****************************************************************************/
/****************************************************************************/

/*********************************************************************
 *   Function:		char *SfError( code )
 *
 *   Description:	Returns the message associated with error 'code'.
 *
 *   Parameters:
 *		Input :	error code
 *
 *********************************************************************/
char *
SfError( code )

int	code;
{
     int	i;

     if ( sf_debug ) 
       fprintf( stderr, "Entered SfError()\n");

     for ( i=0 ; errors[i].code!=0 ; i++ ) {
	  if ( errors[i].code == code ) break;
     }
     return( errors[i].message );
}

/*********************************************************************
 *   Function:		void SfDebugOn()
 *
 *   Description:	Sets 'sf_debug' variable to 1.
 *			=> Debug information will be displayed on stderr.
 *
 *********************************************************************/
void
SfDebugOn()
{
     sf_debug = 1;
}

/*********************************************************************
 *   Function:		void SfDebugOff()
 *
 *   Description:	Sets 'sf_debug' variable to 0.
 *			=> Debug information will be not displayed .
 *
 *********************************************************************/
void
SfDebugOff()
{
     sf_debug = 0;
}

/*********************************************************************
 *   Function:		void SfIndexMode()
 *
 *   Description:	Chenges 'sf_index' variable and returns its
 *                      final value.
 *
 *********************************************************************/
long
SfIndexMode( mode )

int mode;
{
     if (mode >= 0)
         sf_index = mode;

     return(sf_index);
}

long
getflength(file, name)

FILE *file;
char *name;
{
   long  current = ftell(file);
   long  length;

   fseek(file, 0L, SEEK_END);
   length = ftell(file);
   fseek(file, current, SEEK_SET);
   return(length);
}
/****************************************************************************/
/**********************************   END   *********************************/

