/*
 * Written by Thomas Heller, May 2000
 *
 * $Id: install.c,v 1.1 2000/06/28 02:04:01 gward Exp $
 */

/*
 * Windows Installer program for distutils.
 *
 * (a kind of self-extracting zip-file)
 *
 * At runtime, the exefile has appended:
 * - compressed setup-data in ini-format, containing the following sections:
 *	[metadata]
 *	author=Greg Ward
 *	author_email=gward@python.net
 *	description=Python Distribution Utilities
 *	licence=Python
 *	name=Distutils
 *	url=http://www.python.org/sigs/distutils-sig/
 *	version=0.9pre
 *
 *	[Setup]
 *	info= text to be displayed in the edit-box
 *	pthname= for the .pth file (if needed)
 *	pyc_compile= 0 or 1, whether .pyc files are to be generated>
 *	pyo_compile= 0 or 1, whether .pyo files are to be generated>
 *	title= to be displayed by this program
 *	vers_minor= if present, has to be 5 or 6 to require 1.5 or 1.6
 *
 * - a struct meta_data_hdr, describing the above
 * - a zip-file, containing the modules to be installed.
 *   for the format see http://www.pkware.com/appnote.html
 *
 * What does this program do?
 * - the setup-data is uncompressed and written to a temporary file.
 * - setup-data is queried with GetPrivateProfile... calls
 * - [metadata] - info is displayed in the dialog box
 * - The registry is searched for installations of python 1.5 and 1.6
 * - The user can select the python version to use by radio-buttons.
 *   (only if vers_minor is not specified)
 * - The python-installation directory (sys.prefix) is displayed
 *   in an edit field. This directory can be edited, and will
 *   be used as the root-directory for the modules to be installed.
 * - When the start-button is pressed, files from the zip-archive
 *   are extracted to the file system. All .py filenames are stored
 *   in a list.
 *   If pyc_compile is specified, python.dll (the correct one) is
 *   dynamically loaded, and used to compile these files to byte code.
 *   Same for pyo_compile, after setting the Py_OptimizeFlag.
 *   If an installation-directory different from sys.prefix
 *   had been specified, a .pth file is created in sys.prefix.
 */

/*
 * To Do:
 *  - install a help-button, which will display the above
 *    text to the user
 *  - should there be a possibility to display a README file
 *    before starting the installation (if one is present in the archive)
 *  - think about uninstaller
 *  - more comments about what the code does(?)
 *  - fix the progress-bar when compiling files: it displays the
 *    percentage relative to the total number of files, not the
 *    files to be compiled
 *
 *  - think about an uninstaller (?)
 *  - evolve this into a full blown installer (???)
 */

#include <windows.h>
#include <commctrl.h>
#include <shlobj.h>
#include "resource.h"

#define ZLIB_DLL
#include <zlib.h>

#include <stdio.h>
#include <stdarg.h>

/* Bah: global variables */
HWND hwndMain;
HWND hDialog;
char info[4096];
char title[80];
char pthname[80];
char *arc_data;
DWORD arc_size;
char prefix[MAX_PATH];
char *ini_file;
BOOL pyc_compile;
BOOL pyo_compile;
char *file_list = NULL;
int file_list_size = 0;
char modulename[MAX_PATH];
char pythondll[MAX_PATH];
char install_dir[MAX_PATH];
int vers_minor;

#define WM_NUMFILES WM_USER+1	/* wParam: 0, lParam: total number of files */
#define WM_NEXTFILE WM_USER+2	/* wParam: number of this file */
				/* lParam: points to pathname */

enum { UNSPECIFIED, ASK, ALWAYS, NEVER } allow_overwrite = UNSPECIFIED;

#pragma pack(1)

/* zip-archive headers
 * See: http://www.pkware.com/appnote.html
 */

struct eof_cdir {
    long tag;	/* must be 0x06054b50 */
    short disknum;
    short firstdisk;
    short nTotalCDirThis;
    short nTotalCDir;
    long nBytesCDir;
    long ofsCDir;
    short commentlen;
};

struct cdir {
    long tag;	/* must be 0x02014b50 */
    short version_made;
    short version_extract;
    short gp_bitflag;
    short comp_method;
    short last_mod_file_time;
    short last_mod_file_date;
    long crc32;
    long comp_size;
    long uncomp_size;
    short fname_length;
    short extra_length;
    short comment_length;
    short disknum_start;
    short int_file_attr;
    long ext_file_attr;
    long ofs_local_header;
};

struct fhdr {
    long tag;	/* must be 0x04034b50 */
    short version_needed;
    short flags;
    short method;
    short last_mod_file_time;
    short last_mod_file_date;
    long crc32;
    long comp_size;
    long uncomp_size;
    short fname_length;
    short extra_length;
};


struct meta_data_hdr {
    int tag;
    int method;
    int crc32;
    int uncomp_size;
    int comp_size;
    int reserved1;
    int reserved2;
    int reserved3;
};

#pragma pack()


void unescape (char *str)
{
    char *dst = str;
    char *src = str;
    char *eon;
    char ch;

    while (src && *src) {
	if (*src == '\\') {
	    switch (*++src) {
	    case 'n':
		*dst++ = '\n';
		*dst++ = '\r';
		break;
	    case 'r':
		*dst++ = '\r';
		break;
	    case '0': case '1': case '2': case '3':
		ch = (char)strtol (src, &eon, 8);
		if (ch == '\n')
		    *dst++ = '\r';
		*dst++ = ch;
		src = eon;
	    }
	} else
	    *dst++ = *src++;
    }
    *dst = '\0';
}

void Log_RegVal (HKEY hKey, char *keyname, char *valuename, char *value)
{
}

void Log_FileCreated (char *pathname)
{
}

void Log_DirCreated (char *pathname)
{
}

void RegisterUninstaller (void)
{
}

void AddToFilelist (char *pathname)
{
    file_list_size += strlen (pathname) + 2;
    if (!file_list) {
	file_list = malloc (strlen (pathname) + 1);
	strcpy (file_list, pathname);
    } else {
	file_list = realloc (file_list, file_list_size);
        if (file_list) {
    	    strcat (file_list, "|");
	    strcat (file_list, pathname);
	}
    }
}

int CompileFilelist (BOOL optimize_flag,
		     int (__cdecl * PyRun_SimpleString)(char *))
{
    char *temp_list = strdup (file_list); /* strtok changes the string */
    char *cp = temp_list;
    char *pathname;
    char Buffer[MAX_PATH];
    int n = 0;
    int errors = 0;

    while (pathname = strtok (cp, "|")) {
	++n;
	cp = NULL;
        wsprintf (Buffer,
		  "import py_compile; py_compile.compile (r'%s')", pathname);
        if (PyRun_SimpleString (Buffer))
	    ++errors;
	else {
	    wsprintf (Buffer, "%s%c", pathname, optimize_flag ? 'o' : 'c');
	    Log_FileCreated (Buffer);
	}
	SendMessage (hDialog, WM_NEXTFILE, n, (LPARAM)pathname);
    };
    free (temp_list);
    return errors;
}

BOOL SystemError (int error, char *fmt, ...)
{
    char Buffer[1024];
    va_list marker;
    int n;

#ifdef _DEBUG
    _asm int 3;
#endif

    if (error) {
        LPVOID lpMsgBuf;
	FormatMessage( 
	    FORMAT_MESSAGE_ALLOCATE_BUFFER | 
	    FORMAT_MESSAGE_FROM_SYSTEM,
	    NULL,
	    error,
	    MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
	    (LPSTR)&lpMsgBuf,
	    0,
	    NULL 
	    );
        strncpy (Buffer, lpMsgBuf, sizeof (Buffer));
	LocalFree (lpMsgBuf);
    } else
	Buffer[0] = '\0';
    va_start (marker, fmt);
    n = lstrlen (Buffer);
    _vsnprintf (Buffer+n, sizeof (Buffer)-n, fmt, marker);
    MessageBox (hwndMain, Buffer, "Runtime Error", MB_OK | MB_ICONSTOP);
    va_end (marker);
    return FALSE;
}

BOOL ZlibError (char *fmt, ...)
{
    char Buffer[256];
    va_list marker;

#ifdef _DEBUG
    _asm int 3;
#endif

    va_start (marker, fmt);
    _vsnprintf (Buffer, sizeof (Buffer), fmt, marker);
    MessageBox (hwndMain, Buffer, "Zlib Error", MB_OK | MB_ICONSTOP);
    va_end (marker);
    return FALSE;
}

BOOL AskOverwrite (char *filename)
{
    int result;
  again:
    if (allow_overwrite == ALWAYS)
	return TRUE;
    if (allow_overwrite == NEVER)
	return FALSE;
    if (allow_overwrite == ASK)
        return (IDYES == MessageBox (hwndMain,
			    filename,
			    "Overwrite existing file?",
			    MB_YESNO | MB_ICONQUESTION));

    result = MessageBox (hwndMain,
"Overwrite existing files?\n"
"\n"
"Press YES to ALWAYS overwrite existing files,\n"
"press NO to NEVER overwrite existing files,\n"
"press CANCEL to ASK individually.",
		         "Overwrite options",
			 MB_YESNOCANCEL | MB_ICONQUESTION);
    if (result == IDYES)
	allow_overwrite = ALWAYS;
    else if (result == IDNO)
	allow_overwrite = NEVER;
    else
	allow_overwrite = ASK;
    goto again;
}

/* Convert unix-path to dos-path */
static void fixpath (char *path)
{
    while (path && *path) {
	if (*path == '/')
	    *path = '\\';
	++path;
    }
}

char *MapExistingFile (char *pathname, DWORD *psize)
{
    HANDLE hFile, hFileMapping;
    DWORD nSizeLow, nSizeHigh;
    char *data;

    hFile = CreateFile (pathname,
	GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING,
	FILE_ATTRIBUTE_NORMAL, NULL);
    if (hFile == INVALID_HANDLE_VALUE)
	return NULL;
    nSizeLow = GetFileSize (hFile, &nSizeHigh);
    hFileMapping = CreateFileMapping (hFile,
	NULL, PAGE_READONLY, 0, 0, NULL);
    CloseHandle (hFile);

    if (hFileMapping == INVALID_HANDLE_VALUE)
	return NULL;
    
    data = MapViewOfFile (hFileMapping,
	FILE_MAP_READ, 0, 0, 0);

    CloseHandle (hFileMapping);
    *psize = nSizeLow;
    return data;
}

BOOL EnsureDirectory (char *pathname, char *new_part)
{
    while (new_part && *new_part && (new_part = strchr (new_part, '\\'))) {
	DWORD attr;
	*new_part = '\0';
	attr = GetFileAttributes (pathname);
	if (attr == -1) {
	    /* nothing found */
	    if (!CreateDirectory (pathname, NULL))
		return SystemError (GetLastError(), "CreateDirectory (%s)", pathname);
	    Log_DirCreated (pathname);
	}
	if (attr & FILE_ATTRIBUTE_DIRECTORY) {
	    ;
	} else {
	    SetLastError (183);
	    return SystemError (GetLastError(), "CreateDirectory (%s)", pathname);
	}
	*new_part = '\\';
	++new_part;
    }
    return TRUE;
}

char *MapNewFile (DWORD flags, char *filename,
			 char *pathname_part, struct fhdr *pfhdr)
{
    HANDLE hFile, hFileMapping;
    char *dst;
    int size = pfhdr->uncomp_size;
    FILETIME ft;

    /* CREATE_ALWAYS overwrites existing files, CREATE_NEW doesn't! */
  try_again:
    if (!flags)	/* better use -1 as a special value here ?*/
	flags = allow_overwrite == ALWAYS ? CREATE_ALWAYS : CREATE_NEW;
    hFile = CreateFile (filename,
			GENERIC_WRITE | GENERIC_READ,
			0, NULL,
			flags,
			FILE_ATTRIBUTE_NORMAL, NULL);
    if (hFile == INVALID_HANDLE_VALUE) {
	DWORD x = GetLastError();
	switch (x) {
	case ERROR_FILE_EXISTS:
	    if (AskOverwrite (filename))
	       	hFile = CreateFile (filename,
				    GENERIC_WRITE | GENERIC_READ,
				    0, NULL,
				    CREATE_ALWAYS,
				    FILE_ATTRIBUTE_NORMAL, NULL);
	    else
		return NULL;
	    break;
	case ERROR_PATH_NOT_FOUND:
	    if (EnsureDirectory (filename, pathname_part))
		goto try_again;
	    else
		return FALSE;
	    break;
	default:
	    SetLastError (x);
	    break;
	}
    }
    if (hFile == INVALID_HANDLE_VALUE) {
	SystemError (GetLastError(), "CreateFile (%s)", filename);
	return NULL;
    }

    Log_FileCreated (filename);

    DosDateTimeToFileTime (pfhdr->last_mod_file_date, pfhdr->last_mod_file_time, &ft);
    SetFileTime (hFile, &ft, &ft, &ft);

    hFileMapping = CreateFileMapping (hFile,
				      NULL, PAGE_READWRITE, 0, size, NULL);

    CloseHandle (hFile);

    if (hFileMapping == INVALID_HANDLE_VALUE) {
	SystemError (GetLastError(), "CreateFileMapping (%s)", filename);
	return NULL;
    }

    dst = MapViewOfFile (hFileMapping,
			 FILE_MAP_WRITE, 0, 0, 0);

    CloseHandle (hFileMapping);

    if (!dst) {
	SystemError (GetLastError(), "MapViewOfFile (%s)", filename);
	return NULL;
    }
    return dst;
}

static void Debug(char *data, int size)
{
    char *fname = "c:\\wininst\\fails.cmp";
    FILE *fp = fopen (fname, "wb");
    fwrite (data, 1, size, fp);
    fclose (fp);
}

BOOL extract_file (char *dst, struct fhdr *phdr, char *src)
{
    z_stream zstream;
    int result;

    if (phdr->method == Z_DEFLATED) {
	int x;
	long crc = 0;
        memset (&zstream, 0, sizeof (zstream));
        zstream.next_in = src;
        zstream.avail_in = phdr->comp_size+1;
	zstream.next_out = dst;
        zstream.avail_out = phdr->uncomp_size;

/* Apparently an undocumented feature of zlib: Set windowsize
 to negative values to supress the gzip header and be compatible with
 zip! */
	result = TRUE;
        if (Z_OK != (x = inflateInit2(&zstream, -15))) {
	    result = ZlibError("inflateInit2 returns %d", x);
	    goto cleanup;
	}
	if (Z_STREAM_END != (x = inflate(&zstream, Z_FINISH))) {
	    result = ZlibError("inflate returns %d", x);
	    Debug(src, phdr->comp_size);
	}
      cleanup:
	if (Z_OK != (x = inflateEnd(&zstream))) {
	    result = ZlibError("inflateEnd returns %d", x);
	}
//	crc = crc32 (0, NULL, 0);
//	crc = crc32(crc, dst, phdr->uncomp_size);
//	if (crc != phdr->crc32) {
//	    /* CRC-Error */
//	    result = SystemError(0, "CRC-Error in archive");
//	}
    } else if (phdr->method == 0) {
	memcpy(dst, src, phdr->uncomp_size);
	result = TRUE;
    } else
	result = FALSE;
    UnmapViewOfFile(dst);
    return result;
}

/* Open a zip-compatible archive and extract all files
 * into the specified directory (which is assumed to exist)
 */
BOOL unzip_archive (char *dirname, char *data, DWORD size)
{
    int n;
    char pathname[MAX_PATH];
    char *new_part;

    /* read the end of central directory record */
    struct eof_cdir *pe = (struct eof_cdir *)&data[size - sizeof (struct eof_cdir)];
    int arc_start = size - sizeof (struct eof_cdir) - pe->nBytesCDir - pe->ofsCDir;

    /* set position to start of central directory */
    int pos = arc_start + pe->ofsCDir;

    /* make sure this is a zip file */
    if (pe->tag != 0x06054b50)
	return FALSE;
    
    /* Loop through the central directory, reading all entries */
    for (n = 0; n < pe->nTotalCDir; ++n) {
	char *fname;
	char *pcomp;
	char *dst;
	struct cdir *pcdir = (struct cdir *)&data[pos];
	struct fhdr *pfhdr = (struct fhdr *)&data[pcdir->ofs_local_header + arc_start];

        if (pcdir->tag != 0x02014b50)
	    return FALSE;
	if (pfhdr->tag != 0x04034b50)
	    return FALSE;
	pos += sizeof (struct cdir);
	fname = (char *)&data[pos]; /* This is not null terminated! */
	pos += pcdir->fname_length + pcdir->extra_length + pcdir->comment_length;
	pcomp = &data[pcdir->ofs_local_header
		     + sizeof (struct fhdr)
		     + arc_start
		     + pfhdr->fname_length
		     + pfhdr->extra_length];

	strcpy (pathname, dirname);
	strcat (pathname, "\\");
	new_part = &pathname[lstrlen (pathname)];
	strncat (pathname, fname, pfhdr->fname_length);
	fixpath (pathname);
	if (pathname[strlen(pathname)-1] != '\\') {
	    dst = MapNewFile (0, pathname, new_part, pfhdr);
	    if (dst) {
		if (!extract_file (dst, pfhdr, pcomp))
		    return FALSE;
		if (0 == strcmp (".py", &pathname[strlen(pathname) - 3]))
		    AddToFilelist (pathname);
	    }
	}
	SendMessage (hDialog, WM_NUMFILES, 0, MAKELPARAM (0, pe->nTotalCDir));
	SendMessage (hDialog, WM_NEXTFILE, n, (LPARAM)new_part);
    }
    return TRUE;
}

char *ExtractIniFile (char *data, DWORD size)
{
    /* read the end of central directory record */
    struct eof_cdir *pe = (struct eof_cdir *)&data[size - sizeof (struct eof_cdir)];
    int arc_start = size - sizeof (struct eof_cdir) - pe->nBytesCDir - pe->ofsCDir;
    int ofs = arc_start - sizeof (struct meta_data_hdr);

    /* read meta_data info */
    struct meta_data_hdr *pmd = (struct meta_data_hdr *)&data[ofs];
    struct fhdr fhdr;
    char *src, *dst;
    char *ini_file;
    char tempdir[MAX_PATH];

    if (pe->tag != 0x06054b50) {
	SystemError (0, "Setup program invalid or damaged");
	return NULL;
    }

    if (pmd->tag != 0x12345678 || ofs < 0) {
	SystemError (0, "Setup program invalid or damaged");
	return NULL;
    }

    /* Fake an fhdr to make extract_file() happy */
    fhdr.method = pmd->method;
    fhdr.comp_size = pmd->comp_size;
    fhdr.uncomp_size = pmd->uncomp_size;
    fhdr.crc32 = pmd->crc32;

    src = ((char *)pmd) - pmd->comp_size;
    ini_file = malloc (MAX_PATH); /* will be returned, so do not free it */
    if (!ini_file)
	return NULL;
    if (!GetTempPath (sizeof (tempdir), tempdir)
	|| !GetTempFileName (tempdir, "~du", 0, ini_file)) {
	SystemError (GetLastError(), "Could not create temporary file");
	return NULL;
    }
    
    dst = MapNewFile (CREATE_ALWAYS, ini_file, NULL, &fhdr);
    if (!extract_file (dst, &fhdr, src))
	return NULL;
    return ini_file;
}

void PumpMessages (void)
{
    MSG msg;
    while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }
}

/* Retrieve the python installation directory from the registry */
void SetInstallDir (int major, int minor)
{
    HKEY hKey;
    DWORD type, size;
    char keyname[256];

    wsprintf (pythondll, "python%d%d", major, minor);
    wsprintf (keyname,
	      "Software\\Python\\PythonCore\\%d.%d\\InstallPath",
	      major, minor);
    RegOpenKeyEx (HKEY_LOCAL_MACHINE, keyname, 0, KEY_QUERY_VALUE, &hKey);
    size = sizeof (prefix);
    RegQueryValueEx (hKey, NULL, NULL, &type, prefix, &size);
}

BOOL IsPythonInstalled (int major, int minor)
{
    HKEY hKey;
    DWORD type, size;
    char keyname[256];
    DWORD result;

    wsprintf (keyname,
	      "Software\\Python\\PythonCore\\%d.%d\\InstallPath",
	      major, minor);
    result = RegOpenKeyEx (HKEY_LOCAL_MACHINE, keyname, 0,
			   KEY_QUERY_VALUE, &hKey);
    if (result)
	return FALSE;
    size = 0;
    result = RegQueryValueEx (hKey, NULL, NULL, &type, NULL, &size);
    if (result || (size == 0))
	return FALSE;
    return TRUE;
}

void CompileFiles (void)
{
    void (__cdecl * Py_Initialize)(void);
    void (__cdecl * Py_Finalize)(void);
    int (__cdecl * PyRun_SimpleString)(char *);
    int *Py_OptimizeFlag;
    int errors = 0;
    HINSTANCE hPython;

    if (!file_list)
	return;
    if (pyc_compile) {
	SendDlgItemMessage (hDialog, IDC_PROGRESS, PBM_SETPOS, 0, 0);
	SetDlgItemText (hDialog, IDC_INFO, "Loading python...");
		
	hPython = LoadLibrary (pythondll);
	Py_Initialize = (void (*)(void))GetProcAddress (hPython,"Py_Initialize");
	Py_Finalize = (void (*)(void))GetProcAddress (hPython, "Py_Finalize");
	PyRun_SimpleString = (int (*)(char *))GetProcAddress (hPython, "PyRun_SimpleString");
	Py_OptimizeFlag = (int *)GetProcAddress (hPython, "Py_OptimizeFlag");

	*Py_OptimizeFlag = 0;
	Py_Initialize ();

	SetDlgItemText (hDialog, IDC_INFO, "Compiling .py to .pyc...");
	errors += CompileFilelist (FALSE, PyRun_SimpleString);
	Py_Finalize ();
	FreeLibrary (hPython);
    }

    if (pyo_compile) {
	SendDlgItemMessage (hDialog, IDC_PROGRESS, PBM_SETPOS, 0, 0);
	SetDlgItemText (hDialog, IDC_INFO, "Loading python...");

	hPython = LoadLibrary (pythondll);
	Py_Initialize = (void (*)(void))GetProcAddress (hPython, "Py_Initialize");
	Py_Finalize = (void (*)(void))GetProcAddress (hPython, "Py_Finalize");
	PyRun_SimpleString = (int (*)(char *))GetProcAddress (hPython, "PyRun_SimpleString");
	Py_OptimizeFlag = (int *)GetProcAddress (hPython, "Py_OptimizeFlag");

	*Py_OptimizeFlag = 1;
	Py_Initialize ();
	SetDlgItemText (hDialog, IDC_INFO, "Compiling .py to .pyo...");
	errors += CompileFilelist (TRUE, PyRun_SimpleString);
	Py_Finalize ();
		
	FreeLibrary (hPython);
    }

    if (errors)
	MessageBox (hDialog,
		    "Errors occurred while compiling",
		    "setup",
		    MB_OK | MB_ICONWARNING);

    if (strcmp (install_dir, prefix)) {
	char path[MAX_PATH];
	FILE *pth;

	wsprintf (path, "%s\\%s.pth", prefix, pthname);
	pth = fopen (path, "w");
	Log_FileCreated (path);
	fprintf (pth, install_dir);
	fclose (pth);
    }
}

BOOL CALLBACK DlgProc (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    BOOL result;
    switch (msg) {
    case WM_INITDIALOG:
	hDialog = hwnd;
	SetDlgItemText (hwnd, IDC_EDIT1, info);
	EnableWindow (GetDlgItem (hwnd, IDC_START), FALSE);
	EnableWindow (GetDlgItem (hwnd, IDC_PYTHON15), FALSE);
	EnableWindow (GetDlgItem (hwnd, IDC_PYTHON16), FALSE);
	{
	    BOOL can_use_15 = IsPythonInstalled (1, 5);
	    BOOL can_use_16 = IsPythonInstalled (1, 6);

	    if (!can_use_15 && !can_use_16) {
		SetDlgItemText (hwnd, IDC_INFO,
				"ERROR: Cannot find any usable python installation");
		return TRUE;
	    }

	    if (vers_minor == 5)
		can_use_16 = FALSE;
	    if (vers_minor == 6)
		can_use_15 = FALSE;
	    if (can_use_15) {
		CheckRadioButton (hwnd, IDC_PYTHON15, IDC_PYTHON16, IDC_PYTHON15);
		SetInstallDir (1, 5);
		EnableWindow (GetDlgItem (hwnd, IDC_PYTHON15), TRUE);
	    }
	    if (can_use_16) {
		CheckRadioButton (hwnd, IDC_PYTHON15, IDC_PYTHON16, IDC_PYTHON16);
		SetInstallDir (1, 6);
		EnableWindow (GetDlgItem (hwnd, IDC_PYTHON16), TRUE);
	    }

	    if (!can_use_15 && !can_use_16) {
		char Buffer[64];
		wsprintf (Buffer, "ERROR: Requires python version 1.%d", vers_minor);
		SetDlgItemText (hwnd, IDC_INFO, Buffer);
		return TRUE;
	    }

	    SetDlgItemText (hwnd, IDC_PATH, prefix);

	}
        SetDlgItemText (hwnd, IDC_INFO, "Press Start to install");
	EnableWindow (GetDlgItem (hwnd, IDC_START), TRUE);
	return TRUE;

    case WM_NUMFILES:
	SendDlgItemMessage (hwnd, IDC_PROGRESS, PBM_SETRANGE, 0, lParam);
	PumpMessages ();
	return TRUE;

    case WM_NEXTFILE:
	SendDlgItemMessage (hwnd, IDC_PROGRESS, PBM_SETPOS, wParam, 0);
	PumpMessages ();
	return TRUE;

    case WM_COMMAND:
	switch (LOWORD (wParam)) {
	case IDC_PYTHON15:
	    SetInstallDir (1, 5);
	    SetDlgItemText (hwnd, IDC_PATH, prefix);
	    break;
	case IDC_PYTHON16:
	    SetInstallDir (1, 6);
	    SetDlgItemText (hwnd, IDC_PATH, prefix);
	    break;
#if 0
	case IDC_BROWSE:
	    {
		BROWSEINFO bi;
		char displayName[MAX_PATH];
		LPITEMIDLIST pidl;

		bi.hwndOwner = hwnd;
		bi.pidlRoot = NULL;
		bi.pszDisplayName = displayName;
		bi.lpszTitle = "Select Installation Directory";
		bi.ulFlags = BIF_RETURNONLYFSDIRS | 0x40;
		bi.lpfn = NULL;
		bi.lParam = 0;
		bi.iImage = 0;

		CoInitialize(NULL);
		pidl = SHBrowseForFolder(&bi);
		
		if (pidl && SHGetPathFromIDList (pidl, displayName)) {
		    //XXX Must still free() pidl;
		    SetDlgItemText (hwnd, IDC_PATH, displayName);
		}
		CoUninitialize ();
	    }
	    break;
#endif
	case IDC_START:	/* Start Installation */
	    EnableWindow (GetDlgItem (hwnd, IDC_START), FALSE);
            EnableWindow (GetDlgItem (hwnd, IDCANCEL), FALSE);

	    SetDlgItemText (hwnd, IDC_INFO, "Extracting files...");

	    {
		GetDlgItemText (hwnd, IDC_PATH,
				install_dir, sizeof (install_dir));

		/* Make sure the installation directory name ends in a */
		/* backslash */
		if (install_dir[strlen(install_dir)] != '\\')
		    strcat (install_dir, "\\");
		if (strcmp (install_dir, prefix))
		    EnsureDirectory (install_dir, install_dir);
		/* Strip the trailing backslash again */
		install_dir[strlen(install_dir)-1] = '\0';

		RegisterUninstaller ();

		/* Extract all files from the archive */
		result = unzip_archive (install_dir, arc_data, arc_size);
    		if (result) {
		    CompileFiles ();
		    
		    SetDlgItemText (hDialog, IDC_INFO, "Installation completed .");
		} else
		    SetDlgItemText (hwnd, IDC_INFO, "Installation failed.");
	    }

	    
	    EnableWindow (GetDlgItem (hwnd, IDCANCEL), TRUE);
	    return TRUE;

	case IDCANCEL:
	    EndDialog (hwnd, TRUE);
	    return TRUE;
	}
    }
    return FALSE;
}

LRESULT CALLBACK WindowProc (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HDC hdc;
    HFONT hFont;
    int h;
    PAINTSTRUCT ps;
    switch (msg) {
    case WM_PAINT:
	hdc = BeginPaint (hwnd, &ps);
	h = GetSystemMetrics (SM_CYSCREEN) / 10;
	hFont = CreateFont (h, 0, 0, 0, 700, TRUE,
			    0, 0, 0, 0, 0, 0, 0, "Times Roman");
	hFont = SelectObject (hdc, hFont);
	SetBkMode (hdc, TRANSPARENT);
	TextOut (hdc, 15, 15, title, strlen (title));
	SetTextColor (hdc, RGB (255, 255, 255));
	TextOut (hdc, 10, 10, title, strlen (title));
	DeleteObject (SelectObject (hdc, hFont));
	EndPaint (hwnd, &ps);
	return 0;
    }
    return DefWindowProc (hwnd, msg, wParam, lParam);
}

HWND CreateBackground (char *title)
{
    WNDCLASS wc;
    HWND hwnd;
    char buffer[4096];

    wc.style = CS_VREDRAW | CS_HREDRAW;
    wc.lpfnWndProc = WindowProc;
    wc.cbWndExtra = 0;
    wc.cbClsExtra = 0;
    wc.hInstance = GetModuleHandle (NULL);
    wc.hIcon = NULL;
    wc.hCursor = LoadCursor (NULL, IDC_ARROW);
    wc.hbrBackground = CreateSolidBrush (RGB (0, 0, 128));
    wc.lpszMenuName = NULL;
    wc.lpszClassName = "SetupWindowClass";

    if (!RegisterClass (&wc))
	MessageBox (hwndMain,
		    "Could not register window class",
		    "Setup.exe", MB_OK);

    wsprintf (buffer, "Setup %s", title);
    hwnd = CreateWindow ("SetupWindowClass",
			 buffer,
			 0,
			 0, 0,
			 GetSystemMetrics (SM_CXFULLSCREEN),
			 GetSystemMetrics (SM_CYFULLSCREEN),
			 NULL,
			 NULL,
			 GetModuleHandle (NULL),
			 NULL);
    ShowWindow (hwnd, SW_SHOWMAXIMIZED);
    UpdateWindow (hwnd);
    return hwnd;
}

int WINAPI WinMain (HINSTANCE hInst, HINSTANCE hPrevInst,
		    LPSTR lpszCmdLine, INT nCmdShow)
{
    GetModuleFileName (NULL, modulename, sizeof (modulename));

    /* Map the executable file to memory */
    arc_data = MapExistingFile (modulename, &arc_size);
    if (!arc_data) {
	SystemError (GetLastError(), "Could not open file %s", modulename);
	return 1;
    }

    /* Extract the configuration data into a temporary file */
    ini_file = ExtractIniFile (arc_data, arc_size);
    if (!ini_file) {
	return 1;
    }

    /* Read installation information */
    GetPrivateProfileString ("Setup", "title", "", title,
			     sizeof (title), ini_file);
    unescape (title);
    GetPrivateProfileString ("Setup", "pthname", "", pthname,
			     sizeof (pthname), ini_file);
    unescape (title);
    GetPrivateProfileString ("Setup", "info", "", info,
			     sizeof (info), ini_file);
    unescape (info);
    pyc_compile = GetPrivateProfileInt ("Setup", "pyc_compile", 0, ini_file);
    pyo_compile = GetPrivateProfileInt ("Setup", "pyo_compile", 0, ini_file);
    vers_minor = GetPrivateProfileInt ("Setup", "vers_minor", 0, ini_file);

    hwndMain = CreateBackground (title);

    InitCommonControls ();
    DialogBox (hInst, MAKEINTRESOURCE (IDD_DIALOG1), hwndMain, DlgProc);

    /* Clean up */
    UnmapViewOfFile (arc_data);
    if (ini_file)
	DeleteFile (ini_file);

    return 0;
}
