#########################3
##Disclaimer
##==========
##
## This  software is provided without warranty of any kind.
## No liability is taken for any loss or damages, direct or
## indirect, that may result through the use of it. No warranty
## is made with respect to this documantation, or the programs
## and functions therein.
## There are no warranties that the programs or their documentation
## are free of error, or that it is consistent with any standard,
## or that it will meet the requirement for a particular application.
 ## Copyright
## =========
##
## We have adopted the GNU LIBRARY GENERAL PUBLIC LICENSE to apply to
## the this software.
## For more information on the license terms, see
## http://www.gnu.org/copyleft/lgpl.txt
    
 ## Author : Alessandro MIRONE mirone@esrf.fr
                                                                   




import  Numeric 
import  Matrix 
import  copy
import  LinearAlgebra
import arrayfns
from math import cos,sin
import math
import os




####################################################
# Returns 1 ,-1 or zero
# depending on the ciclicity of the a,b,c list.
# Utility function.
# If a,b,c can be reordered in a monotonic arrangement 
# by an even number of permutations retuns 1
# If the number is odd returns -1
# If two numbers are equal retorns zero
def ciclicity(a,b,c):
  if( a==b or  a==c or c==b):
    return 0
  elif(  (b-a)*(c-b)*(a-c) <0  ):
    return 1
  else:
    return -1                                                             

###################################
# Utility function.
# Return simply the element-wise difference 
# of two tuples : a-b
def difftuple(a,b):
  res=()
  for i in range(0,len(a)):
    res=  res+  (a[i]-b[i],)
  return res

######################################""
# utility function.
# Return the intersection of two lists.
# Uses a dictionary to go faster
#

def intersection(list1, list2):
  int_dict = {}
  list1_dict = {}
  for e in list1: 
    list1_dict[e] = 1
  for e in list2:
     if list1_dict.has_key(e):
         int_dict[e] = 1
  return int_dict.keys()


###########################################################
# This function test the equality between two OP_cella
# objects. The OP_cella is defined further

def OP_isDifferent(cella, cellb):

 #################################
 # First find the linear combination of axis that gives the shift.
 # The cell shifts are decomposede on cella.vectors basis

 A      = Numeric.transpose(cella.cellvectors)
 A      = LinearAlgebra.inverse (A )
 coeffa = Numeric.dot(A,cella.shift) 
 coeffb = Numeric.dot(A,cellb.shift)

 ############################################################
 # calculates the relative shift of cellb relative to cella
 # in units of the cellb.vectors

 cellb.unitsshift=Numeric.zeros(3,Numeric.Float32)

 # the differences are modulus 1. (periodicity)

 for i in range(0,3):
   cellb.unitsshift[i]= (((coeffb-coeffa)[i]+0.5) %1.0) -0.5
 if(Numeric.sum(abs(cellb.unitsshift))>0.1):
   return 1  # they are different

 diff=abs(cella.cellvectors-cellb.cellvectors)
 diff=Numeric.sum(Numeric.sum(diff))
 if(diff>0.1): 
   return 1

# if we arrive here, it is really the same cella
# So they are not different

 return 0

# end of OP_isDifferent(cella, cellb)

############################################################
# class  OP_cella
# This class stores things like atom names, their positions,
# cell axis, shift ......
# it has methods for rotation, reflection,
# to calculate Fourier transform, for finding the way one can rotate a set
# of atoms to another set of atoms, for traking interactions and so on......

class OP_cella:
  #########################################################"
  # Constructor of OP_cella.
  # Here one passes  the relevant parameter (see default list)
  def __init__(self, 
                     cellvectors=Numeric.array( [ (1.0,0.0,0.0),(0.0,1.0,0.0), (0.0,0.0,1.0) ] ),
                     AtomNames=['a','b'],
                     PositionsList=[Numeric.array(  [ Numeric.array((0.0,0.0,0.0)),Numeric.array((0.5,0.,0.))]    ),
                                    Numeric.array( (Numeric.array((1.0,0.5,0.5)),)  )
                                   ]
      ):
      self.cellvectors=cellvectors
      self.AtomNames  =AtomNames
      self.PositionsList =(PositionsList)
      self.shift=Numeric.array([0.0,0.0,0.0])
      self.unitsshift=Numeric.zeros(3,Numeric.Float32)
      self.reflection=1
      self.atomlist=[]
      for i in range(0, len(AtomNames)):
       for j in range(0, len(PositionsList[i])):
          self.atomlist.append( (AtomNames[i],j ) )

  ####################################################
  # OP_cella.Reflection(self)
  # Reflection reflects all the position on the cell origin.
  # The shift is left untouched because it is the shift of the origin itself.

  def Reflection(self):
     self.cellvectors=(-1.0)*self.cellvectors
     self.reflection=-self.reflection  
     for i in range(0, len(self.PositionsList)):
        self.PositionsList[i]=(-1.0)*self.PositionsList[i]

  ##########################################################
  # OP_cella.getK(self, N=2)
  # Returns an array of 3D vectors, that are spanned by
  # integer multiples of cell's Brillouin vectors.
  # integers run form -N to N.
  # (the vectors are already multiplied by imaginary unit
  #  in order to be ready for fourier transforming)
  
  def getK(self, N=2):
      AntiSymm= Numeric.array([ [ [ ciclicity(i,j,k) for k in range(0,3) ] for j in range (0,3) ] for i in range(0,3) ])
      cellVolume=Numeric.dot( Numeric.dot(AntiSymm, self.cellvectors[2]),self.cellvectors[1] )
      cellVolume=Numeric.dot(self.cellvectors[0],cellVolume)
      Brillvectors=Numeric.zeros([3,3], Numeric.Float64)
      Brillvectors[0]=2 * Numeric.pi * Numeric.dot( Numeric.dot(AntiSymm, self.cellvectors[2]),self.cellvectors[1] )/ cellVolume
      Brillvectors[1]=2 * Numeric.pi * Numeric.dot( Numeric.dot(AntiSymm, self.cellvectors[0]),self.cellvectors[2] )/ cellVolume
      Brillvectors[2]=2 * Numeric.pi * Numeric.dot( Numeric.dot(AntiSymm, self.cellvectors[1]),self.cellvectors[0] )/ cellVolume
      K=[  Numeric.dot(  [ i,j,k],  Brillvectors  )  for k in range(-N,N+1) for j in range (-N,N+1) for i in range(-N,N+1) ]
      K=Numeric.array(K)*complex(0,1.0)
      self.Brillvectors=Brillvectors
      self.volume= cellVolume
      return K

  ###########################################################
  # OP_cella.getFourier(self, K)
  # return an array (for each atom kind(element))
  # of arrays (for each vector in K) of fourier component
  # Dirac delta are put on the atoms locations.

  def getFourier(self, K):
      res=[]
      for positions in self.PositionsList:
         resAtom=Numeric.zeros(len(K))
         for pos in positions:
               pos=pos+self.shift
               resAtom=resAtom+Numeric.exp(Numeric.dot(  K , pos ) )
         res.append(resAtom)
      return Numeric.array(res)

  ################################################################################
  # OP_cella.Transform(self,Az1, Ax, Az2   ,DPOS )
  # Rotates by the three Eulerian angles Az1, Ax, Az2 around
  # the x and z axis through the origin and shifts the origin by DPOS

  def Transform(self,Az1, Ax, Az2   ,DPOS ):

      rotAz1=Numeric.array([ [cos(Az1)  , -sin(Az1) , 0  ],
                     [sin(Az1)  ,  cos(Az1) , 0  ],
                     [       0  ,         0 , 1 ]
                ]
               )
      rotAx=Numeric.array([ [ 1 ,  0  , 0  ],
                    [ 0 ,  cos(Ax) , -sin(Ax) ],
                    [ 0 ,  sin(Ax)  ,cos(Ax)  ]
                ]
               )
      rotAz2=Numeric.array([ [cos(Az2)  , -sin(Az2) , 0  ],
                           [sin(Az2)  ,  cos(Az2) , 0  ],
                           [       0  ,         0 , 1 ]
                ]
               )
      rot=Numeric.dot( rotAz2, Numeric.dot(rotAx,rotAz1) )

      self.TransformByMatrix( rot ,DPOS )

  ################################################################################
  # OP_cella.TransformByMatrix(self, rot ,DPOS )
  # Rotates by the rot matrix and shifts the origin by DPOS

  def TransformByMatrix(self, rot ,DPOS ):

      ####################################################
      # Please to notice that as the vectors
      # are stored row by row in self.cellvectors
      # we have to rotate  a row at once.
      # ... we cannot do Numeric.dot(rot,   self.cellvectors)

      for i in range(0, len(self.cellvectors)):
        self.cellvectors[i]=Numeric.dot(rot,   self.cellvectors[i])

      for ka in range(0, len(self.AtomNames)):
         L=self.PositionsList[ka]
         for i in range(0,len(L)):
            L[i]=Numeric.dot(rot,   L[i])

      # stores the most recent rotation parameters
      self.shift=DPOS       
      self.Rotation=rot

  ############################################################
  # OP_cella.computeDistances(self, maxdist)
  # It builds up a Big multiple dictionary, namely
  #
  # self.distances[Aa][Bb][kA][kB][(DX,DY,DZ)]
  #
  # where Aa and kA are the atom name ( for example "Si",..)
  #  and the position in the array of vectors in cell.PositionsList
  # starting from 0. Same thing for Bb and kB.
  # (DX,DY,DZ) are integer numbers specifying the cell containing (Bb,kB)
  # relatively to (Aa,kA).
  # DX,DY,DZ run from -N to N 

  def computeDistances(self, maxdist, N=2):
    self.distances={}
    for iAa in range(0,len(self.AtomNames)):
       Aa=self.AtomNames[iAa]
       self.distances[Aa]={}
       for iBb in range(0,len(self.AtomNames)):
          Bb=self.AtomNames[iBb]
          self.distances[Aa][Bb]={}
          for kA in range(0, len(self.PositionsList[iAa])):
             self.distances[Aa][Bb][kA]={}

             for kB in range(0, len(self.PositionsList[iBb])):
                 self.distances[Aa][Bb][kA][kB]={}
                 for BX in range(-N,N+1):
                   for BY in range(-N,N+1):
                     for BZ in range(-N,N+1):
                       P1= self.PositionsList[iAa][kA]
                       P2= self.PositionsList[iBb][kB] +Numeric.dot( [BX,BY,BZ]  , self.cellvectors       ) 
                       dist=math.sqrt( Numeric.sum( (P2-P1)*(P2-P1)   )  ) 
                       if(dist<maxdist):                  
                          self.distances[Aa][Bb][kA][kB][(BX,BY,BZ)]=[ dist  , P1+self.shift,P2+self.shift,[] ]

  ##################################################################################
  # OP_cella. FindTetragons(self, Latoms, CellsCoo=[(0,0,0),(0,0,0),(0,0,0),(0,0,0)])
  # This is the key routine to find good candidates to symmetry group.
  # Latoms is a list of the kind [  (Aa,kA) ,(Bb,kB)....] 
  # ( i.e. couples formed by atom-name and position in the position list.
  # Latoms must be formed of four  atoms defining a non-degenerate tetraedron.
  # A check is permormed on the non-degeneracy.
  # The funtions finds all the possible equivalent tetraedrons (which have the same
  # set of distances, and the same atom kinds)
  # The function return the list of all these tetraedrons
 
  def FindTetragons(self, Latoms, CellsCoo=[(0,0,0),(0,0,0),(0,0,0),(0,0,0)]):
    Positions= self.FindPositions(Latoms, CellsCoo=CellsCoo )

    listindexes= [ self.AtomNames.index(Latoms[i][0])  for i in range(0,len(Latoms))]

    vectors = Positions[1:4]-Positions[0]
    #####
    # check for non-degeneracy
    det= LinearAlgebra.determinant(vectors)
    if(abs(det)<0.00001):
        print " ERROR : DETERMINANT IS ", det, " PROBABLE DEGENERACY "
        os.exit() 
    # CellsCoo=[(0,0,0),(0,0,0),(0,0,0),(0,0,0) ]
    CooList=[(i,j,k) for i in range(-2,3) for j in range(-2,3)    for k in range(-2,3)   ]
    (constraintNames,constraintTripod,constraintCircle)=self.FindConstraints(Latoms, CellsCoo=CellsCoo)
    cN=constraintNames
    res=[(Latoms,CellsCoo )]
    for iA0 in range(0, len(self.PositionsList[listindexes[0]]) ):
      print iA0
      Coo0=(0,0,0)
      for iA1 in range(0, len(self.PositionsList[listindexes[1]]) ):
        for Coo1 in CooList:
           d= self.distances[cN[0]][cN[1]][iA0 ][iA1 ][difftuple(Coo1, Coo0)][0]
           if ( abs(d-constraintTripod[0])>0.0001 ): continue
           for iA2 in range(0, len(self.PositionsList[listindexes[2]]) ):

             for Coo2 in CooList:
               d= self.distances[cN[0]][cN[2]][iA0 ][iA2 ][difftuple(Coo2, Coo0)][0]
               if ( abs(d-constraintTripod[1])>0.0001 ): continue
               for iA3 in range(0, len(self.PositionsList[listindexes[3]]) ):

                 for Coo3 in CooList:

                    d= self.distances[cN[0]][cN[3]][iA0 ][iA3 ][difftuple(Coo3, Coo0)][0]
                    if ( abs(d-constraintTripod[2])>0.0001 ): continue

                    LatomsNew=[  (constraintNames[i] ,iA ) for i,iA in [(0,iA0),(1,iA1),(2,iA2),(3,iA3)]  ]
                    (NamesNew,TripodNew,CircleNew)=self.FindConstraints(LatomsNew, [Coo0,Coo1,Coo2,Coo3] )
                    if( Numeric.sum( abs(TripodNew-constraintTripod)+ 
                         abs(CircleNew-constraintCircle) )<0.0001):
                      res.append((LatomsNew, [Coo0,Coo1,Coo2,Coo3] ))
                      PositionsNew= self.FindPositions(    LatomsNew, [Coo0,Coo1,Coo2,Coo3]      )
                      vectorsNew = PositionsNew[1:4]-PositionsNew[0]
    return res  
         

  ###############################################################################
  #  The next two funtion are auxilary funtions for 
  #  Findtetragons 

  def FindPositions(self, Latoms, CellsCoo=[(0,0,0),(0,0,0),(0,0,0),(0,0,0)]):
    Positions=Numeric.zeros([4,3], Numeric.Float64)
    for i in range(0,4):
      Positions[i]=self.PositionsList[self.AtomNames.index(Latoms[i][0])][  Latoms[i][1]      ]
      Positions[i]=Positions[i]+Numeric.dot(CellsCoo[i],self.cellvectors)
    return Positions

  def FindConstraints(self, Latoms, CellsCoo=[(0,0,0),(0,0,0),(0,0,0),(0,0,0)] ):
    constraintNames=[Latoms[i][0] for i in range(0,4) ]
    La=Latoms
    constraintTripod=Numeric.array([ self.distances[La[0][0]][La[i][0]][La[0][1] ][La[i][1] ][CellsCoo[i]][0]
                         for i in range(1,4)    ]  )

    constraintCircle=Numeric.array([ self.distances[La[i][0]][La[( i)%3 +1  ][0]][La[i][1] ]
                                      [La[(i)%3 +1][1] ][difftuple(CellsCoo[i%3+1], CellsCoo[i])][0]
                         for i in range(1,4)      ])

    return (constraintNames,constraintTripod,constraintCircle)



  def FindPosition(self, Latoms, CellCoo=(0,0,0)):
    Position=self.PositionsList[self.AtomNames.index(Latoms[0])][  Latoms[1]      ]
    Position=Position +Numeric.dot(CellCoo,self.cellvectors)
    return Position


  
  ###############################################################
  # OP_cella.FindEquivalences(self, CellsList)
  # This function should receive a list of cells (CellList)
  # that are superpose as they are, atom by atom to self.
  # (program stops if this condition is not satisfied)
  # It sets  self.equivalences containing a list of lists containing set of equivalent atoms in the format [(Aa,kA),...],
  #          for each cell atom. It is exactly a multiplication table for the symmetry group
  #
  #          self.grouplist    containing the names given to the groups (like Cu_G1 )
  #          self.Groups       a list of groups. Each group contains the equivalent atoms in the format (Aa,kA)
  #          self.DGroups      self.DGroups[(Aa,kA)] gives the name (a name like Cu_G1 ) to which atom (Aa,kA) belongs

   
  def FindEquivalences(self, CellsList):
   equivalences={}
   for i in range(0, len(self.AtomNames)):
      name=self.AtomNames[i]
      positions =self.PositionsList[i]
      for iA in range(0, len(positions)):
         EqAtoms=[]
         equivalences[ (name,iA) ]= EqAtoms 
         for cell in  CellsList:
            eqa=cell.FindEquivalentOf(self.PositionsList[i][iA]+self.shift,  name )
            if(eqa==None):
              print "BIG Problem"
              print " Found no equivalent atom in a cella that should be equivalent"
              os.exit()
            EqAtoms.append(eqa) 
   self.equivalences= equivalences

   Groups=[]
   grouplist=[]
   DGroups={}
   for i in range(0, len(self.AtomNames)):
      name=self.AtomNames[i]
      NGr=[]
      for iA in range(0, len(self.PositionsList[i] )):
        Na= (name,iA)
        ngr=0
        for group in NGr:
           if( Na in   group):          
             break
           if( intersection(self.equivalences[Na], group) !=[] ):
             group.append(Na)
             DGroups[Na]="%s_G%d" %(name,ngr)
             break
           ngr=ngr+1

        else:
           NGr.append([Na])
           DGroups[Na]="%s_G%d" %(name,ngr)
           grouplist.append("%s_G%d" %(name,ngr))
      Groups=Groups+NGr

   self.grouplist=grouplist
   self.Groups=Groups
   self.DGroups=DGroups

       
  ####################################################
  # utility funtion used by FindEquivalences
             
  def FindEquivalentOf(self, position,  name ):
     A      = Numeric.transpose(self.cellvectors)
     A      = LinearAlgebra.inverse (A )
     positions =self.PositionsList[self.AtomNames.index(name)]
     for iA in range(0, len(positions)):
       pos=positions[iA]+self.shift
       diff = Numeric.dot(A,  position-pos ) 
       for i in range(0,3):
         diff[i]= ((diff[i]+0.5) %1.0) -0.5
         if(Numeric.sum(abs(diff))<0.0001):
            return((name,iA))
     return None
       


  ###################################################
  # OP_cella.printEquivalences(self)
  # prints in a more readeble format
  #     self.equivalences
  #     self.DGroups

  def printEquivalences(self):
    print "TABLE D'EQUIVALENCES"
    for i in range(0, len(self.AtomNames)):
      name=self.AtomNames[i]
      positions=self.PositionsList[i]
      for iA in range(0, len(positions)):
         print "%2s%d  ----->  " %(name,iA),
         EqAtoms=self.equivalences[ (name,iA) ]
         count=0
         for at in EqAtoms:
            print "   %2s%d  " % at,
            count=count+1
            if(count%8 ==0):
             print ""
             print "             ",
         print ""
    print ""
    for group in self.Groups:
      prefix="    "
      print self.DGroups[group[0]]," : ",
      for Na in group:
        print  "%s" % prefix,   "%2s%d " % Na ,
        prefix=" == "       
      print "\n"


  ################################################
  #  OP_cella.computeHistograms(self, DR=0.1)
  #  for all the couples of atom groups g1,g2, creates a multiple dictionary
  #  histo[g1][g2]
  #  giving a list of items in the format [ [ r1  ,r2  , Ninteractions)....
  #   where Ninteractions are the number of interactions between an atom of g1
  #   and one of g2 at a distance between r1 and r2, r1-r2 being the maximum interval smaller
  # than DR

  def computeHistograms(self, DR=0.1):
     distances={}
     where={}
     histo={}
     for g1 in self.grouplist:
       distances[g1]={}
       where[g1]={}
       histo[g1]={}
       for g2 in self.grouplist:
          distances[g1][g2]=[]
          where[g1][g2]={}
          histo[g1][g2]=[]

     for iAa in range(0,len(self.AtomNames)):
      Aa=self.AtomNames[iAa]
      for iBb in range(0,len(self.AtomNames)):
       Bb=self.AtomNames[iBb]
       for kA in range(0, len(self.PositionsList[iAa])):
        for kB in range(0, len(self.PositionsList[iBb])):
         g1=self.DGroups[(Aa,kA)]
         g2=self.DGroups[(Bb,kB)]
         lista=distances[g1][g2]
         lw= where[g1][g2]
         dictio=self.distances[Aa][Bb][kA][kB]
         for Coo in dictio.keys():
           lista.append(dictio[Coo][0])
           lw[dictio[Coo][0]]=("like %s%d in 0 and %s%d in %d %d %d" % (Aa,kA,Bb,kB,Coo[0],Coo[1],Coo[2]))
     for g1 in self.grouplist:
       histo[g1]={}
       for g2 in self.grouplist:
          histo[g1][g2]=[]
          distances[g1][g2] = Numeric.sort(distances[g1][g2])
          lw= where[g1][g2]
          l=     distances[g1][g2]
          Rinf=-100
          hi=histo[g1][g2]
          tok=None
          for r in l:
            if(r<0.0001): continue
            if(r-Rinf>DR):
              if(tok!=None): hi.append(tok)
              Rinf=r
              tok=[Rinf,r,1,lw[ r  ]]
            else:
              tok[1]=r
              tok[2]=tok[2]+1
          hi.append(tok)
     self.histo=histo
     for g1 in self.grouplist:
       for g2 in self.grouplist:
         for h in histo[g1][g2]:
           h[2]=h[2]/2
     

  ####################################
  # print self.histo
  #
  def  printHisto(self,n):
    for i in range(0,len(self.grouplist)):
      for j in range(i ,len(self.grouplist)):
        g1=self.grouplist[i]
        g2=self.grouplist[j]
        print g1, "---",g2," ",
        h=self.histo[g1][g2]
        count=0
        for t in h:
           print "  %d in [%4f,%4f];" %(t[2],t[0],t[1]),
           count+=1
           if(count==n): break
        print " "
        

  def printParametersInput(self,filename, n=2):
    fd=open(filename,"w")
    
    fd.write("# This first part is just an initialisation\n")
    fd.write("# Edit below \n")
    fd.write("M_={}\nZ_={}\nY_={}\nK_={}\nVWs_={}\nLJs_={}\nBMs_={}\n")
    fd.write("BK_L={}\nSM_L={}\nBK_T={}\nSM_T={}\nTens_P={}\nTens_S={}\nTens_T={}\nTens_YT={}\nangleBonds={}\nFunction_YT=None\nSCREENING_FUNCTION=None\n")
    for i in range(0,len(self.grouplist)):
      g1=self.grouplist[i]
      fd.write("BK_L['%s']={}\nSM_L['%s']={}\nBK_T['%s']={}\nSM_T['%s']={}\nTens_P['%s']={}\nTens_S['%s']={}\nTens_T['%s']={}\nTens_YT['%s']={}\n"%( (g1,)*8 ))
      for j in range(0 ,len(self.grouplist)):
        g2=self.grouplist[j]
        fd.write("SM_L['%s']['%s']={}\nSM_T['%s']['%s']={}\n"%( (g1,g2)*2 ))

    fd.write("# HHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHH\n")
    fd.write("# Edit the followin part\n")

    fd.write("# Masses : M \n")
    for i in range(0,len(self.grouplist)):
      fd.write("M_['%s']=0.0\n" % self.grouplist[i])

    fd.write("# Coulomb interaction : Z \n")
    for i in range(0,len(self.grouplist)):
      fd.write("Z_['%s']=0.0\n" % self.grouplist[i])

    fd.write("\n# Shells Model : shell charge \n")
    for i in range(0,len(self.grouplist)):
      fd.write("Y_['%s']=0.0\n" % self.grouplist[i])

    fd.write("\n# Shells Model : polarizability  \n")
    for i in range(0,len(self.grouplist)):
      fd.write("K_['%s']=1000000.0\n" % self.grouplist[i])

    fd.write("\n# Van der Walls  : sigma  \n")
    fd.write("VW_V0=0.0\n")
    for i in range(0,len(self.grouplist)):
      fd.write("VWs_['%s']=1.0\n" % self.grouplist[i])


    fd.write("\n# Lenard Jones   : sigma  \n")
    fd.write("LJ_V0=0.0\n")
    for i in range(0,len(self.grouplist)):
      fd.write("LJs_['%s']=1.0\n" % self.grouplist[i])


    fd.write("\n# Born-Mayer  : sigma  \n")
    fd.write("BM_V0=0.0\n")
    for i in range(0,len(self.grouplist)):
      fd.write("BMs_['%s']=1.0\n" % self.grouplist[i])


    for i in range(0,len(self.grouplist)):
      g1=self.grouplist[i]
      for j in range(i ,len(self.grouplist)):
        g2=self.grouplist[j]
        h=self.histo[g1][g2]
        fd.write(  "# XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX \n"  )
        fd.write(  "# Interactions between '%s' and '%s' \n"  % (g1,g2) )
        count=0
        for t in h:
           if(count==n): break
           fd.write(  "# Shell N%d going from %f to %f  %s \n" %  (count,t[0],t[1],t[3]))
           count+=1
        fd.write("BK_L['%s']['%s']=[]\n" %(g1,g2) )
        fd.write("BK_T['%s']['%s']=[]\n" %(g1,g2) )
        fd.write("SM_L['%s']['%s']['SS']=[]\n" %(g1,g2) )
        fd.write("SM_T['%s']['%s']['SS']=[]\n" %(g1,g2) )
        fd.write("SM_L['%s']['%s']['SC']=[]\n" %(g1,g2) )
        fd.write("SM_T['%s']['%s']['SC']=[]\n" %(g1,g2) )
        fd.write("SM_L['%s']['%s']['CS']=[]\n" %(g1,g2) )
        fd.write("SM_T['%s']['%s']['CS']=[]\n" %(g1,g2) )
        if(g1!=g2):
          fd.write("# just for symmetry \n")
          fd.write("SM_L['%s']['%s']=SM_L['%s']['%s']\n" %(g2,g1,g1,g2) )
          fd.write("SM_T['%s']['%s']=SM_T['%s']['%s']\n" %(g2,g1,g1,g2) )
          fd.write("BK_L['%s']['%s']=BK_L['%s']['%s']\n" %(g2,g1,g1,g2) )
          fd.write("BK_T['%s']['%s']=BK_T['%s']['%s']\n" %(g2,g1,g1,g2) )
           
    for letter in ["P","T","S","YT"]:
     for i in range(0,len(self.grouplist)):
      g1=self.grouplist[i]
      for j in range(i ,len(self.grouplist)):
        g2=self.grouplist[j]
        h=self.histo[g1][g2]
        fd.write(  "# XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX \n"  )
        fd.write(  "# Tensorial Interactions between '%s' and '%s' \n"  % (g1,g2) )
        count=0
        for t in h:
           if(count==n): break
           fd.write(  "# Shell N%d going from %f to %f  %s \n" %  (count,t[0],t[1],t[3]))
           count+=1
        fd.write("Tens_%s['%s']['%s']=[]\n" %(letter, g1,g2) )
        if(g1!=g2):
          fd.write("# just for symmetry \n")
          fd.write("Tens_%s['%s']['%s']=Tens_%s['%s']['%s']\n" %(letter,g2,g1,letter,g1,g2) )
           



  def __deepcopy__(self, memo):
     res=OP_cella()
     res.cellvectors=Numeric.array(self.cellvectors     ,  copy=1  )
     res.AtomNames  =copy.copy(self.AtomNames              )
     res.PositionsList = copy.copy(self.PositionsList)
     for i in range(0, len(res.PositionsList)):
        res.PositionsList[i]=Numeric.array(res.PositionsList[i],  copy=1 )
     res.shift=self.shift 
     return res


        

class  OP_ComparisonCellRotatedCell:
  def __init__(self, Cell, Az1, Ax, Az2,DX,DY,DZ, reflection=0):
      self.Cell=Cell
      self.K=Cell.getK()
      self.Az1=Az1
      self.Ax=Ax
      self.Az2=Az2
      self.DX =DX
      self.DY =DY
      self.DZ =DZ
      self.Fourier=Cell.getFourier(self.K)
      self.reflection=reflection


  def error(self ):
      newCell=copy.deepcopy(self.Cell)
      if(self.reflection):
        newCell.reflection()
      newCell.Transform(Minimiser.par(self.Az1), Minimiser.par(self.Ax), Minimiser.par(self.Az2), 
                        Numeric.array([Minimiser.par(self.DX), Minimiser.par(self.DY), Minimiser.par(self.DZ)] ) )

      newFourier=newCell.getFourier(self.K)
   
      diffFourier=newFourier-self.Fourier
 
      diffFourier=  (diffFourier* Numeric.conjugate(diffFourier)).real  

      res=Numeric.sum(Numeric.sum(diffFourier))
#      print res
      return res

def OP_FindSymmetries(cella):
  cellList=[cella]
  paraList=[(0.0,0.0,0.0, 0.0, 0.0, 0.0, 1.0 )]
  cella.Transform(0.0,0.0,0.0, 
                  Numeric.array([0.0, 0.0, 0.0] ) )
  K=cella.getK()
  az1=Minimiser.Variable(0,-6*Numeric.pi, 6*Numeric.pi)
  ax =Minimiser.Variable(0,-6*Numeric.pi, 6*Numeric.pi)
  az2=Minimiser.Variable(0,-6*Numeric.pi, 6*Numeric.pi)
      
  dX=Minimiser.Variable(0.0,-40,40.0)
  dY=Minimiser.Variable(0.0,-40,40.0)
  dZ=Minimiser.Variable(0,-140,140.0)

  equalcount=0
  while(1):
    reflection=0
    comparison= OP_ComparisonCellRotatedCell(cella , az1, ax, az2, dX,dY,dZ, reflection)
    miniob=Minimiser.minimiser(comparison,[  az1, ax,  az2,  dX,  dY, dZ  ])

    map(Minimiser.Variable.setrandom,[  az1, ax,  az2,  dX,  dY, dZ  ] )

    res=miniob.amoebaAnnealed(0.0001, lambda x:700./(2.0+x), 100, arret=0.001, maxthesame=30, DEBUG=0, maxcount=20)

    if(not res):
      print " sfortunato"
    else: 
      miniob.amoeba(0.0000001, arret=0.00001) 
      newCell=copy.deepcopy(cella)
      if(reflection): newCell.reflection()
      newCell.Transform(Minimiser.par(az1), Minimiser.par(ax), Minimiser.par(az2), 
                  Numeric.array([Minimiser.par(dX), Minimiser.par(dY), Minimiser.par(dZ)] ) )
      isnew=1
      for tok in cellList:
        if ( not OP_isDifferent(tok,newCell) ):
         isnew=0
      if( isnew ):
        print " Found a new group element "
        paraList.append( ( Minimiser.par(az1), Minimiser.par(ax), Minimiser.par(az2), 
                  Minimiser.par(dX), Minimiser.par(dY), Minimiser.par(dZ), reflection)    )
        OP_TellAbout(newCell)
        equalcount=0
        cellList.append(newCell)
        print " N ELEMENTI ", len(cellList)
      else:
        print " No new group element " 
        equalcount+=1
        if(equalcount>4*len(cellList)):
          print " ALL FOUND "
          for newCell in cellList:
             OP_TellAbout(newCell)
          return paraList         



def OP_TellAbout(newCell, short="NO"):

        # findout the type of rotation
        if(isinstance(newCell, OP_cella)):
          A=newCell.Rotation *complex(1.0,0.0)
        else:
          A=newCell*complex(1.0,0.0)
        (eval, evect)=LinearAlgebra.eigenvectors(A)
        if(short=="NO"):
          print "Rotazione ="
          print A
          print " evect = "
          print evect
          print " eval="
          print eval

        # eigenvalue 1 corresponds to rotation axis
        # Find rotation axis
        for i in range(0,3):
          if (abs(1-eval[i])<0.00001): 
            rotation_axis =  evect[i]
            # Now we are ready to calculate the rotation
            # angle. We chose a different eigenvalue/vector
            # than the fixed axis. The argument of the eigenvalue
            # gives us the rotation angle. Its sign is calculated
            # considering the eigenvector : the real and imaginary part
            # are two orthogonal vectors. Their vector product times
            # the fixed axis gives the sign.
            othervector = evect[(i+1) %3 ]
            otherval    = eval [(i+1) %3 ]
            angle       = math.atan2(otherval.imag, otherval.real  )
            v1 = othervector.real*math.sqrt(2.0)
            v2 = othervector.imag*math.sqrt(2.0)
            triedron = LinearAlgebra.determinant(Numeric.array([rotation_axis,v1,v2]))
            if(triedron.real>0):
              angle=-angle      
            break
        else:
         print " something wrong finding fixed point of rotation "
         sys.exit()    
        print " Rotation Axis = ",  rotation_axis.real
        print " Angle         = ",  angle
        if(isinstance(newCell, OP_cella)):
          print " Shift         = ",  newCell.shift
          print" Shift/Axis         = ",  newCell.unitsshift
          if(newCell.reflection==-1):
            print " Reflection = YES "
          else:
            print " Reflection = NO  "



if __name__ == '__main__' :
  
# running some simple tests

      cell=OP_cella()
      K=cell.getK()

      az1=Minimiser.Variable(0,-Numeric.pi, Numeric.pi)
      ax =Minimiser.Variable(0,-Numeric.pi, Numeric.pi)
      az2=Minimiser.Variable(0,-Numeric.pi, Numeric.pi)
      
      dX=Minimiser.Variable(0,0.0,1.0)
      dY=Minimiser.Variable(0,0.0,1.0)
      dZ=Minimiser.Variable(0,0.0,1.0)


      comparison= OP_ComparisonCellRotatedCell(cell , az1, ax, az2,dX,dY,dZ)


      az1.value=1





def Talk_about_Distance_variation(evects,one_over_sqrtM,cella, Neigen):
        """ Here evect should be already multiplied by the right phase
        """
        evects =evects*one_over_sqrtM
        for g1 in cella.grouplist :
            for g2 in cella.grouplist :
                max_merit=-1.0e-50
                max_blabla=""
                for a1 in cella.Groups[cella.grouplist.index(g1)]:
                    for a2 in cella.Groups[cella.grouplist.index(g2)]:
                        if(a1==a2):
                            continue
                        index_of_a1_incellalist=cella.atomlist.index(a1)
                        index_of_a2_incellalist=cella.atomlist.index(a2)
                        vect3D_a1= evects[Neigen][ index_of_a1_incellalist*3: index_of_a1_incellalist*3+3]
                        vect3D_a2= evects[Neigen][ index_of_a2_incellalist*3: index_of_a2_incellalist*3+3]
                       
                        for Coo in [(0,0,0) , (1,1,1), (0,1,1), (1,0,1), (1,1,0) , (1,0,0),(0,1,0), (0,0,1), ]:
                            DR=cella.FindPosition(a2, CellCoo=Coo ) - cella.FindPosition(a1, CellCoo= (0,0,0))
                            dDR_real=Numeric.sum(  DR*(( vect3D_a2- vect3D_a1 ).real)   )
                            dDR_imag=Numeric.sum(  DR*(( vect3D_a2- vect3D_a1 ).imag)   )
                            dDR = math.sqrt(dDR_real*dDR_real+dDR_imag*dDR_imag)
                            dist= math.sqrt( Numeric.sum(DR*DR))
                            merit =  dDR / Numeric.sum(DR*DR)
                            if(abs(merit)>max_merit):
                                max_merit=abs(merit)
                                max_blabla=" dDR=%e for a1=(%s,%d) and a2=(%s,%d) in (%d,%d,%d) at a distance of %e" % ( (dDR,)+a1+a2+Coo+(dist,))
                print "for g1=%s and g2=%s the blablabla is %s\n" %( g1,g2,  max_blabla)               
               




