"""
    Drawable1D.py
    1D Graphics displaying widget
        
"""

from PyDVT import __version__,__date__,__author__

import qt
import pyqt_pl
import myqplplot
import Numeric
import math
from Binding import Pen,Brush
from BinarySearch import BinarySearch

OVERSIZE_POLAR=1.25

def GetSymbolCode(symbol):
    SymbolCode={ 'circle':      4 ,      \
                 'square':      0 ,      \
                 'diamond':     11,      \
                 'plus':        2 ,      \
                 'cross':       5 ,      \
                 'star':        12,      \
                 'triangle':    7        \
    }
    if symbol in SymbolCode.keys(): return  SymbolCode[symbol]
    else:                           return  SymbolCode['circle']


             

class GraphicalObj:
    def __init__(self,view,pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0"),yscale=0):
        self.view=view
        self.visible=0
        if pen is not None:
            self.pen=Pen(pen.color,pen.width,pen.style)
        else:
            self.pen=None
         
        if brush is not None:
            self.brush=Brush(brush.color,brush.style)
        else:
            self.brush=None
        self.X0=self.Y0=self.X1=self.Y1=None
        self.yscale=yscale

    def erase(self):
        if self.visible:
            self.visible=0
            self.paint()
            
    def show(self):
         if self.visible==0:
             self.visible=1
             self.paint()

    def SetCoords(self,X0=None,Y0=None,X1=None,Y1=None):
        self.view.qpl.InitDrawing()
        if (self.X0,self.Y0,self.X1,self.Y1)==(X0,Y0,X1,Y1): return        
        self.erase()
        (self.X0,self.Y0,self.X1,self.Y1)=(X0,Y0,X1,Y1)
        self.show()
        self.view.qpl.EndDrawing()

    def Update(self,pen=None,brush=None,textOptsDic=None):
        self.view.qpl.InitDrawing()
        self.erase()
        if pen is not None: self.pen=Pen(pen.color,pen.width,pen.style)
        if brush is not None: self.brush=Brush(brush.color,brush.style)
        if textOptsDic is not None:
            if textOptsDic.has_key('orient'): self.orient=textOptsDic['orient']
            if textOptsDic.has_key('align'): self.align=textOptsDic['align']
            if textOptsDic.has_key('text'): self.text=textOptsDic['text']
            if textOptsDic.has_key('style'):self.style=textOptsDic['style']
        self.show()
        self.view.qpl.EndDrawing()


    def GetCoords(self):
        if self.X1 is not None: return (self.X0,self.Y0,self.X1,self.Y1)
        else: return (self.X0,self.Y0)

    def SetColor(self,color):
        #pyqt_pl.plscol0(14,255-color[0],255-color[1],255-color[2])
        #pyqt_pl.plcol0(14)
        pyqt_pl.plrgb1(255-color[0],255-color[1],255-color[2])
    

    def GetStyle(self,style):
            if   style=="solid":    return 1
            elif style=="dashed":   return 3
            elif style=="dotted":   return 2
            else:                   return 0         


    def paint(self):
        pyqt_pl.plxormod(1)
        if self.yscale==1: pyqt_pl.plwind(self.view.realxmin, self.view.realxmax, self.view.realy2min, self.view.realy2max)        
        self._paint()
        if self.yscale==1:pyqt_pl.plwind(self.view.realxmin, self.view.realxmax, self.view.realymin, self.view.realymax)
        pyqt_pl.plwid(0)        
        pyqt_pl.plpsty(0)
        pyqt_pl.pllsty(1)
        pyqt_pl.plxormod(0)



    def _paint(self):
        pass


class RectangleObj(GraphicalObj):
    def _paint(self):
        pyqt_pl.plwid(0)
        self.SetColor(self.brush.color)

        if   self.brush.style=="fill_100":
            pyqt_pl.plpsty(0)  
        elif self.brush.style=="fill_75":
            pyqt_pl.plpsty(6)
            pyqt_pl.plpat((450,),(1000,))
        elif self.brush.style=="fill_50":
            pyqt_pl.plpsty(6)
            pyqt_pl.plpat((450,),(1500,))
        elif self.brush.style=="fill_25":
            pyqt_pl.plpsty(3)  
        else:
            pyqt_pl.plpsty(6)
            pyqt_pl.plpat((450,),(2500,))
                
        x0,y0=self.view.GetPlotPosition((self.X0,self.Y0))
        x1,y1=self.view.GetPlotPosition((self.X1,self.Y1))

        if self.brush.style!="fill_0":
            #Correct bug of plplot
            if x0<self.view.xmin and x1 <=self.view.xmax and x1 >= self.view.xmin:
                X0=self.view.xmin+(self.view.xmax-self.view.xmin)/200.
            else:
                X0=x0
            if x1>self.view.xmax and x0 <=self.view.xmax and  x0 >= self.view.xmin:
                X1=self.view.xmax
            else:
                X1=x1          
            if x0<=self.view.xmin and x1 >=self.view.xmax:
                X0=self.view.xmin+(self.view.xmax-self.view.xmin)/200.
                X1=self.view.xmax
            a1=Numeric.array([X0,X0,X1,X1],Numeric.Float)
            a2=Numeric.array([y0,y1,y1,y0],Numeric.Float)
            pyqt_pl.plfill(a1, a2)

        a1=Numeric.array([x0,x0,x1,x1,x0],Numeric.Float)
        a2=Numeric.array([y0,y1,y1,y0,y0],Numeric.Float)
        if   self.pen!=None:
            style=self.GetStyle(self.pen.style)
            if style:
                pyqt_pl.pllsty(style)
                self.SetColor(self.pen.color)
                pyqt_pl.plwid(self.pen.width)
                pyqt_pl.plline(a1, a2)


class EllipseObj(GraphicalObj):
    def _paint(self):
        pass


class LineObj(GraphicalObj):    
    def _paint(self):
        self.SetColor(self.pen.color)
        pyqt_pl.plwid(self.pen.width)
        style=self.GetStyle(self.pen.style)
        if style:        
            pyqt_pl.pllsty(style)
            x0,y0=self.view.GetPlotPosition((self.X0,self.Y0))
            x1,y1=self.view.GetPlotPosition((self.X1,self.Y1))
            a1=Numeric.array([x0,x1],Numeric.Float)
            a2=Numeric.array([y0,y1],Numeric.Float)
            pyqt_pl.plline(a1, a2)
        

            
class TextObj(GraphicalObj):
    def __init__(self,view,text, align="left",orient=0.,pen=Pen((0,0,0),1,"solid"),brush=None,style="normal",yscale=0):
        GraphicalObj.__init__(self,view,pen,brush,yscale)
        self.text=text
        self.align=align
        self.orient=orient
        self.style=style

    def _paint(self):
        style=self.GetStyle(self.pen.style)
        if style:        
            self.SetColor(self.pen.color)
            pyqt_pl.pllsty(style)
            if   self.align=="left":     just=0.0
            elif self.align=="right":    just=1.0
            elif self.align=="center":   just=0.5
            else:                        just=0.0
            if self.style=="bold":
                pyqt_pl.plwid(2)
                pyqt_pl.plfont(1)
            elif self.style=="italic":
                pyqt_pl.plwid(1)
                pyqt_pl.plfont(3)
            else:
                pyqt_pl.plwid(1)
                pyqt_pl.plfont(1)
            ang= float(self.orient) * math.pi/ 180.
            dx= math.cos(ang)*(self.view.xmax-self.view.xmin)
            dy= math.sin(ang)*(self.view.ymax-self.view.ymin)
            x,y=self.view.GetPlotPosition((self.X0,self.Y0))
            pyqt_pl.plptex (x,y,dx,dy, just , self.text)

class SymbolObj(GraphicalObj):
    def __init__(self,view,symbol="circle",pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0"),yscale=0):
        GraphicalObj.__init__(self,view,pen,brush,yscale)
        self.symbol=symbol

    def _paint(self):
        self.SetColor(self.pen.color)
        pyqt_pl.plwid(self.pen.width)
        x,y=self.view.GetPlotPosition((self.X0,self.Y0))
        ax=Numeric.array([x],Numeric.Float)
        ay=Numeric.array([y],Numeric.Float)
        pyqt_pl.plpoin(ax, ay, GetSymbolCode(self.symbol) )



class Drawable_1D(qt.QGrid):
    def __init__(self,parent,zoom_mode="ON",scroll_mode="ON"):
        self.back_color = qt.QColor()
        self.back_color.setRgb( 0xFF, 0xFF, 0xFF )
        if scroll_mode!="OFF":
            qt.QGrid.__init__(self,2,parent,None)
            self.qpl=  myqplplot.MyQplplot(self,back_color=self.back_color)
            self.VScroll=qt.QScrollBar(qt.Qt.Vertical,self)
            self.HScroll=qt.QScrollBar(qt.Qt.Horizontal,self)
            if scroll_mode=="AUTO":
                self.VScroll.hide()
                self.HScroll.hide()
                self.qpl.setSizePolicy(qt.QSizePolicy(qt.QSizePolicy.Expanding,qt.QSizePolicy.Expanding))
                self.VScroll.setSizePolicy(qt.QSizePolicy(qt.QSizePolicy.Fixed,qt.QSizePolicy.Expanding))
                self.HScroll.setSizePolicy(qt.QSizePolicy(qt.QSizePolicy.Expanding,qt.QSizePolicy.Fixed))
        else:
            qt.QGrid.__init__(self,1,parent,None)
            self.qpl=  myqplplot.MyQplplot(self,back_color=self.back_color)
        
        self.parent=parent
        self.ScrollMode=scroll_mode
        self.Gridlines=0
        
        self.init=1
        self.functions={}

        self.SetDrawPos(0.12, 0.88, 0.12, 0.88)
        
        self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot=0,1,0,1,None,None
        self.xmin, self.xmax, self.ymin, self.ymax,self.y2min, self.y2max=self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot
        self.ZoomedX=0
        self.ZoomedY=0

        self.LogX=0
        self.LogY=0
        self.PolarEnv=0
         
        if self.ScrollMode!="OFF":
            self.HScroll.setMinValue(0)
            self.HScroll.setMaxValue(0)
            self.HScroll.setEnabled(0)
            self.VScroll.setMinValue(0)
            self.VScroll.setMaxValue(0)
            self.VScroll.setEnabled(0)

        self.style="Line"

        self.TitleLabel=self.XLabel=self.Y1Label=self.Y2Label=""
        self.GraphicObjects=[]        
        
    def Show(self):
       self.show()

    def GetHeight(self):
        return self.height()

    def GetWidth(self):
        return self.width()

    def ToggleGridlines(self):
        if self.Gridlines: self.Gridlines=0
        else: self.Gridlines=1
        self._Refresh()


    def SetLabels(self,title_label=None,x_label=None,y_label=None,y2_label=None):
        if title_label is not None: self.TitleLabel=title_label
        if x_label is not None: self.XLabel=x_label
        if y_label is not None: self.Y1Label=y_label
        if y2_label is not None: self.Y2Label=y2_label

            
    def SetEnv(self,xmin,xmax,ymin,ymax,y2min=None,y2max=None,reset_zoom=0):
       if reset_zoom or (self.xmintot, self.xmaxtot,self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot)!=(xmin,xmax,ymin,ymax,y2min,y2max):
           self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot,self.y2mintot, self.y2maxtot=xmin,xmax,ymin,ymax,y2min,y2max
           if self.PolarEnv:
               self.xmintot, self.xmaxtot, self.ymintot, self.ymaxtot=self.xmintot*OVERSIZE_POLAR, self.xmaxtot*OVERSIZE_POLAR, self.ymintot*OVERSIZE_POLAR, self.ymaxtot*OVERSIZE_POLAR
           self.ResetZoom(0)

    def SetLogX(self,on,minvalue=1):
        self.LogX=on
        self.minvaluelogx=minvalue
        self.ResetZoom(0)
        
        
    def SetLogY(self,on,minvalue=0.000001):
        self.LogY=on
        self.minvaluelogy=minvalue
        self.ResetZoom(0)
        
    ###Add to TkBinding
    def SetPolarEnv(self,value):
        self.PolarEnv=value

    def SetDrawPos(self,xmin,xmax,ymin,ymax):
       self.xposmin,self.xposmax,self.yposmin,self.yposmax=xmin,xmax,ymin,ymax

    def RemoveGraphicObjects(self):
        self.GraphicObjects=[]
       
    def RemoveFunction(self,name=None):
        if name==None:
            self.functions={}
        else:
            if name in self.functions.keys():  del self.functions[name]
            
    def SetFunction(self,name,function_pars):
        self.functions[name]=function_pars

    def SetFunctionItem(self,name,item,val):
        if name in self.functions.keys():
            self.functions[name][item]=val

    def hscroll( self, value ):
        window=self.xmax-self.xmin
        xmintot,xmaxtot,ymintot,ymaxtot,y2mintot,y2maxtot=self._GetPlotLimits()
        self.xmin=(float(value)*((float(xmaxtot)-(window))-xmintot)/999.)+xmintot
        self.xmax=self.xmin+window
        if y2mintot != None and y2maxtot != None:
            self.y2min=(float (self.ymin)-ymintot) * (float(y2maxtot)-y2mintot) /  (float(ymaxtot)-ymintot) +y2mintot
            self.y2max=(float (self.ymax)-ymintot) * (float(y2maxtot)-y2mintot) /  (float(ymaxtot)-ymintot) +y2mintot        
        self._Refresh()

    def vscroll( self, value ):
        window=self.ymax-self.ymin
        xmintot,xmaxtot,ymintot,ymaxtot,y2mintot,y2maxtot=self._GetPlotLimits()
        value=999-value
        self.ymin=(float(value)*((float(ymaxtot)-(window))-ymintot)/999.)+ymintot
        self.ymax=self.ymin+window
        if y2mintot != None and y2maxtot != None:
            self.y2min=(float (self.ymin)-ymintot) * (float(y2maxtot)-y2mintot) /  (float(ymaxtot)-ymintot) +y2mintot
            self.y2max=(float (self.ymax)-ymintot) * (float(y2maxtot)-y2mintot) /  (float(ymaxtot)-ymintot) +y2mintot
        self._Refresh()


    def Redraw(self):
        self._Refresh()


    def Save(self,filename,format="BMP"):        
        import pl
        pl.plsdev("psc")
        pl.plsfnam(filename)
        pl.plscolbg(self.qpl.back_color.red(),self.qpl.back_color.green(),self.qpl.back_color.blue())        
        pl.plinit()
        self._DrawEnv(pl)
        self._DrawFunctions(pl)
        pl.plend()
        if format=="PS": return
        if format=="JPG": format="JPEG"
        else:
            try:
                import Image
                pilim=Image.open(filename).transpose(Image.ROTATE_270)
                pilim.save(filename,format)
            except:
                import os
                try: os.remove(tmpfile)
                except: pass
                return 0
            
        #else:                    
            #pixmap=qt.QPixmap.grabWidget(self.qpl)
            #qt.bitBlt(pixmap,0,0,self.qpl,0,0)
            #if format=="JPG": format="JPEG"
            #return pixmap.save(filename,format)
        
        
    def IncrementalUpdate(self,function_name,range,restart=0):
        if function_name not in self.functions.keys(): return
        sel=self.functions[function_name]
        self.qpl.InitDrawing()
        sel["valid_range"]=range        
        if (restart):
            self._DrawEnv(pyqt_pl)
            self._DrawFunctions(pyqt_pl)    
        else:
            self._DrawObjects()                
            self._DrawFunction(sel,pyqt_pl)
            
        
        self._DrawObjects()
        self.qpl.EndDrawing()
        sel["valid_range"]=(0,range[1])


    def _Refresh(self):
        self.qpl.InitDrawing()
        self._DrawEnv(pyqt_pl)
        self._DrawFunctions(pyqt_pl)
        self._DrawObjects()
        self.qpl.EndDrawing()
        

    def DrawLabels(self,modpl):                
        modpl.pllsty(1)
        modpl.plfont(1)
        #Multiline? Change text size scaling: modpl.plschr(0,0.5)?
        #BOLD? modpl.plwid(2)
        if self.TitleLabel!=None: modpl.plmtex("t", 3.0, 0.5, 0.5, self.TitleLabel)
        #modpl.plwid(0)
        if self.XLabel!=None:     modpl.plmtex("b", 3.5, 0.5, 0.5, self.XLabel)
        if self.Y1Label!=None:    modpl.plmtex("l", 5.0, 0.5, 0.5, self.Y1Label)
        if self.Y2Label!=None:    modpl.plmtex("r", 5.0, 0.5, 0.5, self.Y2Label)

                
       
    def _DrawEnv(self,modpl):
        self.cursor=None
        self.formercurpos=None
        self.rect=None
        
        modpl.pladv(0)
        modpl.plrgb1(0,0,0)
        modpl.plwid(0)        
        modpl.pllsty(1)
        modpl.plfont(1)


        xmin,xmax,ymin,ymax,y2min,y2max=self.xmin, self.xmax, self.ymin,self.ymax, self.y2min,self.y2max

        if self.PolarEnv:
            rmax=self.xmaxtot/OVERSIZE_POLAR
            dtr = math.pi / 180.0
            x0 = Numeric.cos(dtr*Numeric.arrayrange(361))
            y0 = Numeric.sin(dtr*Numeric.arrayrange(361))
            modpl.plenv(xmin,xmax,ymin,ymax, 1, -2)
            if self.Gridlines:
                modpl.plrgb1(194,194,194)
                i = 0.1*Numeric.arrayrange(1,11)*rmax
                i.shape = (-1,1)
                x=i*x0
                y=i*y0
                for i in range(10):
                    modpl.plline(x[i], y[i])
                    modpl.plschr (0,.7)
                    modpl.plptex (x[i][0], y[i][0],0.,1., 0.0 , "%0.2f" % (x[i][0],))
                    modpl.plschr (0,1.)
                modpl.plrgb1(0,0,0)                    

            for i in range(12):
                theta = 30.0 * i
                dx = Numeric.cos(dtr * theta)*rmax
                dy = Numeric.sin(dtr * theta)*rmax
                modpl.pljoin(0.0, 0.0, dx, dy)
                text = `int(theta)`
                #Slightly off zero to avoid floating point logic flips at 90 and 270 deg.
                if dx >= -0.00001:
                    modpl.plptex(dx, dy, dx, dy, -0.15, text)
                else:
                    modpl.plptex(dx, dy, -dx, -dy, 1.15, text)
            
        else:
            #TODO: Try to make a calculation in order to let the text always visible (in the way of plvsta)
            #modpl.plvpor(self.xposmin,self.xposmax,self.yposmin,self.yposmax)
            #modpl.plvsta()
            #(xposmin,xposmax,yposmin,yposmax)=modpl.plgspa()
            #modpl.plsvpa(xposmin+10,xposmax-10,yposmin+10,yposmax-10)
            modpl.plvpor(self.xposmin,self.xposmax,self.yposmin,self.yposmax)
            
            xpars="abcnst"
            ypars="abnstv"
            y2pars="cmstv"
            if self.Gridlines:
                xpars=xpars+"g"
                ypars=ypars+"g"

            if self.LogX:   xpars=xpars+"l"
            if self.LogY:   ypars=ypars+"l"
            
            modpl.plwind(xmin,xmax,ymin,ymax)
            
            if self.y2min==None or self.y2max==None:
                ypars=ypars+"c"
                modpl.plbox(xpars, 0.0, 0,ypars,  0.0, 0)
            else:
                if self.LogY:
                    y2pars=y2pars+"l"
                modpl.plbox(xpars, 0.0, 0,ypars,  0.0, 0)
                modpl.plwind(xmin,xmax,y2min,y2max)        
                modpl.plbox("", 0.0, 0, y2pars, 0, 0)
                modpl.plwind(xmin,xmax,ymin,ymax)    

        self.realxmin, self.realxmax, self.realymin,self.realymax, self.realy2min,self.realy2max=xmin,xmax,ymin,ymax,y2min,y2max
        modpl.plrgb1(0,0,0)
        self.DrawLabels(modpl)

            
    #def ExecPlplot(self,cmd):
    #    exec "pyqt_pl."+cmd

    #def EraseUpperBorder(self):
    #     self.qpl.erase( 0, 0, self.qpl.width(), int(self.yposmin*self.qpl.height())-1 )

    #def EraseLowerBorder(self):
    #     self.qpl.erase( 0, int(self.yposmax*self.qpl.height())+1, self.qpl.width(), int((1.0-self.yposmax)*self.qpl.height())-1)

    #def EraseLeftBorder(self):
    #     self.qpl.erase( 0, 0,  int(self.xposmin*self.qpl.width()),self.qpl.height() )

    #def EraseRightBorder(self):
    #     self.qpl.erase ( int(self.xposmax*self.qpl.width())+1, 0,  int((1.0-self.xposmin)*self.qpl.width())-1,self.qpl.height() )

    #def EraseDrawArea(self):
    #     self.qpl.erase( int(self.xposmin*self.qpl.width())+1, int(self.yposmin*self.qpl.height())+1, int((self.xposmax-self.xposmin)*self.qpl.width()-0.5), int((self.yposmax-self.yposmin)*self.qpl.height()-0.5) )
            

    def _DrawFunctions(self,modpl):
        for function in self.functions.values(): self._DrawFunction(function,modpl)

    def _DrawFunction(self,func,modpl):
            xmin,ymin=self.GetDataPosition((self.xmin,self.ymin))
            xmax,ymax=self.GetDataPosition((self.xmax,self.ymax))

            if self.PolarEnv:
                xarr=func["xdata"]
                yarr=func["data"]
            else:
                #first=Numeric.searchsorted(func["xdata"],xmin-1)
                #last =Numeric.searchsorted(func["xdata"],xmax+1)            
                #ASSUMES XDATA SORTED (EITHER ASCENDING OR DESCENDING)                
                if "valid_range" not in func.keys():                    
                    first=BinarySearch(func["xdata"],xmin-1)
                    last=BinarySearch(func["xdata"],xmax+1)                                    
                else:                   
                   top=min(func["valid_range"][1]-1,len(func["xdata"])-1)
                   if func["xdata"][func["valid_range"][0]]>func["xdata"][top]:
                       low= max(self.xmin-1,func["xdata"][top]-1)
                       high=min(self.xmax+1,func["xdata"][func["valid_range"][0]]+1)
                   else:
                       low= max(self.xmin-1,func["xdata"][func["valid_range"][0]]-1)
                       high=min(self.xmax+1,func["xdata"][top]+1)
                   first=BinarySearch(func["xdata"],low,func["valid_range"][1])
                   last =BinarySearch(func["xdata"],high,func["valid_range"][1])
                       
                if first>last: first,last=last,first
                xarr=Numeric.take(func["xdata"],range(first,last))
                yarr=Numeric.take(func["data"],range(first,last))

            if len(xarr)==0: return

            pen=func["pen"]
            modpl.plrgb1(pen.color[0],pen.color[1],pen.color[2])

                   
            if self.LogX:
                minval=min(xarr.flat)
                if minval<=0: xarr=Numeric.maximum(xarr,self.minvaluelogx)
                xarr=Numeric.log10(xarr)

            if self.LogY:
                minval=min(yarr.flat)
                if minval<=0:
                    yarr=Numeric.maximum(yarr,self.minvaluelogy)
                    ymin=math.log10(self.minvaluelogy)
                else:
                    ymin=math.log10(ymin)
                yarr=Numeric.log10(yarr)
                
            
            
            if func["yscale"]==1:
                modpl.plwind(self.realxmin, self.realxmax, self.realy2min, self.realy2max)
                

            if pen.style!="hidden" and (self.style=="Line" or self.style=="PointsLine") :
                if   pen.style=="solid":           modpl.pllsty(1)
                elif pen.style=="dashed":          modpl.pllsty(3)
                elif pen.style=="dotted":          modpl.pllsty(2)
                modpl.plwid(pen.width)
                if "delimeters" in func.keys():
                    for delimeter in func["delimeters"]:
                        if self.LogX:
                            delimeter=(math.log10(max(delimeter[0],self.minvaluelogx)),math.log10(max(delimeter[1],self.minvaluelogx)))
                        start,end=BinarySearch(xarr,delimeter[0]-1),BinarySearch(xarr,delimeter[1])
                        if end>start:
                            partialxarr=Numeric.take(xarr,range(start,end))
                            partialyarr=Numeric.take(yarr,range(start,end))
                            modpl.plline(partialxarr,partialyarr)
                    
                else:
                    modpl.plline(xarr,yarr)
                modpl.pllsty(1)
                modpl.plwid(0)
            
            if self.style=="Points" or self.style=="PointsLine": symbol="point"
            else: symbol=func["symbol"]
            if symbol != "none":
                if symbol=="point":
                    modpl.plwid(int(max(5,pen.width*3)))
                    modpl.plpoin(xarr,yarr, -1 )
                else:
                    modpl.plwid(pen.width)
                    modpl.plpoin(xarr,yarr, GetSymbolCode(symbol))                    
                modpl.plwid(0)

            if self.style=="Bars":
                if   pen.style=="solid":           modpl.pllsty(1)
                elif pen.style=="dashed":          modpl.pllsty(3)
                elif pen.style=="dotted":          modpl.pllsty(2)
                modpl.plwid(pen.width)
                modpl.plpsty(0)
                if len(xarr)==1: off=1.
                else: off=(xarr[1]-xarr[0])/2.
                xarroff=xarr-off                
                for i in range(len(xarr)-1):
                    x0,y0,x1=xarroff[i],yarr[i],xarroff[i+1]
                    modpl.plfill([x0, x0, x1,x1],[ymin, y0, y0, ymin])
                x0,y0,x1=xarroff[-1],yarr[-1],xarroff[-1]+off
                modpl.plfill([x0, x0, x1,x1],[ymin, y0, y0, ymin])
            
            if func["yscale"]==1:
                modpl.plwind(self.realxmin, self.realxmax, self.realymin, self.realymax)
    
            


    def SetStyle(self,style):
        if style!=self.style:
            self.style=style
            self._Refresh()



    def SetZoom (self,(x0,y0),(x1,y1)):        
        if ((abs(x0-x1))<(0.00001 * (self.xmaxtot-self.xmintot))) or ((abs(y0-y1))<(0.00001 * (self.ymaxtot-self.ymintot))):
            return
        self.xmin=min(x1,x0)
        self.xmax=max(x1,x0)
        self.ymin=min(y1,y0)
        self.ymax=max(y1,y0)

        self.xmin,self.ymin=self.GetPlotPosition((self.xmin,self.ymin))
        self.xmax,self.ymax=self.GetPlotPosition((self.xmax,self.ymax))

        xmintot,xmaxtot,ymintot,ymaxtot,y2mintot,y2maxtot=self._GetPlotLimits()

        if   abs(ymintot-self.ymin)<(ymaxtot-ymintot)/100: self.ymin=ymintot
        if   abs(ymaxtot-self.ymax)<(ymaxtot-ymintot)/100: self.ymax=ymaxtot
        if   abs(xmintot-self.xmin)<(xmaxtot-xmintot)/100: self.xmin=xmintot
        if   abs(xmaxtot-self.xmax)<(xmaxtot-xmintot)/100: self.xmax=xmaxtot

        if self.y2mintot != None and self.y2maxtot != None:
            self.y2min=(float (self.ymin)-ymintot) * (float(y2maxtot)-y2mintot) /  (float(ymaxtot)-ymintot) +y2mintot
            self.y2max=(float (self.ymax)-ymintot) * (float(y2maxtot)-y2mintot) /  (float(ymaxtot)-ymintot) +y2mintot

        if self.ScrollMode!="OFF":
            if self.ZoomedX:
                self.disconnect(self.HScroll,qt.SIGNAL("valueChanged(int)"),self.hscroll)
            if self.ZoomedY:
                self.disconnect(self.VScroll,qt.SIGNAL("valueChanged(int)"),self.vscroll)
            if self.xmin!=xmintot or self.xmax!=xmaxtot:
                self.HScroll.setMinValue(0)
                self.HScroll.setMaxValue(999)        
                self.HScroll.setValue((float(self.xmin)-xmintot)*999./((float(xmaxtot)-(self.xmax-self.xmin))-xmintot))
                self.HScroll.setEnabled(1)
                self.connect(self.HScroll,qt.SIGNAL("valueChanged(int)"),self.hscroll)
                if self.ScrollMode=="AUTO":
                    self.HScroll.show()                
            if self.ymin!=ymintot or self.ymax!=ymaxtot:
                self.VScroll.setMinValue(0)
                self.VScroll.setMaxValue(999)        
                self.VScroll.setValue(999-((float(self.ymin)-ymintot)*999./((float(ymaxtot)-(self.ymax-self.ymin))-ymintot)))
                self.VScroll.setEnabled(1)
                self.connect(self.VScroll,qt.SIGNAL("valueChanged(int)"),self.vscroll)
                if self.ScrollMode=="AUTO":
                    self.VScroll.show()                

        if self.xmin!=xmintot or self.xmax!=xmaxtot:self.ZoomedX=1
        if self.ymin!=ymintot or self.ymax!=ymaxtot:self.ZoomedY=1
                
        self._Refresh()


    def ResetZoom(self,update=1):
       self.xmin, self.xmax, self.ymin, self.ymax, self.y2min, self.y2max=self._GetPlotLimits()
       changed=0       
       if self.ZoomedX:
           if self.ScrollMode!="OFF":           
               self.disconnect(self.HScroll,qt.SIGNAL("valueChanged(int)"),self.hscroll)
               self.HScroll.setEnabled(0)       
               self.HScroll.setMinValue(0)
               self.HScroll.setMaxValue(0)
               if self.ScrollMode=="AUTO":
                    self.HScroll.hide()
           self.ZoomedX=0
           changed=1
       if self.ZoomedY:
           if self.ScrollMode!="OFF":           
               self.disconnect(self.VScroll,qt.SIGNAL("valueChanged(int)"),self.vscroll)
               self.VScroll.setEnabled(0)       
               self.VScroll.setMinValue(0)
               self.VScroll.setMaxValue(0)
               if self.ScrollMode=="AUTO":
                    self.VScroll.hide()
           self.ZoomedY=0
           changed=1
       if changed and update: self._Refresh()


			
    def _RightButtonPress(self,e):
        self.parent._RightButtonPress(e.globalPos ())
                    

        

    def _DrawObjects(self):
        for obj in self.GraphicObjects:
            obj.visible=0
            obj.show()

    def PutLine(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid"),yscale=0):
        obj=LineObj(self,pen,None,yscale)
        self.GraphicObjects.append(obj)
        obj.SetCoords(X0,Y0,X1,Y1)        
        return obj
    
    def PutRectangle(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0"),yscale=0):
        obj=RectangleObj(self,pen,brush,yscale)
        self.GraphicObjects.append(obj)
        obj.SetCoords(X0,Y0,X1,Y1)
        return obj

    def PutEllipse(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0"),yscale=0):
        obj=EllipseObj(self,pen,brush,yscale)
        self.GraphicObjects.append(obj)
        obj.SetCoords(X0,Y0,X1,Y1)  
        return obj

    def PutSymbol(self,X0,Y0,symbol="circle",pen=Pen((0,0,0),1,"solid"),yscale=0):
        obj=SymbolObj(self,symbol,pen,None,yscale)        
        self.GraphicObjects.append(obj)
        obj.SetCoords(X0,Y0)
        return obj

    def PutText(self,X0,Y0,text, align="center",orient=0.,pen=Pen((0,0,0),1,"solid"),style="normal",yscale=0,brush=Brush((255,255,255),"fill_100")):
        obj=TextObj(self,text,align,orient,pen,brush,style,yscale)        
        self.GraphicObjects.append(obj)
        obj.SetCoords(X0,Y0)
        return obj

    def SetObjectCoords(self,obj,*args):
        obj.SetCoords(*args)

    def GetObjectCoords(self,obj):
        return obj.GetCoords()

    
    def EraseObject(self,obj):
        if obj in self.GraphicObjects:
            obj.erase()
            self.GraphicObjects.remove(obj)

    def UpdateObject(self,obj,pen=None,brush=None,textOptsDic=None):
        if obj in self.GraphicObjects:
            obj.Update(pen,brush,textOptsDic)
            


    def GetVisibleXAxis(self):
        xmin,ymin=self.GetDataPosition((self.xmin,self.ymin))
        xmax,ymax=self.GetDataPosition((self.xmax,self.ymax))
        return ((xmin,xmax))

    def GetVisibleYAxis(self):
        xmin,ymin=self.GetDataPosition((self.xmin,self.ymin))
        xmax,ymax=self.GetDataPosition((self.xmax,self.ymax))
        return ((ymin,ymax))

    def GetVisibleY2Axis(self):
        if self.y2min is not None and self.y2max is not None:
            xmin,y2min=self.GetDataPosition((self.xmin,self.y2min))
            xmax,y2max=self.GetDataPosition((self.xmax,self.y2max))
        else: y2min=y2max=None
        return ((y2min,y2max))


    def GetFunctionScale(self,name):
        try:
            return self.functions[name]["yscale"]
        except:
            return None
    

    def ImageCoord2ViewCoord(self, coord):
        return coord        

    def ViewCoord2ImageCoord(self, coord):
        return coord


    def DataCoord2ViewCoord(self, coord):
        pass        


    def GetPlotPosition(self,(x,y)):        
        if self.LogX: x=Numeric.log10(max(x,self.minvaluelogx))
        if self.LogY: y=Numeric.log10(max(y,self.minvaluelogy))
        return (x,y)


    def GetDataPosition(self,(x,y)):
        if self.LogX: x=math.pow(10.,x)
        if self.LogY: y=math.pow(10.,y)
        return (x,y)


    def ViewCoord2DataCoord(self, coord):
        xdata,ydata=0,0
        self.qpl.InitDrawing()
        x=float(coord[0])/self.qpl.width()
        y=float(self.qpl.height()-coord[1])/self.qpl.height()
        (xdata,ydata,page)=pyqt_pl.plcalc_world(x,y)
        
        if page==-1:
            (xmin, xmax, ymin, ymax)=self.xposmin,self.xposmax,self.yposmin,self.yposmax
            x=max(xmin,min(x,xmax))
            y=max(ymin,min(y,ymax))
            (xdata,ydata,x)=pyqt_pl.plcalc_world(x,y)

        (xdata,ydata)=self.GetDataPosition((xdata,ydata))

        if page==-1: return (0,(xdata,ydata))    
        else: return (1,(xdata,ydata))   


    def Destroy(self):
        self.parent=None
        self.qpl=  None
        self.functions={}
        self.GraphicObjects=[]        
        self.VScroll=None
        self.HScroll=None



    def _GetPlotLimits(self):
        xmintot,xmaxtot,ymintot,ymaxtot,y2mintot,y2maxtot=self.xmintot,self.xmaxtot,self.ymintot,self.ymaxtot,self.y2mintot,self.y2maxtot
        if self.LogX: xmintot,xmaxtot=Numeric.log10(max(xmintot,self.minvaluelogx)),Numeric.log10(max(xmaxtot,self.minvaluelogx))
        if self.LogY:
            ymintot,ymaxtot=Numeric.log10(max(ymintot,self.minvaluelogy)),Numeric.log10(max(ymaxtot,self.minvaluelogy))
            if (y2mintot is not None) and (y2maxtot is not None):
                y2mintot,y2maxtot=Numeric.log10(max(y2mintot,self.minvaluelogy)),Numeric.log10(max(y2maxtot,self.minvaluelogy))

        return xmintot,xmaxtot,ymintot,ymaxtot,y2mintot,y2maxtot



    def _LeftButtonPress(self,e):
        if self.parent is not None: self.parent._ButtonPress(self.ViewCoord2ImageCoord((e.x(),e.y())))
        
    def _LeftButtonRelease(self,e):
        if self.parent is not None: self.parent._ButtonRelease(self.ViewCoord2ImageCoord((e.x(),e.y())))
        

    def _Motion(self,e):
        if self.parent is not None: self.parent._Motion(self.ViewCoord2ImageCoord((e.x(),e.y())))
  
    def _PressMotion(self,e):
        if self.parent is not None: self.parent._ButtonPressMotion(self.ViewCoord2ImageCoord((e.x(),e.y())))

    def _DoubleClick(self,e):
        if self.parent is not None: self.parent._DoubleClick(self.ViewCoord2ImageCoord((e.x(),e.y())))

    def _KeyPress(self,e):
        ret=str(e.text())
        if ret=="": ret= e.key()
        if self.parent is not None: self.parent._KeyPress(ret)

    def GetViewport(self):
        return self.qpl
