"""
    ViewSelect.py
    Base class for mouse/keyboard event based data selection over View objects    
"""

from PyDVT import __version__,__date__,__author__


################################################################################  
from View import Pen,Brush,KeyCodes
################################################################################  
INPUT_OUTPUT = 1
OUTPUT = 2
################################################################################  


class ViewSelect:
  """
    ViewSelect objects implement data selection over View objects
    through mouse and keyboard events. 
    A ViewSelect object can be linked only to one Data object, but to
    multiple views.
    ViewSelect receives  mouse, keyboard and DataChange events, and
    calls a callback function to inform application when a selection
    happens.
    Derived classes implement its functionality by overriding event
    methods, drawing over views and calling the selection callback.

    SetSelection method sets the selection dictionary, redraws all views
    and trigger selection callback. If derived classes implement drawing
    of the final selection based in the selection dictionary,
    application can set selection by code by calling SetSelection.

    The callback receive as a parameter the ViewSelect object that
    generated the event. The selection is read with GetSelection
    method, that returns a dictionary. The keys for this dictionary
    can be different for each derived class.
    


    Interface:
    ===========================
        __init__
        ConnectView
        DisconnectView
        SetSelection
        GetSelection        
        RemoveSelection
        GetViewList
        Enable
        Disable
        DisableKeyboard
        EnableKeyboard
        RemoveSelectionInDataChange
        Erase
        Update
        SetPen
        SetBrush
        GetData
        Destroy

    Overridables:
    ===========================
        DataChanged

    Virtuals:
    ===========================    
        KeyPress
        Motion
        PressMotion
        Press
        DoubleClick
        Release
        EraseView
        DrawView
        GetType
    

    Internal Events:
    ===========================
        Receives:
            "DataChange"
            "DeleteEvent"
            "ButtonPressMotion"
            "ButtonRelease"
            "ButtonPress"
            "DoubleClick"
            "Motion"
            "KeyPress"
            "ImageChange"
  """

  def __init__(self,data=None,callback=None):
    """
    Constructor
    Parameters:
    data: the Data or Selection object the ViewSelect will be linked to.
          there's no direct interaction to the data object, all ViewSelect operations are
          made through the View object. The base class just registers "DataChange"
          and "DeleteEvent" events
    callback: callback function for the selection event
    """
    self.Data=data
    self.MouseEnabled=1
    self.KeyboardEnabled=1
    self.callback=callback
    self.Pen=Pen((0,0,0),1,"solid")    
    self.Brush= Brush((0,0,0),"fill_25")   
    if self.Data is not None:
        self.Data.eh.register("DataChange",   self.DataChanged)
        self.Data.eh.register("DeleteEvent",  self.Destroy)
    self.ViewList={}
    self.Selection={}
    self.Alive=1
    self.FlagRemoveSelectionInDataChange=0


  def GetData(self):
    """
    Returns linked Data object (None if no linked Data)
    """
    return self.Data

    
  def SetPen(self,pen):
    """
    Changes default pen
    Parameters:
    pen: Pen object    
    """
    self.Pen=pen


  def SetBrush(self,brush):
    """
    Changes default brush
    Parameters:
    brush: Brush object    
    """
    self.Brush=brush


  def ConnectView(self, view, mode=INPUT_OUTPUT):      
    """
    Connects a view to the ViewSelect object
    Parameters:
    view: the view object
    mode: INPUT_OUTPUT - ViewSelect object both draws itself on the View and
          receives its mouse events
          OUTPUT - ViewSelect object draws itself on the view but does not
          receive its mouse events
    """
    self.ViewList[view]={"Mode":mode}
    view.eh.register("Motion",            self.__Motion)
    view.eh.register("ButtonPressMotion", self.__PressMotion)
    view.eh.register("ButtonPress",       self.__Press)
    view.eh.register("ButtonRelease",     self.__Release)
    view.eh.register("KeyPress",          self.__KeyPress)
    view.eh.register("DoubleClick",       self.__DoubleClick)
    view.eh.register("ImageChange",       self.Update)

    
  def DisconnectView(self, view):
    """
    Disonnects a view from the ViewSelect object
    Parameters:
    View: the view object    
    """
    if view in self.ViewList.keys():
      del self.ViewList[view]
      if view.eh is not None:
          view.eh.unregister("Motion",            self.__Motion)
          view.eh.unregister("ButtonPressMotion", self.__PressMotion)
          view.eh.unregister("ButtonPress",       self.__Press)
          view.eh.unregister("ButtonRelease",     self.__Release)
          view.eh.unregister("KeyPress",          self.__KeyPress)
          view.eh.unregister("DoubleClick",       self.__DoubleClick)
          view.eh.unregister("ImageChange",       self.Update)


  def GetViewList(self):
    """
    Returns list of connected views
    """
    return self.ViewList.keys()


  def GetSelection(self):
    """
    Returns selection dictionary ({} if no selection in done)
    """
    return self.Selection


  def SetSelection(self,selection):
    """
    Sets Selection dictionary, redraws and call callback function.
    Parameters:
    selection: dictionary with selection data
    """
    self.Selection=selection      
    self.Update("ALL")
    self.NotifySelection()



  def RemoveSelection(self):
    """
    Clears the selection (if done) and erases from all views.
    The ViewSelect object is still active.
    """
    self.Erase("ALL")
    self.Selection={}


  def RemoveSelectionInDataChange(self,remove=0):
    """
    Sets flags that controls if ViewSelect is removed or not in a
    DataChange event.
    The default implementation of DataChanged works based on this flags.
    If DataChanged is overriden this method may not work.
    Parameters:
    remove: if non-zero selection is removed in a DataChange event.
        Default: selection is not removed
    """
    self.FlagRemoveSelectionInDataChange=remove


  def Disable(self):
    """
    Makes ViewSelect object stop ansering mouse envents of the connected
    views, but keeps the selection on screen, if any.
    (this "freezes" the selection)
    """
    self.MouseEnabled=0


  def Enable(self):
    """
    Makes ViewSelect object restart ansering mouse envents of the connected
    views, without changing the selection (if any).
    """
    self.MouseEnabled=1


  def DisableKeyboard(self):
    """
    Disables answer to keyboard events
    """
    self.KeyboardEnabled=0


  def EnableKeyboard(self):
    """
    Enables answer to keyboard events
    """
    self.KeyboardEnabled=1


  def Erase(self,view="ALL"):
    """
    Erases the drawing of a SelectView object
    Parameters:
    view:   if "ALL", erases all views
            if a view instance, erases only this one
    """
    self.__CheckViews()
    if view=="ALL":
        for view in self.ViewList.keys(): self.EraseView(view)
    elif view in self.ViewList.keys(): self.EraseView(view)
    

  def Update(self,view="ALL"):
    """
    Updates the drawing of a SelectView object
    Parameters:
    view:   if "ALL", updates all views
            if a view instance, updates only this one
    """
    self.__CheckViews()
    self.Erase(view)
    if view=="ALL":
        for view in self.ViewList.keys(): self.DrawView(view)
    elif view in self.ViewList.keys():  self.DrawView(view)


  def NotifySelection(self):
    if self.Selection!={} and self.callback is not None: self.callback(self)


  def __CheckViews(self):
    for view in self.ViewList.keys():
        if view.IsVisible()==0: del self.ViewList[view]
    

  def __Motion(self,(source,pos)):
    self.__CheckViews()
    if (self.MouseEnabled==1) and (source in self.ViewList.keys()):      
      if self.ViewList[source]["Mode"]==INPUT_OUTPUT : self.Motion(pos)


  def __PressMotion(self,(source,pos)):
    self.__CheckViews()
    if (self.MouseEnabled==1) and (source in self.ViewList.keys()):
      if self.ViewList[source]["Mode"]==INPUT_OUTPUT : self.PressMotion(pos)


  def __Press(self,(source,pos)):
    self.__CheckViews()
    if (self.MouseEnabled==1) and (source in self.ViewList.keys()):
      if self.ViewList[source]["Mode"]==INPUT_OUTPUT : self.Press(pos)


  def __DoubleClick(self,(source,pos)):
    self.__CheckViews()
    if (self.MouseEnabled==1) and (source in self.ViewList.keys()):
      if self.ViewList[source]["Mode"]==INPUT_OUTPUT : self.DoubleClick(pos)
      

  def __Release(self,(source,pos)):
    self.__CheckViews()
    if (self.MouseEnabled==1) and (source in self.ViewList.keys()):
      if self.ViewList[source]["Mode"]==INPUT_OUTPUT : self.Release(pos)


  def __KeyPress(self,(source,key,flags)):
    self.__CheckViews()
    if (self.KeyboardEnabled==1) and (source in self.ViewList.keys()):
      if self.ViewList[source]["Mode"]==INPUT_OUTPUT : self.KeyPress(key,flags)


  def Destroy(self,source=None):
    """
    Cleanup.
    As event handlers and callbacks, by keeping a reference to other
    objects, avoid them to be destroyed (and freed from memory), the
    Destroy method has to be explicitly called for ViewSelect objects
    before they get out of scope.
    I had tried to use weak references in order not to need this method,
    but this generated many other troubles (like impossibility to define
    selections locally, difficult control of the destruction sequence,
    not interceptable error mesassages...).
    """
    if self.Alive==0:return
    self.Selection={}    

    self.Erase("ALL")
    if self.Data is not None:
        if self.Data.eh is not None:
            self.Data.eh.unregister("DataChange",   self.DataChanged)
            self.Data.eh.unregister("DeleteEvent",  self.Destroy)
        self.Data=None        
    for view in self.ViewList.keys():
        if view.eh is not None:
            view.eh.unregister("Motion",            self.__Motion)
            view.eh.unregister("ButtonPressMotion", self.__PressMotion)
            view.eh.unregister("ButtonPress",       self.__Press)
            view.eh.unregister("ButtonRelease",     self.__Release)
            view.eh.unregister("KeyPress",          self.__KeyPress)
            view.eh.unregister("DoubleClick",       self.__DoubleClick)
            view.eh.unregister("ImageChange",       self.Update)

    self.ViewList={}
    self.Data=None
    self.Alive=0


#################################
#       OVERRIDABLES
#################################
  def DataChanged(self,par):
    """
    Overridable: default implementation just verifies flag set by RemoveSelectionInDataChange.
    Added the interception of the "reconfiguremoithis" key to force view
    to follow the page change 
    """
    if (type(par[1])==type({})   and par[1].has_key("reconfiguremoithis")  ):
                  has_changed=0
                  newp = par[1]["reconfiguremoithis"]
                  if( newp != self.GetSelection()['BoundingRect'][1].PageIndex):
                    self.GetSelection()['BoundingRect'][1].PageIndex = par[1]["reconfiguremoithis"]
                    has_changed=1

                  if( newp != self.GetSelection()['BoundingRect'][0].PageIndex):
                    self.GetSelection()['BoundingRect'][0].PageIndex = par[1]["reconfiguremoithis"]
                    has_changed=1
                  self.NotifySelection() 
                  return


    if self.FlagRemoveSelectionInDataChange: self.RemoveSelection()



#################################
#            VIRTUALS
#################################

  def KeyPress(self,key,flags):
    """
    Virtual: Keyboard event
    Parameter:
    """
    pass


  def Motion(self,pos):
    """
    Virtual: Mouse move event
    Parameter:
    pos: coordinates, as sent by GetPosition of the View object
         that generated the event
    """
    pass


  def PressMotion(self,pos):
    """
    Virtual: Mouse motion while button pressed event
    Parameter:
    pos: coordinates, as sent by GetPosition of the View object
         that generated the event
    """
    pass


  def Press(self,pos):
    """
    Virtual: Mouse press event
    Parameter:
    pos: coordinates, as sent by GetPosition of the View object
         that generated the event
    """
    pass


  def DoubleClick(self,pos):
    """
    Virtual: Mouse double click event
    Parameter:
    pos: coordinates, as sent by GetPosition of the View object
         that generated the event
    """
    pass


  def Release(self,pos):
    """
    Virtual: Mouse release event
    Parameter:
    pos: coordinates, as sent by GetPosition of the View object
         that generated the event
    """
    pass


  def EraseView(self,view):
    """
    Virtual: Implements the erasing code
    Parameter:
    view: view object on which perform the operation
    """
    pass


  def DrawView(self,view):
    """
    Virtual: Implements the drawing code
    Parameter:
    view: view object on which perform the operation
    """
    pass


  def GetType(self):
    """
    Virtual: Return a string identifying the type of selection
    """
    pass




class ViewSelectDefault(ViewSelect):
  """
  ViewSelect derived class, also virtual, that adds some functionality,
  (which are similar to the default ViewSelect objects)
  assuming that the grafical contents of the SelectView object are
  stored in the dictionary self.ViewList, and are composed by only one
  canvas object for the selection ("SelectDraw", in the dictionary) and
  one canvas object for the intermediate drawing ("TempDraw", in the
  dictionary). ViewSelectDefault assumes also selection drawing based
  in the key "BoundingRect" in the selection dictionary.
  Derived classes shall implement DrawSelection, DrawTemp, and the mouse
  events (Motion,PressMotion,Press,Release) that are necessary.
  By using SetSelectionPos (instead of SetSelection) the key "BoundingRect"
  is fixed in the selection dictionary. It's a tuple ((x0,y0]), (x1,y1))
  with bounding rect of selection.        
  """

  def SetSelectionPos(self,bounding_rect):
    """
    Updates selection with new bounding rectangle.
    Redraws and call callback function.
    Parameters:
        bounding_rect: value put to "BoundingRect" key in Selection dictionary
    """
    self.Selection["BoundingRect"]=bounding_rect      
    self.Update("ALL")
    self.NotifySelection()

  
  def EraseView(self,view):
    """
    Virtual: See ViewSelect
    """
    self.EraseTemp(view)
    self.EraseSelection(view)


  def DrawView(self,view):
    """
    Virtual: See ViewSelect
    """
    try:
        #I can't be sure about the order the DeleteEvent is going to be received by views and selections.
        #Without the the line below, sometimes the selection object can be repainted after the graphic was erased
        #by a DataDeleted event(GraphView).
        if view.GetSource() == (): return
        self.DrawSelection(view)
    except:
        pass


  def EraseSelection(self,view):
    """
    Erases the canvas object for the selection
    """
    if "SelectDraw" in self.ViewList[view].keys():
        view.Drawable.EraseObject(self.ViewList[view]["SelectDraw"])
        del self.ViewList[view]["SelectDraw"]


  def EraseTemp(self,view):
    """
    Erases the canvas object for the intermediate drawing
    """
    if "TempDraw" in self.ViewList[view].keys():
        view.Drawable.EraseObject(self.ViewList[view]["TempDraw"])
        del self.ViewList[view]["TempDraw"]


  #VIRTUALS
  def DrawSelection(self,view):
      """
      Derived classes draw selection and store its reference in
      self.ViewList[view]["SelectDraw"]
      """
      pass


  def DrawTemp(self,view,mouse_position):
      """
      Derived classes draw intermediate drawing and store its reference in
      self.ViewList[view]["TempDraw"]
      """
      pass

