"""
    View.py
    Generic base class for data displaying widgets    
"""

from PyDVT import __version__,__date__,__author__

################################################################################
from Data import DataPosition
import AM_EventHandler as EventHandler
import types

from Binding import *

if GUI_Binding==None: raise "GUI Binding import error"

################################################################################
       

class Position:
    """
    Structure received by mouse event callbacks, containing information about
    position of the event, in the several coordinate systems.
    The actual data coordinates are stored in the DataCoord member,
    a DataPosition object (Data module)
    """
    WindowCoord=None
    ViewCoord=None
    ImageCoord=None
    DataSelectionCoord=None
    DataCoord=DataPosition()


class View(Container):
    """
    Base class to data displaying widgets.
    
    View objects are connected to a source through SetSource method. A source can
    be an object (or sequence objects) of class Filter, or any other class that
    implements an interface as difined in SetSource.
    
    The base class manages the mouse and keyboard events (to ViewSelect
    objects and the application), the popup menu, the mouse pointer,
    a cursor, communication to and from linked objects (through EventHandler)
    and more.

    Graphical operations over the view can be easily implemented by subclassing
    ViewSelect classes, overriding mouse/keyboard events callbacks (see
    ViewSelect's doc).
    
    View objects draws to a widget called Drawable, created by derived
    classes, overriding CreateDrawable method. It can be retrieved by
    GetDrawable().
    
    This class is GUI independent, and decides for the importation of the GUI
    binding package (QtBinding or TkBinding). Tkinter or Qt must be loaded
    previously.

    Extends Container class, as defined in the binding package.
    The base class doesn't manage resizing - up to derived classes/drawables.

    All internal event exchanging is done through EventHandler objects.
    All the application level interface is done to avoid application to deal
    with EventHandler, but it can, if it wants to.

    Standard View derived classes define Filter objects to nicely connect to
    these View derived object.

    
    Interface:
    ===========================
    Container derived methods (Defined in the GUI Binding):
        GetHeight
        GetWidth
        IsVisible
        Show
        SetSize
        SetPointer
    
    General methods:
        __init__
        SetSource
        GetSource
        Update
        Refresh
        Invalidate    
        SetOnLine
        LockPosition
        SetCursor
        GetDrawable
        GetDrawableWidth
        GetDrawableHeight
        Save
        Destroy

    Popup menu methods:
        GetPopupMenu
        AddMenuPopupItem
        AddMenuPopupCascade
        AddMenuSeparator
        DisableMenuItem
        EnableMenuItem
        DeleteMenuItem
        ClearMenu

    Overridables:        
        EventButtonPress    
        EventButtonPressMotion  
        EventButtonRelease  
        EventMotion
        EventDoubleClick    
        EventKeyPress
        GetSaveFormats


    Virtuals 
    ===========================
        CreateDrawable  
        DataChanged 
        GetPosition 
        DataCoord2ImageCoord    
        ImageCoord2DataCoord    
        Redraw  



    Internal Events:
    ===========================
        Emits:
            "ButtonPressMotion"
            "ButtonRelease"
            "ButtonPress"
            "Motion"
            "KeyPress"
            "ImageChange"
            "DoubleClick"
        Receives:
            "DataChange"
            "DeleteEvent"
        
    """
    def __init__(self,parent=None,pars={},**kw):
        """
        Constructor
        Parameters:
          parent: Parent window
          pars:   Dictionary with View initialization options
            Options defined by base class:
            (If an option is ommited, the default value is taken)
                "ZoomMode":
                  Sets View's zoom mode ("ON","OFF",...)
                  Default:"ON"
                "ScrollMode":
                  Sets View's scroll mode ("ON","OFF",...)
                  Default:"ON"
                "AddOnLine":
                  If non-zero adds on-line entry in menu
                  (enables/disables response to DataChange events)
                  Default:0
                "AddLockPosition":
                  If non-zero adds lock position entry in menu
                  (sets/resets zoom/scroll position lock in a
                  DataChange event )
                  Default:0
                "AddRefresh":
                  If non-zero adds refresh entry in menu
                  (requeries sources and redraws contents)
                  Default:0
                "DataChangedCallback":
                  If defined, sets a callback called to inform a DataChanged event
                  has just been processed.
                  Callback receives two parameters, the view object and the
                  source object that generated the DataChange event
            New initialization options can be defined by derived classes                                           
          kw:     keywords to Container initializatregisterion
        """
        Container.__init__(self,parent,**kw)
        self.parent=parent
        self.Source=()
        self.eh=EventHandler.EventHandler()
        self.OnLine=1
        self.FlagLockPosition=0
        if "ScrollMode"  in pars.keys(): self.ScrollMode=pars["ScrollMode"]
        else: self.ScrollMode="ON"
        if "ZoomMode"  in pars.keys(): self.ZoomMode=pars["ZoomMode"]
        else: self.ZoomMode="ON"
        
        self.EvButtonPressMotion = self.eh.create("ButtonPressMotion")
        self.EvButtonRelease     = self.eh.create("ButtonRelease")
        self.EvButtonPress       = self.eh.create("ButtonPress")
        self.EvDoubleClick       = self.eh.create("DoubleClick")
        self.EvMotion            = self.eh.create("Motion")
        self.EvKeyPress          = self.eh.create("KeyPress")
        self.EvImageChange       = self.eh.create("ImageChange")
        
        self.CreateDrawable()
        
        self.MenuPopup = None
        self.Cursor=None
        #Pointer shall be initialized by derived classes

        if "AddOnLine" in pars.keys(): AddOnLine=pars["AddOnLine"]
        else: AddOnLine=0
        if "AddLockPosition" in pars.keys(): AddLockPosition=pars["AddLockPosition"]
        else: AddLockPosition=0
        if "AddRefresh" in pars.keys(): AddRefresh=pars["AddRefresh"]
        else: AddRefresh=0
        if "DataChangedCallback" in pars.keys(): self.DataChangedCallback=pars["DataChangedCallback"]
        else: self.DataChangedCallback=None
        if AddOnLine:
          self.OnLineMenuItem=self.AddMenuPopupItem("OnLine",self._CmdOnLine,'checkbutton')
          self.MenuPopup.CheckItem(self.OnLineMenuItem,self.OnLine)
        if AddLockPosition:
          self.LockPositionMenuItem=self.AddMenuPopupItem("LockPosition",self._CmdLockPosition,'checkbutton')
          self.MenuPopup.CheckItem(self.LockPositionMenuItem,self.FlagLockPosition)
        if AddRefresh:          
          self.AddMenuPopupItem("Refresh",self.Refresh)
        self.Alive=1        

        
    def _CmdOnLine(self):
        if self.OnLine:
            self.OnLine=0
        else:
            self.OnLine=1
            self.Refresh()
        self.MenuPopup.CheckItem(self.OnLineMenuItem,self.OnLine)
                
        
    def _CmdLockPosition(self):        
        if self.FlagLockPosition:
            self.LockPosition(0)
        else:
            self.LockPosition(1)
        self.MenuPopup.CheckItem(self.LockPositionMenuItem,self.FlagLockPosition)


    def Refresh(self):
        """
        Generates rescan of all sources and complete redraw of view.
        For simple redrawing Update method should be called instead,
        but for the cases where the source data is directly changed,
        with no DataChange event sent (or when View object is not
        on-line).      
        """
        self.DataChanged()
        if self.DataChangedCallback is not None: self.DataChangedCallback(self,None)
        
            
    def LockPosition(self,value):
        """
        Controls positioning of view after data change.
        Parameters:
            value: If zero, view stays at the same position and zoom state in the
                   case of a DataChange event. Otherwise, views returns to initial
                   position. By default FlagLockPosition is set to zero.
                   (FlagLockPosition shall be interpreted by derived classes)
        """
        self.FlagLockPosition=value


    def SetOnLine(self,value):
        """
        Controls reponse to DataChange event.
        Parameters:
            value:If zero, view does not answer Data change events
        """
        self.OnLine=value


    def SetSource(self,source=()):
        """
        Connects view object to a source or sequence of sources
        
        Internally the sources are always stored as a tuple.
        GetSource always returns a tuple, even if SetSource was
        initialized with a single object or None (== ()).
        
        A source can be DataSelection or Filter derived objects.

        Actually any class can be a source if the derived View knows
        how to interface to it.
        The common interface (the one implemented to DataSelection
        and Filter is:
          - The source have an EventHandler member called eh
            on which the View registers "DataChange" and "DeleteEvent"
            (these should be emmited by the source, and derived View classes
            don't mind registering, since it is done in View base class).
          - The source shall have a GetOutput method on which the
            View gets the actual data in a dictionary, as described
            in the Filter module.
        With this interface existing View classes can be connect to
        new source classes and new View classes can be connected to
        existing source classes.
        
        
        Parameters:
            source: DataSelection/Filter object or sequence of objects
        """
        if source==None: source=()
        elif type(source) is types.InstanceType: source=(source,)
        if type(source) is not  types.TupleType: source=tuple(source)
        
        if (self.Source != source):
            for sel in self.Source:
               if sel.eh is not None:
                   sel.eh.unregister("DataChange",self._DataChanged)
                   sel.eh.unregister("DeleteEvent",self._DataDeleted)
            self.Source=source
            for sel in self.Source:
               sel.eh.register("DataChange",self._DataChanged)        
               sel.eh.register("DeleteEvent",self._DataDeleted)
            self._DataChanged()

    def GetSource(self):
        """
        Gets view's tuple of sources
        """
        return self.Source

    def _DataChanged(self,source=None):
        if self.OnLine:
            self.DataChanged(source)
            if self.DataChangedCallback is not None: self.DataChangedCallback(self,source)


    def _DataDeleted(self,source=None):
        newsel=[]
        for sel in self.Source:
            if sel is source:
               if sel.eh is not None:
                   sel.eh.unregister("DataChange",self._DataChanged)
                   sel.eh.unregister("DeleteEvent",self._DataDeleted)
            else:
                newsel.append(sel)
        if len (newsel):
            self.Source=tuple(newsel)
        else:
            self.Source=()
        self._DataChanged()

            

    def Update(self):
        """
        Causes redraw of view and notification of linked ViewSelect ojects
        Derived classes must always call this method to generate the redrawing,
        instead calling Redraw, for ViewSelect notification
        """
        self.Redraw()
        self.Invalidate()


    def Invalidate(self):
        """
        Sends ImageChange event to linked objects
        """
        self.eh.event(self.EvImageChange, self)


    def SetCursor(self,cursor):
        """
        Changes the type of cursor.
        A cursor is a ViewSelect object that reacts to mouse and draws itself on the view.
        A tipical example is crosshairs.
        There's no standarized names for cursors (up to derived classes)
        Parameters:
            cursor: ViewSelect object
        """
        if self.Cursor is not None:
            self.Cursor.Destroy()
        self.Cursor=cursor
        self.Cursor.ConnectView(self)


    def Destroy(self,source=None):
        """
        Cleanup.
        As event handlers and callbacks, by keeping a reference to other
        objects, avoid them to be destroyed (and freed from memory), the
        Destroy method has to be explicitly called for View objects
        before they get out of scope.
        I had tried to use weak references in order not to need this method,
        but this generated many other troubles (like impossibility to define
        selections locally, difficult control of the destruction sequence,
        not interceptable error mesassages...).
        """
        if self.Alive==0: return
        self.SetSource(None)
        self.parent=None
        self.ClearMenu()
        if self.Cursor is not None:
            self.Cursor.Destroy()
            self.Cursor=None
        if self.Drawable is not None:
            if hasattr(self.Drawable,"Destroy"):self.Drawable.Destroy()
            self.Drawable.parent=None
        self.Drawable=None

        for sel in self.Source:
            if sel.eh is not None:
                sel.eh.unregister("DataChange",self._DataChanged)
                sel.eh.unregister("DeleteEvent",self._DataDeleted)                    
        self.Source=()

        self.eh=None                                
        self.Alive=0        


    def GetDrawable(self):
        """
        Returns drawable object
        """
        return self.Drawable

    def GetDrawableWidth(self):
        """
        Returns width of drawable area - required by ViewSelect objects
        """
        return self.Drawable.GetWidth()


    def GetDrawableHeight(self):    
        """
        Returns height of drawable area - required by ViewSelect objects
        """
        return self.Drawable.GetHeight()


    def Save(self,filename,format):
        """
        Saves contents
        By default call Save mathod of Drawable, but can be overridden to
        have a different behaviour.
        View classes shall override GetSaveFormats to return supported formats.
        Parameters:
            filename: string, name of the output file
            format: string
        """
        try:
            ret=self.Drawable.Save(str(filename),str(format))
            return ret
        except:
            return 0


#################################
#            POPUP MENU
#################################

    def GetPopupMenu(self):
        """
        Returns popup menu: a Menu object
        
        All menu functionalities can me managed directly through GetPopupMenu and
        by calling Binding's Menu methos.
        The below functions are provided for conveniance.
        Even using them it is important to access Menu object in order to call
        CheckItem/EnableItem/DisableItem and to add cascade menus.
        See:
        GUIBinding.Binding.Menu
        """
        if self.MenuPopup == None:
            self.MenuPopup = Menu(self.Drawable)
        return self.MenuPopup

    def AddMenuPopupItem(self,label,Command,Style='command'):
        """
        Inserts a new item in the popup menu
        Parameters:
        Label:   label of the menu item
        Command: Callback function
        Style:  (Not considered in qt)
               'command' to add a command menu item
               'checkbutton' to add a checkbutton menu item (not used in QtBinding)
        """
        if self.MenuPopup == None:
            self.MenuPopup = Menu(self.Drawable)
        return self.MenuPopup.AddCommand(str(label),Command,Style)
            


    def AddMenuPopupCascade(self,label,menu):
        """
        Inserts a new menu in the popup menu
        Parameters:
        Label:  label of the menu 
        Menu:   menu object
        """
        if self.MenuPopup == None:
            self.MenuPopup = Menu(self.Drawable)
        return  self.MenuPopup.AddCascade(label,menu)


    def AddMenuSeparator(self):
        """
        Inserts a separator in the popup menu
        """
        if self.MenuPopup == None:
            self.MenuPopup = Menu(self.Drawable)
        return self.MenuPopup.AddSeparator()

    def DisableMenuItem(self,index):
        """
        Disables a menu item in the popup menu
        Parameters:
        index:  label or index of the menu item to be disabled
                as returned by AddMenuPopupItem or AddMenuPopupCascade
        """
        self.MenuPopup.DisableItem(index)    
            
    def EnableMenuItem(self,index):
        """
        Enables a menu item in the popup menu
        Parameters:
        index:  label or index of the menu item to be enabled
                as returned by AddMenuPopupItem or AddMenuPopupCascade
        """
        self.MenuPopup.EnableItem(index)    

    def DeleteMenuItem(self,index):
        """
        Deletes a menu item from the popup menu
        Parameters:
        index:  label or index of the menu to item be deleted
                as returned by AddMenuPopupItem or AddMenuPopupCascade
        """
        self.MenuPopup.DeleteItem(index)    

    def ClearMenu(self):
        """
        Removes all items of the popup menu
        """
        if self.MenuPopup is not None:
            self.MenuPopup.Destroy()
            self.MenuPopup=None
        
        
        
#################################
#       MOUSE/KEYBOARD EVENTS
#################################

    def _DoubleClick(self, event):
        """
        Called by derived class / drawable
        """
        if self.Source==(): return
        pos = self.GetPosition(event)
        if pos is not None:
            self.eh.event(self.EvDoubleClick, (self,pos))
            self.EventDoubleClick(pos)


    def _ButtonPressMotion(self, event):
        """
        Called by derived class / drawable
        """
        if self.Source==(): return
        pos = self.GetPosition(event)
        if pos is not None:
            self.eh.event(self.EvButtonPressMotion, (self,pos))
            self.EventButtonPressMotion(pos)

            



    def _ButtonPress(self, event, Botton = None):
        """
        Called by derived class / drawable
        """
        if self.Source==(): return
        pos = self.GetPosition(event)        
        if pos is not None:
            self.eh.event(self.EvButtonPress, (self,pos))
            self.EventButtonPress(pos)
        if(Botton is not None):
            if(Botton == "Mid"):
                if( hasattr(Binding,"StartDrag")):
                    Binding.StartDrag( self)
                

    def _ButtonRelease(self, event):
        """
        Called by derived class / drawable
        """
        if self.Source==(): return
        pos = self.GetPosition(event)
        if pos is not None:
            self.eh.event(self.EvButtonRelease, (self,pos))
            self.EventButtonRelease(pos)
            
    def _Motion(self, event):
        """
        Called by derived class / drawable
        """
        if self.Source==(): return
        pos = self.GetPosition(event)        
        if pos is not None:
            self.eh.event(self.EvMotion, (self,pos))
            self.EventMotion(pos)

    def _RightButtonPress(self, event):
        """
        Called by derived class / drawable
        """
        if self.Source ==(): return 
        if self.MenuPopup != None:
            self.MenuPopup.Show(event)

    def _KeyPress(self, key, flags=None):
        """
        Called by derived class / drawable
        """
        if self.Source==(): return
        self.eh.event(self.EvKeyPress, (self,key,flags))
        self.EventKeyPress(key,flags)


#################################
#            VIRTUALS
#################################

    def CreateDrawable(self):
        """
        Virtual: Implements creation of  a Drawable object (which implements the
        Drawable interface)
        """        
        pass

    def DataChanged(self,source=None):
        """
        Virtual: called to inform that data has changed.
        If source == None all selections should be updated
        Otherwise, the source object that actually changed is passed.
        If a "data" not in source's GetOutput, it should
        be erased from the View's drawable
        For performance reasons DataChanged should be the only method
        to call source's GetOutput, since this call triggers the
        operations of all cascaded sources.
        """
        pass


    def GetPosition(self, event):
        """
        Virtual: Used to transform event coordinates into data coordinates (rule
        known by derived classes). Must return the Position object that
        will be dispatched to ViewSelect objects and to the application
        """
        pass


    def DataCoord2ImageCoord(self,data_coord):
        """
        Virtual: Method that transforms data coordinates object into  
        image coordinates
        DataCoord is a DataPosition object
        Derived class must call DataCoord2SelectionCoord  from 
        its DataSelection object and after transform into view coords (since
        it does not know the data object itself).
        """
        return None

    def ImageCoord2DataCoord(self,view_coord):
        """
        Virtual: Method that transforms View coordinate into a data coordinates
        ViewCoordinate is a tuple (x,y)
        DataCoord is DataPosition object
        Derived class must transform view coords into selection coords and after
        call SelectionCoord2DataCoord  from its DataSelection object (since
        it does not know the data object itself).
        """
        return None


    def Redraw(self):
        """
        Virtual: Derived classes must code the actual drawing overriding this method
        """
        pass


#################################
#    OVERRIDABLES
#################################


    def EventButtonPress(self,pos):
        """
        Virtual
        Parameters:
           pos: position of the event as return by GetPosition
        """
        pass

    def EventButtonPressMotion(self,pos):
        """
        Virtual
        Parameters:
           pos: position of the event as return by GetPosition
        """
        pass

    def EventButtonRelease(self,pos):
        """
        Virtual
        Parameters:
           pos: position of the event as return by GetPosition
        """
        pass

    def EventMotion(self,pos):
        """
        Virtual
        Parameters:
           pos: position of the event as return by GetPosition
        """
        pass

    def EventDoubleClick(self,pos):
        """
        Virtual
        Parameters:
           pos: position of the event as return by GetPosition
        """
        pass
    
    def EventKeyPress(self,key,flags):
        """
        Virtual: Keyboard event callback
        Parameters:
           key: python string with key (if a caracter) or keycode as defined in
                GuiBinding.Bindings.KeyCodes
           flags: indicates state of auxiliary keys 
        """        
        pass

    def GetSaveFormats(self):
        """
        Returns tuple with supported save formats
        """
        return ()
        
