"""
    ImageViewSelect.py
    ViewSelect derived classes to ImageView
    

    Interface:
    ===========================
    ViewSelect interface
"""

from PyDVT import __version__,__date__,__author__

################################################################################  
from ViewSelect import *
import DataSelection
from Data import DataPosition



class ImageViewSelect(ViewSelectDefault):
    """
    ViewSelectDefault derived, base class for common ViewSelection objects over ImageView.
    This class adds keyboard management.
    This class (and derived) cannot have data = None in the contructor.
    """
    def KeyPress(self,key,flags):
        """
        Virtual: See ViewSelect
        """
        if "BoundingRect" not in self.Selection.keys(): return
        (p0,p1)=self.Selection["BoundingRect"]
        (x0,y0,x1,y1)=p0.PageCoord[0],p0.PageCoord[1],p1.PageCoord[0],p1.PageCoord[1]
        if key==KeyCodes["KEY_UP"]:
            y0,y1=y0-1,y1-1
        elif key==KeyCodes["KEY_DOWN"]:
            y0,y1=y0+1,y1+1
        elif key==KeyCodes["KEY_LEFT"]:
            x0,x1=x0-1,x1-1
        elif key==KeyCodes["KEY_RIGHT"]:
            x0,x1=x0+1,x1+1
        else:
            return
        self.SetLimitedSelectionPos((DataPosition(p0.PageIndex,(x0,y0)),DataPosition(p1.PageIndex,(x1,y1))))


    def SetLimitedSelectionPos(self,bounding_rect):
      """
      See ViewSelect
      """
      if self.Data.IsCoordValid(bounding_rect[0]) and self.Data.IsCoordValid(bounding_rect[1]):
          self.SetSelectionPos(bounding_rect)

          
    #VIRTUALS
    def GetDataSelection(self):
        pass


    
class ImageViewSelectHLine(ImageViewSelect):
  """
  Implemets an horizontal line selection
  """
  
  def Motion(self,pos):
    """
    Virtual: See ViewSelect
    """
    mouse_position=pos.DataCoord
    for view in self.ViewList.keys():   self.DrawTemp(view,mouse_position)


  def Release(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.SetSelectionPos((pos.DataCoord,pos.DataCoord))      


  def DrawSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if "BoundingRect" not in self.Selection.keys(): return
    pos=view.DataCoord2ImageCoord(self.Selection["BoundingRect"][0])
    width=view.GetDrawableWidth()
    if "SelectDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDraw"]=view.Drawable.PutLine(0,pos[1],width-1, pos[1],self.Pen)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDraw"], 0, pos[1], width-1, pos[1])
            

  def DrawTemp(self,view,mouse_position):
    """
    Virtual: See ViewSelectDefault
    """
    pos=view.DataCoord2ImageCoord(mouse_position)
    width=view.GetDrawableWidth()
    if "TempDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["TempDraw"]=view.Drawable.PutLine(0,pos[1],width-1, pos[1],Pen(self.Pen.color,self.Pen.width,"dotted"))
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempDraw"], 0, pos[1], width-1, pos[1])


  def GetDataSelection(self):
    """
    Virtual: See ImageViewSelect
    """
    if "BoundingRect" not in self.Selection.keys(): return None
    return DataSelection.OrthoLineSelection(self.Data,[0,self.Selection["BoundingRect"][0].PageCoord[1]],["ALL",1],self.Selection["BoundingRect"][0].PageIndex)
 

  def GetType(self):
    """
    Virtual: See ViewSelect
    """
    return "HLine"

      

class ImageViewSelectVLine(ImageViewSelect):
  """
  Implemets a vertical line selection
  """
  
  def Motion(self,pos):
    """
    Virtual: See ViewSelect
    """
    mouse_position=pos.DataCoord
    for view in self.ViewList.keys():   self.DrawTemp(view,mouse_position)


  def Release(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.SetSelectionPos((pos.DataCoord,pos.DataCoord))      

    
  def DrawSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if "BoundingRect" not in self.Selection.keys(): return    
    pos=view.DataCoord2ImageCoord(self.Selection["BoundingRect"][0])
    height=view.GetDrawableHeight()
    if "SelectDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDraw"]=view.Drawable.PutLine(pos[0],0,pos[0],height-1,self.Pen)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDraw"], pos[0], 0, pos[0], height-1)


  def DrawTemp(self,view,mouse_position):
    """
    Virtual: See ViewSelectDefault
    """
    pos=view.DataCoord2ImageCoord(mouse_position)
    height=view.GetDrawableHeight()
    if "TempDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["TempDraw"]=view.Drawable.PutLine(pos[0],0,pos[0],height-1,Pen(self.Pen.color,self.Pen.width,"dotted"))
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempDraw"], pos[0], 0, pos[0], height-1)    


  def GetDataSelection(self):
    """
    Virtual: See ImageViewSelect
    """
    if "BoundingRect" not in self.Selection.keys(): return None
    return DataSelection.OrthoLineSelection(self.Data,[self.Selection["BoundingRect"][0].PageCoord[0],0],[1,"ALL"],self.Selection["BoundingRect"][0].PageIndex)


  def GetType(self):
    """
    Virtual: See ViewSelect
    """
    return "VLine"

class ImageViewSelectRect(ImageViewSelect):
  """
  Implemets a rectangle selection
  """
  def __init__(self, data=None, callback=None):
     ImageViewSelect.__init__(self, data, callback)
     self.Created= 0
     self.Anchor= None
     self.Moving= 0

  def PressMotion(self,pos):
    """
    Virtual: See ViewSelect
    """
    mouse_position=pos.DataCoord
    if not self.Created:
       for view in self.ViewList.keys():
          self.DrawTemp(view,mouse_position)
    else:
       if self.Anchor is not None:
          self.Moving= 1
	  x= pos.DataCoord.PageCoord[0]-self.Anchor.PageCoord[0]
	  y= pos.DataCoord.PageCoord[1]-self.Anchor.PageCoord[1]
	  self.Anchor= pos.DataCoord
	  if (x,y)!=(0,0):
	      self.DrawMove(x,y)

  def Press(self,pos):
    """
    Virtual: See ViewSelect
    """
    if self.Created:
      (b,e)= self.Selection["BoundingRect"]
      xr= (min(b.PageCoord[0],e.PageCoord[0]), max(b.PageCoord[0],e.PageCoord[0]))
      yr= (min(b.PageCoord[1],e.PageCoord[1]), max(b.PageCoord[1],e.PageCoord[1]))
      (x,y)= pos.DataCoord.PageCoord
      if x>xr[0] and x<xr[1] and y>yr[0] and y<yr[1]:
	 self.Anchor= pos.DataCoord
      else:
         self.Anchor= None
    if not self.Created or self.Anchor is None:
      self.Erase("ALL")
      self.Selection={}
      self.Begin=pos.DataCoord
      self.Created= 0

  def Release(self,pos):
    """
    Virtual: See ViewSelect
    """
    if hasattr(self,"Begin")==0: return
    if not self.Created:
       self.SetSelectionPos((self.Begin,pos.DataCoord))
    else:
       if self.Moving:
           self.Anchor= None
           self.Moving= 0
	   self.NotifySelection()

  def DrawMove(self, x, y):
    (begin, end)= self.Selection["BoundingRect"]
    beginCoord= (begin.PageCoord[0]+x, begin.PageCoord[1]+y)
    endCoord= (end.PageCoord[0]+x, end.PageCoord[1]+y)
    
    #self.Selection["BoundingRect"]= (begin, end)
    #for view in self.ViewList.keys():
    #            self.DrawSelection(view)

    xmax= 0
    ymax= 0
    for view in self.ViewList.keys():
       if xmax: xmax= min(view.GetDrawableWidth()-1, xmax)
       else:    xmax= view.GetDrawableWidth()-1
       if ymax: ymax= min(view.GetDrawableHeight()-1, ymax)
       else:	ymax= view.GetDrawableHeight()-1

    if beginCoord[0]>=0 and beginCoord[1]>=0 and \
	endCoord[0]<=xmax and endCoord[1]<=ymax:
	    begin.PageCoord= beginCoord
	    end.PageCoord= endCoord
	    self.Selection["BoundingRect"]= (begin, end)
            for view in self.ViewList.keys():
                self.DrawSelection(view)

  def DrawSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if "BoundingRect" not in self.Selection.keys(): return
    self.EraseTemp(view)    
    start=view.DataCoord2ImageCoord(self.Selection["BoundingRect"][0])
    end=  view.DataCoord2ImageCoord(self.Selection["BoundingRect"][1])    
    if "SelectDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDraw"]=view.Drawable.PutRectangle(start[0], start[1], end[0], end[1],self.Pen,self.Brush)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDraw"], start[0], start[1], end[0], end[1])
    self.Created= 1


  def DrawTemp(self,view,mouse_position):
    """
    Virtual: See ViewSelectDefault
    """
    if hasattr(self,"Begin")==0: return
    start=view.DataCoord2ImageCoord(self.Begin)
    end=  view.DataCoord2ImageCoord(mouse_position)    
    if "TempDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["TempDraw"]=view.Drawable.PutRectangle(start[0], start[1], end[0], end[1],self.Pen,self.Brush)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempDraw"], start[0], start[1], end[0], end[1])


  def GetDataSelection(self):
    """
    Virtual: See ImageViewSelect
    """
    if "BoundingRect" not in self.Selection.keys(): return None
    begin=self.Selection["BoundingRect"][0]
    end=self.Selection["BoundingRect"][1]
    return DataSelection.RectSelection(self.Data,[min(begin.PageCoord[0],end.PageCoord[0]),min(begin.PageCoord[1],end.PageCoord[1])],[abs(begin.PageCoord[0]-end.PageCoord[0]),abs(begin.PageCoord[1]-end.PageCoord[1])],end.PageIndex)


  def GetType(self):
    """
    Virtual: See ViewSelect
    """
    return "Rect"

class StaticImageViewSelectRect(ImageViewSelect):
  """
  Implemets a rectangle selection
  """
  
  def PressMotion(self,pos):
    """
    Virtual: See ViewSelect
    """
    mouse_position=pos.DataCoord
    for view in self.ViewList.keys():   self.DrawTemp(view,mouse_position)


  def Press(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.Erase("ALL")
    self.Selection={}
    self.Begin=pos.DataCoord


  def Release(self,pos):
    """
    Virtual: See ViewSelect
    """
    if hasattr(self,"Begin")==0: return
    self.SetSelectionPos((self.Begin,pos.DataCoord))


  def DrawSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if "BoundingRect" not in self.Selection.keys(): return
    self.EraseTemp(view)    
    start=view.DataCoord2ImageCoord(self.Selection["BoundingRect"][0])
    end=  view.DataCoord2ImageCoord(self.Selection["BoundingRect"][1])    
    if "SelectDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDraw"]=view.Drawable.PutRectangle(start[0], start[1], end[0], end[1],self.Pen,self.Brush)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDraw"], start[0], start[1], end[0], end[1])


  def DrawTemp(self,view,mouse_position):
    """
    Virtual: See ViewSelectDefault
    """
    if hasattr(self,"Begin")==0: return
    start=view.DataCoord2ImageCoord(self.Begin)
    end=  view.DataCoord2ImageCoord(mouse_position)    
    if "TempDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["TempDraw"]=view.Drawable.PutRectangle(start[0], start[1], end[0], end[1],self.Pen,self.Brush)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempDraw"], start[0], start[1], end[0], end[1])


  def GetDataSelection(self):
    """
    Virtual: See ImageViewSelect
    """
    if "BoundingRect" not in self.Selection.keys(): return None
    begin=self.Selection["BoundingRect"][0]
    end=self.Selection["BoundingRect"][1]
    return DataSelection.RectSelection(self.Data,[min(begin.PageCoord[0],end.PageCoord[0]),min(begin.PageCoord[1],end.PageCoord[1])],[abs(begin.PageCoord[0]-end.PageCoord[0]),abs(begin.PageCoord[1]-end.PageCoord[1])],end.PageIndex)


  def GetType(self):
    """
    Virtual: See ViewSelect
    """
    return "Rect"

class ImageViewSelectLine(ImageViewSelect):
  """
  Implemets a generic line selection
  """
  
  def PressMotion(self,pos):
    """
    Virtual: See ViewSelect
    """
    mouse_position=pos.DataCoord
    for view in self.ViewList.keys():   self.DrawTemp(view,mouse_position)


  def Press(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.Erase("ALL")
    self.Selection={}
    self.Begin=pos.DataCoord


  def Release(self,pos):
    """
    Virtual: See ViewSelect
    """
    if hasattr(self,"Begin")==0: return
    self.SetSelectionPos((self.Begin,pos.DataCoord))


  def DrawSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if "BoundingRect" not in self.Selection.keys(): return
    self.EraseTemp(view)
    start=view.DataCoord2ImageCoord(self.Selection["BoundingRect"][0])
    end=  view.DataCoord2ImageCoord(self.Selection["BoundingRect"][1])    
    if "SelectDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDraw"]=view.Drawable.PutLine(start[0], start[1], end[0], end[1],self.Pen)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDraw"], start[0], start[1], end[0], end[1])


  def DrawTemp(self,view,mouse_position):
    """
    Virtual: See ViewSelectDefault
    """
    if hasattr(self,"Begin")==0: return
    start=view.DataCoord2ImageCoord(self.Begin)
    end=  view.DataCoord2ImageCoord(mouse_position)    
    if "TempDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["TempDraw"]=view.Drawable.PutLine(start[0], start[1], end[0], end[1],self.Pen)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempDraw"], start[0], start[1], end[0], end[1])


  def GetDataSelection(self):
    """
    Virtual: See ImageViewSelect
    """
    if "BoundingRect" not in self.Selection.keys(): return None
    begin=self.Selection["BoundingRect"][0]
    end=self.Selection["BoundingRect"][1]
    return DataSelection.LineSelection(self.Data,begin.PageCoord,end.PageCoord,end.PageIndex)    


  def GetType(self):
    """
    Virtual: See ViewSelect
    """
    return "Line"



class ImageViewSelectMousePosition(ViewSelectDefault):
  """
  Implemets a position selection - continous event generation
  """
  
  def Motion(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.Selection["Position"]=pos
    self.NotifySelection()


  def PressMotion(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.Motion(pos)


  def GetType(self):
    """
    Virtual: See ViewSelect
    """
    return "MousePosition"


class ImageViewSelectCrosshairs(ViewSelectDefault):
  """
  Implemets a crosshair selection - continous event generation + 2 orthogonal lines
  """
  
  def __init__(self,data=None,callback=None):
    ViewSelect.__init__(self,data,callback)
    self.Pen=Pen((0,0,0),0,"dotted")

        
  def Motion(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.Selection["Position"]=pos
    for view in self.ViewList.keys():  self.DrawSelection(view)
    self.NotifySelection()


  def PressMotion(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.Motion(pos)


  def DrawSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if self.Selection=={}: return
    pos=self.Selection["Position"].ImageCoord
    height=view.GetDrawableHeight()
    width=view.GetDrawableWidth()
    if "SelectDrawV" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDrawV"]=view.Drawable.PutLine(pos[0],0, pos[0], height-1,self.Pen)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDrawV"], pos[0], 0, pos[0], height-1)
    if "SelectDrawH" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDrawH"]=view.Drawable.PutLine(0, pos[1], width-1, pos[1],self.Pen)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDrawH"], 0, pos[1], width-1, pos[1])


  def EraseSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    for i in ["SelectDrawV","SelectDrawH"]:
      if i in self.ViewList[view].keys():
        view.Drawable.EraseObject(self.ViewList[view][i])
        del self.ViewList[view][i]


  def GetType(self):
    """
    Virtual: See ViewSelect
    """
    return "Crosshairs"



class ImageViewSelectPoint(ImageViewSelect):
  """
  Implemets a one point selection
  """
  
  def PressMotion(self,pos):
    """
    Virtual: See ViewSelect
    """
    mouse_position=pos.DataCoord
    for view in self.ViewList.keys():   self.DrawTemp(view,mouse_position)


  def Press(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.Erase("ALL")
    self.Selection={}
    mouse_position=pos.DataCoord
    for view in self.ViewList.keys():  self.DrawTemp(view,mouse_position)


  def Release(self,pos):
    """
    Virtual: See ViewSelect
    """
    self.SetSelectionPos((pos.DataCoord,pos.DataCoord))


  def DrawSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    self.EraseTemp(view)
    pos=self.Selection["BoundingRect"][0]
    start=view.DataCoord2ImageCoord(pos)
    end=(start[0]+1,start[1]+1)
    
    crossLength=3*(end[1]-start[1])
    crossWidth=end[0]-start[0]
    
    upCrossStart=(start[0],start[1]-crossWidth-crossLength)
    upCrossEnd=(start[0]+crossWidth,start[1]-crossWidth)
    leftCrossStart=(start[0]-crossWidth-crossLength,start[1])
    leftCrossEnd=(start[0]-crossWidth,start[1]+crossWidth)
    bottomCrossStart=(start[0],start[1]+2*crossWidth)
    bottomCrossEnd=(start[0]+crossWidth,start[1]+2*crossWidth+crossLength)
    rightCrossStart=(start[0]+2*crossWidth,start[1])
    rightCrossEnd=(start[0]+2*crossWidth+crossLength,start[1]+crossWidth)

    if "SelectDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["SelectDraw"]=view.Drawable.PutRectangle(start[0], start[1], end[0], end[1],self.Pen,self.Brush)
        self.ViewList[view]["CrossUp"]=view.Drawable.PutRectangle(upCrossStart[0], upCrossStart[1], upCrossEnd[0], upCrossEnd[1],self.Pen,self.Brush)
        self.ViewList[view]["CrossLeft"]=view.Drawable.PutRectangle(leftCrossStart[0], leftCrossStart[1], leftCrossEnd[0], leftCrossEnd[1],self.Pen,self.Brush)
        self.ViewList[view]["CrossBottom"]=view.Drawable.PutRectangle(bottomCrossStart[0], bottomCrossStart[1], bottomCrossEnd[0], bottomCrossEnd[1],self.Pen,self.Brush)
        self.ViewList[view]["CrossRight"]=view.Drawable.PutRectangle(rightCrossStart[0], rightCrossStart[1], rightCrossEnd[0], rightCrossEnd[1],self.Pen,self.Brush)
    else:     
        view.Drawable.SetObjectCoords(self.ViewList[view]["SelectDraw"], start[0], start[1], end[0], end[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["CrossUp"],upCrossStart[0], upCrossStart[1], upCrossEnd[0], upCrossEnd[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["CrossLeft"],leftCrossStart[0], leftCrossStart[1], leftCrossEnd[0], leftCrossEnd[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["CrossBottom"],bottomCrossStart[0], bottomCrossStart[1], bottomCrossEnd[0], bottomCrossEnd[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["CrossRight"],rightCrossStart[0], rightCrossStart[1], rightCrossEnd[0], rightCrossEnd[1])


  def DrawTemp(self,view,mouse_position):
    """
    Virtual: See ViewSelectDefault
    """
    start=view.DataCoord2ImageCoord(mouse_position)
    end=(start[0]+1,start[1]+1)
    
    
    crossLength=3*(end[1]-start[1])
    crossWidth=end[0]-start[0]
    
    upCrossStart=(start[0],start[1]-crossWidth-crossLength)
    upCrossEnd=(start[0]+crossWidth,start[1]-crossWidth)
    leftCrossStart=(start[0]-crossWidth-crossLength,start[1])
    leftCrossEnd=(start[0]-crossWidth,start[1]+crossWidth)
    bottomCrossStart=(start[0],start[1]+2*crossWidth)
    bottomCrossEnd=(start[0]+crossWidth,start[1]+2*crossWidth+crossLength)
    rightCrossStart=(start[0]+2*crossWidth,start[1])
    rightCrossEnd=(start[0]+2*crossWidth+crossLength,start[1]+crossWidth)
    
    if "TempDraw" not in self.ViewList[view].keys():
        self.ViewList[view]["TempDraw"]=view.Drawable.PutRectangle( start[0], start[1], end[0], end[1],self.Pen,self.Brush)
        self.ViewList[view]["TempCrossUp"]=view.Drawable.PutRectangle(upCrossStart[0], upCrossStart[1], upCrossEnd[0], upCrossEnd[1],self.Pen,self.Brush)
        self.ViewList[view]["TempCrossLeft"]=view.Drawable.PutRectangle(leftCrossStart[0], leftCrossStart[1], leftCrossEnd[0], leftCrossEnd[1],self.Pen,self.Brush)
        self.ViewList[view]["TempCrossBottom"]=view.Drawable.PutRectangle(bottomCrossStart[0], bottomCrossStart[1], bottomCrossEnd[0], bottomCrossEnd[1],self.Pen,self.Brush)
        self.ViewList[view]["TempCrossRight"]=view.Drawable.PutRectangle(rightCrossStart[0], rightCrossStart[1], rightCrossEnd[0], rightCrossEnd[1],self.Pen,self.Brush)
    else:
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempDraw"], start[0], start[1], end[0], end[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempCrossUp"],upCrossStart[0], upCrossStart[1], upCrossEnd[0], upCrossEnd[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempCrossLeft"],leftCrossStart[0], leftCrossStart[1], leftCrossEnd[0], leftCrossEnd[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempCrossBottom"],bottomCrossStart[0], bottomCrossStart[1], bottomCrossEnd[0], bottomCrossEnd[1])
        view.Drawable.SetObjectCoords(self.ViewList[view]["TempCrossRight"],rightCrossStart[0], rightCrossStart[1], rightCrossEnd[0], rightCrossEnd[1])


  def EraseSelection(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if "SelectDraw" in self.ViewList[view].keys():
        for draw in self.ViewList[view].keys():
            #if (draw != "Mode"):
            if draw in ("SelectDraw","CrossUp","CrossLeft","CrossBottom","CrossRight"):
                view.Drawable.EraseObject(self.ViewList[view][draw])
                del self.ViewList[view][draw]

  def EraseTemp(self,view):
    """
    Virtual: See ViewSelectDefault
    """
    if "TempDraw" in self.ViewList[view].keys():
        for draw in self.ViewList[view].keys():
            if draw in ("TempDraw","TempCrossUp","TempCrossLeft","TempCrossBottom","TempCrossRight"):
                view.Drawable.EraseObject(self.ViewList[view][draw])
                del self.ViewList[view][draw]


  def GetDataSelection(self):
    """
    Virtual: See ImageViewSelect
    """
    if "BoundingRect" not in self.Selection.keys(): return None
    return DataSelection.PointSelection(self.Data,self.Selection["BoundingRect"][0])


  def GetType(self):
      return "Point"
