"""
    ImageView.py
    View derived class for 2d image displaying


    Interface:
    ===========================
    View interface
    SetZoom
"""

from PyDVT import __version__,__date__,__author__

from View import *
import ImageViewSelect
from Filter import Filter
from Data import Data
from DataSelection import RectSelection

import Command
import spslut
import Numeric

colormap_table={}
NUM_COLORS_8BITS=100
def SetColormapTable():
    for colormap in (spslut.GREYSCALE,spslut.TEMP,spslut.RED,spslut.GREEN,spslut.BLUE,spslut.REVERSEGREY):
        table=Numeric.fromstring(spslut.palette(NUM_COLORS_8BITS,colormap),'l')        
        if GetDisplayProperties()["SPSLUT_MODE"]=="BGRX": table= (Numeric.bitwise_and(table.byteswapped(),0xFFFFFF00)/0x100)
        aux=Numeric.zeros((256,),'l')
        aux[50:(50+NUM_COLORS_8BITS)]=table
        colormap_table[colormap]=list(aux)

################################################################################


class ColormapFilter(Filter):
    """
    View's standard filter 
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0):
        """
        Parameters:
            name: optional name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
        """
        Filter.__init__(self,source,synchronized,buffer_input)
        self.MinMax=(0,0)
        self.AbsoluteMinMax=(0,0)
	self.Size= (0,0)
        self.Scale="Linear"
        self.Colormap="Temperature"
        self.SpslutScale=spslut.LINEAR
        self.SpslutColormap=spslut.TEMP
        self.AutoScale=1
        self.ColormapChange=0
        self.colormap_table=None
        if name==None: self.name=str(self)
        else: self.name=name
        self.ColormapSelection=None
        self.reduc=1
        self.fastreduc=0
        self.Gamma=3.0
        self.UpdateRange=None
    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            "name" : string
            "data" : 2d-NumPy array with the original data
            "image" : String with RGB representation of data
            "size": Size of the image
            "depth": 8 or 32
            "colormap_table": if "depth"==8, palette table, otherwize None           
            "filterchange" : flag indicating just a change in
                             filter parameters has taken place
                             (not the actual data).
                             ImageView always keeps Zoom/Position
                             if this flag is set.
        """        
        sel=self.GetInput()
        if "data" not in sel.keys(): return {"name":self.name}
        if sel["data"] is not None:
            disp_props=GetDisplayProperties()
            if disp_props["DISPLAY_DEPTH"]==8:
                global NUM_COLORS_8BITS
                if sel["data"].typecode() in ("b","1","s"): data=sel["data"].astype("l")
                else: data=sel["data"]
                if self.UpdateRange is not None:
                        data=data[self.UpdateRange[0]:self.UpdateRange[1]]
                (image,size,self.MinMax)=spslut.transform(data , (self.reduc,self.fastreduc), 
                                                    (self.SpslutScale, self.Gamma), 'L',
                                                    self.SpslutColormap, self.AutoScale, self.MinMax,(50,49+NUM_COLORS_8BITS))
                                
                #All this pad mess is done because one qt expects 4-byte aligned data.
                #I think one of the constructors of QImage could solve the problem
                #but it is not implemented in PyQt 
                if disp_props["ALIGN_8BIT_IMAGES"]:
                    pad=(4-(size[0] % 4))%4
                    if pad:
                        img_arr=Numeric.reshape(Numeric.fromstring(image,"b"),(size[1],size[0]))
                        image=Numeric.concatenate((img_arr,(Numeric.zeros((size[1],pad),"b"))),1).tostring()
                
                global colormap_table
                if colormap_table=={}: SetColormapTable()
                self.colormap_table=colormap_table[self.SpslutColormap]
            else:
               data=sel["data"]
               if self.UpdateRange is not None:
                     data=data[self.UpdateRange[0]:self.UpdateRange[1]]
               (image,size,self.MinMax)=spslut.transform(sel["data"] , (self.reduc,self.fastreduc), 
                                                    (self.SpslutScale, self.Gamma), disp_props["SPSLUT_MODE"], 
                                                    self.SpslutColormap, self.AutoScale, self.MinMax)            
	    self.Size= size
            ###TODO: Check for troubles in that
            if self.AutoScale: self.AbsoluteMinMax=self.MinMax
            #self.AbsoluteMinMax=self.MinMax
            ret = {"name":self.name,"image":image,"size":size,"datachanged":not self.ColormapChange,"data":sel["data"]}
            if "xdata" in sel.keys(): ret["xdata"]=sel["xdata"]
            if "ydata" in sel.keys(): ret["ydata"]=sel["ydata"]            
            ret["depth"]=disp_props["DISPLAY_DEPTH"]
            ret["colormap_table"]=self.colormap_table
            ret["update_range"]=self.UpdateRange
            self.UpdateRange=None
            return ret
        
        self.AbsoluteMinMax=(0,0)
        return {"name":self.name}


    def GetColormapOutput(self):
	ret= {}
	if self.MinMax==(0,0) or self.Size==(0,0): return ret

	step= float(self.MinMax[1]-self.MinMax[0])/float(self.Size[0])
	line= Numeric.arrayrange(self.MinMax[0], self.MinMax[1], step)
	data= Numeric.array([line for idx in range(10)])

	disp_props= GetDisplayProperties()
        if disp_props["DISPLAY_DEPTH"]==8:
        	global NUM_COLORS_8BITS
		data= data.astype("l")
		(image,size,minmax)= spslut.transform(data, (1,0), (self.SpslutScale, self.Gamma), "L",
						self.SpslutColormap, self.AutoScale, self.MinMax,
						(50,49+NUM_COLORS_8BITS))
		if disp_props["ALIGN_8BIT_IMAGES"]:
			pad=(4-(size[0] % 4))%4
                    	if pad:
                        	img_arr=Numeric.reshape(Numeric.fromstring(image,"b"),(size[1],size[0]))
                        	image=Numeric.concatenate((img_arr,(Numeric.zeros((size[1],pad),"b"))),1).tostring()
	else:
		(image,size,minmax)=spslut.transform(data, (1,0), (self.SpslutScale, self.Gamma), 
					disp_props["SPSLUT_MODE"], self.SpslutColormap, self.AutoScale, self.MinMax)            
	return {"image":image, "size":size, "depth":disp_props["DISPLAY_DEPTH"], 
			"colormap_table":self.colormap_table, "minmax":minmax, "scale":self.Scale}


    def ConnectColormap(self,colormap_sel):
        """
        Connects the ColormapFilter object to a ColormapSelect object. The view object
        will answer to ColormapChange events from Colormap object
        Parameter:
        colormap: colormap object
        """
        self.DisconnectColormap()
        self.ColormapSelection=colormap_sel
        self.ColormapSelection.eh.register("ColormapChange", self.SetColormapParameters)
        

    def DisconnectColormap(self):
        """
        Disconnects the ColormapFilter object from a ColormapSelect object.
        Parameter:
        colormap: colormap object
        """
        if self.ColormapSelection is not None:
            if self.ColormapSelection.eh is not None:
                self.ColormapSelection.eh.unregister("ColormapChange", self.SetColormapParameters)
            self.ColormapSelection=None
        


    def GetColormapParameters(self):
        """
        Gets colormap parameters of the view object, a dictionary, with the keys:
            AbsoluteMinMax: tuple (min,max) with the lowest and highest values of the data
            MinMax: tuple (min,max) with the lowest and highest values used for the colormap
            Scale: "Linear", "Logarithmic", "Gamma"
            Colormap: "GrayScale","Temperature","Red","Green","Blue" or "RevGrey"
            Gamma: Factor for gamma scale
            AutoScale: if set, generates MinMax according to the displayed data
                       (takes highest and lowest values of the data)
        """
        return ({"AbsoluteMinMax":self.AbsoluteMinMax,"MinMax":self.MinMax,"Scale":self.Scale,"Colormap":self.Colormap,"AutoScale":self.AutoScale,"Gamma":self.Gamma})


    def SetColormapParameters(self,colormap_pars):
        """
        Sets colormap parameters of the view object
        Parameters
        colormap_pars: dictionary with the following keys (as described in GetColormapParameters):
            MinMax
            Scale
            Colormap
            AutoScale
            Gamma
            
            AbsoluteMinMax is not considered
        """
        colormap_table_change=0
        if "MinMax"    in colormap_pars.keys(): self.MinMax=colormap_pars["MinMax"]
        if "Scale"     in colormap_pars.keys(): self.Scale=colormap_pars["Scale"]
        if "Colormap"  in colormap_pars.keys(): self.Colormap=colormap_pars["Colormap"]
        if "AutoScale" in colormap_pars.keys(): self.AutoScale=colormap_pars["AutoScale"]
        if "Gamma" in colormap_pars.keys(): self.Gamma=colormap_pars["Gamma"]
        if self.AbsoluteMinMax ==(0,0): self.AbsoluteMinMax= self.MinMax

        if self.Colormap=="GrayScale":self.SpslutColormap=spslut.GREYSCALE
        elif self.Colormap=="Temperature":self.SpslutColormap=spslut.TEMP
        elif self.Colormap=="Red":self.SpslutColormap=spslut.RED
        elif self.Colormap=="Green":self.SpslutColormap=spslut.GREEN
        elif self.Colormap=="Blue":self.SpslutColormap=spslut.BLUE
        elif self.Colormap=="RevGrey":self.SpslutColormap=spslut.REVERSEGREY
        else: self.SpslutColormap=spslut.TEMP

        if self.Scale=="Linear":self.SpslutScale=spslut.LINEAR
        elif self.Scale=="Logarithmic":self.SpslutScale=spslut.LOG
        elif self.Scale=="Gamma":self.SpslutScale=spslut.GAMMA
        else: self.SpslutScale=spslut.LINEAR                
        
        self.ColormapChange=1
        self.Refresh()
        self.ColormapChange=0

    def PartialUpdate(self,update_range):
 
                                                                  """
 
                                                                  Calls for redraw of sub-image, range of lines changed in the data array.
 
                                                                  Parameters:
 
                                                                  update_range: Y range, tuple (start,end)
 
                                                                  """
 
                                                                  self.UpdateRange=update_range
 
                                                                  self.Refresh()
 
                                                          
        


    def SetReducParameters(self,reduc,fastreduc):
        """
        Sets reduction parameters 
        Parameters:
        reduc: scale to the reduction
        fastreduc: if non-zero just eliminates rows/cols 
        """
        self.reduc=reduc
        self.fastreduc=fastreduc
        self.Refresh()

    def GetReducParameters(self):
	return (self.reduc, self.fastreduc)
    
    def Destroy(self,source=None):
        """
        Virtual: See View.Destroy
        """
        self.DisconnectColormap()
        Filter.Destroy(self,source)


    def DataCoord2SelectionCoord(self,data_coord, trim=1):
        """
        Overriden to corret coords convertion based on reduction applied.
        """
        source=self.GetSource()
        if source is not None:
            coord=source.DataCoord2SelectionCoord(data_coord, trim)
            return (coord[0]/self.reduc,coord[1]/self.reduc)
        return (-1,-1)


    def SelectionCoord2DataCoord(self,selection_coord):
        """
        Overriden to corret coords convertion based on reduction applied.
        """
        source=self.GetSource()
        if source is not None:
            coord=(selection_coord[0]*self.reduc,selection_coord[1]*self.reduc,)
            return source.SelectionCoord2DataCoord(coord)
        return DataPosition(-1,(-1,-1))


class ImageStatFilter(Filter):
    def __init__(self, name=None, source=None, synchronized=1, buffer_input=0):
        Filter.__init__(self, source, synchronized, buffer_input)
        self.name= name or str(self)
    
    def GetOutput(self):
        ret= {}
        sel=self.GetInput()
	data= sel.get("data", None)
        if data is not None:
            line= Numeric.ravel(data)
            ret["min"]= Numeric.minimum.reduce(line)
            ret["max"]= Numeric.maximum.reduce(line)
            line= line.astype(Numeric.Float32)
            nb= float(line.shape[0])
            ret["mean"]= Numeric.add.reduce(line)/nb
            x= line - ret["mean"]
            ret["std"]= Numeric.sqrt(Numeric.add.reduce(x*x)/(nb-1.0))
        return ret

class ImageFilter(Filter):
    """
    View's standard filter to previously RGB generated images
    (ImageFileData).
    Obviously spslut is not used in this case.
    """
    def __init__(self,name=None,source=None,synchronized=1,buffer_input=0):
        """
        Parameters:
            name: optional name of the function
            source: Source Filter/DataSelection
            synchronized: non-zero if on-line
            buffer_input: see Filter.__init__
        """
        Filter.__init__(self,source,synchronized,buffer_input)
        if name==None: self.name=str(self)
        else: self.name=name

    def GetOutput(self):
        """
        Returns the selection data (dictionary)
        Keys:
            "name" : string
            "data" : 2d-NumPy array with the original data (RGB values)
            "image" : String transformation of data
            "size" : dimentions of image
            "depth": always 32
            "colormap_table": None           
        """
        sel=self.GetInput()
        if "data" not in sel.keys(): return {"name":self.name}
        if sel["data"] is None:  return {"name":self.name}

        if GetDisplayProperties()["SPSLUT_MODE"]=="BGRX":
            if sys.byteorder == "big":
                image= (Numeric.bitwise_and(sel["data"],0xFFFFFF00)/0x100).tostring()
            else:
                image= (Numeric.bitwise_and(sel["data"].byteswapped(),0xFFFFFF00)/0x100).tostring()
            
        else:
            if sys.byteorder == "big":
                image = sel["data"].byteswapped().tostring()
            else:
                image = sel["data"].tostring()
            
        size= (sel["data"].shape[1],sel["data"].shape[0])
        return {"name":self.name,"image":image,"size":size,"depth":32,"colormap_table":None}



class Image(ColormapFilter):
    """
    View's simplified filter for direct NumPy interface.
    (creates and hides it's own data object)
    """
    def __init__(self,data=None,name=None):
        """
        Parameters:
            data: NumPy 2d array to be displayed
        """
        self.data=Data()
        if data is not None: self.data.AppendPage(array=data)                
        ColormapFilter.__init__(self,name,RectSelection(self.data))
            
    def Destroy(self,source=None):
        """
        If filter gets out of scope, this should be called to destroy
        internally created data object.
        """
        if self.Alive==0: return
        self.Alive=0
        self.data.Destroy()


class ImageView(View):
    """
    View derived class for image representation of 2d data.
    This class accepts just one source object. If the SetData
    is called with multiple sources, only the first one is considered.    
    ImageView expects the following keys from it's source (GetOutput
    method):
    "image": string, image RGBX representation, as returned by spslut.
    "size": size of image, tuple (x,y) .
    "depth": 8 or 32(default)
    "colormap_table": if "depth"==8, palette table, otherwize None           
    "name": Optional. Not used for the moment.
    "data": Optional. Reference to original NumPy array. Not used by this class,
            but derived classes and/or applications to retrieve original data.
    "datachanged": Optional. Flag set by any intermediate Filter to signalize
                   ImageView that a DataChange event was taken place due to
                   a change in the parameters of the Filter, and not the
                   data itself ("datachanged" equals to 0). If this flag is
                   present and set to zero, ImageView don't change zoom/scroll
                   position in a DataChange event, even if LockPosition is set
                   to 0.
                   
    Interface:
    ===========================
    View interface
    SetZoom
    ZoomTo
    ClearZoomSelect
    SetCursorType
    GetColormapFilter
    GetImagePositionValue
    GetDataPositionValue
    
    Overridables:    
    ===========================
    View overridables
    CreateMenu
    EventPosition
    
    Some of Drawable's methods are callable at application level (they're accessed
    through View.GetDrawable() (see GUIBinding.Drawable2D):    
    """
    def __init__(self, parent=None, pars={}, **kw):
        """
        See View.__init__
        Parameters:
        parent: parent window
        pars: Dictionary with View initialization options
              New options defined by this class (in addiction to base class options):
                "ScrollMode": "ON","OFF","AUTO"
                "ZoomMode": "ON","OFF","FIT_TO_SCREEN"
                "Oversize" (Just in TkBinding):
                      "AUTO_SET": manages automatically the oversize
                      region when zooming/scrolling            
                      "NONE": no oversize area
                      "FULL_IMAGE": always loads complete
                      float: defines directly the oversize area
                "StoreImageData": If set, view object stores "data" from it's selection
                                  (if exists) in self.ImageData. By default it is not
                                  stored, to save memory, but can be useful to access
                                  pixel values when image data has ben changed in
                                  relation to it's data object values, by any transformation.
        kw:   keywords to container initialization
        """
        View.__init__(self,parent,pars,**kw)        
                    
        if "Oversize" in pars.keys(): Oversize=pars["Oversize"]
        else: Oversize="NONE"

        if "StoreImageData" in pars.keys():
            self.StoreImageData=pars["StoreImageData"]
            self.ImageData=None
        else: self.StoreImageData=0        
        self.CreateMenu()

        if self.ZoomMode=="FIT_TO_SCREEN":
            self.ScrollMode="OFF"
            
        if self.ScrollMode!="OFF":
            if Oversize=="AUTO_SET":
                self.Drawable.AutoOversizeSet=1
            elif Oversize=="FULL_IMAGE":
                self.Drawable.AutoOversizeSet=0
                self.Drawable.FullDisplay=1
            elif Oversize=="NONE":
                self.Drawable.AutoOversizeSet=0
                self.Drawable.FullDisplay=0
                self.Drawable.Oversize=1.0
            else:
                self.Drawable.AutoOversizeSet=0
                self.Drawable.FullDisplay=0
                self.Drawable.Oversize=Oversize           

            self.Drawable.MoveTo((0,0))
        
        self.Drawable.Show()
        self.cursor=None
        self.SetCursorType("None")
        self.SetPointer("cross")


    def GetColormapFilter(self):
        """
        Auxiliar method. Search the filters recursively in order to find a ColormapFilter.
        This can be useful to derived classes to retrieve the ColormapFilter (in order to
        change its parameters) if some othere filters are cascaded after it.
        If no ColormapFilter is found, returns None.
        """
        try:
            sel=self.Source[0]
            while hasattr(sel,"GetSource"):
                if sel.__class__ is ColormapFilter: return sel
                for base in  sel.__class__.__bases__:
                    if base is ColormapFilter: return sel
                sel=sel.GetSource()                
        except: pass            
        return None


    def GetImage(self):
	"""
	Return an Image of the current source
	"""
	if self.Source==(): return None
	if hasattr(self.Drawable, "GetImage"):
		image= self.Drawable.GetImage()
		#TODO: change name to use legend as for 1D graph
		return {"image":image, "name":self.Source[0].name}
	else:	return None

    def GetColormapImage(self):
	"""
	Return an Image of the current Source Colormap
	"""
	if self.Source==(): return None
	if hasattr(self.Drawable, "GetColormapImage") and \
	   hasattr(self.Source[0], "GetColormapOutput"):
		data= self.Source[0].GetColormapOutput()
		if not data.has_key("image"): return None
		image= self.Drawable.GetColormapImage(data["image"], data["size"], data["depth"], data["colormap_table"])
		return {"image":image, "minmax": data["minmax"], "scale":data["scale"]}

    def GetDataPositionValue(self,data_pos):
        """
        Returns the value of the data for a given data coordinate, in the
        linked data object.
        If invalid coordinate or no linked data, returns None
        """
        if self.Source == (): return None
        data=self.Source[0].GetData()            
        if data==None: return None
        else:          return data.GetCoordValue(data_pos)            
        

    def GetImagePositionValue(self,image_pos):
        """
        Returns the value of the data for a given image coordinate.
        If invalid coordinate or no ImageData present, returns None
        """
        if hasattr(self,"ImageData")==0 or self.ImageData is None: return None
        if image_pos[1]>=self.ImageData.shape[0] or image_pos[0] >= self.ImageData.shape[1]: return None
        return self.ImageData[image_pos[1]][image_pos[0]]


    def CreateMenu(self):
        """
        Can be overwritten by derived classes to create a
        different popup menu
        """
        if self.ZoomMode=="ON":
            if self.MenuPopup is not None:self.AddMenuSeparator()
            self.AddMenuPopupItem("Zoom To",self.ZoomTo)            
            self.AddMenuPopupItem('Zoom +',Command.Command(self.SetZoom, com="in"))
            self.AddMenuPopupItem('Zoom -',Command.Command(self.SetZoom, com="out"))
            self.AddMenuPopupItem('Fit to Screen', Command.Command(self.SetZoom, com="fit"))
            self.AddMenuPopupItem('Reset Zoom',Command.Command(self.SetZoom, com="normal"))
        
#################################
#            ZOOM
#################################  

    def SetZoom(self, val=None, rect=None, com=None):
        """
        Sets zoom state to the ImageView object
        Parameters: (only one must be set)
          val:  a float, sets directly a value to the zoom factor
          rect: a tuple in the format (x0,y0,x1,y1), that defines the coordinates of
              the upper-left and the lower-right points of a zoomed area, in
              data selection coordinates.
          com:  can be "fit", "normal", "in" and "out"
        """
        self.Drawable.SetZoom (val=val,rect=rect,com=com)


    def ZoomTo(self):
        """
        Called from menu option: creates ImageViewSelectRect to define a
        zoom region. Promotes zoom on selection event.
        """
        if (self.Source != ()):
          data=self.Source[0].GetData()
          if data==None: return          
          self.ZoomSelect=ImageViewSelect.ImageViewSelectRect(data,self._EventZoomSelection)        
          self.ZoomSelect.ConnectView(self)

    def _EventZoomSelection(self,source):
        if (self.Source != ()) and (self.Source[0] != None):          
          data=self.Source[0].GetData()
          if data==None:
              self.ClearZoomSelect()
              return
          self.ZoomSelect.EraseSelection(self)
          (p0,p1)=source.GetSelection()["BoundingRect"]
          (x0,y0),(x1,y1)=self.Source[0].DataCoord2SelectionCoord(p0),self.Source[0].DataCoord2SelectionCoord(p1)
          rect=(min(x0,x1),min(y0,y1),max(x0,x1),max(y0,y1))
          self.SetZoom(rect=rect)
          self.ClearZoomSelect()

    def ClearZoomSelect(self):
        """
        Called to destroy zoom ViewSelect object.
        Called in the zoom selection event, but applications can assure
        the destruction by calling this.
        """
        if (self.ZoomSelect is not None):
          self.ZoomSelect.Destroy()
          self.ZoomSelect=None
        

#################################
#            CURSOR
#################################

    def SetCursorType(self,cursortype):
        """
        Sets the cursor
        Parameters: 
            cursortype: "None" (default) or "Crosshairs"
                        Both types generate EventPosition events
        """
        if cursortype=="None":
            cursor=ImageViewSelect.ImageViewSelectMousePosition(None,self.EventPosition)
        elif cursortype=="Crosshairs":
            cursor=ImageViewSelect.ImageViewSelectCrosshairs(None,self.EventPosition)
        else:
            raise "GraphView: Bad cursor type"                   
        self.SetCursor(cursor)


#################################
#            EVENTS
#################################

    def EventPosition(self,source):
        """
        Overridable: Cursor callback event
        Parameters:
            source: ImageViewSelect object thar generated the event
        """
        pass


#################################
#            VIRTUALS
#################################

    def CreateDrawable(self):
        """
        Virtual: See View.CreateDrawable
        """        
        self.Drawable = Drawable_2D(self,self.ZoomMode,self.ScrollMode)

    def GetPosition(self, event):
        """
        Virtual: See View.GetPosition
        """
        ret=Position()
        if event==(-1,-1)or (self.Source == ()):  return None
        ret.ImageCoord=event
        ret.DataSelectionCoord=event
        ret.ViewCoord=self.Drawable.ImageCoord2ViewCoord(event)
        ret.DataCoord=self.Source[0].SelectionCoord2DataCoord(ret.DataSelectionCoord)        
        return ret


    def DataChanged(self,source=None):
        """
        Virtual: See View.DataChanged
        """
        if self.Source != ():
            sel=self.Source[0].GetOutput()
            if "image" in sel.keys() and "size" in sel.keys() and sel["size"][0] and sel["size"][1]:


 
              if "update_range" in sel.keys():
                     update_range=sel["update_range"]
              else:update_range=None



              if "datachanged" in sel.keys() and sel["datachanged"]==0:
                  self.Drawable.DrawImage(sel["image"],sel["size"],sel["depth"],sel["colormap_table"],0)
                                          # ,update_range=update_range )
              else:
                  self.Drawable.DrawImage(sel["image"],sel["size"],sel["depth"],sel["colormap_table"],not self.FlagLockPosition )
                                          # ,update_range=update_range)
              if self.StoreImageData:
                  if "data" in sel.keys(): self.ImageData=sel["data"]
            else:            
                self.Drawable.EraseImage()
                if self.StoreImageData: self.ImageData=None
        else:
            self.Drawable.EraseImage()
            if self.StoreImageData: self.ImageData=None

        self.Update()
        ### This event should be generated inside ViewSelect's Update???
        if self.Cursor is not None and "Position" in self.Cursor.Selection.keys(): self.EventPosition(self.Cursor)


    def Redraw(self):
        """
        Virtual: See View.Redraw
        """
        pass


    def DataCoord2ImageCoord(self,data_coord,trim=1 ):
        """
        Virtual: See View.DataCoord2ImageCoord
        """
        if self.Source == (): return (-1,-1)        
        selection_coord=self.Source[0].DataCoord2SelectionCoord(data_coord, trim)
        return selection_coord


    def ViewCoord2DataCoord(self,view_coord):
        """
        Virtual: See View.ViewCoord2DataCoord
        """
        if self.Source  == (): return DataPosition(-1,(-1,-1))
        data_coord=self.Source[0].SelectionCoord2DataCoord(view_coord)        
        return data_coord


    def GetSaveFormats(self):
        """
        Returns tuple with supported save formats
        """
        return ("PNG","BMP","PS","JPG")


        
