"""	
	ExtendedGraphView.py
	Derived from GraphView, 1D plotting class.
	Add some functionnalities to GraphView :
		-	a status bar for visualizing on cursor's informations (position and under cursor pixel's value)
	 	-	some graph draw methods : graph draw design modification (change plot symbol,line style, color, width...) 
		-	some rois management methods : add, del, design customization (change roi name ; border style, color, width ; interior filling stripes, color). Also roi limits moving by click and drag
		-	some peaks management methods : add, del, design customization (change peak name ; peak line style, color, width). Also peaks moving by click and drag
"""

__author__ =  'Nicolas PASCAL (pascal@esrf.fr)'
__version__=  '1.0'


if __name__ == '__main__':
	try:
		import SetPath
	except:
		pass
	import sys
	if len (sys.argv)>1 and sys.argv[1]=="Tk":
		import Tkinter
	else:
		import qt

from GraphView import *
import GraphViewSelect
import types

DEBUG=0
class ExtendedGraphView(GraphView):

	def __init__(self, parent=None, pars={}, **kw):
		"""
		Constructor
		Evoluted 1D plotting class
		parametres : 
			-	parent : parent widget
			-	pars : parametres' dictionnary. 
						'StatusBar' : if 0, no status bar added. if 1 or not present in keys, displayed
						'SelectionSensibility' : it represents the division of graph's screen around around a peak or a roi limit which permit to select it :
														The width of the sensible to selection area is calculated as (xmin-xmax)/SelectionSensibility. So the bigger is SelectionSensibility, 
														the smaller is the selecting area. Default is 20
						'RoiSelection' : enable the roi selection mode. Default is on
						'PeakSelection' : enable the peak selection mode. Default is on
						'Legend' : if 0, no graph legend added. if 1 or not present in keys, displayed ##### not done yet for Qt
				 For details on other keys, see GraphView documentation
		"""
		self.opts={}
		if "StatusBar" in pars.keys(): self.opts['StatusBar']=pars["StatusBar"]
		else: self.opts['StatusBar']=1
		if "SelectionSensibility" in pars.keys(): self.opts['SelectionSensibility']=pars["SelectionSensibility"]
		else: self.opts['SelectionSensibility']=20
		if "RoiSelection" in pars.keys(): self.opts['RoiSelection']=pars["RoiSelection"]
		else: self.opts['RoiSelection']=1
		if "PeakSelection" in pars.keys(): self.opts['PeakSelection']=pars["PeakSelection"]
		else: self.opts['PeakSelection']=1
		if "Legend" in pars.keys(): self.opts['Legend']=pars["Legend"]
		else: self.opts['Legend']=1
		
		self.Select=None
		self.ZoomSelect=None
		
		GraphView.__init__(self, parent, pars, **kw)
		self.eh.register("graphElementStyleChangedEvent",self._ChangeGraphElementStyle)
		if self.opts['RoiSelection']: self.eh.register("roiStyleChangedEvent",self._ChangeRoiStyle)
		if self.opts['PeakSelection']: self.eh.register("peakStyleChangedEvent",self._ChangePeakStyle)
		
		if self.opts['StatusBar']:
			# add status bar
			self.statusLabel=Label(self)
			self.statusLabel.Show()
		
		# functionning mode
		self.activeMode=None
		
		# mode menu items
		self.CreateExtendedMenu()

		# displayed popups relative to graphical objects
		self.popups={}
		
		# graphes
		self.activeGraph=None # active graph object
		self.activeGraphHighlightPen=None # pen used to highlight the active graph
		self.activeGraphPen=None # stores the highlighted graph pen to restore it once it will be deactivated
		
		self.SetActiveGraphPen(color=(255,0,0),width=2,style='dotted') # init active graph pen style
		
		#self.activeGraphColor=None
		
		self.popups['Graph']={}
		
		if self.opts['RoiSelection']:
			# roi attributes
			self.rois={}
			self.activeRoi=None
			self.activeRoiHighlight=[]

			self.popups['Roi']={}
		
			self.roiLimit_dnd=0 # roi limit drag 'n' drop management mode
		
		if self.opts['PeakSelection']: 
			# peaks attributes
			self.peaks={}
			self.activePeak=None

			self.popups['Peak']={}
		
			self.peaks_dnd=0 # peaks drag 'n' drop management mode
				
	################################
	###           MENU           ###
	################################
	
	# --- Menu creation ---
	
	def CreateExtendedMenu(self):
		"""
		create mode menu
		"""
		# menu specific to ExtendedGraphView 
		self.ZoomMenuItem=self.AddMenuPopupItem('Zoom',self._SetZoom,'checkbutton')
		self.ResetZoomMenuItem=self.AddMenuPopupItem("Reset Zoom",self.Drawable.ResetZoom)
		self.AddMenuSeparator()
		if self.opts['RoiSelection']: self.RoiMenuItem=self.AddMenuPopupItem('Roi',self._SetRoi,'checkbutton')
		if self.opts['PeakSelection']: self.PeakMenuItem=self.AddMenuPopupItem('Peak',self._SetPeak,'checkbutton')
		self.AddMenuPopupItem("Clear selection",self._EraseSelect)
		self.AddMenuSeparator()
		self.CrosshairsMenuItem=self.AddMenuPopupItem("Crosshairs",self._SetCrosshairs,'checkbutton')
		self.GridlinesMenuItem=self.AddMenuPopupItem("Gridlines",self._SetGridlines,'checkbutton')
		self.AddMenuSeparator()
		self.AddMenuPopupItem("Toggle Log X",self._ToggleLogX)
		self.AddMenuPopupItem("Toggle Log Y",self._ToggleLogY)

		
	def CreateMenu(self):
		"""
		Create the standard GraphView menu
		"""
		# standard GraphView Menu : just zoom box changed with checkbutton option, in ExtendedGraphView whereas present in Qt binding
		pass

	def CreateDrawable(self):
		"""
		View window creation, override it for customization 
		"""
		GraphView.CreateDrawable(self)

	# --- Menu items callbacks ---
	
	def _SetZoom(self):
		newstate=not self.MenuPopup.IsItemChecked(self.ZoomMenuItem)
		self.SetModeStateZoom(newstate)
		self.MenuPopup.CheckItem(self.ZoomMenuItem,newstate)
		
	def _SetRoi(self):
		newstate=not self.MenuPopup.IsItemChecked(self.RoiMenuItem)
		self.SetModeStateRoi(newstate)
		self.MenuPopup.CheckItem(self.RoiMenuItem,newstate)
		
	def _SetPeak(self):
		newstate=not self.MenuPopup.IsItemChecked(self.PeakMenuItem)
		self.SetModeStatePeak(newstate)
		self.MenuPopup.CheckItem(self.PeakMenuItem,newstate)
		
	def _EraseSelect(self):
		if self.Select is not None:
			self.Select.Erase(self)
	
	def _DestroySelect(self):
		if self.Select is not None:
			self.Select.Erase(self)
			self.Select.Destroy()
			self.Select=None
	
	def _SetCrosshairs(self):
		if self.MenuPopup.IsItemChecked(self.CrosshairsMenuItem):
			self.SetCursorType(cursortype="None")
			self.MenuPopup.CheckItem(self.CrosshairsMenuItem,0)
		else: 
			self.SetCursorType(cursortype="Crosshairs")
			self.MenuPopup.CheckItem(self.CrosshairsMenuItem,1)
		
	def _SetGridlines(self):
		newstate=not self.MenuPopup.IsItemChecked(self.GridlinesMenuItem)
		self.Drawable.ToggleGridlines()
		self.MenuPopup.CheckItem(self.GridlinesMenuItem,newstate)

	############################################
	###   SELECTION ON THE GRAPH CALLBACKS   ###
	############################################
	
	def EventZoomSelection(self,source):
		"""
		Callback of a zoom box selection
		"""
		rect=source.GetSelection()["BoundingRect"]
		self.ZoomSelect.RemoveSelection()
		self.Drawable.SetZoom(rect[0],rect[1])
	
	def EventRoiSelection(self,source):
		"""
		Virtual : Callback of a new roi selection : to be overidden for specific behavior
		"""
		if DEBUG : print "EventRoiSelection"
		pass
	
	def EventPeakSelection(self,source):
		"""
		Virtual : Callback of a new peak selection : to be overidden for specific behavior
		"""
		if DEBUG : print "EventPeakSelection"
		pass

	##########################################
	###  GRAPH ITEMS SELECTION CALLBACKS   ###
	##########################################
	
	def ClickOnGraphCB(self,graphName=None):
		"""
		Callback of a click near a graph event : to be overidden for specific behavior
		"""
		if DEBUG : print " ClickOnGraphCB ",graphName
		# if more than one graph, make it being the active one
		self.SetActiveGraph(graphName)
		
	def ClickOnRoiCB(self,roiName=None):
		"""
		Callback of a click in a roi event : to be overidden for specific behavior
		"""
		if DEBUG : print " ClickOnRoiCB ",roiName
		# if more than one roi, make it being the active one
		self.SetActiveRoi(roiName)
		
	def ClickOnPeakCB(self,peakName=None):
		"""
		Callback of a click near a peak event : to be overidden for specific behavior
		"""
		if DEBUG : print " ClickOnPeakCB ",peakName
		# if more than one peak, make it being the active one
		self.SetActivePeak(peakName)

	##################################
	###   MOUSE EVENTS CALLBACKS   ###
	##################################
	
	def EventPosition(self,source):
		"""
		When cursor is moved on the image, staus bar values are updated
		"""
		if not self.roiLimit_dnd and not self.peaks_dnd:
			if self.nearRoisLimit(source.Selection["Position"].DataCoord)!=[] or self.nearPeaks(source.Selection["Position"].DataCoord)!=[]:
				self.SetPointer("h_double_arrow")
			else:
				self.SetPointer("cross")
		
		if self.opts['StatusBar']:
			Coord=source.Selection["Position"].DataCoord
			# format values:
			statusString="x:%07.3f   y:%07.3f" % (Coord[0],Coord[1])
			self.statusLabel.SetText(statusString)
		 
	def EventButtonPress(self,pos):
		# set active graph near position
		graphName=self.GetClosestFunction(pos.DataCoord)
		if graphName is not None:
			self.ClickOnGraphCB(graphName)

		if self.opts['RoiSelection']: 
			# set active roi near position
			enclosingRois=self.InsideRoi(pos.DataCoord)
			if enclosingRois!=[]:
				self.ClickOnRoiCB(enclosingRois[0].name)

			# activate or not roi limit drag 'n' drop
			overlappedRoiLimit=self.nearRoisLimit(pos.DataCoord)
			if overlappedRoiLimit!=[]:
				self.roiLimit_dnd=1
				self.draggedRoiLimit=overlappedRoiLimit

		if self.opts['PeakSelection']: 
			# peaks
			overlappedPeaks=self.nearPeaks(pos.DataCoord)
			if overlappedPeaks!=[]:
				# set active peak
				self.ClickOnPeakCB(overlappedPeaks[0].name)	

				# activate peaks drag 'n' drop
				self.peaks_dnd=1
				self.draggedPeaks=overlappedPeaks
	
	def EventButtonPressMotion(self,pos):
		if self.opts['RoiSelection']: 
			# roi limits drag 'n' drop
			if self.roiLimit_dnd:
				i=0
				while i<len(self.draggedRoiLimit):
					roi,limit=self.draggedRoiLimit[i]
					roi.MoveLimit(limit,pos.DataCoord[0])
					if roi==self.GetActiveRoi():
						# update active roi highlight box
						for obj in self.activeRoiHighlight:
							xmin,ymin,xmax,ymax=self.Drawable.GetObjectCoords(obj)
							ymin,ymax=self.GetYAxis()
							if xmin==limit:
								xmin=pos.DataCoord[0]
								self.Drawable.SetObjectCoords(obj,xmin,ymin,xmax,ymax)
							elif xmax==limit:
								xmax=pos.DataCoord[0]
								self.Drawable.SetObjectCoords(obj,xmin,ymin,xmax,ymax)
					self.draggedRoiLimit[i]=(roi,pos.DataCoord[0])
					i+=1
		
		if self.opts['PeakSelection']: 
			# peaks drag n drop
			if self.peaks_dnd:
				for pk in self.draggedPeaks:
					pk.Move(pos.DataCoord[0])

	def EventButtonRelease(self,pos):
		if self.opts['RoiSelection']: 
			# disable roi limits drag 'n' drop if set
			if self.roiLimit_dnd:
				self.roiLimit_dnd=0
		
		if self.opts['PeakSelection']: 
			# disable peaks drag 'n' drop if set
			if self.peaks_dnd:
				self.peaks_dnd=0

	def EventMotion(self,pos):
		pass

	def EventDoubleClick(self,pos):
		"""
		display a popup with items properties 
		"""
		# GRAPH
		graphName=self.GetClosestFunction(pos.DataCoord)
		if graphName is not None:
			# double click occurs near a graph : display graph properties dialog
			graph=self.Functions[graphName]
				
			if graph is not None :
				if graph==self.activeGraph and self.activeGraphHighlightPen is not None :
					graphCnf={'symbol':graph['symbol'],'style':self.activeGraphPen.style,'width':self.activeGraphPen.GetWidth(),'color':self.activeGraphPen.color}
				else:
					graphCnf={'symbol':graph['symbol'],'style':graph['pen'].style,'width':graph['pen'].GetWidth(),'color':graph['pen'].color}
				
				if (not self.popups['Graph'].has_key(graphName)) or self.popups['Graph'][graphName].IsDestroyed():
					# if doesn't displayed yet : display the GraphStyleDialog for this graph
					self.popups['Graph'][graphName]=GraphStyleDialog(self,self.eh,graph['name'],graphCnf)
					self.popups['Graph'][graphName].Show()
		
		if self.opts['RoiSelection']: 
			# ROIS
			overlappedRois=self.InsideRoi(pos.DataCoord)
			if overlappedRois!=[]:
				# double click occurs inside a roi : display roi properties dialog
				for roi in overlappedRois:
					roiFillCnf={'fill':roi.brush.style,'color':roi.brush.color}
					roiBorderCnf={'style':roi.pen.style,'width':roi.pen.width,'color':roi.pen.color}
					if (not self.popups['Roi'].has_key(roi.name))or self.popups['Roi'][roi.name].IsDestroyed():
						# if doesn't displayed yet : display the RoiStyleDialog for this roi
						self.popups['Roi'][roi.name]=RoiStyleDialog(self,self.eh,roi.name,roiFillCnf,roiBorderCnf)
						self.popups['Roi'][roi.name].Show()
		
		if self.opts['PeakSelection']: 
			# PEAKS
			overlappedPeaks=self.nearPeaks(pos.DataCoord)
			if overlappedPeaks!=[]:
				# double click occurs inside a peak : display peak properties dialog
				for peak in overlappedPeaks:
					peakCnf={'style':peak.pen.style,'width':peak.pen.width,'color':peak.pen.color}
					if (not self.popups['Peak'].has_key(peak.name))or self.popups['Peak'][peak.name].IsDestroyed():
						
						# if doesn't displayed yet : display the PeakStyleDialog for this peak
						self.popups['Peak'][peak.name]=PeakStyleDialog(self,self.eh,peak.name,peakCnf)
						self.popups['Peak'][peak.name].Show()

	# --- keyboard events callback ---
	
	def EventKeyPress(self,key,flags):
		pass
	
	############################################
	###  CHANGE GRAPH ITEMS STYLE CALLBACKS  ###
	############################################
	
	def _ChangeGraphElementStyle(self,graphName,graphCnf):
		"""
		callback of a GraphStyleDialog event
		"""
		if self.Functions.has_key(graphName):
			graph=self.Functions[graphName]

			# change graph style
			if graph is not None :
				if len(self.Functions)<=1 or graph!=self.activeGraph or self.activeGraphHighlightPen is None:
					# if one graph or the graph to change is not the active one : change its color directly on the graph
					graph['pen'].style=graphCnf['style']
					graph['pen'].width=graphCnf['width']
					graph['pen'].color=graphCnf['color']
					
					self.SetPen(graph['name'],graph['pen'])
					self.SetSymbol(graph['name'],graphCnf['symbol'])
				else:
					# if the graph to change is the active one : store its color to change it when it won't be highlighted in red
					self.activeGraphPen.style=graphCnf['style']
					self.activeGraphPen.width=graphCnf['width']
					self.activeGraphPen.color=graphCnf['color']
				

	def _ChangeRoiStyle(self,roiName,(newRoiName,roiFillCnf,roiBorderCnf)):
		"""
		callback of the event generated by the RoiStyleDialog
		"""
		if self.rois.has_key(roiName):
			roi=self.rois[roiName]
			
			# change name
			if roiName!=newRoiName:
				if self.rois.has_key(newRoiName):
					print "Can't change roi name for this one : already exists an roi with it"
				else:
					# update self's datas
					self.rois[newRoiName]=roi
					del self.rois[roiName]
					# update popups dictionnary key
					self.popups['Roi'][newRoiName]=self.popups['Roi'][roiName]
					del self.popups['Roi'][roiName]
					roiName=newRoiName
					# update roi object
					roi.SetName(newRoiName)
			
			# change pen
			roi.SetPenValues(roiBorderCnf['style'],roiBorderCnf['width'],roiBorderCnf['color'])
			
			# change brush
			roi.SetBrushValues(roiFillCnf['fill'],roiFillCnf['color'])
			
	def _ChangePeakStyle(self,peakName,(newPeakName,peakCnf)):
		"""
		callback of a PeakStyleDialog event
		"""
		if self.peaks.has_key(peakName):
			peak=self.peaks[peakName]
			
			# change name
			if peakName!=newPeakName:
				if self.peaks.has_key(newPeakName):
					print "Can't change peak name for this one : already exists an peak with it"
				else:
					# update self's datas
					self.peaks[newPeakName]=peak
					del self.peaks[peakName]
					# update popups dictionnary key
					self.popups['Peak'][newPeakName]=self.popups['Peak'][peakName]
					del self.popups['Peak'][peakName]
					peakName=newPeakName
					# update peak object
					peak.SetName(newPeakName)
			
			# change pen
			peak.SetPenValues(peakCnf['style'],peakCnf['width'],peakCnf['color'])

	def _RedrawGraphItems(self):
		"""
		update the graph items (roi, peaks) size, when the frame is resized for exemple
		"""
		if self.opts['RoiSelection']:
			# update active roi highlight rectangles size
			ymin,ymax=self.GetYAxis()
			for rect in self.activeRoiHighlight:
				for xmin,xmax in self.activeRoi.limits:
					# create the highlight rectangle
					self.Drawable.SetObjectCoords(rect,xmin,ymin,xmax,ymax)
			
			for roi in self.rois.keys():
				self.rois[roi].Redraw()
				
		if self.opts['PeakSelection']: 
			for peak in self.peaks.keys():
				self.peaks[peak].Redraw()
	
	################################
	###     MODE MANAGEMENT      ###
	################################

	# ---  Modes functions  ---
	# use this model when creating your own modes
	
	# Zoom mode
	def SetModeStateZoom(self,state):
		if state==0:
			self.DisableModeZoom()
		else:
			self.EnableModeZoom()
	
	def EnableModeZoom(self):
		# disable other selections
		if self.Select is not None:
			if isinstance(self.Select,GraphViewSelect.GraphViewSelectHRect):
				self.DisableModeRoi()
			elif isinstance(self.Select,GraphViewSelect.GraphViewSelectVLine):
				self.DisableModePeak()
		
		if self.ZoomSelect is None:
			self.MenuPopup.CheckItem(self.ZoomMenuItem,1)
			self.activeMode='Zoom'
		
			if hasattr(self,"ZoomBrush")==0: self.ZoomBrush=Brush((240,240,240),"fill_100")
			if hasattr(self,"ZoomPen")==0:   self.ZoomPen=Pen((0,0,0),1,"solid")      

			self.ZoomSelect=GraphViewSelect.GraphViewSelectRect(self.Source[0],self.EventZoomSelection)
			self.ZoomSelect.SetBrush(self.ZoomBrush)
			self.ZoomSelect.SetPen(self.ZoomPen)
			self.ZoomSelect.ConnectView(self)
		
	def DisableModeZoom(self):
		self.activeMode=None
		self.MenuPopup.CheckItem(self.ZoomMenuItem,0)
		if self.ZoomSelect is not None:
			self.ZoomSelect.Destroy()
			self.ZoomSelect=None
	
	# Roi mode
	def SetModeStateRoi(self,state):
		if state==0:
			self.DisableModeRoi()
		else:
			self.EnableModeRoi()
	
	def EnableModeRoi(self):
		# disable other selections
		if self.ZoomSelect is not None:
			self.DisableModeZoom()
		if self.Select is not None:
			if isinstance(self.Select,GraphViewSelect.GraphViewSelectVLine):
				self.DisableModePeak()

		if self.Select is None:
			self.MenuPopup.CheckItem(self.RoiMenuItem,1)
		
			self.activeMode='Roi'
			
			if (self.Source != ()):
				self._DestroySelect()
				data=self.Source[0].GetData()
				if data==None: return
				self.Select=GraphViewSelect.GraphViewSelectHRect(data,self.EventRoiSelection)
				self.Select.ConnectView(self)
		
	def DisableModeRoi(self):
		if self.Select is not None:
			#self.MenuPopup.UnselectItem(3)
			self.MenuPopup.CheckItem(self.RoiMenuItem,0)
			
			self.activeMode=None
			self._DestroySelect()
		
	# peak mode
	def SetModeStatePeak(self,state):
		if state==0:
			self.DisableModePeak()
		else:
			self.EnableModePeak()
	
	def EnableModePeak(self):
		# disable other selections
		if self.ZoomSelect is not None:
			self.DisableModeZoom()
		if self.Select is not None:
			if isinstance(self.Select,GraphViewSelect.GraphViewSelectHRect):
				self.DisableModeRoi()
		
		if self.Select is None:
			self.MenuPopup.CheckItem(self.PeakMenuItem,1)
			self.activeMode='Peak'

			# create peak selection
			if (self.Source != ()):
				self._DestroySelect()
				data=self.Source[0].GetData()
				if data==None: return
				self.Select=GraphViewSelect.GraphViewSelectVLine(data,self.EventPeakSelection)
				self.Select.ConnectView(self)
				# active mode is now peak selection
				self.activeMode='Peak'
		
	def DisableModePeak(self):
		if self.Select is not None:
			peakCheckIndex=3
			if self.opts['RoiSelection']: peakCheckIndex+=1
			#self.MenuPopup.UnselectItem(peakCheckIndex)
			self.MenuPopup.CheckItem(peakCheckIndex,0) ###
			
			self.activeMode=None
			self._DestroySelect()
	
	# --- generic mode management method ---
	
	def SetActiveMode(self,mode=''):
		"""
		disable old active mode, and set the mode named 'mode' asa active
		"""
		self.DisableMode(self.activeMode)
		self.EnableMode(mode)
		
	def DisableMode(self,mode=''):
		"""
		Calls a function to disable the specified mode with a name "DisableMode"+'mode'
		"""
		if self.modes.has_key(mode): exec("self.DisableMode"+mode+"()")
		
	def EnableMode(self,mode=''):
		"""
		Calls a function to enable the specified mode with a name "EnableMode"+'mode'
		"""
		if self.modes.has_key(mode): exec("self.EnableMode"+mode+"()")	
	
	################################
	###     GRAPH MANAGEMENT     ###
	################################
	def DataChanged(self,source=None):
		# update graphes on graph : duplicate reference, but easier to use
		GraphView.DataChanged(self,source)
		for graph in self.Source:
			if source is graph: self.SetActiveGraph(graph.name)

	
	def DelGraph(self,name=None):
		"""
		delete the graph called 'name'
		"""
		if name is not None:
			i=0
			while i<len(self.Source) and name!=self.Source[i].name:
				i+=1
			if i<len(self.Source):
				# delete in Selection list
				self.Source[i].Destroy()
				s=list(self.GetSource())
				del s[i]
				self.SetSource(s)

			# redraw roi and peak items if graph resize
			self._RedrawGraphItems()
				
	def SetActiveGraphPen(self,color=None,width=None,style=None):
		"""
		changed the pen style used to highlight the active graph
		Changes only the non None parameters
		To disable active graph highlight, call this method without any parameter
		"""
		newColor=color
		newWidth=width
		newStyle=style
		if newColor is None and newWidth is None and newStyle is None:
			# disable active graph highlight
			self.activeGraphHighlightPen=None
			
			# unhighlight active graph if it is the case
			if self.GetActiveGraph() is not None :
				self.GetActiveGraph()['pen']=self.activeGraphPen
				self.SetPen(self.GetActiveGraph()['name'],self.activeGraphPen)
			
		else:
			if self.activeGraphHighlightPen is not None:
				# for non given or None parameters : recuperate activeGraphHighlightPen values
				if newColor is None: newColor=self.activeGraphHighlightPen.color
				if newWidth is None: newWidth=self.activeGraphHighlightPen.width
				if newStyle is None: newStyle=self.activeGraphHighlightPen.style
			else:
				# give default values for non given parameters when non preexistant activeGraphHighlightPen
				if newColor is None: newColor=(255,0,0)
				if newWidth is None: newWidth=3
				if newStyle is None: newStyle='normal'
			self.activeGraphHighlightPen=Pen(newColor,newWidth,newStyle)
			# update active graph pen with new activeGraphHighlightPen
			if self.activeGraph is not None:
				self.activeGraphPen=self.activeGraph['pen']
				self.SetPen(self.activeGraph['name'],self.activeGraphHighlightPen)
	
	def GetActiveGraph(self):
		"""
		return the active graph
		"""
		return self.activeGraph
	
	def SetActiveGraph(self,name):
		"""
		set the graph 'name' as the active one
		"""		
		if name is not None:
			if self.GetActiveGraph()!=self.Functions[name]:
				oldActiveGraph=self.activeGraph
				self.activeGraph=self.Functions[name]
				
				if self.activeGraphHighlightPen is not None:
					# a pen is set to visualize the active graph
					oldActiveGraphPen=self.activeGraphPen
					self.activeGraphPen=self.Functions[name]['pen']

					# more than one graph : display the active one with activeGraphHighlightPen 
					if len(self.Functions)>1:
						# redraw old active graph with its old color
						if oldActiveGraph is not None and oldActiveGraphPen is not None:
							oldActiveGraph['pen']=oldActiveGraphPen
							self.SetPen(oldActiveGraph['name'],oldActiveGraph['pen'])

						# redraw new active graph in red
						#self.activeGraphColor=self.activeGraph['pen'].color
						self.activeGraph['pen']=self.activeGraphHighlightPen
						self.SetPen(self.activeGraph['name'],self.activeGraph['pen'])

	################################
	###      ROI MANAGEMENT      ###
	################################
	
	def InsideRoi(self,pos):
		"""
		return the sequence of rois containing the graph position 'pos'. If None, return an empty sequence
		"""
		ret=[]
		for roi in self.rois.keys():
			# in each roi 
			for min,max in self.rois[roi].limits:
				# in each roi limit
					if pos[0]>=min and pos[0]<=max:
						ret.append(self.rois[roi])
		return ret

	def nearRoisLimit(self,pos):
		"""
		return the sequence of roi near the graph position 'pos'. If None, return an empty sequence
		"""
		sensibleLimits=(self.Drawable.xmax-self.Drawable.xmin)/self.opts['SelectionSensibility']
		ret=[]
		for roi in self.rois.keys():
			# for each roi
			limits=self.rois[roi].limits
			i,lim=0,None
			xmin,xmax=limits[i]
			found=0
			while i<len(limits) and not found:
				xmin,xmax=limits[i]
				if (pos[0]>=(xmin-sensibleLimits/2) and pos[0]<=(xmin+sensibleLimits/2)):
					lim=xmin
					found=1
				elif (pos[0]>=(xmax-sensibleLimits/2) and pos[0]<=(xmax+sensibleLimits/2)):
					lim=xmax
					found=1
				else:
					i+=1
					
			if found:
				ret.append((self.rois[roi],lim))
		return ret
	
	def PutRoi(self,name='roi',limits=[],pen=Pen((0,0,0),2,"solid"),brush=Brush((0,0,0),"fill_25")):
		"""
		Put a roi on the graph with the specified parametres :
			-	name : the name given to this roi. If an already existant roi have the same name, it will be renamed with an non existing name using the style existing-name'_x'
				 		   where x is a integer.
			-	limits : the graph limits defining this roi : it's a sequence of min,max tuples
			-	pen : the pen style used for drawing the roi border
			-	brush : the brush style used for filling the roi
		Once the roi has been added, it is set as the active one		   
		"""	
		if self.rois.has_key(name):
			# if a roi with a name of label exists : renamed the one to add
			i=0
			while self.rois.has_key(name+'_'+str(i)):
				i+=1
			name+='_'+str(i)
		
		# hide all other roi names
		for roi in self.rois.keys():
			self.rois[roi].HideName()
		
		# add the new roi parametres to the list
		self.rois[name]=GraphViewRoi(self,name,limits,pen,brush)
		
		# make the new roi become the active one
		self.SetActiveRoi(name)
			
	def DelRoi(self,name=''):
		"""
		Delete the roi with the given name
		"""
		if self.rois.has_key(name):
			# erases roi rectangles
			for obj in self.rois[name].drawnRect:
				self.Drawable.EraseObject(obj)
			# erases roi labels
			for obj in self.rois[name].drawnLab:
				self.Drawable.EraseObject(obj)
			# delete it in the rois list
			del self.rois[name]
			
			if self.GetActiveRoi().name == name:
				# if roi to delete is the active one : erases active roi border rectangles
				for obj in self.activeRoiHighlight:
					self.Drawable.EraseObject(obj)
					self.activeRoiHighlight.remove(obj)
			
	def ShowRoisNames(self):
		"""
		show all rois labels
		"""
		for roi in self.rois.keys():
			self.rois[roi].ShowName()
			
	def ShowRoiName(self,name=''):
		"""
		show the label of the roi called 'name' 
		"""
		if self.rois.has_key(name):
			self.rois[name].ShowName()
		
	def HideRoisNames(self):
		"""
		hide all rois labels
		"""
		for roi in self.rois.keys():
			self.rois[roi].HideName()
			
	def HideRoiName(self,name=''):
		"""
		hide the label of the roi called 'name' 
		"""
		if self.rois.has_key(name):
			self.rois[name].HideName()
		
	def SetActiveRoi(self,name=''):
		"""
		if existant, set the roi with a name of 'name' as the active one. Draws a red rectangle around it, and remove it on the old active one
		"""		
		if self.rois.has_key(name):
			self.activeRoi=self.rois[name]

			# if more than one roi : draw a red rectangle around the roi to view which one is the active one
			if len(self.rois)>1: 
				# erases old active roi border rectangles
				for obj in self.activeRoiHighlight:
					self.Drawable.EraseObject(obj)
					self.activeRoiHighlight.remove(obj)
				# create it on new active one
				ymin,ymax=self.GetYAxis()
				for xmin,xmax in self.activeRoi.limits:
					# create the highlight rectangle
					self.activeRoiHighlight.append(self.Drawable.PutRectangle(xmin,ymin,xmax,ymax,pen=Pen((255,0,0),2,"solid")))

				# erases others roi lablels
				self.HideRoisNames()
			# show the label of the active roi
			self.ShowRoiName(name)
			
	def GetActiveRoi(self):
		"""
		return the active roi
		"""
		return self.activeRoi
		
	def GetRois(self):
		"""
		return a sequence containing all the displayed roi
		"""
		ret=[]
		for roi in self.rois.keys():
			ret.append(self.rois[roi])
		return ret
		
	############################
	###   PEAKS MANAGEMENT   ###
	############################
	
	def nearPeaks(self,pos):
		"""
		return the sequence of peaks near the graph position 'pos'. If None, return an empty sequence
		"""
		sensibleLimits=(self.Drawable.xmax-self.Drawable.xmin)/self.opts['SelectionSensibility']
		ret=[]
		for peak in self.peaks.keys():
			# for each peak position
				if pos[0]>=(self.peaks[peak].pos-sensibleLimits/2) and pos[0]<=(self.peaks[peak].pos+sensibleLimits/2):
					ret.append(self.peaks[peak])
		return ret
	
	def PutPeak(self,name='pk',pos=0,pen=Pen((0,0,0),2,"solid")):
		"""
		Put a peak on the graph :
			-	name : name of the peak. If an existing peaqk has already this name, using the style existing-name'_x'
				 		   where x is a integer.
			-	pos : the x position where to put the peak
			-	pen : the pen style used for drawing the peak
		"""
		if self.peaks.has_key(name):
			# if a peak with a name of label exists : renamed the one to add
			i=0
			while self.peaks.has_key(name+'_'+str(i)):
				i+=1
			name+='_'+str(i)
		
		# hide all other peak names
		for peak in self.peaks.keys():
			self.peaks[peak].HideName()
		
		# add the new peak parametres to the list
		self.peaks[name]=GraphViewPeak(self,name,pos,pen)
		
		# make the new peak become the active one
		self.SetActivePeak(name)
		
	def DelPeak(self,name=''):
		"""
		delete the peak which has the given name. Delete nothing if no peak has this name
		"""
		if self.peaks.has_key(name):
			# erases peak line
			for obj in self.peaks[name].drawnVLines:
				self.Drawable.EraseObject(obj)
			# erases peak labels
			for obj in self.peaks[name].drawnLab:
				self.Drawable.EraseObject(obj)
			# delete it in the peaks list
			if self.peaks[name] == self.activePeak: self.activePeak=None
			del self.peaks[name]
		
	def SetActivePeak(self,name):
		"""
		make the peak called 'name' become the active one 
		"""
		self.activePeak=self.peaks[name]
		
		if len(self.peaks)>1:
			# erases other peaks names
			self.HidePeaksNames()
		
		# shows the label of this peak
		self.ShowPeakName(name)
			
	def GetActivePeak(self):
		"""
		return the active peak (last which has been added for the moment)
		"""
		return self.activePeak
	
	def ShowPeaksNames(self):
		"""
		show all peaks labels
		"""
		for pk in self.peaks.keys():
			self.peaks[pk].ShowName()
			
	def ShowPeakName(self,name):
		"""
		show all peaks labels
		"""
		if name in self.peaks.keys():
			self.peaks[name].ShowName()
		
	def HidePeaksNames(self):
		"""
		hide all peaks labels
		"""
		for pk in self.peaks.keys():
			self.peaks[pk].HideName()
	
class GraphViewPeak:
	"""
	Class representing a graph's peak
	"""
	def __init__(self,view,name='roi',pos=0,pen=Pen((0,0,0),2,"solid")):
		"""
		Constructor
			-	view : view where the roi must be drawn
			-	pos : the position where to put the peak
			-	pen : the pen style used for drawing the peak
		"""
		self.view=view
		
		self.name=name
		self.pos=pos
		self.pen=pen

		# drawn on view objects references
		# vertical lines
		self.drawnVLines=[]
		# labels
		self.drawnLab=[]
		
		# draw it on the view
		ymin,ymax=self.view.GetYAxis()
		# create the verticale line
		self.drawnVLines.append(self.view.Drawable.PutLine(self.pos,ymin,self.pos,ymax,pen=self.pen))
		# create the roi label
		self.drawnLab.append(self.view.Drawable.PutText(self.pos,ymax,self.name))

	def Redraw(self):
		"""
		redraw the peak to fit to new graph size
		"""
		ymin,ymax=self.view.GetYAxis()
		for line in self.drawnVLines:
			# update the vertical line
			self.view.Drawable.SetObjectCoords(line,self.pos,ymin,self.pos,ymax)
		for lab in self.drawnLab:
			# update the peak label
			self.view.Drawable.SetObjectCoords(lab,self.pos,ymax)

	def Move(self,pos):
		"""
		move the peak to the given position
		"""
		self.pos=pos
		ymin,ymax=self.view.GetYAxis()
		for line in self.drawnVLines: self.view.Drawable.SetObjectCoords(line,self.pos,ymin,self.pos,ymax)
		for lab in self.drawnLab: self.view.Drawable.SetObjectCoords(lab,self.pos,ymax)
	
	def ShowName(self):
		"""
		show the peak name label
		"""
		if self.drawnLab==[]:
			# draw it on the view
			ymin,ymax=self.view.GetYAxis()
			# create the roi label
			self.drawnLab.append(self.view.Drawable.PutText(self.pos,ymax,self.name))
	
	def HideName(self):
		"""
		hide the peak name label
		"""
		for lab in self.drawnLab:
			self.view.Drawable.EraseObject(lab)
		self.drawnLab=[]
			
	def SetName(self,name):
		"""
		change the name attribute, and redraw it on the graph
		"""
		self.name=name
		for label in self.drawnLab:
			self.view.Drawable.UpdateObject(label,textOptsDic={'text':name})
	
	def SetPenValues(self,style=None,width=None,color=None):
		"""
		change pen values and redraw it
		"""
		changed=0
		if style is not None and self.pen.style!=style:
			self.pen.style=style
			changed=1
		if width is not None and self.pen.width!=width:
			self.pen.width=width
			changed=1
		if color is not None and self.pen.color!=color:
			self.pen.color=color
			changed=1
			
		if changed:
			# redraw
			for rect in self.drawnVLines:
				self.view.Drawable.UpdateObject(rect,self.pen)

class GraphViewRoi:
	"""
	Class representing a roi made on a graph
	"""
	def __init__(self,view,name='roi',limits=[],pen=Pen((0,0,0),2,"solid"),brush=Brush((0,0,0),"fill_25")):
		"""
		Constructor
			-	view : view where the roi must be drawn
			-	limits : the graph limits defining this roi : it's a sequence of min,max tuples
			-	pen : the pen style used for drawing the roi border
			-	brush : the brush style used for filling the roi		   
		"""
		self.view=view
		
		self.name=name
		self.limits=limits
		self.pen=pen
		self.brush=brush
		# drawn on view objects references
		# rectangles
		self.drawnRect=[]
		# labels
		self.drawnLab=[]
		
		# draw it on the view
		ymin,ymax=self.view.GetYAxis()
		labelYOffset=(ymax-ymin)/15
		for xmin,xmax in self.limits:
			# create the rectangle
			self.drawnRect.append(self.view.Drawable.PutRectangle(xmin,ymin,xmax,ymax,pen=self.pen,brush=self.brush))
			# create the roi label
			self.drawnLab.append(self.view.Drawable.PutText(xmin+int((xmax-xmin)/2),ymax-labelYOffset,self.name))

	##################################
	###  CHANGE ATTRIBUTES VALUES  ###
	##################################
	
	def SetLimits(self,newLimits=[]):
		"""
		set new limits for tyhis roi. Format : newLimits = [(roi1min,roi1max),(roi2min,roi2max)]
		"""
		oldLimitsNbr,newLimitsNbr=len(self.limits),len(newLimits)
		ymin,ymax=self.view.GetYAxis()
		labelYOffset=(ymax-ymin)/15
		for i in range(min(oldLimitsNbr,newLimitsNbr)):
			(oldMin,oldMax),(newMin,newMax)=self.limits[i],newLimits[i]
			if oldMin!=newMin or oldMax!=newMax:
				# update limits attribute
				self.limits[i]=(newMin,newMax)
				# update graphical items
				self.view.Drawable.SetObjectCoords(self.drawnRect[i],newMin,ymin,newMax,ymax)
				if self.drawnLab!=[]:
					self.view.Drawable.SetObjectCoords(self.drawnLab[i],newMin+int((newMax-newMin)/2),ymax-labelYOffset)
		for i in range(newLimitsNbr,oldLimitsNbr):
			# delete needless old limits 
			del self.limits[i]
			# update graphical items
			self.view.Drawable.EraseObject(self.drawnRect[i])
			del self.drawnRect[i]
			if self.drawnLab!=[]:
				self.view.Drawable.EraseObject(self.drawnLab[i])
				del self.drawnLab[i]
		for i in range(oldLimitsNbr,newLimitsNbr):
			# put new rectangles
			(newMin,newMax)=newLimits[i]
			# update limits attribute
			self.limits.append((newMin,newMax))
			# update graphical items
			self.drawnRect.append(self.view.Drawable.PutRectangle(newMin,ymin,newMax,ymax,pen=self.pen,brush=self.brush))
			if self.drawnLab!=[]:
				self.drawnLab.append(self.view.Drawable.PutText(newMin+int((newMax-newMin)/2),ymax-labelYOffset,self.name))
			
	def MoveLimit(self,oldLimit,newLimit):
		"""
		changes the old limit for new one and redraw
		"""
		# search for the limit to be changed
		i=0
		while i<len(self.limits) and (self.limits[i][0]!=oldLimit and self.limits[i][1]!=oldLimit):
			i+=1
		if i<len(self.limits):
			ymin,ymax=self.view.GetYAxis()
			labelYOffset=(ymax-ymin)/15
			xmin,xmax=self.limits[i]
			if xmin==oldLimit:
				self.limits[i]=(newLimit,xmax)
				self.view.Drawable.SetObjectCoords(self.drawnRect[i],newLimit,ymin,xmax,ymax)
				if self.drawnLab!=[]:
					self.view.Drawable.SetObjectCoords(self.drawnLab[i],newLimit+int((xmax-newLimit)/2),ymax-labelYOffset)
				#self.view.Drawable.marker_configure(self.drawnLab[i], coords=(newLimit+int((xmax-newLimit)/2),ymax-labelYOffset))
			else:
				self.limits[i]=(xmin,newLimit)
				self.view.Drawable.SetObjectCoords(self.drawnRect[i],xmin,ymin,newLimit,ymax)
				if self.drawnLab!=[]:
					self.view.Drawable.SetObjectCoords(self.drawnLab[i],xmin+int((newLimit-xmin)/2),ymax-labelYOffset)
				#self.view.Drawable.marker_configure(self.drawnLab[i], coords=(xmin+int((newLimit-xmin)/2),ymax-labelYOffset))
	
	def Redraw(self):
		"""
		redraw the roi to fit to new graph size
		"""
		# draw it on the view
		ymin,ymax=self.view.GetYAxis()
		labelYOffset=(ymax-ymin)/15
		for xmin,xmax in self.limits:
			for rect in self.drawnRect:
				# update the rectangles
				self.view.Drawable.SetObjectCoords(rect,xmin,ymin,xmax,ymax)
			for lab in self.drawnLab:
				# update the roi labels
				self.view.Drawable.SetObjectCoords(lab,xmin+int((xmax-xmin)/2),ymax-labelYOffset)
	
	def SetName(self,name):
		"""
		change the name attribute, and redraw it on the graph
		"""
		self.name=name
		for label in self.drawnLab:
			self.view.Drawable.UpdateObject(label,textOptsDic={'text':name})
	
	def SetPenValues(self,style=None,width=None,color=None):
		"""
		change pen values and redraw it
		"""
		changed=0
		if style is not None and self.pen.style!=style:
			self.pen.style=style
			changed=1
		if width is not None and self.pen.width!=width:
			self.pen.width=width
			changed=1
		if color is not None and self.pen.color!=color:
			self.pen.color=color
			changed=1
			
		if changed:
			# redraw
			for rect in self.drawnRect:				
				self.view.Drawable.UpdateObject(rect,self.pen,self.brush)

	def SetBrushValues(self,fill=None,color=None):
		"""
		change brush values and redraw it
		"""
		changed=0
		if fill is not None and self.brush.style!=fill:
			self.brush.style=fill
			changed=1
		if color is not None and self.brush.color!=color:
			self.brush.color=color
			changed=1
			
		if changed:
			# redraw
			for rect in self.drawnRect:
				self.view.Drawable.UpdateObject(rect,self.pen,self.brush)
	
	def ShowName(self):
		"""
		show the roi name in in all rectangle
		"""
		if self.drawnLab==[]:
			# if no label drawn : draw it on the view
			ymin,ymax=self.view.GetYAxis()
			labelYOffset=(ymax-ymin)/15
			for xmin,xmax in self.limits:
				# create the roi label
				self.drawnLab.append(self.view.Drawable.PutText(xmin+int((xmax-xmin)/2),ymax-labelYOffset,self.name))
	
	def HideName(self):
		"""
		hide the roi name labels
		"""
		for lab in self.drawnLab:
			self.view.Drawable.EraseObject(lab)
		self.drawnLab=[]

#######################
#######  TEST  ########
#######################

#--- menu popup callbacks relative to graphes ---

def delGraph():
	global view
	graphToDel=view.GetActiveGraph()
	if graphToDel is not None:
		view.DelGraph(graphToDel['name'])

#--- menu popup callbacks relative to rois ---

def addRoi():
	global view
	if view.Select is not None:
		((xmin,y0),(xmax,y1))=view.Select.Selection["BoundingRect"]
		view.Select.Erase(view)
		view.PutRoi(name='roi',limits=[(xmin,xmax)],pen=Pen((198,154,236),2,"solid"),brush=Brush((198,154,236),"fill_25"))

def delRoi():
	global view
	roiToDel=view.GetActiveRoi()
	if roiToDel is not None:
		view.DelRoi(roiToDel.name)
	
def showRoiLabels():
	global view
	view.ShowRoisNames()

def hideRoiLabels():
	global view
	view.HideRoisNames()

#--- menu popup callbacks relative to rois ---
	
def addPeak():
	global view
	if view.Select is not None:
		((pos,y0),(pos,y1))=view.Select.Selection["BoundingRect"]
		view.PutPeak(name='pk',pos=pos,pen=Pen((0,0,0),2,"solid"))
	
def delPeak():
	global view
	pkToDel=view.GetActivePeak()
	if pkToDel is not None:
		view.DelPeak(pkToDel.name)
	
def showPeakLabels():
	global view
	view.ShowPeaksNames()

def hidePeakLabels():
	global view
	view.HidePeaksNames()

def enableActiveGraphHighlight():
	global view
	view.SetActiveGraphPen((255,0,0),2,'dotted')
	
def disableActiveGraphHighlight():
	global view
	view.SetActiveGraphPen(None)
	

def test():
	import math
	NB_POINTS=500
	global view
	if "qt" in sys.modules.keys():
		root = qt.QApplication(sys.argv)	  
		view = ExtendedGraphView(None)
	else:
		root = Tkinter.Tk()
		view = ExtendedGraphView(root,{'StatusBar':1,'RoiSelection':1,'PeakSelection':1})
	
	# create test command in menu popup
	view.AddMenuSeparator()
	view.AddMenuPopupItem("Del graph",delGraph)
	if view.opts['RoiSelection']==1:
		view.AddMenuSeparator()
		view.AddMenuPopupItem("Add roi",addRoi)
		view.AddMenuPopupItem("Del roi",delRoi)
		view.AddMenuPopupItem("Show roi labels",showRoiLabels)
		view.AddMenuPopupItem("Hide roi labels",hideRoiLabels)
	if view.opts['PeakSelection']==1:
		view.AddMenuSeparator()
		view.AddMenuPopupItem("Add peak",addPeak)
		view.AddMenuPopupItem("Del peak",delPeak)
		view.AddMenuPopupItem("Show peaks label",showPeakLabels)
		view.AddMenuPopupItem("Hide peaks label",hidePeakLabels)
	view.AddMenuSeparator()
	view.AddMenuPopupItem("enable active graph highlight",enableActiveGraphHighlight)
	view.AddMenuPopupItem("disable active graph highlight",disableActiveGraphHighlight)
	
	# create graph datas to display
	arr1 = Numeric.zeros ((NB_POINTS,))
	arr2 = Numeric.zeros ((NB_POINTS,))
	for i in range(NB_POINTS):
		arr1[i]= i
		arr2[i]= (i/2)*math.sin(math.sqrt(i))
	graphData1=Graph(arr1,pen=Pen((0,0,255),3,"solid"))
	graphData2=Graph(arr2,pen=Pen((0,0,255),3,"solid"))
	view.SetSource((graphData1,graphData2))
	view.SetSize(500,500)
	view.Show()

	if "qt" in sys.modules.keys():
		root.setMainWidget(view)
		root.exec_loop()
	else:
		root.mainloop()

if __name__ == '__main__':
	test()
