"""
    Filter.py
    Filter base class
"""

__version__=  '1.0'
__date__ =    '01 June 2002'
__author__ =  'Alexandre Gobbo (gobbo@esrf.fr)'


import EventHandler


class Filter:
    """
    Filter objects are inserted between Data and View objects. They represent
    any kind of transformation of data, or the way it is represented. Filters
    can be cascaded.
    These objects have just one source (a Data or another Filter) and
    can produce output to one or more objects (either View or another Filter
    object).
    A particular case of Filter is DataSelection classes, which represent
    a slice of the Data objects (they have Data objects as sources).
    Other Filters can be cascaded to perform operations on the data, and to
    conform the data to the format required by View object.
    Filter base class implements an interface that allows derived classes to
    comunicate to source and output objects.

    Interface:
    ===========================
        __init__
        SetSynchronized
        GetData
        GetSource
        Refresh
        Destroy

    Overridables
    ===========================
        DataCoord2SelectionCoord
            (default implementation works if filter doesn't change coordinate system)
        SelectionCoord2DataCoord
            (default implementation works if filter doesn't change coordinate system)
        DataChanged

    Virtuals:
    ===========================
        GetOutput


    Internal Events:
    ===========================
        Emits:
            "DataChange"
            "DeleteEvent"
        Receives:
            "DataChange"
            "DeleteEvent"

    """
    def __init__(self,source,synchronized=1,buffer_input=0):
        """
        Constructor
        Parameters:
          source: the source object (a Filter or Data derived object)
          synchronized: if zero the filter won't answer to DataChange
                        events from the source (no dynamic updating)
          buffer_input: if non-zero, default implementation of
                  DataChanged event stores input in self.Input.          
                Normally the filter is going to query it's source's output.
                inside it's GetOutput. This in order:
                - not to execute retrieving operations but for data that actually
                  is going to be displayed.
                - not to always hold source dara in memory
                When working this way, buffer_input should be 0 (default)

                By setting buffer_input to 1, there's no need to call the source in
                GetOutput. This is useful if:
                - the data is accessed in more than one place than GetOutput,
                  by multiple receivers,or if reading/processing data is slow.
                - Setting transformation parametrs and promote a redrawing without having
                  to read the source.
                  
                buffer_input must be 0 to DataSelection classes.              

          
        """        
        self.eh=EventHandler.EventHandler()
        self.DataChange = self.eh.create("DataChange")
        self.DeleteEvent = self.eh.create("DeleteEvent")        
        self.Synchronized=synchronized
        self.BufferInput=buffer_input
        if self.BufferInput: self.Input={}
        if source!=None:
            source.eh.register("DeleteEvent",self.Destroy)
            self.Source=source
            if self.Synchronized:
               source.eh.register("DataChange",self.DataChanged)
            if hasattr (source,"DataRef"):
                self.DataRef=source.GetData()
            else:
                self.DataRef=source
        else:
            self.DataRef=None
            self.Source=None
        self.Alive=1
        if buffer_input: self.DataChanged(None)

    def SetSynchronized(self,synchronized):
        """
        Set/reset on-line mode
        Parameters:
          synchronized: if zero the filter won't answer to DataChange
                        events from the source (no dynamic updating)
        """        
        source=self.GetSource()
        if source==None: return
        if synchronized!=self.Synchronized:
            self.Synchronized=synchronized
            if self.Synchronized:
                source.eh.register("DataChange",self.DataChanged)
            else:
                source.eh.unregister("DataChange",self.DataChanged)


    def Refresh(self):
        """
        Generates DataChange event to inform linked objects to read data
        from the filter.
        """        
        source=self.GetSource()
        if source==None: return
        self.eh.event(self.DataChange, self)

    def GetSource(self):
        """
        Returns source object.
        """        
        return self.Source
        
    
    def GetData(self):
        """
        Returns linked Data object (as a filter has just une source, it can only
        be linked to a single Data object).
        """        
        return self.DataRef


    def GetDataSelection(self):
        """
        Returns linked DataSelection
        """        
        obj=self
        while (1):
            source=obj.GetSource()
            if source==self.DataRef: return obj
            if source==None:return None
            obj=source
            
            


    def Destroy(self,source=None):
        """
        Cleanup.
        As event handlers and callbacks, by keeping a reference to other
        objects, avoid them to be destroyed (and freed from memory), the
        Destroy method has to be explicitly called for Filter objects
        before they get out of scope.
        I had tried to use weak references in order not to need this method,
        but this generated many other troubles (like impossibility to define
        selections locally, difficult control of the destruction sequence,
        not interceptable error mesassages...).
        """
        if self.Alive==0: return
        self.eh.event(self.DeleteEvent, self)     

        if self.Source is not None:
            if self.Source.eh is not None:
                self.Source.eh.unregister("DataChange",self.DataChanged)
                if self.Synchronized:
                    self.Source.eh.unregister("DeleteEvent",self.Destroy)
        self.Source=None
        self.DataRef=None
        self.eh=None
        self.Alive=0

#################################
#       OVERRIDABLES
#################################
    def DataCoord2SelectionCoord(self,data_coord):
        """
        Overridable: Derived classes need to reimplement this method if they
        change the coordinate system (translation, rotation, reduction,...).
        The default implementation just calls the source's.
        """
        source=self.GetSource()
        if source is not None:
            return source.DataCoord2SelectionCoord(data_coord)
        return (-1,-1)


    def SelectionCoord2DataCoord(self,selection_coord):
        """
        Overridable: Derived classes need to reimplement this method if they
        change the coordinate system (translation, rotation, reduction,...).
        The default implementation just calls the source's.
        """
        source=self.GetSource()
        if source is not None:
            return source.SelectionCoord2DataCoord(selection_coord)
        return DataPosition(-1,(-1,-1))


    def DataChanged(self,par):
        """
        Overridable: By the default implementation, when a filter receive a
        DataChange event it generates it's own event (self.Refresh()).
        If buffer_input option in construnctor, stores input data
        (self.GetSource().GetOutput())in self.Input.
        
        Derived classes can override this method if:
        - they want to inspect the input to find out if it is needed or not to
          generate a DataChage event.
        - it is needed to make a copy() of  source.GetOutput() into self.Input
          (if the data can change into intermediate states between two DataChanged           
          events)  
        """
        source=self.GetSource()
        ###if source==None: return
        if self.BufferInput:
            if source is not None:
                 self.Input=source.GetOutput()
            else:
                 self.Input = {}
        self.Refresh()
        

#################################
#       VIRTUAL
#################################

    def GetOutput(self):
        """
        Virtual: Derived classes implements the filter behaviour in here.
        This method should read source data, execute a transformation
        and return a dictionary with the output.
        The input data can be reached by two ways:
        - Stored in self.Input in DataChanged event, if option buffer_input
          set in constructor. In this case self.GetSource.GetOutput()
          should not be called from here, in order no to perform a
          redundant operation.
        - Queried from source inside GetOutput (default)
            If the source is a Filter it should call it's GetOutput
            to retrieve the input.
            If the source is a Data object, the filter should call Data's
            methods as GetPageArrayRegion to retrieve the input.
        The method returns a dictionary.
        {} means empty selection
        Some standarized keys:
            data:  NumPy array containing the actual data
            xdata: NumPy array containing x axis
            ydata: NumPy array containing y axis
            value: Numeric value
        Derived classes can use some of these, and define new ones.
        """
        return {}
