#!/usr/bin/env python

import string
import os
import exceptions
from Tkinter import *
import Selector
import Blt
import tkFileDialog
import tkColorChooser
import tkMessageBox
import tkSimpleDialog
import BalloonWidget

MAXCOL= 40
COLWIDTH= 4

# directory dialog box from tk
class DirOpen(tkFileDialog._Dialog):
    command= "tk_chooseDirectory"

def askdirectory(**options):
    return apply(DirOpen, (), options).show()


#
# Entry field generic class
#
class Field:
    """ Generic Field class:
	the following methods can be overwritten:
	- buildfield: build the field with a given master and a line number
			for the grid manager
	- setvalue: set a given value (the default one for example)
	- getvalue: get the current value
    """
    def __init__(self, name, **kw):
	self.name= name
	for key in kw.keys():
	    setattr(self, key, kw[key])
	if not hasattr(self, 'default'):
	    self.default= None
	if not hasattr(self, 'shift'):
	    self.shift= 0
	#if not hasattr(self, 'eh'):
	    #self.eh= None
	#else:
	    #if self.eh is not None:
		#self.seteh(self.eh)
	if not hasattr(self, 'validate'):
	    self.validate= None
	else:
	    if type(self.validate) is not TupleType:
		self.validate= (self.validate,)
	if not hasattr(self, 'help'):
	    self.help= None

    #def seteh(self, eh):
	#self.eh= eh
	#self.lastvalue= None
	#self.event= self.eh.create('InputChanged')

    def callback(self, event):
	new= self.getvalue()
	if new!=self.lastvalue:
	    self.lastvalue= new
	    self.eh.event(self.event, self, self.name, self.getvalue())

    def build(self, sheet, master, row, column):
	self.sheet= sheet
	nb= self.buildfield(master, row, column)
	self.setdefault()
	return nb

    def sethelp(self, widget):
	if self.help is not None:
	    self.sheet.sethelp(widget, self.help)
	    
    def check(self):
	if self.validate is not None:
	    value= self.getvalue()
	    if type(value)==ListType:
		for val in value:
		    for validator in self.validate:
			validator.check(self.name, val)
	    else:
	        for validator in self.validate:
		    validator.check(self.name, value)

    def setdefault(self, value=None):
	if value is None:
	    self.set(self.default)
	else:
	    self.default= value
	    self.set(self.default)

    def get(self):
	return { self.name: self.getvalue() }

    def set(self, value=None):
	self.lastvalue= value
	self.setvalue(value)

	
    # --- methods to be overwritten
    def buildfield(self, master, row, column):
	""" Build the field with:
		master = master frame
		row = start row for grid manager
		column = start column for grid manager
	    this method must return the number of line used
	"""
	return 1

    def setvalue(self, value=None):
	""" Set the given value
	    If value is None, reset the field
	"""
	pass

    def getvalue(self):
	""" Get the current value
	"""
	return None

#
# derivated field classes
#

class EntryField(Field):
    """ Simple text entry field
    """
    def __init__(self, name='SimpleEntry', text='Label', endtext=None, **kw):
	Field.__init__(self, name, text=text, endtext=endtext, **kw)

    def buildfield(self, master, row, column):
	label= Label(master, text=self.text)
	label.grid(row=row, column=column+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=row, column=column+1, sticky='ew')

	if self.endtext is not None:
	    endlabel= Label(master, text=self.endtext)
	    endlabel.grid(row=row, column=column+2, sticky='w')

	self.sethelp(self.entry)

	return 1

    def getvalue(self):
	res= string.strip(self.entry.get())
	if len(res):
	    return res
	else:
	    return None
	
    def setvalue(self, value=None):
	self.entry.delete(0, 'end')
	if value is not None:
	    self.entry.insert(1, value)
	    self.entry.xview('end')


class DoubleEntryField(Field):
    def __init__(self, name='DoubleEntry', text='Label', **kw):
	Field.__init__(self, name, text=text, **kw)

    def buildfield(self, master, row, col):
	label= Label(master, text=self.text)
	label.grid(row=row, column=col+0, sticky='w')

	self.entry1= Entry(master, state='normal')
	self.entry1.grid(row=row, column=col+1, sticky='ew')
	self.entry2= Entry(master, state='normal')
	self.entry2.grid(row=row, column=col+2, sticky='ew')

	self.sethelp(self.entry1)
	self.sethelp(self.entry2)
	return 1
	
    def getvalue(self):
	res1= string.strip(self.entry1.get())
	res2= string.strip(self.entry2.get())
	if not len(res1):
	    res1= None
	if not len(res2):
	    res2= None
	return [res1, res2]

    def setvalue(self, value=None):
	self.entry1.delete(0, 'end')
	self.entry2.delete(0, 'end')
	if value is not None:
	    if type(value)==ListType and len(value)==2:
	        self.entry1.insert(1, value[0])
	        self.entry1.xview('end')
	        self.entry2.insert(1, value[1])
	        self.entry2.xview('end')



class CheckEntryField(Field):
    def __init__(self, name='OptionEntry', text='Label', checked=1,
                full_output=0, **kw):
	self.full_output=full_output
	self.checked=checked
	Field.__init__(self, name, text=text, checked=checked, **kw)

    def buildfield(self, master, row, col):
	self.option= IntVar()
	check= Checkbutton(master, text=self.text, variable=self.option, command=self.setoption)
	check.grid(row=row, column=col+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=row, column=col+1, sticky='ew')

	self.setoption(self.checked)

	self.sethelp(self.entry)
	self.sethelp(check)

	return 1

    def getvalue(self):
	opt= self.option.get()
	res=string.strip(self.entry.get())
	if self.full_output:
		return res,opt
	else:    
	    if opt:
	        if not len(res):
		        return None
	        else:
		        return res
	    else:
	        return None

    def setoption(self, value=None):
	if value is None:
	    value= self.option.get()
	else:
	    self.option.set(value)
	if value:
	    self.entry.configure(state='normal')
	else:
	    self.entry.configure(state='disabled')

    def setvalue(self, value=None):
	self.entry.configure(state='normal')
	self.entry.delete(0, 'end')
	if value is not None:
	    self.entry.insert(1, value)
	    self.entry.xview('end')
	    self.setoption(self.checked)
	else:
	    self.setoption(0)

	
class LinkedCheckEntryField(Field):
    def __init__(self, name='LinkedCheckField', text=[], checked=0, **kw):
	Field.__init__(self, name, text=text, checked=checked, **kw)
	self.values= []
	self.entries= []

    def buildfield(self, master, row, col):
	nb= 0
	for idx in range(len(self.text)):
	    self.values.append(IntVar())
	    check= Checkbutton(master, text=self.text[idx], variable=self.values[idx],
			command= lambda this=self,idx=idx: LinkedCheckEntryField.onclick(this, idx))
	    check.grid(row=row+nb, column=col+0, sticky='w')

	    entry= Entry(master, state='disabled')
	    entry.grid(row=row+nb, column=col+1, sticky='ew')
	    self.entries.append(entry)
	    self.sethelp(entry)
	    self.sethelp(check)
	    nb+= 1
	return nb

    def onclick(self, index):
	value= self.values[index].get()
	if value:
	    self.entries[index].configure(state='normal')
	    for idx in range(index):
		self.values[idx].set(1)
		self.entries[idx].configure(state='normal')
	else:
	    for idx in range(index, len(self.values)):
		self.values[idx].set(0)
		self.entries[idx].configure(state='disabled')
	    
    def getvalue(self):
	result= []
	for (val, ent) in zip(self.values, self.entries):
	    if val.get():
		res= string.strip(ent.get())
		if not len(res):
		    res= None
	        result.append(res)
	return result

    def setvalue(self, value=None):
	for val in self.values:
	    val.set(0)
	for ent in self.entries:
	    ent.configure(state='normal')
	    ent.delete(0, 'end')
	if value is not None:
	    for idx in range(len(value)):
	        self.values[idx].set(1)
		self.entries[idx].insert(1, value[idx])
		self.entries[idx].xview('end')
	    for idx in range(len(value), len(self.entries)):
		self.entries[idx].configure(state='disabled')	

class EditEntryField(Field):
    """ Edit Entry
    """
    def __init__(self, name='EditEntry', text='User Entry', command=None, commandtext='Edit'):
	Field.__init__(self, name, text=text, command=command, commandtext=commandtext)

    def buildfield(self, master, row, col):
	label= Label(master, text=self.text)
	label.grid(row=row, column=col+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=row, column=col+1, sticky='ew')

	button= Button(master, text=self.commandtext, command=self.command_cb)
	button.grid(row=row, column=col+2, sticky='w')

	self.sethelp(self.entry)
	self.sethelp(button)
	return 1

    def command_cb(self):
	if self.command is not None:
	    ret= apply(self.command, (self.getvalue(),))
	    if ret is not None:
	        self.setvalue(ret)

    def getvalue(self):
	res= string.strip(self.entry.get())
	if len(res):
	    return res
	else:
	    return None

    def setvalue(self, value=None):
	self.entry.delete(0, 'end')
	if value is not None:
	    self.entry.insert(1, value)
	    self.entry.xview('end')


class EditFileInput(Field):
    """ Input File + Edit command
    """
    def __init__(self, name='FileInput', text='Input File', types=[("all files", "*.*")], command=None, **kw):
	Field.__init__(self, name, text=text, types=types, command=command, **kw)

    def buildfield(self, master, row, col):
	label= Label(master, text=self.text)
	label.grid(row=row, column=col+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=row, column=col+1, sticky='ew')

	button= Button(master, text='Find', command=self.askfile)
	button.grid(row=row, column=col+2, sticky='w')

	button= Button(master, text='Edit', command=self.editfile)
	button.grid(row=row, column=col+3, sticky='w')

	self.sethelp(self.entry)
	self.sethelp(button)
	return 1

    def getvalue(self):
	res= string.strip(self.entry.get())
	if len(res):
	    return os.path.abspath(res)
	else:
	    return None

    def askfile(self):
	if self.getvalue() is not None:
	    initialdir= os.path.dirname(self.getvalue())
	else:
	    initialdir= ''
	file= tkFileDialog.askopenfilename(initialdir=initialdir,
				initialfile=self.getvalue(),
				title=self.text, filetypes=self.types)
	if len(file):
	    self.setvalue(file)

    def setvalue(self, filename=None):
	self.entry.delete(0, 'end')
	if filename is not None:
	    self.entry.insert(1, filename)
	    self.entry.xview('end')

    def editfile(self):
	retfile= apply(self.command, (self.getvalue(), ))
	self.setvalue(retfile)
	
	
class FileInput(Field):
    """ Input File entry
    """
    def __init__(self, name='FileInput', text='Input File', types=[("all files", "*.*")], **kw):
	Field.__init__(self, name, text=text, types=types, **kw)

    def buildfield(self, master, row, col):
	label= Label(master, text=self.text)
	label.grid(row=row, column=col+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=row, column=col+1, sticky='ew')

	button= Button(master, text='...', width=1, command=self.askfile)
	button.grid(row=row, column=col+2, sticky='w')

	self.sethelp(self.entry)
	self.sethelp(button)
	return 1

    def getvalue(self):
	res= string.strip(self.entry.get())
	if len(res):
	    return os.path.abspath(res)
	else:
	    return None

    def askfile(self):
	if self.getvalue() is not None:
	    initialdir= os.path.dirname(self.getvalue())
	else:
	    initialdir= ''
	file= tkFileDialog.askopenfilename(initialdir=initialdir,
				initialfile=self.getvalue(),
				title=self.text, filetypes=self.types)
	if len(file):
	    self.setvalue(file)

    def setvalue(self, filename=None):
	self.entry.delete(0, 'end')
	if filename is not None:
	    self.entry.insert(1, filename)
	    self.entry.xview('end')


class FileOutput(FileInput):
    """ Output File entry
    """
    def askfile(self):
	val= self.getvalue()
	if val is not None:
	    initialdir= os.path.dirname(val)
	else:
	    initialdir= ''
	file= tkFileDialog.asksaveasfilename(initialfile=val,
				initialdir=initialdir,
				title=self.text, filetypes=self.types)
	if len(file):
	    self.setvalue(file)

class DirInput(FileInput):
    """ Input directory entry (any directory with read permission)
    """
    def askfile(self):
	if self.getvalue() is not None:
	    initialdir= os.path.dirname(self.get())
	else:
	    initialdir= ''
	dir= askdirectory(initialdir=initialdir,title=self.text,mustexist=1)
	if len(dir):
	    self.setvalue(dir)

class DirOutput(FileInput):
    """ Input directory entry (any directory with write permission)
    """
    def askfile(self):
	if self.getvalue() is not None:
	    initialdir= os.path.dirname(self.getvalue())
	else:
	    initialdir= ''
	dir= askdirectory(initialdir=initialdir,title=self.text,mustexist=0)
	if len(dir):
	    self.setvalue(dir)

class LinkedCheckField(Field):
    def __init__(self, name='LinkedCheckField', text=[], **kw):
	Field.__init__(self, name, text=text, **kw)
	self.values= []

    def buildfield(self, master, row, col):
	nb= 0
	for idx in range(len(self.text)):
	    self.values.append(IntVar())
	    check= Checkbutton(master, text=self.text[idx], variable=self.values[idx],
			command= lambda this=self,idx=idx: LinkedCheckField.onclick(this, idx))
	    check.grid(row=row+nb, column=col+0, sticky='w')
	    self.sethelp(check)
	    nb+= 1
	return nb

    def onclick(self, index):
	value= self.values[index].get()
	if value:
	    for idx in range(index):
		self.values[idx].set(1)
	else:
	    for idx in range(index, len(self.values)):
		self.values[idx].set(0)
	    
    def getvalue(self):
	sum= 0
	for value in self.values:
	    sum+= value.get()
	return sum

    def setvalue(self, value=None):
	for val in self.values:
	    val.set(0)
	if value is not None:
	    for idx in range(value):
	        self.values[idx].set(1)

class CheckField(Field):
    """ Flag option to check
    """
    def __init__(self, name='CheckField', text='Label', **kw):
	Field.__init__(self, name, text=text, **kw)

    def buildfield(self, master, row, col):
	self.value= IntVar()
	check= Checkbutton(master, text=self.text, variable=self.value)

	check.grid(row=row, column=col+0, columnspan=2, sticky='w')

	self.sethelp(check)
	return 1

    def getvalue(self):
	return self.value.get()
	
    def setvalue(self, value=None):
	if value is not None:
	    self.value.set(value)
	else:
	    self.value.set(0)


class RadioField(Field):
    """ List of radio buttons linked together with an optionnal text header
    """
    def __init__(self, name='RadioField', text=None, values=[], default=0, **kw):
	Field.__init__(self, name, text=text, values=values, default=default, **kw)

    def buildfield(self, master, row, col):
	i= 0
	if self.text:
	    label= Label(master, text=self.text)
	    label.grid(row=i+row, column=self.shift+0, columnspan=MAXCOL, sticky='w')
	    i+=1
	self.selected= IntVar()
	nb= 0
	for text in self.values:
	    radio= Radiobutton(master, text=text, value=nb, variable=self.selected)
	    radio.grid(row=i+row, column=col+0, sticky='w')
	    self.sethelp(radio)
	    nb+=1
	    i+=1
	return i

    def getvalue(self):
	return self.selected.get()

    def setvalue(self, value=None):
	if value is not None:
	    self.selected.set(value)
	else:
	    self.selected.set(0)


class SelectField(Field):
    def __init__(self, name='SelectField', text='label', values=[], index=0, **kw):
	""" if index==1: index of selected value will be returned
		otherwise the value selected itself will be returned
	"""
	Field.__init__(self, name, text=text, values=values, index=index, **kw)

    def buildfield(self, master, row, col):
	label= Label(master, text=self.text)
	label.grid(row=row, column=col+0, sticky='w')

	self.select= Selector.Selector(master, self.values)
	self.select.grid(row=row, column=col+1, sticky='w')

	self.sethelp(self.select)
	return 1

    def getvalue(self):
	if self.index:
	    return self.values.index(self.select.value.get())
	else:
	    return self.select.value.get()

    def setvalue(self, value=None):
	if self.index:
	    item= self.values[value]
	else:
	    item= value
	if item is None:
	    item= self.values[0]
	self.select.show_item(item)

class ColorField(Field):
    def __init__(self, name='ColorField', text='color', **kw):
	Field.__init__(self, name, text=text, **kw)

    def buildfield(self, master, row, col):
	label= Label(master, text=self.text)
	label.grid(row=row, column=col, sticky='w')

	self.button= Button(master, command=self.__askcolor, width=10, bg='red')
	self.button.grid(row=row, column=col+1, sticky='ew')

	self.sethelp(self.button)
	return 1

    def __askcolor(self):
	newcolor= tkColorChooser.askcolor(color=self.getvalue())
	self.setvalue(newcolor[1])

    def getvalue(self):
	return self.button.cget('bg')

    def setvalue(self, value=None):
	self.button.configure(background=value, activebackground=value)
	
#
# Null field class (no output)
#
class NullField:
    """ Class to define a null field (which return any parameter).
	The only method to overwrite is buildfield (same as class Field)
    """
    def __init__(self, **kw):
	for key in kw.keys():
	    setattr(self, key, kw[key])
	self.name= None
	if not hasattr(self, 'shift'):
	    self.shift= 0
	self.eh= None

    def build(self, sheet, master, row, column):
	return self.buildfield(master, row, column)

    def check(self):
	pass

    def buildfield(self, master, row, column):
	return 0

    def setdefault(self):
	pass

    def set(self, value):
	pass

    def get(self):
	return {}

#
# Derivated null class field
#	
class TextField(NullField):
    """ Simple text label (no entry) to add comment/information
    """
    def __init__(self, text='My text', shift=0):
	NullField.__init__(self, text=text)
	self.shift= shift

    def buildfield(self, master, row, col):
	col= col
	if type(self.text)==StringType:
	    Label(master, text=self.text).grid(row=row, column=col, columnspan=MAXCOL, sticky='w')
	if type(self.text)==ListType:
	    for text in self.text:
		lb= Label(master, text=text)
		lb.grid(row=row, column=col, sticky='w')
		col+=1
	return 1


class SpaceField(NullField):
    """ Add an empty line
    """
    def __init__(self, line=1, width=None):
	NullField.__init__(self, line=line)
	self.width= width

    def buildfield(self, master, row, column):
	for iadd in range(self.line):
	    lb= Label(master)
	    if self.width is not None:
		lb.configure(width=self.width)
	    lb.grid(row=row+iadd, col=column, columnspan=COLWIDTH)
	return self.line


class SeparatorField(NullField):
    """ Add a separator
    """
    def __init__(self, height=2, padding=2):
	NullField.__init__(self, height=height, padding=padding)

    def buildfield(self, master, row, col):
	pad1= Frame(master, height=self.padding)
	pad2= Frame(master, height=self.padding)
	sep= Frame(master, bg='black', height=self.height)
	pad1.grid(row=row, column=col, columnspan=MAXCOL, sticky='ewns')
	sep.grid(row=row+1, column=col, columnspan=MAXCOL, sticky='ewns')
	pad2.grid(row=row+2, column=col, columnspan=MAXCOL, sticky='ewns')
	return 3
	

class TitleField(NullField):
    def __init__(self, text=None, relief='raised', bg='blue', fg='white'):
	NullField.__init__(self, text=text, relief=relief, bg=bg, fg=fg)

    def buildfield(self, master, row, col):
	frame= Label(master, text=self.text, relief=self.relief, bg=self.bg, fg=self.fg)
	frame.grid(row=row, column=col, columnspan=MAXCOL, sticky='ew')
	return 1


#
# Validate classes
#
class InvalidField(exceptions.Exception):
    """ Exception that should be raised in a Validate class
    """
    def __init__(self, text=None):
	self.text= text

class Validate:
    """ Base class for validator. Overwrite method check
    """
    def __init__(self, error=None, **kw):
	self.error= error
	for key in kw.keys():
	    setattr(self, key, kw[key])

    def senderror(self, name, value, message):
	""" Raise an error
	"""
	raise InvalidField, '%s: <%s> %s.'%(name, value, message)

    def check(self, name, value):
	pass


class IsFloat(Validate):
    def check(self, name, value):
	try:
	    val= float(value)
	except:
	    self.senderror(name, value, 'should be a number')

class IsPositive(Validate):
    def check(self, name, value):
	IsFloat().check(name, value)
	if float(value)<0.0:
	    self.senderror(name, value, 'should be a positive number')

class IsNegative(Validate):
    def check(self, name, value):
	IsFloat().check(name, value)
	if float(value)>0.0:
	    self.senderror(name, value, 'should be a negative number')

class MinMax(Validate):
    def __init__(self, min=None, max=None, **kw):
	Validate.__init__(self, min=min, max=max, **kw)

    def check(self, name, value):
	IsFloat().check(name, value)
	val= float(value)
	if self.min is not None:
	    if val<self.min:
		self.senderror(name, value, 'should be greater than %f'%self.min)
	if self.max is not None:
	    if val>self.max:
		self.senderror(name, value, 'should be lower than %f'%self.max)
	
class NonBlank(Validate):
    def check(self, name, value):
	if value is None or len(value)==0:
	    self.senderror(name, value, 'should be non-blank')

#
# sheet classes
#
class Sheet:
    """ Class to manage all fields in one sheet
    """
    def __init__(self, notetitle=None, fields=(), balloon=None):
	self.notetitle= notetitle
	#self.eh= eh
	self.balloon= balloon

	self.fields= ()
	self.nbfield= 1
	for field in fields:
	    if type(field)==TupleType:
		self.fields+=(field,)
		self.nbfield= max(self.nbfield, len(field))
	    else:
		self.fields+=(((field,),))

    def build(self, master):
	self.frame= Frame(master, bd=2, relief='ridge')
	line= 0
	lineused= 1
	for rowfield in self.fields:
	    colused= 0
	    for colfield in rowfield:
		#if self.eh is not None and colfield.eh is None:
		    #colfield.seteh(self.eh)
		lineused= max(lineused, colfield.build(self, self.frame, line, colused+colfield.shift))
		colused+= COLWIDTH
	    line+= lineused
	    
    def get(self):
	res= {}
	for rowfield in self.fields:
	    for colfield in rowfield:
	        res.update(colfield.get())
	return res

    def validate(self):
	errors= {}
	for rowfield in self.fields:
	    for colfield in rowfield:
	        try:
		    colfield.check()
	        except InvalidField, error:
		    errors[colfield.name]= error.text
	return errors

    def set(self, newdict):
	allkey= newdict.keys()
	for rowfield in self.fields:
	    for colfield in rowfield:
	        if colfield.name in allkey:
		    colfield.set(newdict[colfield.name])

    def sethelp(self, widget, message):
	if self.balloon is None:
	    self.balloon= BalloonWidget.BalloonWidget(self.frame)
	self.balloon.bind(widget, message, statusHelp=None)
		
    def reset(self):
	for rowfield in self.fields:
	    for colfield in rowfield:
	        colfield.set(None)

    def setdefault(self, dicdefault=None):
	if dicdefault is None:
	    for rowfield in self.fields:
		for colfield in rowfield:
	            colfield.setdefault()
	else:
	    allkey= dicdefault.keys()
	    for rowfield in self.fields:
		for colfield in rowfield:
		    if colfield.name in allkey:
		        colfield.setdefault(dicdefault[colfield.name])

class SheetFrame(Frame):
    def __init__(self, root, notebook=0, notetitle=(), sheets=(),
                type=None, default=None, init=None):
	Frame.__init__(self, root)

	self.sheets= sheets
	self.default= default
	self.init= init
	self.notebook= notebook

	if type == 'notebook':
	    self.notebook=1
	self.notetitle= notetitle

	self.__build()

	if self.default:
	    self.setdefault(self.default)
	if self.init:
	    self.set(self.init)

    def __build(self):
	""" build one sheet per notebook page
	"""
	if self.notebook:
	    self.tabset= Blt.Tabset(self,relief='flat',tiers=3,bd=0,name='notebook') 
	    self.tabset.pack(fill='both',expand='yes')

	    for loop in range(len(self.notetitle), len(self.sheets)):
		self.notetitle += (None, )
	    idx= 1
	    for (sheet,title) in zip(self.sheets, self.notetitle):
	        shframe= self.__build_sheet(self.tabset, sheet)
		if title is None:
		    try:
			title= sheet.notetitle or 'Tab %d'%idx
		    except:
			title= 'Tab %d'%idx
	        self.tabset.insert(idx, title, window=shframe)
	        idx += 1
	else:
	    frame= self.__build_sheet(self, self.sheets)
	    frame.pack(fill='both', expand='yes',side='top')

    def __build_sheet(self, master, sheets):
	if type(sheets)==TupleType:
	    frame= Frame(master)
	    row= 0
	    for rowsheet in sheets:
		if type(rowsheet)==TupleType:
		    col= 0
		    for colsheet in rowsheet:
			colsheet.build(frame)
			colsheet.frame.grid(row=row, column=col, sticky='ewns')
			col += 1
		else:
		    rowsheet.build(frame)
		    rowsheet.frame.grid(row=row, sticky='ewns')
		row += 1
	    return frame
	else:
	    sheets.build(master)
	    return sheets.frame

    def check(self):
	return self.__check(self.sheets)

    def __check(self, sheets):
	if type(sheets)==TupleType:
	    errors= {}
	    for sheet in sheets:
		errors.update(self.__check(sheet))
	    return errors
	else:
	    return sheets.validate()
	
    def get(self):
	return self.__get(self.sheets)

    def __get(self, sheets):
	if type(sheets)==TupleType:
	    result= {}
	    for sheet in self.sheets:
	        result.update(self.__get(sheet))
	    return result
	else:
	    return sheets.get()

    def set(self, newdict):
	self.__set(self.sheets, newdict)

    def __set(self, sheets, newdict):
	if type(sheets)==TupleType:
	    for sheet in sheets:
		self.__set(sheet, newdict)
	else:
	    sheets.set(newdict)

    def setdefault(self, dicdefault=None):
	self.__setdefault(self.sheets, dicdefault)

    def __setdefault(self, sheets, dicdefault):
	if type(sheets)==TupleType:
	    for sheet in sheets:
		self.__setdefault(sheet, dicdefault)
	else:
	    sheets.setdefault(dicdefault)
	    
    def reset(self):
	self.__reset(self.sheets)

    def __reset(self, sheets):
	if type(sheets)==TupleType:
	    for sheet in sheets:
		self.__reset(sheet)
	else:
	    sheets.reset()


class SheetDialog(tkSimpleDialog.Dialog):
    def __init__(self, parent=None, title=None, sheets=(), notebook=0,
                type=None, default=None, init=None, validate=1):
	self.type=type
	if parent is None:
	    parent= Tk()
	    parent.withdraw()
	self.sheets= sheets
	self.notebook= notebook
	self.default= default
	self.init= init
	self.valid= validate
	self.wtitle= title
	tkSimpleDialog.Dialog.__init__(self, parent, title)

    def body(self,master):
	self.bodywid= SheetFrame(master, notebook=self.notebook, sheets=self.sheets, 
					type=self.type,default=self.default, init=self.init)
	self.bodywid.pack(expand='yes',fill='both')

    def buttonbox(self):
	sep = Frame(self,name='dialogseparator',bd=2)
	box = Frame(self,name='dialogbuttonbox')

	self.okb     = Button(box,command=self.ok,text='OK',name='okbutton')
	self.cancelb = Button(box,command=self.cancel,text='cancel',name='cancelbutton')
	self.defaultb= Button(box,command=self.setdefault,text='default',name='defaultbutton')
	self.resetb  = Button(box,command=self.reset,text='reset',name='resetbutton')

	self.bind('<Return>',self.ok)
	self.bind('<Escape>',self.cancel)
	
	self.resetb.pack(side='left')
	self.defaultb.pack(side='left')
	self.cancelb.pack(side='left')
	self.okb.pack(side='left')
	sep.pack(fill='x',pady=3)
	box.pack(anchor='center')

    def validate(self):
	if self.valid:
	    errors= self.bodywid.check()
	    if len(errors.keys())>0:
	        self.showerror(errors)
	        return 0
	    else:
		return 1
	else:
	    return 1

    def apply(self):
	self.result= self.bodywid.get()

    def reset(self):
	self.bodywid.reset()

    def setdefault(self):
	self.bodywid.setdefault()
	
    def showerror(self, errors):
	text= ''
	for name in errors.keys():
	    text+= '%s\n' % errors[name]
	if self.wtitle is not None:
	    title= '%s - ERROR'%self.wtitle
	else:
	    title= 'ERROR'
	tkMessageBox.showerror(title, text)


def testframe():
    sheet3= Sheet(notetitle='first', 
		  fields=(EntryField('text', 'Your name', default='manu', validate=IsPositive()),
			  FileInput('infile', 'Input file', default='infile.dat'),
			  FileOutput('outfile', 'Output file', default='outfile.dat')))
    sheet4= Sheet(notetitle='second',
	   	  fields=(EntryField('text2', 'Your name', default='manu'),
			  FileInput('infile2', 'Input file', default='infile.dat'),
			  FileOutput('outfile2', 'Output file', default='outfile.dat')))
    app = Tk()
    app.title('Test SheetFrame class')
    frm= SheetFrame(app, sheets=(sheet3,sheet4), type='notebook')
    frm.pack()
    frm.set( {'text': 'Emmanuel', 'text2': 'Papillon'} )
    app.mainloop()


def testdialog():
    sheet1= Sheet(notetitle='Entry/Text/Check',
	       fields=(	TitleField('First title'),
			EntryField('ent1', 'An entry field', help="Here is my text message"),
			EntryField('ent2', 'With end text', endtext='???', help='Help Message'),
			SeparatorField(),
			TextField('Single text field'),
			TextField('The same with a shift', shift=1),
			TextField(['col1', 'col2', 'col3', 'col4']),
			SeparatorField(),
			CheckField('flag', 'Check it !!', help='Help Message'),
			SpaceField(),
			CheckEntryField('Checkent1', 'Check and entry mixed', help='Help Message'),
		))

    sheet2= Sheet(notetitle='File/Dir',
		fields=(TitleField('File input / output'),
			FileInput('infile', 'Input File', help='Find or type input filename'),
			FileOutput('outfile', 'Output File', help='Find or type output filename'),
			TitleField('Directory input / output'),
			DirInput('indir', 'Input directory', help='Help Message'),
			DirOutput('outdir', 'Output directory', help='Help Message'),
			SeparatorField(),
			EntryField('ent3', 'Or use an edit entry', help='Help Message'),
		))

    sheet3= Sheet(notetitle='Linked Field',
		fields=(TextField('RadioField'),
			SeparatorField(padding=6, height=6),
			RadioField('radio', values=['Choose this one','or choose this one'], help='Help Message'),
			SpaceField(),
			TextField('LinkedCheckField'),
			SeparatorField(),
			LinkedCheckField('linkcheck', ['first choice',
							'second possible only if first checked',
							'last possible only if first and second checked'],
						shift=1, help='Help Message'),
			SpaceField(),
			TextField('LinkedCheckEntryField'),
			SeparatorField(),
			LinkedCheckEntryField('linkchecktext',
						['the same', 'as previous', 'with entry'], help='Help Message')
			))
			
    sheet4= Sheet(notetitle='Others', 
		fields=(TitleField('SelectField'),
			SelectField('host','Hostname', ['expgi', 'expgj', 'expglin'], help='Help Message'),
			SpaceField(),
			TitleField('DoubleEntryField'),
			TextField([None, 'From', 'To']),
			DoubleEntryField('dbent_X', 'X range', help='Help Message'),
			DoubleEntryField('dbent_Y', 'Y range', help='Help Message')
		))

    sheet5= Sheet(notetitle='Multi Field',
		  fields=(TitleField('Multi Field'),
			  TextField('Add several fields on the same line'),
			  ( CheckField('ch_1', 'First check'),
			    CheckField('ch_2', 'Second check'),
			    CheckField('ch_3', 'Third check') ),
			  ( EntryField('ent_1', 'First entry'),
			    EntryField('ent_2', 'Second entry'),
			    EntryField('ent_3', 'Third entry') )
		))

    root= Tk()
    root.withdraw()
    app= SheetDialog(root, title='Option Sheet',
	             sheets= (sheet1, sheet2, sheet3, sheet4, sheet5),
		     notebook=1, validate=0, init={'ent1': 75, 'infile':'toto_in.dat'},
			default={'ent1':50, 'infile':'default.dat'})
    print app.result

if __name__=="__main__":
    #testframe()
    testdialog()

