#!/usr/bin/env python

import string
import os
import exceptions
from Tkinter import *
import Selector
import Blt
import tkFileDialog
import tkMessageBox
import tkSimpleDialog

MAXCOL= 10

# directory dialog box from tk
class DirOpen(tkFileDialog._Dialog):
    command= "tk_chooseDirectory"

def askdirectory(**options):
    return apply(DirOpen, (), options).show()


#
# Entry field generic class
#
class Field:
    """ Generic Field class:
	the following methods can be overwritten:
	- buildfield: build the field with a given master and a line number
			for the grid manager
	- set: set a given value (the default one for example)
	- get: get the current value
    """
    def __init__(self, name, **kw):
	self.name= name
	for key in kw.keys():
	    setattr(self, key, kw[key])
	if not hasattr(self, 'default'):
	    self.default= None
	#if not hasattr(self, 'command'):
	#    self.command= None
	#if not hasattr(self, 'commandtext'):
	#    self.commandtext= None
	if not hasattr(self, 'shift'):
	    self.shift= 0
	if not hasattr(self, 'validate'):
	    self.validate= None
	else:
	    if type(self.validate) is not TupleType:
		self.validate= (self.validate,)
	
    def build(self, master, line):
	nb= self.buildfield(master, line)
	self.setdefault()
	#if self.command:
	#    cmdtext= self.commandtext or 'Edit'
	#    cmd= Button(master, text=cmdtext, command=self.usercall)
	#    cmd.grid(row=line, rowspan=nb, column=3)
	return nb

    #def usercall(self):
	#print 'usercall', self.command

    def check(self):
	if self.validate is not None:
	    value= self.get()
	    if type(value)==ListType:
		for val in value:
		    for validator in self.validate:
			validator.check(self.name, val)
	    else:
	        for validator in self.validate:
		    validator.check(self.name, value)

    def setdefault(self, value=None):
	if value is None:
	    self.set(self.default)
	else:
	    self.default= value
	    self.set(self.default)

    def getdict(self):
	return { self.name: self.get() }

    # --- methods to be overwritten
    def buildfield(self, master, line):
	""" Build the field with:
		master = master frame
		line = line number for the grid manager
	    this method must return the number of line used
	"""
	return 1

    def set(self, value=None):
	""" Set the given value
	    If value is None, reset the field
	"""
	pass

    def get(self):
	""" Get the current value
	"""
	return None

#
# derivated field classes
#

class EntryField(Field):
    """ Simple text entry field
    """
    def __init__(self, name='SimpleEntry', text='Label', endtext=None, **kw):
	Field.__init__(self, name, text=text, endtext=endtext, **kw)

    def buildfield(self, master, line):
	label= Label(master, text=self.text)
	label.grid(row=line, column=self.shift+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=line, column=self.shift+1, sticky='ew')

	if self.endtext is not None:
	    endlabel= Label(master, text=self.endtext)
	    endlabel.grid(row=line, column=self.shift+2, sticky='w')

	return 1

    def get(self):
	return string.strip(self.entry.get())
	
    def set(self, value=None):
	self.entry.delete(0, 'end')
	if value is not None:
	    self.entry.insert(1, value)
	    self.entry.xview('end')

class DoubleEntryField(Field):
    def __init__(self, name='DoubleEntry', text='Label', **kw):
	Field.__init__(self, name, text=text, **kw)

    def buildfield(self, master, line):
	label= Label(master, text=self.text)
	label.grid(row=line, column=self.shift+0, sticky='w')

	self.entry1= Entry(master, state='normal')
	self.entry1.grid(row=line, column=self.shift+1, sticky='ew')
	self.entry2= Entry(master, state='normal')
	self.entry2.grid(row=line, column=self.shift+2, sticky='ew')

	return 1
	
    def get(self):
	return [string.strip(self.entry1.get()), string.strip(self.entry2.get())]

    def set(self, value=None):
	self.entry1.delete(0, 'end')
	self.entry2.delete(0, 'end')
	if value is not None:
	    if type(value)==ListType and len(value)==2:
	        self.entry1.insert(1, value[0])
	        self.entry1.xview('end')
	        self.entry2.insert(1, value[1])
	        self.entry2.xview('end')



class CheckEntryField(Field):
    def __init__(self, name='OptionEntry', text='Label', checked=1, **kw):
	Field.__init__(self, name, text=text, checked=checked, **kw)

    def buildfield(self, master, line):
	self.option= IntVar()
	check= Checkbutton(master, text=self.text, variable=self.option, command=self.setoption)
	check.grid(row=line, column=self.shift+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=line, column=self.shift+1, sticky='ew')

	self.setoption(self.checked)

	return 1

    def get(self):
	opt= self.option.get()
	if opt:
	    return string.strip(self.entry.get())
	else:
	    return None

    def setoption(self, value=None):
	if value is None:
	    value= self.option.get()
	else:
	    self.option.set(value)
	if value:
	    self.entry.configure(state='normal')
	else:
	    self.entry.configure(state='disabled')

    def set(self, value=None):
	self.entry.configure(state='normal')
	self.entry.delete(0, 'end')
	if value is not None:
	    self.entry.insert(1, value)
	    self.entry.xview('end')
	    self.setoption(1)
	else:
	    self.setoption(0)

	
class LinkedCheckEntryField(Field):
    def __init__(self, name='LinkedCheckField', text=[], checked=0, **kw):
	Field.__init__(self, name, text=text, checked=checked, **kw)
	self.values= []
	self.entries= []

    def buildfield(self, master, line):
	nb= 0
	for idx in range(len(self.text)):
	    self.values.append(IntVar())
	    check= Checkbutton(master, text=self.text[idx], variable=self.values[idx],
			command= lambda this=self,idx=idx: LinkedCheckEntryField.onclick(this, idx))
	    check.grid(row=line+nb, column=self.shift+0, sticky='w')

	    entry= Entry(master, state='disabled')
	    entry.grid(row=line+nb, column=self.shift+1, sticky='ew')
	    self.entries.append(entry)
	    nb+= 1
	return nb

    def onclick(self, index):
	value= self.values[index].get()
	if value:
	    self.entries[index].configure(state='normal')
	    for idx in range(index):
		self.values[idx].set(1)
		self.entries[idx].configure(state='normal')
	else:
	    for idx in range(index, len(self.values)):
		self.values[idx].set(0)
		self.entries[idx].configure(state='disabled')
	    
    def get(self):
	result= []
	for (val, ent) in zip(self.values, self.entries):
	    if val.get():
	        result.append(string.strip(ent.get()))
	return result

    def set(self, value=None):
	for val in self.values:
	    val.set(0)
	for ent in self.entries:
	    ent.configure(state='normal')
	    ent.delete(0, 'end')
	if value is not None:
	    for idx in range(len(value)):
	        self.values[idx].set(1)
		self.entries[idx].insert(1, value[idx])
		self.entries[idx].xview('end')
	    for idx in range(len(value), len(self.entries)):
		self.entries[idx].configure(state='disabled')	


class FileInput(Field):
    """ Input File entry
    """
    def __init__(self, name='FileInput', text='Input File', types=[("all files", "*")], **kw):
	Field.__init__(self, name, text=text, types=types, **kw)

    def buildfield(self, master, line):
	label= Label(master, text=self.text)
	label.grid(row=line, column=self.shift+0, sticky='w')

	self.entry= Entry(master, state='normal')
	self.entry.grid(row=line, column=self.shift+1, sticky='ew')

	button= Button(master, text='...', width=1, command=self.askfile)
	button.grid(row=line, column=self.shift+2, sticky='w')
	
	return 1

    def get(self):
	return string.strip(self.entry.get()) 

    def askfile(self):
	file= tkFileDialog.askopenfilename(initialfile=self.get(),
				title=self.text, filetypes=self.types)
	if len(file):
	    self.set(file)

    def set(self, filename=None):
	self.entry.delete(0, 'end')
	if filename is not None:
	    self.entry.insert(1, filename)
	    self.entry.xview('end')

class FileOutput(FileInput):
    """ Output File entry
    """
    def askfile(self):
	file= tkFileDialog.asksaveasfilename(initialfile=self.get(),
				title=self.text, filetypes=self.types)
	if len(file):
	    self.set(file)

class DirInput(FileInput):
    """ Input directory entry (any directory with read permission)
    """
    def askfile(self):
	dir= askdirectory(initialdir=self.get(),title=self.text,mustexist=1)
	if len(dir):
	    self.set(dir)

class DirOutput(FileInput):
    """ Input directory entry (any directory with write permission)
    """
    def askfile(self):
	dir= askdirectory(initialdir=self.get(),title=self.text,mustexist=0)
	if len(dir):
	    self.set(dir)

class LinkedCheckField(Field):
    def __init__(self, name='LinkedCheckField', text=[], **kw):
	Field.__init__(self, name, text=text, **kw)
	self.values= []

    def buildfield(self, master, line):
	nb= 0
	for idx in range(len(self.text)):
	    self.values.append(IntVar())
	    check= Checkbutton(master, text=self.text[idx], variable=self.values[idx],
			command= lambda this=self,idx=idx: LinkedCheckField.onclick(this, idx))
	    check.grid(row=line+nb, column=self.shift+0, sticky='w')
	    nb+= 1
	return nb

    def onclick(self, index):
	value= self.values[index].get()
	if value:
	    for idx in range(index):
		self.values[idx].set(1)
	else:
	    for idx in range(index, len(self.values)):
		self.values[idx].set(0)
	    
    def get(self):
	sum= 0
	for value in self.values:
	    sum+= value.get()
	return sum

    def set(self, value=None):
	for val in self.values:
	    val.set(0)
	if value is not None:
	    for idx in range(value):
	        self.values[idx].set(1)

class CheckField(Field):
    """ Flag option to check
    """
    def __init__(self, name='CheckField', text='Label', **kw):
	Field.__init__(self, name, text=text, **kw)

    def buildfield(self, master, line):
	self.value= IntVar()
	check= Checkbutton(master, text=self.text, variable=self.value)

	check.grid(row=line, column=self.shift+0, columnspan=2, sticky='w')

	return 1

    def get(self):
	return self.value.get()
	
    def set(self, value=None):
	if value is not None:
	    self.value.set(value)
	else:
	    self.value.set(0)


class RadioField(Field):
    """ List of radio buttons linked together with an optionnal text header
    """
    def __init__(self, name='RadioField', text=None, values=[], default=0, **kw):
	Field.__init__(self, name, text=text, values=values, default=default, **kw)

    def buildfield(self, master, line):
	i= 0
	if self.text:
	    label= Label(master, text=self.text)
	    label.grid(row=line+i, column=self.shift+0, columnspan=MAXCOL, sticky='w')
	    i+=1
	self.selected= IntVar()
	nb= 0
	for text in self.values:
	    radio= Radiobutton(master, text=text, value=nb, variable=self.selected)
	    radio.grid(row=line+i, column=self.shift+0, sticky='w')
	    nb+=1
	    i+=1
	return i

    def get(self):
	return self.selected.get()

    def set(self, value=None):
	if value is not None:
	    self.selected.set(value)
	else:
	    self.selected.set(0)


class SelectField(Field):
    def __init__(self, name='SelectField', text='label', values=[], **kw):
	Field.__init__(self, name, text=text, values=values, **kw)

    def buildfield(self, master, line):
	label= Label(master, text=self.text)
	label.grid(row=line, column=self.shift+0, sticky='w')

	self.select= Selector.Selector(master, self.values)
	self.select.grid(row=line, column=self.shift+1, sticky='w')

	return 1

    def get(self):
	return self.values.index(self.select.value.get())

    def set(self, value=None):
	if value is not None:
	    self.select.show_item(self.values[value])
	else:
	    self.select.show_item(self.values[0])

#
# Null field class (no output)
#
class NullField:
    """ Class to define a null field (which return any parameter).
	The only method to overwrite is buildfield (same as class Field)
    """
    def __init__(self, **kw):
	for key in kw.keys():
	    setattr(self, key, kw[key])
	self.name= None

    def build(self, master, line):
	return self.buildfield(master, line)

    def check(self):
	pass

    def buildfield(self, master, line):
	return 0

    def setdefault(self):
	pass

    def set(self, value):
	pass

    def getdict(self):
	return {}

#
# Derivated null class field
#	
class TextField(NullField):
    """ Simple text label (no entry) to add comment/information
    """
    def __init__(self, text='My text', shift=0):
	NullField.__init__(self, text=text)
	self.shift= shift

    def buildfield(self, master, line):
	col= 0
	if type(self.text)==StringType:
	    Label(master, text=self.text).grid(row=line, column=self.shift+col, columnspan=MAXCOL, sticky='w')
	if type(self.text)==ListType:
	    for text in self.text:
		lb= Label(master, text=text)
		lb.grid(row=line, column=self.shift+col, sticky='w')
		col+=1
	return 1


class SpaceField(NullField):
    """ Add an empty line
    """
    def __init__(self, line=1):
	NullField.__init__(self, line=line)

    def buildfield(self, master, line):
	for i in range(self.line):
	    Label(master).grid(row=line+i, columnspan=MAXCOL)
	return self.line


class SeparatorField(NullField):
    """ Add a separator
    """
    def __init__(self, height=2, padding=2):
	NullField.__init__(self, height=height, padding=padding)

    def buildfield(self, master, line):
	pad1= Frame(master, height=self.padding)
	pad2= Frame(master, height=self.padding)
	sep= Frame(master, bg='black', height=self.height)
	pad1.grid(row=line, column=0, columnspan=MAXCOL, sticky='ewns')
	sep.grid(row=line+1, column=0, columnspan=MAXCOL, sticky='ewns')
	pad2.grid(row=line+2, column=0, columnspan=MAXCOL, sticky='ewns')
	return 3
	

class TitleField(NullField):
    def __init__(self, text=None, relief='raised', bg='blue', fg='white'):
	NullField.__init__(self, text=text, relief=relief, bg=bg, fg=fg)

    def buildfield(self, master, line):
	frame= Label(master, text=self.text, relief=self.relief, bg=self.bg, fg=self.fg)
	frame.grid(row=line, columnspan=MAXCOL, sticky='ew')
	return 1


#
# Validate classes
#
class InvalidField(exceptions.Exception):
    """ Exception that should be raised in a Validate class
    """
    def __init__(self, text=None):
	self.text= text

class Validate:
    """ Base class for validator. Overwrite method check
    """
    def __init__(self, error=None, **kw):
	self.error= error
	for key in kw.keys():
	    setattr(self, key, kw[key])

    def senderror(self, name, value, message):
	""" Raise an error
	"""
	raise InvalidField, '%s: <%s> %s.'%(name, value, message)

    def check(self, name, value):
	pass


class IsFloat(Validate):
    def check(self, name, value):
	try:
	    val= float(value)
	except:
	    self.senderror(name, value, 'should be a number')

class IsPositive(Validate):
    def check(self, name, value):
        print "Checking"
        IsFloat().check(name, value)
        if float(value)<0.0:
            self.senderror(name, value, 'should be a positive number')

class IsNegative(Validate):
    def check(self, name, value):
	IsFloat().check(name, value)
	if float(value)>0.0:
	    self.senderror(name, value, 'should be a negative number')

class MinMax(Validate):
    def __init__(self, min=None, max=None, **kw):
	Validate.__init__(self, min=min, max=max, **kw)

    def check(self, name, value):
	IsFloat().check(name, value)
	val= float(value)
	if self.min is not None:
	    if val<self.min:
		self.senderror(name, value, 'should be lower than %f'%self.min)
	if self.max is not None:
	    if val>self.max:
		self.senderror(name, value, 'should be greater than %f'%self.min)
	
class NonBlank(Validate):
    def check(self, name, value):
	if len(value)==0:
	    self.senderror(name, value, 'should be non-blank')

#
# sheet classes
#
class Sheet:
    """ Class to manage all fields in one sheet
    """
    def __init__(self, notetitle=None, fields=()):
	self.fields= fields
	self.notetitle= notetitle

    def build(self, master):
	self.frame= Frame(master, bd=2, relief='ridge')
	line= 1
	for field in self.fields:
	    line += field.build(self.frame, line)
	    
    def get(self):
	res= {}
	for field in self.fields:
	    res.update(field.getdict())
	return res

    def validate(self):
	errors= []
	for field in self.fields:
	    try:
		field.check()
	    except InvalidField, error:
		errors.append(error.text)
	return errors

    def set(self, newdict):
    	allkey= newdict.keys()
	for field in self.fields:
	    if field.name in allkey:
		field.set(newdict[field.name])

    def reset(self):
	for field in self.fields:
	    field.set(None)

    def setdefault(self, dicdefault=None):
	if dicdefault is None:
	    for field in self.fields:
	        field.setdefault()
	else:
	    allkey= dicdefault.keys()
	    for field in self.fields:
		if field.name in allkey:
		    field.setdefault(dicdefault[field.name])


class SheetFrame(Frame):
    def __init__(self, root, type='single', sheets=(), default=None, init=None):
	Frame.__init__(self, root)

	self.sheets= sheets
	self.type= type
	self.default= default
	self.init= init

	if self.type=='notebook':
	    self.__build_note()
	elif self.type=='horizontal':
	    self.__build_horizontal()
	else:
	    self.__build_vertical()

	if self.default:
	    self.setdefault(self.default)
	if self.init:
	    self.set(self.init)
	    

    def __build_note(self):
	""" build one sheet per notebook page
	"""
	self.notebook= Blt.Tabset(self,relief='flat',tiers=3,tearoff=0,bd=0,name='notebook') 
	self.notebook.pack(fill='both',expand='yes')
	idx= 1
	for sheet in self.sheets:
	    sheet.build(self.notebook)
	    title= sheet.notetitle or 'tab %d'%idx
	    self.notebook.insert(idx, title, window=sheet.frame)
	    idx += 1

    def __build_vertical(self):
	""" build all the sheets in the same frame vertically
	"""
	for sheet in self.sheets:
	    sheet.build(self)
	    sheet.frame.pack(fill='both', expand='yes',side='top')

    def __build_horizontal(self):
	""" build all the sheets in the same frame horizontally
	"""
	self.mainframe= Frame(self)
	for sheet in self.sheets:
	    sheet.build(self.mainframe)
	    sheet.frame.pack(side='left')
	self.mainframe.pack(side='top')

    def check(self):
	errors= []
	for sheet in self.sheets:
	    errors+= sheet.validate()
	return errors

    def get(self):
	result= {}
	for sheet in self.sheets:
	    result.update(sheet.get())
	return result

    def set(self, newdict):
	for sheet in self.sheets:
	    sheet.set(newdict)

    def setdefault(self, dicdefault=None):
	for sheet in self.sheets:
	    sheet.setdefault(dicdefault)
	    

    def reset(self):
	for sheet in self.sheets:
	    sheet.reset()


class SheetDialog(tkSimpleDialog.Dialog):
    def __init__(self, parent, title=None, sheets=(), type='single', default=None, init=None, validate=1):
	#if not parent:
	#    parent= Tk()
	self.sheets= sheets
	self.type= type
	self.default= default
	self.init= init
	self.valid= validate
	self.wtitle= title
        self.result = None	
	tkSimpleDialog.Dialog.__init__(self, parent, title)

    def body(self,master):
	self.bodywid= SheetFrame(master, type=self.type, sheets=self.sheets, 
					default=self.default, init=self.init)
	self.bodywid.pack(expand='yes',fill='both')

    def buttonbox(self):
	sep = Frame(self,name='dialogseparator',bd=2)
	box = Frame(self,name='dialogbuttonbox')

	self.okb     = Button(box,command=self.ok,text='OK',name='okbutton')
	self.cancelb = Button(box,command=self.cancel,text='cancel',name='cancelbutton')
	self.defaultb= Button(box,command=self.setdefault,text='default',name='defaultbutton')
	self.resetb  = Button(box,command=self.reset,text='reset',name='resetbutton')

	self.bind('<Return>',self.ok)
	self.bind('<Escape>',self.cancel)
	
	self.resetb.pack(side='left')
	self.defaultb.pack(side='left')
	self.cancelb.pack(side='left')
	self.okb.pack(side='left')
	sep.pack(fill='x',pady=3)
	box.pack(anchor='center')

    def validate(self):
	if self.valid:
	    errors= self.bodywid.check()
	    if len(errors)>0:
	        self.showerror(errors)
	        return 0
	    else:
		return 1
	else:
	    return 1

    def apply(self):
	#print self.bodywid.get()
        self.result = self.bodywid.get()

    def reset(self):
	self.bodywid.reset()

    def setdefault(self):
	self.bodywid.setdefault()
	
    def showerror(self, errors):
	text= ''
	for line in errors:
	    text+= '%s\n' % line
	if self.wtitle is not None:
	    title= '%s - ERROR'%self.wtitle
	else:
	    title= 'ERROR'
	tkMessageBox.showerror(title, text)



def testframe():
    sheet3= Sheet(notetitle='first', 
		  fields=(EntryField('text', 'Your name', default='manu', validate=IsPositive()),
			  FileInput('infile', 'Input file', default='infile.dat'),
			  FileOutput('outfile', 'Output file', default='outfile.dat')))
    sheet4= Sheet(notetitle='second',
	   	  fields=(EntryField('text2', 'Your name', default='manu'),
			  FileInput('infile2', 'Input file', default='infile.dat'),
			  FileOutput('outfile2', 'Output file', default='outfile.dat')))
    app = Tk()
    app.title('Test SheetFrame class')
    frm= SheetFrame(app, sheets=(sheet3,sheet4), type='notebook')
    frm.pack()
    frm.set( {'text': 'Emmanuel', 'text2': 'Papillon'} )
    app.mainloop()


def testdialog():
    sheet1= Sheet(notetitle='Entry/Text/Check',
	       fields=(	TitleField('First title'),
			EntryField('ent1', 'An entry field'),
			EntryField('ent2', 'With end text', endtext='???'),
			SeparatorField(),
			TextField('Single text field'),
			TextField('The same with a shift', shift=1),
			TextField(['col1', 'col2', 'col3', 'col4']),
			SeparatorField(),
			CheckField('flag', 'Check it !!'),
			SpaceField(),
			CheckEntryField('Checkent1', 'Check and entry mixed'),
			CheckEntryField('Checkent2', 'With a shift ...', shift=1)
		))

    sheet2= Sheet(notetitle='File/Dir',
		fields=(TitleField('File input / output'),
			FileInput('infile', 'Input File'),
			FileOutput('outfile', 'Output File'),
			TitleField('Directory input / output'),
			DirInput('indir', 'Input directory'),
			DirOutput('outdir', 'Output directory'),
			SeparatorField(),
			EntryField('ent3', 'Or use a simple entry'),
		))

    sheet3= Sheet(notetitle='Linked Field',
		fields=(TextField('RadioField'),
			SeparatorField(padding=6, height=6),
			RadioField('radio', values=['Choose this one','or choose this one']),
			SpaceField(),
			TextField('LinkedCheckField'),
			SeparatorField(),
			LinkedCheckField('linkcheck', ['first choice',
							'second possible only if first checked',
							'last possible only if first and second checked'],
						shift=1),
			SpaceField(),
			TextField('LinkedCheckEntryField'),
			SeparatorField(),
			LinkedCheckEntryField('linkchecktext',
						['the same', 'as previous', 'with entry'])
			))
			
    sheet4= Sheet(notetitle='Others', 
		fields=(TitleField('SelectField'),
			SelectField('host','Hostname', ['expgi', 'expgj', 'expglin']),
			SpaceField(),
			TitleField('DoubleEntryField'),
			TextField([None, 'From', 'To']),
			DoubleEntryField('dbent_X', 'X range'),
			DoubleEntryField('dbent_Y', 'Y range')
		))

			
			
			
    asheet1= Sheet(notetitle='first',
		  fields=(TitleField(text='First sheet'),
			  EntryField('text', 'Precision level', endtext='%', default='50', validate=(IsPositive(), IsFloat(min=0, max=100))),
			  EntryField('infile', 'Input file', default='infile.dat', validate=IsPositive()),
			  SeparatorField(),
			  EntryField('outfile', 'Output file', default='outfile.dat', validate=IsPositive()),
			  CheckEntryField('option', 'Anything', default='something'),
			  LinkedCheckField('checklink', ['premier', 'second', 'troisieme'], default=2),
			  LinkedCheckEntryField('textlink', ['premier', 'second', 'troisieme'], default=['first'])))
    asheet2= Sheet(fields=(TitleField(text='Second sheet'),
			  DirInput('indir', 'Input directory'),
			  DirOutput('outdir', 'Output directory', default='la'),
			  SpaceField(),
			  TextField('Following is different options:'),
			  TextField([None, 'col1', 'col2']),
			  CheckField('force', 'Force writing', default=0),
			  CheckField('ext', 'Add automatic extension', default=1),
			  SeparatorField(padding=6, height=3),
			  RadioField('choice', 'linked radio buttons', ['choose this one','or choose this one']),
			  SelectField('host', 'Hostname', ['expgi', 'expgj', 'expglin'], default=2)))
    root= Tk()
    root.withdraw()
    app= SheetDialog(root, title='Option Sheet',
	             sheets= (sheet1, sheet2, sheet3, sheet4),
		     type='notebook', validate=0, init={'ent1': 75, 'infile':'toto_in.dat'},
			default={'ent1':50, 'infile':'default.dat'})
    print app.result
if __name__=="__main__":
    #testframe()
    testdialog()

