
from qt import *
from qttable import *
import types, os.path
import EventHandler
from resources import fileopen,fileclose

class ScanList(QWidget):

	allSelectionMode= {'single': QListView.Single,
                        'multiple': QListView.Multi,
                        'extended': QListView.Extended
                        }

	allSelectionType= ["mca", "scan", "both"]

	def __init__(self, parent=None, name=None, fl=0, selection="single", type="both"):
		QWidget.__init__(self, parent, name, fl)

		self.layout= QVBoxLayout(self)
		self.list= QListView(self, "ScanList")
		self.layout.addWidget(self.list)

		# --- font
		#self.setFont(QFont("Application", 10))

		# --- list headers
		self.list.addColumn("X")
		self.list.addColumn("S#")
		self.list.addColumn("Command")
		self.list.addColumn("Pts")
		self.list.addColumn("Mca")
		self.list.header().setResizeEnabled(0, 0)
		self.list.header().setResizeEnabled(0, 1)
		self.list.header().setResizeEnabled(1, 2)
		self.list.header().setResizeEnabled(0, 3)
		self.list.header().setResizeEnabled(0, 4)
		self.list.header().setClickEnabled(0,-1)
		self.list.setSorting(-1)

		# --- list selection options
		self.list.setAllColumnsShowFocus(1)
		self.setSelectionMode(selection)
		self.disableMca	= (type=="mca")
		self.disableScan= (type=="scan")

		# --- signal handling
		self.connect(self.list, SIGNAL("selectionChanged()"), self.__selectionChanged)
		self.connect(self.list, SIGNAL("contextMenuRequested(QListViewItem *, const QPoint &, int)"), self.__contextMenu)

		# --- context menu
		self.menu= QPopupMenu(self.list)
		id= self.menu.insertItem("Show scan header")
		self.menu.connectItem(id, self.__showScanInfo)

		self.data= None
		self.scans= []

	#
	# Widget options
	#
	def setSelectionMode(self, selection):
		if self.allSelectionMode.has_key(selection):
			self.list.setSelectionMode(self.allSelectionMode[selection])

	def setFont(self, qtfont):
		self.list.setFont(qtfont)

	# 
	# Data management
	#
	def setData(self, specfiledata):
		self.data= specfiledata
		self.refresh()

	def refresh(self):
		self.list.clear()
		if self.data is None or self.data.SourceName is None:	return
		info= self.data.GetSourceInfo()
		self.scans= []
		after= None
		for (sn, cmd, pts, mca) in zip(info["KeyList"], info["Commands"], info["NumPts"], info["NumMca"]):
			if after is not None:
				item= QListViewItem(self.list, after, "", sn, cmd, str(pts), str(mca))
			else:
				item= QListViewItem(self.list, "", sn, cmd, str(pts), str(mca))
			if (self.disableMca and not mca) or (self.disableScan and not pts):
				item.setSelectable(0)
				#XXX: not possible to put in italic: other solutions ??
			self.list.insertItem(item)
			self.scans.append(sn)
			after= item

	def clear(self):
		self.list.clear()
		self.data= None
		self.scans= []

	def markScanSelected(self, scanlist):
		for sn in self.scans:
			item= self.list.findItem(sn, 1)
			if item is not None:
				if sn in scanlist:
					item.setText(0, "X")
				else:
					item.setText(0, "")

	#
	# signal/slot handling
	#
	def __selectionChanged(self):
		sel= [sn for sn in self.scans if self.list.findItem(sn,1).isSelected()]
		self.emit(PYSIGNAL("scanSelection"), (sel,))

	def __contextMenu(self, item, point, col):
		if item is not None:
			sn= str(item.text(1))
			self.menu.setItemParameter(self.menu.idAt(0), self.scans.index(sn))
			self.menu.popup(point)

	def __showScanInfo(self, idx):
		print "Scan information:"
		print self.data.GetSourceInfo(self.scans[idx])


class McaTable(QWidget):
	def __init__(self, parent=None, name=None, fl=0):
		QWidget.__init__(self, parent, name, fl)

		self.mainLayout= QVBoxLayout(self)

		self.table= QTable(self)
		self.table.setSelectionMode(QTable.Multi)
		self.table.setFocusStyle(QTable.FollowStyle)
		self.table.setReadOnly(1)
		self.table.verticalHeader().setResizeEnabled(0, -1)
		self.table.horizontalHeader().setResizeEnabled(0, -1)

		# --- selection in case of a single MCA by scan
		self.firstMca= QTableSelection()
		self.firstMca.init(0, 0)
		self.firstMca.expandTo(0, 0)

		self.mainLayout.addWidget(self.table)

		#self.connect(self.table, SIGNAL("selectionChanged()"), self.__selectionChanged)
		# XXX: do not work correctly anymore !!! (after Qt upgrade)
		self.connect(self.table, SIGNAL("doubleClicked(int,int,int,const QPoint&)"), self.__doubleClicked)
		self.reset()

	#
	# data management
	#
	def setData(self, specfiledata):
		self.data= specfiledata
		self.reset()

	def setScan(self, scankey):
		if type(scankey)==types.ListType: 
			if len(scankey): scankey= scankey[0]
			else: scankey=None
		if scankey is None:
			self.reset()
		else:
			self.info= self.data.GetSourceInfo(scankey)
			self.refresh()

	def markMcaSelected(self, mcalist):
		for row in range(self.pts):
			for col in range(self.det):
				if str(self.__getMcaNo(row, col)) in mcalist:
					self.table.setText(row, col, "X")
				else:
					self.table.setText(row, col, "")
			
	def selectMcaList(self, mcalist):
		for mcano in mcalist:
			self.selectMcaNo(mcano)

	def selectMcaNo(self, mcano):
		(row, col)= self.__getRowCol(mcano)
		selection= QTableSelection()
		selection.init(row, col)
		selection.expandTo(row, col)
		self.table.addSelection(selection)
		#self.__selectionChanged()

	def selectFirstMca(self):
		self.table.addSelection(self.firstMca)
		#self.__selectionChanged()

	def refresh(self):
		mca= self.info["NbMca"] or 0
		pts= self.info["Lines"] or 0

		if mca==0: 
			self.reset()
		else:
			# --- try to compute number of detectors
			if pts>0 and mca%pts==0:
				self.det= mca/pts
				self.pts= pts
			else:
				self.det= mca
				self.pts= 1

			self.table.setNumCols(self.det)
			self.table.setNumRows(self.pts)
			Hheader= self.table.horizontalHeader()
			for idx in range(self.det):
				Hheader.setLabel(idx, "mca.%d"%(idx+1), -1)
				self.table.adjustColumn(idx)

			for row in range(self.pts):
				for col in range(self.det):
					self.table.setText(row, col, "")

			#if self.info["Lines"]==pts:
			#	print "AddNorm", self.info["AllLabels"]

			if mca==1:
				self.selectFirstMca()

	def clear(self):
		self.table.clearSelection(1)

	def reset(self):
		self.table.clearSelection(1)
		self.table.setNumCols(1)
		self.table.setNumRows(0)
		self.table.horizontalHeader().setLabel(0, "NO MCA for the selected scan", -1)
		self.table.adjustColumn(0)
		self.det= 0
		self.pts= 0

	def getSelection(self):
		if self.det==0:	return []
		selection= []
		for row in range(self.pts):
			for col in range(self.det):
				if self.table.isSelected(row, col):
					selection.append("%s.%d"%(self.info["Key"], self.__getMcaNo(row,col)))
		return selection
	#
	# signal / slot handling
	#
	def __selectionChanged(self):
		if self.det==0: return
		else:
			self.emit(PYSIGNAL("mcaSelection"), (self.getSelection,))

	def __doubleClicked(self, *args):
		self.emit(PYSIGNAL("mcaDoubleClicked"), (["%s.%d"%(self.info["Key"], self.__getMcaNo(args[0], args[1]))],))

	#
	# (mcano) <--> (row,col) operations
	#
	def __getMcaNo(self, row, col):
		return (row*self.det+col+1)
		
	def __getRowCol(self, mcano):
		mcano= mcano-1
		row= int(mcano/self.det)
		col= mcano-(row*self.det)
		return (row, col)

class SpecFileSelector(QWidget):
	def __init__(self, parent=None, name="SpecFileSelector", fl=0):
		QWidget.__init__(self, parent, name, fl)

		self.data= None
		self.lastSelection= None
		self.currentScan= None
		self.selection= None

		mainLayout= QVBoxLayout(self)

		# --- event handler
		self.eh= EventHandler.EventHandler()
		self.addEvent= self.eh.create("addSelection")
		self.delEvent= self.eh.create("delSelection")
		
		# --- file combo/open/close
		fileWidget= QWidget(self)
		self.fileCombo= QComboBox(fileWidget)
		self.fileCombo.setEditable(0)
		self.mapComboName= {}
		self.openIcon= QIconSet(QPixmap(fileopen))
		openButton= QToolButton(fileWidget)
		openButton.setIconSet(self.openIcon)
		openButton.setSizePolicy(QSizePolicy(QSizePolicy.Fixed, QSizePolicy.Minimum))
		self.closeIcon= QIconSet(QPixmap(fileclose))
		closeButton= QToolButton(fileWidget)
		closeButton.setIconSet(self.closeIcon)
		closeButton.setSizePolicy(QSizePolicy(QSizePolicy.Fixed, QSizePolicy.Minimum))

		fileLayout= QHBoxLayout(fileWidget)
		fileLayout.addWidget(self.fileCombo)
		fileLayout.addWidget(openButton)
		fileLayout.addWidget(closeButton)

		self.connect(openButton, SIGNAL("clicked()"), self.openFile)
		self.connect(closeButton, SIGNAL("clicked()"), self.closeFile)
		self.connect(self.fileCombo, SIGNAL("activated(const QString &)"), self.__fileSelection)

		# --- splitter with scan list/mca list
		splitter= QSplitter(self)
		splitter.setOrientation(QSplitter.Vertical)

		self.scanList= ScanList(splitter, selection="single", type="mca")
		self.mcaTable= McaTable(splitter)

		self.connect(self.scanList, PYSIGNAL("scanSelection"), self.__scanSelection)
		self.connect(self.mcaTable, PYSIGNAL("mcaDoubleClicked"), self.__mcaDoubleClicked)

		# --- select / remove buttons
		butWidget	= QWidget(self)
                #self.directView	= QCheckBox("direct view", butWidget)
		removeButton	= QPushButton("Remove", butWidget)
                selectButton	= QPushButton("Show", butWidget)

                butLayout= QHBoxLayout(butWidget)
                #butLayout.addWidget(self.directView)
		butLayout.addWidget(removeButton)
                butLayout.addWidget(selectButton)

		self.connect(selectButton, SIGNAL("clicked()"), self.__selectClicked)
		self.connect(removeButton, SIGNAL("clicked()"), self.__removeClicked)

		mainLayout.addWidget(fileWidget)
		mainLayout.addWidget(splitter)
		mainLayout.addWidget(butWidget)

	def openFile(self, filename=None):
		if filename is None:
			filename= QFileDialog.getOpenFileName(None, "Specfiles (*.dat)\nAll files (*)", self,
							"openFile", "Open a new SpecFile")
			filename= str(filename)
			if not len(filename):	return
	
		if filename in self.mapComboName.keys():
			self.selectFile(filename)
		else:
			if not self.data.SetSource(filename):
				QMessageBox.critical(self, "ERROR opening Specfile",
						"Cannot open following specfile:\n%s"%(filename))
			else:
				filename= self.data.SourceName
				self.mapComboName[filename]= os.path.basename(filename)
				self.fileCombo.insertItem(self.mapComboName[filename])
				self.selectFile(filename)

	def selectFile(self, filename=None):
		if filename is not None:
		    if str(self.fileCombo.currentText())!=self.mapComboName[filename]:
			for idx in range(self.fileCombo.count()):
				if str(self.fileCombo.text(idx))==self.mapComboName[filename]:
					self.fileCombo.setCurrentItem(idx)
					break
		    self.data.SetSource(filename)
		self.refresh()

	def closeFile(self, filename=None):
		if filename is None:
			file= str(self.fileCombo.currentText())
			for filename, comboname in self.mapComboName.items():
				if comboname==file: break

		if self.selection is not None and self.selection.has_key(filename):
			mcakeys= []
			for scan in self.selection[filename].keys():
				mcakeys += [ "%s.%s"%(scan, mca) for mca in self.selection[filename][scan] ]
			if len(mcakeys):
				msg= "%d mca are linked to that Specfile source.\n"%len(mcakeys)
				msg+="Do you really want to delete all these graphs ??"
				ans= QMessageBox.information(self, "Remove SpecFile %s"%filename, msg,
						QMessageBox.No, QMessageBox.Yes)
				if ans==QMessageBox.No: return
				self.emit(PYSIGNAL("delSelection"), (self.data.SourceName, mcakeys))
			
		for idx in range(self.fileCombo.count()):
			if str(self.fileCombo.text(idx))==self.mapComboName[filename]:
				#if idx==self.fileCombo.currentItem():
				self.fileCombo.removeItem(idx)
				del self.mapComboName[filename]
				break

		if not self.fileCombo.count():
			self.selectFile()
		else:
			self.selectFile(self.mapComboName.keys()[0])

	def __fileSelection(self, file):
		file= str(file)
		for filename, comboname in self.mapComboName.items():
			if comboname==file:
				self.selectFile(filename)
				break

	"""	
	def setFileList(self, filelist, fileselected=None):
		self.fileCombo.clear()
		for file in filelist:
			self.fileCombo.insertItem(file)
		if fileselected is None: fileselected= filelist[0]
		self.selectFile(fileselected)
	"""

	def setData(self, specfiledata):
		self.data= specfiledata
		self.scanList.setData(specfiledata)
		self.mcaTable.setData(specfiledata)

	def setSelected(self, sellist):
		self.selection= {}
		for sel in sellist:
			filename= sel["SourceName"]
			scan, order, mca= sel["Key"].split(".")
			scankey= "%s.%s"%(scan,order)
			if not self.selection.has_key(filename):
				self.selection[filename]= {}
			if not self.selection[filename].has_key(scankey):
				self.selection[filename][scankey]= []
			self.selection[filename][scankey].append(mca)
		self.__refreshSelection()

	def OLDsetSelection(self, seldict={}):
		self.selection= {}
		for filename in seldict.keys():
			self.selection[filename]= {}
			for key in seldict[filename]:
				scan, order, mca= key.split(".")
				scankey= "%s.%s"%(scan, order)
				if not self.selection[filename].has_key(scankey):
					self.selection[filename][scankey]= []
				self.selection[filename][scankey].append(mca)
		self.__refreshSelection()
	
	def __refreshSelection(self):
		if self.selection is not None:
			sel= self.selection.get(self.data.SourceName, {})
			self.scanList.markScanSelected(sel.keys())
			self.mcaTable.markMcaSelected(sel.get(self.currentScan, []))
		
	def refresh(self):
		self.scanList.refresh()
		self.mcaTable.reset()
		self.__refreshSelection()
	
	def __scanSelection(self, scankey):
		self.mcaTable.setScan(scankey)
		if len(scankey):
			self.currentScan= scankey[0]
			if self.selection is not None and self.selection.has_key(self.data.SourceName):
				self.mcaTable.markMcaSelected(self.selection[self.data.SourceName].get(self.currentScan, []))
		else:
			self.currentScan= None

	def __mcaSelection(self, mcakeys):
		self.lastSelection= mcakeys
		#if self.directView.isChecked():
		#	self.emit(PYSIGNAL("tempSelection"), (mcakeys,))

	def __mcaDoubleClicked(self, mcakeys):
		sel= {"SourceName":self.data.SourceName, "Key":mcakeys}
		self.eh.event(self.addEvent, [sel])

	def __OLDmcaDoubleClicked(self, mcakeys):
		self.emit(PYSIGNAL("newSelection"), (mcakeys,))

	def __selectClicked(self):
		selection= []
		for mca in self.mcaTable.getSelection():
			selection.append({"SourceName":self.data.SourceName,"Key":mca})
		if len(selection):
			self.eh.event(self.addEvent, selection)
		
	def __OLDselectClicked(self):
		selection= self.mcaTable.getSelection()
		if (len(selection)):
			self.emit(PYSIGNAL("newSelection"), (self.data.SourceName, selection,))

	def __removeClicked(self):
		selection= []
		for mca in self.mcaTable.getSelection():
			selection.append({"SourceName":self.data.SourceName, "Key":mca})
		if (len(selection)):
			self.eh.event(self.delEvent, selection)

	def __OLDremoveClicked(self):
		selection= self.mcaTable.getSelection()
		if (len(selection)):
			self.emit(PYSIGNAL("delSelection"), (self.data.SourceName, selection,))

def test():
	import sys
	import SpecFileData

	def tempSelection(*args):
		print "tempSelection", args

	def newSelection(*args):
		print "newSelection", args

	if not len(sys.argv)>1:
		print "USAGE: %s <specfile>"%sys.argv[0]
		sys.exit(0)

	a = QApplication(sys.argv)
	QObject.connect(a, SIGNAL("lastWindowClosed()"),a,SLOT("quit()"))

	w = SpecFileSelector()
	QObject.connect(w, PYSIGNAL("tempSelection"), tempSelection)
	QObject.connect(w, PYSIGNAL("newSelection"), newSelection)
	a.setMainWidget(w)

	d = SpecFileData.SpecFileData()
	w.setData(d)
	
	w.show()
	a.exec_loop()

if __name__=="__main__":
	test()
