from qt import *
from GraphWindow import *
from EventHandler import EventHandler
from McaFilterInfo import McaFilterInfoTable

from McaDataSource import SourceClass,SourceMenu,McaDataSource

from GraphInfo import GraphInfo

import os
import mdi
import resources

PrintTempFile=".pymca_tmp.ps"

class PyMcaWindow(mdi.ApplicationWindow):
	def __init__(self, parent=None, name="PyMcaWindow", fl=Qt.WDestructiveClose):
		mdi.ApplicationWindow.__init__(self, parent, name, fl)
		self.statusBar().hide()

		self.setupDockSource()
		self.setupFilterInfo()
		self.setupSource()

		self.setupGraph()

	def initToolBar(self):
		self.toolbar= QToolBar(self,'file operations')

		fileOpen = QToolButton(self.openIcon,'Open File',qt.QString.null, 
				self.onOpenFile,self.toolbar,'open file')
		QWhatsThis.add(fileOpen,resources.fileOpenText)
		specOpen = QToolButton(self.specIcon,'Open Spec Array',qt.QString.null, 
				self.onOpenSpec,self.toolbar,'open spec array')
		QWhatsThis.add(specOpen,resources.specOpenText)
		fileSave = QToolButton(self.saveIcon,'Save',qt.QString.null,
					self.onSave,self.toolbar,'save file')
		QWhatsThis.add(fileSave,resources.fileSaveText)
		filePrint = QToolButton(self.printIcon,'Print Graph',qt.QString.null,
					self.onPrint,self.toolbar,'print graph')
		QWhatsThis.add(filePrint,resources.filePrintText)
		self.toolbar.addSeparator()

		newScreen	= QToolButton(self.newIcon, "New Graph", qt.QString.null,
					self.onNewGraph, self.toolbar, "new graph")
		fullScreen	= QToolButton(self.fullIcon, "Full Screen", qt.QString.null,
					self.onFullScreen, self.toolbar, "full screen")
		tileScreen	= QToolButton(self.tileIcon, "Tile Graphs", qt.QString.null,
					self.onMdiTile, self.toolbar, "tile windows")
		self.toolbar.addSeparator()
		QWhatsThis.whatsThisButton(self.toolbar)
		
	def initIcons(self):
		self.openIcon	= QIconSet(QPixmap(resources.fileopen))
		self.specIcon	= QIconSet(QPixmap(resources.spec))
		self.saveIcon 	= QIconSet(QPixmap(resources.filesave))
		self.printIcon 	= QIconSet(QPixmap(resources.fileprint))
		self.newIcon	= QIconSet(QPixmap(resources.window_new))
		self.fullIcon	= QIconSet(QPixmap(resources.window_fullscreen))
		self.tileIcon	= QIconSet(QPixmap(resources.window_nofullscreen))
		self.specIcon	= QIconSet(QPixmap(resources.spec))

		QMimeSourceFactory.defaultFactory().setPixmap('fileopen',qt.QPixmap(resources.fileopen))
		QMimeSourceFactory.defaultFactory().setPixmap('spec',qt.QPixmap(resources.spec))

	def initMenuBar(self):
		self.menuFile= QPopupMenu(self)
		self.menuBar().insertItem('&File',self.menuFile)

		id= self.menuFile.insertItem(QIconSet(self.openIcon),'Open &File',self.onOpenFile,Qt.CTRL+Qt.Key_F)
		self.menuFile.setWhatsThis(id,resources.fileOpenText)
		id= self.menuFile.insertItem(QIconSet(self.specIcon),'&Open Spec',self.onOpenSpec,Qt.CTRL+Qt.Key_O)
		self.menuFile.setWhatsThis(id,resources.specOpenText)

		id= self.menuFile.insertItem(QIconSet(self.saveIcon),'&Save',self.onSave,Qt.CTRL+Qt.Key_S)
		self.menuFile.setWhatsThis(id,resources.fileSaveText)
		self.menuFile.insertSeparator()
		id= self.menuFile.insertItem(QIconSet(self.printIcon),'&Print Graph',self.onPrint,Qt.CTRL + Qt.Key_P)
		self.menuFile.setWhatsThis(id,resources.filePrintText)
		self.menuFile.insertSeparator()
		self.menuFile.insertItem('&Quit',qApp,SLOT('closeAllWindows()'),Qt.CTRL + Qt.Key_Q)

		self.menuSource= QPopupMenu()
		self.menuSource.setCheckable(1)
		self.menuBar().insertItem('&Tools', self.menuSource)
		self.menuSourceIndex= {}

		for (type,param) in SourceMenu.items():
			idx= self.menuSource.insertItem(param[0], self.onSource, param[1])
			self.menuSourceIndex[idx]= type
		self.menuSource.insertSeparator()
		idx= self.menuSource.insertItem('&Graph Browser', self.onSelectedSource, Qt.CTRL+Qt.Key_G)
		self.menuSourceSelectedIndex= idx

		self.menuGraph= QPopupMenu()
		self.menuGraph.setCheckable(1)
		self.connect(self.menuGraph, SIGNAL("aboutToShow()"), self.slotGraphMenuAboutToShow)
		self.menuBar().insertItem("&Graph", self.menuGraph)

		self.menuHelp= QPopupMenu()
		self.menuBar().insertSeparator()
		self.menuBar().insertItem('&Help',self.menuHelp)
		self.menuHelp.insertItem('&About',self.onAbout,Qt.Key_F1)
		self.menuHelp.insertItem('About &Qt',self.aboutQt)

	def slotGraphMenuAboutToShow(self):
		self.menuGraph.clear()
		self.menuGraph.insertItem('&New Graph',self.onNewGraph,Qt.CTRL+Qt.Key_N)
		#self.menuGraph.insertItem('&Show Graph Data', self.onGraphData,Qt.CTRL+Qt.Key_S)

		if len(self.mdi.windowList())==0: return

		self.menuGraph.insertSeparator()

		self.menuGraph.insertItem('&Cascade',self.onMdiCascade)
		self.menuGraph.insertItem('&Tile',self.onMdiTile)
		self.menuGraph.insertItem('&Tile Horizontally',self.tilehoriz)
		self.menuGraph.insertItem('&Tile Vertically',self.tilevertic)
		self.menuGraph.insertSeparator()

		self.graphMenuMap= {}
		for window in self.mdi.windowList():
			index= self.menuGraph.insertItem(str(window.caption()), self.slotGraphMenuActivated)
			self.graphMenuMap[index]= window
			if self.mdi.activeWindow()==window:
				self.menuGraph.setItemChecked(index, 1)
	
	def slotGraphMenuActivated(self, index):
		self.graphMenuMap[index].setFocus()

	def onMdiCascade(self):
		self.disconnect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onGraphActivated)
		self.cascade()
		self.connect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onGraphActivated)

	def onMdiTile(self):
		self.disconnect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onGraphActivated)
		self.mdi.tile()
		self.connect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onGraphActivated)

	def onFullScreen(self):
		self.mdi.activeWindow().showMaximized()

	def setupDockSource(self):
		self.sourceDock	= QDockWindow(QDockWindow.InDock, self.leftDock(), "Sources")
		self.sourceStack= QWidgetStack(self.sourceDock)
		self.sourceDock.setWidget(self.sourceStack)
		self.sourceDock.setResizeEnabled(1)
		self.sourceDock.setCloseMode(QDockWindow.Always)
		self.addDockWindow(self.sourceDock, Qt.DockLeft)
		self.sourceDock.hide()
		self.connect(self.sourceDock, SIGNAL("visibilityChanged(bool)"), self.onSourceDockHide)

	def onSourceDockHide(self, b):
		if not b:
			for idx in self.menuSourceIndex.keys():
				self.menuSource.setItemChecked(idx, 0)

	def setupDockTool(self):
		self.toolDock= QDockWindow(QDockWindow.InDock, self.rightDock(), "Graph Tools")
		self.toolDock.setResizeEnable(1)
		self.toolDock.setCloseMode(QDockWindow.Always)
		self.addDockWindow(self.toolDock, Qt.RightDock)
		self.toolDock.hide()

	def setupGraph(self):
		self.connect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onGraphActivated)
		self.onNewGraph()

	def onGraphActivated(self, wid):
		if wid is not None:
			info= self.filterInfoWidget.currentPage()
			if info.name!=wid.name:
				idx= self.getFilterInfoIdx(wid.name)
				self.filterInfoWidget.setCurrentPage(idx)

	def setupFilterInfo(self):
		self.filterInfoDock= QDockWindow(QDockWindow.InDock, self.bottomDock(), "PLotList")
		self.filterInfoDock.setResizeEnabled(1)
		self.filterInfoDock.setCloseMode(QDockWindow.Always)
		self.filterInfoWidget= QTabWidget(self.filterInfoDock)
		self.filterInfoDock.setWidget(self.filterInfoWidget)
		self.addDockWindow(self.filterInfoDock, Qt.DockBottom)
		self.filterInfoDock.hide()
		self.connect(self.filterInfoDock, SIGNAL("visibilityChanged(bool)"), self.onFilterInfoDockHide)
		self.connect(self.filterInfoWidget, SIGNAL("currentChanged(QWidget*)"), self.onFilterInfoActivated)

	def onFilterInfoActivated(self, wid):
		graph= self.getActiveGraph()
		if graph is not None:
			if wid.name!=graph.name:
				graph= self.getGraph(wid.name)
				graph.setFocus()
		else:
			graph= self.getGraph(wid.name)
			graph.setFocus()

	def getFilterInfoIdx(self, name):
		for idx in range(self.filterInfoWidget.count()):
			wid= self.filterInfoWidget.page(idx)
			if wid.name==name:
				return idx
		return None

	def onFilterInfoDockHide(self, b):
		if not b:
			self.menuSource.setItemChecked(self.menuSourceSelectedIndex, 0)

	def updateFilterInfo(self):
		names= self.__getGraphNames()
		for idx in range(self.filterInfoWidget.count()):
			self.filterInfoWidget.page(idx).setGraphNames(names)
	
	#
	# McaDataSource operations
	#	
	def showSource(self, type):
		wid= self.sourceData[type].getSelectionWidget(self.sourceStack)
		if wid!=self.sourceStack.visibleWidget():
			self.sourceStack.raiseWidget(wid)
		if self.sourceDock.isHidden():
			self.sourceDock.show()
		for (idx,val) in self.menuSourceIndex.items():
			self.menuSource.setItemChecked(idx, (val==type))
		return wid

	def setupSource(self):
		self.sourceEH	= EventHandler()

		self.sourceData	= {}
		for (type,param) in SourceClass.items():
			self.sourceData[type]= McaDataSource(param[0], param[1], param[2], param[3])
			self.sourceData[type].eh.register("addDataEvent", self.addDataEventCB)
			#self.sourceData[type].eh.register("delDataEvent", self.delDataEventCB)

	def selectSource(self, type, selkeys):
		failed= self.sourceData[type].addSelection(selkeys)
		if len(failed):
			str= "Failed to load %s sources:\n"%type
			for sel in selkeys:
				str= str+"%s:%s\n"%(sel["SourceName"], sel["Key"])
			QMessageBox.critical(self, "ERROR loading source", str)

	#
	# GraphWindow operations
	#
	def getActiveGraph(self):
		return self.mdi.activeWindow()

	def getGraph(self, name):
		for graph in self.mdi.windowList():
			if str(graph.caption())==name:
				return graph
		return None
		
	def onNewGraph(self):
		# --- disconnect from mdi to avoid multi-signals
		self.disconnect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onGraphActivated)

		# --- create new graph
		name = self.__getNewGraphName()
		graph= GraphWindow(self.mdi, name=name)
		self.connect(graph, PYSIGNAL("GraphClosed"), self.closeGraph)
		graph.show()

		# --- create its graph info widget
		info= GraphInfo(self, name, graph.view)
		info.setAutoRefresh(1)

		self.filterInfoWidget.addTab(info, name)
		self.filterInfoWidget.setCurrentPage(self.filterInfoWidget.indexOf(info))

		info.eh.register("addFilterEvent", self.addFilterCB)
		info.eh.register("delFilterEvent", self.delFilterCB)

		self.updateFilterInfo()

		# --- tile the graph to have a correct desktop
		self.mdi.tile()
		self.connect(self.mdi, SIGNAL("windowActivated(QWidget*)"), self.onGraphActivated)

		return graph

	def closeGraph(self, graph, ce):
		# --- remove associated plot list
		name= str(graph.caption())
		for idx in range(self.filterInfoWidget.count()):
			if str(self.filterInfoWidget.label(idx))==name:
				w= self.filterInfoWidget.page(idx)
				self.filterInfoWidget.removePage(w)
				w.close()
		# --- remove data
		filters= graph.view.GetSource()
		self.delFilterCB(filters)
		
		# --- remove graph
		graph.destroy(ce)

	#def onGraphData(self):
		#self.openGraphData()

	#def openGraphData(self, gw=None):
		#if gw is None:
			#gw= self.mdi.activeWindow()
		#gname= str(gw.caption())
		#dname= "Data %d"%(int(gname.split()[1]))
		#dview= DataWindow(self.mdi, name=dname)
		#dview.view.ConnectTo(gw.view)
		#dview.show()
		#self.onMdiTile()

	def __getGraphNames(self):
		return [ str(window.caption()) for window in self.mdi.windowList() ]

	def __getNewGraphName(self):
		names= self.__getGraphNames()
		idx= 0
		while "Graph %d"%idx in names:
			idx += 1
		return "Graph %d"%(idx)

	#
	# Callback from Source menu
	#
	def onSource(self, menuidx):
		type= self.menuSourceIndex[menuidx]
		if self.menuSource.isItemChecked(menuidx):
			self.sourceDock.hide()
			self.menuSource.setItemChecked(menuidx, 0)
		else:
			self.showSource(type)

	def onSelectedSource(self, idx=None):
		if idx is not None and self.menuSource.isItemChecked(idx):
			self.filterInfoDock.hide()
			self.menuSource.setItemChecked(idx, 0)
		else: 
			if self.filterInfoDock.isHidden():
				self.filterInfoDock.show()
				self.menuSource.setItemChecked(idx, 1)

	#
	# CB from DataSource to GraphView	
	#
	def addDataEventCB(self, data_selections):
		graph= self.getActiveGraph()
		if graph is None:
			graph= self.onNewGraph()
		graph.view.AddDataSelection(data_selections)

	#
	# CB from McaFilterInfo table to GraphView
	#
	def addFilterCB(self, filters, graphname):
		""" Add filters to graph <graphname>
		"""
		print "addFilterCB", filters, graphname
		if graphname=="New":
			graph= self.onNewGraph()
		else:
			graph= self.getGraph(graphname)
		if graph is not None:
			graph.view.AddFilter(filters)

	#
	# CB from McaFilterInfo table to DataSource
	#
	def delFilterCB(self, filters):
		""" Remove filters from graph <graphname>
		    Extract DataSelection objects from filters
		    and remove them from Data objects
		"""
		print "delFilterCB", filters
		datasel= [ filter.GetDataSelection() for filter in filters ]
		for source in self.sourceData.values():
			source.delDataSelection(datasel)

	#
	# Callback from File menu
	#
	def onOpenFile(self):
		wid= self.showSource("SpecFile")
		wid.openFile()
		# --- to be replaced by openSource

	def onOpenSpec(self):
		self.showSource("SPS")

	def onSave(self):
		print "onSave"

	def onPrint(self):
		if self.printer.setup():
			window= self.mdi.activeWindow()
			window.view.Save(PrintTempFile,"PS")
			printername= str(self.printer.printerName())
			if sys.platform!= "win32":
                    		os.system("lp -c -d " + printername + " " + PrintTempFile)
			os.remove(PrintTempFile)

	def onAbout(self):
		QMessageBox.about(self, 'PyMCA', 'MCA visualization tool')

	def aboutQt(self):
		QMessageBox.aboutQt(self,'PyMCA')

if __name__=='__main__':
	import sys
	a = mdi.MDIApp(sys.argv)
	w = PyMcaWindow()
	a.setMainWidget(w)
	#w.resize(400,300)
	w.show()
	a.connect(a, SIGNAL('lastWindowClosed()'), a, SLOT('quit()'))
	a.exec_loop()
