from qt import *

from PyDVT.GraphView import *
from ViewSelectList import *
from McaGraphSelect import *
import Command
import EventHandler


# Pen, Color, Style values
# ------------------------
PenStyles= [ "solid", "dashed", "dotted", "hidden" ]
PenColors= [ 
	["black",	(0, 0, 0)],
	["red",		(255, 0, 0)],
	["blue",	(0, 0, 255)],
	["green",	(0, 255, 0)],
	["magenta",	(255, 0, 255)],
	["cyan",	(0, 255, 255)],
	["yellow",	(255, 255, 0)],
	["darkRed",	(139, 0, 0)],
	["darkBlue", 	(0, 0, 139)],
	["darkGreen",	(0, 100, 0)],
	["darkMagenta",	(139, 0, 139)],
	["darkCyan",	(0, 139, 139)],
	["darkYellow",	(0, 0, 0)]
]
YScales = ["Left", "Right"]

# Graph Settings values
# ---------------------
GraphStyle= ["Line", "Bars", "Points", "PointsLine"]
GraphCursor= ["Vertical", "Crosshairs", "None"]
GraphMode= ["Zoom", "Roi", "Peak", "Normal"]

def getColorName(color):
	""" Return name of a color given its RGB tuple
	"""
	for c in PenColors:
		if color==c[1]:
			return c[0]
	return "other"

def getColorRgb(name):
	""" Return RGB tuple for a color given its name
	"""
	for c in PenColors:
		if name==c[0]:
			return c[1]
	return PenColors[0][1]


class GraphPen:
	""" Class to auto-increment Pen color
	"""
	def __init__(self, width=1, style="solid"):
		self.width= width
		self.style= style
		self.reset()

	def reset(self):
		self.ColorIndex	= -1

	def get(self):
		self.ColorIndex += 1
		if self.ColorIndex == len(PenColors):
			self.ColorIndex	= 0
		return Pen(PenColors[self.ColorIndex][1], self.width, self.style)


# -------------------------------------------------------------------------
# McaGraphFilter: GraphFilter with normalization, calibration, scale factor
# -------------------------------------------------------------------------
class McaGraphFilter(GraphFilter):
	"""
	Method to retrieve filter attributes:
		GetInfo()
	Methods to change filter attributes:
		SetYFactor(yfactor)
		SetActiveNorm(norm)
		SetActiveCalib(calibname)
		SetCalib(name, coeff)
	Graphic attributes (pen, symbol, yscale) should be changed through View methods:
		View.SetPen, View.SetSymbol, View.SetYScale
		graph will be redrawn without calling filter's GetOutput method.
	"""
	def __init__(self,name=None,source=None,synchronized=1,buffer_input=0,yscale=0,yfactor=1, \
				calib=None,pen=Pen((0,0,0),2,"solid"),symbol="none"):
		Filter.__init__(self, source, synchronized, buffer_input)
		if name is None: self.name= str(self)
		else: self.name= name
		self.pen= pen
		self.symbol= symbol
		self.yscale= yscale
		self.yfactor= yfactor
		self.calib_type= CalibDefaultType
		self.calib= McaGraphFilterCalib(self)
		self.calib.Set("Energy", "Default", (0, 2, 0))
		if calib is not None:
			self.calib.Update(calib)
		self.norm= {}
		self.active_norm= None
		self.peaks= None
		self.show_peaks= 1
		try:
			self.legend= self.Source.GetDataSelection().GetInfo()["Legend"]
		except:
			self.legend= "-"

	def GetOutput(self):
		sel= self.GetInput()
		ret= {"name": self.name}

		# --- Y axis data
		if "data" not in sel.keys(): return ret
		if sel["data"] is None: return ret
		ydata= sel["data"].flat

		# --- scale factor
		if self.yfactor!=1: ydata= ydata*self.yfactor

		# --- normalization counter
		if sel.has_key("norm"):
			self.norm= sel["norm"]
			if self.active_norm in sel["norm"].keys():
				if self.norm[self.active_norm]!=0:
					ydata= ydata/self.norm[self.active_norm]
		else:
			self.norm= {}

		# --- X axis data
		if sel.has_key("xdata"): xchan= sel["xdata"]
		else: xchan= Numeric.arrayrange(ydata.shape[0])

		# --- calibration
		coeff= self.calib.Get(self.calib_type)
		if coeff is None:
			# calib not defined: return no data
			return ret
		elif coeff==CalibDefaultCoeff:
			xdata= xchan
		else:
			xdata= coeff[0] + coeff[1]*xchan + coeff[2]*(xchan*xchan)

		# --- peaks
		if self.show_peaks and self.peaks is not None:
			peaks= {}
			for (name, coord) in self.peaks.items():
				x= coord[0]
				try: y= ydata[int(x)]
				except: y= None
				peaks[name]= (x, y)
			ret["peaks"]= peaks

		ret.update({ "name":self.name, "data":ydata, "xdata":xdata, 
				"yscale":self.yscale, "pen":self.pen, "symbol":self.symbol})
		return ret

	def GetInfo(self):
		""" Return all filter attributes but not the data itself.
		    To get filter's data call GetOutput()
		"""
		ret= { "name":self.name, "pen":self.pen, "symbol":self.symbol, "yscale":self.yscale,
			"yfactor":self.yfactor, "norm":self.norm, "active_norm":self.active_norm,
			"calib":self.calib, "calib_type":self.calib_type }
		return ret

	# --- Pen attributes
	def GetPen(self):
		return self.pen

	def SetPen(self, pen):
		self.pen= pen

	# --- Y scale factor
	def GetYFactor(self):
		return self.yfactor

	def SetYFactor(self, yfactor=1):
		""" Define a scale factor.
		    Refresh is called if <refresh>=1 and if needed.
		"""
		if yfactor!=self.yfactor:
			self.yfactor= yfactor

	# --- counter normalization
	def GetActiveNorm(self):
		return self.active_norm

	def SetActiveNorm(self, norm=None):
		""" Define normalization counter. if <norm> is None, or <norm> not in known counters,
		    no normalization counter is set.
		"""
		if norm is not None:
			if norm not in self.norm.keys():
				norm= None
		if norm!=self.active_norm:
			self.active_norm= norm

	# --- calibration
	def SetCalibType(self, type):
		self.calib_type= type

	def GetCalibType(self):
		return self.calib_type

	def GetCalib(self):
		return self.calib

	def SetCalib(self, calib):
		self.calib= calib

	# --- peaks
	def ShowPeaks(self, show=1):
		self.show_peaks= show
		
	def SetPeaks(self, peaks=None):
		""" peaks= { "pk_name": (x, ymax) }
		"""
		self.peaks= peaks

	def GetPeaks(self):
		return self.peaks

	def AddPeak(self, xpos, ymax=None, pkname=None):
		if self.peaks is None: self.peaks= {}
		if pkname is None:
			idx= 0
			while "pk%d"%idx in self.peaks.keys():
				idx+=1
			pkname= "pk%d"%idx
		self.peaks[pkname]= (xpos, ymax)

	# --- general info
	def GetType(self):
		try: type= self.Source.GetType()
		except: type= "MCA"
		return type

	def GetLegend(self):
		return self.legend


#TODO:
# class McaGraphFilterPeaks to hold peaks properties (elt, ray, name)

CalibTypes= ["Channels", "Energy"]
CalibDefaultType= "Channels"
CalibDefaultName= "Default"
CalibDefaultCoeff= (0, 1, 0)

class McaGraphFilterCalib:
	def __init__(self, parent):
		self.filter= parent
		self.param= {}
		for type in CalibTypes:
			if type==CalibDefaultType:
				self.param[type]= [CalibDefaultName, {CalibDefaultName: CalibDefaultCoeff}]
			else:
				self.param[type]= [None, {}]

	def GetParam(self):
		return self.param

	def Get(self, type=CalibDefaultType):
		calib= self.param[type]
		if calib[0] is not None:
			return calib[1][calib[0]]
		else:	return None

	def Set(self, type, name, coeff, active=1):
		calib= self.param[type]
		calib[1][name]= coeff
		if self.filter.GetCalib()==type and self.GetActive(type)==name:
			print "calib Refresh"
			self.filter.Refresh()
		if active: 
			self.SetActive(type, name)

	def SetActive(self, type, name=None):
		calib= self.param[type]
		if name in calib[1].keys() or name is None:
			if calib[0]!=name:
				calib[0]= name
				if self.filter.GetCalib()==type:
					print "calib Refresh"
					self.filter.Refresh()
		

	def GetActive(self, type):
		return self.param[type][0]

	def Remove(self, type, name):
		del self.param[type][1][name]
		if self.GetActive(type)==name:
			if len(self.param[type][1].keys()):
				self.SetActive(type, self.param[type][1].keys()[0])
			else:
				self.SetActive(type)

	def Update(self, caldict):
		print "Calib.Update: TODO"
		
# -----------------------------------------------
# McaGraphView: GraphView with peak/roi selection
# -----------------------------------------------
class McaGraphView(GraphView):

	def __init__(self, parent=None):
		self.SendSettingFlag= 0
		self.LogX=self.LogY= 0
		self.CursorType= "None"

		GraphView.__init__(self, parent, {"AddRefresh":0, "AddStyleSelect":0, "AddStatus":0})

		self.EvSourceChange	= self.eh.create("SourceChange")
		self.EvActiveChange	= self.eh.create("ActiveChange")
		self.EvCursorPosition	= self.eh.create("CursorPosition")
		self.EvSettingChange	= self.eh.create("SettingChange")

		self.GraphPen= GraphPen(1, "solid")

		self.ZoomPen= Pen((0,0,0),1,"solid")
		self.ZoomBrush= Brush((240,240,240),"fill_100")

		self.RoiList= RoiViewSelectList(self)
		self.RoiPen= Pen((0,0,0),1,"solid")
		self.RoiBrush= Brush((240,240,240),"fill_100")

		self.Peaks= {}
		self.PeakPen= Pen((0,0,0),1,"solid")

		self.ActiveFilter= None
		self.FilterNames= []

		self.CurrentCalib= CalibDefaultType

		self.FlagLockPosition= 1
		self.SendSettingFlag= 1

		self.SetLabels(None, self.CurrentCalib, "Counts")
		
	def CreateDrawable(self):
		GraphView.CreateDrawable(self)
		#self.EnableLegend(1,"bottom")
	#
	# SetSource
	#
	def SetSource(self, source=()):
		GraphView.SetSource(self, source)

	def DataChanged(self, source=None):
		""" DataChanged methods includes:
			GraphView.DataChanged
		plus a call to CheckActiveFilter to update active filter
		and a call to PeakChanged to update peaks drawing
		"""
		print "DataChanged", source
		if len(self.Source):
			for filter in self.Source:
				filter.SetCalibType(self.CurrentCalib)
		GraphView.DataChanged(self, source)
		if not len(self.Source): self.SetMode("Normal")
		self.CheckActiveFilter()
		self.PeakChanged()
		return

	def Invalidate(self):
		""" GraphView.Invalidate base class method
		plus send a \"SourceChange\" event if a filter has been added/removed
		"""
		GraphView.Invalidate(self)
		names= self.Functions.keys()
		if names!=self.FilterNames:
			self.FilterNames= names
			self.eh.event(self.EvSourceChange, self.GetSource())

	def PeakChanged(self):
		# --- create / update PeakViewSelect
		for (name, data) in self.Functions.items():
			if not data.has_key("peaks"):
				if self.Peaks.has_key(name):
					print "Peaks", name, "Destroyed"
					self.Peaks[name].Destroy()
			else:
				if not self.Peaks.has_key(name):
					filter= self.GetFilter(name)
					self.Peaks[name]= PeakViewSelect(filter)
					self.Peaks[name].ConnectView(self)
				self.Peaks[name].SetPen(data["pen"])
				self.Peaks[name].SetSelection(data["peaks"])

		# --- cleanup destroyed PeakViewSelect	
		for (name, peak) in self.Peaks.items():
			if not self.Functions.has_key(name):
				peak.Destroy()
			if not peak.Alive:
				del self.Peaks[name]

	#
	# Get filter from its name
	#
	def GetFilter(self, name):
		""" Return filter object or None if not found
		"""
		for filter in self.GetSource():
			if filter.name==name:
				return filter
		return None

	#
	# Add/Remove Source methods
	#
	def AddDataSelection(self, selections):
		""" Arguments: tuple of DataSelection
		Create one McaGraphFilter for each DataSelection, changing pen for each
		"""
		filters= ()
		for sel in selections:
			filters += ( McaGraphFilter(source=sel, pen=self.GraphPen.get(), buffer_input=1), )
		self.AddFilter(filters)

	def AddFilter(self, filters):
		""" Arguments: tuple of McaGraphFilter
		Add all the filters on the graph
		"""
		self.SetSource(self.GetSource()+filters)

	def RemoveFilter(self, names):
		""" Arguments: tuple of McaGraphFilter names
		Remove McaGraphFilter from graph.
		"""
		old= self.GetSource()
		new= ()
		for filter in old:
			if filter.name not in names:
				new += (filter,)
		if len(old)!=len(new):
			self.SetSource(new)

	#
	# Calibration
	#
	def GetCalib(self):
		return self.CurrentCalib

	def SetCalib(self, calib):
		if calib not in CalibTypes: return
		else: self.CurrentCalib= calib
		self.SetLabels(None, self.CurrentCalib, "Counts")
		self.Refresh()

	#
	# Roi utils
	#
	def getRoiList(self):
		return self.RoiList
	#
	# Graph Settings
	#
	def GetSettings(self):
		""" Get dictionnary of current graph settings
		"""
		set= {}
		set["Mode"]= self.CurrentMode
		set["LogX"]= self.LogX
		set["LogY"]= self.LogY
		set["GridLines"]= self.Drawable.Gridlines
		set["Lock"]= self.FlagLockPosition
		set["OnLine"]= self.OnLine
		set["Cursor"]= self.CursorType
		set["Style"]= self.Drawable.style
		return set

	def SetSettings(self, setdict, sendevent=0):
		""" Change graph settings using given dictionnary
		If sendevent=1, \"SettingChange\" events will be fired.
		By default no events are sent.
		"""
		if not sendevent:
			self.SendSettingFlag= 0
		if setdict.has_key("Mode"):
			self.SetMode(setdict["mode"])
		if setdict.has_key("Style"):
			self.SetStyle(setdict["Style"])
		if setdict.has_key("Cursor"):
			self.SetCursorType(setdict["Cursor"])
		if setdict.has_key("LogX"):
			self.SetLogX(setdict["LogX"])
		if setdict.has_key("LogY"):
			self.SetLogY(setdict["LogY"])
		if setdict.has_key("GridLines"):
			if setdict["GridLines"]!=self.Drawable.GridLines: self.ToggleGridlines()
		if setdict.has_key("Lock"):
			if setdict["Lock"]!=self.FlagLockPosition: self.ToggleLockPosition()
		if setdict.has_key("OnLine"):
			if setdict["OnLine"]!=self.OnLine: self.ToggleOnLine()
		if not sendevent:
			self.SendSettingFlag= 1

	def SendSetting(self, param, value):
		""" Depending on SettingFlag, send \"SettingChange\" event
		with setting name and value
		"""
		if self.SendSettingFlag:
			self.eh.event(self.EvSettingChange, param, value)


	#
	# Source Graphic attributes
	#
	# (SetPen, SetSymbol exist in GraphView but does not handle case of no change (newpen==oldpen))
	#
	def SetYScale(self, name, yscale):
		yscale= (yscale==1)
		for sel in self.Source:
			if sel.name==name:
				if sel.yscale!=yscale:
					sel.yscale= yscale
					self.Drawable.SetFunctionItem(name, "yscale", yscale)
					self.Refresh()
					#XXX: why Update() does not work ??
				return

	def SetPen(self, name, pen):
		for sel in self.Source:
			if sel.name==name:
				if sel.pen!=pen:
					sel.pen= pen
					self.Drawable.SetFunctionItem(name, "pen", pen)
					self.Update()
				return

	def SetSymbol(self, name, symbol):
		for sel in self.Source:
			if sel.name==name:
				if sel.symbol!=symbol:
					sel.symbol= symbol
					self.Drawable.SetFunctionItem(name, "symbol", symbol)
					self.Update()
				return
	
	#
	# Mode Management
	#
	def CreateMenu(self):
		self.ClearMenu()

		self.ModeMenuItem= {}
		self.CommandMenuItem= {}
		self.CurrentMode= "Unset"

		self.ModeMenuItem["Normal"]= self.AddMenuPopupItem("Normal mode",
						Command.Command(self.SetMode, mode="Normal"), "checkbutton")
		self.ModeMenuItem["Zoom"]= self.AddMenuPopupItem("Zoom mode", 
						Command.Command(self.SetMode, mode="Zoom"), "checkbutton")
		self.ModeMenuItem["Roi"]= self.AddMenuPopupItem("Roi mode", 
						Command.Command(self.SetMode, mode="Roi"), "checkbutton")
		self.ModeMenuItem["Peak"]= self.AddMenuPopupItem("Peak mode", 
						Command.Command(self.SetMode, mode="Peak"), "checkbutton")
		self.AddMenuSeparator()
		self.SetMode("Unset")

	def GetMode(self):
		return self.CurrentMode

	def SetMode(self, mode):
		if mode!="Unset" and not len(self.Source): 
			self.SendSetting("Mode", self.CurrentMode)
		else:
			if mode==self.CurrentMode: mode= "Normal"
			self.DisableMode(self.CurrentMode)
			for key in self.ModeMenuItem.keys():
				self.MenuPopup.CheckItem(self.ModeMenuItem[key], (key==mode))
			for key in self.CommandMenuItem.keys():
				if key[0:4]!="Menu": self.DeleteMenuItem(self.CommandMenuItem[key])
			self.CommandMenuItem= {}
			self.EnableMode(mode)
			self.CustomizeMode(mode)
			self.SendSetting("Mode", mode)
			self.ResetPointer()

	def ResetPointer(self):
		if self.CurrentMode == "Normal":
			self.SetPointer("arrow")
		else:
			self.SetPointer("cross")

	def DisableMode(self, mode):
		if mode=="Zoom":	self.DisableZoomMode()
		elif mode=="Roi":	self.DisableRoiMode()
		elif mode=="Peak":	self.DisablePeakMode()
		else:			self.DisableNormalMode()
	
	def EnableMode(self, mode):
		if mode=="Zoom":	self.EnableZoomMode()
		elif mode=="Roi":	self.EnableRoiMode()
		elif mode=="Peak":	self.EnablePeakMode()
		else:			self.EnableNormalMode()
		self.CurrentMode= mode

	def CustomizeMode(self, mode):
		pass

	def _EraseSelect(self):
		if self.Select is not None:
			self.Select.Erase(self)

	def _DestroySelect(self):
		if self.Select is not None:
			self.Select.Destroy()
			self.Select= None

	#
	# Normal MODE
	#
	def EnableNormalMode(self):
		self.CommandMenuItem["Refresh"]= self.AddMenuPopupItem("Refresh", self.Refresh)

		popup= Menu(self.MenuPopup)
		self.CommandMenuItem["MenuLogX"]= popup.AddCommand("X Log scale", self.ToggleLogX, "checkbutton")
		self.MenuPopup.CheckItem(self.CommandMenuItem["MenuLogX"], self.LogX)
		self.CommandMenuItem["MenuLogY"]= popup.AddCommand("Y Log scale", self.ToggleLogY, "checkbutton")
		self.MenuPopup.CheckItem(self.CommandMenuItem["MenuLogY"], self.LogY)
		popup.AddSeparator()
		self.CommandMenuItem["MenuGridlines"]= popup.AddCommand("Show Gridlines", self.ToggleGridlines, "checkbutton")
		self.MenuPopup.CheckItem(self.CommandMenuItem["MenuGridlines"], self.Drawable.Gridlines)
		self.CommandMenuItem["Axis"]= self.AddMenuPopupCascade("Axis", popup)

		if not hasattr(self,"cursor_popup_menu"):
			popup= Menu(self.MenuPopup)
			popup.AddCommand("Vertical", Command.Command(self.SetCursorType, cursortype="Vertical"), "radiobutton")
			popup.AddCommand("Crosshairs", Command.Command(self.SetCursorType, cursortype="Crosshairs"), "radiobutton")
			popup.AddCommand("None", Command.Command(self.SetCursorType, cursortype="None"), "radiobutton")
			self.cursor_popup_menu= popup
			popup.SetCheckedRadio(self.CursorType)
		self.CommandMenuItem["Cursor"]= self.AddMenuPopupCascade("Cursor", self.cursor_popup_menu)

		if not hasattr(self, "style_popup_menu"):
			popup= Menu(self.MenuPopup)
			popup.AddCommand("Line", Command.Command(self.SetStyle, style="Line"), "radiobutton")
			popup.AddCommand("Bars", Command.Command(self.SetStyle, style="Bars"), "radiobutton")
			popup.AddCommand("Points", Command.Command(self.SetStyle, style="Points"), "radiobutton")
			popup.AddCommand("Line+Points", Command.Command(self.SetStyle, style="PointsLine"), "radiobutton")
			self.style_popup_menu= popup
			popup.SetCheckedRadio(self.Drawable.style)
		self.CommandMenuItem["Style"]= self.AddMenuPopupCascade("Style", self.style_popup_menu)

		self.CommandMenuItem["OnLine"]= self.AddMenuPopupItem("OnLine",self.ToggleOnLine,'checkbutton')
		self.MenuPopup.CheckItem(self.CommandMenuItem["OnLine"], self.OnLine)
		self.CommandMenuItem["Lock"]= self.AddMenuPopupItem("LockPosition",self.ToggleLock,'checkbutton')
		self.MenuPopup.CheckItem(self.CommandMenuItem["Lock"], self.FlagLockPosition)

	def DisableNormalMode(self):
		pass

	def SetStyle(self, style):
		self.Drawable.SetStyle(style)
		self.SendSetting("Style", style)

	def SetCursorType(self, cursortype):
		GraphView.SetCursorType(self, cursortype)
		self.CursorType= cursortype
		self.SendSetting("Cursor", cursortype)

	def ToggleOnLine(self):
		self.OnLine= not self.OnLine
		if self.OnLine: self.Refresh()
		if self.CommandMenuItem.has_key("OnLine"):
			self.MenuPopup.CheckItem(self.CommandMenuItem["OnLine"], self.OnLine)
		self.SendSetting("OnLine", self.OnLine)

	def SetOnLine(self, online):
		if online!=self.OnLine:
			self.ToggleOnLine()

	def ToggleLock(self):
		self.LockPosition(not self.FlagLockPosition)
		if self.CommandMenuItem.has_key("Lock"):
			self.MenuPopup.CheckItem(self.CommandMenuItem["Lock"], self.FlagLockPosition)
		self.SendSetting("Lock", self.FlagLockPosition)

	def SetLock(self, lock):
		if lock!=self.FlagLockPosition:
			self.ToggleLock()

	def ToggleGridlines(self):
		self.Drawable.ToggleGridlines()
		self.SendSetting("GridLines", self.Drawable.Gridlines)
		if self.CommandMenuItem.has_key("MenuGridlines"):
			self.MenuPopup.CheckItem(self.CommandMenuItem["MenuGridlines"], self.Drawable.Gridlines)

	def SetGridlines(self, show=0):
		if show!=self.Drawable.Gridlines:
			self.ToggleGridlines()
			
	def SetLogX(self, logx=0):
		if logx!=self.LogX:
			self.ToggleLogX()

	def ToggleLogX(self):
		self.LogX= not self.LogX
		if self.LogX: self.SetXScaleLog()
		else:         self.SetXScaleLinear()
		self.Update()
		if self.CommandMenuItem.has_key("MenuLogX"):
			self.MenuPopup.CheckItem(self.CommandMenuItem["MenuLogX"], self.LogX)
		self.SendSetting("LogX", self.LogX)

	def SetLogY(self, logy=0):
		if logy!=self.LogY:
			self.ToggleLogY()

	def ToggleLogY(self):
		self.LogY= not self.LogY
		if self.LogY: self.SetYScaleLog()
		else:         self.SetYScaleLinear()
		self.Update()
		if self.CommandMenuItem.has_key("MenuLogY"):
			self.MenuPopup.CheckItem(self.CommandMenuItem["MenuLogY"], self.LogY)
		self.SendSetting("LogY", self.LogY)

	def SetYScaleLog(self):
		""" Overwrite GraphView.SetYScaleLog to change minimum value to 1 
		"""
		self.Drawable.SetLogY(1, 1)
		
	#
	# Zoom MODE
	#
	def EnableZoomMode(self):
		self.CommandMenuItem["Reset"]= self.AddMenuPopupItem("Reset Zoom",self.Drawable.ResetZoom)
		#self.CommandMenuItem["ZoomIn"]= self.AddMenuPopupItem("Zoom inside graph", self.ToggleZoomIn, "checkbutton")
		#self.CommandMenuItem["ZoomOut"]= self.AddMenuPopupItem("Zoom in new graph", self.ToggleZoomOut, "checkbutton")

		self.Select= GraphViewSelect.GraphViewSelectRect(self.Source[0],self._EventZoomSelection)
		self.Select.SetBrush(self.ZoomBrush)
		self.Select.SetPen(self.ZoomPen)
		self.Select.ConnectView(self)

	def DisableZoomMode(self):
		self._DestroySelect()

	def _EventZoomSelection(self,source):
		rect=source.GetSelection()["BoundingRect"]
		self.Select.Erase()
		self.Drawable.SetZoom(rect[0],rect[1])

	def ResetZoom(self):
		self.Drawable.ResetZoom()

	def ToggleZoomIn(self):
		""" Zoom inside current Graph
		"""
		pass

	def ToggleZoomOut(self):
		""" Zoom in a new separate graph
		"""
		pass

	#
	# Roi MODE
	#
	def EnableRoiMode(self):
		self.CommandMenuItem["DelAll"]= self.AddMenuPopupItem("Delete All",
						Command.Command(self.RoiList.Delete, index=None))
		self.CommandMenuItem["DelActive"]= self.AddMenuPopupItem("Delete Active ROI", 
						Command.Command(self.RoiList.Delete, index=-1))
		self.CommandMenuItem["Table"]= self.AddMenuPopupItem("Show ROI Table", self.ShowRoiTable)
		self.CommandMenuItem["Hide"]= self.AddMenuPopupItem("Hide all ROIs", 
						Command.Command(self.RoiList.Hide, index=None))
		self.CommandMenuItem["Show"]= self.AddMenuPopupItem("Show all ROIs", 
						Command.Command(self.RoiList.Show, index=None))
		self.__CreateRoiSelect()

	def __CreateRoiSelect(self):	
		self.Select= GraphViewSelect.GraphViewSelectHRect(self.Source[0], self._EventRoiViewSelect)
		self.Select.SetPen(self.RoiPen)
		self.Select.SetBrush(self.RoiBrush)
		self.Select.ConnectView(self)

	def DeleteRoi(self, index=None):
		self.RoiList.Delete(index)

		if self.CurrentMode=="Roi":
			# --- current selection can be behind one roi !!
			self.Select.Destroy()
			self.__CreateRoiSelect()

		self.Update()

	def DisableRoiMode(self):
		self._DestroySelect()

	def _EventRoiViewSelect(self, source):
		rect= source.GetSelection()["BoundingRect"]
		self.Select.Erase(self)
		if rect[1][0]>rect[0][0]:
			self.RoiList.Add(rect)

	def ShowRoiTable(self):
		from RoiView import RoiLimitsDialog
		self.RoiDiag= RoiLimitsDialog()
		self.RoiDiag.SetSource(self.RoiList)
		self.RoiDiag.Show()

	#
	# Peak MODE
	#
	def EnablePeakMode(self):
		self.CommandMenuItem["Delete"]= self.AddMenuPopupItem("Remove Peaks", Command.Command(self.DeletePeaks))
		self.__CreatePeakSelect()

	def __CreatePeakSelect(self):
		self.Select= GraphViewSelect.GraphViewSelectVLine(self.Source[0], self._EventViewSelectPeak)
		self.Select.SetPen(self.PeakPen)
		self.Select.ConnectView(self)

	def DisablePeakMode(self):
		self._DestroySelect()

	def _EventViewSelectPeak(self, source):
		xpos= source.GetSelection()["BoundingRect"][0][0]
		#self.Select.Erase(self)
		self.Select.Destroy()

		filter= self.GetActiveFilter()
		if filter is not None:
			filter.AddPeak(xpos)
			filter.Refresh()

		self.__CreatePeakSelect()

	def DeletePeaks(self, name=None):
		if name is not None:
			filter= self.GetFilter(name)
		else:
			filter= self.GetActiveFilter()
		if self.CurrentMode=="Peak":
			self.Select.Destroy()
			self.__CreatePeakSelect()
		if filter is not None:
			filter.SetPeaks(None)
			filter.Refresh()

	def ShowPeakTable(self):
		print "ShowPeakTable"

	#
	# Current spectrum selection
	#
	def EventButtonPress(self, pos):
		if self.CurrentMode=="Normal" and len(self.Source)>1:
			name= self.GetClosestFunction(pos.DataCoord)
			self.SetActiveFilter(name)

	def CheckActiveFilter(self):
		if len(self.Source)==0:
			active= None
		elif len(self.Source)==1:
			active= self.Source[0].name
		else:
			if self.ActiveFilter not in self.Functions.keys():
				active= None
			else:
				active= self.ActiveFilter
		self.SetActiveFilter(active)

	def SetActiveFilter(self, name=None):
		if self.ActiveFilter!=name:
			if self.ActiveFilter is not None and self.Functions.has_key(self.ActiveFilter):
				pen= self.Functions[self.ActiveFilter]["pen"]
				if pen.width!=1:
					pen.width= 1
					self.Functions[self.ActiveFilter]["pen"]= pen
					self.Update()
			if name is not None:
				width= 1+(len(self.Source)>1)
				pen= self.Functions[name]["pen"]
				if pen.width!=width:
					pen.width= width
					self.Functions[name]["pen"]= pen
					self.Update()
			self.ActiveFilter= name
			self.eh.event(self.EvActiveChange, self.GetFilter(self.ActiveFilter))
		else:
			if self.ActiveFilter is not None:
				pen= self.Functions[name]["pen"]
				width= 1+(len(self.Source)>1)
				if pen.width!=width:
					pen.width= width
					self.Functions[name]["pen"]= pen
					self.Update()

	def GetActiveFilter(self):
		if self.ActiveFilter is None: return None
		else: return self.GetFilter(self.ActiveFilter)

	def EventPosition(self, source):
		sel=source.GetSelection()
		coord=sel["Position"].DataCoord
		type=source.GetType()
		if type=="Crosshairs" or type=="MousePosition":
			xpos= coord[0]
			ypos= coord[1]
		elif type=="VerticalCursor":
			xpos= coord[0]
			ypos= ()
			for (name,(x,y)) in self.GetPositionValues(coord):
				ypos+=(y,)
		self.eh.event(self.EvCursorPosition, xpos, ypos)


class RoiViewSelect(GraphViewSelect.GraphViewSelectHRect):
	def __init__(self):
		GraphViewSelect.GraphViewSelectHRect.__init__(self)
		self.Hidden= 0
		self.Active= 0

	def Update(self, *args):
		if not self.Hidden:
			GraphViewSelect.GraphViewSelectHRect.Update(self, *args)

	def Hide(self, view="ALL"):
		if not self.Hidden:
			self.Erase(view)
			self.Disable()
			self.Hidden= 1

	def Show(self, view="ALL"):
		if self.Hidden:
			self.Hidden= 0
			self.Update(view)
			self.Enable()
		
	def Press(self, pos):
		x= pos.DataCoord[0]
		(xmin, xmax)= self.GetLimits()
		if x>xmin and x<xmax: self.SetActive(1)
		else: self.SetActive(0)

	def GetLimits(self):
		rect= self.GetSelection()["BoundingRect"]
		return (rect[0][0], rect[1][0])

	def SetLimits(self, limits):
		self.SetSelectionPos(((limits[0], 0), (limits[1], 0)))

	def GetPen(self):
		return self.Pen

	def SetActive(self, active=0):
		if active!=self.Active:
			self.Active= active
			self.Pen.width= 1+active
			self.Update()

	def IsHidden(self):
		return (self.Hidden==1)

	def IsActive(self):
		return (self.Active==1)

	def GetType(self):
		return "Roi"

class RoiViewSelectList(ViewSelectList):
	def __init__(self, view):
		ViewSelectList.__init__(self, view, RoiViewSelect)
		self.Pen= GraphPen(1, "solid")

	def GetPen(self):
		return self.Pen.get()
		
