"""
    Drawable2D.py
    2D Image displaying widget
        
"""

from PyDVT import __version__,__date__,__author__


import Tkinter
import Image
import ImageTk
import sys
import Numeric
from Binding import Pen,Brush



class Drawable_2D(Tkinter.Canvas):
   def __init__(self,parent,zoom_mode="ON",scroll_mode="ON"):
        kw = {}  
        kw["highlightthickness"] = 0
        self.ImageSize=None
        self.parent=parent
        self.Image =None
        self.ZoomMode=zoom_mode
        self.ScrollMode=scroll_mode
        self.colormap_table=None
        Tkinter.Canvas.__init__(self,parent,kw)        

        self.DataSize=(0,0)
        self.ImageSize=(0,0)
        self.ViewSize=(0,0)  
        self.WindowSize=(0,0)
        self.ViewPosition=(0,0)
        self.WindowPosition=(0,0)
        self.DataPosition=(0,0)
        self.FormerViewSize=(0,0)

        self.ZoomFactor = 1.0        

        self._CreateScrollBar()
        self.configure(xscrollcommand=self.xsbar.set,xscrollincrement='1p')
        self.configure(yscrollcommand=self.ysbar.set,yscrollincrement='1p')
        self.xview("moveto",0)
        self.yview("moveto",0)

        self.bind("<Configure>", self._ResizeCanvas)
        self.bind('<ButtonPress-3>',self._RightButtonPressCanvas)
        self.bind("<Motion>", self._MotionCanvas, "+")
        self.bind("<B1-Motion>", self._ButtonPressMotionCanvas, "+")
        self.bind("<ButtonRelease-1>", self._ButtonReleaseCanvas, "+")
        self.bind("<ButtonPress-1>", self._ButtonPressCanvas, "+")
        self.bind("<Double-Button-1>",self._DoubleClickCanvas)
        self.bind("<KeyPress>",self._KeyPress)
        
        
        self.focus()
        self.EraseImage()
            


   def setScrollTracking(Tracking):
        if (Tracking):  self.configure(jump=Tkinter.FALSE)
        else:           self.configure(jump=Tkinter.TRUE) 
    

   def ViewCoord2ImageCoord(self,view_coord):
        if (self.Image==None):
            return (-1,-1)
        else:
            return self._ViewCoord2DataSelectionCoord(view_coord)

        
   def ImageCoord2ViewCoord(self,image_coord):
        if (self.Image==None):
            return (-1,-1)
        else:
            return self._DataSelectionCoord2ViewCoord(image_coord)



   def DrawImage(self,image_string,image_size,depth,colormap_table,reset_zoom=0):
        if reset_zoom:
            self.ZoomFactor=1.0
            self.WindowPosition=(0,0)
        if depth==8:
            self.Image = Image.fromstring("L",image_size,image_string)            
            if self.colormap_table is not colormap_table:
                self.palette=[]
                for i in range (256):
                    val=int(colormap_table[i])
                    self.palette.append(val & 0xFF)
                    self.palette.append((val & 0xFF00)/0x100)
                    self.palette.append((val & 0xFF0000)/0x10000)                            
                self.colormap_table = colormap_table
            self.Image.putpalette(self.palette)
        else:
            if sys.byteorder == "big":
                image_string = Numeric.fromstring(image_string,"l").byteswapped().tostring()
            self.Image = Image.fromstring("RGBX",image_size,image_string)
        self.DataSize=image_size
        self.Redraw()


   def EraseImage(self):
        self.FormerViewSize=(0,0)
        self.ImageSize=self.DataSize=(0,0)
        self.Image=None
        self.View = Image.new("RGB",(0,0))
        self.photim = ImageTk.PhotoImage(self.View)
        self.canvim = self.create_image(0,0,anchor=Tkinter.NW,image=self.photim)


   def PutLine(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid")):
        (X0,Y0)=self.ImageCoord2ViewCoord((X0,Y0))
        (X1,Y1)=self.ImageCoord2ViewCoord((X1,Y1))
        obj = self.create_line(X0,Y0,X1,Y1,stipple=pen.GetStyle(), width=pen.GetWidth(),fill=pen.GetColor())        
        self.tkraise(obj)
        return obj

   def PutRectangle(self,X0,Y0,X1,Y1,pen=Pen((0,0,0),1,"solid"),brush=Brush((0,0,0),"fill_0")):
       (X0,Y0)=self.ImageCoord2ViewCoord((X0,Y0))
       (X1,Y1)=self.ImageCoord2ViewCoord((X1,Y1))
       obj = self.create_rectangle(X0,Y0,X1,Y1,fill=brush.GetColor(),stipple=brush.GetStyle(),width=pen.GetWidth(),outline=pen.GetColor(),outlinestipple=pen.GetStyle()) 
       self.tkraise(obj)
       return obj
               
   def SetObjectCoords(self,obj,X0,Y0,X1,Y1):
        (X0,Y0)=self.ImageCoord2ViewCoord((X0,Y0))
        (X1,Y1)=self.ImageCoord2ViewCoord((X1,Y1))
        self.coords(obj, X0,Y0,X1,Y1)
        self.tkraise(obj)
        
   def EraseObject(self,obj):
       self.delete(obj)


   def SetZoom(self, val=None, rect=None, com=None):
        self.WindowSize=(float(self.parent.winfo_width()),float(self.parent.winfo_height()))
        if val != None:
          DataPosition=self._ImageCoord2DataSelectionCoord(self.WindowPosition)
          self.ZoomFactor = val
          self.WindowPosition=self._DataSelectionCoord2ImageCoord(DataPosition)

        #Data Selection Coordinates
        elif rect != None:
          DataWidth = max(rect[2] - rect[0] , 1)
          DataHeight = max(rect[3] - rect[1] , 1)
          if DataWidth==0: return
          if DataHeight==0: return
          self.ZoomFactor=min((float(self.WindowSize[0])/float(DataWidth)),float(self.WindowSize[1])/float(DataHeight))
          self.WindowPosition=self._DataSelectionCoord2ImageCoord([rect[0],rect[1]])
          
        elif com != None:
          if com == "fit":
            self.WindowPosition=(0,0)
            if (self.DataSize[0]==0) or (self.DataSize[1]==0): return
            self.ZoomFactor=min(self.WindowSize[0]/float(self.DataSize[0]),(self.WindowSize[1]/float(self.DataSize[1])))        
          elif com == "normal":
            self.WindowPosition=(0,0)
            self.ZoomFactor=1.0
          elif com == "in":
            self.SetZoom(val=self.ZoomFactor*2.0)
            return
          elif com == "out":
            self.SetZoom(val=self.ZoomFactor*0.5)
            return
        if self.ZoomFactor < 0.001:
          self.ZoomFactor = 0.001          
        self.Update()
    
   def MoveTo(self,position):
        self.xview(Tkinter.MOVETO,position[0])
        self.yview(Tkinter.MOVETO,position[1])

        
   def SetScrollRegion(self,scroll_region):
        self.configure(scrollregion=scroll_region)
        
   def Show(self):
        self.grid(row=0,col=0,sticky="nsew")

   def GetWindowPosition(self):
       return self.WindowPosition

   def GetWindowHeight(self):
        return self.winfo_height()

   def GetWindowWidth(self):
        return self.winfo_width()

   def GetZoomFactor(self):
       return self.ZoomFactor

   def GetHeight(self):
        return self.DataSize[1]

   def GetWidth(self):
        return self.DataSize[0]


   def Update(self):
       self.Redraw()
       self.parent.Invalidate()

   def Redraw(self):
        """
        Draws itself
        """
        self.WindowSize=(float(self.parent.winfo_width()),float(self.parent.winfo_height()))
        if  (self.Image is not None): 
          if (self.WindowSize[0]<=1) and (self.WindowSize[1]<=1): return
          self.parent.grid_propagate(0)
          if self.ZoomMode=="FIT_TO_SCREEN":
            self.WindowPosition=(0,0)
            if (self.DataSize[0]==0) or (self.DataSize[1]==0): return
            self.ZoomFactor=min(self.WindowSize[0]/float(self.DataSize[0]),(self.WindowSize[1]/float(self.DataSize[1])))        
            self.ImageSize=(self.DataSize[0] * self.ZoomFactor,self.DataSize[1] * self.ZoomFactor)
            self.ViewSize=self.ImageSize
            self.ViewPosition=(0,0)
          else:
              if self.ScrollMode=="OFF":
                self.ImageSize=(self.DataSize[0] * self.ZoomFactor,self.DataSize[1] * self.ZoomFactor)
                self.ViewSize= (min(self.ImageSize[0]-self.WindowPosition[0],self.winfo_width()),min(self.ImageSize[1]-self.WindowPosition[1],self.winfo_height()))
                self.ViewSize= (max(self.ViewSize[0],0),max(self.ViewSize[1],0))
                
                self.ViewPosition=self.WindowPosition
              else:
                self.ImageSize=(self.DataSize[0] * self.ZoomFactor,self.DataSize[1] * self.ZoomFactor)
                if self.AutoOversizeSet:
                  if self.ZoomFactor<=1:
                    self.FullDisplay=1
                  else:
                    self.FullDisplay=0
                    self.Oversize=1.5
                self._CheckDisplayScrollbars()
                self.WindowSize=(float(self.winfo_width()),float(self.winfo_height()))
                #Sets View Area
                if self.FullDisplay:
                  self.ViewSize=self.ImageSize
                  self.ViewPosition=(0,0)
                else:        
                  self.ViewSize=(min(self.ImageSize[0],self.WindowSize[0]*self.Oversize),min(self.ImageSize[1],self.WindowSize[1]*self.Oversize))
                  self.ViewPosition=(self.WindowPosition[0]-(self.ViewSize[0]-self.WindowSize[0])/2,self.WindowPosition[1]-(self.ViewSize[1]-self.WindowSize[1])/2)
                  self.ViewPosition=(max(self.ViewPosition[0],0.0),max(self.ViewPosition[1],0.0))
                  self.ViewPosition=(min(self.ViewPosition[0],self.ImageSize[0]-self.ViewSize[0]),min(self.ViewPosition[1],self.ImageSize[1]-self.ViewSize[1]))

          self.DataPosition=self._ImageCoord2DataSelectionCoord(self.ViewPosition)          
          self.CorrPos=(self.ViewPosition[0]%self.ZoomFactor,self.ViewPosition[1]%self.ZoomFactor)
            
          SizeImg=[self.ViewSize[0]+self.CorrPos[0],self.ViewSize[1]+self.CorrPos[1]]
          
          SizeXData=int(min((SizeImg[0]/self.ZoomFactor)+1,int(self.DataSize[0])-self.DataPosition[0]))
          SizeYData=int(min((SizeImg[1]/self.ZoomFactor)+1,int(self.DataSize[1])-self.DataPosition[1]))          
          self.View=self.Image.crop((int(self.DataPosition[0]),int(self.DataPosition[1]),int(self.DataPosition[0]+SizeXData),int(self.DataPosition[1]+SizeYData)))
          if self.ZoomFactor != 1:
            SizeImg=[self.ZoomFactor*SizeXData,self.ZoomFactor*SizeYData]
            self.View = self.View.resize(SizeImg)
            self.View=self.View.crop((int(self.CorrPos[0]),int(self.CorrPos[1]),int(self.CorrPos[0]+self.ViewSize[0]),int(self.CorrPos[1]+self.ViewSize[1])))
                      
          if self.ViewSize==self.FormerViewSize:  self.photim.paste(self.View)
          else:
            self.photim = ImageTk.PhotoImage(self.View)
            self.itemconfig(self.canvim, image=self.photim)
            self.FormerViewSize=self.ViewSize
          
          
          if self.ScrollMode!="OFF":
            self.configure(scrollregion=(-self.ViewPosition[0],-self.ViewPosition[1],self.ImageSize[0]-self.ViewPosition[0],self.ImageSize[1]-self.ViewPosition[1]))
            self.xview(Tkinter.MOVETO,(self.WindowPosition[0]/self.ImageSize[0]))
            self.yview(Tkinter.MOVETO,(self.WindowPosition[1]/self.ImageSize[1]))                      
          
        else:
            self.FormerViewSize=(0,0)
            self.View = Image.new("RGB",(0,0))
            self.photim = ImageTk.PhotoImage(self.View)
            self.canvim = self.create_image(0,0,anchor=Tkinter.NW,image=self.photim)


   def _ResizeCanvas(self, event):
        """
        Resize events don't call directly update method because sometimes
        the drawing area was bigger the actual displayed, or the derived class does
        not want to redraw itself as the window changes. In the derived classes,
        it can be decided if an update is required or not.
        """
        if self.Image is None: return
        if self.ZoomMode=="FIT_TO_SCREEN":
            self.Update()
        else:
            if self.ScrollMode=="OFF":
                self.Update()
            else:
                self.ysbar.update()
                self.xsbar.update()
                posx=self.xsbar.get()
                posy=self.ysbar.get()
                beginx=posx[0] * self.ImageSize[0]
                beginy=posy[0] * self.ImageSize[1]
                endx=posx[1] * self.ImageSize[0]
                endy=posy[1] * self.ImageSize[1]

                if (beginx <self.ViewPosition[0]) or \
                   (beginy <self.ViewPosition[1]) or \
                   (endx>(self.ViewPosition[0]+self.ViewSize[0]))or \
                   (endy>(self.ViewPosition[1]+self.ViewSize[1]))or \
                   (self.ViewSize[0]<=2 and self.ViewSize[1]<=2):      
                  self.Update()
                else:
                  self.WindowSize=(float(self.winfo_width()),float(self.winfo_height()))
                  self._CheckDisplayScrollbars()


   def _RightButtonPressCanvas(self,event):
        """
        """
        self.parent._RightButtonPress(event)


   def _ButtonPressMotionCanvas(self, event):
        """
        """
        pos=self.ViewCoord2ImageCoord((self.canvasx(event.x),self.canvasy(event.y)))
        if pos is not None: self.parent._ButtonPressMotion(pos)

   def _ButtonPressCanvas(self, event):
        """
        """
        pos=self.ViewCoord2ImageCoord((self.canvasx(event.x),self.canvasy(event.y)))
        if pos is not None: self.parent._ButtonPress(pos)

   def _DoubleClickCanvas(self, event):
        """
        """
        pos=self.ViewCoord2ImageCoord((self.canvasx(event.x),self.canvasy(event.y)))
        if pos is not None: self.parent._DoubleClick(pos)

   def _ButtonReleaseCanvas(self, event):
        """
        """
        pos=self.ViewCoord2ImageCoord((self.canvasx(event.x),self.canvasy(event.y)))
        if pos is not None: self.parent._ButtonRelease(pos)
            
   def _MotionCanvas(self, event):
        """
        """
        pos=self.ViewCoord2ImageCoord((self.canvasx(event.x),self.canvasy(event.y)))
        if pos is not None: self.parent._Motion(pos)



   def _CreateScrollBar(self):
        self.xsbar = Tkinter.Scrollbar(self.parent,orient=Tkinter.HORIZONTAL)        
        self.xsbar.configure(command=self._EventXSB,jump=Tkinter.FALSE)
        self.ysbar = Tkinter.Scrollbar(self.parent,orient=Tkinter.VERTICAL)        
        self.ysbar.configure(command=self._EventYSB,jump=Tkinter.FALSE)

        self.ysbar.bind("<ButtonRelease-1>", self._ButtonReleaseYsbar)
        self.xsbar.bind("<ButtonRelease-1>", self._ButtonReleaseXsbar)


   def _CheckDisplayScrollbars(self):
          if self.ScrollMode=="OFF":
              showX=0
              showY=0          
          elif self.ScrollMode=="ON":
              showX=1
              showY=1          
          else:
              InnerWindowSize=[self.WindowSize[0]-self.ysbar.winfo_width(),self.WindowSize[1]-self.xsbar.winfo_height()]
              showX,showY=0,0
              if round(self.ImageSize[0]) > self.WindowSize[0]:
                showX=1
                if round(self.ImageSize[1])> InnerWindowSize[1]:
                  showY=1
              elif round(self.ImageSize[0])<=InnerWindowSize[0]:
                if round(self.ImageSize[1]) > self.WindowSize[1]:
                  showY=1
              else:
                if round(self.ImageSize[1]) > self.WindowSize[1]:
                  showX=1
                  showY=1          

          if showX==0: self.xsbar.grid_forget()
          else:        self.xsbar.grid(row=1,col=0,sticky="nswe")

          if showY==0: self.ysbar.grid_forget()
          else:        self.ysbar.grid(row=0,col=1,sticky="nswe")



   def _EventXSB(self, *arguments, **keywords):
        scrpos=self.xsbar.get()
        if scrpos == (0.0,1.0): return
        self.WindowPosition=(scrpos[0] * self.ImageSize[0],self.WindowPosition[1])
        self.xview(*arguments, **keywords) 

   def _EventYSB(self, *arguments, **keywords):
        scrpos=self.ysbar.get()
        if scrpos == (0.0,1.0): return
        self.WindowPosition=(self.WindowPosition[0],scrpos[0] * self.ImageSize[1])
        self.yview(*arguments, **keywords)


   def _ButtonReleaseYsbar(self, event):
        scrpos=self.ysbar.get()
        if scrpos == (0.0,1.0): return
        DisplayBegin= scrpos[0] * self.ImageSize[1]
        DisplayEnd=scrpos[1] * self.ImageSize[1]
        self.WindowPosition=(self.WindowPosition[0],DisplayBegin)
        if (DisplayBegin<self.ViewPosition[1]) or (DisplayEnd>(self.ViewPosition[1]+self.ViewSize[1])):
          self.Update()      

   def _ButtonReleaseXsbar(self, event):
        scrpos=self.xsbar.get()
        if scrpos == (0.0,1.0): return
        DisplayBegin= scrpos[0] * self.ImageSize[0]
        DisplayEnd=scrpos[1] * self.ImageSize[0]
        self.WindowPosition=(DisplayBegin,self.WindowPosition[1])
        if (DisplayBegin<self.ViewPosition[0]) or (DisplayEnd>(self.ViewPosition[0]+self.ViewSize[0])):
          self.Update()      


#################################
#            TRANSFORMATIONS
#################################

   def _DataSelectionCoord2ViewCoord(self,data_selection_coord):
        image_coord=self._DataSelectionCoord2ImageCoord(data_selection_coord)
        if image_coord == (-1,-1): return (-1,-1)
        return ((int(image_coord[0]-self.ViewPosition[0]),int(image_coord[1]-self.ViewPosition[1])))

   def _ViewCoord2DataSelectionCoord(self,view_coord):        
        xval = (self.ViewPosition[0]+view_coord[0])
        yval = (self.ViewPosition[1]+view_coord[1])
        return self._ImageCoord2DataSelectionCoord((int(xval),int(yval)))


   def _ImageCoord2DataSelectionCoord(self,image_coord):        
        if self.Image is None: return (-1,-1)
        xval = int (float(image_coord[0]) / self.ZoomFactor)
        yval=  int (float(image_coord[1]) / self.ZoomFactor)
        xval = min (xval,int(self.DataSize[0])-1)
        xval = max (xval,0)
        yval = min (yval,int(self.DataSize[1])-1)
        yval = max (yval,0)
        return (xval, yval)


   def _DataSelectionCoord2ImageCoord(self,data_selection_coord):
        if self.Image  is None: return (-1,-1)
        xval = int (float(min(data_selection_coord[0],int(self.DataSize[0]))) * self.ZoomFactor)
        yval=  int (float(min(data_selection_coord[1],int(self.DataSize[1]))) * self.ZoomFactor)
        return (xval, yval)


   def _ViewCoord2WindowCoord(self,view_coord):
        xval = view_coord[0]-(self.WindowPosition[0] - self.ViewPosition[0])
        yval = view_coord[1]-(self.WindowPosition[1] - self.ViewPosition[1])
        return (int(xval), int(yval))

   def _WindowCoord2ImageCoord(self,window_coord):
        xval = int (window_coord[0] + self.WindowPosition[0])
        yval = int (window_coord[1] + self.WindowPosition[1])
        return (xval, yval)

   def _KeyPress(self,e):
       ret=str(e.char)
       if ret=="":
            ret= e.keycode
       self.parent._KeyPress(ret)
