# Copyright (c) 2002 Detlev Offenbach <detlev@die-offenbachs.de>

import os

from qt import *

from Icons import IconBack, IconForward, IconHome, IconOpen, IconPrint
from Icons import IconClose, IconExit, IconAddBookmark, IconEditBookmarks
from Icons import IconNewWindow, IconWhatsThis
from BookmarkDialog import BookmarkDialog

class HelpWindow(QMainWindow):
    """
    Class to implement a help viewer. Ported to PyQt from the Qt examples.
    This version includes some enhancements to the original.
    """
    helpwindows = []
    mBookmarks = []
    editBMDialog = None

    def __init__(self, home, path, parent, name):
        QMainWindow.__init__(self,parent,name,Qt.WDestructiveClose)
        
        self.mHistory = []
        self.pathCombo = None
        
        homeDirStr = QDir.homeDirPath()
        self.ericDirStr = str(homeDirStr) + os.sep + '.eric'
        ericDir = QDir(self.ericDirStr)
        if not ericDir.exists():
            ericDir.mkdir(self.ericDirStr)
        
        self.readHistory()
        if not len(self.__class__.mBookmarks):
            self.readBookmarks()
        
        self.browser = QTextBrowser(self)
        QWhatsThis.add(self.browser, self.trUtf8(
                """<b>Help Browser</b>"""
                """<p>This is the main window of the helpviewer. It"""
                """displays the selected help information. Only"""
                """html help files can be displayed.</p>"""
                ))
        
        self.browser.mimeSourceFactory().setFilePath(QStringList(path))
        self.browser.setFrameStyle(QFrame.Panel | QFrame.Sunken)
        self.connect(self.browser, SIGNAL('textChanged()'), self.handleTextChanged)
        
        self.setCentralWidget(self.browser)
        
        if home:
            self.browser.setSource(home)
            
        self.connect(self.browser, SIGNAL('highlighted(const QString&)'),
                     self.statusBar(), SLOT('message(const QString&)'))
        self.connect(self.browser, SIGNAL('backwardAvailable(bool)'),
                     self.setBackwardAvailable)
        self.connect(self.browser, SIGNAL('forwardAvailable(bool)'),
                     self.setForwardAvailable)
                     
        self.resize(640, 700)
        
        self.initActions()
        self.initMenus()
        self.initToolbars()
        
        self.setRightJustification(1)
        self.pathCombo.insertItem(home)
        self.browser.setFocus()
        
        self.__class__.helpwindows.append(self)
        
        # generate the edit bookmarks dialog
        self.__class__.editBMDialog = BookmarkDialog(self)
        
    def initActions(self):
        """
        Define the user interface actions.
        """
        self.newAct = QAction(self.trUtf8('New Window'), 
                        QIconSet(IconNewWindow), 
                        self.trUtf8('&New Window'), 
                        QKeySequence(self.trUtf8("CTRL+N","File|New Window")), 
                        self)
        self.newAct.setStatusTip(self.trUtf8('Opens a new help window'))
        self.newAct.setWhatsThis(self.trUtf8(
                """<b>New Window</b>"""
                """<p>This opens a new help window. The new window is"""
                """ a clone of the current one.</p>"""
                ))
        self.connect(self.newAct, SIGNAL('activated()'), self.handleNewWindow)
        
        self.openAct = QAction(self.trUtf8('Open File'), 
                        QIconSet(IconOpen), 
                        self.trUtf8('&Open File'), 
                        QKeySequence(self.trUtf8("CTRL+O","File|Open")), 
                        self)
        self.openAct.setStatusTip(self.trUtf8('Open a help file for display'))
        self.openAct.setWhatsThis(self.trUtf8(
                """<b>Open File</b>"""
                """<p>This opens a new help file for display."""
                """ It pops up a file selection dialog.</p>"""
                ))
        self.connect(self.openAct, SIGNAL('activated()'), self.handleOpenFile)
        
        self.printAct = QAction(self.trUtf8('Print'), 
                        QIconSet(IconPrint), 
                        self.trUtf8('&Print'), 
                        QKeySequence(self.trUtf8("CTRL+P","File|Print")), 
                        self)
        self.printAct.setStatusTip(self.trUtf8('Print the displayed help'))
        self.printAct.setWhatsThis(self.trUtf8(
                """<b>Print</b>"""
                """<p>Print the displayed help text.</p>"""
                ))
        self.connect(self.printAct, SIGNAL('activated()'), self.handlePrintFile)
        
        self.closeAct = QAction(self.trUtf8('Close'), 
                        QIconSet(IconClose), 
                        self.trUtf8('&Close'), 
                        QKeySequence(self.trUtf8("CTRL+W","File|Close")), 
                        self)
        self.closeAct.setStatusTip(self.trUtf8('Close the current help window'))
        self.closeAct.setWhatsThis(self.trUtf8(
                """<b>Close</b>"""
                """<p>Closes the current help window. If the master"""
                """ help window is closed, all others are closed as well.</p>"""
                ))
        self.connect(self.closeAct, SIGNAL('activated()'), self, SLOT('close()'))
        
        self.exitAct = QAction(self.trUtf8('Exit'), 
                        QIconSet(IconExit), 
                        self.trUtf8('E&xit'), 
                        QKeySequence(self.trUtf8("CTRL+X","File|Exit")), 
                        self)
        self.exitAct.setStatusTip(self.trUtf8('Exit the application'))
        self.exitAct.setWhatsThis(self.trUtf8(
                """<b>Exit</b>"""
                """<p>Exit the application.</p>"""
                ))
        self.connect(self.exitAct, SIGNAL('activated()'), qApp, SLOT('closeAllWindows()'))
        
        
        self.backAct = QAction(self.trUtf8('Backward'), 
                        QIconSet(IconBack), 
                        self.trUtf8('&Backward'), 
                        Qt.CTRL+Qt.Key_Left, self)
        self.backAct.setStatusTip(self.trUtf8('Move one help screen backward'))
        self.backAct.setWhatsThis(self.trUtf8(
                """<b>Backward</b>"""
                """<p>Moves one help screen backward. If none is"""
                """ available, this action is disabled.</p>"""
                ))
        self.connect(self.backAct, SIGNAL('activated()'),
                     self.browser, SLOT('backward()'))
                     
        self.forwardAct = QAction(self.trUtf8('Forward'), 
                        QIconSet(IconForward), 
                        self.trUtf8('&Forward'), 
                        Qt.CTRL+Qt.Key_Right, self)
        self.forwardAct.setStatusTip(self.trUtf8('Move one help screen forward'))
        self.forwardAct.setWhatsThis(self.trUtf8(
                """<b>Forward</b>"""
                """<p>Moves one help screen forward. If none is"""
                """ available, this action is disabled.</p>"""
                ))
        self.connect(self.forwardAct, SIGNAL('activated()'),
                     self.browser, SLOT('forward()'))
                     
        self.homeAct = QAction(self.trUtf8('Home'), 
                        QIconSet(IconHome), 
                        self.trUtf8('&Home'), 0, self)
        self.homeAct.setStatusTip(self.trUtf8('Move to the initial help screen'))
        self.homeAct.setWhatsThis(self.trUtf8(
                """<b>Home</b>"""
                """<p>Moves to the initial help screen.</p>"""
                ))
        self.connect(self.homeAct, SIGNAL('activated()'),
                     self.browser, SLOT('home()'))
        
        self.clearHistoryAct = QAction(self.trUtf8('Clear History'), 
                        self.trUtf8('&Clear History'), 0, self)
        self.clearHistoryAct.setStatusTip(self.trUtf8('Clear the history information.'))
        self.clearHistoryAct.setWhatsThis(self.trUtf8(
                """<b>Clear History</b>"""
                """<p>Clears the history information.</p>"""
                ))
        self.connect(self.clearHistoryAct,SIGNAL('activated()'),self.handleClearHistory)

        self.addBookmarkAct = QAction(self.trUtf8('Add Bookmark'), 
                                QIconSet(IconAddBookmark), 
                                self.trUtf8('&Add Bookmark'),0,self)
        self.addBookmarkAct.setStatusTip(self.trUtf8('Add a bookmarks.'))
        self.addBookmarkAct.setWhatsThis(self.trUtf8(
                """<b>Add Bookmark</b>"""
                """<p>Adds the currently displayed help screen to"""
                """ the list of bookmarks. Bookmarks are saved in"""
                """ the file bookmark in the .eric subdirectory"""
                """ within the users home directory, i.e."""
                """ ~/.eric/bookmarks. This file contains two"""
                """ lines of text for each entry. The first line"""
                """ gives the text to be displayed in the menu,"""
                """ the second line names the help file to be displayed.</p>"""
                ))
        self.connect(self.addBookmarkAct,SIGNAL('activated()'),self.handleAddBookmark)
                     
        self.editBookmarksAct = QAction(self.trUtf8('Edit Bookmarks'), 
                                QIconSet(IconEditBookmarks),
                                self.trUtf8('&Edit Bookmarks...'),0,self)
        self.editBookmarksAct.setStatusTip(self.trUtf8('Edit Bookmarks'))
        self.editBookmarksAct.setWhatsThis(self.trUtf8(
                """<b>Edit Bookmarks</b>"""
                """<p>This action displayes a dialog in which you can"""
                """ change the name and the file of a bookmark as well"""
                """ as the order of the bookmarks in the menu.</p>"""
                ))
        self.connect(self.editBookmarksAct,SIGNAL('activated()'),self.handleEditBookmarks)
        
        self.whatsThisAct = QAction(self.trUtf8('What\'s This?'),
                                QIconSet(IconWhatsThis),
                                self.trUtf8('What\'s This?'),
                                Qt.SHIFT+Qt.Key_F1,self)
        self.whatsThisAct.setStatusTip(self.trUtf8('Context sensitive help'))
        self.whatsThisAct.setWhatsThis(self.trUtf8(
                """<b>Display context sensitive help</b>"""
                """<p>In What's This? mode, the mouse cursor shows an arrow with a question"""
                """ mark, and you can click on the interface elements to get a short"""
                """ description of what they do and how to use them. In dialogs, this"""
                """ feature can be accessed using the context help button in the"""
                """ titlebar.</p>"""
                ))
        self.connect(self.whatsThisAct,SIGNAL('activated()'),self.whatsThis)

        self.aboutAct = QAction(self.trUtf8('About'),
                            self.trUtf8('&About'),0,self)
        self.aboutAct.setStatusTip(self.trUtf8('Display information about this software'))
        self.aboutAct.setWhatsThis(self.trUtf8(
                """<b>About</b>"""
                """<p>Display some information about this software.</p>"""
                ))
        self.connect(self.aboutAct,SIGNAL('activated()'),self.handleAbout)
                     
        self.aboutQtAct = QAction(self.trUtf8('About Qt'),
                            self.trUtf8('About &Qt'),0,self)
        self.aboutQtAct.setStatusTip(self.trUtf8('Display information about the Qt toolkit'))
        self.aboutQtAct.setWhatsThis(self.trUtf8(
                """<b>About Qt</b>"""
                """<p>Display some information about the Qt toolkit.</p>"""
                ))
        self.connect(self.aboutQtAct,SIGNAL('activated()'),self.handleAboutQt)

        self.backAct.setEnabled(0)
        self.forwardAct.setEnabled(0)
    
    def initMenus(self):
        """
        Create the menus.
        """
        mb = self.menuBar()

        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&File'),menu)
        self.newAct.addTo(menu)
        self.openAct.addTo(menu)
        self.printAct.addTo(menu)
        menu.insertSeparator()
        self.closeAct.addTo(menu)
        self.exitAct.addTo(menu)

        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&Go'), menu)
        self.backAct.addTo(menu)
        self.forwardAct.addTo(menu)
        self.homeAct.addTo(menu)
        
        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&History'), menu)
        self.connect(menu, SIGNAL('aboutToShow()'), self.handleShowHistoryMenu)
        self.connect(menu, SIGNAL('activated(int)'), self.handleHistChosen)
        self.historyMenu = menu
        
        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&Bookmarks'), menu)
        self.connect(menu, SIGNAL('aboutToShow()'), self.handleShowBookmarksMenu)
        self.connect(menu, SIGNAL('activated(int)'), self.handleBookmChosen)
        self.bookmarksMenu = menu
        
        mb.insertSeparator()
        
        menu = QPopupMenu(self)
        mb.insertItem(self.trUtf8('&Help'),menu)
        self.aboutAct.addTo(menu)
        self.aboutQtAct.addTo(menu)
        menu.insertSeparator()
        self.whatsThisAct.addTo(menu)

    
    def initToolbars(self):
        """
        Create the toolbars.
        """
        filetb = QToolBar(self)
        bookmarktb = QToolBar(self)
        helptb = QToolBar(self)
        gotb = QToolBar(self)
        
        self.newAct.addTo(filetb)
        self.openAct.addTo(filetb)
        self.printAct.addTo(filetb)
        
        filetb.addSeparator()
        
        self.closeAct.addTo(filetb)
        self.exitAct.addTo(filetb)
        
        
        self.addBookmarkAct.addTo(bookmarktb)
        self.editBookmarksAct.addTo(bookmarktb)
        
        
        self.backAct.addTo(gotb)
        self.forwardAct.addTo(gotb)
        self.homeAct.addTo(gotb)
        
        gotb.addSeparator()
        
        self.pathCombo = QComboBox(1, gotb)
        self.connect(self.pathCombo, SIGNAL('activated(const QString &)'),
                     self.handlePathSelected)
        gotb.setStretchableWidget(self.pathCombo)
        QWhatsThis.add(self.pathCombo,self.trUtf8(
                """<p>Enter the help file to be displayed directly into this"""
                """ edit field. Select a previously shown help file from the"""
                """ drop down list.</p>"""
                ))
        
        
        self.whatsThisAct.addTo(helptb)
        
        
    def handleShowHistoryMenu(self):
        """
        Private slot called in order to show the history menu
        """
        self.historyMenu.clear()
        self.clearHistoryAct.addTo(self.historyMenu)
        self.historyMenu.insertSeparator()
        idx = 0
        for hist in self.mHistory:
            self.historyMenu.insertItem(hist,idx)
            idx += 1
        
    def handleShowBookmarksMenu(self):
        """
        Private slot called in order to show the bookmarks menu
        """
        self.bookmarksMenu.clear()
        self.addBookmarkAct.addTo(self.bookmarksMenu)
        self.editBookmarksAct.addTo(self.bookmarksMenu)
        self.bookmarksMenu.insertSeparator()
        idx = 0
        for bookm in self.__class__.mBookmarks:
            self.bookmarksMenu.insertItem(bookm[0],idx)
            idx += 1
        
    def handlePathSelected(self, path):
        """
        Private slot called when a file is selected in the combo box
        """
        self.browser.setSource(path)
        try:
            self.mHistory.index(str(path))
        except ValueError:
            self.mHistory.append(str(path))
            if len(self.mHistory) > 20:
                self.mHistory = self.mHistory[1:]

    def handleTextChanged(self):
        """
        Private slot called when the displayed text is changed
        """
        if self.browser.documentTitle().isNull():
            self.setCaption(str(self.browser.context()))
        else:
            self.setCaption(str(self.browser.documentTitle()))
            
        selectedURL = self.browser.context()
        
        if not selectedURL.isEmpty() and self.pathCombo:
            exists = 0
            i = 0
            for i in range(0, self.pathCombo.count()):
                if str(self.pathCombo.text(i)) == str(selectedURL):
                    exists = 1
                    break
                    
            if not exists:
                self.pathCombo.insertItem(selectedURL, 0)
                self.pathCombo.setCurrentItem(0)
                try:
                    self.mHistory.index(str(selectedURL))
                except ValueError:
                    self.mHistory.append(str(selectedURL))
                    if len(self.mHistory) > 20:
                        self.mHistory = self.mHistory[1:]
            else:
                self.pathCombo.setCurrentItem(i)
            

    def handleNewWindow(self):
        """
        Private slot called to open a new help window
        """
        h = HelpWindow(self.browser.source(), ".", self.parent(), "qbrowser")
        h.show()
        
    def handleOpenFile(self):
        """
        Private slot called to open a file
        """
        fn = QFileDialog.getOpenFileName(None, None, self)
        if not fn.isEmpty():
            self.browser.setSource(fn)
        
    def handlePrintFile(self):
        """
        Private slot called to print the current file
        """
        printer = QPrinter()
        printer.setFullPage(1)
        if printer.setup(self):
            p = QPainter(printer)
            metrics = QPaintDeviceMetrics(p.device())
            dpix = metrics.logicalDpiX()
            dpiy = metrics.logicalDpiY()
            margin = 72 # pt
            body = QRect(margin * dpix / 72, margin * dpiy / 72,
                         metrics.width() - margin * dpix / 72 * 2,
                         metrics.height() - margin * dpiy / 72 * 2)
            font = QFont("times", 10)
            richText = QSimpleRichText(self.browser.text(), font,
                        self.browser.context(),
                        self.browser.styleSheet(),
                        self.browser.mimeSourceFactory(),
                        body.height())
            richText.setWidth(p, body.width())
            view = QRect(body)
            page = 1
            while 1:
                richText.draw(p, body.left(), body.top(), view, self.colorGroup())
                view.moveBy(0, body.height())
                p.translate(0, -body.height())
                p.setFont(font)
                p.drawText(view.right() - p.fontMetrics().width(QString.number(page)),
                    view.bottom() + p.fontMetrics().ascent() + 5, QString.number(page))
                if view.top() >= body.top() + richText.height():
                    p.end()
                    break
                printer.newPage()
                page += 1
        
    def handleAbout(self):
        """
        Private slot to show the about information
        """
        QMessageBox.about(self, self.trUtf8("Helpviewer"), self.trUtf8(
            """<h3> About Helpviewer </h3>"""
            """<p>The Helpviewer window is a little HTML browser to"""
            """ display HTML help files like the Qt documentation</p>"""
            ))
        
    def handleAboutQt(self):
        """
        Private slot to show info about Qt
        """
        QMessageBox.aboutQt(self, self.trUtf8("Helpviewer"))

    def readHistory(self):
        """
        Private function to read the history from the history file.
        The file is ~/.eric/history.
        """
        try:
            f1 = open(self.ericDirStr + os.sep + 'history', 'r')
            hist = f1.readlines()
            f1.close()
            if len(hist) > 20:
                hist = hist[(len(hist)-20):]
            for h in hist:
                self.mHistory.append(h.strip())
        except:
            pass
        
    def readBookmarks(self):
        """
        Private function to read the bookmarks from the bookmarks file.
        The file is ~/.eric/bookmarks.
        """
        try:
            f2 = open(self.ericDirStr + os.sep + 'bookmarks', 'r')
            bookmarks = f2.readlines()
            f2.close()
            for i in range(0, len(bookmarks), 2):
                self.__class__.mBookmarks.append((bookmarks[i].strip(),
                                        bookmarks[i+1].strip()))
        except:
            pass
            
    def writeBookmarks(self):
        """
        Public function to write the bookmarks to the bookmarks file.
        The file is ~/.eric/bookmarks.
        """
        try:
            f2 = open(self.ericDirStr + os.sep + 'bookmarks', 'w')
            for bookm in self.__class__.mBookmarks:
                f2.write(bookm[0] + os.linesep)
                f2.write(bookm[1] + os.linesep)
            f2.close()
        except:
            pass

    def setBackwardAvailable(self, b):
        """
        Private slot called when backward references are available.
        """
        self.backAct.setEnabled(b)
        
    def setForwardAvailable(self, b):
        """
        Private slot called when forward references are available.
        """
        self.forwardAct.setEnabled(b)
        
    def handleClearHistory(self):
        """
        Private slot called to clear the current history.
        """
        self.mHistory = []
        
    def handleHistChosen(self, i):
        """
        Private slot called by the history menu
        """
        if i >= 0:
            self.browser.setSource(self.mHistory[i])
        
    def handleBookmChosen(self, i):
        """
        Private slot called by the bookmarks menu
        """
        if i >= 0:
            self.browser.setSource(self.__class__.mBookmarks[i][1])
        
    def handleAddBookmark(self):
        """
        Private slot called to add a bookmark
        """
        self.__class__.mBookmarks.append((str(self.caption()), 
                                str(self.browser.context())))
        self.writeBookmarks()
                                
    def handleEditBookmarks(self):
        """
        Private slot called to edit the bookmarks
        """
        self.__class__.editBMDialog.show()
                                
    def closeEvent(self, e):
        """
        Private event handler for the close event.
        This handler saves the current bookmarks and history entries
        to the respective files in ~/.eric.
        """
        try:
            f1 = open(self.ericDirStr + os.sep + 'history', 'w')
            for hist in self.mHistory:
                f1.write(hist + os.linesep)
            f1.close()
        except:
            pass
        
        del self.__class__.helpwindows[self.__class__.helpwindows.index(self)]
        e.accept()