# Copyright (c) 2000, 2001, 2002 Phil Thompson <phil@river-bank.demon.co.uk>
# Copyright (c) 2002 Detlev Offenbach <detlev@die-offenbachs.de>


import sys
import os
import socket
from qt import SIGNAL, PYSIGNAL
from qtnetwork import QServerSocket, QSocket

from DebugProtocol import *
from Config import RemoteDebuggerEnabled, RemoteDebuggerHost, RemoteExecution


class DebugServer(QServerSocket):
    """DebugServer()

    A class that implements the server side of a debugger/client connection.

    """
    def __init__(self):
        QServerSocket.__init__(self,0)

        self.qsock = None
        self.progLoaded = 0
        self.queue = []
        self.breaks = []

        self.startRemote()

    def startRemote(self):
        """
        Private method to start a remote interpreter.
        """
        if RemoteDebuggerEnabled:
            ipaddr = socket.gethostbyname(socket.gethostname())
            os.spawnv(os.P_NOWAIT,RemoteExecution,
                [RemoteExecution, 
                 RemoteDebuggerHost, 
                 sys.executable,
                 os.path.abspath(os.path.join(sys.path[0],'DebugClient.py')),
                 `self.port()`,
                 ipaddr])
        else:
            os.spawnv(os.P_NOWAIT,sys.executable,
                [sys.executable,
                 os.path.join(sys.path[0],'DebugClient.py'),
                 `self.port()`])

    def newConnection(self,sockfd):
        """
        Re-implemented to handle a new connection.
        """
        sock = QSocket()
        sock.setSocket(sockfd)

        # If we already have a connection, refuse this one.  It will be closed
        # automatically.
        if self.qsock is not None:
            return

        self.connect(sock,SIGNAL('readyRead()'),self.handleLine)
        self.connect(sock,SIGNAL('connectionClosed()'),self.startClient)

        self.qsock = sock

        # Send commands that were waiting for the connection.
        for cmd in self.queue:
            self.qsock.writeBlock(cmd)

        self.queue = []

    def remoteLoad(self,finfo,argv):
        """
        Public method to load a new program to debug.
        """
        # Restart the client if there is already a program loaded.
        if self.progLoaded:
            self.startClient(0)
 
        self.sendCommand('%s%s %s\n' % (RequestLoad, str(finfo.absFilePath()),str(argv)))
        self.progLoaded = 1
        self.restoreBreakpoints()

    def remoteStatement(self,stmt):
        """
        Public method to execute a Python statement.  The statement should not
        have a trailing newline.
        """
        self.sendCommand('%s\n' % stmt)
        self.sendCommand(RequestOK + '\n')

    def remoteStep(self):
        """
        Public method to single step the debugged program.
        """
        self.sendCommand(RequestStep + '\n')

    def remoteStepOver(self):
        """
        Public method to step over the debugged program.
        """
        self.sendCommand(RequestStepOver + '\n')

    def remoteStepOut(self):
        """
        Public method to step out the debugged program.
        """
        self.sendCommand(RequestStepOut + '\n')

    def remoteStepQuit(self):
        """
        Public method to stop the debugged program.
        """
        self.sendCommand(RequestStepQuit + '\n')

    def remoteContinue(self):
        """
        Public method to continue the debugged program.
        """
        self.sendCommand(RequestContinue + '\n')

    def remoteBreakpoint(self,fn,line,set,cond):
        """
        Public method to set or clear a breakpoint.
        """
        if set:
            self.breaks.append((fn, line, cond))
        else:
            try:
                self.breaks.remove((fn, line, cond))
            except:
                pass
        self.sendCommand('%s%s,%d,%d,%s\n' % (RequestBreak,fn,line,set,cond))

    def remoteRawInput(self,s):
        """
        Public method to send the raw input to the debugged program.
        """
        self.sendCommand(s + '\n')
        
    def remoteClientVariables(self, scope, filter):
        """
        Public method to request the variables of the debugged program.
        """
        self.sendCommand('%s%d, %s\n' % (RequestVariables,scope,str(filter)))
        
    def remoteEval(self, arg):
        """
        Public method to evaluate arg in the current context of the debugged program
        """
        self.sendCommand('%s%s\n' % (RequestEval,arg))
        
    def remoteExec(self, stmt):
        """
        Public method to execute stmt in the current context of the debugged program
        """
        self.sendCommand('%s%s\n' % (RequestExec,stmt))
        
    def handleLine(self):
        """
        Private method to handle a data from the client.
        """
        while self.qsock.canReadLine():
            line = str(self.qsock.readLine())

            #print line          ##debug
            
            eoc = line.find('<') + 1

            if line[0] == '>' and eoc > 0:
                # Emit the signal corresponding to the response.
                resp = line[:eoc]

                if resp == ResponseLine:
                    fn, line = line[eoc:-1].split(',')
                    self.emit(PYSIGNAL('clientLine'),(fn,int(line)))
                    return
                    
                if resp == ResponseVariables:
                    self.emit(PYSIGNAL('clientVariables'),(line[eoc:-1],))
                    return

                if resp == ResponseOK:
                    self.emit(PYSIGNAL('clientStatement'),(0,))
                    return

                if resp == ResponseContinue:
                    self.emit(PYSIGNAL('clientStatement'),(1,))
                    return

                if resp == ResponseException:
                    self.emit(PYSIGNAL('clientException'),(line[eoc:-1],))
                    return

                if resp == ResponseSyntax:
                    self.emit(PYSIGNAL('clientSyntaxError'),(line[eoc:-1],))
                    return
                    
                if resp == ResponseExit:
                    self.emit(PYSIGNAL('clientExit'),(line[eoc:-1],))
                    return
                
                if resp == ResponseRaw:
                    self.emit(PYSIGNAL('clientRawInput'),(line[eoc:-1],))
                    return

            self.emit(PYSIGNAL('clientOutput'),(line,))

    def startClient(self,unplanned=1):
        """
        Private method to handle the loss of the client.
        """
        if self.qsock is not None:
            self.qsock = None
            self.emit(PYSIGNAL('clientGone'),(unplanned,))

        self.startRemote()

    def sendCommand(self,cmd):
        """
        Private method to send a single line command to the client.
        """
        if self.qsock is not None:
            self.qsock.writeBlock(cmd)
        else:
            self.queue.append(cmd)

    def restoreBreakpoints(self):
        """
        Private method to restore the break points after a restart.
        """
        if len(self.breaks):
            for fn, line, cond in self.breaks:
                self.sendCommand('%s%s,%d,%d,%s\n' % (RequestBreak,fn,line,1,cond))
    
    def clearBreakpoints(self):
        """
        Private method (slot) to clear the local list of breakpoints.
        """
        self.breaks = []
    
    def getAllBreakpoints(self):
        """
        Public method to get all breakpoints set in the server.
        """
        return self.breaks
        
    def getFileBreakpoints(self, filename):
        """
        Public method to get all breakpoint set in the server belonging
        to the file 'filename'.
        """
        bpList = []
        if len(self.breaks):
            for fn, line, cond in self.breaks:
                if fn == filename:
                    bpList.append((line, cond))
        return bpList
        