"""
    FitDialog.py
    Fit filter parameters control dialog.
    
"""

from PyDVT import __version__,__date__,__author__


################################################################################
import qt
import EventHandler
import Command
from Binding import *
from MultiParameters import ParametersTab as Parameters
from PyDVT.FitFilter import DEFAULT_FIT_PARS


class FitParametersDialog(qt.QMainWindow):
    """
    """
    def __init__(self, parent,initial_parameters=None):
        self.Parameters=DEFAULT_FIT_PARS
        if initial_parameters is not None: self.Parameters.update(initial_parameters)
        
        self.eh = None
        self.Destroyed=0
        self.ParametersChange = None
        qt.QMainWindow.__init__(self,parent,"Fit Parameters Editor",0)
        self.table=Parameters(self)
        self.menuAction = qt.QPopupMenu(self)
        self.menuConfig = qt.QPopupMenu(self)        
        self.menuBar().insertItem('&Action',self.menuAction)
        self.menuBar().insertItem('&Config',self.menuConfig)
        
        self.setCentralWidget (self.table)        
        self.statusBar().message('Ready')
        self.setCaption('Fit Parameters Editor')
        self.resize(600,400)

        self.Mode=self.Parameters["Mode"]
        self.Theory=self.Parameters["Theory"]
        self.Background=self.Parameters["Background"]
        self.WeightFlag=self.Parameters["WeightFlag"]
        self.Sensitivity=self.Parameters["Sensitivity"]
        self.FwhmPoints=self.Parameters["FwhmPoints"]
        self.Yscaling=self.Parameters["Yscaling"]
        self.AutoScaling=self.Parameters["AutoScaling"]
        self.AutoFwhm=self.Parameters["AutoFwhm"]
        
        self.TableParameters=[]


    def onInitialize(self,eh=None,fit=None):
        self.menuConfig.clear()
        self.menuAction.clear()
        self.theory_menu_dict={}
        self.back_menu_dict={}
        self.weight_menu_dict={}
        self.mode_menu_dict={}
        self.WeightFlag=self.Theory=self.Background=None
        

        if self.eh is not None:
            self.eh.unregister("FitFinished", self.onFitFinished)
        self.eh = eh
        self.fit = fit
        if self.eh is not None:
            self.ParametersChange = self.eh.create("ParametersChange")
            self.eh.register("FitFinished", self.onFitFinished)
        if fit is not None:
            self.EstimateMenuItem=self.menuAction.insertItem('&Estimate',self.onEstimate,qt.Qt.CTRL + qt.Qt.Key_E)
            self.ActionMenuItem=self.menuAction.insertItem('&StartFit',self.onStartFit,qt.Qt.CTRL + qt.Qt.Key_S)

            self.menuConfigTheory = Menu(self)
            self.menuConfig.insertItem('&Theory',self.menuConfigTheory)
            self.menuConfigBackground = Menu(self)
            self.menuConfig.insertItem('&Background',self.menuConfigBackground)
            self.menuConfigWeight = Menu(self)
            self.menuConfig.insertItem('&Weight',self.menuConfigWeight)
            self.menuConfigMode = Menu(self)
            self.menuConfig.insertItem('&Mode',self.menuConfigMode)
            self.menuConfig.insertItem('&Peak Search',self.OnPeakSearch)
            
            for theory in fit.theorylist:
                self.theory_menu_dict[theory]=self.menuConfigTheory.AddCommand(theory,Command.Command(self.onTheorySelect,theory))
            for background in fit.bkglist:
                self.back_menu_dict[background]=self.menuConfigBackground.AddCommand(background,Command.Command(self.onBackgroundSelect,background))
            for weight_flag in ["Statistical","No weight"]:                
                self.weight_menu_dict[weight_flag]=self.menuConfigWeight.AddCommand(weight_flag,Command.Command(self.onWeightSelect,weight_flag))
            for mode in ["Normal","MCA"]:                
                self.mode_menu_dict[mode]=self.menuConfigMode.AddCommand(mode,Command.Command(self.onModeSelect,mode))
            if fit.fitconfig['WeightFlag']:self.onWeightSelect("Statistical")
            else:self.onWeightSelect("No weight")
            self.onTheorySelect(fit.fitconfig['fittheory'])
            self.onBackgroundSelect(fit.fitconfig['fitbkg'])
            self.onModeSelect(self.Mode)


    def onWeightSelect(self,value):
        for weight_flag in self.weight_menu_dict.keys():
            if value==weight_flag:
                self.menuConfigWeight.CheckItem(self.weight_menu_dict[weight_flag],1)
            else:
                self.menuConfigWeight.CheckItem(self.weight_menu_dict[weight_flag],0)
        self.WeightFlag=value
        

    def onTheorySelect(self,value):
        for theory in self.theory_menu_dict.keys():
            if value==theory:
                self.menuConfigTheory.CheckItem(self.theory_menu_dict[theory],1)
            else:
                self.menuConfigTheory.CheckItem(self.theory_menu_dict[theory],0)
        self.Theory=value
                        
    def onBackgroundSelect(self,value):
        for back in self.back_menu_dict.keys():
            if value==back:
                self.menuConfigBackground.CheckItem(self.back_menu_dict[back],1)
            else:
                self.menuConfigBackground.CheckItem(self.back_menu_dict[back],0)
        self.Background=value

    def onModeSelect(self,value):
        for mode in self.mode_menu_dict.keys():
            if value==mode:
                self.menuConfigMode.CheckItem(self.mode_menu_dict[mode],1)
            else:
                self.menuConfigMode.CheckItem(self.mode_menu_dict[mode],0)
        self.Mode=value
        if self.Mode=="MCA":
            self.menuAction.setItemEnabled( self.EstimateMenuItem, 0 )
        else:
            self.menuAction.setItemEnabled( self.EstimateMenuItem, 1 )

        
    def onNewDoc(self):
        self.statusBar().message('New',3000)

    def onEstimate(self):
        self.statusBar().message('Estimate')
        qt.qApp.processEvents()
        self.Apply("ESTIMATE")

    def onStartFit(self):
        self.statusBar().message('Start Fit')
        qt.qApp.processEvents()
        self.TableParameters= self.table.fillfitfromtable()
        if self.TableParameters is None: self.Apply("ESTIMATE_FIT")
        else: self.Apply("FIT")
        
    def OnPeakSearch(self):
        dlg=Dialog(self,"Peak Search",1,0)
        box=qt.QVBox( dlg )
        
        box1 = qt.QHBox( box )
        box2 = qt.QHBox( box )
        box3 = qt.QHBox( box )
        box4 = qt.QHBox( box )
        box5 = qt.QHBox( box )
        box6 = qt.QHBox( box )
        
        lab1=qt.QLabel(box1)
        ed1=qt.QLineEdit(box1)
        lab2=qt.QLabel(box2)
        ed2=qt.QLineEdit(box2)
        lab3=qt.QLabel(box3)
        ed3=qt.QLineEdit(box3)
        chk1=qt.QCheckBox("Auto YScaling",box4)
        chk2=qt.QCheckBox("Auto Fwhm",box5)
        lab1.setText("Sensitivity")
        ed1.setText(qt.QString(str(self.Sensitivity)))
        lab2.setText("FwhmPoints")
        ed2.setText(qt.QString(str(self.FwhmPoints)))
        lab3.setText("Yscaling")
        ed3.setText(qt.QString(str(self.Yscaling)))
        chk1.setChecked(self.AutoScaling)
        chk2.setChecked(self.AutoFwhm)
        
        button=qt.QPushButton(box6)
        button.setText("OK")
        self.connect(button,qt.SIGNAL("clicked()"),dlg.accept)
        box.resize( 200, 150 )
        if dlg.exec_loop():
            try:
                self.Sensitivity=float(str(ed1.text()))
                self.FwhmPoints=int(str(ed2.text()))
                self.Yscaling=float(str(ed3.text()))
                self.AutoScaling=chk1.isChecked()
                self.AutoFwhm=chk2.isChecked()
            except: pass

    def SetParameters(self,parameters):
        if "WeightFlag" in parameters.keys():
            if parameters["WeightFlag"]: self.onWeightSelect("Statistical")
            else: self.onWeightSelect("No weight")            
        if "Theory" in parameters.keys(): self.onTheorySelect(parameters["Theory"])
        if "Background" in parameters.keys():self.onBackgroundSelect(parameters["Background"])
        if "Mode" in parameters.keys():self.onModeSelect(parameters["Mode"])
        if "Sensitivity" in parameters.keys(): self.Sensitivity=parameters["Sensitivity"]
        if "FwhmPoints" in parameters.keys(): self.FwhmPoints=parameters["FwhmPoints"]
        if "Yscaling" in parameters.keys(): self.Yscaling=parameters["Yscaling"]
        if "AutoScaling" in parameters.keys(): self.AutoScaling=parameters["AutoScaling"]
        if "AutoFwhm" in parameters.keys(): self.AutoFwhm=parameters["AutoFwhm"]
        

    def GetParameters(self):
        if self.WeightFlag=="Statistical": weight_flag=1
        else: weight_flag=0
        return {"Theory":self.Theory,"Mode":self.Mode,"Background":self.Background,"WeightFlag":weight_flag,"Sensitivity":self.Sensitivity,"FwhmPoints":self.FwhmPoints,"Yscaling":self.Yscaling,"AutoScaling":self.AutoScaling,"AutoFwhm":self.AutoFwhm}

    
    def onFitFinished(self,parameters):
        self.statusBar().message('Ready')
        if self.Mode=="MCA": self.table.fillfrommca(parameters)
        else:
            self.table.fillfromfit(parameters,current="Region 1")
            for view in self.table.tables.keys():
                if view != "Region 1":
                    self.table.removeview(view)
        
    def Show(self):
        self.show()

    def hideEvent(self,e):
        self.Destroyed=1
        qt.QMainWindow.hideEvent(self,e)

    def IsDestroyed(self):
        return self.Destroyed

    
    def Apply(self,action="FIT"):
        """
        Called by ColormapEditorWidget then change takes place
        """
        if self.eh is not None:
            pars={}
            if self.WeightFlag=="Statistical": pars["WeightFlag"]=1
            else: pars["WeightFlag"]=0
            pars["Theory"]=self.Theory
            pars["Background"]=self.Background
            pars["Sensitivity"]=self.Sensitivity
            pars["FwhmPoints"]=self.FwhmPoints
            pars["Yscaling"]=self.Yscaling
            pars["AutoScaling"]=self.AutoScaling
            pars["AutoFwhm"]=self.AutoFwhm
            pars["Mode"]=self.Mode
            pars["Action"]=action
            pars["FitPars"]=self.TableParameters
            self.eh.event(self.ParametersChange,pars)


if __name__ == "__main__":
    import sys    
    app = qt.QApplication(sys.argv)
    mw = FitParametersDialog(None,None)
    app.setMainWidget (mw)
    mw.show()
    app.exec_loop()

