#!/usr/bin/env python
"""
Implement a class for fitting data.
"""
from Tkinter import *
from Tkinter import _cnfmerge
from Scientific.Functions.LeastSquares import leastSquaresFit
from Numeric import *
import tkSimpleDialog
import Selector
import ScrollFrame
import EventHandler
import SpecArithmetic
from Gefit import LeastSquaresFit
from Gefit import test
import SpecfitFuns
import SpecfitConf
from prnconf import *
from AppPrint import *
#import SimplePlot
import PsCanvas
from ScriptOption import *
import os
import string

DEBUG=0
 
SPECFITDEFAULTS={'FileAction':0,
                 'infile':os.environ['HOME']+'/.specfitdefaults.py',
                 'outfile':os.environ['HOME']+'/.specfitdefaults.py',
                 'Geometry':"600x400+50+50",
                 'NoConstrainsFlag':0,
                 'BackgroundIndex':1,
                 'TheoryIndex':0,
                 'PosFwhmFlag':1,
                 'HeightAreaFlag':0,
                 'SameFwhmFlag':1,
                 'PositionFlag':0,
                 'EtaFlag':0,
                 'WeightFlag':0,
                 'Yscaling':1.0,
                 'Xscaling':1.0}

class Specfit (Toplevel):
    def __init__(self,master,xdata,ydata,sigmay=None,
                 xmin=None,xmax=None,eh=None,
                 source='',
                 scan='',yname='',moncounter='none',mondata=[],
                 xname='',user_theory='External Function',user_estimate=None,
                 dataupdate=None,user_parameters=None,
                 user_function=None,prnconf=None,
                 geometry=None,defaultsfile=None,cnf={},**kw):
       """
       Class constructor.
       Create parameters widget.
       eh is the event handler of the application.
       source and scan are the source name of data, and the scan name.
       xdata and ydata are the data arrays, xname and yname are their names
       """
       cnf = _cnfmerge((cnf,kw))
       Toplevel.__init__(self,master,cnf)
       #create the menu
       self.title('Specfit')
       self.conf = SpecfitConf.SpecfitConf(self,configfile=defaultsfile)
       self.conf.grid(row=0,column=0,sticky='nw')
       self.conf.fitconfigupdated.trace('w',self.updatefromconfig)
       self.fitconfig = self.conf.fitconfig
       if geometry is not None:
            try:
                self.geometry(geometry)
            except:
                try:
                    self.geometry(self.fitconfig['Geometry'])
                except:
                    self.geometry(SPECFITDEFAULTS['Geometry'])
       else:
            self.geometry(SPECFITDEFAULTS['Geometry'])
                
       if eh is None:
            self.eh = EventHandler.EventHandler()
       else:
            self.eh = eh
            
       self.master = master
       self.source = source
       self.scan   = scan
       self.dataupdate = dataupdate
       
       if prnconf is not None:
            self.prnconf=prnconf
       else:
            self.prnconf=PrintConfig()

       self.prnconf.childstc['Specfit'] = ['', '', None, None, 1.0]

       self.conf.create_menu(["Print",self.menuprintps])

       self.fit_theory = {
           'Background & Gaussians' : ['Height','Position','Fwhm'],
           'Background & Lorentz'   : ['Height','Position','Fwhm'],
           'Back. Area Gaussians'   : ['Area','Position','Fwhm'],
           'Back. Area Lorentz'     : ['Area','Position','Fwhm'],
           'Pseudo-Voigt Line'     : ['Height','Position','Fwhm','Eta'],
           'Area Pseudo-Voigt'     : ['Area','Position','Fwhm','Eta'],
           'Back. & Step Down'      : ['Height','Position','Fwhm'],
           'Back. & Step Up'        : ['Height','Position','Fwhm']}
       
       self.theory_list = []
       for i in self.conf.theory_list:
            self.theory_list.append(i)
       

       self.bkg_list = ['Constant',
                        'Linear'  ,
                        'Exponential',
                        'Internal',
                        'No Background']

       self.bkg_theory = {'Constant': ['Constant'],
                          'Linear'   :['Constant','Slope'],
                          'Exponential':['Factor','Slope'],
                          'Internal':['Curvature','Iterations'],
                          'No Background':[]}

       self.bkg_functions ={'Constant':self.bkg_constant,
                            'Linear':self.bkg_linear,
                            'Exponential':self.bkg_linear,
                            'Internal':self.bkg_internal,
                            'No Background':self.bkg_none}
                            

       self.fit_functions ={'Background & Gaussians': self.gauss,
                            'Background & Lorentz':self.lorentz,
                            'Back. Area Gaussians':self.agauss,
                            'Back. Area Lorentz':self.alorentz,
                            'Pseudo-Voigt Line':self.pvoigt,
                            'Area Pseudo-Voigt':self.apvoigt,
                            'Back. & Step Down':self.stepdown,
                            'Back. & Step Up':self.stepup}

       self.estimate_fun={'Background & Gaussians': self.estimate_gauss,
                            'Background & Lorentz':self.estimate_lorentz,
                            'Back. Area Gaussians':self.estimate_agauss,
                            'Back. Area Lorentz':self.estimate_alorentz,
                            'Pseudo-Voigt Line':self.estimate_pvoigt,
                            'Area Pseudo-Voigt':self.estimate_apvoigt,
                            'Back. & Step Down':self.estimate_stepdown,
                            'Back. & Step Up':self.estimate_stepup}

                            
       if user_function is not None:
            if user_parameters is not None:
                #self.fit_theory[user_theory]=[]
                #for k in user_parameters: 
                #    self.fit_theory[user_theory].append(k+`1`)
                self.fit_theory[user_theory]=user_parameters
                self.fit_functions[user_theory]=user_function
                self.fitconfig['TheoryIndex']=len(self.theory_list)
                if user_estimate is not None:
                    self.estimate_fun[user_theory]=user_estimate
                else:
                    self.estimate_fun[user_theory]=self.estimate_none
                self.theory_list.append(user_theory)
            else:
                print "Using default theory"
    
       #select the background
       self.bkg_fun = self.bkg_functions[self.bkg_list[self.fitconfig['BackgroundIndex']]]
    
       #select the function
       try:
            self.selected_th = self.theory_list[self.fitconfig['TheoryIndex']]
            self.fit_fun = self.fit_functions[self.selected_th]
       except:
            self.fitconfig['TheoryIndex']=0
            self.selected_th = self.theory_list[self.fitconfig['TheoryIndex']]
            self.fit_fun = self.fit_functions[self.selected_th]
       
       #set the list of parameters
       self.final_theory=[]
       for i in self.bkg_theory[self.bkg_list[self.fitconfig['BackgroundIndex']]]:
            self.final_theory.append(i)
       for i in self.fit_theory[self.selected_th]:
            self.final_theory.append(i+'1') 
                
       self.selector  = Selector.Selector(self,self.theory_list,
                                          command=self.make_selection)
       self.selector.value.set(self.theory_list[self.fitconfig['TheoryIndex']])
       self.weight_options = ['No Weight','Statistical']
       self.weight_but     = Selector.Selector(self,self.weight_options,
                                            command=self.setweight)
       self.weight = self.fitconfig['WeightFlag']

       self.build()
       self.pretreat(xdata,ydata,sigmay=sigmay,
                 xmin=xmin,xmax=xmax,
                 source=source,mondata=mondata,moncounter=moncounter,
                 scan=scan,yname=yname,
                 xname=xname)
       self.weight_but.value.set(self.weight_options[self.weight])
       #self.build()
       #self.estimate()

    def fit_fun0(self,pars,t):
        nb=len(self.bkg_theory[self.bkg_list[self.fitconfig['BackgroundIndex']]])
        #treat differently user and built in functions
        if self.selected_th in self.conf.theory_list:
            if (nb>0):
                return self.bkg_fun(pars[0:nb],t) + self.fit_fun(pars[nb:len(pars)],t)
            else:
                return self.fit_fun(pars,t)
        else:
            nu=len(self.fit_theory[self.selected_th])
            niter=(len(pars)-nb)/nu
            u_term=zeros(shape(t),Float)
            if niter > 0:
                for i in range(niter):
                    u_term= u_term+ \
                            self.fit_fun(pars[(nb+i*nu):(nb+(i+1)*nu)],t)
            if (nb>0):
                return self.bkg_fun(pars[0:nb],t) + u_term
            else:
                return u_term
       
    def menuprintps(self,*args):
        self.printps(fname='')

    def updatefromconfig(self,*args):
       self.yscaling.set(self.fitconfig['Yscaling'])
       self.weight = self.fitconfig['WeightFlag'] 
       self.weight_but.value.set(self.weight_options[self.weight])
       self.selector.value.set(self.theory_list[self.fitconfig['TheoryIndex']])
       if self.selected_th != self.theory_list[self.fitconfig['TheoryIndex']]:
            self.make_selection(self.theory_list[self.fitconfig['TheoryIndex']])
       self.bkg_fun = self.bkg_functions[self.bkg_list[self.fitconfig['BackgroundIndex']]]


    def pretreat(self,xdata,ydata,sigmay=None,
                 xmin=None,xmax=None,
                 source='',mondata=[],moncounter='none',
                 scan='',yname='',
                 xname=''):
       self.source = source
       self.scan   = scan

       self.xdata0  = xdata
       self.ydata0  = ydata
       if yname:
            self.yname = '%s (fit)' % yname
       else:
            self.yname = 'Y (fit)'
       if xname:
            self.xname = xname
       else:
            self.xname = 'X'
       if xmin is not None:
            self.xmin = xmin
       else:
            self.xmin = min(xdata)
       if xmax is not None:
            self.xmax = xmax
       else:
            self.xmax = max(xdata) 
       #make sure data are ordered
       i1 = argsort(xdata)
       self.xdata = take(xdata,i1)
       self.ydata = take(ydata,i1)
       if len(mondata) > 0:
            self.mondata0 = mondata
            self.mondata = take(mondata,i1)
       else:
            self.mondata0 = ones(self.ydata.shape,Float)
            self.mondata = ones(self.ydata.shape,Float)
       if sigmay is not None:
            self.sigmay = take(sigmay,i1)
            #self.weight = 1
       else:
            self.sigmay = None
       # take the values between limits
       i1 = nonzero(greater(self.xdata,self.xmin))
       self.xdata = take(self.xdata,i1)
       self.ydata = take(self.ydata,i1)
       self.mondata = take(self.mondata,i1)
       if self.sigmay is not None:
            self.sigmay = take(self.sigmay,i1)
       i1 =  nonzero(less(self.xdata,self.xmax))     
       self.xdata = take(self.xdata,i1)
       self.ydata = take(self.ydata,i1)
       self.mondata = take(self.mondata,i1)
       if self.sigmay is not None:
            self.sigmay = take(self.sigmay,i1)
       self.ydata = self.ydata / self.mondata
       #OK with the pre-treatment
       self.weight_but.value.set(self.weight_options[self.weight])       
       #Update build in order to show the changesself.buildupdate()
       self.head_lbl.configure(text='Fit scan %s of %s' %(self.scan,self.source))
          
    def build(self):
       """
       Create the widget.
       """

       #self.buildmenu()
       self.head_lbl  = Label(self,
                             text='Fit scan %s of %s' %(self.scan,self.source),
                             relief='groove',bd=1)
                             
       lbl_selector   = Label(self,text='Fit theory : ')

       lbl_weight     = Label(self,text='Weight : ')
       lbl_yscaling   = Label(self,text='Y scaling : ')
       self.yscaling  = DoubleVar()
       self.yscaling.set(self.fitconfig['Yscaling'])
       yscaling       = Entry(self,bg='#eec',
                            justify='right',textvariable=self.yscaling)
       self.yscaling.trace('w',self.updateyscaling)

       state_frame    = Frame(self,relief='ridge',bd=2)
       lbl_state      = Label(state_frame,text='Fit state : ',anchor='w')
       self.state     = StringVar()
       state          = Label(state_frame,textvariable=self.state,bg='#ffffff')
       lbl_chi        = Label(state_frame,text='Chi sqr :')
       self.chi       = DoubleVar()
       chi            = Label(state_frame,textvariable=self.chi,bg='#ffffff')
       state_frame.columnconfigure((1,3),weight=1)

       roff=1
       lbl_state.     grid(row=0+roff,column=0,sticky='w')
       state.         grid(row=0+roff,column=1,sticky='we')
       lbl_chi.       grid(row=0+roff,column=2,sticky='w')
       chi.           grid(row=0+roff,column=3,sticky='w')

       estimate_but   = Button(self,text='Estimate',command=self.estimate)
       fit_but        = Button(self,text='Start fit',command=self.startfit)
       dismiss_but    = Button(self,text='Dismiss',command=self.exit)
       #print_but      = Button(self,text='Print',command=self.printps)

       self.head_lbl. grid(row=0+roff,column=0,sticky='n',columnspan=3)
       lbl_selector.  grid(row=1+roff,column=0,sticky='w')
       self.selector. grid(row=1+roff,column=1,sticky='w')  
       #param          = self.fit_theory[self.selected_th]
       param          = self.final_theory
       self.param_win = Parameter(self,param,relief='ridge',bd=2)
       self.param_win.grid(row=2+roff,column=0,sticky='nsew',columnspan=3)
       state_frame.   grid(row=3+roff,column=0,sticky='new',columnspan=3)
       estimate_but.  grid(row=4+roff,column=0,sticky='w')
       fit_but.       grid(row=4+roff,column=1)
       dismiss_but.   grid(row=4+roff,column=2,sticky='e')
       #print_but.     grid(row=0+roff,column=0,sticky='w')
       lbl_weight.    grid(row=0+roff,column=2,sticky='w')
       self.weight_but.grid(row=0+roff,column=2,sticky='e')
       lbl_yscaling.  grid(row=1+roff,column=2,sticky='w')
       yscaling.      grid(row=1+roff,column=2,sticky='e')
       self.columnconfigure((0,1,2),weight=1)

       self.state.set('Ready')

    def updateyscaling(self,*args):
        old = self.fitconfig['Yscaling']
        try: 
            self.fitconfig['Yscaling'] = self.yscaling.get()
        except:
            self.fitconfig['Yscaling'] = old
            return

    
    def buildmenu(self):
        #COMMAND   = 1
        #SEPARATOR = 2
        #CASCADE   = 3
        #config_menu  = ["Config", None,
        #                  [COMMAND, "Configure",  self.fitconfig],
        #                  [SEPARATOR, ],
        #                  [COMMAND, "Load Config",  self._loadconfig],
        #                  [COMMAND, "Save Config",  self.saveconfig],
        #                  [SEPARATOR,],
        #                  [COMMAND, "Load Default",  self.loaddefaults],
        #                  [COMMAND, "Save As Default", self.savedefaults],
        #                  [SEPARATOR, ],
        #                  [COMMAND, "Exit",       None]   ]

        self.menu = SpecfitConf.SpecfitConf(self)
        #self.menu.create_menu(config_menu,side='left')
        self.menu.grid(row=0,column=0)
        #self.menu.create_menu(config_menu)
        #self.menu.pack()     
       

    def printps(self,fname=''):
       if fname == '':
            #Assuming it is called by itself
            # Try a recursive solution
            dict = {}
            dict['Specfit'] = (self,Specfit.printps)
            AppPrint(self,text='',
                dict=dict,prnconf=self.prnconf)
       else:     
            param_list  = self.fit_theory[self.selected_th]
            flagconstrains=0
            prin = "FIT Theory = %s CHISQ = %6.3g\n" % (self.selected_th,self.chi.get())
            prin += "Parameter   Estimation  Fit result    Sigma     Code    Min/Par  Max/Del/Fac\n"  
            # get the maximum parameter name length
            maxlen = 0
            for param in param_list:
                maxlen = max(maxlen,len(param))
            for param in param_list:
                adjust = param
                #print adjust
                while len(adjust) <= maxlen:
                    adjust = adjust + ' '
                group = self.param_win.parameters [param] ['group'].get()
                estimation = self.param_win.parameters [param] ['estimation'].get()
                fitresult = self.param_win.parameters [param] ['fitresult'].get()
                sigma = self.param_win.parameters [param] ['sigma'].get()
                code = self.param_win.parameters [param] ['code'].get()
                cons1 = self.param_win.parameters [param] ['cons1'].get()
                cons2 = self.param_win.parameters [param] ['cons1'].get()
                prin += "%s %+6.4e %+6.4e %+6.3e %8s %+6.4e %+6.4e\n" %  (adjust,
                    estimation,fitresult,sigma,code,cons1,cons2)
            pgsize = self.prnconf.pgsize
            pgbd = self.prnconf.pgborders
            width = "%fm" % (pgsize[0] - (pgbd[2] + pgbd[3]))
            height = "%fm" % (pgsize[1] - (pgbd[0] + pgbd[1]))
            cvs=PsCanvas.PsCanvas(self.master,width=width,height=height)
            #tp = Toplevel()
            #cvs=Canvas(tp,width=width,height=height)
            a=cvs.create_text(0,
                       0,
                       text=prin,font=('Courier',10),
                       anchor=NW)
            bbox = cvs.bbox(a)
            #cvs.pack()
            #cvs.update()
            pwidth = "%fm" % pgsize[0]
            pheight = "%fm" % pgsize[1]
            x = "%fm" % pgbd[2]
            if self.prnconf.pglandscape:
                # landscape: lower border (inverted)
                y = "%fm" % pgbd[1]
            else:
                # portrait: y increase upwards
                y = "%fm" % (pgsize[1] - pgbd[0])
            #fwidth = "%fm" % (pgsize[0] * blst[4])
            fwidth = "%fm" % (pgsize[0] * 1.0)
            #print "creating the file ",fname
            cvs.postscript(width=pwidth,height=pheight,pagex=x,pagey=y,pageanchor=NW,
                       pagewidth=fwidth, rotate=self.prnconf.pglandscape,
                       file=fname,
                       colormode=self.prnconf.colormode)
            #cvs.postscript(width=pwidth,height=pheight,pagex=x,pagey=y,pageanchor=NW,
            #           pagewidth=fwidth, rotate=self.prnconf.pglandscape,
            #           file='specfit.ps',
            #           colormode=self.prnconf.colormode)

    def make_selection(self,item):
       """
       Callback to the type selection of fit fuction.
       The associated parameters are then displayed.
       """
       if item == self.selected_th:
           return

       #print "item = ", item
       #self.param_win.    destroy()
       self.fitconfig['TheoryIndex'] = self.theory_list.index(item)
       self.selected_th = item
       self.fit_fun = self.fit_functions[item]       
       #param            = self.fit_theory[item]
       #set the list of parameters
       self.final_theory=[]
       for i in self.bkg_theory[self.bkg_list[self.fitconfig['BackgroundIndex']]]:
            self.final_theory.append(i)
       for i in self.fit_theory[self.selected_th]:
            self.final_theory.append(i+'1') 
       param = self.final_theory
       self.param_win   = Parameter(self,param)
       self.param_win.    grid(row=3,column=0,sticky='nsew',columnspan=3)
      
       #self.estimate()
       self.state.set('Ready')
       self.chi.set('')
       
    def setweight(self,item):
        #if item == self.weight_options [0]:
        #    self.weight = 0
        #elif item == self.weight_options [1]:
        #    self.weight = 1
        self.weight = self.weight_options.index(item)
        self.fitconfig['WeightFlag'] = self.weight
    def exit(self):
       self.destroy()

    def startfit(self):
       """
       Check the parameters validity, launch the fit routine, and
       display the result.
       """
       if self.dataupdate is not None:
            self.dataupdate()
       self.state.set('Fit in progress')
       self.update()

       #param_list  = self.fit_theory[self.selected_th]
       param_list = self.final_theory
       length      = len(param_list)
       param_val   = []
       
       param_constrains   = [[],[],[]]
       flagconstrains=0
       for param in param_list:
            val,constrains = self.param_win.cget(param)
            if constrains [0] != 0:
                #print "%s constrains = %g %g %g" % (param,constrains[0],constrains[1],constrains[2])
                flagconstrains=1
            if (string.find(param,"Height")>=0) or \
               (string.find(param,"Area")>=0) or \
               (string.find(param,"Slope")>=0) or \
               (string.find(param,"Constant")>=0):
                param_val.append(val*self.fitconfig['Yscaling'])
            else:
                param_val.append(val)            
            param_constrains [0].append(constrains [0])
            param_constrains [1].append(constrains [1])
            param_constrains [2].append(constrains [2])

       data = []
       i = 0
       for xval in self.xdata:
            if self.sigmay is None:
                data.append([xval,self.ydata[i]*self.fitconfig['Yscaling']])
            else:
                data.append([xval,self.ydata[i]*self.fitconfig['Yscaling'],
                            self.sigmay[i]*sqrt(self.fitconfig['Yscaling'])])
#                            self.sigmay[i]])
            i = i + 1
     
       try:
           if flagconstrains != 1:
                found = LeastSquaresFit(self.fit_fun0,param_val,data,
                        weightflag=self.weight)
           else:
                found = LeastSquaresFit(self.fit_fun0,param_val,data,                
                    constrains=param_constrains,weightflag=self.weight)
       except 'LinearAlgebraError' :
           text = sys.exc_info()[1]
           if type(text) is not string._StringType: 
              text = text.args
              if len(text):
                 text = text[0]
              else:
                 text = ''
           self.state.set('Fit error : %s' %text)
           return

       i = 0
       noigno = []
       for name in param_list:
           if self.param_win.parameters[name]['code'].get() != 'IGNORE':
              noigno.append(i)
              if (string.find(name,"Height")>=0) or \
                 (string.find(name,"Area")>=0) or \
                 (string.find(name,"Slope")>=0) or \
                 (string.find(name,"Constant")>=0):
                       found[0][i]=found[0][i]/self.fitconfig['Yscaling']
                       found[2][i]=found[2][i]/self.fitconfig['Yscaling']
           self.param_win.configure(name=name,fitresult=found[0][i])
           self.param_win.configure(name=name,sigma=found[2][i])
           i = i + 1

       newdata = self.fit_fun0(take(found[0],noigno).tolist(),array(self.xdata0))
       newdata = newdata * self.mondata0
       #Event.ResetEvent(self,self.eh)
       #XNewDataEvent=self.eh.create('XNewDataEvent')
       #self.eh.event(XNewDataEvent,data={'xname' : self.xname,
       #                                       'xdata' : newdata[0]} )
       YNewDataEvent=self.eh.create('YNewDataEvent')
       self.eh.event(YNewDataEvent,data={'yname' : self.yname,
                                              'xname' : self.xname,
                                              'ydata' : newdata,
                                              'color' : 'red'}      )


       self.chi.set(found[1]/self.fitconfig['Yscaling'])
       self.state.set('Ready')


    def estimate(self):
       """
       Fill the parameters entries with an estimation made on the given data.
       """
       CONS=['FREE',
            'POSITIVE',
            'QUOTED',
            'FIXED',
            'FACTOR',
            'DELTA',
            'IGNORE']
       
       #make sure data are current
       if self.dataupdate is not None:
            self.dataupdate()

       xx = self.xdata
       yy = self.ydata

       #estimate the background
       esti_bkg=self.estimate_bkg()
       bkg_esti_parameters = esti_bkg[0]
       bkg_esti_constrains = esti_bkg[1]
       zz                  = array(esti_bkg[2])
       
       #estimate the function
       esti_fun=self.estimate_fun[self.selected_th]
       estimation = esti_fun(xx,yy,zz,yscaling=self.yscaling.get())
       fun_esti_parameters = estimation[0]
       fun_esti_constrains = estimation[1]
       
       #estimations are made
       #build the names
       self.final_theory=[]
       for i in self.bkg_theory[self.bkg_list[self.fitconfig['BackgroundIndex']]]:
            self.final_theory.append(i)       
       i=0
       j=1
       while (i < len(fun_esti_parameters)):             
             for k in self.fit_theory[self.selected_th]: 
                self.final_theory.append(k+`j`)
                i=i+1
             j=j+1      
             
       self.param_win.destroy()
       param          = self.final_theory
       self.param_win = Parameter(self,param,relief='ridge',bd=2)
       self.param_win.grid(row=3,column=0,sticky='nsew',columnspan=3)
       j=0
       i=0
       k=0
       for pname in self.final_theory:
            if i < len(bkg_esti_parameters):
                self.param_win.configure(name=pname,
                                         estimation=bkg_esti_parameters[i],
                                         group=0,
                                         code=CONS[int(bkg_esti_constrains[0][j])],
                                         cons1=bkg_esti_constrains[1][j],
                                         cons2=bkg_esti_constrains[2][j])
                i=i+1
            else:
                if (j % len(fun_esti_parameters)) == 0:
                    k=k+1
                #print k
                if (CONS[int(fun_esti_constrains[0][j])] == "FACTOR") or \
                   (CONS[int(fun_esti_constrains[0][j])] == "DELTA"):
                        fun_esti_constrains[1][j] = fun_esti_constrains[1][j] +\
                                                    len(bkg_esti_parameters)
                self.param_win.configure(name=pname,
                                         estimation=fun_esti_parameters[j],
                                         group=k,
                                         code=CONS[int(fun_esti_constrains[0][j])],
                                         cons1=fun_esti_constrains[1][j],
                                         cons2=fun_esti_constrains[2][j])
                j=j+1

    def estimate_bkg(self):       
       xx = self.xdata
       yy = self.ydata
       if self.fitconfig['Yscaling'] == 0:
            self.yscaling.set(1.0)
                    
       yy = yy * self.fitconfig['Yscaling']
       self.zz=SpecfitFuns.subac(yy,1.0001,1000)
       zz = self.zz
       #SimplePlot.plot([xx,yy,zz])
       npoints = len(zz)
       if self.fitconfig['BackgroundIndex'] == 0:       
            #Constant background
            S = float(npoints)
            Sy = sum(zz)
            fittedpar=[(Sy/S)/self.fitconfig['Yscaling']]
       else:
            S = float(npoints)
            Sy = sum(zz)
            Sx = float(sum(xx))
            Sxx = float(sum(xx * xx))            
            Sxy = float(sum(xx * zz))
                                
            deno = S * Sxx - (Sx * Sx)
            bg = (Sxx * Sy - Sx * Sxy)/deno
            slop = (S * Sxy - Sx * Sy)/deno        
            fittedpar=[bg/self.fitconfig['Yscaling'],slop/self.fitconfig['Yscaling']]
       cons = zeros((3,len(fittedpar)),Float)
       return fittedpar,cons,zz

    def estimate_gauss(self,xx,yy,zzz,xscaling=1.0,yscaling=1.0):
       #xx = self.xdata
       #yy = self.ydata
       if yscaling == 0:
            yscaling=1.0
                    
       yy = yy * yscaling
       
       zz=SpecfitFuns.subac(yy,1.000,10000)
       #SimplePlot.plot([xx,yy,zz])
       npoints = len(zz)
       if npoints > 15:
            peaks = SpecfitFuns.seek(yy,1,npoints,10,3.25)
       else:
            peaks = []
       #print "peaks subac = ",SpecfitFuns.seek(yy-zz,1,npoints,4,3.25)
       largest_index = 0
       if len(peaks) > 0:
            j = 0
            for i in peaks:
                if j == 0:
                    sig=5*abs(xx[npoints-1]-xx[0])/npoints
                    param = array([yy[int(i)] - zz [int(i)], xx[int(i)] ,sig])
                    largest = param
                else:
                    param2 = array([yy[int(i)] - zz [int(i)], xx[int(i)] ,sig])
                    param = concatenate((param,param2),1)
                    if (param2[0] > largest[0]):
                        largest = param2
                        largest_index = j
                j = j + 1
            xw = resize(xx,(npoints,1))
            sy = sqrt(abs(yy))
            yw = resize(yy-zz,(npoints,1))
            sy = resize(sy,(npoints,1))
            datawork = concatenate((xw,yw,sy),1)
            cons = zeros((3,len(param)),Float)
            cons [0] [2:len(param):3] = 1.0
            fittedpar, chisq, sigmapar = LeastSquaresFit(SpecfitFuns.gauss,param,
                                            datawork,maxiter=4,constrains=cons.tolist())
            #I already have the estimation
       else:
            #Use SPEC estimate ...
            peakpos,height,myidx = SpecArithmetic.search_peak(xx,yy-zz)
            fwhm,cfwhm         = SpecArithmetic.search_fwhm(xx,yy-zz,
                                           peak=peakpos,index=myidx) 
            xx = array(xx)
            if npoints > 2:
                #forget SPEC estimation of fwhm
                fwhm=5*abs(xx[npoints-1]-xx[0])/npoints
            fittedpar=[height,peakpos,fwhm]
            largest=array([height,peakpos,fwhm])
            peaks=[peakpos]
            #cons = zeros((3,len(param)),Float)
       cons = zeros((3,len(fittedpar)),Float)
       j=0
       for i in range(len(peaks)):
                #Setup height area constrains
                if self.fitconfig['NoConstrainsFlag'] == 0:
                    if self.fitconfig['HeightAreaFlag']:
                        #POSITIVE = 1
                        cons[0] [j] = 1
                        cons[1] [j] = 0
                        cons[2] [j] = 0
                j=j+1
                
                
                #Setup position constrains
                if self.fitconfig['NoConstrainsFlag'] == 0:
                    if self.fitconfig['PositionFlag']:
                                #QUOTED = 2
                                cons[0][j]=2
                                cons[1][j]=min(xx)
                                cons[2][j]=max(xx) 
                j=j+1
                
                #Setup positive FWHM constrains
                if self.fitconfig['NoConstrainsFlag'] == 0:
                    if self.fitconfig['PosFwhmFlag']:
                                #POSITIVE=1
                                cons[0][j]=1
                                cons[1][j]=0
                                cons[2][j]=0
                    if self.fitconfig['SameFwhmFlag']:
                        if (i != largest_index):
                                #FACTOR=4
                                cons[0][j]=4
                                cons[1][j]=3*largest_index+2
                                cons[2][j]=1.0
                j=j+1
       return fittedpar,cons 
       
    def estimate_lorentz(self,xx,yy,zzz,xscaling=1.0,yscaling=1.0):
       fittedpar,cons = self.estimate_gauss(xx,yy,zzz,xscaling,yscaling)
       return  fittedpar,cons

    def estimate_pvoigt(self,xx,yy,zzz,xscaling=1.0,yscaling=1.0):
       fittedpar,cons = self.estimate_gauss(xx,yy,zzz,xscaling,yscaling)
       npeaks=len(cons[0])/3
       newpar=[]
       newcons=zeros((3,4*npeaks),Float)
       for i in range(npeaks):
            newpar.append(fittedpar[3*i])
            newpar.append(fittedpar[3*i+1])
            newpar.append(fittedpar[3*i+2])
            newpar.append(0.5)
            newcons[0][4*i]=cons[0][3*i]
            newcons[0][4*i+1]=cons[0][3*i+1]
            newcons[0][4*i+2]=cons[0][3*i+2]
            newcons[1][4*i]=cons[1][3*i]
            newcons[1][4*i+1]=cons[1][3*i+1]
            newcons[1][4*i+2]=cons[1][3*i+2]
            newcons[2][4*i]=cons[2][3*i]
            newcons[2][4*i+1]=cons[2][3*i+1]
            newcons[2][4*i+2]=cons[2][3*i+2]
            #Eta constrains
            newcons[0][4*i+3]=0
            newcons[1][4*i+3]=0
            newcons[2][4*i+3]=0
            if self.fitconfig['NoConstrainsFlag'] == 0:
                if self.fitconfig['EtaFlag']:
                      #QUOTED=2
                      newcons[0][4*i+3]=2
                      newcons[1][4*i+3]=0.0              
                      newcons[2][4*i+3]=1.0              
       return  newpar,newcons

    def estimate_agauss(self,xx,yy,zzz,xscaling=1.0,yscaling=1.0):
       fittedpar,cons = self.estimate_gauss(xx,yy,zzz,xscaling,yscaling)
       #get the number of found peaks
       npeaks=len(cons[0])/3
       for i in range(npeaks):
            height = fittedpar[3*i]
            fwhm = fittedpar[3*i+2]
            fittedpar[3*i] = (height * fwhm / (2.0*sqrt(2*log(2))))*sqrt(2*pi)
       return fittedpar,cons

    def estimate_alorentz(self,xx,yy,zzz,xscaling=1.0,yscaling=1.0):
       fittedpar,cons = self.estimate_gauss(xx,yy,zzz,xscaling,yscaling)
       #get the number of found peaks
       npeaks=len(cons[0])/3
       for i in range(npeaks):
            height = fittedpar[3*i]
            fwhm = fittedpar[3*i+2]
            fittedpar[3*i] = (height * fwhm * 0.5 *pi)
       return fittedpar,cons
       
    def estimate_apvoigt(self,xx,yy,zzz,xscaling=1.0,yscaling=1.0):
       fittedpar,cons = self.estimate_pvoigt(xx,yy,zzz,xscaling,yscaling)
       npeaks=len(cons[0])/4
       for i in range(npeaks):
            height = fittedpar[4*i]
            fwhm = fittedpar[4*i+2]
            fittedpar[4*i] = 0.5*(height * fwhm * 0.5 *pi)+\
                         0.5*(height * fwhm / (2.0*sqrt(2*log(2))))*sqrt(2*pi)
       return fittedpar,cons
       
    def estimate_stepdown(self,xxx,yyy,zzz,xscaling=1.0,yscaling=1.0):
        crappyfilter=[-0.25,-0.75,0.0,0.75,0.25]
        cutoff = 2
        yy=convolve(yyy,crappyfilter,mode=1)[2:-2]
        if max(yy) > 0:
            yy = yy * max(yyy)/max(yy)
        xx=xxx[2:-2]
        fittedpar,cons=self.estimate_agauss(xx,yy,zzz,xscaling,yscaling)
        npeaks=len(cons[0])/4
        largest_index=0
        largest=[fittedpar[3*largest_index],
                 fittedpar[3*largest_index+1],
                 fittedpar[3*largest_index+2]]
        newcons=zeros((3,3),Float)
        for i in range(npeaks):
            if fittedpar[3*i] > largest[0]:
                largest_index=i
                largest=[fittedpar[3*largest_index],
                         fittedpar[3*largest_index+1],
                         fittedpar[3*largest_index+2]]                 
        #Setup constrains
        if self.fitconfig['NoConstrainsFlag'] == 0:
                #Setup height constrains
                if self.fitconfig['HeightAreaFlag']:
                        #POSITIVE = 1
                        cons[0] [0] = 1
                        cons[1] [0] = 0
                        cons[2] [0] = 0

                #Setup position constrains
                if self.fitconfig['PositionFlag']:
                                #QUOTED = 2
                                cons[0][1]=2
                                cons[1][1]=min(xxx)
                                cons[2][1]=max(xxx) 

                #Setup positive FWHM constrains
                if self.fitconfig['PosFwhmFlag']:
                                #POSITIVE=1
                                cons[0][2]=1
                                cons[1][2]=0
                                cons[2][2]=0
                        
        return largest,cons
    
    def estimate_stepup(self,xx,yyy,zzz,xscaling=1.0,yscaling=1.0):
        crappyfilter=[0.25,0.75,0.0,-0.75,-0.25]
        cutoff = 2
        yy=convolve(yyy,crappyfilter,mode=1)[2:-2]
        if max(yy) > 0:
            yy = yy * max(yyy)/max(yy)
        xx=xxx[2:-2]
        fittedpar,cons=self.estimate_agauss(xx,yy,zzz,xscaling,yscaling)
        npeaks=len(cons[0])/4
        largest_index=0
        largest=[fittedpar[3*largest_index],
                 fittedpar[3*largest_index+1],
                 fittedpar[3*largest_index+2]]
        newcons=zeros((3,3),Float)
        for i in range(npeaks):
            if fittedpar[3*i] > largest[0]:
                largest_index=i
                largest=[fittedpar[3*largest_index],
                         fittedpar[3*largest_index+1],
                         fittedpar[3*largest_index+2]]                 
        #Setup constrains
        if self.fitconfig['NoConstrainsFlag'] == 0:
                #Setup height constrains
                if self.fitconfig['HeightAreaFlag']:
                        #POSITIVE = 1
                        cons[0] [0] = 1
                        cons[1] [0] = 0
                        cons[2] [0] = 0

                #Setup position constrains
                if self.fitconfig['PositionFlag']:
                                #QUOTED = 2
                                cons[0][1]=2
                                cons[1][1]=min(xxx)
                                cons[2][1]=max(xxx) 

                #Setup positive FWHM constrains
                if self.fitconfig['PosFwhmFlag']:
                                #POSITIVE=1
                                cons[0][2]=1
                                cons[1][2]=0
                                cons[2][2]=0
                        
        return largest,cons

    def estimate_none(self,x,y,z,xscaling=1.0,yscaling=1.0):
        tkMessageBox.showerror('Error', "No estimate. Please fill the table")
        newpar=[]
        for k in self.fit_theory[self.selected_th]:
                newpar.append(0.0)
        return newpar,zeros((3,len(newpar)),Float)

    def estimate_unknown(self,x,y,z,xscaling=1.0,yscaling=1.0):
        str = "Number of %s Functions : " % self.selected_th
        nfuns = input(str)
        nfuns=int(nfuns)
        if nfuns < 1:
            nfuns=1
        newpar=[]
        j=0
        for i in range(nfuns):
            j=j+1
            for k in self.fit_theory[self.selected_th]:
                newpar.append(input(k+`j`+' =  '))
        return newpar,zeros((3,len(newpar)),Float)

    def gendata(self,func,pars,start=0,end=100,npoints=100,colwise=0):
       """
       Generate data.
       X data are generated by dividing interval [start,end] in npoints points.
       Then the Y data resulting from the execution of function func with
       parameters pars, and the X data generated.
       """
       data  = []
       xdata = []
       ydata = []
    
       if npoints > 1:
          ival = ( end - start ) / ( npoints - 1)
       else: 
          return None
    
       for k in range(npoints):
           x = start + ival * k
           if colwise:
              xdata.append(x)
              ydata.append(apply(func,[pars,x]))
           else:
              data.append([x,apply(func,[pars,x])])
       
       if colwise:
          return([xdata,ydata])
       else:
          return(data)

    def indexx(x):
        #adapted from runningReport (Mike Fletcher, Python newsgroup) 
        set = map(None,x,range(len(x)))
        set.sort() #sorts by values and then by index
        return map(lambda x:x[1],set)


    def bkg_constant(self,pars,x):
       """
       Constant background
       """
       return pars[0]  

    def bkg_linear(self,pars,x):
       """
       Linear background
       """
       return pars[0] + pars [1] * x  

    def bkg_internal(self,pars,x):
       """
       Internal Background
       """       
       return self.zz  

    def bkg_none(self,pars,x):
       """
       Internal Background
       """       
       return zeros(x.shape,Float)  

    def apvoigt(self,pars,x):
       """
       Fit function.
       """
       return SpecfitFuns.apvoigt(pars,x)
       #return pars[0] + pars [1] * x + SpecfitFuns.apvoigt(pars[2:len(pars)],x)

    def pvoigt(self,pars,x):
       """
       Fit function.
       """
       return SpecfitFuns.pvoigt(pars,x)
       #return pars[0] + pars [1] * x + SpecfitFuns.pvoigt(pars[2:len(pars)],x)
    
    def lorentz(self,pars,x):
       """
       Fit function.
       """
       #return pars[0] + pars [1] * x + SpecfitFuns.lorentz(pars[2:len(pars)],x)
       return SpecfitFuns.lorentz(pars,x)
    
    def alorentz(self,pars,x):
       """
       Fit function.
       """
       return SpecfitFuns.alorentz(pars,x)
    
    def gauss(self,pars,x):
       """
       A fit function.
       """
       return SpecfitFuns.gauss(pars,x)

    def agauss(self,pars,x):
       """
       A fit function.
       """
       return SpecfitFuns.agauss(pars,x)

    def stepdown(self,pars,x):
       """
       A fit function.
       """
       return 0.5*SpecfitFuns.downstep(pars,x)

    def stepup(self,pars,x):
       """
       A fit function.
       """
       return 0.5*SpecfitFuns.upstep(pars,x)
    
    def fun(self,param, t):
        gterm = param[2] * exp(-0.5 * ((t - param[3]) * (t - param[3]))/param[4])
        #gterm = gterm + param[3] * exp(-0.5 * ((t - param[4]) * (t - param[4]))/param[5])
        bterm = param[1] * t + param[0]
        return gterm + bterm

    def fitconfig(self):
        sheet1=Sheet(notetitle='Fit Theory',
            fields=(TitleField('Default Fit Function'),
                RadioField('DefaultTheory',
                values=self.theory_list)
                ))
        sheet2=Sheet(notetitle='Background',
            fields=(TitleField('Default Background Function'),
                RadioField('BackgroundIndex',
                values=['Constant','Linear','Exponential','Internal',
                'No Background'])
                ))
        sheet3=Sheet(notetitle='Constrains',
            fields=(TitleField('Default Constrains'),
                CheckField('HeightAreaFlag','Force positive Height/Area'),
                CheckField('PositionFlag','Force position in interval'),
                CheckField('PosFwhmFlag','Force positive FWHM'),
                CheckField('SameFwhmFlag','All peaks same FWHM'),
                CheckField('EtaFlag','Eta between 0.0 and 1.0'),
                CheckField('NoConstrainsFlag','No Constrains'),
                ))
        sheet4=Sheet(notetitle='Weights',
            fields=(TitleField('Default Weight'),
                RadioField('DefaultWeight',values=['No Weight','Statistical']),
                TitleField('Default Scaling'),
                EntryField('Xscaling', 'X factor'),
                EntryField('Yscaling', 'Y factor')
                ))
        dummy = {}
        for key in self.fitdefaults.keys():
            dummy[key] = self.fitdefaults[key]

        app=SheetDialog(self.master,title='Specfit Config',
                    sheets=(sheet1,sheet2,sheet3,sheet4),
                    type='notebook',
                    validate=0,
                    init=dummy,
                    default=dummy)
        if app.result is not None:
            for key in app.result.keys():
                self.fitdefaults[key] = app.result[key]
        del app
        
    def _loadconfig(self):
        dummy={}
        if self.fitdefaults.has_key('infile'):
            dummy['infile']=self.fitdefaults['infile']
        sheet1= Sheet(notetitle='Load Config',
                fields=(TitleField('Configuration File'),   
                    FileOutput('infile','Output File')
                    ))
        sheet2= Sheet(notetitle='Dummy Config',
                fields=(TitleField('Configuration File'),   
                    FileOutput('infile','Output File')
                    ))
        app=SheetDialog(self.master,title='Specfit Config',
                    sheets=[sheet1],
                    type='notebook',
                    validate=0,
                    init=dummy,
                    default=dummy)
        if app.result is not None:
               file=app.result['infile']
               self.loadconfig(file) 
        del app
        
    def loadconfig(self,file=None):
        dummy = {}
        if file is not None:
            try:
                execfile(file,dummy)
            except:
                tkMessageBox.showerror('Error',
                    "Error reading file %s" % file)
                return
            for key in dummy.keys():
                #SPECFITDEFAULTS[key] = dummy[key]
                self.fitdefaults[key] = dummy[key] 
        del dummy

    def saveconfig(self,file=None):
        if file is not None:
            try:
                f=open(file,'w')
            except:
                tkMessageBox.showerror('Error',
                    "Error opening output file %s" % file)
                return
            for key in self.fitdefaults.keys():
                if (key != 'infile') and (key != 'outfile'):
                    if type(self.fitdefaults[key]) == type("string"):
                        f.write("%s='%s'\n" % (key,self.fitdefaults[key]))
                    else:
                        f.write("%s=%s\n" % (key,self.fitdefaults[key]))
                else:
                    if key == 'outfile':
                        f.write("%s='%s'\n" % (key,
                                self.fitdefaults['outfile']))
            f.close()

    def loaddefaults(self):
        filename=os.environ['HOME']+'/.specfitdefaults.py'
        self.loadconfig(file=filename)

    def savedefaults(self):
        filename=os.environ['HOME']+'/.specfitdefaults.py'
        self.saveconfig(file=filename)


class Parameter(ScrollFrame.ScrollFrame):
    """Parameter class provides for each parameters :
       - info display
       - value entry
       - constrains parameters select button
    """ 
    def __init__(self,master,paramlist=[],constrains=[],cnf={},**kw):
        """
        Class constructor.
        Create a widget containing the parameters.
        paramlist is the list of parameters names.
        """
        cnf = _cnfmerge((cnf,kw))
        ScrollFrame.ScrollFrame.__init__(self,master,cnf)
                
        self.paramlist  = paramlist
        self.parameters = {}
        if constrains != []:
            self.constrains= constrains       
        else:
            self.constrains=[]
            for i in range(len(paramlist)):
                self.constrains.append([0,0,0])
                    
        self.build()
        #self.pack(fill='both',expand='yes')

    def build(self):
        """
        Build the widget.
        """
        lbl0 = Label(self,text='Parameters',relief='groove')
        lbl1 = Label(self,text='Estimation',relief='groove')
        lbl2 = Label(self,text='Fit result',relief='groove')
        lbl3 = Label(self,text='Sigma',     relief='groove')
        lbl4 = Label(self,text='    Restrains    ',     relief='groove')
        lbl5 = Label(self,text='Min or Parameter',     relief='groove')
        lbl6 = Label(self,text='Max/Delta/Factor',     relief='groove')

        lbl0.grid(row=0,column=0,sticky='ew')
        lbl1.grid(row=0,column=1,sticky='ew')
        lbl2.grid(row=0,column=2,sticky='ew')
        lbl3.grid(row=0,column=3,sticky='ew')
        lbl4.grid(row=0,column=4,sticky='ew')
        lbl5.grid(row=0,column=5,sticky='ew')
        lbl6.grid(row=0,column=6,sticky='ew')

        width = 15
        line  = 1
        for param in self.paramlist:
            self.parameterline(param,line,width)
            line = line + 1

        self.columnconfigure((1,2,3),weight=1)
         
    def parameterline(self,param,line,width):
            esti_var  = DoubleVar()
            fit_var   = DoubleVar()
            sigma_var = DoubleVar()
            code_var  = StringVar()
            self.code_var=code_var
            cons1_var = DoubleVar()
            cons2_var = DoubleVar()
            min_var = DoubleVar()
            max_var = DoubleVar()
            relatedto_var = StringVar()
            factor_var = DoubleVar()
            delta_var = DoubleVar()
            group_var = IntVar()
            
            lbl_name       = Label(self,text=param)
            entry_estimate = Entry(self,bg='#eec',width=width,
                                   justify='right',textvariable=esti_var)
            lbl_fitresult  = Label(self,width=width,anchor='e',bd=1,
                             relief='ridge',textvariable=fit_var,bg='#ffffff')
            lbl_sigma      = Label(self,width=width,anchor='e',bd=1,
                             relief='ridge',textvariable=sigma_var,bg='#ffffff')
            #cb_fixed       = Checkbutton(self,variable=fix_var)
            par_opts= ['FREE','POSITIVE','QUOTED','FIXED','DELTA',
                      'FACTOR','IGNORE','ADD']
            cb_fixed       = Selector.Selector(self,par_opts,
                             command=lambda i, j=self, k=param: j.setvalue(i,k))
            #cb_fixed.setoptions(par_opts)
            #print cb_fixed
           
            #entry_cons1 = Entry(self,bg='#eec',width=width,
            #                       justify='right',textvariable=cons1_var)
            #entry_cons2 = Entry(self,bg='#eec',width=width,
            #                       justify='right',textvariable=cons2_var)
            # Previous Entry statements become Labels
            label_cons1 = Label(self,width=width,anchor='e',bd=1,
                               relief='ridge',textvariable=cons1_var)
            label_cons2 = Label(self,width=width,anchor='e',bd=1,
                               relief='ridge',textvariable=cons2_var)
            
            code_var=cb_fixed.value

            #print "code_var = ",code_var


            lbl_name.      grid(row=line,column=0,sticky='ew')
            entry_estimate.grid(row=line,column=1,sticky='ew')
            lbl_fitresult. grid(row=line,column=2,sticky='ew')
            lbl_sigma.     grid(row=line,column=3,sticky='ew')
            cb_fixed.      grid(row=line,column=4,sticky='ew')
            #entry_cons1.   grid(row=line,column=5,sticky='ew')
            #entry_cons2.   grid(row=line,column=6,sticky='ew')
            label_cons1.   grid(row=line,column=5,sticky='ew')
            label_cons2.   grid(row=line,column=6,sticky='ew')
       
            self.parameters[param] = {'estimation' : esti_var,
                                      'fitresult'  : fit_var,
                                      'sigma'      : sigma_var,
                                      'code'       : code_var,
                                      'cons1'      : cons1_var,
                                      'cons2'      : cons2_var,
                                      'vmin'       : min_var,
                                      'vmax'       : max_var,
                                      'relatedto'  : relatedto_var,
                                      'factor'     : factor_var,
                                      'delta'      : delta_var,                                      
                                      'group'      : group_var,
                                      }
            #self.configure(name=param,code=self.code_var)
            self.configure(name=param,
                        code=par_opts[self.constrains[line-1] [0]],
                        cons1=self.constrains[line-1] [1],
                        cons2=self.constrains[line-1] [2])
        
    def setvalue(self,item,workparam):
        if (item == 'FREE'):
            self.parameters[workparam]['code'].set(item)
            #put all the group FREE if it was a IGNORE group
            group = self.parameters[workparam]['group'].get()
            for param in self.paramlist:
                if group == self.parameters[param]['group'].get():
                    code = self.parameters[param]['code'].get()
                    if code == 'IGNORE':
                        self.parameters[param]['code'].set('FREE')

            #pass
        elif (item == 'POSITIVE'):
            self.parameters[workparam]['code'].set(item)
            #put all the group FREE if it was a IGNORE group
            group = self.parameters[workparam]['group'].get()
            for param in self.paramlist:
                if group == self.parameters[param]['group'].get():
                    code = self.parameters[param]['code'].get()
                    if code == 'IGNORE':
                        self.parameters[param]['code'].set('FREE')

            #pass
        elif (item == 'QUOTED'):
            self.parameters[workparam]['code'].set(item)
            #put all the group FREE if it was a IGNORE group
            group = self.parameters[workparam]['group'].get()
            for param in self.paramlist:
                if group == self.parameters[param]['group'].get():
                    code = self.parameters[param]['code'].get()
                    if code == 'IGNORE':
                        self.parameters[param]['code'].set('FREE')

            text=['1st limit','2nd limit']
            a = ParameterDialog(self,text=text,title=workparam)
            if a.result is not None:
                self.configure(name=workparam,
                               cons1=min(a.output),
                               cons2=max(a.output),
                               vmin=min(a.output),
                               vmax=max(a.output))
        elif (item == 'FIXED'):
            self.parameters[workparam]['code'].set(item)
            #put all the group FREE if it was a IGNORE group
            group = self.parameters[workparam]['group'].get()
            for param in self.paramlist:
                if group == self.parameters[param]['group'].get():
                    code = self.parameters[param]['code'].get()
                    if code == 'IGNORE':
                        self.parameters[param]['code'].set('FREE')

            #pass
        elif (item == 'FACTOR') | (item == 'DELTA'):
            #print "previous code was ",self.parameters[workparam]['code'].get()
            self.parameters[workparam]['code'].set(item)
            #put all the group FREE if it was a IGNORE group
            group = self.parameters[workparam]['group'].get()
            for param in self.paramlist:
                if group == self.parameters[param]['group'].get():
                    code = self.parameters[param]['code'].get()
                    if code == 'IGNORE':
                        self.parameters[param]['code'].set('FREE')

            #print "Working parameter number is",workparam
            options = []
            text = []
            for param in self.paramlist:
                #print 'param ', param
                #print 'self.cget(param)',self.cget(param)
                if self.cget(param)[1] [0]  < 4:
                    if self.parameters [param]['code'].get() != 'IGNORE':
                        options.append(param)
            if options != []:
                text.append('Parameter')
                if item == 'FACTOR':
                    text.append('Factor')
                else:
                    text.append('Delta')
                a = ParameterDialog(self,text=text,options1=options,
                    title=workparam)
                #print "a.result",a.result
                #print "a.output",a.output
                if a.result is not None:
                    i = 0
                    cons1 = []
                    cons2 = a.output [1]
                    for param in self.paramlist:
                        if a.output [0] == param:
                            cons1 = i
                        else:
                            i = i + 1
                    if cons1 == []:
                        print 'a.output [0] = ',a.output [0]
                        for param in self.paramlist:
                            print 'param = ',param,'test = ',a.output[0] == param
                    else:
                        self.configure(name=workparam,
                           cons1=cons1,
                           cons2=cons2,
                           relatedto=a.output[0])
                else:
                    print 'WARNING parameter %s set to FREE' % workparam
                    self.parameters[workparam]['code'].set('FREE')
                #print "Working parameter number is",workparam
                #print "Its associated info is ",self.cget(workparam)
                #print "More detailed"
                #print "code",self.parameters[workparam]['code'].get()
                #print "cons1",self.parameters[workparam]['cons1'].get()
                #print "cons2",self.parameters[workparam]['cons2'].get()
                #print "relatedto",self.parameters[workparam]['relatedto'].get()
        if item == 'IGNORE':
            group = self.parameters [workparam] ['group'].get()
            #print "I should ignore the group ",group
            self.ignoregroup(group)
        elif item == 'ADD':
            i=1
            for param in self.paramlist:
                if i <=  self.parameters [param] ['group'].get():
                    i = i + 1
            group = self.parameters [workparam] ['group'].get()
            #print "try to reset the code to free ... "
            self.parameters[workparam]['code'].set('FREE')
            #add the group i, of type group
            self.addgroup(i,group)

    def addgroup(self,newg,gtype):
        line = 0
        newparam=[]
        for param in self.paramlist:
            line = line + 1
            paramgroup = self.parameters [param] ['group'].get()
            if paramgroup  == gtype:
                #Try to construct an appropriate name
                #I have to remove any possible trailing number
                #and append the group index
                j=len(param)-1
                while ('0' < param[j]) & (param[j] < '9'):
                    j=j-1
                    if j == -1:
                        break
                if j >= 0 :
                    newparam.append(param[0:j+1] + `newg`)
                else:
                    newparam.append(`newg`)
        width=15
        for param in newparam:
            line=line+1
            self.constrains.append([0,0,0])
            self.paramlist.append(param)
            self.parameterline(param,line,width)        

    def ignoregroup(self,gtype):
        # I should check if the group can be ignored
        # for the time being I just fix all of them to zero in cget
        candidates = []
        for param in self.paramlist:
            if self.parameters [param] ['group'].get() == gtype:
                candidates.append(param)        
                self.parameters [param] ['code'].set('IGNORE') 
                    
    def configure(self,name=None,fitresult=None,sigma=None,
                  estimation=None,code=None,cons1=None,cons2=None,
                  vmin=None,vmax=None,
                  relatedto=None,factor=None,delta=None,group=None,**kw):
        """
        Configure the widget.
        Configure the name parameters ie fill the infos labels (fit result,
        or sigma) and the value entry (coming from the estimation).
        Others keywords are passed to the global frame configure method.
        """ 
        Frame.configure(self,kw)
        if name is not None:
           if fitresult is not None:
               self.parameters[name]['fitresult'].set(fitresult)
           if sigma is not None:
               self.parameters[name]['sigma'].set(sigma)
           if estimation is not None:
               self.parameters[name]['estimation'].set(estimation)
           if code is not None:
               #print "setting code =",code
               self.parameters[name]['code'].set(code)
           if cons1 is not None:
               #print "setting cons2 =",cons1
               self.parameters[name]['cons1'].set(cons1)
           if cons2 is not None:
               #print "setting cons2 =",cons2
               self.parameters[name]['cons2'].set(cons2)
           if vmin is not None:
               self.parameters[name]['vmin'].set(vmin)
           if vmax is not None:
               self.parameters[name]['vmax'].set(vmax)
           if relatedto is not None:
               self.parameters[name]['relatedto'].set(relatedto)
           if factor is not None:
               self.parameters[name]['factor'].set(factor)
           if delta is not None:
               self.parameters[name]['delta'].set(delta)
           if group is not None:
               self.parameters[name]['group'].set(group)
               

    def cget(self,option):
          """
          Return tuple (estimation, constrains) where estimation
          is the values in the estimation field and constrains are
          the relevant constrains according to the active code
          0 = Free      1 = Positive        2 = Quoted
          3 = Fixed     4 = Factor          5 = Delta
          6 = Ignored 
          """
          #print "option = ", option
          if option in self.paramlist:
               estimation = self.parameters[option]['estimation'].get()
               #print "estimation = ",estimation
               code = self.parameters[option]['code'].get()
               cons1 = self.parameters[option]['cons1'].get()
               cons2 = self.parameters[option]['cons2'].get()
               #print "code =",code
               #print "cons1=",cons1
               #print "cons2=",cons2
               constrains = [0,0,0]
               if code == 'FREE':
                    return estimation, constrains               
               elif  code == 'POSITIVE':
                    constrains [0] = 1
                    return estimation, constrains
               elif  code == 'QUOTED':
                    constrains [0] = 2
                    constrains [1] = cons1
                    constrains [2] = cons2
                    #constrains [0] [1] = self.parameters[option]['vmin'].get()
                    #constrains [0] [2] = self.parameters[option]['vmax'].get()
                    return estimation, constrains
               elif  code == 'FIXED':
                    constrains [0] = 3
                    return estimation, constrains
               elif  code == 'FACTOR':
                    constrains [0] = 4
                    constrains [1] = int(cons1)
                    constrains [2] = cons2
                    #print "related to",self.parameters[option]['relatedto'].get()
                    #constrains [0] [1] = self.parameters[option]['relatedto'].get()
                    #constrains [0] [2] = self.parameters[option]['factor'].get()
                    return estimation, constrains
               elif  code == 'DELTA':
                    constrains [0] = 5
                    constrains [1] = int(cons1)
                    constrains [2] = cons2
                    #constrains [0] [1] = self.parameters[option]['relatedto'].get()
                    #constrains [0] [2] = self.parameters[option]['delta'].get()
                    return estimation, constrains
               elif  code == 'ADD':
                    constrains [0] = 0
                    constrains [1] = cons1
                    constrains [2] = cons2
                    #constrains [0] [1] = self.parameters[option]['relatedto'].get()
                    #constrains [0] [2] = self.parameters[option]['delta'].get()
                    return estimation, constrains
               elif  code == 'IGNORE':
                    #print "IGNORE"
                    estimation = 0
                    constrains [0] = 6
                    constrains [1] = cons1
                    constrains [2] = cons2
                    #constrains [0] [1] = self.parameters[option]['relatedto'].get()
                    #constrains [0] [2] = self.parameters[option]['delta'].get()
                    return estimation, constrains
          else:
               Frame.cget(self,option)


class ParameterDialog(tkSimpleDialog.Dialog):

    def __init__(self,master,text=None,options1=None,options2=None,**kw):
        if text is not None:         
            self.text=text
        else:
            self.text=["Label 1","Label 2"]

        if options1 is not None:
            self.options1 = options1
            self.selector1 = 1
        else:
            self.selector1 = 0

        if options2 is not None:
            self.options1 = kw["options2"]
            self.selector2 = 1
        else:
            self.selector2 = 0
        
        self.output = [[],[]]
        self.result = None    

        tkSimpleDialog.Dialog.__init__(self,master,**kw)
        
    def body(self,master):
    
        Label(master,text=self.text[0]).grid(row=0)
        Label(master,text=self.text[1]).grid(row=1)

        if self.selector1 == 1:
            self.output [0] = self.options1[0]
            self.e1  = Selector.Selector(master, self.options1,
                           command=lambda i, j = self, k = 0: j.setvalue(i,k))
            #              command=self.printitem)
        else:
            self.e1=Entry(master)

 
        if self.selector2 == 1:
            self.output [1] = self.options2[0]
            self.e2  = Selector.Selector(master ,self.options2,
                           command=lambda i, j = self, k = 1: j.setvalue(i,k))
            self.e2.value = self.options2[0]
            #              command=self.printitem)
        else:
            self.e2=Entry(master)

        self.e1.grid(row=0,column=1)
        self.e2.grid(row=1,column=1)
        return self.e1 # initial focus
        
    def setvalue(self,item,k):
        self.output [k] = item
    
    def apply(self):
        if self.selector1 == 1:
            pass
        else:
            self.output [0] = self.e1.get()
        if self.selector2 == 1:
            pass
        else:
            self.output [1] = self.e2.get()
        self.result = self.output




def paramtest():
     """
     Parameter class test function.
     """
     root = Tk()
     inst = Parameter(root,['un','one','uno','bla'])
     inst.pack(expand='yes',fill='both')
 
     inst.configure(name='one',fitresult=0.125483)

     print inst.cget('one')

     root.mainloop()

def testfun(param, t):
    bterm=param[0] + param[1] * t
    gterm = SpecfitFuns.gauss(param[2:len(param)],t)
    return bterm + gterm


def fittest():
     """
     Specfit class test function.
     """
     root = Tk()
     npoints=1000
     xx = arange(npoints).astype(Float)
     xdata=xx.tolist()
     yy = 10000.0 * exp (- 0.5 * ((xx-500) * (xx-500)) /15)+ 2.0 * xx + 10.5
     ydata=yy.tolist()
     xx=resize(xx,(npoints,1))
     yy=resize(yy,(npoints,1))
     sy = sqrt(abs(yy))
     sy=resize(sy,(npoints,1)) 
     data = concatenate((xx, yy, sy),1)
     parameters = array (( 11.0, 1.9,8000.0, 499.0, 5))
     #fittedpar, chisq, sigmapar = LeastSquaresFit(fun,parameters,data)
     found = LeastSquaresFit(testfun,parameters,data)
     print "found =",found
     inst = Specfit(root,xdata,ydata,
                    source='My source',
                    eh=None,
                    scan='My scan')
     root.mainloop()

def  dialogtest():
     root = Tk()
     a=ParameterDialog(root,text=['Label 1','Label 2'],options1=['op1','op2','op3'])
     print "a.result = ",a.result
     root.mainloop()

if __name__ == '__main__':
     #dialogtest()
     #test(10000)
     fittest()
