/*
Update 09/11/2001 R. Wilcke (wilcke@esrf.fr)
                  change the scope of the first "#if ANDY_CORR" clause - as it
                  was, all global variable declarations were in it.
Update 08/11/2001 R. Wilcke (wilcke@esrf.fr)
                  andy_corr() and andy_fcorr(): save the x- and y-pixel sizes
                  when obtained from the (spline or coordinate) files;
                  corr_calc(): set the header values PSize_1 and PSize_2 for the
                  corrected image from the saved x- and y-pixel sizes;
                  add global variables INP_MAX and INP_MIN, which define the
                  maximum and minumum allowed values in the source and the dark
                  images, and add routines set_inpmax() and set_inpmin() to set
                  their values;
                  mark_overflow_nocorr(): add test for values less than INP_MIN
                  or greater than INP_MAX;
Update 05/11/2001 R. Wilcke (wilcke@esrf.fr)
                  azim_int(): add code to restrict integration to the range
                  given by either the input arguments or the dimension of the
                  input image.
Update 05/11/2001 R. Wilcke (wilcke@esrf.fr)
                  azim_int(): add 2 more new input arguments.
Update 31/10/2001 R. Wilcke (wilcke@esrf.fr)
                  azim_int(): add calculation of maximum radius for the
                  azimuthal integration.
Update 22/10/2001 R. Wilcke (wilcke@esrf.fr)
                  azim_int(): add 3 new input arguments for start radius and
                  angle as well as angular increment.
Update 18/10/2001 R. Wilcke (wilcke@esrf.fr)
                  set_dummy(): set value of "DDummy" to reflect new value of
                  "Dummy";
                  normint_im(): take "ddummy" from the CORTYP header structure;
                  add new routine azim_int() to integrate an image azimuthally.
Update 03/10/2001 R. Wilcke (wilcke@esrf.fr)
                  put_buffer(): change last argument in save_esrf_file() from
                  NULL to -1.
Update 02/10/2001 R. Wilcke (wilcke@esrf.fr)
                  andy_corr(): remove the last two arguments in the call to
                  save_esrf_file().
Update 14/09/2001 R. Wilcke (wilcke@esrf.fr)
                  correct_image(): return -2 if no buffer for corrected image
                  was given, -1 if any of the corrections failed, and 0 else.
Update 13/09/2001 R. Wilcke (wilcke@esrf.fr)
                  normint_im(): corrected test for the required header values.
Update 11/09/2001 R. Wilcke (wilcke@esrf.fr)
                  mark_overflow(): correct error in the processing of the last
                  pixel: in the "for" loop for "j", replace "YSIZE" by "XSIZE"
                  as it is the case in the loop for all other pixels;
                  lut_calc(): correct code for filling the lut_d->xrel and
                  lut_d->yrel arrays, because the input arrays x_trg and y_trg
                  have one pixel more in each dimension than xrel and yrel.
Update 20/08/2001 R. Wilcke (wilcke@esrf.fr)
                  add function get_headval() to obtain the values of the header
                  structures;
                  set_headval(): define default values for structure members
                  Dummy, DDummy, Offset_1, Offset_2, Orientat and Intens_1;
                  correct_image(): if no correction is to be done, just copy the
                  input file to the output file;
                  set_dummy(): put the new "Dummy" in the header structue of the
                  corrected image (type = CORTYP);
                  mark_overflow_nocorr(): write the output "Dummy" value into
                  the header of the corrected image;
                  correct_image(): allocate the space for the temporary image no
                  longer in the structure "lut_d", but in a separate static
                  variable "temp_im";
Update 17/08/2001 R. Wilcke (wilcke@esrf.fr)
                  set_headval(): do no longer set member "init" to 1 in the
                  routine, as "init" now contains the "OR"ed flags of all header
                  keywords that have been found;
                  normint_im(): test "init" member for the required keywords;
                  mark_overflow_nocorr() and prepare_flood(): test "init" member
                  for the keywords "Dummy" and "DDummy";
                  mark_overflow(): return immediately without action if
                  "image_over" is not a valid dummy value;
                  change dimension of structure array img_head from 6 to 7;
                  set_headval(): "type" can now also be "SPDTYP";
                  andy_fcorr(): fill the members PSize_1 and PSize_2 of the
                  header structure for the corrected image from the files for
                  the corrected x and y coordinates;
                  andy_fcorr() and andy_corr(): or the flags for PSize_1 and
                  PSize_2 into the "init" member of the corrected image header;
                  move the declarations of andy_active_area() and andy_fcorr()
                  to the beginning of the file;
                  correct_image(): free the members of structure "lut_d" only if
                  they do not contain NULL pointers.
Update 16/08/2001 R. Wilcke (wilcke@esrf.fr)
                  correct_image(): change code to process the marking of illegal
                  pixels correctly;
                  mark_overflow() and mark_overflow_nocorr(): use only "Dummy"
                  to mark illegal pixel values in the output;
                  prepare_flood(): use the value of the keyword "Dummy" in the
                  floodfield image header to find illegal pixels in the
                  floodfield image;
                  normint_im(): do not process pixels with illegal values;
                  set_actrad(): take new value only if different from old one.
Update 14/08/2001 R. Wilcke (wilcke@esrf.fr)
                  andy_fcorr(): add new fifth argument (type = SPDTYP) to the
                  read_esrf_file() call.
Update 13/08/2001 R. Wilcke (wilcke@esrf.fr)
                  rename structure variable "src_head" to "img_head", make it
                  an array with dimension 6 and initialize all their "init"
                  members to 0;
                  set_headval(): change the type of the function from "void" to
                  "int", add a second input argument to the function, change
                  code to copy the input header to particular element in the
                  header structure array, copy a SRCTYP input header also to
                  the CORTYP structure and return an error if the "type" of the
                  structurre requested has an illegal value;
                  andy_corr(): put the x- and y-pixel sizes in the header
                  structure element CORTYP;
                  normint_im(): take the values from the header structure
                  element CORTYP;
                  prepare_flood(): add code to take the values for "dummy" and
                  "ddummy" from the FLOTYP header structure, if this has been
                  initialized.
Update 10/08/2001 R. Wilcke (wilcke@esrf.fr)
                  subtract_im(): always subtract the background constant;
                  correct_image(): do corrections even if only NORM_INT or
                  SCA_IM are specified;
                  normint_im(): add output buffer as second input argument;
Update 07/08/2001 R. Wilcke (wilcke@esrf.fr)
                  add global variable SCA_IM, which points to the scattering
                  background image, and routine set_scaim() to set its value;
                  subtract_im(): remove variable "negative";
                  subtract_im(): add fourth input argument for the background
                  constant and modify the references to BKG_CONST accordingly;
                  correct_image(): add forth input argument to the calls to
                  subtract_im();
                  correct_image(): correct error in the "distortion floodfield
                  (floodfield after distortion)" case: wrong source image for
                  corr_calc() call was specified;
                  correct_image(): add code to perform the scattering background
                  correction;
                  andy_corr(): divide the x- and y-pixel sizes by 1000000.
Update 06/08/2001 R. Wilcke (wilcke@esrf.fr)
                  andy_corr(): put the corrected x- and y-pixel sizes in the
                  data header structure.
Update 03/08/2001 R. Wilcke (wilcke@esrf.fr)
                  correct_image(): add variable "bkg_cor" to indicate if a
                  background correction is to be done;
                  add global variable NORM_INT for the normalization to absolute
                  scattering intensities and routine set_normint() to set its
                  value;
                  add struct src_head and routine set_headval() to set its
                  values;
                  add routine normint_im() to calculate the normalization to
                  absolute scattering intensities;
                  correct_image(): reorganize the logical structure of the
                  decision which corrections are to be made, and add the call
                  to normint_im() if NORM_INT is set.
Update 28/06/2001 R. Wilcke (wilcke@esrf.fr)
                  add routines set_bkgim() and set_floim() to set the values
                  of BKG_IM and FLO_IM, which point to the background image and
                  the floodfield image;
                  change routine correct_image() by eliminating the background
                  image and the floodfield image from the input arguments and
                  modifying the code to use the new global variables BKG_IM and
                  FLO_IM;
                  removed declaration of variable one_spline (not used);
                  unloadspd(): return without action if spline == NULL.
Update 26/06/2001 R. Wilcke (wilcke@esrf.fr)
                  change the name of the variables XFILE and YFILE to XINFILE
                  and YINFILE;
                  add variables XOUTFILE and YOUTFILE for the names of files
                  where the distortion correction values can be written to;
                  andy_corr(): make the output of the distortion correction
                  values dependent on "XOUTFILE && YOUTFILE" instead of
                  "verbose == 2" and use the filenames in XOUTFILE and YOUTFILE
                  instead of the fixed names "cor_x.edf" and "cor_y.edf";
                  add new function set_xycorout() to set the value of XOUTFILE
                  and YOUTFILE.
Update 25/06/2001 R. Wilcke (wilcke@esrf.fr)
                  remove variable DO_XYFILE;
                  andy_corr(): replace test for DO_XYFILE with test for
                  "XFILE && YFILE";
                  move declaration of variable "verbose" here from "spd.h";
                  change type of BKG_CONST from "int" to "float";
                  remove declaration of the parameters for the geometrical
                  method (CURV_RADIUS, XCENTER and YCENTER) and all related
                  code. This method is no longer used;
                  make_grid(): set the variable CURV_RADIUS in this function to
                  the old default value of CURV_RADIUS = 270000/180;
                  add new functions:
                    - set_verbose() to set the value of "verbose";
                    - set_overflow() to set the value of IMAGE_OVER and
                      IMAGE_OVER_SET;
                    - set_dummy() to set the value of "Dummy";
                    - set_actrad() to set the value of ACTIVE_R;
                    - set_slinfil() to set the value of FILENAME;
                    - set_xycorin() to set the value of XFILE and YFILE;
                    - set_xysize() to set the value of XSIZE and YSIZE;
                    - get_xsize() to get the value of XSIZE;
                    - get_ysize() to get the value of YSIZE;
                    - set_bkgconst() to set the value of BKG_CONST;
                    - set_dospd() to set the value of DO_SPD;
                    - set_dolater() to set the value of DO_LATER;
                    - set_doflat() to set the value of DO_FLAT;
Update 12/06/2001 R. Wilcke (wilcke@esrf.fr)
                  change variable name "DUMMY" to "Dummy" to avoid name conflict
                  with macro DUMMY().
Update 14/05/2001 R. Wilcke (wilcke@esrf.fr)
                  andy_corr(): replace SHM_FLOAT by MFloat in the call to
                  save_esrf_file().
Update 03/05/2001 R. Wilcke (wilcke@esrf.fr)
                  andy_fcorr(): replace SHM_FLOAT by MFloat in the call to
                  read_esrf_file().
Update 26/04/2001 R. Wilcke (wilcke@esrf.fr)
                  prepare_flood(): add the possibility that the input image
                  is already of type "float", and just invert the image in this
                  case.
Update 14/02/2001 R. Wilcke (wilcke@esrf.fr)
                  divide_im() and divide_insito_im(): remove the code that tests
                  for pixel values >= IMAGE_OVER and replaces them with
                  IMAGE_OVER_SET. This is no longer needed for "float" arrays;
                  correct_image(): remove the calls to "mark_overflow_nocorr()"
                  for the cases where the floodfield correction is done before
                  before the distortion correction, and call "mark_overflow()"
                  with the original source image. This is a consequence of the
                  changes to "divide_im()" and "divide_insito_im()" described
                  above;
                  mark_overflow_nocorr() and mark_overflow(): to mark pixels
                  with overflow in the output images, use the value DUMMY if it
                  is set (i.e., != 0.), otherwise IMAGE_OVER_SET.
Update 13/02/2001 R. Wilcke (wilcke@esrf.fr)
                  subtract_im(): allow the background subtraction to yield
                  negative values in the output image (they were set =0 before),
                  and remove the diagnostic message about negative values;
                  clean up global variables: remove C_XSIZE, C_YSIZE;
                  added global variable DUMMY;
                  corr_calc(): set the value of empty pixels in the target image
                  to DUMMY (they were set to 0. before).
Update 06/02/2001 R. Wilcke (wilcke@esrf.fr)
                  changed the arguments to the "BISPEV" call (they must all be
                  pointers because of the FORTRAN calling convention).
Update 02/02/2001 R. Wilcke (wilcke@esrf.fr)
                  divide_insito_im() and divide_im(): change routines to work
                  with an input image of type "float": convert the corresponding
                  function arguments and local variables to type "float" and
                  change code to operate on floats;
                  substract_im(): change name to subtract_im() and convert
                  routine to work with an input image of type "float": convert
                  the corresponding function arguments and local variables to
                  type "float" and change code to operate on floats;
                  mark_overflow() and mark_overflow_nocorr(): change routines to
                  work with an input image of type "float": convert the
                  corresponding function arguments and local variables to type
                  "float" and change code to operate on floats;
                  corr_calc(): change routine to work with an input image of
                  type "float": convert the corresponding function arguments
                  and local variables to type "float" and change code to operate
                  on floats;
                  remove function corr_calc_plus();
                  correct_image(): change routine to work with an input image of
                  type "float": convert the corresponding function arguments
                  and allocate "float" memory space for "lut_d->temp_im";
Update 17/01/2001 R. Wilcke (wilcke@esrf.fr)
                  pxcorrgrid(): replaced call to E02DEF() by call to BISPEV();
                  changed type of arguments accordingly from "double" to "float"
                  everywhere in the program.
Update 10/01/2001 R. Wilcke (wilcke@esrf.fr)
                  mark_overflow() and gtest(): replaced call to pxcorr() by
                  call to pxcorrgrid();
                  removed routine pxcorr() and all references to E02DEF().

*/

#include "spd.h"
#include "SaxsRoutine.h"

#if ANDY_CORR
#ifdef UNDERSCORE
#define BISPEV bispev_
#else
#ifdef UPPERCASE
#define BISPEV BISPEV
#else
#define BISPEV bispev
#endif /* UPPERCASE */
#endif /* UNDERSCORE */
#endif /* ANDY_CORR */

static char *FILENAME = "spatial.dat";
static char *XINFILE = NULL;
static char *YINFILE = NULL;
static char *XOUTFILE = NULL;
static char *YOUTFILE = NULL;
static unsigned short IMAGE_OVER = 0xffff;
static unsigned short IMAGE_OVER_SET = 0xffff;
static int DO_SPD = 1;
static int DO_FLAT = 0;
static int DO_LATER = 1;
static int NORM_INT = 0;
static int XSIZE = 1024; /* Size of the src image */
static int YSIZE = 1024;
static int verbose = 1;
static float ACTIVE_R = 0.;
static float BKG_CONST = 0.;
static float Dummy = 0.;
static float INP_MAX = 0.;
static float INP_MIN = 0.;
static float *BKG_IM = NULL;
static float *FLO_IM = NULL;
static float *SCA_IM = NULL;

static float psizex = -1.,psizey = -1.;

/*
 * Declare structure for data header and initialize the "init" member to 0
 * (not initialized).
 */
static struct data_head img_head[7] = {{0},{0},{0},{0},{0},{0},{0}};

static int LUT_INVALID = 1;
static int SPLINE_INVALID = 1;
static long *fldumlst = NULL;
static float *X_COR = NULL;
static float *Y_COR = NULL;

#if ANDY_CORR
struct spd_spline {
  int dx_xknots;
  int dx_yknots;
  double xcor_size;
  double xcenter;
  float *xlambda;
  float *xmu;
  float *xpars;
  int dy_xknots;
  int dy_yknots;
  double ycor_size;
  double ycenter;
  float *ylambda;
  float *ymu;
  float *ypars;
  double grid_space;
  float *wrk;
  int *iwrk;
  int maxspace;
  double reg_x0; /* Valid region from to */
  double reg_x1;
  double reg_y0;
  double reg_y1;
};

static float *x_buf = NULL, *y_buf = NULL;
static struct spd_spline *spline=NULL;

int andy_active_area(float *image,int xcen,int ycen,int r);
int andy_fcorr(float **cor_x,float **cor_y,int x0,int x1,int y0,int y1);

#endif /* ANDY_CORR */

/*
 * Gets the values of the user-definable header parameters' structure.
 *
 * The image header structure selected by the input parameter "type" is returned
 * in the structure pointed to by the argument "outhead". The member "init" of
 * the structure contains the "OR"ed flags of all the header keywords that have
 * been initialized.
 *
 * Default value if this structure array element has never been initialized by
 * the user (i.e., if set_headval() has never been called for the structure
 * with index "type"): "init" = 0. However, the CORTYP structure needs not to be
 * initialized by the user, as its values will be set from the SRCTYP structure.
 * Thus the CORTYP structure is initialized if the SRCTYP structure has been
 * initialized.
 *
 * See the declaration of the "img_head" structure array for the values that
 * "type" can have.
 *
 * Input : type: index of the structure array element to be returned
 * Output: outhead: structure with the values of the structure "img_head[type]"
 * Return: -1  if "type" has an illegal value
 *          0  otherwise
 */

int get_headval(struct data_head *outhead,int type)
{
  if(type < SRCTYP || type > SPDTYP)
    return(-1);

  *outhead = img_head[type];

  return(0);
}

/*
 * Gets the value of the user-definable variable XSIZE. This is the number of
 * pixels in the x-direction.
 *
 * Default value if not defined by the user: 1024
 *
 * Input : none
 * Output: none
 * Return: present value of variable XSIZE
 */

int get_xsize(void)
{
  return(XSIZE);
}

/*
 * Gets the value of the user-definable variable YSIZE. This is the number of
 * pixels in the y-direction.
 *
 * Default value if not defined by the user: 1024
 *
 * Input : none
 * Output: none
 * Return: present value of variable YSIZE
 */

int get_ysize(void)
{
  return(YSIZE);
}

/*
 * Sets the user-definable variable ACTIVE_R. If set, the spatial correction is
 * only done inside a circular area with radius ACTIVE_R in the image.
 *
 * Default value if not defined by the user: 0. (i.e., no circular area defined)
 *
 * Input : arad: new value for variable ACTIVE_R
 * Output: none
 * Return: none
 */

void set_actrad(float arad)
{
  if(arad != ACTIVE_R) {
    ACTIVE_R = arad;
    LUT_INVALID = 1;
  }
}

/*
 * Sets the user-definable variable BKG_CONST. If set, this value is subtracted
 * from every pixel in the source image.
 *
 * For details, see routine subtract_im().
 *
 * Default value if not defined by the user: 0.
 *
 * Input : bkgconst: new value for variable BKG_CONST
 * Output: none
 * Return: none
 */

void set_bkgconst(float bkgconst)
{
  BKG_CONST = bkgconst;
}

/*
 * Sets the user-definable variable BKG_IM. If set, this variable points to
 * the background image.
 *
 * Default value if not defined by the user: NULL pointer.
 *
 * Input : bkgim: new value for variable BKG_IM
 * Output: none
 * Return: none
 */

void set_bkgim(void *bkgim)
{
  BKG_IM = (float *)bkgim;
}

/*
 * Sets the user-definable variable DO_FLAT. If this variable is set, the target
 * image will be normalized to a flat image.
 *
 * Default value if not defined by the user: 0
 *
 * Input : doflat: new value for variable DO_FLAT
 * Output: none
 * Return: none
 */

void set_doflat(int doflat)
{
  if(DO_FLAT != doflat) {
    LUT_INVALID = 1;
    DO_FLAT = doflat;
  }
}

/*
 * Sets the user-definable variable DO_LATER. If this variable is set, the
 * floodfield correction will be done after the distortion correction, otherwise
 * distortion will be done after floodfield.
 *
 * Default value if not defined by the user: 1
 *
 * Input : dolater: new value for variable DO_LATER
 * Output: none
 * Return: none
 */

void set_dolater(int dolater)
{
  if(DO_LATER != dolater) {
    LUT_INVALID = 1;
    DO_LATER = dolater;
  }
}

/*
 * Sets the user-definable variable DO_SPD. If this variable is set, a
 * distortion correction will be performed, otherwise it will not be performed.
 *
 * Default value if not defined by the user: 1
 *
 * Input : dospd: new value for variable DO_SPD
 * Output: none
 * Return: none
 */

void set_dospd(int dospd)
{
  DO_SPD = dospd;
}

/*
 * Sets the user-definable variable "Dummy" which is used to mark invalid pixels
 * in the output image.
 *
 * Default value if not defined by the user: 0.
 *
 * The value of "DDummy" is also redefined to reflect the new value of "Dummy":
 *         DDummy = DDSET(Dummy)
 *
 * Input : dummy_in: new value for variable Dummy
 * Output: none
 * Return: none
 */

void set_dummy(float dummy_in)
{
  Dummy = dummy_in;
  img_head[CORTYP].Dummy = Dummy;
  img_head[CORTYP].init |= FL_DUMMY;
  img_head[CORTYP].init |= FL_DDUMM;
  img_head[CORTYP].DDummy = DDSET(Dummy);
}

/*
 * Sets the user-definable variable FLO_IM. If set, this variable points to
 * the floodfield image.
 *
 * Default value if not defined by the user: NULL pointer.
 *
 * Input : floim: new value for variable FLO_IM
 * Output: none
 * Return: none
 */

void set_floim(void *floim)
{
  FLO_IM = (float *)floim;
}

/*
 * Sets the values of the user-definable header parameters' structure.
 *
 * The input structure is copied to the "data_head" type structure with the
 * array index given by the input parameter "type". The member "init" of that
 * header structure contains then the "OR"ed flags of all the header keywords
 * that are initialized.
 *
 * Default value if this structure array element has never been initialized by
 * the user (i.e., if set_headval() has never been called for the structure
 * with index "type"): "init" = 0
 *
 * If the input header is of type SRCTYP, copy it into the CORTYP structure as
 * well.
 *
 * See the declaration of the "img_head" structure array for the values that
 * "type" can have.
 *
 * Input : inhead: structure with the new values for the structure
 *                 "img_head[type]"
 *         type  : index of the structure array element to be initialized
 * Output: none
 * Return: -1  if "type" has an illegal value
 *          0  otherwise
 */

int set_headval(struct data_head inhead,int type)
{
  if(type < SRCTYP || type > SPDTYP)
    return(-1);

  img_head[type] = inhead;

  if(!(inhead.init & FL_DUMMY))
  {
    img_head[type].init |= FL_DUMMY;
    img_head[type].Dummy = 0.;
  }
  if(!(inhead.init & FL_DDUMM))
  {
    img_head[type].init |= FL_DDUMM;
    img_head[type].DDummy = DDSET(img_head[type].Dummy);
  }
  if(!(inhead.init & FL_OFFS1))
  {
    img_head[type].init |= FL_OFFS1;
    img_head[type].Offset_1 = 0.;
  }
  if(!(inhead.init & FL_OFFS2))
  {
    img_head[type].init |= FL_OFFS2;
    img_head[type].Offset_2 = 0.;
  }
  if(!(inhead.init & FL_ORIEN))
  {
    img_head[type].init |= FL_ORIEN;
    img_head[type].Orientat = 1;
  }
  if(!(inhead.init & FL_INTE1) && inhead.init & FL_INTE0)
  {
    img_head[type].init |= FL_INTE1;
    img_head[type].Intens_1 = img_head[type].Intens_0;
  }

  if(type == SRCTYP)
    img_head[CORTYP] = img_head[SRCTYP];

  return(0);
}

/*
 * Sets the user-definable variable "INP_MAX" which is used to define the
 * maximum allowed value in the input and the dark current image.
 *
 * The value 0. is used to signal that no maximum allowed value is set.
 *
 * Default value if not defined by the user: 0. (i.e., not set).
 *
 * Input : inpmax_in: new value for variable INP_MAX
 * Output: none
 * Return: none
 */

void set_inpmax(float inpmax_in)
{
  INP_MAX = inpmax_in;
}

/*
 * Sets the user-definable variable "INP_MIN" which is used to define the
 * minimum allowed value in the input and the dark current image.
 *
 * The value 0. is used to signal that no minimum allowed value is set.
 *
 * Default value if not defined by the user: 0. (i.e., not set).
 *
 * Input : inpmin_in: new value for variable INP_MIN
 * Output: none
 * Return: none
 */

void set_inpmin(float inpmin_in)
{
  INP_MIN = inpmin_in;
}

/*
 * Sets the user-definable variable NORM_INT. If this variable is set, the
 * target image will be normalized to absolute scattering intensities.
 *
 * Default value if not defined by the user: 0 (i.e., not set)
 *
 * Input : normint: new value for variable NORM_INT
 * Output: none
 * Return: none
 */

void set_normint(int normint)
{
  NORM_INT = normint;
}

/*
 * Sets the user-definable variables IMAGE_OVER and IMAGE_OVER_SET which are
 * used to mark pixels with overflow values in the images.
 *
 * Default value for both variables if not defined by the user: 0xffff
 *
 * Input : overfl: new value for variables IMAGE_OVER and IMAGE_OVER_SET
 * Output: none
 * Return: none
 */

void set_overflow(unsigned short overf)
{
  IMAGE_OVER = overf;
  IMAGE_OVER_SET = overf;
}

/*
 * Sets the user-definable variable SCA_IM. If set, this variable points to
 * the scattering background image.
 *
 * Default value if not defined by the user: NULL pointer.
 *
 * Input : scaim: new value for variable SCA_IM
 * Output: none
 * Return: none
 */

void set_scaim(void *scaim)
{
  SCA_IM = (float *)scaim;
}

/*
 * Sets the user-definable variable FILENAME. The corresponding file contains
 * the spline function coefficients that used to calculate the distortion
 * correction values.
 *
 * Default value if not defined by the user: "spatial.dat"
 *
 * Input : filename: new value for variable FILENAME
 * Output: none
 * Return: none
 */

void set_splinfil(char *filename)
{
  int newfile = 0;
  static int dist_mtime = 0;
  struct stat stat_buf;

  if(XINFILE) {
    pfree(XINFILE);
    XINFILE = NULL;
  }
  if(YINFILE) {
    pfree(YINFILE);
    YINFILE = NULL;
  }

  if(strcmp(FILENAME,filename) != 0) {
    newfile = 1;
  } else if(dist_mtime) {
    if(stat(filename,&stat_buf) != 0)
      prmsg(MSG,("Cannot access distortion file %s\n",filename));
    else if(dist_mtime != (int)stat_buf.st_mtime)
      newfile = 1;
  }

  if(newfile || dist_mtime == 0) {
    prmsg(DMSG,("Spatial filename changed from %s to %s\n",FILENAME,filename));
    if(dist_mtime)
      pfree(FILENAME);
    if(stat(filename,&stat_buf) != 0)
      prmsg(MSG,("Cannot access distortion file %s\n",filename));
    else
      dist_mtime = (int)stat_buf.st_mtime;
    SPLINE_INVALID = 1;
    LUT_INVALID = 1;
    FILENAME = (char *)pmalloc(strlen(filename) + 1);
    strcpy(FILENAME,filename);
  }
}

/*
 * Sets the user-definable variable "verbose" which controls the level of
 * message printing from the program:
 *
 * - if verbose == -1, nothing is printed;
 * - if verbose ==  0, print only the message with types ERROR or FATAL;
 * - if verbose ==  1, print all messages except those of type DMSG (debug);
 * - if verbose ==  2, print all messages including debugging messages.
 *
 * Default value if not defined by the user: 1
 *
 * Input : verb_in: new value for variable "verbose"
 * Output: none
 * Return: none
 */

void set_verbose(int verb_in)
{
  verbose = verb_in;
}

/*
 * Sets the user-definable variables XINFILE and YINFILE. The corresponding
 * files contain for each pixel of the source image the spatial distortion
 * correction values in x direction (XINFILE) and y direction (YINFILE).
 *
 * Default value for both variables if not defined by the user: NULL pointer
 *
 * Input : xfile: new value for variable XINFILE
 *         yfile: new value for variable YINFILE
 * Output: none
 * Return: none
 */

void set_xycorin(char *xfile,char *yfile)
{
  int xflen,yflen;

  if((xflen = strlen(xfile)) == 0 || (yflen = strlen(yfile)) == 0)
    return;

  if(XINFILE) {
    pfree(XINFILE);
    XINFILE = NULL;
  }
  if(YINFILE) {
    pfree(YINFILE);
    YINFILE = NULL;
  }

  XINFILE = (char *)pmalloc((xflen + 1) * sizeof(char));
  strcpy(XINFILE,xfile);
  YINFILE = (char *)pmalloc((yflen + 1) * sizeof(char));
  strcpy(YINFILE,yfile);

  LUT_INVALID = 1;

}

/*
 * Sets the user-definable variables XOUTFILE and YOUTFILE. If both are set, the
 * spatial distortion correction values for each pixel of the source image will
 * be saved in XOUTFILE (for x direction) and YOUTFILE (for y direction).
 *
 * Default value for both variables if not defined by the user: NULL pointer
 * (i.e., not set)
 *
 * Input : xfile: new value for variable XOUTFILE
 *         yfile: new value for variable YOUTFILE
 * Output: none
 * Return: none
 */

void set_xycorout(char *xfile,char *yfile)
{
  int xflen,yflen;

  if((xflen = strlen(xfile)) == 0 || (yflen = strlen(yfile)) == 0)
    return;

  if(XOUTFILE) {
    pfree(XOUTFILE);
    XOUTFILE = NULL;
  }
  if(YOUTFILE) {
    pfree(YOUTFILE);
    YOUTFILE = NULL;
  }

  XOUTFILE = (char *)pmalloc((xflen + 1) * sizeof(char));
  strcpy(XOUTFILE,xfile);
  YOUTFILE = (char *)pmalloc((yflen + 1) * sizeof(char));
  strcpy(YOUTFILE,yfile);

}

/*
 * Sets the user-definable variables XSIZE and YSIZE. These are the number of
 * pixels in the x- and y-direction.
 *
 * The x-direction is the "fast-moving" index in a multidimensional C array,
 * the y-direction is the "slow-moving" one.
 *
 * Default value for both variables if not defined by the user: 1024
 *
 * Input : xsize: new value for variable XSIZE
 *         ysize: new value for variable YSIZE
 * Output: none
 * Return: none
 */

void set_xysize(int xsize,int ysize)
{
  XSIZE = xsize;
  YSIZE = ysize;
}

#if defined(ANDY_CORR)

/*
 * Reads the spline function parameters from a file and fills them into the
 * corresponding elements of a newly created spd_spline structure.
 *
 * Memory for the arrays of the structure is allocated by the function itself
 * and can be freed by a call to unloadspd().
 *
 * Input : filename: name of the file that contains the spline function
 *                   parameters
 * Output: none
 * Return: if successful: pointer to the created spline function structure
 *         else:          NULL
 */

struct spd_spline *loadspdfile(char *filename)
{
  FILE *fp;
  char *line;
  struct spd_spline *spline;
  int nopar, maxspace;

  if((fp = fopen(filename,"r")) == NULL) {
    prmsg(ERROR,("Cannot read <%s> (open failed)\n",filename));
    return(NULL);
  }

  if((spline = (struct spd_spline *)pmalloc(sizeof(struct spd_spline))) == NULL)
    goto memerror;

  /* Read in valid region in pixels */
  if(findkeyword(fp,"VALID REGION"))
    goto readerror;

  if(((line = myfgets(fp)) == NULL) || (sscanf(line,"%lf %lf %lf %lf",
    &(spline->reg_x0),&(spline->reg_y0),&(spline->reg_x1),&(spline->reg_y1))
    != 4))
    goto readerror;

  if(findkeyword(fp,"GRID SPACING"))
    goto readerror;

  if(((line = myfgets(fp)) == NULL) || (sscanf(line,"%lf %lf %lf",
    &(spline->grid_space),&(spline->xcor_size),&(spline->ycor_size)) != 3))
    goto readerror;

  /* Read in X Distortion data */
  if(findkeyword(fp,"X-DISTORTION"))
    goto readerror;

  if(((line = myfgets(fp)) == NULL) ||
    (sscanf(line,"%d %d",&(spline->dx_xknots),&(spline->dx_yknots)) != 2))
    goto readerror;

  nopar = (spline->dx_yknots-4) * (spline->dx_xknots-4);

  if(!(spline->xpars = (float *)pmalloc(sizeof(float) * nopar)) ||
    !(spline->xmu = (float *)pmalloc(sizeof(float) * spline->dx_yknots)) ||
    !(spline->xlambda = (float *)pmalloc(sizeof(float) * spline->dx_xknots)))
    goto memerror;

  if(readarray(fp,spline->dx_xknots,spline->xlambda))
    goto readerror;

  if(readarray(fp,spline->dx_yknots,spline->xmu))
    goto readerror;

  if(readarray(fp,nopar,spline->xpars))
    goto readerror;

  /* Read in Y Distortion data */
  if (findkeyword(fp,"Y-DISTORTION"))
    goto readerror;

  if(((line = myfgets(fp)) == NULL) ||
    (sscanf(line,"%d %d",&(spline->dy_xknots),&(spline->dy_yknots)) != 2))
    goto readerror;

  nopar = (spline->dy_yknots-4) * (spline->dy_xknots-4);

  if(!(spline->ypars = (float *)pmalloc(sizeof(float) * nopar)) ||
    !(spline->ymu = (float *)pmalloc(sizeof(float) * spline->dy_yknots)) ||
    !(spline->ylambda = (float *)pmalloc(sizeof(float) * spline->dy_xknots)))
    goto memerror;

  maxspace = 3000 * 5 * 2;
  spline->maxspace = maxspace;

  if(!(spline->wrk = (float *)pmalloc(sizeof(float) * maxspace)) ||
    !(spline->iwrk = (int *)pmalloc(sizeof(int) * maxspace)))
    goto memerror;

  if(readarray(fp,spline->dy_xknots,spline->ylambda))
    goto readerror;

  if(readarray(fp,spline->dy_yknots,spline->ymu))
    goto readerror;

  if(readarray(fp,nopar,spline->ypars))
    goto readerror;

  fclose(fp);
  return(spline);

 readerror:
  prmsg(ERROR,("Error reading from file <%s>\n",filename));
  fclose(fp);
  return(NULL);

 memerror:
  prmsg(ERROR,("Cannot allocate memory\n"));
  fclose(fp);
  return(NULL);

}

/*
 * Searches the input file for a line containing a specified keyword.
 *
 * Input : fp:      stream pointer referring to the input file
 *         keyword: the keyword to search for
 * Output: none
 * Return: 0  if found
 *         1  otherwise
 */

findkeyword(FILE *fp,char *keyword)
{
  char *line;

  while((line = myfgets(fp)) != NULL) {
    if((char *)strstr(line,keyword) != NULL)
      return(0);
  }
  prmsg(MSG,("Missing keyword \"%s\" in file \n",keyword));
  return(1);
}

/*
 * Reads a number of values from the input file and fills them into an array.
 *
 * Input : fp:  stream pointer referring to the input file
 *         no:  the number of values to read
 *         arr: pointer to the array where the values will be stored
 * Output: none
 * Return: 0  if requested number of values was read
 *         1  otherwise
 */

readarray(FILE *fp,int no,float *arr)
{
  int i, j, idx=0;
  char *line,*s;
  char dammed_minus_copy[512];

  while((line = myfgets(fp)) != NULL) {

    /*
     * Handles the special case that the input might contain tokens that are
     * separated by a "-" (minus) instead of a blank, but the "-" is actually
     * part of the next token (e.g. 2345-4567). Insert a blank before the "-".
     */
    for(i=0, j=0; i <= strlen(line); i++) {
      if(i && line[i] == '-' && line[i-1] != ' ')
        dammed_minus_copy[j++] = ' ';
      dammed_minus_copy[j++] = line[i];
    }
    line = dammed_minus_copy;

    for(i=0; i< VALUES_PERLINE; i++) {
      if((s = (char*) strtok(line," ")) == NULL)  {
	prmsg(MSG,("Not enough values in file\n"));
	return(1);
      }
      line = NULL;
      arr[idx++] = atof(s);
      if(idx == no)
	return(0);
    }
  }
  prmsg(MSG,("Unexpected end of file\n"));
  return(1);
}

/*
 * Free the buffers allocated for the input spline function structure.
 *
 * Input : spline: structure with the parameters for the spline function
 * Output: none
 * Return: none
 */

unloadspd(struct spd_spline *spline)
{
  if(!spline)
    return;
  if(spline->xpars)
    pfree(spline->xpars);
  if(spline->xmu)
    pfree(spline->xmu);
  if(spline->xlambda)
    pfree(spline->xlambda);
  if(spline->ypars)
    pfree(spline->ypars);
  if(spline->ymu)
    pfree(spline->ymu);
  if(spline->ylambda)
    pfree(spline->ylambda);
  if(spline->wrk)
    pfree(spline->wrk);
  if(spline->iwrk)
    pfree(spline->iwrk);
  if(spline)
    pfree(spline);
}

/*
 * Computes the spline function values for x and y direction at the grid formed
 * by the input coordinate points. It will calculate dx, dy at all the
 * intersections of the px, py arrays. E.g. with px = [2,3] and py = [7,8] this
 * routine will calculate dx,dy for the 4 points ((2,7), (2,8), (3,7), (3,8)).
 *
 * Input : spline:   structure with the parameters of the x and y spline
 *                   functions
 *         nox, noy: number of points in x and y arrays (= size in x and in y)
 *         px, py:   arrays with the x and y coordinates
 * Output: dx, dy: arrays with the x and y spline values at the corresponding
 *                 (px,py) coordinate points (will be nox * noy points)
 * Return: 0 if no errors
 *         error code from BISPEV call otherwise
 */

pxcorrgrid(struct spd_spline *spline,int nox,int noy,float px[],float py[],
  float dx[],float dy[])
{
  int ifail=0,i;
  int lwrk  = spline->maxspace;
  int liwrk = spline->maxspace;
  int ideg = 3;
  float *ddx,*ddx_p;
  float *dx_p = dx, *dy_p = dy;
  static long buffer[20000];

  if((ddx_p = ddx = (float *)pmalloc(nox * noy * sizeof (float))) == NULL) {
    prmsg(ERROR,("No memory in pxcalcgrid\n"));
    return;
  }

  /*
   * Note that BISPEV() is a FORTRAN routine, and that the FORTRAN calling
   * convention expects all arguments to be pointers. Thus even the constants
   * used for the degree of the spline must be given as pointers to variables
   * (in this case the variable "ideg").
   */
  BISPEV(spline->xlambda,&spline->dx_xknots,spline->xmu,&spline->dx_yknots,
    spline->xpars,&ideg,&ideg,
    px,&nox,py,&noy,
    ddx,spline->wrk,&lwrk,spline->iwrk,&liwrk,&ifail);
  for (i = 0; i < nox * noy; i++)
    *dx_p++ = *ddx_p++;

  if(ifail)
    prmsg(ERROR,("Error %d in e02dff\n",ifail));

  BISPEV(spline->ylambda,&spline->dy_xknots,spline->ymu,&spline->dy_yknots,
    spline->ypars,&ideg,&ideg,
    px,&nox,py,&noy,
    ddx,spline->wrk,&lwrk,spline->iwrk,&liwrk,&ifail);
  ddx_p = ddx;
  for (i = 0; i < nox * noy; i++)
    *dy_p++ = *ddx_p++;

  pfree(ddx);

  if(ifail)
    prmsg(ERROR,("Error %d in e02dff\n",ifail));

  return(ifail);
}
#endif /* ANDY_CORR */

/*
 * Cuts the input triangle along a vertical line into a (smaller) triangle and
 * an irregular quadrangle. The new coordinates of the triangle are stored in
 * the old input triangle. The quadrangle is divided into two new triangles,
 * and their coordinates are stored in "new_triangles".
 *
 * Under particular circumstances there may be only one or even no new triangle
 * created (if the cut line goes through one of the corners or coincides with
 * one of the sides of the input triangle).
 *
 * Input : in_tri: structure with the x and y coordinates of a triangle
 *         v:      x coordinate of the vertical cut line
 * Output: new_triangles: structures with the x and y coordinates of the new
 *                        triangles created
 *         no_new:        number of the new triangles created (normally 2)
 * Return: 0
 */

int trianglecutv_only(struct triangle *in_tri,float v,
  struct triangle new_triangles[],int *no_new)
{
  float U1,U2,U3;
  int i,j,next,afternext;
  float py,ndx,sum,sum2,area,len,dx,dy;
  struct triangle *new = new_triangles;
  /*
   * Set pointers to the old (input) and the new (to be created) triangle:
   * - tri, x and y point to the old triangle;
   * - nx and ny point to the new triangle.
   */
  struct triangle *tri = in_tri;
  float *x = tri->x, *y = tri->y;
  float *nx = new->x, *ny = new->y;
  int iv= (int)v;


  /*
   * Define first point as "previous point".
   * Is this previous point to the right of the cut line?
   */
  *no_new = 0;
  U1 = x[0] > v;
  for(i = 0; i < 3; i++) {

    /*
     * Determine next point.
     * Is this next point to the right of the cut line?
     */
    next = (i == 2) ? 0 : (i + 1);
    U2 = x[next] > v;

    /*
     * If the previous and the next point are not on the same side of the cut
     * line, then cut the triangle at the cut line between previous point and
     * next point.
     *
     * The resulting two triangles will consist of the following points:
     * - old triangle: previous point, cut point, and "afternext" point;
     * - new triangle: cut point, next point, and "afternext" point.
     *
     * The new triangle will thus share two points with the old one (cutpoint
     * and afternext).
     *
     * If the previous and the next point are on the same side of the cut line,
     * then the cut will not happen between these two points. Just continue the
     * loop and test the next pair of points.
     */
    if(U1 != U2) {

      /*
       * Calculate cut point.
       */
      ndx = v - x[i];
      py = y[i] + ndx * (y[next] - y[i]) / (x[next] - x[i]);

      /*
       * Copy old triangle to new triangle.
       */
      memcpy(new,tri,sizeof(struct triangle));

      /*
       * Put cut point as new point into the old and the new triangle.
       */
      x[next] = nx[i] = v;
      y[next] = ny[i] = py;

      if(new == new_triangles) {

        /*
         * If this was the first cut, then there are now two triangles: the
         * modified input triangle and the newly created one.
         *
         * One of the two will be entirely on one side of the cut line, the
         * other needs to be cut again along the cut line.
         *
	 * Find out which triangle needs to be cut.
         */
	afternext = (next == 2) ? 0 : (next + 1);
	U3 = x[afternext] > v;

	/*
         * If the new triangle has to be cut:
         *
         * - cycle all triangle pointers: the second triangle will become the
         *   old one, and the (to be created) third will be the new triangle
         *   (i.e., "tri", "x" and "y" will point to the second, "nx" and "ny"
         *   to the third triangle);
         *
         * - then just apply the same algorithm again: this will cut the second
         *   triangle in the same way as the first one was cut before, with the
         *   new cut point stored in the coordinates of the second and third
         *   triangle.
         *
         * If the old (input) triangle has to be cut:
         *
         * - cycle just the pointers to the new triangle: the (to be created)
         *   third triangle will be the new triangle, but the first triangle
         *   will remain the old one (i.e., "tri", "x" and "y" will point to the
         *   first, "nx" and "ny" to the third triangle);
         *
         * - then just apply the same algorithm again: this will cut the first
         *   triangle again in the same way as it was cut before, with the new
         *   cut point stored in the coordinates of the first and third
         *   triangle.
         */
	if(U3 == U1) {
	  tri = new;
	  x = nx;
	  y = ny;
	}
	new++;
	nx = new->x;
	ny = new->y;
	*no_new = 1;

      } else {
        /*
         * If this was not the first cut, then the task is finished. Just
         * increase the triangle count, terminate the loop and return.
         */
	*no_new = 2;
	break;
      }
    }
    U1 = U2;
  }
  return(0);
}

/*
 * Does the same as routine trianglecutv_only(), but cuts along a horizontal
 * line.
 *
 * See routine trianglecutv_only() for details of how the routine works.
 *
 * Input : in_tri: structure with the x and y coordinates of a triangle
 *         v:      y coordinate of the horizontal cut line
 * Output: new_triangles: structures with the x and y coordinates of the new
 *                        triangles created
 *         no_new:        number of the new triangles created (normally 2)
 * Return: 0
 */

int trianglecuth_only(struct triangle *in_tri,float v,
  struct triangle new_triangles[],int *no_new)
{
  float U1,U2,U3;
  int i,j,next,afternext;
  float px,ndy,sum,sum2,area,len,dx,dy;
  struct triangle *new = new_triangles;
  struct triangle *tri = in_tri;
  float *x = tri->x,*y = tri->y;
  float *nx = new->x,*ny = new->y;
  int iv = (int) v;

  *no_new = 0;
  U1 = y[0] > v;

  for(i = 0; i < 3; i++) {
    next = (i == 2) ? 0 : (i + 1);
    U2 = y[next] > v;

    if(U1 != U2) {
      ndy = v - y[i];
      px = x[i] + ndy * (x[next] - x[i]) / (y[next] - y[i]);
      memcpy(new,tri,sizeof(struct triangle));

      x[next] = nx[i] = px;
      y[next] = ny[i] = v;

      if(new == new_triangles) {
	afternext = (next == 2) ? 0 : (next + 1);

	U3 = y[afternext] > v;
	if (U3 == U1) {
	  tri = new;
	  x = nx;
	  y = ny;
	}
	new++;
	nx = new->x;
	ny = new->y;
	*no_new = 1;

      } else {
	*no_new = 2;
	break;
      }
    }

    U1 = U2;
  }
}

/*
 * Calculates the area of the triangle formed by the three coordinate pairs in
 * the input arrays x and y. It uses Heron's formula
 *
 *      area = sqrt(s * (s-a) * (s-b) * (s-c))
 *
 *      where a,b,c are the three sides of the triangle and s = (a+b+c) / 2
 *
 * Input : x, y: arrays with the three x and y coordinates of the triangle
 * Output: area: the area of the triangle
 * Return: none
 */

area_only(float *x,float *y,float *area)
{
  float dx,dy;
  float sum = 0;
  float len1,len2,len3;
  float insum;

  dx = x[0] - x[2];
  dy = y[0] - y[2];
  sum = len1 = sqrt (dx * dx + dy * dy);
  dx = x[1] - x[0];
  dy = y[1] - y[0];
  len2 = sqrt (dx * dx + dy * dy);
  sum += len2;
  dx = x[2] - x[1];
  dy = y[2] - y[1];
  len3 = sqrt (dx * dx + dy * dy);
  sum += len3;

  sum /= 2;
  insum = sum * (sum - len1) * (sum - len2) * (sum - len3);
  if(insum >= 0)
    *area = sqrt (sum * (sum - len1) * (sum - len2) * (sum - len3));
  else {
    *area = 0;
#if 0
    prmsg(MSG,("SQRT error, %f,%f %f,%f %f,%f (%f)",x[0],y[0],x[1],y[1],x[2],
      y[2],insum));
#endif
  }
}

/*
 * Cuts the input triangle into several new ones such that each of them is
 * fully contained in one of the squares of an integer valued x and y grid.
 *
 * Input : triangles[0]: structure with the x and y coordinates of a triangle
 * Output: triangles: structures with the x and y coordinates of the new
 *                    triangles created. These are the integer-truncated values
 *                    of the triangle's center (the (smallest x, smallest y)
 *                    corner of the grid square the triangle is in)
 *         n:         number of the new triangles created
 *         ixmin:     the x coordinate of the vertical grid line just to the
 *                    left of the input triangle
 *         ixmax:     the x coordinate of the rightmost vertical grid line that
 *                    still has part of the input triangle to its right
 *         iymin:     the y coordinate of the horizontal grid line just below
 *                    the input triangle
 *         iymax:     the y coordinate of the highest horizontal grid line that
 *                    still has part of the input triangle above it
 * Return: 0
 */

int triangle_cutall(struct triangle triangles[],int *n,int *ixmin,int *ixmax,
  int *iymin,int *iymax)
{
  int i,j;
  int no_add,no_tri,free,total_add;
  int cutxmin,cutxmax,cutymin,cutymax;
  float *x = triangles[0].x, *y = triangles[0].y;
  float xmin,xmax,ymin,ymax;

  /*
   * Get the smalles and largest x and y value of the input triangle.
   */
  xmin = x[0];
  if(xmin > x[1])
    xmin = x[1];
  if(xmin > x[2])
    xmin = x[2];
  ymin = y[0];
  if(ymin > y[1])
    ymin = y[1];
  if(ymin > y[2])
    ymin = y[2];
  xmax = x[0];
  if(xmax < x[1])
    xmax = x[1];
  if(xmax < x[2])
    xmax = x[2];
  ymax = y[0];
  if(ymax < y[1])
    ymax = y[1];
  if(ymax < y[2])
    ymax = y[2];

  /*
   * Determine the leftmost and the rightmost vertical cut line, as well as
   * the lowest and highest horizontal cut line.
   *
   * A valid vertical cut line has a part of the triangle to the left and
   * another part to the right. The leftmost vertical cut line is the one
   * with the smallest x value that still has some of the triangle to its left.
   * Note that in extreme cases this "some" may be empty, if the cut line goes
   * through the leftmost point of the triangle.
   *
   * The rightmost vertical cut line is determined in an analog way by
   * selecting the vertical cut line with the largest x value that still has
   * some of the triangle to its right.
   *
   * The same philosophy is then applied to select the lowest and highest
   * horizontal cut line.
   */
  cutxmin = ceil (xmin);
  cutxmax = floor (xmax);
  cutymin = ceil (ymin);
  cutymax = floor (ymax);

  /*
   * Cut the input triangle along the vertical grid (formed by all integer
   * values between the leftmost and rightmost vertical cut line) into several
   * triangles in such a way that each triangle is entirely contained between
   * two grid lines.
   *
   * no_tri counts the number of triangles, the first cut triangle will replace
   * the input triangle, and the other ones are stored into subsequent
   * locations in the structure array triangles.
   */
  no_tri = 1;
  free = 1;

  for(i = cutxmin; i <= cutxmax; i++) {
    total_add = 0;
    for(j = 0; j < no_tri; j++) {
      trianglecutv_only(triangles + j,(float)i,triangles + free,&no_add);
      free += no_add;
      if(free >= MAX_TRIANGLES)
	prmsg(ERROR,("Too many triangles (%d) \n",free));
      total_add += no_add;
    }
    no_tri += total_add;
  }

  /*
   * Now cut all these created triangles along the horizontal grid (formed by
   * the integer values between the lowest and highest horizontal cut line)
   * into several triangles in such a way that each triangle is entirely
   * contained in the square between adjacent grid lines.
   */
  for(i = cutymin; i <= cutymax; i++) {
    total_add = 0;
    for(j = 0; j < no_tri; j++) {
      trianglecuth_only(triangles + j,(float)i,triangles + free,&no_add);
      free += no_add;
      if(free >= MAX_TRIANGLES)
	prmsg(ERROR,("Too many triangles (%d) \n",free));
      total_add += no_add;
    }
    no_tri += total_add;
  }

  /*
   * Determine the grid coordinates for each triangle. These are the
   * integer-truncated values of the triangle's center (i.e., the
   * (smallest x, smallest y) corner of the grid square the triangle is in).
   */
  for(i = 0; i < no_tri; i++) {
    triangles[i].ypos = (int)
      ((triangles[i].y[0] + triangles[i].y[1] + triangles[i].y[2]) / 3);
    triangles[i].xpos = (int)
      ((triangles[i].x[0] + triangles[i].x[1] + triangles[i].x[2]) / 3);
  }

  *ixmin = cutxmin - 1;
  *iymin = cutymin - 1;
  *ixmax = cutxmax;
  *iymax = cutymax;
  *n = no_tri;
}

/*
 * Cuts the input pixel into segments along an integer-values horizontal and
 * vertical grid. It determines the area of the input pixel that falls into
 * each grid square and the minimal and maximal grid values in x and y.
 *
 * To achieve this, the input pixel is cut in two triangles, which in turn are
 * cut into more triangles in such a way that at the end each of the created
 * triangles is fully contained in one square of the grid.
 *
 * Input : x, y:  arrays with the (four) x and y coordinates of a pixel
 *         debug: debugging flag: produce debugging output if != 0
 * Output: parts: array that gives for each grid square the area of the input
 *                pixel it contains
 *         txmin: the x coordinate of the vertical grid line just to the left
 *                of the input pixel
 *         txmax: the x coordinate of the rightmost vertical grid line that
 *                still has part of the input pixel to its right
 *         tymin: the y coordinate of the horizontal grid line just below the
 *                input pixel
 *         tymax: the y coordinate of the highest horizontal grid line that
 *                still has part of the input pixel above it
 *         total: the area of the input pixel
 * Return: number of triangles created
 */

calcparts(float x[],float y[],int debug,float parts[],int *txmin,int *txmax,
  int *tymin,int *tymax,float *total)
{
  int xmin,xmax,ymin,ymax;
  int xmin2,xmax2,ymin2,ymax2;
  struct triangle triangles[MAX_TRIANGLES];
  float totalarea;
  float totalarea2;
  int no_tri,no_tri2;
  int n,i,j,idx;

  /*
   * Divide the pixel into two triangles along the line from point "0" to
   * point "2":
   *
   *                 0             1
   *                  -------------
   *                  |           |
   *                  |           |
   *                  |           |
   *                  |           |
   *                  -------------
   *                 3             2
   *
   * Calculate the areas of these two triangles, then cut each into several
   * triangles along an integer-valued horizontal and vertical grid in such a
   * way that each of those new triangles is entirely contained between
   * neighbouring grid lines.
   */
  triangles[0].x[0] = x[0]; triangles[0].y[0] = y[0];
  triangles[0].x[1] = x[1]; triangles[0].y[1] = y[1];
  triangles[0].x[2] = x[2]; triangles[0].y[2] = y[2];
  area_only(triangles[0].x,triangles[0].y,&totalarea);
  triangle_cutall(triangles,&no_tri,&xmin,&xmax,&ymin,&ymax);

  triangles[no_tri].x[0] = x[2]; triangles[no_tri].y[0] = y[2];
  triangles[no_tri].x[1] = x[3]; triangles[no_tri].y[1] = y[3];
  triangles[no_tri].x[2] = x[0]; triangles[no_tri].y[2] = y[0];

  area_only(triangles[no_tri].x,triangles[no_tri].y,&totalarea2);
  triangle_cutall(triangles + no_tri,&no_tri2,&xmin2,&xmax2,&ymin2,&ymax2);

  /*
   * Get the total number of triangles, the area and the smalles and biggest
   * grid values in x and y.
   */
  no_tri += no_tri2;
  totalarea += totalarea2;
  if(xmin2 < xmin)
    xmin = xmin2;
  if(ymin2 < ymin)
    ymin = ymin2;
  if(xmax2 > xmax)
    xmax = xmax2;
  if(ymax2 > ymax)
    ymax = ymax2;

  /*
   * Calculate for each grid square the area of the input pixel it contains,
   * and store it in the array "parts". This array contains "number of x grid
   * squares" times "number of y grid squares" elements, and the grid squares
   * are stored in row-fashion (x increases fastest).
   */
  n = (xmax - xmin + 1) * (ymax - ymin + 1);
  if(n > MAX_PARTS) {
    prmsg(ERROR,("To many parts (%d) - pixel too big\n",n));
    exit(-1);
  }

  for(i = 0; i < n; i++)
    parts[i] = 0;

  for(j=0; j <no_tri; j++) {
    area_only(triangles[j].x,triangles[j].y,&triangles[j].area);
    idx = (triangles[j].xpos - xmin) +
      (triangles[j].ypos - ymin) * (xmax - xmin + 1);
    parts[idx] += triangles[j].area;
  }

  if(debug)
    debugout(triangles,no_tri,totalarea,parts,xmin,xmax,ymin,ymax);

  *txmin = xmin; *txmax = xmax; *tymin = ymin; *tymax = ymax;
  *total = totalarea;
  return(no_tri);
}

/*
 * Print for debugging purposes the values calculated by "calcparts()".
 *
 * Input : triangles: structure array with the triangles created
 *         no_tri:    number of triangles created
 *         total:     the area of the pixel
 *         parts:     array that gives for each grid square the area of the
 *                    input pixel it contains
 *         xmin:      the x coordinate of the vertical grid line just to the
 *                    left of the pixel
 *         xmax:      the x coordinate of the rightmost vertical grid line that
 *                    still has part of the pixel to its right
 *         ymin:      the y coordinate of the horizontal grid line just below
 *                    the pixel
 *         ymax:      the y coordinate of the highest horizontal grid line that
 *                    still has part of the pixel above it
 * Output: none
 * Return: none
 */

int debugout(struct triangle triangles[],int no_tri,float total,float parts[],
  int xmin,int xmax,int ymin,int ymax)
{
  FILE *file;
  float sumparts;
  int i,j,idx = 0;
  float fxmin,fxmax,fymin,fymax;

  /*
   * Print smallest and largest grid value in x and y.
   */
  printf("Min Max: X %d %d Y %d %d\n",xmin,xmax,ymin,ymax);

  /*
   * Print for each triangle the coordinates of the corners, the area and the
   * grid coordinates.
   *
   * Then test if the triangle is fully contained in its grid square, and
   * print an error message if not.
   */
  for(i = 0; i < no_tri; i++) {
    fxmin = triangles[i].x[0];
    fymin = triangles[i].y[0];
    fxmax = triangles[i].x[0];
    fymax = triangles[i].y[0];
    printf("Triangle %d:\n",i);
    for(j = 0; j < 3; j++) {
      printf("x[%d]=%7.2f, y[%d]=%7.2f\n",j,triangles[i].x[j],
        j,triangles[i].y[j]);
      if(fxmin > triangles[i].x[j])
	fxmin = triangles[i].x[j];
      if(fymin > triangles[i].y[j])
	fymin = triangles[i].y[j];
      if(fxmax < triangles[i].x[j])
	fxmax = triangles[i].x[j];
      if(fymax < triangles[i].y[j])
	fymax = triangles[i].y[j];
    }
    printf("     area = %7.2f, xpos = %d, ypos = %d\n",
      triangles[i].area,triangles[i].xpos,triangles[i].ypos);
    if(fxmin < (float)triangles[i].xpos ||
      fxmax > (float)triangles[i].xpos + 1 ||
      fymin < (float)triangles[i].ypos ||
      fymax > (float)triangles[i].ypos + 1) {
      printf("Error: x[%f:%f] y[%f:%f]\n",fxmin,fxmax,fymin,fymax);
    }

  }

  /*
   * Print the area of the input pixel contained in each grid square.
   */
  for(j = ymin; j <= ymax; j++) {
    for(i = xmin; i <= xmax; i++) {
      printf("%6.2f ",parts[idx++]);
    }
    printf("\n");
  }

  /*
   * Print the total area of the input pixel, determined as the sum over all
   * the triangle areas.
   */
  sumparts = 0;
  for(i = 0; i < no_tri; i++)
    sumparts += triangles[i].area;
  printf("============  %f (total) = %f (sum of parts)\n",total,sumparts);

  /*
   * Write all the triangle coordinates to the file "/tmp/triangles".
   */
  file = fopen("/tmp/triangles","w");
  for(i = 0; i < no_tri; i++) {
    fprintf(file,"%f %f 1\n%f %f\n%f %f\n%f %f\n",
      triangles[i].x[0],triangles[i].y[0],triangles[i].x[1],triangles[i].y[1],
      triangles[i].x[2],triangles[i].y[2],triangles[i].x[0],triangles[i].y[0]);
  }
  fclose(file);
}

/*
 * This routine handles the printing of user-defined messages.
 *
 * It works by being called from the macros "prmsg" or "__prmsg" (see
 * corresponding description above). These macros call this routine twice:
 * - first with four arguments, of which the first is == NULL. The remaining
 *   three contain the message type, the line number and the file name;
 * - then with a variable number of arguments. The first (which must not be
 *   NULL) contains the format string, the remaining ones the parameters to
 *   print.
 *
 * Message have one of the types MSG, DMSG, WARNING, ERROR or FATAL.
 *
 * Depending on the message type and the value of the user-defined "verbose"
 * global variable, the message may or may not be printed:
 * - if verbose == -1, return without any action;
 * - if verbose ==  0, print only the message with types ERROR or FATAL;
 * - if verbose ==  1, print all messages except those of type DMSG;
 * - if verbose ==  2, print all messages.
 *
 * If the message type is ERROR or FATAL, it will be printed together with the
 * system error message for the error number "errno" as well as the current
 * line number and file name (as defined by the C preprocessor macros __LINE__
 * and __FILE__).
 *
 * If the message type is FATAL, the program will then be terminated by
 * exit(1). Note that this will not happen if verbose == -1!
 *
 * This routine uses the "varargs" set of macros to handle a variable argument
 * list. For more details, see the "varargs" manual page.
 *
 * Input : va_alist: variable argument list
 * Output: none
 * Return: none
 */

void _prmsg(va_alist)
     va_dcl
{
  char *format;
  va_list args;
  static int type;
  static int line;
  static char file[256];

  va_start(args);

  /*
   * Get the first argument. If it is == NULL, only get the message type,
   * the current file name and the current line number (as defined by the C
   * preprocessor macros __LINE__ and __FILE__).
   *
   * If it is not NULL, it contains the format string for the printing.
   */
  format = va_arg(args,char*);

  if(format == NULL) {
    type = va_arg(args,int);
    strcpy(file,va_arg(args,char*));
    line = va_arg(args,int);
    va_end(args);
    return;
  }

  /*
   * Leave this routine without printing in some cases.
   */
  if((verbose == -1) ||
    (verbose != 1 && verbose != 2 && (type == MSG || type == WARNING)) ||
    (verbose != 2 && type == DMSG)) {
    va_end(args);
    return;
  }

  /*
   * Print the message to standard error output.
   *
   * The format string obtained earlier and the rest of the variable argument
   * list is given to a vfprintf() call for printing.
   *
   * If the message type is ERROR or FATAL, also print the system error message
   * for the error number "errno" as well as the current line number and file
   * name (as defined by the C preprocessor macros __LINE__ and __FILE__).
   *
   * If the message type is FATAL, terminate the program with exit(1).
   */
  vfprintf(stderr,format,args);
  if(type == ERROR || type == FATAL)
    fprintf(stderr,"  %s (in file %s line %d)\n",strerror(errno),file,line);
  va_end(args);

  if(type == FATAL) {
    fprintf(stderr,"Program terminated\n");
    exit(1);
  }
}

#if ANDY_CORR

char *myfgets(FILE *fp)
{
  static char buf[1024];
  return fgets(buf,1024,fp);
}

/*
 * Calculates for all pixel points between x0 and x1 (included) and for all
 * lines from y0 to y1 (included) the the corrected coordinates.
 * This routines is called from the program either for the whole image
 * or line by line (y0 == y1) to save memory.
 *
 * Input : x0, x1, y0, y1: minimal and maximal x and y coordinates of the input
 *                         image
 * Output: cor_x, cor_y: arrays with the corrected coordinate values for the
 *                       input image
 * Return: none
 */

int andy_corr(float **cor_x,float **cor_y,int x0,int x1,int y0,int y1)
{
  int ix,iy, idx,ix0;
  int keep_memory = 0;
  int xreg = x1 - x0 + 1, yreg = y1 - y0 + 1;
  static int old_xreg = 0, old_yreg = 0;
  float *px,*py;

  /*
   * If the corrected image coordinates are to be read from a file, do so and
   * return.
   */
  if(XINFILE && YINFILE)
    return(andy_fcorr(cor_x,cor_y,x0,x1,y0,y1));

  /*
   * Use the previously allocated memory areas for the corrected image
   * coordinates and for the x and y buffers if possible, i.e. if
   * - memory for those coordinates has already been allocated, and
   * - the new image or image part (i.e. a line) has the same size as the old
   *   one.
   *
   * Otherwise, free the old memory areas and allocate new ones.
   */
  if(*cor_x && *cor_y && xreg == old_xreg && yreg == old_yreg)
    keep_memory = 1;
  old_xreg = xreg; old_yreg = yreg;


  /*
   * Memory allocation for corrected image coordinates.
   */
  if(!keep_memory) {
    if(*cor_x)
      pfree(*cor_x);

    if(*cor_y)
      pfree(*cor_y);

    if((*cor_x = (float *)pmalloc(xreg * yreg * sizeof(float))) == NULL) {
      prmsg(FATAL,("No memory in andy_corr\n"));
    }

    if((*cor_y = (float *)pmalloc(xreg * yreg * sizeof(float))) == NULL) {
      prmsg(FATAL,("No memory in andy_corr\n"));
    }
  }

  /*
   * Get a new set of spline function parameters, if necessary. This includes
   * freeing the old buffers in the spline function structure, allocating new
   * buffers and reading the parameters from a file into the structure
   * elements.
   *
   * This has to be done at the first pass through the program and also every
   * time the spatial distortion file is changed (then SPLINE_INVALID == 1).
   */
  if(SPLINE_INVALID || spline == NULL) {
    if(spline != NULL) {
      unloadspd(spline);
      spline = NULL;
    }
    SPLINE_INVALID = 0;
    bench(NULL);
    spline = loadspdfile(FILENAME);
    if(spline == NULL)
      return;

  /*
   * Save the corrected x- and y-pixel sizes.
   *
   * They will be put (by corr_calc()) in the data header structure for all
   * corrrected images that are produced with this set of spline coefficients.
   *
   * Note that the units for these sizes are micro-meter in the spatial
   * distortion file, and meter in the data header.
   */
    psizex = spline->xcor_size / 1000000.;
    psizey = spline->ycor_size / 1000000.;

    prmsg(MSG,("XDist: XKnots: %d YKnots: %d YDist: XKnots: %d YKnots: %d\n",
      spline->dx_xknots,spline->dx_yknots,spline->dy_xknots,spline->dy_yknots));
    prmsg(MSG,("Size: [%4.0f %4.0f] [%4.0f %4.0f]\n",spline->reg_x0,
      spline->reg_y0,spline->reg_x1,spline->reg_y1));
  }

  /*
   * Memory allocation for x and y buffers.
   */
  if(!keep_memory) {
    if(x_buf)
      pfree(x_buf);
    if(y_buf)
      pfree(y_buf);

    if((x_buf = (float *)pmalloc(xreg * sizeof(float))) == NULL)
      prmsg(FATAL,("No memory in andy_corr for x\n"));
    if((y_buf = (float *)pmalloc(yreg * sizeof(float))) == NULL)
      prmsg(FATAL,("No memory in andy_corr for y\n"));
  }

  /*
   * Fill the x buffer with the integer values for the x-dimension of the input
   * image (all values between the smallest and largest x).
   *
   * Then do the same for y.
   */
  for(px = x_buf,ix = x0; ix <= x1; ix++)
    *(px++) = (float)ix;
  for(py = y_buf,iy = y0; iy <= y1; iy++)
    *(py++) = (float)iy;

  /*
   * For all points of the image (i.e. all combinations of values in x_buf and
   * y_buf) obtain the corrected x and y coordinates using spline functions
   * (one for the x coordinate, the other for the y coordinate).
   */
  pxcorrgrid(spline,xreg,yreg,x_buf,y_buf,*cor_x,*cor_y);

  /*
   * If there is an active area of the detector defined, do not correct the
   * coordinates for the pixels outside the active area.
   *
   * More precisely, set the coordinate corrections for all those pixels
   * identical to the corrections for the pixels on the edge of the active
   * area. This means that there is a steady transition from the coordinates
   * inside to the ones outside the active area.
   */
  if(ACTIVE_R) {
    andy_active_area(*cor_x,XSIZE/2,YSIZE/2,ACTIVE_R);
    andy_active_area(*cor_y,XSIZE/2,YSIZE/2,ACTIVE_R);
  }

  /*
   * Save the correction values in x and y direction to the two files the names
   * of which are in XOUTFILE and YOUTFILE. These files can then be read and the
   * correction values used by the correction program at the next execution, or
   * they can be inspected with a program to read edf files (e.g. dis).
   */
  if(XOUTFILE && YOUTFILE) {
    //    save_esrf_file(XOUTFILE,*cor_x,yreg,xreg,MFloat,"cor_x",-1);
    //save_esrf_file(YOUTFILE,*cor_y,yreg,xreg,MFloat,"cor_y",-1);
  }
}

/*
 * Free the x and y buffers as well as the buffers for the corrected
 * coordinates.
 *
 * Buffers that have not been allocated will be silently ignored.
 *
 * Input : cor_x, cor_y: pointers to the buffers for the corrected coordinates
 * Output: none
 * Return: none
 */

int andy_free_buffers(float *cor_x,float *cor_y)
{
  if(x_buf) {
    pfree(x_buf);
    x_buf = NULL;
  }
  if(y_buf) {
    pfree(y_buf);
    y_buf = NULL;
  }

#if WASTE4_FORSPEED
  unloadspd(spline);
  spline = NULL;
#endif /* WASTE4_FORSPEED */

  if(cor_x) {
    pfree(cor_x);
    cor_x = NULL;
  }
  if(cor_y) {
    pfree(cor_y);
    cor_y = NULL;
  }

}

/*
 * Gets for all pixel points in the input image the corrected coordinates from
 * two input files (one for the x, the other for the y coordinates).
 *
 * Input : x0, x1, y0, y1: minimal and maximal x and y coordinates of the input
 *                         image
 * Output: cor_x, cor_y: arrays with the corrected coordinate values for the
 *                       input image
 * Return: none
 */

int andy_fcorr(float **cor_x,float **cor_y,int x0,int x1,int y0,int y1)
{
  int rows,cols,err;
  int ix,iy,idx,ix0;
  int xreg = x1 - x0 + 1,yreg = y1 - y0 + 1;
  static int old_xreg = 0,old_yreg = 0;
  float *px,*py;

  /*
   * Free the previously allocated memory areas for the corrected image
   * coordinates if the spatial distortion file has changed (then
   * SPLINE_INVALID == 1) and if there are files for the corrected x and y
   * coordinates defined. The allocation of new memory will be done in
   * read_esrf_file().
   *
   * If the distortion file has not changed, there is no need to reallocate
   * the memory, as the old size is still valid.
   */
  if(SPLINE_INVALID && XINFILE && YINFILE) {

    if(*cor_x) {
      pfree(*cor_x);
      *cor_x = NULL;
    }
    if(*cor_y) {
      pfree(*cor_y);
      *cor_y = NULL;
    }
    /*
    prmsg(MSG,("Reading file %s\n",XINFILE));
    if(read_esrf_file(XINFILE,(void **)cor_x,&rows,&cols,SPDTYP,MFloat,&err)) {
      prmsg(ERROR,("Error reading file %s\n",XINFILE));
    } else {
      prmsg(MSG,("Reading file %s\n",YINFILE));
      if(read_esrf_file(YINFILE,(void **)cor_y,&rows,&cols,SPDTYP,MFloat,&err)){
	prmsg(ERROR,("Error reading file %s\n",YINFILE));
      }
    }
    */
    /*
     * Save the corrected x- and y-pixel sizes from the files for the corrected
     * x and y coordinates.
     *
     * They will be put (by corr_calc()) in the data header structure for all
     * corrrected images that are produced with this set of correction values.
     */
    psizex = img_head[SPDTYP].PSize_1;
    psizey = img_head[SPDTYP].PSize_2;
  }
}

/*
 * Sets new values for the pixels outside the active area. The active area is
 * a cercle with radius "r", its center is at (xcen, ycen).
 *
 * The pixels in the image belong to one of the following five categories:
 * - pixels that are inside the active area;
 * - pixels that are in their x-coordinate outside the active area, without
 *   regard to their y-coordinate (called "pixels to the left or right of the
 *   active area");
 * - pixels that are in their y-coordinate outside the active area, but in
 *   their x-coordinate inside the active area (called "pixels above or below
 *   the active area").
 *
 * The values of the pixels inside the active area are not changed.
 *
 * All pixels to the left (right) of the active area are assigned the pixel
 * value of the leftmost (rightmost) pixel of the active area, i.e. the pixel
 * with the coordinates (xcen - r, 0) (or, for the right side, (xcen + r, 0)).
 *
 * All pixels above (below) the active area are assigned the pixel value of
 * that pixel on the border of the active area that is exactly below (above)
 * them, i.e. the pixel that has the same x-coordinate and is situated on
 * the circumference of the arcive area circle below (above).
 *
 * Input : image:      array with the corrected coordinate values (x or y) for
 *                     the input image
 *         xcen, ycen: x and y coordinates of the center of the active area
 *         r:          radius of the active area
 * Output: image: input array with redefined values for pixels outside the
 *                active area
 * Return: none
 */

int andy_active_area(float *image,int xcen,int ycen,int r)
{
  int x,y;
  float r2 = r * r;
  float dx,dy,set = 0,setn = 0;

  prmsg(MSG,("Setting pixel outside active area (r = %d) \n",r));

  for(x = 0; x < XSIZE; x++) {
    dx = x - xcen;
    if(dx * dx < r2) {
      /*
       * Set pixel values above and below the active area.
       */
      dy = sqrt(r2 - dx * dx);
      for(y = 0; y < (int) ycen - dy; y++)
	image[x + y * XSIZE] = image[x + ((int) (ycen - dy)) * XSIZE];
      for(y = (int) ycen + dy; y < YSIZE; y++)
	image[x + y * XSIZE] = image[x + ((int) (ycen + dy)) * XSIZE];
    } else {
      /*
       * Set pixel values left and right of the active area.
       */
      if(dx > 0)
	for(y = 0; y < YSIZE; y++)
	  image[x + y * XSIZE]= image[xcen + r + ycen * XSIZE];
      else
	for(y = 0; y < YSIZE; y++)
	  image[x + y * XSIZE]= image[xcen - r + ycen * XSIZE];
    }
  }
}

#endif

/*
 * A routine used to get either a perfect grid image or a distorted grid
 * with Gauss peaks for tests purposes. The first peak center is at (15,35)
 * and then every 50 pixels in x and y direction there is another one.
 *
 * Input : image:         pointer to the image memory which has to exist
 *         height, width: height and width of the individual peaks.
 *         distorted:     outputs the peaks not as a perfect grid but
 *                        distorted with a simple algorithm (projection
 *                        of points on a ball)
 * Output: image: filled with image data
 * Return: none
 */

int make_grid(unsigned short *image,float height,float width,int distorted)
{
  float x,y,r,xcen,ycen,px,py;
  float CURV_RADIUS = 270000/180;
  int i,j;
  int i0,i1,j0,j1;
  prmsg(DMSG, ("Grid (%dx%d) with peak height : %f and width %f\n",
    XSIZE,YSIZE,height,width));
  memset(image,0,sizeof(short) * XSIZE * YSIZE);
  for(x = 15; x < XSIZE; x += 50)
    for(y = 35; y < YSIZE; y += 50) {
      xcen = x; ycen = y;
      if(distorted) {
	double dx,dy,atn,d2;
	dx = x - XSIZE / 4; dy = y - YSIZE / 3;
	d2 = CURV_RADIUS * sin (sqrt (dx * dx + dy * dy) / CURV_RADIUS);
	if(fabs(dx) > 1E-5 || fabs(dy) < 1E-5) {
	  atn = atan2(dy,dx);
	  xcen = XSIZE/4.0 + d2 * cos(atn);
	  ycen = YSIZE/3.0 + d2 * sin(atn);
	}
      }
      i0 = xcen - width * 2; i1 = xcen + width * 2 + 1;
      j0 = ycen - width * 2; j1 = ycen + width * 2 + 1;
      if(i0 >=0 && j0 >= 0 && i1 < XSIZE && j1 < YSIZE)
	for(i = i0; i < i1; i++)
	  for(j = j0; j < j1; j++) {
	    r = sqrt (pow(xcen - (float)i,2) + pow(ycen - (float)j,2));
	    image[i + j * XSIZE] = height * exp(-r / pow(width/3,2));
	  }
    }
}

/*
 * Obtains the corrected (resx,resy) coordinate values for the (ix,iy) input
 * image pixel.
 *
 * This routine uses some tricks to speed up the calculation. It will either
 * calculate the whole image or at least a whole line of the image and remember
 * the calculated values. It is much faster to do that - see the NAG manual
 * for a mathematical explaination.
 *
 * Input : ix, iy: x and y coordinate of a pixel in the input image
 * Output: resx, resy: corrected x and y coordinate of the input pixel
 * Return: none
 */

#if ANDY_CORR
my_func(int ix,int iy,float *resx,float *resy)
{
  double x,y;
  int idx;
  float xcor = 0, ycor = 0;
  static int ysave = -1;
  static int ycalc = -1;
  static float *y_saved = NULL, *x_saved = NULL;
  static old_xsize = 0;

  x = (double)ix; y = (double)iy;
  if(ix >= XSIZE)
    ix = XSIZE -1;
  if(iy >= YSIZE)
    iy = YSIZE -1;
  if(ix < 0)
    ix = 0;
  if(iy < 0)
    iy = 0;

/*
 * If the computer has only a small memory, then we do not store the full array
 * with corrected coordinates for the whole image. Instead, only one row is
 * stored, and a new one is calculated when needed. This trades computing time
 * against memory. But even with low memory the correction is calculated line by
 * line. This routine is called for every point to allow experimenting with
 * other simple algorithms.
 */

#if LOW_MEM
  if(iy == ycalc) {
    xcor = X_COR[ix];
    ycor = Y_COR[ix];
  } else if(iy == ysave) {
    xcor = x_saved[ix];
    ycor = y_saved[ix];
  } else {
    if(x_saved) {
      if(XSIZE != old_xsize) {
	pfree(x_saved);
	if((x_saved = (float *)pmalloc(XSIZE * sizeof(float))) == NULL)
	  prmsg(FATAL,("No memory \n"));
      }
      memcpy(x_saved,X_COR,XSIZE * sizeof(float));
    } else {
      x_saved = X_COR;
      X_COR = NULL;
    }

    if(y_saved) {
      if(XSIZE != old_xsize) {
	pfree(y_saved);
	if((y_saved = (float *)pmalloc(XSIZE * sizeof(float))) == NULL)
	  prmsg(FATAL,("No memory \n"));
      }
      memcpy(y_saved,Y_COR,XSIZE * sizeof(float));
    } else {
      y_saved = Y_COR;
      Y_COR = NULL;
    }

    andy_corr(&X_COR,&Y_COR,0,XSIZE-1,iy,iy);
    old_xsize = XSIZE;
    ysave = ycalc;
    ycalc = iy;
    xcor = X_COR[ix];
    ycor = Y_COR[ix];
  }
#else

  /*
   * Calculate the arrays with the corrected x and y image coordinates if
   * necessary, then get the requested values from the arrays.
   */
  if(X_COR == NULL)
    andy_corr(&X_COR,&Y_COR,0,XSIZE-1,0,YSIZE-1);
  idx = iy + YSIZE * ix;
  xcor = X_COR[idx];
  ycor = Y_COR[idx];
#endif /* LOW_MEM */

  *resx = x + xcor;
  *resy = y + ycor;
}

#endif /* ANDY_CORR */

/*
 * Gets for all pixel points in the input image the corrected coordinates.
 *
 * There are several types of correction. Any combination of them can be
 * performed according to the options chosen by the user at program startup:
 *
 * - distortion correction
 * - background subtraction
 * - floodfield correction
 * - intensity normalization
 * - scattering background subtraction
 *
 * The buffer for the output values must have been allocated before calling
 * this routine in order to get any values back. If this buffer is a NULL
 * pointer, the routine will just free all buffers of the look-up-table
 * structure and return without any further action.
 *
 * If there are images for the background, floodfield or scattering background
 * corrections, then the corresponding buffers have to be set up before calling
 * this routine by calls to set_bkgim(), set_floim() and set_scaim(). These
 * calls set the buffer pointers BKG_IM, FLO_IM and SCA_IM that are used in this
 * routine.
 *
 * If any of the required corrections fails, the image will not be processed any
 * further and the routine returns with an error.
 *
 * Input : cor_im: (empty) buffer for the corrected input pixel values
 *         src_im: buffer with the input image
 * Output: cor_im: buffer with the corrected pixel values for the input image
 * Return: -1  if any of the required corrections failed
 *         -2  if there is no output buffer given
 *          0  else
 */

int correct_image(float *cor_im,float *src_im)
{
  static int temp_siz = 0;
  static float *temp_im;
  static struct lut_descript *lut_d = NULL;
  int bkg_cor = BKG_IM || BKG_CONST != 0;

  /*
   * Free the buffers of the look-up-table structure if
   * - there is no buffer for the output, or
   * - the existing look-up table is no longer valid (then LUT_INVALID = 1,
   *   this is in particular at program startup).
   *
   * If there is no output buffer, the routine just returns afterwards with
   * return status -2.
   */
  if(LUT_INVALID || cor_im == NULL) {
    if(lut_d != NULL) {
      if(lut_d->lut)
	pfree(lut_d->lut);
      if(lut_d->prog)
	pfree(lut_d->prog);
      if(lut_d->offset_tab)
	pfree(lut_d->offset_tab);
      if(lut_d->rel_tab)
	pfree(lut_d->rel_tab);
      if(lut_d->relend_tab)
	pfree(lut_d->relend_tab);
      if(lut_d->abs_src)
	pfree(lut_d->abs_src);
      if(lut_d->temp_im)
	pfree(lut_d->temp_im);
      pfree(lut_d);
      lut_d = NULL;
    }
    if(cor_im == NULL)
      return(-2);
  }

  bench(NULL);

  /*
   * Calculate a new set of tables for the distortion correction, if necessary.
   */
  if(LUT_INVALID && DO_SPD) {
    prmsg(MSG,("Calculating look-up table: %d bit\n",LUT_BYTE ? 8 : 16));
    lut_d = lut_calc();
    LUT_INVALID = 0;
    bench("lut_calc");
  }

  if(temp_im != NULL && (XSIZE * YSIZE * sizeof(float)) != temp_siz)
    pfree(temp_im);

  if((DO_SPD || FLO_IM) && temp_im == NULL)
    if((temp_im = (float *)pmalloc(XSIZE * YSIZE * sizeof(float))) == NULL)
      prmsg(FATAL,("No memory in correct_calc for temp src\n"));
    else
      temp_siz = XSIZE * YSIZE * sizeof(float);

  /*
   * Might be a bit too many lines, but is easier to program. All the different
   * cases: with/without distortion, background image, floodfield, scattering
   * background.
   *
   * Treat first all combinations with distortion correction, then the ones
   * without.
   *
   * The floodfield correction can be done before or after the distortion
   * correction, depending on the DO_LATER flag set by the user (DO_LATER == 1
   * means that floodfield will be done after distortion).
   */
  if(DO_SPD) {
    /*
     * background floodfield distortion (distortion after floodfield)
     */
    if(!DO_LATER && bkg_cor && FLO_IM) {
      subtract_im(src_im,BKG_IM,temp_im,BKG_CONST);
      divide_insito_im(temp_im,FLO_IM);
      corr_calc(cor_im,temp_im,lut_d);
    /*
     * background distortion floodfield (floodfield after distortion)
     */
    } else if(DO_LATER && bkg_cor && FLO_IM) {
      subtract_im(src_im,BKG_IM,temp_im,BKG_CONST);
      corr_calc(cor_im,temp_im,lut_d);
      divide_insito_im(cor_im,FLO_IM);
      memcpy(temp_im,src_im,XSIZE * YSIZE * sizeof(float));
    /*
     * background distortion
     */
    } else if(bkg_cor && FLO_IM == NULL) {
      subtract_im(src_im,BKG_IM,temp_im,BKG_CONST);
      corr_calc(cor_im,temp_im,lut_d);
      memcpy(temp_im,src_im,XSIZE * YSIZE * sizeof(float));
    /*
     * floodfield distortion (distortion after floodfield)
     */
    } else if(!DO_LATER && !bkg_cor && FLO_IM){
      divide_im(src_im,FLO_IM,temp_im);
      corr_calc(cor_im,temp_im,lut_d);
    /*
     * distortion floodfield (floodfield after distortion)
     */
    } else if(DO_LATER && !bkg_cor && FLO_IM) {
      corr_calc(cor_im,src_im,lut_d);
      divide_insito_im(cor_im,FLO_IM);
      memcpy(temp_im,src_im,XSIZE * YSIZE * sizeof(float));
    /*
     * distortion
     */
    } else if(!bkg_cor && FLO_IM == NULL) {
      corr_calc(cor_im,src_im,lut_d);
      memcpy(temp_im,src_im,XSIZE * YSIZE * sizeof(float));
    }

    /*
     * Intensity normalization, scattering background, marking of invalid pixels
     */
    if(NORM_INT)
      if(normint_im(cor_im,cor_im) != 0)
        return(-1);
    if(SCA_IM)
      subtract_im(cor_im,SCA_IM,cor_im,0.);
 
    /*
     * Mark invalid pixels in the output image.
     *
     * Invalid pixels are those where at least one of the following conditions
     * is true:
     * 1) the value of the pixel in the source image is equal to the value of
     *    the "Dummy" keyword in the source image header;
     * 2) the value of the pixel in the source image is equal to the value of
     *    the "overflow" command line argument;
     * 3) the value of the pixel in the source image is less than the value of
     *    the "inp_min" command line argument or greater than the value of the
     *    "inp_max" command line argument;
     * 4) the value of the pixel in the floodfield image is 0. or equal to the
     *    vaue of the "Dummy" keyword in the floodfield image header.
     *
     * Pixels that correspond to conditions 1 to 3 will both be marked with the 
     * output image "Dummy" value by mark_overflow_nocorr(). The marking is
     * done in a temporary copy of the input image.
     *
     * If the floodfield correction has been called before the distortion
     * correction, then divide_im() and divide_intito_im() have already marked
     * the pixels corresponding to condition 4 with the output image "Dummy"
     * value in the very same temporary copy of the input image.
     *
     * Therefore, in this temporary copy of the input image, all invalid pixels
     * are now marked with the output image "Dummy" value.
     *
     * mark_overflow() takes this temporary copy of the input image, transfers
     * the illegal pixels to the corresponding distortion corrected pixels in
     * the corrected image and sets their values to the output image "Dummy"
     * value. Note that one input pixel may correspond to more than one output
     * pixel, in which case all the corresponding output pixels are set to
     * "Dummy".
     */
    mark_overflow_nocorr(src_im,temp_im);
    mark_overflow(temp_im,cor_im,lut_d,Dummy);

  } else if(bkg_cor || FLO_IM) {
    /*
     * background floodfield
     */
    if(bkg_cor && FLO_IM) {
      subtract_im(src_im,BKG_IM,temp_im,BKG_CONST);
      divide_im(temp_im,FLO_IM,cor_im);
    /*
     * background
     */
    } else if(bkg_cor && FLO_IM == NULL) {
      subtract_im(src_im,BKG_IM,cor_im,BKG_CONST);
    /*
     * floodfield
     */
    } else if(!bkg_cor && FLO_IM) {
      divide_im(src_im,FLO_IM,cor_im);
    }

    /*
     * Intensity normalization, scattering background, overflow pixel marking
     */
    if(NORM_INT)
      if(normint_im(cor_im,cor_im) != 0)
        return(-1);
    if(SCA_IM)
      subtract_im(cor_im,SCA_IM,cor_im,0.);

    mark_overflow_nocorr(src_im,cor_im);

  /*
   * Only intensity normalization and scattering background
   */
  } else if(NORM_INT || SCA_IM) {

    if(NORM_INT) {
      if(normint_im(src_im,cor_im) != 0)
        return(-1);
      if(SCA_IM)
        subtract_im(cor_im,SCA_IM,cor_im,0.);
    } else
        subtract_im(src_im,SCA_IM,cor_im,0.);

      mark_overflow_nocorr(src_im,cor_im);

  /*
   * Nothing: just copy the input source file to the corrected file.
   */
  } else {
    prmsg(MSG,("Nothing to be done\n"));
    memcpy(cor_im,src_im,XSIZE * YSIZE * sizeof(float));
    mark_overflow_nocorr(src_im,cor_im);
  }

  bench("corr_calc");
  return(0);
}

/*
 * Performs a background subtraction for each pixel of the input image and
 * stores the result into the output image.
 *
 * It is perfectly possible for this subtraction to yield a negative value
 * for any given pixel, which is then written as such (i.e. negative) in the
 * output image.
 *
 * The background can be defined by the user at program startup on a pixel
 * basis and / or on a global basis. For the former, a background image has
 * to be specified that contains the pixel values for the background. For the
 * latter, a background constant has to be specified in the input arguments.
 *
 * There are three possibilities:
 *
 * 1) there is a background image (i.e., "bkg_im" is not NULL), and the
 *    background constant is 0. Then for each pixel the background values of
 *    the background image are subtracted from the corresponding values in the
 *    input image;
 * 2) there is no background image. Then the background constant is subtracted
 *    from each pixel in the input image;
 * 3) there is a background image, but the background constant is not 0. Then
 *    for each pixel in the input image first the background constant and then
 *    the corresponding pixel value of the background image are subtracted from
 *    it.
 *
 * Input : src_im: buffer with the input image (float)
 *         bkg_im: buffer with the background image (float)
 *         bkgcin: background constant
 * Output: out_im: buffer with output image (= corrected input) (float)
 * Return: none
 */

subtract_im(float *src_im,float *bkg_im,float *out_im,float bkgcin)
{
  register float *bkgptr,*srcptr,*tempptr,*templast;
  register float bkg_const = bkgcin;

  srcptr = src_im;
  tempptr = out_im;
  templast = out_im + XSIZE * YSIZE - 1;
  if(bkg_im) {
    bkgptr = bkg_im;
    if(bkg_const == 0.)
    {
      /*
       * There is a background image, but the global background variable is 0.
       * Subtract the background image from the input image.
       */
      while(tempptr <= templast) {
        *tempptr++ = *srcptr++ - *bkgptr++;
      }
    }
    else
    {
      /*
       * There is a background image, and the global background variable is not
       * 0. Subtract the global background variable and the background image
       * from the input image.
       */
      while(tempptr <= templast) {
        *tempptr++ = *srcptr++ - bkg_const - *bkgptr++;
      }
    }
  } else {
    /*
     * There is no background image. Subtract the global background variable
     * from the input image.
     */
    while(tempptr <= templast) {
      *tempptr++ = *srcptr++ - bkg_const;
    }
  }
}

/*
 * Perform the floodfield (also called flatfield) correction and store the
 * result back into the input image.
 *
 * This takes into account the fact that a sample with absolutely uniform
 * scattering response does not necessarily produce a flat image (e.g. because
 * of a non-uniform detector response). If one takes a image of a real sample,
 * one has to correct for this non-uniformity.
 *
 * This is done by taking an image from an uniformly scattering probe (the
 * floodfield image) and then dividing the real image by the floodfield image.
 *
 * This routine multiplies each pixel of the input image with the value of the
 * corresponding pixel of the inverted floodfield image and stores the result
 * back into the input image.
 *
 * Pixels with an invalid value in the floodfield image are marked as invalid in
 * the output image by setting the value in the output image to "Dummy". For
 * details, see routine "prepare_flood()".
 *
 * This routine is identical in fuctionality to the routine divide_im(),
 * except that this routine stores the result back into the input array.
 *
 * Input : src_im: buffer with the input image (float)
 *         flo_im: buffer with the (inverted) floodfield image (float)
 * Output: src_im: buffer with corrected input image
 * Return: none
 */

divide_insito_im(float *src_im,float *flo_im)
{
  register float *srclast,*srcptr,*floptr;
  long *pfldum;

  srclast = src_im + XSIZE * YSIZE - 1;
  srcptr = src_im;
  floptr = flo_im;

  while(srcptr <= srclast) {
    *srcptr = *srcptr * *floptr++;
    srcptr++;
  }

  /*
   * Mark pixels that have an invalid value in the floodfield image by marking
   * the corresponding pixels in the output image as "invalid" as well.
   */
  if(fldumlst != NULL)
  {
    for(pfldum = fldumlst; *pfldum != -1; pfldum++)
      *(src_im + *pfldum) = Dummy;
  }
}

/*
 * Perform the floodfield (also called flatfield) correction and store the
 * result into the output image.
 *
 * This takes into account the fact that a sample with absolutely uniform
 * scattering response does not necessarily produce a flat image (e.g. because
 * of a non-uniform detector response). If one takes a image of a real sample,
 * one has to correct for this non-uniformity.
 *
 * This is done by taking an image from an uniformly scattering probe (the
 * floodfield image) and then dividing the real image by the floodfield image.
 *
 * This routine multiplies each pixel of the input image with the value of the
 * corresponding pixel of the inverted floodfield image and stores the result
 * into the output image.
 *
 * Pixels with an invalid value in the floodfield image are marked as invalid in
 * the output image by setting the value in the output image to "Dummy". For
 * details, see routine "prepare_flood()".
 *
 * This routine is identical in fuctionality to the routine divide_insito_im(),
 * except that this routine stores the result in a separate output array.
 *
 * Input : src_im: buffer with the input image (float)
 *         flo_im: buffer with the floodfield image (float)
 * Output: cor_im: buffer with output image (=corrected input) (float)
 * Return: none
 */

divide_im(float *src_im,float *flo_im,float *cor_im)
{
  register float *srcptr,*tempptr,*templast,*floptr;
  long *pfldum;

  tempptr = cor_im;
  templast = cor_im + XSIZE * YSIZE - 1;
  srcptr = src_im;
  floptr = flo_im;

  while (tempptr <= templast) {
    *tempptr++ = *srcptr++ * *floptr++;
  }

  /*
   * Mark pixels that have an invalid value in the floodfield image by marking
   * the corresponding pixels in the output image as "invalid" as well.
   */
  if(fldumlst != NULL)
  {
    for(pfldum = fldumlst; *pfldum != -1; pfldum++)
      *(cor_im + *pfldum) = Dummy;
  }
}

/*
 * Normalize the input image to absolute scattering intensities and store the
 * result into the output image.
 *
 * The values for this calculation are taken from the data header parameters'
 * structure "img_head" for the corrected image (type == CORTYP). If this
 * structure does not contain all needed values, the routine returns with an
 * error.
 *
 * Input : src_im: buffer with the input image (float)
 * Output: cor_im: buffer with corrected input image
 * Return:  0  if no errors
 *         -1  if header parameter structure does not contain all needed values
 */

int normint_im(float *src_im,float *cor_im)
{
  register float *srcptr,*corptr;
  register int i1;
  int i2;
  unsigned long reqflg;
  register float offcen_1,psize_1;
  float offcen_2,psize_2;
  float ddummy;
  register double rd2,rd2h,i2r_1,fact1;
  double dis,dis2,i2r_2;

  reqflg = FL_PSIZ1 | FL_PSIZ2 | FL_OFFS1 | FL_OFFS2 | FL_CENT1 | FL_CENT2 |
    FL_INTE1 | FL_SAMDS;
  if((img_head[CORTYP].init & reqflg) != reqflg) {
    prmsg(MSG,
      ("Needed header parameters not set - cannot normalize to intensities\n"));
    return(-1);
  }

  srcptr = src_im;
  corptr = cor_im;

  ddummy = img_head[CORTYP].DDummy;
  psize_1 = img_head[CORTYP].PSize_1;
  psize_2 = img_head[CORTYP].PSize_2;
  offcen_1 = img_head[CORTYP].Offset_1 - img_head[CORTYP].Center_1;
  offcen_2 = img_head[CORTYP].Offset_2 - img_head[CORTYP].Center_2;
  dis = img_head[CORTYP].SamplDis; 
  dis2 = dis * dis;
  fact1 = dis * psize_1 * psize_2 * img_head[CORTYP].Intens_1;

  for(i2 = 0; i2 < YSIZE; i2++) {

    i2r_2 = INDEX2R(i2,offcen_2,psize_2);
    rd2h = dis2 + i2r_2 * i2r_2;
    for(i1 = 0; i1 < XSIZE; i1++, corptr++, srcptr++) {
      /*
       * Do not process pixels with illegal values.
       */
      if(DUMMY(*srcptr,Dummy,ddummy))
        continue;
      i2r_1 = INDEX2R(i1,offcen_1,psize_1);
      rd2 = rd2h + i2r_1 * i2r_1;
      *corptr = *srcptr * sqrt(rd2) * rd2 / fact1;
    }
  }
  return(0);
}

/*
 * Scans the input image for pixels marked as illegal (value = image_over)
 * and marks the corresponding pixels in the distortion corrected output image.
 *
 * The value used for marking these pixels is "Dummy". This value can be defined
 * with the command line argument "dummy", otherwise the value of the "Dummy"
 * keyword in the source image header is taken. The default (if neither is
 * specified) is 0.
 *
 * If "image_over" is not a valid dummy value (i.e., DUMMYDEFINED() for
 * "image_over" returns "False", then the routine returns immediately without
 * action.
 *
 * Input : src_im:     buffer with the input image (float)
 *         lut_d:      structure with the look-up-table for the distortion
 *                     corrections
 *         image_over: value used to mark illegal pixels in the input image
 * Output: trg_im: buffer for the distortion corrected output image with all
 *                 pixels marked that correspond to illegal pixels in the
 *                 input image (float)
 * Return: none
 */

int mark_overflow(float *src_im,float *trg_im,struct lut_descript * lut_d,
  float image_over)
{
  float savepix;
  float *lastpix = src_im + XSIZE * YSIZE - 1;
  register float *srcptr;
  register float srcpix;
  int i,j,maxxpixel,maxypixel,sx,sy,x,y,xmin,xmax,ymin,ymax;
  short *xrel,*yrel;
  int idx;
  float dsx,dsy,dx,dy;
  float fx,fy;
  float image_over_set;
  float dimage_over;
  double active2 = ACTIVE_R * ACTIVE_R;
  double d;

  prmsg(DMSG,("Looking for illegal pixels %f\n",image_over));
  
  dimage_over = DDSET(image_over);
  if(!DUMMYDEFINED(image_over,dimage_over))
    return;

  savepix = *lastpix;
  *lastpix = image_over;
  srcptr = src_im;
  maxxpixel = (lut_d->maxxpixel > 3) ? 2 : ((lut_d->maxxpixel > 1) ? 1 : 0);
  maxypixel = (lut_d->maxypixel > 3) ? 2 : ((lut_d->maxypixel > 1) ? 1 : 0);
  xrel = lut_d->xrel;
  yrel = lut_d->yrel;
  image_over_set = Dummy;

  /*
   * Endless loop over input pixels.
   */
  for(;;) {
    srcpix = *srcptr;
    if(DUMMY(srcpix,image_over,dimage_over))
      /*
       * Terminate loop over input pixels.
       */
      if(srcptr == lastpix)
	break;
      else {
        /*
         * Calculate x and y coordinates (in input image space) of the pixel.
         */
	idx = srcptr - src_im;
	sx = idx % XSIZE; sy = idx / XSIZE;
        /*
         * Obtain the corresponding corrected x and y coordinates (in output
         * image space).
         */
#if WASTE4_FORSPEED
	x = sx + xrel[idx]; y = sy + yrel[idx];
#else /* WASTE4_FORSPEED */
#if ANDY_CORR
	/* We go to a lower level to make things faster ATTENTION */
	dsx = sx; dsy = sy;

 	/*
         * Don't do anything with illegal pixels outside the active area.
         */
	if(ACTIVE_R) {
	  d = (dsx - XSIZE/2) * (dsx - XSIZE/2) +
	    (dsy - YSIZE/2) * (dsy - YSIZE/2);
	  prmsg(DMSG,("Pixel distance test: Is %f (%f,%f) > %f \n",
	    d,dsx,dsy,active2));
	  if(d > active2) {
	    srcptr++;
	    continue;
	  }
	}

	pxcorrgrid(spline,1,1,&dsx,&dsy,&dx,&dy);
	x = dsx + dx + .5; y = dsy + dy + .5;
#else /* ANDY_CORR */
	my_func(sx,sy,&fx,&fy);
	x = fx + .5; y = fy + .5;
#endif /* ANDY_CORR */
#endif /* WASTE4_FORSPEED */
/*
	prmsg(DMSG,("Invalid pixel: [%d][%d] (%.1f) -> [%d][%d] (%.1f)\n",
	  sx,sy,image_over,x,y,image_over_set));
*/
        /*
         * One pixel in input image space can correspond to more than one pixel
         * in the corrected image space. Mark all corresponding pixels in the
         * corrected image space as "illegal".
         */
	xmin = x - maxxpixel; if(xmin < 0) xmin = 0;
	xmax = x + maxxpixel; if(xmax >= XSIZE) xmax = XSIZE - 1;
	ymin = y - maxypixel; if(ymin < 0) ymin = 0;
	ymax = y + maxypixel; if(ymax >= YSIZE) ymax = YSIZE - 1;
	for(i = xmin; i <= xmax; i++)
	  for(j = ymin * XSIZE; j <= ymax * XSIZE; j += XSIZE)
	    trg_im[i + j] = image_over_set;
      }
    srcptr++;
  }

  /*
   * This is just a repetition of the above code for the last pixel. Treating
   * this as a special case allows to make the above loop faster by having less
   * comparisons to make.
   */
  *lastpix = savepix;

  if(DUMMY(*lastpix,image_over,dimage_over)) {
    /* Same as above */
    idx = srcptr - src_im;
    sx = XSIZE - 1; sy = YSIZE - 1;
#if WASTE4_FORSPEED
    x = sx + xrel[idx]; y = sy + yrel[idx];
#else /* WASTE4_FORSPEED */
#if ANDY_CORR
    dsx = sx; dsy = sy;

    /* Don't do anything with illegal pixels outside the active area */
    if(ACTIVE_R) {
      d = (dsx - XSIZE/2) * (dsx - XSIZE/2) +
          (dsy - YSIZE/2) * (dsy - YSIZE/2);
      prmsg(DMSG,("Pixel distance test : Is %f (%f,%f) > %f \n",
	d,dsx,dsy,active2));
      if(d > active2) {
        return;
      }
    }

    pxcorrgrid(spline,1,1,&dsx,&dsy,&dx,&dy);
    x = dsx + dx + .5; y = dsy + dy + .5;
#else /* ANDY_CORR */
    my_func(sx,sy,&fx,&fy);
    x = fx + .5; y = fy + .5;
#endif /* ANDY_CORR */
#endif /* WASTE4_FORSPEED */
/*
    prmsg(DMSG,("Invalid pixel: [%d][%d] (%.1f) -> [%d][%d] (%.1f)\n",
      sx,sy,image_over,x,y,image_over_set));
*/
    xmin = x - maxxpixel; if (xmin < 0) xmin = 0;
    xmax = XSIZE - 1;
    ymin = y - maxypixel; if (ymin < 0) ymin = 0;
    ymax = YSIZE - 1;
    for(i = xmin; i <= xmax; i++)
      for(j = ymin * XSIZE; j <= ymax * XSIZE; j += XSIZE)
	trg_im[i + j] = image_over_set;
  }
}

/*
 * Scans the input image for pixels with illegal values and marks the
 * corresponding pixels in the output image.
 *
 * Illegal values are:
 * - "IMAGE_OVER" (pixels marked as overflow);
 * - "dummy" (pixels marked as dummy);
 * - all values smaller than "INP_MIN";
 * - all values greater thatn "INP_MAX".
 *
 * "IMAGE_OVER" is the value of the command line argument "overflow". The
 * default is 0xffff.
 *
 * "dummy" is the value of the "Dummy" keyword in the source image header, if
 * this is defined, otherwise it is 0.
 *
 * "INP_MIN" is the value of the command line argument "inp_min". The default
 * value is 0., meaning that there is no "INP_MIN" value set.
 *
 * "INP_MAX" is the value of the command line argument "inp_max". The default
 * value is 0., meaning that there is no "INP_MAX" value set.
 *
 * The value used for marking these pixels is "Dummy". This value can be defined
 * with the command line argument "dummy", otherwise the value of the "Dummy"
 * keyword in the source image header is taken. The default (if neither is
 * specified) is 0.
 *
 * Input : src_im: buffer with the input image (float)
 * Output: trg_im: output image buffer with overflow pixels marked (float)
 * Return: none
 */

int mark_overflow_nocorr(float *src_im,float *trg_im)
{
  float savepix;
  float *lastpix = src_im + XSIZE * YSIZE - 1;
  register float *srcptr;
  register float srcpix;
  register int idx;
  float image_over = IMAGE_OVER;
  float image_over_set,dimage_over,dummy,ddummy;

  dimage_over = DDSET(image_over);
  image_over_set = Dummy;

  /*
   * Write the (possibly new) "Dummy" value into the header for the corrected
   * output image.
   */
  img_head[CORTYP].Dummy = Dummy;
  img_head[CORTYP].init |= FL_DUMMY;

  /*
   * Define the "dummy" and "ddummy" values.
   */
  if(img_head[SRCTYP].init & FL_DUMMY)
    dummy = img_head[SRCTYP].Dummy;
  else
    dummy = 0.;
  if(img_head[SRCTYP].init & FL_DDUMM)
    ddummy = img_head[SRCTYP].DDummy;
  else
    ddummy = DDSET(dummy);

  /*
   * Save the value of the last pixel and set it to "IMAGE_OVER". 
   *
   * Then test the image for the illegal values "IMAGE_OVER" or "dummy".
   * As the last pixel has been set to "IMAGE_OVER", the loop will terminate
   * there. This eliminates the need to test for the end of the image at each
   * iteration of the loop.
   */
  savepix = *lastpix;
  *lastpix = image_over;
  srcptr = src_im;

  for(;;) {
    srcpix = *srcptr;
    if(DUMMY(srcpix,image_over,dimage_over) || DUMMY(srcpix,dummy,ddummy))
      if(srcptr == lastpix)
	break;
      else {
	idx = srcptr - src_im;
	trg_im[idx] = image_over_set;
      }
    srcptr++;
  }

  /*
   * If "INP_MIN" is set, test the image for pixels with a smaller value.
   *
   * For terminating the loop the same trick as above is used.
   */
  if(INP_MIN != 0.) {
    *lastpix = INP_MIN - 1000.;
    srcptr = src_im;
  
    for(;;) {
      if(*srcptr < INP_MIN)
        if(srcptr == lastpix)
	  break;
        else {
	  idx = srcptr - src_im;
	  trg_im[idx] = image_over_set;
        }
      srcptr++;
    }
  }

  /*
   * If "INP_MAX" is set, test the image for pixels with a greater value.
   *
   * For terminating the loop the same trick as above is used.
   */
  if(INP_MAX != 0.) {
    *lastpix = INP_MAX + 1000.;
    srcptr = src_im;
  
    for(;;) {
      if(*srcptr > INP_MAX)
        if(srcptr == lastpix)
	  break;
        else {
	  idx = srcptr - src_im;
	  trg_im[idx] = image_over_set;
        }
      srcptr++;
    }
  }

  /*
   * Restore the value of the last pixel and test it.
   */
  *lastpix = savepix;

  if(DUMMY(*lastpix,image_over,dimage_over) || DUMMY(*lastpix,dummy,ddummy) ||
    (INP_MIN != 0. && *lastpix < INP_MIN) ||
    (INP_MAX != 0. && *lastpix > INP_MAX)) {
    idx = srcptr - src_im;
    trg_im[idx] = image_over_set;
  }
}

/*
 * Integrate the input image azimuthally.
 *
 * The values for this calculation are taken from the data header parameters'
 * structure "img_head" for the corrected image (type == CORTYP). If this
 * structure does not contain all needed values, the routine returns with an 
 * error.
 *
 * Input : I1Data: buffer with the input image (float)
 *         r0    : minimum radius for azimuthal integration (in meter)
 *         re    : maximum radius for azimuthal integration (in meter)
 *                 (-1. means no maximum radius defined)
 *         a0    : start angle for azimuthal integration (in degrees)
 *         ae    : end angle for azimuthal integration (in degrees)
 *         da    : angular interval for azimuthal integration (in degrees)
 * Output: I0Data: buffer with corrected input image
 * Return:  0  if no errors
 *         -1  if header parameter structure does not contain all needed values
 *         -2  if illegal parameters for inegration are given in the input
 */

int azim_int(float *I1Data,float *I0Data,float r0,float re,float a0,float ae,
  float da)
{
#define CALC_DIST(A) sqrt((A[0])*(A[0])+(A[1])*(A[1]))
  const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;

  int   I0Dim_1,I0Dim_2,I1Dim_1,I1Dim_2;
  float *pI0Data;
  float I0Offset_1,I0Offset_2,I1Offset_1,I1Offset_2;
  float I0PSize_1,I0PSize_2,I1PSize_1,I1PSize_2;
  float I0Center_1,I0Center_2,I1Center_1,I1Center_2;
  float I0Dummy,I0DDummy,I1Dummy,I1DDummy;

  int NAngle,cnt,iangle;
  float Value,Weight,Sum,SumWeight;
  float Wcenter_1,Wcenter_2;
  float Radius,Angle,DAngle,DDAngle,Angle0;

  float dr;

  int i_1,i_2;
  float W_1,W_2;
  float f_11,f_21;
  float Off_1[2],Off_2[2];
  float Ps_1[2],Ps_2[2];

  float i_10,i_11,i_20,i_21,edge1[2],edge2[2],edge3[2],edge4[2];
  float maxrad,minang1,maxang1,minang2,maxang2;

  unsigned long reqflg;
  float ddummy;

  reqflg = FL_PSIZ1 | FL_PSIZ2 | FL_OFFS1 | FL_OFFS2 | FL_CENT1 | FL_CENT2;
  if((img_head[CORTYP].init & reqflg) != reqflg) {
    prmsg(MSG,
      ("Needed header parameters not set - cannot integrate over azimuth\n"));
    return(-1);
  }

  if(r0 < 0. || (re < 0. || r0 >= re) && re != -1. || a0 >= ae || da <= 0.) {
    prmsg(MSG,("Illegal integration parameters\n"));
    return(-2);
  }

   /* 1 input, 0 output */
  I1Dim_1 = XSIZE;
  I1Dim_2 = YSIZE;
  I1Offset_1 = img_head[CORTYP].Offset_1;
  I1Offset_2 = img_head[CORTYP].Offset_2;
  I1PSize_1 = img_head[CORTYP].PSize_1;
  I1PSize_2 = img_head[CORTYP].PSize_2;
  I1Center_1 = img_head[CORTYP].Center_1;
  I1Center_2 = img_head[CORTYP].Center_2;
  I1Dummy = img_head[CORTYP].Dummy;
  I1DDummy = img_head[CORTYP].DDummy;

  dr = PSIZE2N(MIN2(fabs(I1PSize_1),fabs(I1PSize_2)));
  I0Dim_1 = XSIZE;
  I0Dim_2 = YSIZE;
  I0PSize_1 = R2PSIZE(dr);
  I0PSize_2 = R2PSIZE(da);
  I0Offset_1 = R2OFFSET(r0,I0PSize_1);
  I0Offset_2 = 0.0;
//  I0Offset_2 = R2OFFSET(a0,I0PSize_2); +++++++++++
  I0Center_1 = 0.;
  I0Center_2 = 0.;
  I0Dummy = I1Dummy;
  I0DDummy = DDSET(I0Dummy);

  NORMALREF(Off_1[0],Ps_1[0],I0Offset_1,I0PSize_1,I0Center_1);
  NORMALREF(Off_2[0],Ps_2[0],I0Offset_2,I0PSize_2,I0Center_2);
  NORMALREF(Off_1[1],Ps_1[1],I1Offset_1,I1PSize_1,I1Center_1);
  NORMALREF(Off_2[1],Ps_2[1],I1Offset_2,I1PSize_2,I1Center_2);

/*
printf("azim_int: Off_1 = %f %f, Off_2 = %f %f\n",Off_1[0],Off_1[1],Off_2[0],Off_2[1]);
printf("azim_int: Ps_1 = %f %f, Ps_2 = %f %f\n",Ps_1[0],Ps_1[1],Ps_2[0],Ps_2[1]);
*/

  /* calculate actual center */
  Wcenter_1 = WORLD(I2INDEX(I1Center_1,I1Offset_1),Off_1[1],Ps_1[1]); 
  Wcenter_2 = WORLD(I2INDEX(I1Center_2,I1Offset_2),Off_2[1],Ps_2[1]);
  
  DDAngle = (Ps_2[0]/(MIN2(Ps_1[1],Ps_2[1]))) * deg2rad;

  /*
   * Calculate the maximum radius value for the input image.
   */
  i_10 = i_20 = A2INDEX(ARRAYSTART+LOWERBORDER);
  i_11 = A2INDEX(ARRAYSTART+LOWERBORDER+I1Dim_1);
  i_21 = A2INDEX(ARRAYSTART+LOWERBORDER+I1Dim_2);

  edge1[0] = WORLD(i_10,Off_1[1],Ps_1[1]); 
  edge1[1] = WORLD(i_20,Off_2[1],Ps_2[1]);

  edge2[0] = WORLD(i_11,Off_1[1],Ps_1[1]);
  edge2[1] = edge1[1];

  edge3[0] = edge2[0];
  edge3[1] = WORLD(i_21,Off_2[1],Ps_2[1]);

  edge4[0] = edge1[0];
  edge4[1] = edge3[1];

  maxrad = MAX4(CALC_DIST(edge1),CALC_DIST(edge2),CALC_DIST(edge3),
    CALC_DIST(edge4));
printf("maxrad = %f\n",maxrad);
  if(re != -1. && re < maxrad)
    maxrad = re;

    minang1 = fmod((double)a0,360.);
    maxang1 = fmod((double)ae,360.);
    if(maxang1 == 0. && ae != 0.)
      maxang1 = 360.;
    minang2 = 0.;
    maxang2 = 360.;
    if(minang1 < 0. && maxang1 < 0.) {
      minang1 += 360.;
      maxang1 += 360.;
    } else if(minang1 < 0.) {
      minang2 = 360. + minang1;
      minang1 = 0.;
    }
    if(minang1 >= maxang1) {
      minang2 = minang1;
      minang1 = 0.;
    }
printf("minang1 = %f, maxang1 = %f, minang2 = %f, maxang2 = %f\n",minang1,
maxang1,minang2,maxang2);
    minang1 *= deg2rad;
    maxang1 *= deg2rad;
    minang2 *= deg2rad;
    maxang2 *= deg2rad;

  /*
   * Loop over the output array.
   *
   * The outer loop goes over the radius, the inner one over the angle.
   * Radial or angular values outside the requested range are skipped.
   */
  for(i_1=0; i_1 < I0Dim_1 * I0Dim_2; i_1++)
    I0Data[i_1] = I0Dummy;

  for(i_1=0; i_1<I0Dim_1; i_1++) {
    Radius = WORLD(i_1,Off_1[0],Ps_1[0]);
    if(Radius > maxrad || Radius < r0)
      continue;
    /* number of angular intervals for averaging */
    NAngle = MAX2(1,(int)(DDAngle * Radius) + 1); 
    DAngle = (Ps_2[0]/(float)NAngle) * deg2rad;

/*        printf("NAngle = % d\n",NAngle); /* ++++++++++ */
/*        printf("DAngle = % g\n",DAngle*rad2deg);  /* ++++++++++ */

    for(i_2=0; i_2<I0Dim_2; i_2++) {
      Angle = WORLD((LOWERBORDER + (float)i_2),Off_2[0],Ps_2[0]) * deg2rad + 
        DAngle * 0.5;
      if(Angle < minang1 || Angle > maxang2)
        continue;
      if(Angle > maxang1 && Angle < minang2)
        continue;
     
      /* angular averaging */ 
      cnt = 0;
      Sum = 0.0;
      SumWeight = 0.0;
      for(iangle = 0; iangle<NAngle; iangle++) {

/*    printf("iangle = %d, Angle = % g\n",iangle,Angle*rad2deg); /*+++++++++++*/

        W_1 = Radius * cos(Angle) + Wcenter_1;  
        W_2 = Radius * sin(Angle) + Wcenter_2;

        f_11 = INDEX(W_1,Off_1[1],Ps_1[1]);
        f_21 = INDEX(W_2,Off_2[1],Ps_2[1]);

        if(Ipol2ldw(I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,f_11,f_21,&Value,
          &Weight)) {
          /* then do something with the data */
          Sum += Value;
          SumWeight += Weight;
          cnt++;
        } /* if (Ipol2ld ... */

        Angle += DAngle;
      } /* for */

      if(cnt > 0) {
        pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);
        *pI0Data = Sum / SumWeight;

#if defined(ERROR_CALC)
        if(pcb->Flg[NVsum].V) {
          /* Multiply with number of covered pixels */
          *pI0Data *= (Radius*Ps_1[0]*SumWeight*DAngle) / (Ps_1[1]*Ps_2[1]);
          }
#endif /* defined(ERROR_CALC) */
      }

      /* end angular averaging */

    } /* for i_2 ... */

  } /* for i_1 ... */
  return(0);
}

/*
 * Performs the distortion correction on the input image.
 *
 * Determines for each pixel in the output image which pixels in the input
 * image contribute to it, and which fraction of the area of each of those
 * input pixels is mapped on the output pixel being processed. It then uses
 * these fractions to calculate the pixel value of the output pixel: for each
 * contributing input pixel, it multiplies the pixel's value with the
 * corresponding fraction, and sums these values over all contributing input
 * pixels.
 *
 * Input : cor_im: (empty) buffer for the corrected input pixel values
 *         src_im: buffer with the input image
 *         lut_d:  structure with the look-up-table for the distortion
 *                 corrections
 * Output: cor_im: buffer with the corrected pixel values for the input image
 * Return: none
 */

corr_calc(float *cor_im,float *src_im,struct lut_descript *lut_d)
{
  /*
   * starttidx  contains the number of empty pixels at the beginning of the
   *            target image, i.e. it is the index of the first non-empty pixel
   *            in the target image array.
   * startsidx  is the index of the corresponding pixel in the source image
   *            array
   */
  unsigned long startsidx = lut_d->startsidx;
  unsigned long starttidx = lut_d->starttidx;
  unsigned char *prog = lut_d->prog;
  LUT_TYPE *lut = lut_d->lut;
  unsigned short *asrc = lut_d->abs_src;
  int prog_length = lut_d->prog_length;
  int offset;
  unsigned char *prog_ptr = prog;
  unsigned char *prog_end = prog + prog_length;
  LUT_TYPE *lut_ptr = lut;
  int x, y,j;
  int * rel_ptr;
  int len, abs, old;
  int multi;

  int i;
  register int *rel_tab, **relend_tab;
  register int *offset_tab;
  register int *r_ptr;
  register int *relend;
  register float *src_p;
  register float *cor_p;
  unsigned short *abs_ptr = asrc;
  register float add = 0.;
  register float fullscale = FULLSCALE;
  register float dummy = Dummy;
  register unsigned char instruct;
  offset_tab = lut_d->offset_tab;
  rel_tab = lut_d->rel_tab;
  relend_tab = lut_d->relend_tab;

  /*
   * Determine the starting point for processing in the source and target
   * images, skipping over all empty pixels at the beginning of the images.
   *
   * Set the value of these empty pixels in the target image to Dummy.
   */
  src_p = src_im + startsidx;

  cor_p = cor_im;
  for(i=0; i<starttidx; i++)
    *cor_p++ = dummy;

  /*
   * Endless loop through the instructions of the look-up-table program.
   * The loop is terminated by a special instruction PROGEND.
   */
  for(;;) {
    /*
     * Get the next instruction.
     *
     * If it is != 0, then we have to process the "normal" case: a predefined
     * sequence of source pixels, starting with a pixel that is "near" the
     * source pixel that was processed last (i.e., it is in one of the 256
     * positions around the last one).
     *
     * If it is 0, then we have to treat an exception - see below.
     */
    instruct = *prog_ptr++;
    if(instruct != 0) {
      /*
       * Use the offset table to get the address of the first source image
       * pixel to be processed, and calculate that part of the pixel value
       * that goes into the corresponding target pixel. This percentage is
       * contained in the array lut_ptr.
       */
      src_p += offset_tab[instruct];
      add = (*lut_ptr++ * *src_p);

      /*
       * There is usually more than one source image pixel that will contribute
       * to the target pixel being processed. Loop over all these source image
       * pixels, get their addresses (always relative to the previous pixel -
       * see description of rel_tab in lut_calc()) and add their contribution
       * to the value already calculated for the first source pixel above.
       *
       * This loop ends when the sequence end pointer (from relend_tab) has
       * been reached.
       */
      instruct = *prog_ptr++;

      relend = relend_tab[instruct];
      for(r_ptr = rel_tab + (instruct << 5); r_ptr < relend; r_ptr++) {
	src_p += *r_ptr;
	add += (*lut_ptr++ * *src_p);
      }
    } else {
      /*
       * The program instruction is 0: we have to treat an exception. Get the
       * next instruction and see what kind of exception it is.
       */
      instruct = *prog_ptr++;
      if(instruct & INCTARGET) {
        /*
         * Exception INCTARGET: there is one or several (if the MULTIINC flag
         * is set as well) target pixels that will not receive a contribution
         * from any source pixel. Just set the target pixel values to Dummy.
         */
	if(instruct & MULTIINC) {
	  multi  = *prog_ptr++;
	  multi += *prog_ptr++ * 256;
	  for(i = multi; i; i--)
	    *cor_p++ = dummy;
	} else
	  *cor_p++ = dummy;
	continue;
      }
      if(instruct & ABSSRC) {
        /*
         * Exception ABSSRC: the next source pixel is not in one of the 256
         * positions around the present one and therefore its address cannot be
         * expressed as an offset relative to the present source pixel. Get the
         * absolute x and y coordinates of the pixel from the array abs_ptr,
         * obtain the absolute address using the coordinates, and then
         * calculate its contribution to the target pixel value.
         */
	int x = *abs_ptr++;
	int y = *abs_ptr++;
	src_p = src_im + (x + y * XSIZE);
	add = *abs_ptr++ * *src_p;

        /*
         * The following pixels can either be a predefined sequence of move-
         * ments as above, or it is a sequence where each subsequent source
         * pixel coordinate is obtained from its own look-up-table program
         * instruction (case UNCOMPRESSED).
         *
         * For more details on the first case, see above; for more details on
         * the second case, see below under "Exception UNCOMPRESSED".
         */
	if(instruct & UNCOMPRESSED) {
	  while(instruct = *prog_ptr++) {
	    src_p += offset_tab[instruct];
	    add += (*lut_ptr++ * *src_p);
	  }
	} else {
	  instruct = *prog_ptr++;
	  for(r_ptr = rel_tab + RELTABSIZE * instruct;
	       r_ptr < relend_tab[instruct]; r_ptr++) {
	    src_p += *r_ptr;
	    add += (*lut_ptr++ * *src_p);
	  }
	}
      } else if(instruct & UNCOMPRESSED) {
        /*
         * Exception UNCOMPRESSED: the source pixels contributing to this target
         * pixel could not be expressed as a predefined sequence of movements
         * relative to the first source pixel. Therefore for each source pixel
         * there is an instruction in the look-up-table program that gives the
         * corresponding index for the offset table, and this offset is used to
         * calculate the source pixel address. Then the contributions to the
         * target pixel are calculated and summed as before. The loop over the
         * contributing source pixels ends when there is a 0 instruction in the
         * look-up-table program.
         */
	add = 0.;
	while(instruct = *prog_ptr++) {
	  src_p += offset_tab[instruct];
	  add += (*lut_ptr++ * *src_p);
	}
      } else if(instruct & PROGEND)
        /*
         * Exception PROGEND: terminate the loop over the program instructions.
         */
	break;
    }

    /*
     * The variable "add" contains the target image values multiplied by
     * FULLSCALE, as this is the way percentages are expressed in "lut_d->lut"
     * (default type "unsigned short").
     *
     * Now they have to be divided by this value to yield the correct image
     * values.
     *
     * The default value is: FULLSCALE = 0x800.
     */

     *cor_p++ = add / fullscale;
  }

  /*
   * Put the saved x- and y-pixel sizes in the output image header structure.
   */
  img_head[CORTYP].PSize_1 = psizex;
  img_head[CORTYP].PSize_2 = psizey;
  img_head[CORTYP].init |= FL_PSIZ1 | FL_PSIZ2;
}

/*
 * This function is used as the "comparison function" for one of the qsort()
 * calls in lut_calc().
 *
 * It compares the second element in each of the two input fields and returns
 * -1, 0 or 1 if the one in the first field is greater than, equal to or
 * smaller than the one in the second field.
 *
 * This will cause the fields to be sorted in descending order.
 *
 * Input : el1, el2: pointers to the two fields to be compared
 * Output: none
 * Return: -1  if the first input field is greater than the second one
 *          0  if the first input field is equal to the second one
 *          1  if the first input field is smaller than the second one
 */

int histcompare_count(const void *el1,const void *el2)
{
  register unsigned int i1,i2;
  i1 = *((unsigned int*)el1 + 1);
  i2 = *((unsigned int*)el2 + 1);

  if(i1 < i2)
    return(1);
  else if(i1 != i2)
    return(-1);

  return(0);
}

/*
 * This function is used as the "comparison function" for one of the qsort()
 * calls in lut_calc().
 *
 * It compares the first element in each of the two input fields and returns
 * -1, 0 or 1 if the one in the first field is smaller than, equal to or
 * greater than the one in the second field.
 *
 * This will cause the fields to be sorted in ascending order.
 *
 * Input : el1, el2: pointers to the two fields to be compared
 * Output: none
 * Return: -1  if the first input field is than smaller the second one
 *          0  if the first input field is equal to the second one
 *          1  if the first input field is greater than the second one
 */

int histcompare_idx(const void *el1,const void *el2)
{
  register unsigned int i1,i2;
  i1 = *(unsigned int*)(el1);
  i2 = *(unsigned int*)(el2);

  if(i1 < i2)
    return(-1);
  else if(i1 != i2)
    return(1);

  return(0);
}

/*
 * Calculates the variables and tables that will be used for the distortion
 * correction.
 *
 * The image received with a two-dimensional detector is often distorted (e.g.
 * through the use of an image intensifier). The consequence of this distortion
 * is that lines in a grid that were straight and crossing at right angles are
 * no longer straight and cross at angles different from 90 degrees in the
 * image received from the detector. Also, not only the shape but also the size
 * of the grid areas will be modified by this distortion.
 *
 * Before analyzing image data from such a detector for the scientific
 * information that they contain, it is therefore necessary to correct for this
 * distortion. The purpose of this correction is to reconstruct the original
 * undistorted image (also called "corrected image" or "target image") from
 * the distorted one (also called "source image").
 *
 * Both the target and the source image have the same number of pixels (e.g.
 * 1024 * 1024 for a given CCD camera), but because of the distortion the area
 * of a given pixel in the target image will usually correspond to parts of the
 * areas of several pixels in the source image.
 *
 * Once it is established (for all target pixels) which source pixels
 * contribute with which fraction of their area to a given target pixel, the
 * pixel values of the target pixels can be calculated:
 *
 *    target(i) = "sum over j" ( fraction(i,j) * source(j) )
 *
 * where  target(i)      is the value of the target pixel i,
 *        source(j)      is the value of the source pixel j,
 *        fraction(i,j)  is the area fraction of source pixel j that
 *                       contributes to target pixel i
 *
 * In principle, this sum is over all pixels in the source image, but in
 * reality for a given target pixel most of the "fraction(i,j)" values will
 * be 0. It is therefore better to sum only over the source pixels that have
 * a fraction different from 0. These will be called "contributing source
 * pixels".
 *
 * To do the distortion correction, two pieces of information are therefore
 * needed for each target pixel:
 * - a list of the contributing source pixels;
 * - a list with the fractional area contributions of each of these source
 *   pixels to the target pixel being processed.
 *
 * This routine calculates those two lists:
 * - the "look-up-table program" (table "prog" in the structure "lut_descript")
 *   contains the list of the contributing source pixels for each target pixel;
 * - the "look-up table" (table "lut" in the structure "lut_descript") contains
 *   for each target pixel the fractional contributions of each contributing
 *   source pixel.
 *
 * In principle, this is fairly simple and straightforward. For technical
 * reasons, the setup of these two lists is more complicated, and there are
 * additional lists used.
 *
 * The structure "lut_descript" is given below:
 *
 *    struct lut_descript {
 *      LUT_TYPE *lut;
 *      unsigned char *prog;
 *      int prog_length;
 *      unsigned int starttidx;
 *      unsigned int startsidx;
 *      int *offset_tab;
 *      int *rel_tab;
 *      int **relend_tab;
 *      unsigned short *abs_src;
 *      float *temp_im;
 *      int maxxpixel;
 *      int maxypixel;
 *      short *xrel;
 *      short *yrel;
 *    }
 *
 * In order to make the program faster and also not to consume an excessive
 * amount of memory, the list of contributing source pixels in the
 * look-up-table program is given in several different ways depending on the
 * number and distribution of these source pixels:
 *
 * 1) the most general way to give the address of a source pixel is to specify
 *    its x and y coordinate. For this method of absolute addressing the
 *    additional source coordinate table "abs_src" in the structure
 *    "lut_descript" is used. For each source pixel contained in this table
 *    there are three entries at consecutive table addresses: the x coordinate
 *    (column number), the y coordinate (row number) and the area fraction that
 *    this source pixel contributes to the target pixel being processed.
 *
 *    From the row and column number, the pixel's index into the source image
 *    array can be calculated
 *
 *      index = "column number" + "row number" * "total number of columns"
 *
 * 2) this general addressing is, however, only needed at the beginning of a
 *    new target pixel (and even there not always). The distortion changes the
 *    shape and size of the pixels in the undistorted image, but it will not
 *    change the continuity of a pixel area. This means that all source pixels
 *    contributing to a given target pixel occupy a continuous area, i.e. every
 *    source pixel of this target pixel is adjacent to at least one other
 *    source pixel of the same target pixel. Once the first source pixel for
 *    a given target pixel has been found, the addresses of all subsequent
 *    source pixels can be given relative to the previous source pixel.
 *
 *    For this method of relative addressing the additional index offset table
 *    "offset_tab" in the structure "lut_descript" is used. It contains 256
 *    index offsets for the source image array relative to the index of the
 *    previously processed source pixel. The positions thus reached go from the
 *    "upper left" pixel at offset (-8 - 8 * XSIZE) to the "lower right" pixel
 *    at offset (7 + 7 * XSIZE).
 *
 *    In other words, offset_tab[0] points to the pixel that is 8 rows to the
 *    left and 8 lines above the previous pixel. The subsequent elements then
 *    move in this upper line to the right until offset_tab[15], which points
 *    to the pixel that is 7 rows to the right and (still) 8 lines above the
 *    previous pixel. offset_tab[16] then moves one line further down and back
 *    to the leftmost row: 8 rows to the left and 7 lines above the previous
 *    pixel. This continues like that, and offset_tab[255] finally points to
 *    the pixel that is 7 rows to the right and 7 rows below the previous
 *    pixel.
 *
 *    This relative addressing can also be used to specify the first source
 *    pixel of a new target pixel: if the index offset of this first source
 *    pixel relative to the index of the last source pixel of the previous
 *    target pixel fits the -8 to +7 condition for x and y described above,
 *    then the index of this new source pixel is specified this way with the
 *    offset table instead of using the source coordinate table.
 *
 * 3) the principle of relative addressing can be pushed even further. The
 *    addressing scheme as described above describes every source pixel
 *    individually by specifying its distance to the previous source pixel.
 *    This allows for a great variety in the pattern of the source pixels
 *    that belong to a given target pixel.
 *
 *    However, in many cases the image will not be strongly distorted. This
 *    means that the positions of the source pixels that contribute to a given
 *    target pixel will be near that pixel's position. Also, the correction
 *    function that maps the source pixel coordinates to the target pixel
 *    coordinates is not likely to vary a lot from one pixel to the next.
 *
 *    Therefore, the group of source pixels that contribute to a given target
 *    pixel is likely to have a similar shape as one moves from one target
 *    pixel to the next; or, expressed in a different way, the sequence of
 *    source pixel movements is likely to show a similar pattern.
 *
 *    This can be used to compact ("compress") the relative addressing scheme.
 *    The idea is to define a certain number of possible movement sequences and
 *    then look at all target pixels to find out whether the corresponding
 *    source pixel combination is described by one of these defined sequences.
 *    If this is the case, the individual description of the distance between a
 *    given source pixel and the previous one can be replaced by a reference to
 *    the corresponding movement sequence.
 *
 *    For this method of addressing the additional compressed addressing tables
 *    "rel_tab" and "relend_tab" in the structure "lut_descript" are used.
 *    "rel_tab" contains (at most) 256 predefined source pixel movement
 *    sequences of at most 18 pixels each. These 18 pixels are mainly "below
 *    and to the right" of the present pixel.
 *
 *    More precisely, if the present pixel is marked by "x", then all the
 *    pixels marked with a "." in the following drawing can be reached:
 *
 *                  x . . .           (the three pixels to the right)
 *                . . . . .           (line below: 1 left to 3 right)
 *                . . . . .           (two lines below: 1 left to 3 right)
 *                . . . . .           (three lines below: 1 left to 3 right)
 *
 *    The element 0 of each sequence contains the position of the next pixel
 *    relative to the present pixel. All subsequent elements of the sequence
 *    then contain the position of the next pixel relative to the previous one.
 *
 *    relend_tab contains the actual length for each sequence stored in rel_tab
 *    in the following way:
 *
 *      for the sequence "i" starting in rel_tab[RELTABSIZE * i]
 *
 *      relend_tab[i] = pointer to the end location of that sequence in rel_tab
 *
 * 4) it is also possible that no source pixels contribute to the given target
 *    pixel. Then obviously no source pixel indices and fractions need be
 *    given, it is only necessary to indicate the existence of such "empty"
 *    target pixels.
 *
 *    A special case are empty target pixels at the beginning of the image.
 *    They can easily be skipped over by starting the image processing only at
 *    the first non-empty target pixel. For this purpose, the member
 *    "starttidx" of the structure "lut_descript" contains the index of the
 *    first non-empty target pixel, and the member "startsidx" contains the
 *    index of the first source pixel for this target pixel.
 *
 * The use of these tables in the calculation of the distortion correction is
 * as follows:
 *
 * 1) processing starts at the target pixel identified by "starttidx", and the
 *    first of the corresponding source pixels is in "startsidx". The pointer
 *    into the look-up-table program "prog" is set to the beginning of "prog";
 *
 * 2) the instruction (i.e., the value contained in the program pointer) is
 *    examined:
 *
 * 2.1) if it is not 0, then the compressed source pixel addressing is used
 *      (case 3 above), and the instruction contains the relevant index into
 *      the compressed addressing tables "rel_tab" and "relend_tab".
 *
 *      A loop is performed over the source pixels given by "rel_tab" until
 *      the end location defined in "relend_tab" is reached. The look-up table
 *      "lut" contains the corresponding fractional contributions of the source
 *      pixels to the target pixel, i.e. there are for this target pixel as
 *      many values in the look-up table as there are source pixels that
 *      contribute to this target pixel.
 *
 *      The content of each source pixel is multiplied by the corresponding
 *      fractional contribution. The sum over all contributing source pixels
 *      gives the image value for the target pixel;
 *
 * 2.2) if it is 0, then an "exception" occurred, i.e. a different addressing
 *      scheme ist used. The type of the exception is contained in the next
 *      instruction of the look-up-table program. Possible exceptions are
 *      UNCOMPRESSED, ABSSRC, INCTARGET, MULTIINC and PROGEND and combinations
 *      of them. However, not all combinations are possible.
 *
 * 2.2.1) if the exception is just "UNCOMPRESSED", then the relative addressing
 *        scheme is used (case 2 above). The following lookup-table instruction
 *        starts a sequence of values, which is terminated by an instruction
 *        containing 0; the length of the sequence (without the terminating 0)
 *        is just the number of source pixels contributing to the target pixel.
 *
 *        These values are indices into the index offset table "offset_tab".
 *        The corresponding value in the offset table contains the index
 *        offset from the previous source pixel to the present (new) one; for
 *        the first source pixel of a particular target pixel the previous
 *        source pixel is the last source pixel of the previous target pixel.
 *
 *        The fractional contributions of the source pixels to the target pixel
 *        are in the look-up table "lut", and the value of the target pixel is
 *        calculated by a loop over the source pixels. This is the same
 *        procedure as described for the "compressed addressing scheme" above;
 *
 * 2.2.2) if the exception is just "ABSSRC", then the absolute addressing
 *        scheme is used (case 1 above). The x and y coordinates of the first
 *        source pixel contributing to this target pixel are stored in two
 *        consecutive locations in the source coordinate table "abs_src", and
 *        the fractional contribution of this source pixel to the target pixel
 *        is in the following (third) location of the source coordinate table.
 *        For each source pixel that is specified with this addressing scheme,
 *        there are therefore 3 values in the source coordinate table.
 *
 *        The following source pixels are specified by compressed addressing
 *        as described in 2.1 above: the next look-up-table program instruction
 *        contains the relevant index into the compressed addressing tables
 *        "rel_tab" and "relend_tab", and the fractional contributions of the
 *        source pixels to the target pixel are in the look-up table "lut".
 *
 *        Calculation of the target pixel value starts with multiplying the
 *        value of the first source pixel with its corresponding fractional
 *        contribution, and then adding the values of the other source pixels
 *        in a loop as described in step 2.1 above;
 *
 * 2.2.3) if the exception is "ABSSCR" combined with "UNCOMPRESSED", then the
 *        absolute addressing scheme is used for the first source pixel and
 *        the relative addressing scheme for the subsequent ones.
 *
 *        The values for the first source pixel are obtained as in 2.2.2 above,
 *        and the values for the subsequent ones as described in 2.2.1 above.
 *
 *        Calculation of the target pixel vlaue is as described in 2.2.2 above.
 *
 * 2.2.4) if the exception is just "INCTARGET" (note: this cannot happen for
 *        the first target pixel), then it is an empty target pixel as
 *        described in case 4 above.
 *
 *        The value of the target pixel is set to 0. No other action is taken;
 *
 * 2.2.5) if the exception is "INCTARGET" combined with "MULTIINC" (note: this
 *        cannot happen for the first target pixel), then it is sequence of
 *        empty target pixels. Empty target pixels are described in case 4
 *        above.
 *
 *        The two following look-up-table program instructions contain the LSB
 *        and the MSB of a 16-bit integer value that gives the number of empty
 *        target pixels in this sequence.
 *
 *        A loop is performed over all these target pixels, and the value of
 *        each of them is set to 0. No other action is taken;
 *
 * 2.2.6) if the exception is "PROGEND", then the target pixel processing is
 *        terminated;
 *
 * 3) during this processing the look-up-table program pointer is advanced
 *    every time an instruction from the program was used. With the value now
 *    reached, processing continues at step 2 above. This loop ends when the
 *    exception "PROGEND" is encountered (step 2.2.6).
 *
 * The calculation of these tables is done in several steps, and the tables are
 * modified several times:
 *
 * Step 1: determine for each target pixel the contributing source pixels and
 *         the fractional contributions of the source pixels to the target
 *         pixel. At the end of this step, the source coordinate table contains
 *         for all target pixels the first contributing source pixel, and the
 *         subsequent source pixels are given in the look-up-table program
 *         using the relative addressing scheme. The fractional contributions
 *         are stored in the look-up table. Each empty target pixel occupies
 *         one place in each of these tables. The beginning of a new target
 *         pixel is indicated by the flag BITMASK in the look-up table;
 *
 * Step 2: remove the empty target pixels at the beginning of the target image,
 *         and mark the other empty target pixels with the flag INCTARGET in
 *         the look-up table. The look-up table and the look-up-table-program
 *         are shortened according to the number of empty pixels removed;
 *
 * Step 3: optional: if the user-definable global variable "DO_FLAT" is set,
 *         the target image is normalized to a flat image. The structure of the
 *         tables is not changed;
 *
 * Step 4: the absolute addressing scheme for the first contributing source
 *         pixel is replaced by relative addressing whenever possible. Where it
 *         is not possible, the flag ABSSRC is set in the corresponding
 *         location in the look-up table.
 *
 *         At the end of this step, the source coordinate table has its final
 *         form as described above;
 *
 * Step 5: the relative addressing is replaced by the compressed addressing
 *         scheme whenever possible.
 *
 *         At the end of this step, all tables have their final form as
 *         described above.
 *
 * Input : none
 * Output: none
 * Return: structure with the variables and tables needed for the distortion
 *         correction
 */

struct lut_descript *lut_calc()
{
  unsigned char *prog_end;
  unsigned short *cor_im;
  double res[2];
  float parts[MAX_PARTS];
  float fx[4],fy[4];
  int xmin,xmax,ymin,ymax,idx,pidx,sidx;
  int i,j,ix,iy;
  int max_xsize = 0, max_ysize = 0;
  float sum_parts, min_parts = 1000.0, min0_parts = 1000.0, max_parts = 0.0;
  float min_xdist = 1E6, max_xdist = 0;
  float min_ydist = 1E6, max_ydist = 0;
  int count_parts = 0, no_parts;
  float total;
  LUT_TYPE *lut_tab,*lut_ptr;
  LUT_TYPE *max_pixellut;
  LUT_TYPE *new_lut_ptr;
  unsigned char *lut_prog,*prog_ptr;
  unsigned char *new_prog_ptr;
  int old_i,old_j,prog_length;
  unsigned char instruct;
  unsigned int part,psum;
  unsigned int max_pixelpart;
  float *x_trg,*y_trg;
  int *offsets,*offset_p;
  float fxmin,fxmax,fymin,fymax;
  int src_x,src_y,old_instruct;
  int flush_interval,flush_cnt;
  int tidx;
  int offset;
  int lastoffset;
  unsigned short *abs_ptr,*abs_src,*new_abssrc;
  int first;
  int new_x,new_y,diff_x,diff_y;
  int cnt = 0;
  int start_ff,start_tidx;
  int count_abs = 0, count_inc=0;
  int count_trg = 0, count_ff =0;
  int lutsize;
  unsigned char *new_inc;
  struct lut_descript *lut_d;
  int max_ocount,max_oidx;
  int noabs;
  float x,y;
  int stat_case[20];
  int count_deleted=0;

  /*
   *
   * Step 1: determine for each target pixel the contributing source pixels and
   *         the fractional contributions of the source pixels to the target
   *         pixel. Set up the look-up table, the look-up-table program and the
   *         source coordinate table.
   *
   * Allocate memory for the look-up table structure and initialize it to 0.
   */
  if((lut_d = (struct lut_descript *)pmalloc(sizeof(struct lut_descript)))
    == NULL)
    prmsg(FATAL,("No memory \n"));
  memset(lut_d,0,sizeof(struct lut_descript));

#if !LOW_MEM
  /*
   * Allocate two arrays, each with as many elements as there are pixels in the
   * image.
   *
   * There is one pixel more in each direction as for each pixel the four
   * corner points are calculated. For the last pixel in a line that leads us
   * to calculate the distortion for 1024. See loop below MARKER IX+1IY+1.
??? what does the second sentence above mean? in particular, why 1024 ???
   *
   * Then fill them with the corrected x and the corrected y coordinate values
   * for each pixel of the image.
   */
  if((x_trg = (float *)pmalloc((XSIZE + 1) * (YSIZE + 1) * sizeof(float)))
    == NULL ||
    (y_trg = (float *)pmalloc((XSIZE + 1) * (YSIZE + 1) * sizeof(float)))
    == NULL) {
    prmsg(FATAL,("No memory in lut_calc for x_trg and y_trg\n"));
  }

  for(iy = 0; iy < YSIZE + 1; iy++) {
    for(ix = 0; ix < XSIZE + 1; ix++) {
      idx = ix + iy * (XSIZE + 1);
      my_func(ix,iy,x_trg + idx,y_trg + idx);
    }
  }

  bench("X Y array calc");
  prmsg(MSG,("X and Y target arrays calculated\n"));

#if ANDY_CORR
  /*
   * Free andy's buffers
   */
  andy_free_buffers(X_COR,Y_COR);
  X_COR = Y_COR = NULL;
#endif /* ANDY_CORR */
#endif /* !LOW_MEM */

  /*
   * Produce a table that specifies for each target pixel how many source
   * pixels contribute to it.
   */
  if((offsets = (int *)pmalloc(XSIZE * YSIZE * sizeof(int))) == NULL) {
    prmsg(FATAL,("No memory in lut_calc for offsets\n"));
  }

  //  print_memsize();
  memset(offsets,0,XSIZE * YSIZE * sizeof (int));

  for(iy = 0; iy < YSIZE; iy++) {
    for(ix = 0; ix < XSIZE; ix++) {
      idx = ix + iy * (XSIZE + 1);
      /*
       * Get the corrected (x,y) coordinate values for the corners of the
       * square formed by the present point and the three neighbouring ones.
       *
       * The present point has index 0, the others are arranged as follows:
       *
       *                         0   1   |---> x
       *                                 |
       *                         3   2   V y

       * MARKER: IX+1IY+1
       */
#if LOW_MEM
      my_func(ix,iy,fx,fy);
      my_func(ix+1,iy,fx+1,fy+1);
      my_func(ix+1,iy+1,fx+2,fy+2);
      my_func(ix,iy+1,fx+3,fy+3);
#else
      idx = ix + iy * (XSIZE+1);
      fx[0] = x_trg[idx]; fy[0] = y_trg[idx];
      fx[1] = x_trg[idx + 1]; fy[1] = y_trg[idx+1];
      fx[2] = x_trg[idx + XSIZE+1 + 1]; fy[2] = y_trg[idx + XSIZE+1 + 1];
      fx[3] = x_trg[idx + XSIZE+1]; fy[3] = y_trg[idx + XSIZE+1];
#endif /* LOW_MEM */
      /*
       * Determine the size of the square in corrected coordinates, and also
       * the maximum size of all the pixel squares in the image.
       */
      minmax4(fx[0],fx[1],fx[2],fx[3],&fxmin,&fxmax);
      minmax4(fy[0],fy[1],fy[2],fy[3],&fymin,&fymax);

      if(fxmin < 0 || fymin <0 || fxmax >= XSIZE  || fymax >= YSIZE)
	continue;

      xmin = ceil(fxmin) - 1;
      xmax = floor(fxmax);
      ymin = ceil(fymin) - 1;
      ymax = floor(fymax);

      if(xmax - xmin > MAX_PIXELSIZE || ymax - ymin > MAX_PIXELSIZE) {
	continue;
      }

      if(max_xdist < fabs(fx[0] - ix))
	max_xdist = fabs(fx[0] - ix);
      if(max_ydist < fabs(fy[0] - iy))
	max_ydist = fabs(fy[0] - iy);
      if(min_xdist > fabs(fx[0] - ix))
	min_xdist = fabs(fx[0] - ix);
      if(min_ydist > fabs(fy[0] - iy))
	min_ydist = fabs(fy[0] - iy);

      if(max_ysize < ymax - ymin + 1)
	max_ysize = ymax - ymin + 1;
      if(max_xsize < xmax - xmin + 1)
	max_xsize = xmax - xmin + 1;

      /*
       * Counts for each target pixel how many source pixels are mapped into it.
       */
      for(j = ymin; j <= ymax; j++) {
	for(i = xmin; i <= xmax; i++) {
	  tidx = i + XSIZE * j;
	  offsets[tidx]++;
	}
      }
    }
  }

  /*
   * Prepare the look-up table "lut_tab", the look-up-table program "lut_prog"
   * and the source coordinate array "abs_src".
   *
   * The look-up table will contain for every source pixel the part (fraction)
   * of that pixel's area that is mapped into each of the target pixels.
   *
   * The look-up-table program will contain for every target pixel the
   * information which source pixels contributed to it.
   *
   * The source coordinate array will contain for every target pixel the (x,y)
   * coordinates of the first source pixel that contributes to it.
   *
   * In detail:
   *
   * look-up table "lut_tab":
   * - the look-up table contains for each target pixel a sequence of numbers.
   *   The length of each sequence is the number of source pixels that are
   *   mapped into this target pixel. If there are no source pixels mapped into
   *   a particular target pixel, then the length is set to 1;
   * - each number in a given sequence corresponds to one of the source pixels.
   *   It gives, for target pixel "i" and source pixel "j", the part (fraction)
   *   of the area of "j" that maps into "i";
   * - the beginning of each sequence is given by the indices in the offset
   *   table "offsets" in the following form:
   *
   *      lut_tab[offsets[tidx]] = first element of the sequence
   *                               for target pixel "tidx"
   *
   * - the values (fractions) stored in the look-up table are constrained to be
   *   less than FULLSCALE (normally 0x8000);
   * - in the beginning of each sequence, the value to be stored is "OR"ed with
   *   BITMASK (normally 0x8000).
   *
   * look-up-table program "lut_prog":
   * - the look-up-table program contains for each target pixel a sequence of
   *   instructions. In this sequence, there is one instruction for every
   *   source pixel that is mapped into this target pixel. If there are no
   *   source pixels mapped into a particular target pixel, then there is
   *   still one instruction;
   * - from the instructions, the coordinates of the source pixels can be
   *   determined in the following way:
   *
   *   -- the instructions are 8 bit long;
   *   -- the information concerning the y coordinate is in the 4 high bits,
   *      the one for the x coordinate in the 4 low bits;
   *   -- the first instruction of a sequence is always 0x88, the coordinates
   *      of the first source pixel are obtained from the source coordinate
   *      array;
   *   -- the (x,y) coordinates of all subsequent source pixels are stored in
   *      the cooresponding instructions as relative values to the last pixel.
   *      For pixel "i" of a sequence, the instruction value as a hexadecimal
   *      number "yx" is given by
   *            y = y(i-1) - y(i) + 8
   *            x = x(i-1) - x(i) + 8
   *  -- the maximum difference in x or y that is allowed by the program is
   *     +-7, i.e. the present pixel can be at most 7 pixels to the left or
   *     7 pixels to the right of the last one, and likewise 7 pixels above
   *     or 7 pixels below the present one. As an additional restriction, the
   *     difference in x and in y must not be +7 simultaneously (this is to
   *     prevent a valid instruction from having the initialization value
   *     0xff);
   *  -- if there is no source pixel mapped into a particular target pixel, the
   *     (one) corresponding program instruction has the initialization value
   *     0xff.
   *
   * - the beginning of each sequence is given by the indices in the offset
   *   table "offsets" in the following form:
   *
   *      lut_prog[offsets[tidx]] = first instruction of the sequence
   *                                for target pixel "tidx"
   *
   * source coordinate array "abs_src":
   * - the source coordinate array contains for each target pixel the x and
   *   the y coordinate of the first source pixel that contributes to it;
   * - if there is no source pixel that contributes to a given target pixel,
   *   then the coordinates are 0xffff (= initialization value);
   * - for target pixel "tidx", "abs_src[2 * tidx]" contains the x coordinate
   *   and "abs_src[2 * tidx + 1]" contains the y coordinate.
   *
   * First, fill the offset table.
   *
   * Also, count the total lenght of all sequences. This determines the length
   * of the look-up table.
   */
  max_ocount = 0;
  max_oidx = 0;
  for(count_parts = 0, offset_p = offsets, i = 0; i < XSIZE * YSIZE; i++) {
    if(*offset_p == 0)
      count_parts += 1;
    else
      count_parts += *offset_p;
    if(max_ocount < *offset_p) {
      max_ocount = *offset_p;
      max_oidx = i;
    }
    *offset_p++ = count_parts;
  }

  prmsg(MSG,("Max Distortion: x = %f y = %f\n",max_xdist,max_ydist));
  prmsg(MSG,("Min Distortion: x = %f y = %f\n",min_xdist,min_ydist));
  prmsg(MSG,("Max Pixel size: x = %d y = %d\n",max_xsize,max_ysize));
  prmsg(MSG,("Max SRC Pixel in Target : %d at [%d,%d]\n",
    max_ocount, max_oidx % XSIZE, max_oidx / XSIZE));
  prmsg(MSG,("Parts Count total: %d\n",count_parts));

  lut_d->maxxpixel = max_xsize;
  lut_d->maxypixel = max_ysize;

  /*
   * Allocate memory for the look-up table, and initialize the parts sequences.
   *
   * The initialization is as follows:
   * - the beginning of each sequence is set to BITMASK;
   * - all other values are set to 0.
   */
  prog_length = count_parts;
  lut_ptr = lut_tab = (LUT_TYPE *)pmalloc(sizeof(LUT_TYPE) * count_parts);

  if(lut_tab == NULL)  {
    prmsg(FATAL,("No memory for lut\n"));
  }

  memset(lut_tab,0,prog_length * sizeof (LUT_TYPE));
  lut_tab[0] = BITMASK;
  for(offset_p = offsets, i = 0; i< XSIZE * YSIZE -1; i++) {
    lut_tab[*offset_p++] = BITMASK;
  }

  /*
   * Allocate memory for the look-up-table program, and initialize all
   * instructions to 0xff.
   */
  if ((prog_ptr = lut_prog = (unsigned char *)
    pmalloc (sizeof(char) * prog_length)) == NULL) {
    prmsg(FATAL,("No memory for prog\n"));
  }

  memset(lut_prog,0xff,prog_length * sizeof(char));

  /*
   * Allocate memory for the source coordinate array, and initialize all
   * coordinates to 0xffff.
   */
  if((abs_src = (unsigned short *)
    pmalloc(2 * sizeof(short) * XSIZE * YSIZE)) == NULL) {
    prmsg(FATAL,("No memory for abs_src\n"));
  }

  memset(abs_src,0xff,2 * sizeof(short) * XSIZE * YSIZE);

  //  print_memsize();
  prmsg(MSG,("Going to produce the correction program\n"));

  flush_interval = YSIZE / 64;
  flush_cnt = 0;

  /*
   * Loop over all source pixels and get the corresponding corrected coordinate
   * values (target pixels).
   *
   * Then determine for every target pixel how many source pixels contribute to
   * it, and what fraction of each source pixel's area goes into that
   * particular target pixel.
   */
  for(iy = 0; iy < YSIZE; iy++) {
    if(flush_cnt++ > flush_interval) {
      flush_cnt = 0;
      prmsg(MSG,("."));
    }
    for(ix = 0; ix < XSIZE; ix++) {
      /*
       * Get the corrected (x,y) coordinate values for the corners of the
       * square formed by the present point and the three neighbouring ones.
       *
       * The present point has index 0, the others are arranged as follows:
       *
       *                         0   1   |---> x
       *                                 |
       *                         3   2   V y
       */
#if LOW_MEM
      my_func(ix,iy,fx,fy);
      my_func(ix+1,iy,fx+1,fy+1);
      my_func(ix+1,iy+1,fx+2,fy+2);
      my_func(ix,iy+1,fx+3,fy+3);
#else
      idx = ix + iy * (XSIZE+1);
      fx[0] = x_trg[idx]; fy[0] = y_trg[idx];
      fx[1] = x_trg[idx + 1]; fy[1] = y_trg[idx+1];
      fx[2] = x_trg[idx + XSIZE+1 + 1]; fy[2] = y_trg[idx + XSIZE+1 + 1];
      fx[3] = x_trg[idx + XSIZE+1]; fy[3] = y_trg[idx + XSIZE+1];
#endif /* LOW_MEM */
      minmax4(fx[0],fx[1],fx[2],fx[3],&fxmin,&fxmax);
      minmax4(fy[0],fy[1],fy[2],fy[3],&fymin,&fymax);

      if(fxmin < 0 || fymin <0 || fxmax >= XSIZE  || fymax >= YSIZE)
	continue;

      /*
       * Determine for the source pixel whose corrected coordinates are in
       * (fx,fy) the target pixels that it is mapped on, and the area of the
       * source pixel that is mapped into each of the target pixels.
       *
       * On return from calcparts(), the array "parts" contains these areas. It
       * represents a two-dimensional array in target x and y, with x
       * increasing fastest. The ranges are xmin <= x <= xmax and
       * ymin <= y <= ymax.
       */
      if(fabs(fx[0] - fx[1]) > MAX_PIXELSIZE
        || fabs(fy[0] - fy[2]) > MAX_PIXELSIZE) {
	prmsg(WARNING,("Warning: Pixel size x=%f y=%f to big (>%d)\n",
	  fabs(fx[0] - fx[1]),fabs(fy[0] - fy[2]),MAX_PIXELSIZE));
	continue;
      } else {
	calcparts(fx,fy,0,parts,&xmin,&xmax,&ymin,&ymax,&total);
	if(total < 1E-6) {
	  prmsg(WARNING,("Pixel [%f,%f],[%f,%f], [%f,%f], [%f,%f] is 0\n",
	    fx[0],fy[0],fx[1],fy[1],fx[2],fy[2],fx[3],fy[3]));
	  prmsg(WARNING,("        ix: %d iy: %d \n",ix,iy));
	  continue;
	}
      }

      max_pixelpart = 0;
      max_pixellut = NULL;
      pidx = 0;
      psum = 0;

#if BOUND_CHECK
      if(xmax >= XSIZE || xmin < 0 || ymax >= YSIZE || ymin < 0)
	prmsg(FATAL,("Target pxl out of bounds %d %d %d %d\n",
	  xmin,xmax,ymin,ymax));
#endif /* BOUND_CHECK */

      /*
       * Loop over all target pixels that have contributions from the source
       * pixel being processed.
       */
      for(j = ymin; j <= ymax; j++) {
	for(i = xmin; i <= xmax; i++) {

          /*
           * Determine for the target pixel the corresponding locations in the
           * look-up table, the look-up-table program and the source coordinate
           * array.
           */
	  if(i == 0 && j == 0)
	    offset = 0;
	  else
	    offset = offsets[i + j * XSIZE -1];

#if BOUND_CHECK
	  if(i == XSIZE - 1 && j == YSIZE - 1)
	    lastoffset = prog_length -1;
	  else
	    lastoffset = offsets[i + j * XSIZE] - 1;
#endif /* BOUND_CHECK */
	  prog_ptr = lut_prog + offset;
	  lut_ptr = lut_tab + offset;
	  abs_ptr = abs_src + 2 * (i + j * XSIZE);

#if BOUND_CHECK
	  if(!(*lut_ptr & BITMASK))
	    prmsg(ERROR,("Target bit is unset for %d (*lut_ptr = 0x%x)\n",
	      offset,*lut_ptr));
#endif /* BOUND_CHECK */

          /*
           * The contributions of each source pixel to each target pixel are so
           * far expressed as absolute area values.
           *
           * Convert them into relative values by dividing each of those area
           * values by the total area that the corresponding source pixel
           * covers in the target space.
           *
           * Technical remark: the fraction is expressed as an integer on a
           * scale from 0 to FULLSCALE (normally 0x8000 = 32768).
           */
	  part = (parts[pidx] / (float)total * (float)FULLSCALE + .5);

          /*
           * Fill the source coordinate array, the look-up table, and the
           * look-up-table program with the values for this particular target
           * pixel.
           *
           * If there is no source pixel contributing to this target pixel,
           * then skip this part.
           */
	  if(part == 0) {
	    pidx++;
	    continue;
	  }

          /*
           * Fill the source coordinate array.
           *
           * If this is the first source pixel that contributes to this target
           * pixel, fill its x and y coordinates in the source coordinate
           * array.
           *
           * Otherwise, get the coordinates of the first contributing source
           * pixel from the source coordinate array.
           */
	  if(*abs_ptr == 0xffff) {
	    first = 1;
	    src_x = ix;
	    src_y = iy;
	    *abs_ptr++ = ix;
	    *abs_ptr   = iy;
	  } else {
	    first = 0;
	    src_x = *abs_ptr++;
	    src_y = *abs_ptr;
	  }

          /*
           * Fill the look-up-table program.
           *
           * Go through the instructions of the look-up-table program for this
           * target pixel and calculate the (x,y) coordinates of all source
           * pixels that have already been processed for this target pixel.
           *
           * This loop ends when encountering an instruction that has still the
           * initialization value 0xff (note that this cannot be a valid
           * instruction for coordinate calculations).
           *
           * At the end of the loop, src_x and src_y contain the (x,y)
           * coordinates of the last source pixel already processed for this
           * target pixel.
           */
	  while((old_instruct = *prog_ptr) != 0xff) {
	    prog_ptr++;
	    lut_ptr++;
	    src_x += (old_instruct & 0x0f) - 8;
	    src_y += ((old_instruct & 0xf0) >> 4) - 8;
	  }

#if BOUND_CHECK
	  if(src_x >= XSIZE || src_x < 0 || src_y >= YSIZE || src_y < 0)
	    prmsg(MSG,("Src pxl out of bounds in %d c.) %d %d (0x%x 0x%x)\n",
	      prog_ptr - lut_prog,src_x,src_y,instruct,part));
	  if(abs(src_x - ix) > 7 || abs(src_y - iy) > 7)
	    prmsg(DMSG,("src pixel area to big in %d d.) [%d, %d]  [%d,%d]\n",
	      prog_ptr - lut_prog,src_x,src_y,ix,iy));
#endif /* BOUND_CHECK */

          /*
           * Calculate in x and y the difference between the last and the
           * present source pixel and store them in the instruction. For
           * details, see the description of the look-up-table program further
           * above.
           */
	  instruct = (ix - src_x + 8) + ((iy - src_y + 8) << 4);
	  if(abs(src_x - ix) > 7 || abs(src_y - iy) > 7 || instruct == 0xff) {
	    count_deleted++;
	    pidx++;
	    if(first) {
	      prmsg(FATAL,("Can't happen"));
	    }
	    continue;
	  }

#if BOUND_CHECK
	  if(prog_ptr > lut_prog + lastoffset)
	    prmsg(FATAL,("About to overwrite internal table in %d (>%d)\n",
	      prog_ptr - lut_prog,lastoffset));
#endif /* BOUND_CHECK */

	  *prog_ptr = instruct;

          /*
           * Fill the look-up table.
           *
           * The locations in the look-up table and the look-up-table program
           * must always be synchronized. The pointer for the look-up table
           * has been increased above simultaneously with the one for the
           * look-up-table program, thus it now points to the location in the
           * look-up table where the contribution from source pixel (ix,iy) to
           * target pixel (i,j) should go.
           *
           * Store this contribution in the look-up table. For details, see the
           * description of the look-up table further above.
           */
	  if(part > max_pixelpart) {
	    max_pixelpart = part;
	    max_pixellut  = lut_ptr;
	  }
	  psum += part;

	  if(part >= FULLSCALE)
	    part = 0;

	  if(first)
	    part |= BITMASK;

#if BOUND_CHECK
	  if(first && !(*lut_ptr & BITMASK))
	    prmsg(ERROR,("Target bit is unset for %d (*lut_ptr = 0x%x)\n",
	      lut_ptr - lut_tab,*lut_ptr));
	  if(!first && (*lut_ptr & BITMASK))
	    prmsg(ERROR,("Target bit is set for %d (*lut_ptr = 0x%x)\n",
	      lut_ptr - lut_tab,*lut_ptr));
#endif /* BOUND_CHECK */

	  *lut_ptr = part;

#if BOUND_CHECK
          /*
           * At the beginning of a sequence, the BITMASK flag must be set in
           * the look-up table and the corresponding look-up-table program
           * instruction must be 0x88. If not, exit with error.
           */
	  if(*lut_ptr & BITMASK)
	    if(*prog_ptr != 0x88)
	      prmsg(FATAL,("Instruct != 0x88 but 0x%x (0x%x in %d)\n",
		*prog_ptr,*lut_ptr,prog_ptr - lut_prog));
	    else
              /*
               * count_ff only here for debugging
               */
	      count_ff++;
#endif /* BOUND_CHECK */

	  pidx++;
	}
      }

      /*
       * End of the loop processing the target pixels for this source pixel.
       *
       * The sum of all contributions (parts) of this source pixel should be
       * equal to FULLSCALE (because of the way the parts are normalized). If
       * this is not the case, then there have been rounding errors. Put the
       * missing amount (i.e., FULLSCALE - sum of all parts) into the target
       * pixel that has the largest contribution from this source pixel - that
       * seems to be a good place to put it.
       *
       * Note that no value in the look-up table must be equal to FULLSCALE
       * because it would create confusion with the meaning of BITMASK.
       * Therefore, if a pixel has really the value FULLSCALE, hide it as 0.
       * Real target pixels cannot have a 0 contribution, because then they
       * would not have been included in the list of pixels that the source
       * pixel contributes to.
       */
      if(max_pixellut && psum != FULLSCALE) {
	int savelut;
	savelut = *max_pixellut & MAPSCALE;
	if(savelut == 0)
	  savelut = FULLSCALE;
	part = savelut + FULLSCALE - psum;
	if(part >= FULLSCALE)
	  part = 0;
	if(*max_pixellut & BITMASK)
	  *max_pixellut = part | BITMASK;
	else
	  *max_pixellut = part;
#if BOUND_CHECK
	if(*max_pixellut & BITMASK) {
	  instruct = lut_prog[max_pixellut - lut_tab];
          if(instruct != 0x88)
	    prmsg(FATAL,("prog != 0x88 max_p 0x%x (0x%x in %d) 0x%x %d\n",
	      instruct,*max_pixellut,prog_ptr - lut_prog,savelut,psum));
	  else
            /*
             * count_ff only here for debugging
             */
	    count_ff++;
	}
#endif /* BOUND_CHECK */
      }
    }
  }

  prmsg(MSG,("\n"));
  /*
   * At this point we have finished the look-up-table program "lut_prog" with
   * all the relative source pixel coordinates and the look-up table "lut_tab"
   * with the fractional contributions of the source pixels to the target
   * pixels. The values in the look-up table have the highest bit set if at the
   * beginning of a new target pixel sequence.
   *
   * Beginning of a new sequence:
   *
   *                look-up table          program
   *
   *              1vvv vvvv vvvv vvvv      10001000
   *              0vvv vvvv vvvv vvvv      yyyyxxxx
   *              0vvv vvvv vvvv vvvv      yyyyxxxx
   */

  if(count_deleted)
    prmsg(WARNING,("\n%d target pxls deleted, too many src pxls contribute\n",
      count_deleted));

#if BOUND_CHECK
  /*
   * Test whether the arrays are correctly set up for the number of target
   * pixels in the image:
   *
   * - the number of entries in the look-up table that have the BITMASK flag
   *   set should be equal to the number of target pixels (BITMASK signals the
   *   begin of a new target pixel sequence);
   * - the number of entries in the source coordinate array that still have
   *   the initialization value should be equal to the number of target pixels
   *   that do not receive any contribution from any of the source pixels;
   * - all entries in the look-up table that correspond to an index value in
   *   the offset table should be at the beginning of a new sequence and thus
   *   have the BITMASK flag set.
   */
  {
    int count_case = 0, i, idx;
    prog_end = lut_prog + prog_length;
    lut_ptr = lut_tab;
    for(prog_ptr = lut_prog; prog_ptr != prog_end; prog_ptr++, lut_ptr++) {
      if(*lut_ptr & BITMASK)
	count_case++;
    }
    if(count_case != XSIZE * YSIZE)
      prmsg(ERROR,("Lost some target pixels (now only %d)\n",count_case));

    for(count_case = 0, i = 0; i < XSIZE * YSIZE; i++)
      if(abs_src[2*i] == 0xffff && abs_src[2*i+1] == 0xffff)
	count_case++;
    prmsg(DMSG,("Empty targets via abs_src count: %d\n",count_case));

    for(offset_p = offsets, i = 0; i< XSIZE * YSIZE -1; i++) {
      if(!(lut_tab[*offset_p++] & BITMASK)) {
	prmsg(ERROR,("Target advance for %d lost\n",i));
	for(idx = *(offset_p-1); idx < *offset_p; idx++)
	  prmsg(DMSG,(" %d: 0x%4x 0x%2x\n",idx,lut_tab[idx],lut_prog[idx]));
      }
    }
  }
#endif /* BOUND_CHECK */

  /*
   *
   * Step 2: remove the empty target pixels at the beginning of the target
   *         image, and mark the other empty target pixels with the flag
   *         INCTARGET in the look-up table. The look-up table and the
   *         look-up-table-program are shortened according to the number of
   *         empty pixels removed.
   *
  */
  prmsg(MSG,("Delete untouched pixels\n"));

  /*
   * Go through the look-up table and the look-up-table program to find empty
   * target pixels, i.e. target pixels that do not receive a contribution from
   * any source pixel.
   *
   * Delete all consecutive empty target pixels at the beginning of the image,
   * and mark all other empty target pixels as an exception of type INCTARGET.
   */
  abs_ptr  = abs_src;
  prog_end = lut_prog + prog_length;
  lut_ptr  = lut_tab;
  start_ff = 1;
  start_tidx = 0;

  /*
   * Find all empty target pixels. These pixels still contain the
   * initialization value 0xff as look-up-table program instruction.
   *
   * Count how many consecutive empty target pixels are at the beginning of the
   * image array.
   *
   * Mark all other empty target pixels
   * - by setting the look-up-table program instruction to 0x0 (signifies
   *   exception),
   * - and by setting the flags BITMASK and INCTARGET as value in the look-up-
   *   table (signifies beginning of a new target sequence with an empty target
   *   pixel).
   */
  for(prog_ptr = lut_prog; prog_ptr != prog_end; prog_ptr++, lut_ptr++) {
    if(*prog_ptr == 0xff) { /* Nothing will be put in this target */
      if(*lut_ptr & BITMASK) {
	if(start_ff)
	  start_tidx++;
	else {
	  *prog_ptr = 0x0;
	  *lut_ptr  = BITMASK | INCTARGET;
	  count_inc++;
	}
      }
    } else
      start_ff = 0;
  }

  /*
   * Remove the consecutive empty target pixels from the beginning of the
   * look-up table and the look-up-table program.
   *
   * Then re-adjust the size of the look-up table and the look-up-table
   * program.
   */
  new_prog_ptr = lut_prog;
  new_lut_ptr = lut_tab;
  lut_ptr = lut_tab;
  for(prog_ptr = lut_prog; prog_ptr != prog_end; prog_ptr++, lut_ptr++) {
    *new_prog_ptr = *prog_ptr;
    *new_lut_ptr = *lut_ptr;
    if(*prog_ptr != 0xff) {
      new_prog_ptr++;
      new_lut_ptr++;
    }
  }

  prog_length = new_prog_ptr - lut_prog;

  if(count_parts != prog_length)
    prmsg(DMSG,("%d deleted - start offset %d\n",count_parts - prog_length,
      start_tidx));

  if((prog_ptr = lut_prog = (unsigned char *)
    prealloc(lut_prog,sizeof(char) * prog_length)) == NULL) {
    prmsg(FATAL,("Realloc failed \n"));
  }

  if((lut_ptr = lut_tab = (LUT_TYPE *)
    prealloc(lut_tab,sizeof(LUT_TYPE) * prog_length)) == NULL) {
    prmsg(FATAL,("Realloc failed \n"));
  }

  /*
   * At this point the empty target pixels have been marked in the look-up
   * table and the program in the following way:
   *
   *             program     look-up table
   *
   *             00000000    1000 0000 0000 0001  (= BITMASK | INCTARGET)
   *
   * This indicates a pure target advance, i.e. a target pixel that can be
   * skipped over.
   *
   * For non-empty target pixels, the contents of look-up table and program
   * have not changed.
   */

#if BOUND_CHECK
  if(verbose > 1) {
    int count_case = 0,count_case1 = 0,count_case2 = 0;

    prog_end = lut_prog + prog_length;
    lut_ptr = lut_tab;

    /*
     * At the beginning of a sequence, there are now two possibilities:
     * - either the flag INCTARGET is set and the program instruction is 0,
     *   this marks an empty target pixel;
     * - or the program instruction is 0x88 as it should be at the beginning of
     *   a sequence with source pixels.
     *
     * Give a warning if this is not the case.
     */
    for(prog_ptr = lut_prog; prog_ptr < prog_end; prog_ptr++, lut_ptr++) {
      if(*lut_ptr & BITMASK) {
	if((*lut_ptr != (BITMASK | INCTARGET) || *prog_ptr != 0) &&
	  (*prog_ptr != 0x88))
	  prmsg(ERROR,("Can't be *lut_ptr = 0x%x *prog_ptr = 0x%x\n",
	    *lut_ptr,*prog_ptr));
	if(*prog_ptr == 0)
	  count_case1++;
	else
	  count_case2++;

      } else
	count_case++;
    }

    /*
     * Verify the program length:
     * - count_case1  contains the number of empty target pixel instructions;
     * - count_case2  contains all other instructions that mark the beginning
     *                of a new sequence;
     * - count_case   contains the number of all instructions that are not at
     *                the beginning of a new sequence.
     *
     * The sum of the three should equal the program length. If not, give an
     * error message.
     */
    prmsg(DMSG,("Counted %d parts (LOOP count %d (INC TARGET) %d (TRGS))\n",
      count_case,count_case1,count_case2));
    if(count_case+count_case1+count_case2 != prog_length)
      prmsg(ERROR,("Lost some events somewhere\n"));

  }
#endif /* BOUND_CHECK */

  /*
   *
   * Step 3: optional: if the user-definable global variable "DO_FLAT" is set,
   *         the target image is normalized to a flat image. The structure of
   *         the tables (look-up table, etc.) is not changed.
   *
   * Going to normalize to flat target image. This means that when this code is
   * run a peak with 10000 pixels in the source image will not be 10000 pixels
   * in the target image any more. On the other hand a constant image will now
   * become another constant image.
   */
  if(DO_FLAT) {
    LUT_TYPE *lp,*start_lut;
    unsigned long partsum, newpartsum;
    unsigned long part;
    LUT_TYPE bitmask;
    int count_case = 0, count_case1 = 0, count_case2 = 0;

    prmsg(MSG,("Normalizing for flat output image \n"));
    prog_end = lut_prog + prog_length;
    lut_ptr  = lut_tab;
    start_lut = lut_tab;

    /*
     * Loop through the look-up table, add for each target pixel all fractional
     * contributions from the source pixels, rescale them and store them back
     * in the look-up table.
     *
     * To do so, start with the first target pixel and skip forward through the
     * look-up-table program until the next target pixel is found. This defines
     * the start point "start_lut" and the end point "lut_ptr" of the summing
     * for the first target pixel.
     *
     * The following target pixels are processed in an analog manner.
     *
     * Note that the loop will go too far, as it has to go all the way to the
     * end of the program. "lut_ptr" does therefore not point to a valid
     * location of the look-up table on the last turn - careful.
     */
    for(prog_ptr = lut_prog; prog_ptr <= prog_end; prog_ptr++, lut_ptr++) {
      if(prog_ptr == prog_end || (*lut_ptr & BITMASK)) {

	/*
         * Calculate the sum of the fractional contributions in the look-up
         * table for this target pixel.
         *
         * As this sums over all source pixels that contribute to a given
         * target pixel, this sum can of course be different from
         * 100 % (= FULLSCALE), in particular it can also be bigger than
         * 100 %. On the other hand, the sum of all parts that a given source
         * pixel contributes to all target pixels should always be equal to
         * FULLSCALE (= 100 %).
         *
         * Note that a value of 0 in the look-up table really means a value
         * of FULLSCALE. For a detailed explanation, see above at the end of
         * the loop processing the target pixels for a given source pixel.
         */
	partsum = 0;
	max_pixelpart = 0;
	max_pixellut = NULL;

	for(lp = start_lut; lp < lut_ptr; lp++) {
	  if(*lp & MAPSCALE)
	    part = (*lp & MAPSCALE);
	  else
	    part = FULLSCALE;

	  if(part > max_pixelpart) {
	    max_pixelpart = part;
	    max_pixellut  = lp;
	  }
	  partsum += part;
	}

#if BOUND_CHECK
        /*
         * Because of the way the loop through the look-up table is processed,
         * "start_lut" can only point at locations that correspond to the
         * beginning of a new sequence.
         *
         * Give an error message if this is not the case.
         */
	if(start_lut != lut_tab && start_lut < lut_tab + prog_length &&
	  (*start_lut & BITMASK) == 0)
	  prmsg(ERROR,("Can't be: *start_lut is 0x%x *prog = 0x%x in %d\n",
	    *start_lut,*prog_ptr,prog_ptr - lut_prog));
#endif /* BOUND_CHECK */

        /*
	 * partsum should be approximately FULLSCALE on the average.
         * The value is normally not simply FULLSCALE as we are counting
         * here the sum of percentages of the source pixel contribution to
         * the target pixel (i.e. this could be 300 % if 3 source pixels
         * contribute each to 100 %).
         *
         * Rescale the fractional contributions to make partsum equal to
         * FULLSCALE.
         */
	bitmask = BITMASK;
	newpartsum = 0;
	for(lp = start_lut; lp < lut_ptr; lp++) {
	  part = *lp & MAPSCALE;
	  if(part == 0)
	    part = FULLSCALE;
	  part = (part << SHIFT) / partsum;

          /*
           * A contribution of 0 is not supposed to happen for a contributing
           * pixel. Thus, if the rescaling leads to a contribution of 0 (trough
           * truncating in the integer calculation), set the contribution to 1.
           */
	  if(part == 0) {
	    count_case++;
	    part = 1;
	  }

          /*
           * Store the rescaled values back into the look-up table. Set the
           * BITMASK flag for the first pixel of a sequence. Note that
           * "bitmask" is BITMASK for the starting pixel of a sequence and 0
           * for all other pixels.
           *
           * As before, a value of FULLSCALE is hidden in an artificial 0.
           */
	  newpartsum += part;
	  if(part >= FULLSCALE)
	    *lp = 0 | bitmask;
	  else
	    *lp = part | bitmask;
	  bitmask = 0;
	}

        /*
         * The sum of the rescaled contributions for this target pixel must be
         * equal to FULLSCALE. If this is not the case, then there have been
         * rounding errors. Attribute the difference (i.e., FULLSCALE - sum of
         * all parts) to that source pixel that gives the largest contribution
         * to this target pixel - that seems to be the best choice possible.
         */
	if(max_pixellut && newpartsum != FULLSCALE) {
	  int savelut;
	  savelut = *max_pixellut & MAPSCALE;
	  if(savelut == 0)
	    savelut = FULLSCALE;
	  part = savelut + FULLSCALE - newpartsum;
	  if(part >= FULLSCALE)
	    part = 0;
	  if(*max_pixellut & BITMASK)
	    *max_pixellut = part | BITMASK;
	  else
	    *max_pixellut = part;
	}

#if BOUND_CHECK
        /*
         * Test to see if the "start_lut" location still has the BITMASK flag
         * set after the rescaling.
         *
         * Give an error message if this is not the case.
         */
	if(start_lut != lut_tab && start_lut < lut_tab + prog_length &&
	  (*start_lut & BITMASK) == 0)
	  prmsg(ERROR,("Can't be: *start_lut is 0x%x *prog = 0x%x in %d\n",
	    *start_lut,*prog_ptr,prog_ptr - lut_prog));
#endif /* BOUND_CHECK */

        /*
         * Set the start pointer "start_lut" for the next target pixel. Usually
         * it will just be the end pointer "lut_ptr" of the present pixel, but
         * if the next target pixel(s) is (are) empty (INCTARGET flag set),
         * then it (they) will just be skipped.
	 */
	if(prog_ptr != prog_end && *prog_ptr == 0) {
#if BOUND_CHECK
          /*
           * A look-up-table instruction of 0 signals an exception. The only
           * type of exception defined so far is the INCTARGET exception, which
           * can only occur at the beginning of a new sequence. Give an error
           * message if this is not the case.
           */
	  if(*lut_ptr != (INCTARGET | BITMASK))
	    prmsg(ERROR,("Can't be: *lut_ptr is 0x%x\n",*lut_ptr));
	  count_case1++;
#endif /* BOUND_CHECK */
	  start_lut = lut_ptr + 1;           /* Ignore the empty target. */
	} else {
	  start_lut = lut_ptr;
#if BOUND_CHECK
	  count_case2++;
#endif /* BOUND_CHECK */
	}
      }
    }
    /*
     * Print statistics on the target pixels:
     * - count_case      is the number of source pixels that got a 0
     *                   contribution through the rescaling;
     * - count_case1     is the number of empty target pixels;
     * - count_case2 - 1 is the number of the non-empty target pixels
     *                   (-1 as the case prog_ptr == prog_end is counted also)
     */
    prmsg(DMSG,("Counted %d part == 0 (LOOP count %d %d)\n",
      count_case,count_case1,count_case2 - 1));
  }

  /*
   * Step 4: the absolute addressing scheme for the first contributing source
   *         pixel is replaced by relative addressing whenever possible. Where
   *         it is not possible, the flag ABSSRC is set in the corresponding
   *         location in the look-up table. At the end of this step, the source
   *         coordinate table has its final form.
   *
   * At present, the first program instruction for a new target pixel sequence
   * contains always 0x88. This location can be used to store the position of
   * the first source pixel in this target pixel sequence relative to the last
   * source pixel of the previous target pixel.
   *
   * The intention is to obtain the coordinates of the source pixels by
   * specifying the relative distance of the new source pixel with respect to
   * the source pixel treated last. This relative distance will then be stored
   * in the look-up-table program instruction for this source pixel.
   *
   * As the length of a program instruction is fixed (8 bit), there is a
   * maximum of 16 values that can be stored as difference for x and for y.
   * Moreover, the instruction values 0 and 0xff have a special meaning, thus
   * the difference can only have 15 values. These are chosen to be symmetric
   * around 0, i.e. the difference in x and in y must be between -7 and +7.
   *
   * If this maximum difference is exceeded, absolute source coordinates must
   * be used. A 0 will be stored in the look-up-table instruction to mark an
   * exception, and the type of the exception will be ABSSRC.
   *
   * starttidx  contains the number of empty pixels at the beginning of the
   *            target image, i.e. it is the index of the first non-empty pixel
   *            in the target image array.
   * abs_src  contains the x and y coordinates of the source image pixel that
   *          corresponds to the target image pixel being processed.
   *          The array is in principle a two-dimensional pixel array
   *          organized for speed as an one-dimensional array with line-order
   *          (i.e., the x coordinate increases fastest). For each pixel there
   *          are two short integers, the first with the x and the second
   *          with the y coordinate.
   * src_x and src_y  are the x and y coordinates of the source image pixel
   *                  that corresponds to the first non-empty target pixel.
   * startsidx  contains for this source image pixel the index into the source
   *            image array.
   */

  prmsg(MSG,("Going to relative pixels\n"));

  lut_d->starttidx = start_tidx;
  src_x = abs_src[2 * start_tidx];
  src_y = abs_src[2 * start_tidx + 1];
  lut_d->startsidx = src_x +  XSIZE * src_y;

  /*
   * The first valid (non-empty) target pixel must be treated separately, as
   * for it there is no "last source pixel of the previous target pixel".
   * Mark it in the look-up table as if it was not the beginning of a new
   * sequence.
   * This will leave the instruction 0x88 in the corresponding look-up-table
   * program.
   */
  lut_tab[0] &= MAPSCALE;

  new_abssrc = abs_src;
  abs_ptr = abs_src + 2 * start_tidx + 2;
  count_trg = start_tidx + 1;

  lut_ptr = lut_tab;
  prog_end = lut_prog + prog_length;
  /*
   * Loop through the program and look for the beginnings of new target pixel
   * sequences.
   *
   * If one is found, then the look-up-table program instruction for the first
   * source pixel will be modified as described above.
   *
   * For the other source pixels in the program (subsequent pixels for an
   * already found target pixel), just calculate their source coordinates. Thus
   * (src_x, src_y) will contain the "coordinates of the last source pixel from
   * the previous target pixel".
   */
  for(prog_ptr = lut_prog; prog_ptr != prog_end; prog_ptr++, lut_ptr++){
    instruct = *prog_ptr;
    /*
     * Case 1: source pixel at the beginning of a new target pixel sequence.
     */
    if((part = *lut_ptr) & BITMASK) {
#if BOUND_CHECK
      /*
       * At the beginning of a new target pixel sequence, the look-up-table
       * program instruction must be either 0x88 for a normal sequence or 0 for
       * an exception.
       *
       * If this is not the case, give an error message.
       */
      if(instruct != 0x88  && instruct != 0) {
	prmsg(ERROR,("Corrupted Tables prog = 0x%x lut = 0x%x in %d\n",
	  instruct,part,prog_ptr - lut_prog));
	continue;
      }
#endif /* BOUND_CHECK */
      new_x = *abs_ptr++;
      new_y = *abs_ptr++;
#if BOUND_CHECK
      count_trg++;
#endif /* BOUND_CHECK */
      /*
       * Skip empty target pixels.
       */
      if(instruct == 0)
	continue;
#if BOUND_CHECK
      /*
       * For non-empty target pixels, the corresponding coordinates in the
       * source coordinate array must have a real value, not any longer the
       * initialization value.
       *
       * If they do, exit with error.
       */
      if(new_x == 0xffff)
	prmsg(FATAL,("No absolute src coordinate for %d in %d\n",
	  count_trg,prog_ptr - lut_prog));
#endif /* BOUND_CHECK */
      diff_x = new_x - src_x;
      diff_y = new_y - src_y;

      /*
       * The first source pixel for this new target pixel might be wrapped
       * around the end of the line with respect to the last source pixel of
       * the previous target pixel, e.g. (new_x,new_y) = (XSIZE - 1, 3) and
       * (src_x, src_y) = (1, 2).
       *
       * Adding or subtracting XSIZE to the x-difference and modifying the
       * y-difference accordingly allows these cases to be treated as described
       * above (maximum difference +-7). This way the use of absolute
       * coordinates can be avoided here as well.
       */
      if(diff_x >= 8) {
	diff_x -= XSIZE;
	diff_y++;
      }
      if(diff_x <= -8) {
	diff_x += XSIZE;
	diff_y--;
      }

      /*
       * Test if the difference between the new and the old source pixel
       * coordinates can be stored in the 8 bits of the look-up-table program
       * instruction, i.e. if it is between -7 and +7 for both x and y.
       *
       * If yes, store the difference in the look-up-table program instruction.
       *
       * If no, then absolute source coordinates must be used. Mark this event
       * as exception of type ABSSRC. Store the x and y source coordinates in
       * two subsequent locations in the source coordinate array. Store the
       * relative contribution of this source pixel to the target pixel also in
       * the source coordinate array, immediately behing the coordinates.
       */
      if(diff_x >= 8  || diff_x <= -8 || diff_y >= 8 || diff_y <= -8) {
	*prog_ptr = 0x0;
	*lut_ptr  = BITMASK | ABSSRC;
	*new_abssrc++ = new_x;
	*new_abssrc++ = new_y;
        /*
         * As the new set of data is written in the same array as the old
         * coordinates, make sure that this does not overwrite old coordinates
         * that have not been processed yet.
         *
         * This is not that likely to happen, as there should be room at the
         * beginning of the source coordinate array due to empty target pixels
         * at the beginning of the image.
         *
         * However, if it does happen, exit with error.
         */
	if(new_abssrc >= abs_ptr)
	  prmsg (FATAL,("Really really unlucky\n"));
	*new_abssrc++ = part & MAPSCALE;
	count_abs++;
      } else {
	instruct = (diff_x + 8)  + ((diff_y + 8) << 4);
	*prog_ptr = instruct;
#if BOUND_CHECK
        /*
         * As the differences have been tested to be between -7 and +7, the
         * resulting instruction cannot be 0.
         *
         * If it is, exit with error.
         */
	if(*prog_ptr == 0)
	  prmsg(FATAL,("instruct is 0 - Can't be\n"));
#endif /* BOUND_CHECK */
      }

      src_x = new_x;
      src_y = new_y;
#if BOUND_CHECK
      if(src_x >= XSIZE || src_x < 0 || src_y >= YSIZE || src_y < 0)
	prmsg(MSG,("Source pixel out of bounds in %d a.) %d %d (0x%x 0x%x)\n",
	  prog_ptr - lut_prog,src_x,src_y,instruct,part));
#endif /* BOUND_CHECK */
    } else {
    /*
     * Case 2: source pixel that is not at the beginning of a new target pixel
     * sequence.
     */
      /* ??? can instruct == 0 happen here at all ??? */
      if(instruct != 0) {
	src_x += (instruct & 0x0f) - 8;
	src_y += (((int) (instruct & 0xf0)) >> 4) - 8;
#if BOUND_CHECK
	if(src_x >= XSIZE || src_x < 0 || src_y >= YSIZE || src_y < 0)
	  prmsg(MSG,("Source pixel out of bounds in %d b.) %d %d (0x%x 0x%x)\n",
	    prog_ptr - lut_prog,src_x,src_y,instruct,part));
#endif /* BOUND_CHECK */
      }
    }
  }
#if BOUND_CHECK
  if(count_trg != XSIZE * YSIZE)
    prmsg(FATAL,("Not all target pixels treated - can't be %d\n",count_trg));
#endif /* BOUND_CHECK */
  prmsg(DMSG,("Treated: %d empty target increments %d abs. source indices\n",
    count_inc,count_abs));

  /*
   * Re-adjust the size of the source coordinate array. It is also possible
   * that this array is no longer needed, because no target pixel needs
   * absolute source coordinates. Then the source coordinate array is just
   * freed.
   */
  if(new_abssrc == abs_src) {
    pfree(abs_src);
    abs_src = lut_d->abs_src = NULL;
    noabs = 0;
  } else {
    abs_src = lut_d->abs_src = (unsigned short *)
      prealloc(abs_src,(noabs = new_abssrc - abs_src) * sizeof (short));
    if(abs_src == NULL)
      prmsg(FATAL,("realloc fails - why is that\n"));
  }

  pfree(offsets);

#if WASTE4_FORSPEED
  /*
   * Save relative integer target position. This is used later for the overflow
   * correction. x_trg and y_trg contain the the target pixel coordinates.
   * We store here the differences in the short arrays lut_d->xrel and
   * lut_d->yrel. (i.e. if the pixel (100, 100) is distorted to position
   * (103.4, 101.6) then x_trg[100] is 103.4, y_trg[100] is 101.6,
   * my_func(100,100,&x,&y) would return (103.4, 101.6) and lut_d->xrel
   * is 3, and lut_d->yrel is 2.)
   *
   * Note that x_trg, y_trg are floats and lut_d->xrel,yrel are shorts. The
   * reason for that is that we will only need approximate values for the
   * shift to blot out a square around overflow pixels.
   */
  if((lut_d->xrel = (short*)pmalloc(XSIZE * YSIZE * sizeof(short))) == NULL ||
    (lut_d->yrel = (short*)pmalloc(XSIZE * YSIZE * sizeof(short))) == NULL)
    prmsg(FATAL,("No memory for xrel and yrel\n"));

  for(idx = 0, j = 0; j < YSIZE; j++)
    for(i = 0; i < XSIZE; i++) {
#if LOW_MEM
      my_func(i,j,&x,&y);
      lut_d->xrel[idx] = x - i;
      lut_d->yrel[idx] = y - j;
#else
      /*
       * Note that x_trg and y_trg have dimensions [XSIZE + 1,YSIZE + 1]
       */
      lut_d->xrel[idx] = x_trg[i + j * (XSIZE + 1)] - i;
      lut_d->yrel[idx] = y_trg[i + j * (XSIZE + 1)] - j;
#endif /* LOW_MEM */
      idx++;
    }
#endif /* WASTE4_FORSPEED */

#if LOW_MEM
#if ANDY_CORR
  andy_free_buffers(X_COR,Y_COR);
  X_COR = Y_COR = NULL;
#endif /* ANDY_CORR */
#else /* LOW_MEM */
#if !BOUND_SUPER
  pfree(x_trg);
  pfree(y_trg);
#endif /* !BOUND_SUPER */
#endif /* LOW_MEM */

  /*
   * Step 5: the relative addressing is replaced by the compressed addressing
   *         scheme whenever possible.
   *
   *         At the end of this step, all tables (look-up table, etc.) have
   *         their final form.
   */
  {
    unsigned char *start_ptr,*end_ptr,*c_ptr;
    int sx,sy,bito,compress,newmask;
    unsigned char *new_prog,*new_p;
    LUT_TYPE *start_lutptr,*lut_end;
    LUT_TYPE *newlut;
    int v;
    int oldidx;
    unsigned int *histo,*histo2;
    int count_hist,hist_max;
    unsigned int *seq,seq_value;
    int len,abs,old;
    int *rel_ptr,*rel_tab,**relend_tab;
    int *offset_tab;
    int seq_code;

    prmsg(MSG,("Compressing program \n"));

    /*
     * At this point, the coordinates of a source pixel that contributes to a
     * given target pixel can be calculated in one of two ways:
     *
     * - relative: if the new source pixel is not too far away from the last
     *             source pixel processed, then the look-up-table program
     *             instruction for this new pixel contains the relative
     *             distance in x and y between the new and the previous
     *             source pixel;
     * - absolute: if the new source pixel is too far away, then the
     *             corresponding look-up-table program instruction contains a 0
     *             to mark an exception. The exception type is ABSSRC, and the
     *             coordinates of the new pixel are in the source coordinate
     *             array.
     *
     * However, in many cases the image will not be strongly distorted. This
     * means that the positions of the source pixels that contribute to a given
     * target pixel will be near that pixel's position. Also, the correction
     * function that maps the source pixel coordinates to the target pixel
     * coordinates is not likely to vary a lot from one pixel to the next.
     *
     * Therefore, the group of source pixels that contribute to a given target
     * pixel is likely to have a similar shape as one moves from one target
     * pixel to the next; or, expressed in a different way, the sequence of
     * source pixel movements is likely to show a similar pattern.
     *
     * This can be used to shorten the look-up-table program. The idea is to
     * define a certain number of these possible movement patterns and then
     * look at all target pixels to find out whether the corresponding source
     * pixels are described by one of the defined patterns. If this is the
     * case, the sequence of look-up-table program instructions that describe
     * the movement from one source pixel to the next can be replaced by a
     * reference to the corresponding pattern. As the hope is to end up with a
     * smaller program, the process is called "compressing the program".
     *
     * The technical implementation of this idea is described in the following.
     *
     * Scan all source pixel sequences in the look-up-table program for short
     * sequences in the immediate neighbourhood of the starting source pixel.
     * "Short" is here defined as reaching any combination of the pixels that
     * are in one of 18 locations mainly "below and to the right" of the
     * starting pixel. More precisely, if the starting pixel is marked by "x",
     * then all the pixels marked with a "." in the following diagram can be
     * reached:
     *
     *               x . . .           (the three pixels to the right)
     *             . . . . .           (line below: 1 left to 3 right)
     *             . . . . .           (two lines below: 1 left to 3 right)
     *             . . . . .           (three lines below: 1 left to 3 right)
     *
     * Note that the order in which the pixels are reached is not important.
     * Note also that this is not restricting the general case. The starting
     * pixel is simply defined to be in the upper left corner of the pattern.
     * Because of this definition it is sometimes possible that there is a
     * pixel in the pattern which is one unit more to the left in the next
     * line, but almost never two units.
     *
     * There are (2 power 18) possible pixel sequences that can be constructed
     * with 18 pixels. Each sequence can thus be described unambiguously by an
     * 18-bit sequence mask pattern in that way that the bit "i" is set if the
     * pixel "i" is part of the sequence.
     *
     * Determine for each of these sequences the frequency with which it occurs
     * in the look-up-table program.
     *
     * First allocate (2 power MAXHIST) integers of memory for the frequency
     * counters (MAXHIST = 1 << 18).
     */
    if((histo = (unsigned int *)pmalloc(MAXHIST * sizeof(int))) == NULL)
      prmsg(FATAL,("No memory"));
    memset(histo,0,MAXHIST * sizeof(int));

    /*
     * Loop through the program and search for the the next target pixel (= end
     * point of the present source pixel sequence and starting point of the
     * next sequence). This is marked by the flag BITMASK being set.
     *
     * Note that the first program instruction is already at a target pixel,
     * thus there is a "present pixel sequence" at the start of the loop.
     */
    lut_ptr  = lut_tab;
    prog_end = lut_prog + prog_length;
    start_ptr = lut_prog;
    for(prog_ptr = lut_prog; prog_ptr != prog_end;) {
      start_ptr = prog_ptr;
      start_lutptr = lut_ptr;
      lut_ptr++;
      prog_ptr++;
      while(prog_ptr < prog_end && !(*lut_ptr & BITMASK)) {
	lut_ptr++;
	prog_ptr++;
      }
      /*
       * End of sequence found. Now analyze the pixels involved in the
       * sequence: do they all belong to the 18 positions defined above?
       *
       * If yes, the sequence is one of the short sequences that is being
       * looked for: increase the frequency count of the corresponding counter.
       *
       * If no, stop the analysis of this sequence and look for the next one.
       */
      end_ptr = prog_ptr - 1;
      compress = -1;
      if(start_ptr <= end_ptr) {
	sx = 0; sy = 0; compress = 0;
	for(c_ptr = start_ptr + 1; c_ptr <= end_ptr; c_ptr++) {
	  instruct = *c_ptr;
	  sx += (instruct & 0x0f) - 8;
	  sy += (((int)(instruct & 0xf0)) >> 4) - 8;
	  bito = (sy == 0) ? (sx - 1) : sy * 5 - 1 + sx; /* 0..17 */
	  if(sx < -1 || sx > 3 || sy < 0 || sy > 3 || bito < 0 || bito > 17
	    || (sy == 0 && sx < 1)) {
	    /*
             * It is not one of the short sequences. We cannot compress this
             * series.
             */
	    compress = -1;
	    break;
	  }
	  compress |= 1 << bito;
	}
      }

      /*
       * Ignore target pixels which have no corresponding source points (empty
       * source pixel sequence).
       */
      if(*start_ptr == 0 && (*start_lutptr & INCTARGET)) {
	continue;
      }
#if BOUND_CHECK
      if(compress >= MAXHIST)
	prmsg(FATAL,("compress out of bounds\n"));
#endif /* BOUND_CHECK */
      if(compress >= 0)
	histo[compress]++;
    }

    /*
     * histo holds now for every possible sequence pattern the number of
     * occurrences of this pattern. Determine the total number of patterns that
     * are actually used.
     */

    count_hist = 0;
    for(i = 0; i < MAXHIST; i++)
      if(histo[i])
	count_hist++;

    /*
     * Allocate array histo2 and fill it with two pieces of information for
     * each sequence pattern found:
     * 1) the bitfield of the pattern;
     * 2) the number of occurrences for non 0 patterns.
     *
     * Then sort the array in descending order of the number of occurrences and
     * keep only the first 256 patterns (i.e., those 256 patterns that occur
     * most frequently).
     */
    if((histo2 = (unsigned int *)
	 pmalloc(count_hist * 2 * sizeof(int))) == NULL)
	prmsg(FATAL,("No memory"));

    for(j=0, i = 0; i < MAXHIST; i++)
      if(histo[i]) {
	histo2[j++] = i;
	histo2[j++] = histo[i];
      }

    qsort(histo2,count_hist,2 * sizeof(int),histcompare_count);

    /*
     * Throw away excess patterns (if there are more than 256), and readjust
     * the total count of the number of patterns.
     */
    if(count_hist > 256)
      if((histo2 = (unsigned int *)prealloc(histo2,sizeof(int) * 2 * 256)) ==
        NULL) {
	prmsg(FATAL,("Realloc failed \n"));
      }

    hist_max = count_hist > 256 ? 256 : count_hist;

    /*
     * Reuse the array histo2: the number of occurrences is no longer needed
     * and is replaced by the order number of the field. Remember that the
     * order in histo2 is the result of it being sorted in number of
     * occurrences, i.e., the sequence pattern with the highest number of
     * occurrences has order number 0, the one with the next highest number of
     * occurrences has order number 1, etc.
     *
     * histo2 thus contains now for each sequence pattern:
     * 1) the bitfield of the pattern;
     * 2) the order number of the pattern.
     */
    for(i=0; i<hist_max; i++)
      histo2[2*i+1] = i;

    /*
     * We build the relative transition tables here: offset_tab, rel_tab and
     * relend_tab.
     *
     * offset_tab contains the offsets for the 256 pixel positions around the
     * "present pixel", from the "upper left" pixel at offset (-8 - 8 * XSIZE)
     * to the "lower right" pixel at offset (7 + 7 * XSIZE).
     *
     * In other words, offset_tab[0] contains the pixel that is 8 rows to the
     * left and 8 lines above the present pixel. The subsequent elements then
     * move in this upper line to the right until offset_tab[15], which
     * contains the pixel that is 7 rows to the right and (still) 8 lines
     * above the present pixel. offset_tag[16] then moves one line further
     * down and back to the leftmost row: 8 rows to the left and 7 lines above
     * the present pixel. This continues like that, and offset_tab[255]
     * finally contains the pixel that is 7 rows to the right and 7 rows below
     * the present pixel.
     *
     * rel_tab contains sequences of predefined movements around the present
     * pixel. A sequence can be at most 18 movements long and can reach the 18
     * pixels that are mainly "below and to the right" of the present pixel.
     * More precisely, if the present pixel is marked by "x", then all the
     * pixels marked with a "." in the following drawing can be reached:
     *
     *               x . . .           (the three pixels to the right)
     *             . . . . .           (line below: 1 left to 3 right)
     *             . . . . .           (two lines below: 1 left to 3 right)
     *             . . . . .           (three lines below: 1 left to 3 right)
     *
     * The element 0 of each sequence contains the position of the next pixel
     * relative to the present pixel. All subsequent elements of the sequence
     * then contain the position of the next pixel relative to the previous
     * one.
     *
     * relend_tab contains the actual length for each sequence stored in
     * rel_tab in the following way:
     *
     *   for the sequence starting in rel_tab[RELTABSIZE * i]
     *
     *   relend_tab[i] = pointer to the end location of that sequence in
     *                   rel_tab
     */
    offset_tab =  (int *)pmalloc(sizeof(int) * 256);
    rel_tab =  (int *)pmalloc(sizeof(int *) * hist_max * RELTABSIZE);
    relend_tab = (int **)pmalloc(sizeof(int **) * hist_max);

    for(i=0; i<256; i++) {
      offset_tab[i] = ((i & 0x0f) - 8) + (((i & 0xf0) >> 4) - 8) * XSIZE;
    }

    for(i=0; i < hist_max; i++) {
      rel_ptr = rel_tab + RELTABSIZE * i;
      old = 0;
      len = 0;
      seq_code = histo2[2*i];
      for(j = 0; j< 18; j++)
	if(seq_code & (1<<j)) {
	  len++;
	  abs = (j < 3) ? j + 1 : (j - 3) % 5 - 1 + ((j - 3) / 5 + 1) * XSIZE;
	  *rel_ptr++ = abs - old;
	  old = abs;
	}
      relend_tab[i] = rel_tab + RELTABSIZE * i + len;
    }

    lut_d->relend_tab = relend_tab;
    lut_d->rel_tab = rel_tab;
    lut_d->offset_tab = offset_tab;

    qsort(histo2,hist_max,2 * sizeof(int),histcompare_idx);

    prmsg(MSG,("Sequence histogrammed \n"));

    /*
     * Later on we want to find for an arbitrary compress value if we should
     * compress this sequence and what is the code (from 0 to 254). To avoid to
     * loop through the histo2 array at every sequence (and therefore at every
     * pixel) we set histo up in a way to simply do
     *         seq_value = histo[compress];
     *
     * The "qsort" call sorts "histo2" in ascending bitfield order number. This
     * avoids having to use 1 MB of memory (1 << MAXHIST integers = 4 * 256 KB)
     * and makes the filling of the table "histo" faster.
     */
    memset (histo,0,MAXHIST  * sizeof(int));
    for(i = 0; i< hist_max; i++)
      histo[histo2[2*i]] = histo2[2*i+1]+1;
    /*
     * The second element of each "histo2" field holds the bitfield order
     * number --> histo holds for every bitfield "order number + 1" or 0 if
     * not set.
     */

    /*
     * Now use the above defined sequences of predefined movements to compress
     * the look-up-table program.
     *
     * Go through the program and see for every sequence of movements if it is
     * one of the predefined ones. If so, then replace the program instructions
     * for this sequence by a single one giving the corresponding index into
     * the rel_tab table ("compress the sequence").
     *
     * A new look-up-table program is thus built that will replace the old one.
     *
     * First allocate memory for the new look-up-table program.
     */

    if((new_prog = new_p = (unsigned char *)
      pmalloc(prog_length * sizeof(char))) == NULL)
      prmsg(FATAL,("No memory"));

    lut_ptr  = lut_tab;
    prog_end = lut_prog + prog_length;
    prmsg(DMSG,("Current prog length 1: %d\n",prog_end - lut_prog));
    for(i=0; i < 20; i++)
      stat_case[i] = 0;
    for(prog_ptr = lut_prog; prog_ptr != prog_end;) {
      /*
       * Here we are at the start of a target pixel sequence. Find the end of
       * the sequence, i.e. the beginning of the next target pixel (or the end
       * of the look-up-table program).
       */
      start_ptr = prog_ptr;
      start_lutptr = lut_ptr;
      lut_ptr++;
      prog_ptr++;
      stat_case[0]++;
      while(prog_ptr < prog_end && !(*lut_ptr & BITMASK)) {
	lut_ptr++;
	prog_ptr++;
	stat_case[1]++;
      }
      end_ptr = prog_ptr - 1;
      /*
       * start_ptr points to the start of the target pixel sequence, end_ptr to
       * the end.
       */

      compress = -1;
      if(start_ptr <= end_ptr) {
	oldidx = 0; sx = 0; sy = 0; compress = 0;

	/*
         * Run with the pointer c_ptr through the target pixel sequence to
         * evaluate the relative moves.
         */
	for(c_ptr = start_ptr + 1; c_ptr <= end_ptr; c_ptr++) {
	  stat_case[2]++;
	  instruct = *c_ptr;
	  sx += (instruct & 0x0f) - 8;
	  sy += (((int) (instruct & 0xf0)) >> 4) - 8;

	  /*
           * Find out which bit (0..17) to set for this pixel in the sequence
           * mask.
           */
	  bito = (sy == 0) ? (sx - 1) : sy * 5 - 1 + sx;
	  if(sx < -1 || sx > 3 || sy < 0 || sy > 3 || bito < 0 || bito > 17
             || (sy == 0 && sx < 1)) {
	    /*
             * We cannot compress this sequence, as it does not fit the allowed
             * patterns (see description above).
             */
	    compress = -1;
	    stat_case[3]++;
	    break;
	  }
#if BOUND_CHECK
          /*
           * Test if the new pixel to be added is already contained in the
           * sequence mask. If yes, give error message.
           */
	  newmask = 1 << bito;
	  if(compress & newmask)
	    prmsg(FATAL,("Source pixel referenced twice - Can't be\n"));
          /*
           * Test if the new pixel has a smaller array index for the source
           * image than the previous one. If yes, give error message.
           */
	  if(oldidx >= sx + sy * XSIZE)
	    prmsg(FATAL,(" Source not sorted - Can't be\n"));
	  oldidx = sx + sy * XSIZE;
#endif /* BOUND_CHECK */
	  /*
           * Build the sequence bitmask for all contributing source pixels.
           */
	  compress |= 1 << bito;
	}
      }

#if BOUND_CHECK
      if(new_p > new_prog + prog_length - 20) {
	prmsg(FATAL,("Compressed size is bigger than uncompressed\n"));
      }
#endif /* BOUND_CHECK */

      /*
       * If the look-up-table program instruction = 0 and the look-up-table
       * contains INCTARGET, then we have a target pixel without corresponding
       * source pixels.
       *
       * Set up an exception sequence in the look-up-table program:
       * - the first element of the sequence contains 0 to indicate an
       *   execption;
       * - the second element contains the type of exception, here INCTARGET.
       */
      if(*start_ptr == 0 && (*start_lutptr & INCTARGET)) {
	stat_case[4]++;

	/*
         * There should be only one element in the look-up-table program
         * -> no compress anyway.
         *
         * If there is more than one element, exit with error.
         */
#if BOUND_CHECK
	if(end_ptr != start_ptr)
	  prmsg(FATAL,("Table is corrupted somehow\n"));
#endif /* BOUND_CHECK */
	*new_p++ = 0;
	*new_p++ = INCTARGET;
	continue;
      }

      /*
       * This is a sequence that can possibly be compressed. Get the
       * corresponding index into the rel_tab table. If it is 0, then the
       * sequence corresponds to one of those that were not included in the 256
       * possibible entries of the rel_tab table - this should be a rare case.
       */
      if(compress >= 0) {
	seq_value = histo[compress];    /* get the index for this sequence */
	stat_case[5]++;
	if(seq_value == 0) {
	  compress = -1;
	  stat_case[6]++;
	} else {
	  seq_value--;                  /* restore "true" index */
#if BOUND_CHECK
	  if(seq_value >= hist_max || seq_value < 0)
	    prmsg(FATAL,("Seq value > 256 can't be\n"));
#endif /* BOUND_CHECK */
	}
      }

      if(compress >= 0) {
        /*
         * Now it is certain that this sequence can be compressed. There are
         * two possible cases:
         *
         * - the first pixel of the sequence is "near" the source pixel that
         *   was processed last (i.e., it is in one of the 256 positions
         *   around the last one).
         *
         *   Then the look-up-table program instruction is != 0 and contains
         *   the index into the offset table lut_d->offset_tab where the
         *   relative address of the first source image pixel can be found.
         *   This is the "normal" case;
         *
         * - the first pixel of the sequence is not "near" the last source
         *   pixel. Then the look-up-table program instruction is 0 to indicate
         *   an exception, the following instruction will contain ABSSRC to
         *   indicate the type of exception, and the address of the first
         *   source image pixel is in the table lut_d->abs_src.
         *
         * In both cases, the then next program instruction will contain the
         * index that points to the corresponding predefined sequence in the
         * lut_d->rel_tab table.
         */
	stat_case[7]++;
	*new_p++ = *start_ptr;
	if(*start_ptr == 0) {
	  *new_p++ = ABSSRC;
	  stat_case[8]++;
	}
	*new_p++ = seq_value & 0xff;
      } else {
        /*
         * This sequence cannot be compressed. There are the same two
         * possibilities for the address of the first pixel as in the
         * compressed case above:
         *
         * - it is near the last pixel processed. Then the present lookup-table
         *   instruction is != 0 and contains the index into the offset table.
         *
         *   This instruction will be replaced by a 0 to indicate an exception,
         *   followed by UNCOMPRESSED to indicate the type of exception, and
         *   then followed by the old "present" instruction (i.e., by the index
         *   into the offset table);
         *
         * - it is not "near" the last pixel. Then the look-up-table program
         *   instruction is 0 to indicate an exception, the following
         *   instruction will contain both the flags ABSSRC and UNCOMPRESSED to
         *   indicate the type of exception, and the address of the first
         *   source image pixel is in the absolute address table
         *   lut_d->abs_src.
         *
         * In both cases, the then following program instructions contain the
         * indices into the offset table lut_d->offset_tab where the relative
         * address of the following source image pixels can be found. This
         * sequence of source image pixels is ended by a program instruction
         * containing 0.
         */
	stat_case[9]++;
	if(*start_ptr == 0) {
	  stat_case[10]++;
	  *new_p++ = *start_ptr;
	  *new_p++ = ABSSRC | UNCOMPRESSED;
	} else {
	  stat_case[11]++;
	  *new_p++ = 0;
	  *new_p++ = UNCOMPRESSED;
	  *new_p++ = *start_ptr;
	}
	for(c_ptr = start_ptr + 1; c_ptr <= end_ptr; c_ptr++) {
	  stat_case[12]++;
	  *new_p++ = *c_ptr; /* Works OK for INCTARGET */
                             /*??? why does it have to work for INCTARGET ???*/
	}
	*new_p++ = 0; /* End uncompressed data with 0 */
      }
    }

    /*
     * Print statistical information on the compression:
     *  0: target pixels processed
     *  1: total length of the look-up-table program (before compression)
     *  2: total length of all source pixel sequences
     *  3: source sequences that cannot be compressed as they exceed the 18
     *     allowed positions
     *  4: target pixels that have no corresponding source pixels
     *  5: source sequences that could be compressed as they are contained in
     *     the 18 allowed positions
     *  6: source sequences that could be compressed, but are not among the
     *     256 most frequent sequences and are thus not compressed
     *  7: source sequences that will be compressed
     *  8: source sequences that will be compressed and where the first source
     *     pixel is not near the last source pixel (exception ABSSCR)
     *  9: source sequences that will not be compressed
     * 10: source sequences that will not be compressed and where the first
     *     source pixel is not near the last source pixel (exceptions ABSSCR
     *     and UNCOMPRESSED)
     * 11: source sequences that will not be compressed and where the first
     *     source pixel is near the last source pixel (exception UNCOMPRESSED)
     * 12: total length of all UNCOMPRESSED source pixel sequences
     */
    prmsg(DMSG,("Compress: %d %d %d %d %d %d %d %d\n   %d %d %d %d %d\n",
      stat_case[0],stat_case[1],stat_case[2],stat_case[3],stat_case[4],
      stat_case[5],stat_case[6],stat_case[7],stat_case[8],stat_case[9],
      stat_case[10],stat_case[11],stat_case[12]));

    /*
     * The look-up table entry corresponding to an "exception" instruction is
     * no longer needed and can be removed.
     *
     * Also, the BITMASK bit in the look-up table is no longer needed. The
     * value FULLSCALE had before been put as an artificial 0 in the table (to
     * avoid having a value creating the BITMASK meaning "start of a new
     * sequence").
     * Now the value FULLSCALE can be stored correctly in the look-up table.
     */
    prmsg(DMSG,("Current prog length 2: %d\n",prog_end - lut_prog));
    lut_ptr = newlut = lut_tab;
    for(prog_ptr = lut_prog; prog_ptr != prog_end; lut_ptr++, prog_ptr++) {
      if(*prog_ptr == 0)
	continue;
      v = *lut_ptr & MAPSCALE;
      *newlut++ = v ? v : FULLSCALE;
    }

    prmsg(DMSG,("Current prog length 3: %d\n",new_p - new_prog));
    /*
     * Copy the new compressed version of the look-up-table program over the
     * old version.
     *
     * Also, up to this point each target pixel without corresponding source
     * pixel has its own INCTARGET exception. If there are successive target
     * pixels without source pixel, these sequence of exceptions will now be
     * replaced by one single "multipixel" exception.
     *
     * Finally, add a PROGEND exception at the end of the program.
     */
    count_inc = 0;
    for(i=0; i < 20; i++)
      stat_case[i] = 0;
    for(prog_ptr = new_prog, new_inc = lut_prog; prog_ptr < new_p;) {
      if(*prog_ptr == 0 && *(prog_ptr + 1) & INCTARGET) {
        /*
         * Found a target advance instruction. Count the number of successive
         * occurrences, and skip the next two instructions (0 for exception and
         * INCTARGET for exception type).
         */
	stat_case[1]++;
	count_inc++;
	prog_ptr += 2;
      } else {
        /*
         * Found something else than a target advance instruction.
         *
         * If there is a started target advance sequence (count_int != 0), then
         * first terminate it properly before investigating the new instruction:
         *
         * - if it is a single pixel target advance, add a 0 instruction to the
         *   program for an exception and then add an instruction with the
         *   exception type INCTARGET;
         *
         * - if there is a target advance with several pixels, add a 0
         *   instruction to the program for an exception, add an instruction
         *   with the exception type flags INCTARGET and MULTIINC set, and then
         *   add the number of pixels concerned in the next two instructions
         *   (LSB first).
         */
	if(count_inc > 1) {
	  stat_case[2]++;
	  *new_inc++ = 0;
	  *new_inc++ = INCTARGET | MULTIINC;
	  *new_inc++ = count_inc % 256;
	  *new_inc++ = count_inc / 256;
	  count_inc = 0;
	} else if(count_inc == 1) {
	  stat_case[3]++;
	  *new_inc++ = 0;
	  *new_inc++ = INCTARGET;
	  count_inc = 0;
	}
        /*
         * Now investigate the new instruction.
         */
	if(*prog_ptr == 0 && (*(prog_ptr + 1) & UNCOMPRESSED)) {
          /*
           * Not a predefined source pixel sequence.
           *
           * Copy the present instruction (0 for exception), and all following
           * instructions up to and including the 0 that terminates the
           * uncompressed sequence.
           */
	  stat_case[4]++;
	  *new_inc++ = *prog_ptr++;
	  while (*new_inc++ = *prog_ptr++)
	    stat_case[5]++;;
	} else if(*prog_ptr == 0 && (*(prog_ptr + 1) & ABSSRC)) {
          /*
           * Predefined source pixel sequence and first pixel is not near the
           * last one processed.
           *
           * Just copy the next three instructions (0 for exception, ABSSRC for
           * exception type and the index for the absolute address table.
           */
	  stat_case[6]++;
	  *new_inc++ = *prog_ptr++;
	  *new_inc++ = *prog_ptr++;
	  *new_inc++ = *prog_ptr++;
	} else {
          /*
           * "Normal" case: predefined source pixel sequence and first pixel
           * near the last one processed.
           *
           * Just copy the next two instructions (offset index for first pixel
           * and sequence index).
           */
	  stat_case[7]++;
	  *new_inc++ = *prog_ptr++;
	  *new_inc++ = *prog_ptr++;
	}
      }
    }

    /*
     * End of the old program reached. Treat unfinished target advances if
     * necessary.
     */
    if(count_inc > 1) {
      stat_case[8]++;
      *new_inc++ = 0;
      *new_inc++ = INCTARGET | MULTIINC;
      *new_inc++ = count_inc % 256;
      *new_inc++ = count_inc / 256;
      count_inc = 0;
    } else if(count_inc == 1) {
      stat_case[9]++;
      *new_inc++ = 0;
      *new_inc++ = INCTARGET;
      count_inc = 0;
    }

    prmsg(DMSG,("Compress Stat: %d %d %d %d %d %d %d %d %d\n",stat_case[1],
      stat_case[2],stat_case[3],stat_case[4],stat_case[5],stat_case[6],
      stat_case[7],stat_case[8],stat_case[9]));
    /*
     * End of the compression.
     *
     * Mark the end of the look-up-table program by an exception of type
     * PROGEND.
     *
     * Re-allocate the tables for the look-up-table program and the
     * look-up-table (they are smaller now), then free the auxiliary buffers.
     */
    prog_length = new_inc - lut_prog + 2;
    prmsg(DMSG,("Current prog length 4: %d\n",prog_length));

    lut_prog[prog_length-2] = 0;
    lut_prog[prog_length-1] = PROGEND;

    if((prog_ptr = lut_prog = (unsigned char *)
      prealloc(lut_prog,sizeof(char) * prog_length)) == NULL) {
      prmsg(FATAL,("Realloc failed \n"));
    }

    lutsize = newlut - lut_tab;
    if((lut_ptr = lut_tab = (LUT_TYPE *)
      prealloc(lut_tab,sizeof(LUT_TYPE) * lutsize)) == NULL) {
      prmsg(FATAL,("Realloc failed \n"));
    }

    pfree(new_prog);
    pfree(histo);
    pfree(histo2);

  }

#if BOUND_CHECK
  /*
   * Make a dry run to check for source and target pixel boundaries as well as
   * inconsistencies in the look-up table or the look-up-table program.
   */
  {
    int offset;
    unsigned char *prog_ptr = lut_prog;
    LUT_TYPE *lut_ptr = lut_tab;
    int x,y,i,j;
    int *rel_ptr;
    int *offset_tab;
    int len,abs,old;
    register unsigned char compress;
    register int *rel_tab,**relend_tab;
    register int *r_ptr;
    register int cor_p;
    register int src_p;
    unsigned short *abs_ptr = abs_src;
    register unsigned long int add = 0;
    register unsigned int part;
    register unsigned char instruct;
    LUT_TYPE *lastlut = lut_tab + lutsize;
    unsigned char *lastprog = lut_prog + prog_length - 1;
    int count_inc = 0, count_exc = 0, count_abs = 0;
    int count_abs1 = 0, count_abs2 = 0, count_unc = 0, count_multi=0;
    int count_multi_t = 0;
    int multi;

    lut_ptr = lut_tab;

    offset_tab = lut_d->offset_tab;
    rel_tab = lut_d->rel_tab;
    relend_tab = lut_d->relend_tab;

    src_p = lut_d->startsidx;
    cor_p = lut_d->starttidx;

    for(;;) {
      instruct = *prog_ptr++;
      if(instruct != 0) {
        /*
         * Begin of a normal target pixel, i.e. one with a compressed source
         * pixel sequence.
         *
         * Test if the first source pixel is outside the image boundaries, or if
         * there are attempts to exceed the range of the look-up table and the
         * look-up-table program.
         */
	src_p += offset_tab[instruct];
	lut_ptr++;
	if(src_p > XSIZE * YSIZE || src_p <0)
	  prmsg(FATAL,("Table corrupted src out of bounds"));
	if(lut_ptr > lastlut)
	  prmsg(FATAL,("Table corrupted lut out of bounds"));
	if(prog_ptr > lastprog)
	  prmsg(FATAL,("Table corrupted prog out of bounds"));
#if BOUND_SUPER
	{
          /*
           * Test if the corrected target pixel coordinates agree with the ones
           * obtained from the target pixel index (within rounding errors).
           */
	  float x = x_trg[src_p % XSIZE + (src_p / XSIZE) * (XSIZE + 1)];
	  float y = y_trg[src_p % XSIZE + (src_p / XSIZE) * (XSIZE + 1)];

	  if(fabs(x - cor_p % XSIZE) > 1 || fabs(y - cor_p / XSIZE) > 1)
	    prmsg(MSG,("a.) %3d %3d %3d %3d %3d %3d %3f %3f\n",
	      src_p % XSIZE,src_p / XSIZE,cor_p % XSIZE,cor_p / XSIZE,
	      src_p % XSIZE - cor_p % XSIZE,src_p / XSIZE - cor_p / XSIZE,x,y));
	}
#endif /* BOUND_SUPER */
        /*
         * Go through the subsequent source pixels of this sequence and do the
         * same tests as for the first source pixel.
         */
	compress = *prog_ptr++;
	for(r_ptr = rel_tab + RELTABSIZE * compress;
	  r_ptr < relend_tab[compress]; r_ptr++) {
	  src_p += *r_ptr;
	  lut_ptr++;
	  if(src_p > XSIZE * YSIZE || src_p <0)
	    prmsg(FATAL,("Table corrupted src out of bounds"));
	  if(cor_p > XSIZE * YSIZE || cor_p <0)
	    prmsg(FATAL,("Table corrupted src out of bounds"));
	  if(lut_ptr > lastlut)
	    prmsg(FATAL,("Table corrupted lut out of bounds"));
	  if(prog_ptr > lastprog)
	    prmsg(FATAL,("Table corrupted prog out of bounds"));
	}
      } else {
        /*
         * We have to treat an exception.
         */
	count_exc++;
	if(prog_ptr > lastprog)
	  prmsg(FATAL,("Table corrupted prog out of bounds"));
	instruct = *prog_ptr++;

        /*
         * Exception of type "empty target pixel" (one or several).
         *
         * Test if the target pixels are within the image boundaries.
         */
	if(instruct & INCTARGET) {
	  if(instruct & MULTIINC) {
	    count_multi++;
	    multi  = *prog_ptr++;
	    multi += *prog_ptr++ * 256;
	    count_multi_t +=multi;
	    cor_p += multi;
	  } else {
	    count_inc++;
	    cor_p++;
	  }
	  if(cor_p > XSIZE * YSIZE || cor_p <0)
	    prmsg(FATAL,("Table corrupted src out of bounds"));
	  continue;
	}

        /*
         * Exception of type "absolute source pixel address" (for compressed or
         * uncompressed pixel sequence).
         *
         * Test if the source pixels are within the image boundaries, or if
         * there are attempts to exceed the range of the look-up table.
         */
	if(instruct & ABSSRC) {
	  int x = *abs_ptr++;
	  int y = *abs_ptr++;
	  count_abs++;
	  src_p = (x + y * XSIZE);
	  *abs_ptr++;
	  if(src_p > XSIZE * YSIZE || src_p <0)
	    prmsg(FATAL,("Table corrupted src out of bounds"));

	  if(instruct & UNCOMPRESSED) {
	    count_abs1++;
	    while(instruct = *prog_ptr++) {
	      src_p += offset_tab[instruct];
	      lut_ptr++;
	      if(src_p > XSIZE * YSIZE || src_p <0)
		prmsg(FATAL,("Table corrupted src out of bounds"));
	      if(lut_ptr > lastlut)
		prmsg(FATAL,("Table corrupted lut out of bounds"));
	    }
	  } else {
	    count_abs2++;
	    compress = *prog_ptr++;
	    for(r_ptr = rel_tab + RELTABSIZE * compress;
	      r_ptr < relend_tab[compress]; r_ptr++) {
	      src_p += *r_ptr;
	      lut_ptr++;
	      if(src_p > XSIZE * YSIZE || src_p <0)
		prmsg(FATAL,("Table corrupted src out of bounds"));
	      if(lut_ptr > lastlut)
		prmsg(FATAL,("Table corrupted lut out of bounds"));
	    }
	  }
	} else if(instruct & UNCOMPRESSED) {
        /*
         * Exception of type "uncompressed source pixel sequence" (with
         * relative pixel addressing).
         *
         * Test if the source pixels are within the image boundaries, or if
         * there are attempts to exceed the range of the look-up table.
         */

	  count_unc++;
	  while(instruct = *prog_ptr++) {
	    src_p += offset_tab[instruct];
	    lut_ptr++;
	    if(src_p > XSIZE * YSIZE || src_p <0)
	      prmsg(FATAL,("Table corrupted src out of bounds"));
	    if(lut_ptr > lastlut)
	      prmsg(FATAL,("Table corrupted lut out of bounds"));
	  }
	} else if(instruct & PROGEND)
	  break;
      }
      cor_p++;

      /*
       * The target pixel index must not exceed the image boundaries.
       */
      if(cor_p > XSIZE * YSIZE || cor_p <0)
	prmsg(FATAL,("Table corrupted cor_p out of bounds"));
    }

    /*
     * At the end, the target pixel index should point to the last pixel in the
     * image.
     */
    if(cor_p != XSIZE * YSIZE)
      prmsg(FATAL,("Missed some trg pixels, %d\n",cor_p));

    /*
     * There must be three entries for each pixel that is in the source
     * coordinate array (but not all pixels need to be in this array!).
     */
    if(count_abs != noabs/3)
      prmsg(FATAL,("Missed some abs coord, %d != %d\n",count_abs,noabs/3));

    /*
     * Print statistics on the exceptions.
     */
    prmsg(MSG,("Exceptions: %d (%d absolute source (%d uncompr., %d compr.),\n",
      count_exc,count_abs,count_abs1,count_abs2));
    prmsg(MSG,("            %d inc only, %d multi inc (->%d), %d uncompr.)\n",
      count_inc,count_multi,count_multi_t,count_unc));

  }
#endif /* BOUND_CHECK */

  //  print_memsize();

  lut_d->prog_length = prog_length;
  lut_d->prog = lut_prog;
  lut_d->lut = lut_tab;

#if BOUND_SUPER
  pfree(x_trg);
  pfree(y_trg);
#endif /* BOUND_SUPER */

  return(lut_d);
}

/*
 * Prints for debugging purposes various pieces of information from the look-up
 * table, the look-up table program and the source coordinate array.
 *
 * Printed are, for a selectable section of the look-up-table program, the
 * following items:
 * - the program instruction and the content of the look-up-table;
 * - the target pixel coordinates;
 * - the coordinates of the contributing source pixels and their area
 *   contributions to the target pixel.
 *
 * The section to be printed is selected by the input variables "start" and
 * "end".
 *
 * This routine is at present not called from within the program, but can be
 * called from the debugger.
 *
 * Note that this routine expects an "intermediate state" of the look-up table
 * and associated program, i.e. the state when the exceptions and the beginning
 * of a new target pixel sequence are still marked in the look-up table and not
 * in the look-up-table program.
 *
 * Input : prog:        the look-up-table program
 *         lut:         the look-up table
 *         asrc:        the source coordinate array
 *         prog_length: the length of the look-up-table program
 *         startsidx:   the pixel index of the first source pixel
 *         starttidx:   the pixel index of the first target pixel
 *         start:       the sequence number (= array index) of the lookup-table
 *                      program instruction where the printing is to start
 *         end:         the corresponding number for the end of printing
 * Output: none
 * Return: none
 */

debug_print(unsigned char *prog,LUT_TYPE *lut,unsigned short *asrc,
  int prog_length,int startsidx,int starttidx,int start,int end)
{
  LUT_TYPE *lut_ptr = lut;
  unsigned char* prog_ptr = prog;
  int i;
  unsigned char instruct;
  int sx,sy,x,y;
  int trg_x,trg_y;
  int src_x, src_y;
  int tidx,part;
  unsigned short *abs_ptr = asrc;

  tidx = starttidx;
  src_x = startsidx % XSIZE;
  src_y = startsidx / XSIZE;

  for(i = 0; i < prog_length; i++, prog_ptr++, lut_ptr++) {
    /*
     * Print look-up-table program instruction and look-up table content.
     */
    if(i >= start && i <= end)
      prmsg(MSG,("0x%2x 0x%4x :",*prog_ptr,*lut_ptr));
    if((part = *lut_ptr) & BITMASK) {
      tidx++;
      if(*prog_ptr == 0 && (*lut_ptr & ABSSRC)) {
	src_x = *abs_ptr++;
	src_y = *abs_ptr++;
	instruct = 0x88;
	part  = *abs_ptr++;
      }
      trg_x = tidx % XSIZE;
      trg_y = tidx / XSIZE;
      /*
       * Print target pixel coordinates.
       */
      if(i >= start && i <= end)
	prmsg(MSG,("%4d %4d :",trg_x,trg_y));

      if(*prog_ptr == 0 && *lut_ptr & INCTARGET) {
        /*
         * Print source pixel coordinates for an empty target.
         * ??? they can not be very meaningful. Is this only to make sure that
         * they did not get messed up ???
         */
	part = 0;
	if(i >= start && i <= end)
	  prmsg(MSG,("%4d %4d 0x%4x\n",src_x,src_y,part));
	continue;
      }

      part &= MAPSCALE;
    } else
      if(i >= start && i <= end)
	prmsg(MSG,("           "));

    instruct = *prog_ptr;
    src_x += (instruct & 0x0f) - 8;
    src_y += (((int) (instruct & 0xf0)) >> 4) - 8;
    /*
     * Print source pixel coordinates and contributing area fraction.
     */
    if(i >= start && i <= end) {
      if(part == 0)
	part = FULLSCALE;
      prmsg(MSG,("%4d %4d 0x%4x\n",src_x,src_y,part));
    }
  }
}

#if 0
/*
 * Tests the spline function parameters.
 *
 * The parameters are read in from the spline function parameter file, and the
 * spline function is evaluated at one arbitrary coordinate point. The
 * coordinates of this point and the calculated spline function values for x
 * and y are then printed.
 *
 * Note that this function is currently not called from within this package.
 *
 * Input : filename: name of the file that contains the spline function
 *                   parameters
 * Output: none
 * Return: none
 */

gtest(char *filename)
{
  /*
   * Test for an arbitrary point (358,230).
   */
  float xin=358, yin=230, xout, yout;
  spline = loadspdfile(filename);
  pxcorrgrid(spline,1,1,&xin,&yin,&xout,&yout);
  printf("%f %f %f %f\n",xin,yin,xout,yout);
}

#endif

/*
 * Prepare the floodfield (also called flatfield) image for the floodfield
 * correction.
 *
 * The floodfield correction takes into account the fact that a sample with
 * absolutely uniform scattering response does not necessarily produce a flat
 * image (e.g. because of a non-uniform detector response). If one takes an
 * image of a real sample, one has to correct for this non-uniformity.
 *
 * This is done by taking an image from an uniformly scattering probe (the
 * floodfield image) and then dividing the real image by the floodfield image.
 *
 * To make the actual correction calculation in divide_insito_im() and
 * divide_im() faster, the floodfield image calculated here is inverted. Thus
 * the correction routines can multiply with instead of divide by the floodfield
 * image.
 *
 * The input floodfield image can be either of type "unsigned short" or of
 * type "float". If it is of type "unsigned short", then it is handed over
 * in the argument "in_im". If the input floodfield image is of type "float",
 * then "in_im" must be a NULL pointer, and the input floodfield image is handed
 * over in "out_im".
 *
 * The output floodfield image is always of type "float" and returned in the
 * argument "out_im".
 *
 * Invalid pixels in the floodfield image (those with pixel value == 0. or pixel
 * value == dummy) will be stored in a list and dealt with later when the actual
 * division by the floodfield image is made. "dummy" is either the value of the
 * "Dummy" keyword in the floodfield header, if this is defined, or the value of
 * the "dummy" command line argument. If the latter is not defined by the user,
 * its default value is 0.
 *
 * Input : in_im:  array with the input floodfield image (unsigned short)
 *         out_im: array with the input floodfield image (float)
 * Output: out_im: array with the inverted floodfield image (float)
 * Return: none
 */

int prepare_flood(unsigned short *in_im,float *out_im)
{
  float *float_flood;
  unsigned short *short_flood;
  long *pfldum;
  int dummycnt = 0,i;
  float dummy,ddummy;

  /*
   * Define the "dummy" and "ddummy" values.
   */
  if(img_head[FLOTYP].init & FL_DUMMY)
    dummy = img_head[FLOTYP].Dummy;
  else
    dummy = 0.;
  if(img_head[FLOTYP].init & FL_DDUMM)
    ddummy = img_head[FLOTYP].DDummy;
  else
    ddummy = DDSET(dummy);

  /*
   * If the image is of type "unsigned short", copy it from the "in_im" buffer
   * to the (float-type) "out_im" buffer.
   */
  if(in_im != NULL) {
    short_flood = in_im + XSIZE * YSIZE - 1;
    float_flood = out_im + XSIZE * YSIZE - 1;
    for(; float_flood >= out_im; short_flood--, float_flood--)
      *float_flood = (float)*short_flood;
  }

  /*
   * Invert the image on a pixel-to-pixel basis.
   *
   * Count the pixels with invalid values (0. or "dummy"). Values of 0. are
   * replaced by "dummy".
   */
  float_flood = out_im + XSIZE * YSIZE - 1;
  for(; float_flood >= out_im; float_flood--)
    if(*float_flood == 0. || DUMMY(*float_flood,dummy,ddummy))
    {
      *float_flood = dummy;
      dummycnt++;
    }
    else
      *float_flood = 1. / *float_flood;
 
  if(fldumlst != NULL)
  {
    pfree(fldumlst);
    fldumlst = NULL;
  }

  if(dummycnt == 0)
    return(0);

  /*
   * Make the list with the pixel indices of the invalid pixels.
   */
  fldumlst = (long *)pmalloc((dummycnt + 1) * sizeof(long));
  pfldum = fldumlst;
  for(i = 0; i < XSIZE * YSIZE; i++)
    if(*(out_im + i) == dummy)
      *pfldum++ = i;
  *pfldum = -1;

  return(0);
}

/*
 * Prints for debugging purposes various pieces of information from the look-up
 * table and the look-up table program.
 *
 * Printed are the following items:
 * - if the look-up-table program instruction is 0, but it is not at the
 *   beggining of a new target pixel: the contents of the look-up-table program
 *   and of the look-up table as well as the present address in the look-up-
 *   table program;
 * - if the look-up table contains 0xff: the content of the look-up-table
 *   program.
 *
 * The routine also counts
 * - the number of INCTARGET exceptions;
 * - the number of ABSSRC exceptions;
 * - the total number of exceptions (i.e., when program instruction == 0);
 * - the number of occurrences of 0xff in the look-up table.
 *   ??? what does this mean ???
 *
 * This routine is at present not called from within the program, but can be
 * called from the debugger.
 *
 * Note that this routine expects an "intermediate state" of the look-up table
 * and associated program, i.e. the state when the exceptions and the beginning
 * of a new target pixel sequence are still marked in the look-up table and not
 * in the look-up-table program.
 *
 * Input : prog:        the start address in the look-up-table program
 *         lut:         the start address in the look-up table
 *         prog_length: the length of the look-up-table program
 * Output: none
 * Return: none
 */

despair(unsigned char *lut_prog,LUT_TYPE *lut_tab,int prog_length)
{
  int count_abs=0, count_ff=0, count_null =0, count_inc =0;
  unsigned char *prog_end;
  unsigned char *prog_ptr;
  LUT_TYPE *lut_ptr;
  prog_end = lut_prog + prog_length;
  lut_ptr = lut_tab;
  for(prog_ptr = lut_prog; prog_ptr != prog_end; prog_ptr++, lut_ptr++) {
    if(*prog_ptr == 0 && !(*lut_ptr & BITMASK))
      prmsg(MSG,("%d %d %d \n",*prog_ptr,*lut_ptr,prog_ptr-lut_prog));
    if(*prog_ptr == 0 && (*lut_ptr & INCTARGET))
      count_inc++;
    if(*prog_ptr == 0 && (*lut_ptr & ABSSRC))
      count_abs++;
    /* ??? what does 0xff in the look-up table mean ??? */
    if(*lut_ptr == 0xff) {
      prmsg(MSG,("<0x%x>",*prog_ptr));
      count_ff++;
    }
    if(*prog_ptr == 0)
      count_null++;
  }
  prmsg(MSG,("\nInc Abs Null LUT==FF %d %d %d %d\n",count_inc,count_abs,
    count_null,count_ff));
}
