/*+++
1 saxs_waxs
  Projection of the Ewald sphere that is visible by an inclined detector 
  on a flat s-plane. The length of the radial coordinate corresponds to the 
  length of the scattering vector s. The azimuthal angle corresponds to the 
  azimuth of s. 

  The scattering vectors are calculated with saxs-coordinates (-rsys saxs).

2 PURPOSE

  Arguments:
  saxs_waxs [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <rot1[0.0]> <rot2[0.0]> <rot3[0.0]>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>

  All rotations and directions are defined for orientation 1

  <rot1>              : <ccw in degree rotation around direction 1> (default 0)
  <rot2>              : <ccw in degree rotation around direction 2> (default 0) 
  <rot3>              : <ccw in degree rotation around direction 3> (default 0)

2 HISTORY
  2001-04-16  Peter Boesecke from saxs_refract V3.1 
  2001-04-19  PB V3.0
  2001-04-19  PB V3.1 K taken from input image ihb[1]
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_waxs V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "waxs.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> \n\
                <rot1[0.0]> <rot2[0.0]> <rot3[0.0]>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SRot1 "rot1"   /* float */
# define NRot1 0
# define SRot2 "rot2"   /* float */
# define NRot2 1
# define SRot3 "rot3"   /* float */
# define NRot3 2

/*---------------------------------------------------------------------------
1 saxs_waxs

2 PURPOSE
  Projection of an inclined detector surface
---------------------------------------------------------------------------*/
void saxs_waxs (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  int   i, imax; 
  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int i_1, i_2;

  float f1_1, f1_2;
  float W1_1, W1_2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  WaxsCoord W0, W1;
  double K, Rot1, Rot2, Rot3;                                                       
  float tmp;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* check the reference system: Saxs required */
  if (pcb->RSys.V != IO_Saxs) {
    printf("ERROR: Reference system %s required\n",reftostr(IO_Saxs));
    printf("       Start program with option -rsys %s\n",reftostr(IO_Saxs)); 
    }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;                                             

  K    = (double) WAVENUMBER(ihb[1].WaveLength.V);
  Rot1 = (double) pcb->Arg[NRot1].V * deg2rad;
  Rot2 = (double) pcb->Arg[NRot2].V * deg2rad;
  Rot3 = (double) pcb->Arg[NRot3].V * deg2rad;    
  
  waxs_Init ( K, Rot1, Rot2, Rot3 );

  if (pcb->TestBit) {
    waxs_PrintParams ( stdout ); 
    printf("\n"); }                                                             

  /* loop over the output array  */
  for (i_2=0;i_2<I0Dim_2;i_2++) {
    W0.s_2 =  (double) WORLD( i_2, Off_2[0], Ps_2[0] );
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
    for (i_1=0;i_1<I0Dim_1;i_1++) {
      W0.s_1 =  (double) WORLD( i_1, Off_1[0], Ps_1[0] ); 

      W1 = waxs_Saxs ( W0 );                                                                                                                
      if (!W1.status) { 

        f1_1 = INDEX(W1.s_1, Off_1[1], Ps_1[1]);
        f1_2 = INDEX(W1.s_2, Off_2[1], Ps_2[1]);

        if (Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
          f1_1, f1_2, &Value)) {
          /* then do something with the data */

          UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

          } /* if (Ipol2d ... */
        } /* if (!W1.status) */

      pI0Data++;
      } /* for i_1 ... */
    } /* for i_2 ... */

 } /* saxs_waxs*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*---             rotation angles of detector */
  printf("Detector Orientation\n");
  printf("The angles rot1, rot2 and rot3 define the orientation of the detector\n");
  printf("in the lab coordinate system with orientation 1. They are applied\n");
  printf("sequentially. The coordinate system with orientation 1 is defined\n");
  printf("in the following way:\n");
  printf("  origin (0,0) at lower left corner of the image\n");
  printf("  axis 1: horizontally to the right\n");
  printf("  axis 2: vertically up\n");
  printf("  axis 3: against beam direction\n");
  printf("\n");
  printf("             ^ 2 (vertical)\n");
  printf("             |\n");
  printf("             |\n");
  printf("             |\n");
  printf("             |\n");
  printf("             +------> 1 (horizontal)\n");
  printf("             \\\n"); 
  printf("              \\\n");
  printf("               \\\n");
  printf("                _| 3 (against primary beam)\n");
  printf("\n");
  printf("ID01:  -x <=> axis 1,  y <=> axis 2, -z <=> axis 3\n");            
  printf("ID02:   x <=> axis 1, -y <=> axis 3,  z <=> axis 2\n");
  printf("\n");                                                                 

  /*--- Argument  : rotation around axis 1 */
  argv_float(pcb," first rotation: ccw around axis 1 (deg)",
             &pcb->Arg[NRot1],pcb->Arg[NRot1].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : rotation around axis 2 */
  argv_float(pcb,"second rotation: ccw around axis 2 (deg)",
             &pcb->Arg[NRot2],pcb->Arg[NRot2].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : rotation around axis 3 */
  argv_float(pcb," third rotation: ccw around axis 3 (deg)",
             &pcb->Arg[NRot3],pcb->Arg[NRot3].V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
if (pcb->Arg[NRot1].I) printf("rotation 1         : %g deg\n",
  pcb->Arg[NRot1].V);
if (pcb->Arg[NRot2].I) printf("rotation 2         : %g deg\n",
  pcb->Arg[NRot2].V);
if (pcb->Arg[NRot3].I) printf("rotation 3         : %g deg\n",
  pcb->Arg[NRot3].V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  cb.RSys.V = IO_Saxs; /* default */

 /* Define special options */
  DefFloatOption( &cb, SRot1, NRot1);
  cb.Arg[NRot1].V = 0.0; /* default */
  DefFloatOption( &cb, SRot2, NRot2 );
  cb.Arg[NRot2].V = 0.0;   /* default */
  DefFloatOption( &cb, SRot3, NRot3 );
  cb.Arg[NRot3].V = 0.0;  /* default */

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_waxs, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

