/*+++
1 saxs_urol 
  Do a cutting of an aera of the whole image and plot it on a plan

        |intensity
        |
        |
        /------ alpha
       /
      /radius

2 PURPOSE

  Defaults:
  <input file name>        : input.edf 
  <ouptut file name>       : output.edf
  <first image>            : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>              : 1
  <pixel size 1>           : <psize1 in input file> if not found 1.0
  <pixel size 2>           : <psize2 in input file> if not found 1.0
  <center 1>               : <center1 in input file> if not found dim1/2
  <center 2>               : <center2 in input file> if not found dim2/2
  <radius min>             : 0.0
  <radius step>            : 1.0 
  <number of raduis step>  : 100
  <alpha min>              : 0
  <alpha step>             : 1 degre
  <number of alpha step>   : 360

  Specific arguments:
  -rmin
  -rstep
  -nrstep
  -tmin
  -tstep
  -ntstep 
 
  Specific options:
  none

  Wildcard:
  A "=" in the argument list chooses the default.

3 HISTORY
  04-May-1995 PJE creation from saxs_cut v1.0 
  18-Aug-1995 PB correction of summation error in saxs_urol
	      This program does not behave according to standard saxs
	      programs. The input array is not interpolated and therefore
	      the use of some reference systems is not possible.
	      This must be corrected as soon as possible.
	      See "quark!!!!"
  21-Sep-1995 PB M_PI -> SAXS_PI
  05-Dec-1995 PB V3
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  2000-08-04  PB %f->%g
  2001-07-08  PB V3.01 FFirst
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_urol 4.00 2001-07-09, PB/Pierre Jourde (ESRF/EDS)"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <math.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                  <i/p file> <o/p file> <first> <last> <inc> \n\
                  <psize1> <psize2> <center1> <center2> <Rmin> \n\
                  <DeltaR> <NR> <Alphamin> <DeltaAlpha> <NAlpha> <o/p file>"

# define Default_extension DefaultExtension
# define Default_ipfn      DefaultInput 
# define Default_opfn      DefaultOutput 
# define Default_inc       1
# define Default_psize1     1.0
# define Default_psize2     1.0
# define Default_rmin       0.0 
# define Default_nrstep     100 
# define Default_tmin       0.0 
# define Default_tstep      1.0
# define Default_ntstep     360 

# define SRmin    "rmin"
# define NRmin    1
# define SRstep   "rstep"
# define NRstep   2
# define SNRstep  "nrstep"
# define NNRstep  1
# define STmin    "tmin"
# define NTmin    3
# define STstep   "tstep"
# define NTstep   4
# define SNTstep  "ntstep"
# define NNTstep  2

# define BlockNum 2          /* 1 input + 1 output */

# ifndef TRUE
#   define TRUE 1
# endif
# ifndef FALSE
#   define FALSE 0
# endif

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io( CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  long  ib1image_0;
  long  ib1image_1;
  float ib0dummy;
  float ib0ddummy;
  long  ib0dim1;
  long  ib0dim2;
  float ib1center1, ib1center2;
  float ib1psize1, ib1psize2;
  float ib0center1, ib0center2;
  float ib0off1, ib0off2;
  float ib0psize1, ib0psize2;

  ImgHeadBlk ihb[BlockNum];
  int prompt;
  int opt_no;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  /*--- OpenMode of input file */
  if (!ib[1].OpenMode.I) {
    ib[1].OpenMode.V = IO_Old | IO_FileProtect;
    ib[1].OpenMode.I = TRUE; 
    } 

  /*--- Argument  : input file name */
  argv_filename(pcb,"Input file name",ib,1,Default_ipfn,pstatus);

  /* Exit in all cases after an abort from the first argument */
  if (*pstatus!=Success) return;

  OpenImageFile( pcb,ib,1,ib[1].Name.V,ib[1].FFirst.V,
                 IO_Old|IO_FileProtect,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : output file name */
  argv_filename(pcb,"Output file name",ib,0,Default_opfn,pstatus);
  if (*pstatus!=Success) return;

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect;
    ib[0].OpenMode.I = TRUE;
    }

  /* Search for minimum and maximum image number in 1st input sequence */
  (void) SearchMinMaxImage ( pcb, ib, 1, &ib1image_0, &ib1image_1, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : number of first image */
  argv_long(pcb,"First image",&ib[1].First,ib1image_0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : number of last image */
  argv_long(pcb,"Last image",&ib[1].Last,ib1image_1,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : increment */
  argv_long(pcb,"Increment",&ib[1].Inc,Default_inc,pstatus);
  if (*pstatus!=Success) return;

  /*--- Search here for the following informations --- */
  ib0dummy   = -5555.0;  /* dummy of the first chosen input image */
  ib1psize1  = Default_psize1; /*pixel size 1 of the first chosen input image */
  ib1psize2  = Default_psize2; /*pixel size 2 of the first chosen input image */

  (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
  if (*pstatus!=Success) return;
  if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

  CloseImageFile( pcb, ib, 1, pstatus) ;
  if (*pstatus!=Success) return;

  /* Defaults for output file */
  if (!ib[0].Dummy.I) ib[0].Dummy.V  = ihb[1].Dummy.V;
  if (!ib[0].Dim[1].I) ib[0].Dim[1].V = ihb[1].Dim[1];
  if (!ib[0].Dim[2].I) ib[0].Dim[2].V = ihb[1].Dim[2];

  ib1center1 = ihb[1].Dim[1]/2.0; /* center1 of the input image */
  ib1center2 = ihb[1].Dim[2]/2.0; /* center2 of the input image */

  if (ihb[1].Dummy.I)  ib0dummy = ihb[1].Dummy.V;
  if (ihb[1].Center[1].I) ib1center1 = ihb[1].Center[1].V;
  if (ihb[1].Center[2].I) ib1center2 = ihb[1].Center[2].V;
  if (ihb[1].PixSiz[1].I) ib1psize1 = ihb[1].PixSiz[1].V;
  if (ihb[1].PixSiz[2].I) ib1psize2 = ihb[1].PixSiz[2].V;

  /*--- Argument : center 1 */
  argv_float(pcb,"Center 1",&ib[1].Center[1],ib1center1,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : center 2 */
  argv_float(pcb,"Center 2",&ib[1].Center[2],ib1center2,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument :  pixel size 1 */
  argv_float(pcb,"Pixel size 1",&ib[1].PixSiz[1],ib1psize1,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument :  pixel size 2 */
  argv_float(pcb,"Pixel size 2",&ib[1].PixSiz[2],ib1psize2,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : radius min */
  argv_float(pcb,"Radius min",&pcb->Arg[NRmin],Default_rmin,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : radius step */
  argv_float(pcb,"Radius step",&pcb->Arg[NRstep],
      (ib[0].PixSiz[1].V+ib[0].PixSiz[2].V)/2,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : number of radius step */
  argv_long(pcb,"Radius step number",&pcb->Num[NNRstep],Default_nrstep,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : alpha min */
  argv_float(pcb,"Alpha min",&pcb->Arg[NTmin],Default_tmin,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : alpha step */
  argv_float(pcb,"Alpha step",&pcb->Arg[NTstep],Default_tstep,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : number of alpha step */
  argv_long(pcb,"Alpha step number",&pcb->Num[NNTstep],Default_ntstep,pstatus);
  if (*pstatus!=Success) return;
  
  /*--- Calculate here for the following informations --- */
  ib0dim1 = pcb->Num[NNRstep].V ; /* dimension 1 of the output image */
  ib0dim2 = pcb->Num[NNTstep].V ; /* dimension 2 of the output image */
  ib0center1 = 0.0; /* center1 of the output image */
  ib0center2 = 0.0; /* center2 of the output image */
  ib0psize1 = pcb->Arg[NRstep].V; /* pixel size 1 of the output image */
  ib0psize2 = pcb->Arg[NTstep].V; /* pixel size 2 of the output image */
  ib0off1 = pcb->Arg[NRmin].V/(float)pcb->Num[NNRstep].V-1.0; 
        /* off1 of the output image */
  ib0off2 = pcb->Arg[NTmin].V/(float)pcb->Num[NNTstep].V-1.0; 
        /* off2 of the output image */

  /*--- Copy of these values in the image header for output image */
  if (!(ib[0].Dim[1].I)) { ib[0].Dim[1].I = TRUE; ib[0].Dim[1].V = ib0dim1; }
  if (!(ib[0].Dim[2].I)) { ib[0].Dim[2].I = TRUE; ib[0].Dim[2].V = ib0dim2; }
  if (!(ib[0].Center[1].I)) { 
     ib[0].Center[1].I = TRUE; ib[0].Center[1].V = ib0center1; }
  if (!(ib[0].Center[2].I)) { 
     ib[0].Center[2].I = TRUE; ib[0].Center[2].V = ib0center2; }
  if (!(ib[0].PixSiz[1].I)) {
     ib[0].PixSiz[1].I= TRUE; ib[0].PixSiz[1].V = ib0psize1;}
  if (!(ib[0].PixSiz[2].I)) {
     ib[0].PixSiz[2].I= TRUE; ib[0].PixSiz[2].V = ib0psize2;}
  if (!(ib[0].Offset[1].I)) {
     ib[0].Offset[1].I = TRUE; ib[0].Offset[1].V = ib0off1;}
  if (!(ib[0].Offset[2].I)) {
     ib[0].Offset[2].I = TRUE; ib[0].Offset[2].V = ib0off2;}

  printf("\n");
  printf("i/p file           : %s\n",ib[1].Name.V);
  printf("first image        : %d\n",ib[1].First.V);
  printf("last image         : %d\n",ib[1].Last.V);
  printf("increment          : %d\n",ib[1].Inc.V);
  printf("pixel size 1       : %g\n",ib[1].PixSiz[1].V);
  printf("pixel size 2       : %g\n",ib[1].PixSiz[2].V);
  printf("center 1           : %g\n",ib[1].Center[1].V);
  printf("center 2           : %g\n",ib[1].Center[2].V);
  printf("radius min         : %g\n",pcb->Arg[NRmin].V);
  printf("radius step        : %g\n",pcb->Arg[NRstep].V);
  printf("radius step number : %d\n",pcb->Num[NNRstep].V);
  printf("alpha min          : %g\n",pcb->Arg[NTmin].V);
  printf("alpha step         : %g\n",pcb->Arg[NTstep].V);
  printf("alpha step number  : %d\n",pcb->Num[NNTstep].V);
  printf("o/p file           : %s\n",ib[0].Name.V);
  printf("dim 1              : %d\n",ib[0].Dim[1].V);
  printf("dim 2              : %d\n",ib[0].Dim[2].V);
  printf("offset 1           : %g\n",ib[0].Offset[1].V);
  printf("offset 2           : %g\n",ib[0].Offset[2].V);
  printf("pixel size 1       : %g\n",ib[0].PixSiz[1].V);
  printf("pixel size 2       : %g\n",ib[0].PixSiz[2].V);
  printf("center 1           : %g\n",ib[0].Center[1].V);
  printf("center 2           : %g\n",ib[0].Center[2].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

} /* user_io */

/*---------------------------------------------------------------------------
1 Ipol2dT

2 PURPOSE
  Linear two dimensional interpolation including dummies
  No limit checks

2 METHOD
  The point (f_1, f_2) is linearly interpolated between the 4 closest
  program array elements val1=Data[i_1,i_2], val2=Data[i_1+1,I_2],
  val3=Data[i_1+1,i_2+1] and val4=Data[I_1,i_2+1]. If a float index
  is an integer number 0, 1, 2, 3 etc. no interpolation is done in the
  corresponding direction.

2 CALL
  int Ipol2dT (float *Data, int Dim_1, int Dim_2, float Dummy, float DDummy,
              float *value)

  return value            (o)   : number of pixels contributing to the output
                                  value (0..4), if 0, only dummies found.
  float Data[Dim_1,Dim_2] (i)   : input array
  int Dim_1, Dim_2        (i)   : dimension of array
  float Dummy, DDummy     (i)   : dummy specification
  float f_1, f_2          (i)   : program array indices (interpolation point)
  float *value            (o)   : interpolated value

---------------------------------------------------------------------------*/
int Ipol2dT (float *Data, int Dim_1, int Dim_2, float Dummy, float DDummy,
            float f_1, float f_2, float *value)
{ const float eps = 1e-5;
  int cnt;
  float *pval1, *pval2, *pval3, *pval4;
  float w1, w2, w3, w4;
  float weight;
  float val1, val2,val3,val4;

  int   i_1, i_2;
  float r_1, r_2;

  /* calculate integer indices and rest */
  IDX(f_1,i_1,r_1);
  IDX(f_2,i_2,r_2);

  /* printf("Ipol2dT : f_1= %g, f_2= %g, i_1= %d, i_2= %d\n",f_1, f_2, i_1, i_2); */

  /* test on the indices */
  /* for point 1 */
  if ((i_1<Dim_1) && (i_2<Dim_2) && (i_1>-1) && (i_2>-1)) {
    /* point 1 is in */ 
    pval1 = ABSPTR(Data,Dim_1,Dim_2,i_1,i_2);
    val1 = *pval1;
    /* printf("Ipol2dT : point 1 in\n"); */
    }
  else
    /* point 1 is out */
    val1= Dummy;

  /* for point 2 */
  if ((i_1<=Dim_1) && (i_2<Dim_2) && (i_1>0) && (i_2>-1)) {
    /* point 2 is in */
    pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
    val2=*pval2;
    /* printf("Ipol2dT : point 2 in\n"); */
    }
  else
    /* point 2 is out */
    val2= Dummy;

  /* for point 3 */
  if ((i_1<=Dim_1) && (i_2<=Dim_2) && (i_1>0) && (i_2>0)) {
    /* point 3 is in */
    pval3 = NEXTCOLROW(pval1,Dim_1,Dim_2);
    val3 = *pval3;
    /* printf("Ipol2dT : point 3 in\n"); */
    }
  else
    /* point 3 is out */
    val3= Dummy;

  /* for point 4 */
  if ((i_1<Dim_1) && (i_2<=Dim_2) && (i_1>-1) && (i_2>0)) {
    /* point 4 is in */
    pval4 = NEXTROW(pval1,Dim_1,Dim_2);
    val4 = *pval4;
    /* printf("Ipol2dT : point 4 in\n"); */
    }
  else
    /* point 4 is out */
    val4= Dummy;

  weight=0.0;
  *value=0.0;
  cnt=0;
  if (r_1<eps) {
     if (r_2<eps) {
       if NODUMMY(val1,Dummy,DDummy) {
         cnt++; *value = val1;  /* no interpolation */
         } /* if NODUMMY */
       } else {
       if NODUMMY(val1,Dummy,DDummy) {
         cnt++;w1=1.0-r_2; weight+=w1; *value+=val1*w1;
         } /* if NODUMMY */
       if NODUMMY(val4,Dummy,DDummy) {
         cnt++;w4=r_2; weight+=w4; *value+=val4*w4;
         } /* if NODUMMY */
       if (cnt) *value = *value/weight;
       } /* if (r_2<eps) */
     } else {
     if (r_2<eps) {
       if NODUMMY(val1,Dummy,DDummy) {
         cnt++; w1=(1-r_1); weight+=w1; *value = val1*w1;
         } /* if NODUMMY */
       if NODUMMY(val2,Dummy,DDummy) {
         cnt++; w2=r_1; weight+=w2; *value+=val2*w2;
         } /* if NODUMMY */
       if (cnt) *value = *value/weight;
       } else {
       if NODUMMY(val1,Dummy,DDummy) {
         cnt++;w1=(1.0-r_1)*(1.0-r_2); weight+=w1; *value+=val1*w1;
         } /* if NODUMMY */
       if NODUMMY(val2,Dummy,DDummy) {
         cnt++;w2=r_1*(1.0-r_2); weight+=w2; *value+=val2*w2;
         } /* if NODUMMY */
       if NODUMMY(val4,Dummy,DDummy) {
         cnt++;w4=(1.0-r_1)*r_2; weight+=w4; *value+=val4*w4;
         } /* if NODUMMY */
       if NODUMMY(val3,Dummy,DDummy) {
         cnt++;w3=r_1*r_2; weight+=w3; *value+=val3*w3;
         } /* if NODUMMY */
       if (cnt) *value = *value/weight;
       } /* (r_2<eps) */
     } /* if (r_1 ... */

  return(cnt);

} /* Ipol2dT */

/*---------------------------------------------------------------------------
1 saxs_urol  

2 PURPOSE
  Do a radial and angular cutting of a user defined 
  circular aera of the picture 
---------------------------------------------------------------------------*/

void saxs_urol (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus) 
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  int   m,n;
  int i_1,i_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float *I1Data;
  float *pI1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Center_1, I1Center_2;
  float WCenter_1, WCenter_2;
  float Value, R, AStep, RStep;
  float alpha ;
  float I1X_1, I1f_1;
  float I1X_2, I1f_2;
 
  float rtf,roh;
  int rtfm;
  float sum;
  float delta;
  int o;
  int cnt;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum]; 

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n");

  if (pcb->TestBit) 
    for(i=1;i<imax;i++) 
      PrintImageHeaderBlock(i,&ihb[i]);

  /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;
 
  /* 1 input, 1 output */
  I0Data     = ihb[0].Data;
  I0Dummy    = ihb[0].Dummy.V;
  I0DDummy   = ihb[0].DDummy.V;
  I0Dim_1    = (int) ihb[0].Dim[1];
  I0Dim_2    = (int) ihb[0].Dim[2];
  
  I1Data     = ihb[1].Data; 
  I1Dummy    = ihb[1].Dummy.V;
  I1DDummy   = ihb[1].DDummy.V;
  I1Dim_1    = (int) ihb[1].Dim[1];
  I1Dim_2    = (int) ihb[1].Dim[2];
  
  /* Calculation of the world coordinate of the center */
  if (ihb[1].Center[1].I) {
    I1Center_1 = I2INDEX(ihb[1].Center[1].V,Off_1[1]);
    WCenter_1 = WORLD(I1Center_1, Off_1[1], Ps_1[1]);
    }
  else {
    printf("Missing parameters for center\n");
    *pstatus = Failed;
    return;
    }
  if (ihb[1].Center[2].I) {
    I1Center_2 = I2INDEX(ihb[1].Center[2].V,Off_2[1]);
    WCenter_2 = WORLD(I1Center_2, Off_2[1], Ps_2[1]);
    }
  else {
    printf("Missing parameters for center\n");
    *pstatus = Failed;
    return;
    }

   /* initialisations */
   AStep = pcb->Arg[NTstep].V;
   RStep = pcb->Arg[NRstep].V;

   if (pcb->TestBit)
    printf("WCenter_1 = %g , WCenter_2 = %g\n",WCenter_1, WCenter_2);

  /* loop over the output array */
  /* from the output array index, calculation of the real R and alpha
     corresponding, and then, determination of the concerned point on
     the input array. When this point is known, an interpolation is
     done to determine the intensity. It is this value that is stored
     in the output array at the i_1 i_2 position */

    roh = sqrt( (I1Dim_1-WCenter_1) * (I1Dim_1-WCenter_1) + 
	       (I1Dim_2-WCenter_2) * (I1Dim_2-WCenter_2) );

    rtfm = roh * 2.0 * SAXS_PI / I0Dim_2;

    for (m=0;m<I0Dim_1;m++) {
     for (n=0;n<I0Dim_2-1;n++) {

      rtf = rtfm * (m/roh) +1;
      sum = 0;

      alpha = n * 2.0 * SAXS_PI / I0Dim_2;

      delta = 2.0 * SAXS_PI / rtf / I0Dim_2;

      cnt = 0; /* PB 28-Aug-1995 */
      for (o=0; o<(int)rtf; o++) {
        
        i_1 = m * cos(alpha) + WCenter_1;
        i_2 = m * sin(alpha) + WCenter_2;

        if ((i_1>(-1)) && (i_1<I1Dim_1) && (i_2>(-1)) && (i_2<I1Dim_2)) {
          pI1Data = ABSPTR(I1Data,I1Dim_1,I1Dim_2,i_1,i_2); /* quark!!!! */
          Value = *pI1Data;
	  if (NODUMMY(Value, I1Dummy, I1DDummy)) {cnt++;sum += Value;}
          }
        alpha += delta;
        } /* for o */
 
      if (cnt) { sum /= cnt; } else { sum = I0Dummy; }

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,m,n);
      *pI0Data = sum;

      }  /* for n ... */ 
    }  /* for m ... */

 } /* saxs_urol*/

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

 /* Declaration of the specific arguments */
  DefFloatOption  ( &cb, SRmin, NRmin );
  DefFloatOption  ( &cb, SRstep, NRstep );
  DefIntOption  ( &cb, SNRstep, NNRstep );
  DefFloatOption  ( &cb, STmin, NTmin );
  DefFloatOption  ( &cb, STstep, NTstep );
  DefIntOption  ( &cb, SNTstep, NNTstep );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, BlockNum );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_urol, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

