/*+++
1 saxs_scal
 Read the value of a keyword from the headers of each picture 
 and write it to a text file.

2 PURPOSE

  Arguments:
  saxs_scal [options] <i1nam> <output prefix> <i1fst> <i1lst> <i1inc> <keyword>
  defaults:
  <input file name>   : input.edf 
  <output prefix>     : input 
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <scaler keyword>    : "Image"  

  Specific options:
  +key <keyword>                : keyword 
  +pref <output file prefix>    : prefix of the output file name
  +ext  <output file extension> : extension of the output file name
  +fnam <output file name>      : output file name

  The values are written to the file <prefix>_<keyword><extension> .

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  1995-04-28  PJE      creation from saxs_norm
  1995-12-04  PB
  1999-06-26  PB V3.0  DevSeparator, DirSeparator, ExtSeparator 
                       defined in SaxsInput
  1999-12-31  PB V3.01 cr\lf in output file "\r\n"
  2000-08-04  PB       %f->%g
  2001-07-03  PB V3.02 filename_full
  2001-07-07  PB V3.03 IO_DontReadWrite, IO_DontReadData
  2001-07-08  PB V3.04 FFirst
  2001-07-09  PB V3.05 (pcb->ib)[0].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_scal V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

/* Option definitions */
# define SKey              "key"   /* string */
# define NKey              0
# define SPrefix           "pref"
# define NPrefix           1
# define SExtension        "ext"    
# define NExtension        2
# define SFName            "fnam"
# define NFName            3

# define Default_extension ".txt"
# define Default_prefix    "input"
# define Default_key       KImage

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> <keyword>\n\
Purpose: Extract keyword values from image headers and write them into a file\n\
Options: -key, -pref, -ext, -fnam"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
SaxsScalArgvFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void SaxsScalArgvFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[1000];
  int nameblk;

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect | IO_DontReadData;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {

    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadData | IO_DontReadWrite |\
                         IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- Argument  : text file prefix */
    if (ib[0].Name.I) nameblk = 0; else nameblk = 1;
    if (!pcb->Lin[NPrefix].I) {
      /* Determine prefix from input file name */
      (void) RightFR(ib[nameblk].Name.V,DevSeparator,pcb->Lin[NPrefix].V,
                     InputLineLength);
      (void) RightFR(pcb->Lin[NPrefix].V,DirSeparator,pcb->Lin[NPrefix].V,
                     InputLineLength);
      (void) LeftFL(pcb->Lin[NPrefix].V,ExtSeparator,pcb->Lin[NPrefix].V,
                     InputLineLength);
      pcb->Lin[NPrefix].I = TRUE;
      } 

    /*--- Prefix of output file name */
    argv_line( pcb,"Output file prefix",&pcb->Lin[NPrefix],pcb->Lin[NPrefix].V, 
               pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from ASCII file name */
    extract_filenumbers( &ib[0], &(pcb->Lin[NPrefix]), TRUE, pstatus );
    if (*pstatus!=Success) return;

    /* Expand Prefix to the full file name */
    if (!filename_full (pcb->Lin[NPrefix].V,IO_len,pcb->Lin[NPrefix].V)) {

      printf("Input file name or path not allowed\n");exit(-1);}

    } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

    /* Defaults for output file */
    if (!ib[0].Dummy.I) ib[0].Dummy.V  = ihb[1].Dummy.V;
    if (!ib[0].Dim[1].I) ib[0].Dim[1].V = ihb[1].Dim[1];
    if (!ib[0].Dim[2].I) ib[0].Dim[2].V = ihb[1].Dim[2];
  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;

  } /* all other input files */

} /* SaxsScalArgvFilenames */

/*---------------------------------------------------------------------------
1 saxs_scal

2 PURPOSE
  Reading of the value of an scaler from an esrf data file and storage in
  an output ascii file.
---------------------------------------------------------------------------*/

void saxs_scal (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ const int NumberOfImages=2;
  const float eps=1e-30;
  const KeyLineLength=1000;
  int i,imax;
  int j;
  int ErrorValue;

  char DefExtension[InputLineLength];
  char KeyLine[KeyLineLength];
  char fname_full[IO_len];

  FILE *OutputFileId;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

  /* Check number of images */
  if (pcb->ImgBlkLen!=NumberOfImages) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

  /*--- Get default extension */
  if (pcb->Lin[NExtension].I) {
    (void) sprintf(DefExtension,"%s",pcb->Lin[NExtension].V);
    } else {
    (void) sprintf(DefExtension,"%s",Default_extension);
    }

  /*--- Get output filename */
  if (!pcb->Lin[NFName].I) {
    sprintf(pcb->Lin[NFName].V,"%s_%s%s",pcb->Lin[NPrefix].V,pcb->Lin[NKey].V,
            DefExtension);
    }

  /*--- Read keyword value */
  if (pcb->TestBit) printf("Reading keyword %s\n",pcb->Lin[NKey].V);
  if (ReadHeaderLine( pcb,1,ihb,pcb->Lin[NKey].V,KeyLine,pstatus)) {
    if (*pstatus!=Success) return;
    } else {
    sprintf(KeyLine,"<not found>");
    }

  /* Generate filename of ASCII file */
  filename_parameter (fname_full,IO_len,pcb->Lin[NFName].V,0);
  filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[0].FileNumber);
  filename_full (fname_full,IO_len,fname_full);

  /*--- Open output file
    If this is the first image of the sequence, an existing file 
    with the same name must be deleted (open mode = "w"),
    in all other cases it must be opened to append a line (open mode "a"). */

  if (pcb->TestBit) printf("Opening output file %s\n",fname_full);
  *pstatus = FileOpeningError;
  if (ihb[1].ImNum==pcb->ib[1].First.V) {
    /* 1st image: new write */
    if ((OutputFileId=fopen(fname_full,"w")) == NULL) return;
    } else {
    /* appending */
    if ((OutputFileId=fopen(fname_full,"a")) == NULL) return;
    }

  /*--- Write to the output file 
    If this is the first image of the sequence, a header line is written */

  if (pcb->TestBit) printf("Writing keyword to file %s\n",fname_full);
  if (ihb[1].ImNum==pcb->ib[1].First.V) {
    /* 1st image: write header line */
    printf("\n");
    printf("Writing to file : %s\n\n", fname_full);
    fprintf(OutputFileId,"Image\t%s\r\n",pcb->Lin[NKey].V);
    } else
    printf("Appending to file : %s\n\n", fname_full);

  /*   Write keyword value to the output file */
  fprintf(OutputFileId,"%d\t",ihb[1].ImNum);
  fprintf(OutputFileId,"%s\r\n",KeyLine);

  /*--- Close output file */

  if (pcb->TestBit) printf("Closing file %s\n",fname_full);
  *pstatus = FileClosingError;
  if (fclose(OutputFileId)!=0) return;

  *pstatus = Success;

 } /* saxs_scal*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,
           InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  /* Expand Prefix to the full file name */
  if (!filename_full (pcb->Lin[NPrefix].V,IO_len,pcb->Lin[NPrefix].V)) {

     printf("% s. Input file name or path not allowed\n",progfn);exit(-1);}
 
  SaxsScalArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : keyword */
  argv_line(pcb,"keyword", &pcb->Lin[NKey],Default_key,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)   printf("i/p file           : %s\n",ib[1].Name.V);
  if (pcb->Lin[NPrefix].I)   
                      printf("output file prefix : %s\n",pcb->Lin[NPrefix].V);
  if (ib[1].First.I)  printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)   printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)    printf("increment          : %d\n",ib[1].Inc.V);
  if (pcb->Lin[NKey].I)
                      printf("keyword            : %s\n",pcb->Lin[NKey].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  DefLineOption( &cb, SKey,       NKey);
  DefLineOption( &cb, SPrefix,    NPrefix );
  DefLineOption( &cb, SExtension, NExtension); 
  DefLineOption( &cb, SFName,     NFName);

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, BlockNum );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_scal, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

