/*+++
1 saxs_refract
  Refraction correction of 2d-surface scattering patterns

2 PURPOSE
  Refraction correction of 2d-surface scattering patterns. The calculation
  is valid for grazing incidence small angle scattering (GISAXS) and
  higher angle scattering. The incoming beam is refracted at the surface 
  scattered inside the sample and and again refracted at the surface before
  it hits the detector. This routine transforms the observed (external) 
  scattering pattern into the internal scattering pattern that would be 
  observed inside the sample. The calculation is done using Snell's law.

  1-delta : real part of the refractive index (0<=delta<1, default 0.0)
  the     : inclination of sample surface (corresponds to angle theta on id01)
            (0<=the<90 deg, default 0.0 deg)
  chi     : rotation of the inclined sample around the primary beam
            (corresponds to angle chi on id01)
            (0<=chi<=180, default 90 deg). Chi is 90 degrees when 
            the sample surface is horizontal

  Arguments:
  saxs_refract [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <delta[0.0]> <the[0.0]> <chi[90.0]>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>

  <delta>             : refraction index n = 1-delta (default 0)
  <the>               : sample inclination (default 0 deg)
  <chi>               : sample tilt (default: 90 deg)

2 HISTORY
  2001-02-21  from saxs_mac V3.12
  2001-03-01 
  2001-03-27  refract.h
  2001-03-30  V3.0
  2001-04-08  V3.1 -psi -> -the for theta angle 
  2001-05-31  V3.11 corrected: Delta not multiplied by deg2rad
              V3.12 TwoTheta must be smaller than 90 deg
  2001-07-09  PB V4.00 new loop, no repetition


---*/
# define Version  "saxs_refract V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "refract.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> \n\
                <delta[0.0]> <the[0.0]> <chi[90.0]>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SDelta "del"   /* float */
# define NDelta 0
# define STheta "the"   /* float */
# define NTheta 1
# define SChi   "chi"   /* float */
# define NChi   2

/*---------------------------------------------------------------------------
1 saxs_refract

2 PURPOSE
  GISAXS refraction correction
---------------------------------------------------------------------------*/
void saxs_refract (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{  const float deg2rad = SAXS_PI/180.0, rad2deg = 180.0/SAXS_PI;
  int   i, imax; 
  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int i_1, i_2;
  float W, WL, W_1, W_2;

  float f1_1, f1_2;
  float W1, W1L, W1_1, W1_2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  double Delta, Theta, Chi;                                                       
  RefractDir Beam, Beam1;
  float L, L1;

  float tmp;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* check the reference system: Normal required */
  if (pcb->RSys.V != IO_Normal) {
    printf("ERROR: Reference system %s required\n",reftostr(IO_Normal));
    printf("       Start program with option -rsys %s\n",reftostr(IO_Normal)); 
    }

  if (!((ihb[0].SampleDistance.I)&&(ihb[0].SampleDistance.I))) {
    printf("ERROR: SampleDistance missing\n");
    printf("       Use option -i1dis <SampleDistance[m]>\n");
    *pstatus=Failed; return;
    }
  L = ihb[0].SampleDistance.V;
  L1 = ihb[1].SampleDistance.V;                                   

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;                                             

  Delta = (double) pcb->Arg[NDelta].V;
  Theta   = (double) pcb->Arg[NTheta].V * deg2rad;
  Chi   = (double) pcb->Arg[NChi].V * deg2rad;    
  
  refract_Init ( stdout, Delta, Theta, Chi );

  if (pcb->TestBit) {
    refract_PrintParams ( stdout );
    printf("\n"); }                                                             

  /* loop over the output array  */
  for (i_2=0;i_2<I0Dim_2;i_2++) {
    W_2 =  WORLD( i_2, Off_2[0], Ps_2[0] );
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
    for (i_1=0;i_1<I0Dim_1;i_1++) {
      W_1 =  WORLD( i_1, Off_1[0], Ps_1[0] ); 

      tmp = W_1*W_1 + W_2*W_2; 
      W   = sqrt(tmp);
      WL  = sqrt(tmp+L*L);

      Beam.cosAlpha    = W_1/W;
      Beam.sinAlpha    = W_2/W;
      Beam.cosTwoTheta = L/WL;
      Beam.sinTwoTheta = W/WL;

      if (pcb->TestBit>3) {
        printf("Incident beam\n");
        refract_PrintBeam ( stdout, Beam );
        printf("\n"); }                                                                                                                           
      Beam1 = refract_Angles ( Beam );

      if (!Beam1.status) { 

        if (pcb->TestBit>3) {
          printf("Incident beam\n");
          refract_PrintBeam ( stdout, Beam );
          printf("\n"); }                                                                                                                       
        if (Beam1.cosTwoTheta>0.0) {

          W1   = L1 * Beam1.sinTwoTheta/Beam1.cosTwoTheta;
          W1_1 = W1 * Beam1.cosAlpha;
          W1_2 = W1 * Beam1.sinAlpha;

          f1_1 = INDEX(W1_1, Off_1[1], Ps_1[1]);
          f1_2 = INDEX(W1_2, Off_2[1], Ps_2[1]);

          if (Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
            f1_1, f1_2, &Value)) {
            /* then do something with the data */

            UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

            } /* if (Ipol2d ... */

          } /* if Beam1.cosTwoTheta ... */
        } /* !Beam1.status */

      pI0Data++;
      } /* for i_1 ... */
    } /* for i_2 ... */

 } /* saxs_refract*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : difference from refraction index */
  argv_float(pcb,"delta (refraction index n = 1-delta)",
             &pcb->Arg[NDelta],pcb->Arg[NDelta].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sample inclination */
  argv_float(pcb,"sample inclination (deg)",
             &pcb->Arg[NTheta],pcb->Arg[NTheta].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sample tilt */
  argv_float(pcb,"sample tilt (deg)",
             &pcb->Arg[NChi],pcb->Arg[NChi].V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
if (pcb->Arg[NDelta].I)printf("delta = 1 - n      : %g\n",pcb->Arg[NDelta].V);
if (pcb->Arg[NTheta].I)printf("sample inclination : %g deg\n",
    pcb->Arg[NTheta].V);
if (pcb->Arg[NChi].I)  printf("sample tilt        : %g deg\n",
    pcb->Arg[NChi].V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  cb.RSys.V = IO_Normal; /* default */

 /* Define special options */
  DefFloatOption( &cb, SDelta, NDelta);
  cb.Arg[NDelta].V = 0.0; /* default */
  DefFloatOption( &cb, STheta, NTheta );
  cb.Arg[NTheta].V = 0.0;   /* default */
  DefFloatOption( &cb, SChi, NChi );
  cb.Arg[NChi].V = 90.0;  /* default */

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_refract, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

