/*+++
1 saxs_grid
  Generates an image with a grid pattern

2 PURPOSE
  Generates an image with a grid pattern

  Arguments:
  saxs_grid [options] <onam> <odum> <dim 1> <dim 2> <ocon>"

  Defaults:
  output file name    : <onam>              : output.edf
  output dummy        : <odum>              : 0.0
  dimension 1         : <odim 1>            : 1
  dimension 2         : <odim 2>            : 1
  output constant     : <ocon>              : 0.0 (only used if dummy = 0)

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2001-09-10  PB V4.00 from saxs_new V4.00 

---*/
# define Version  "saxs_grid V4.00 2001-09-10, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <onam> <odum> <odim1> <odim2>\n\
                <anc1> <anc2> <gri1> <gri2)\n\
                <diam> <mval[0.0]> <pval[1.0]>\n\
                Generates an image with a grid pattern"

# define BlockNum 1       /* 1 output sequence */

/* Flg */
/* Lin */
/* Num */
/* Val */
# define SAnch_1  "anc1"
# define NAnch_1  0
# define SAnch_2  "anc2"
# define NAnch_2  1
# define SPitch_1 "gri1"
# define NPitch_1 2
# define SPitch_2 "gri2"
# define NPitch_2 3
# define SDiam    "diam"
# define NDiam    4
# define SMatVal  "mval"
# define NMatVal  5
# define SPeakVal "pval"
# define NPeakVal 6

/*---------------------------------------------------------------------------
 internal functions 
---------------------------------------------------------------------------*/

static const float grid_eps = 1e-32;
/*---------------------------------------------------------------------------
NAME
  grid --- returns 1 if pixel (x_1, x_2) is inside a grid point 

SYNOPSIS 
  int isgridpoint( float anchor_1, float anchor_2, 
                   float pitch_1, float pitch_2,
                   float radius, 
                   float x_1, float x_2 );

DESCRIPTION
  The function tests whether (N = ..., -2, -1, 0, 1, 2, ...)
      ( sqr( x_1 - ( anchor_1 + N * pitch_1 ) ) +
        sqr( x_2 - ( anchor_2 + N * pitch_2 ) )) <= sqr(radius)
  where N is a positive or negative integer number
  It returns 1, if it is the case, otherwise 0, -1 in case of an error.
---------------------------------------------------------------------------*/
int isgridpoint( float anchor_1, float anchor_2, 
                 float pitch_1, float pitch_2, 
                 float radius, 
                 float x_1, float x_2 )
{ double x_1d, x_2d;
  double N_1, N_2;
  double invpitch_1, invpitch_2; 
  double tmp_1, tmp_2, dist2, radius2;

  if ( ( fabs(pitch_1)<grid_eps )||( fabs(pitch_2)<grid_eps ) ) {
    printf("ERROR: Division by zero \n");return (-1); }

  invpitch_1 = 1.0/pitch_1;
  invpitch_2 = 1.0/pitch_2;

  x_1d = (double) x_1;
  x_2d = (double) x_2;
  radius2 = (double) radius * (double) radius;

  // round N_1 and N_2
  N_1 = floor( (x_1d - anchor_1) * invpitch_1 + (double) 0.5 );   
  N_2 = floor( (x_2d - anchor_2) * invpitch_2 + (double) 0.5 );
  tmp_1 = x_1d - ( N_1 * pitch_1 + anchor_1 );
  tmp_2 = x_2d - ( N_2 * pitch_2 + anchor_2 );
  dist2 = tmp_1*tmp_1 + tmp_2*tmp_2;

  if (dist2 <= radius2 ) 
    return(1); // it's a grid point
   else return(0); // it's a matrix point

} // grid

/*---------------------------------------------------------------------------
1 saxs_grid

2 PURPOSE
  Generates an image with a grid pattern
---------------------------------------------------------------------------*/
void saxs_grid (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  float anchor_1, anchor_2;
  float pitch_1, pitch_2;
  float radius;
  float matval, peakval;
  int   itis;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=1) {
     printf("%d images found, 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 1, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];

 /* parameters */
  anchor_1 = pcb->Arg[NAnch_1].V;
  anchor_2 = pcb->Arg[NAnch_2].V;
  pitch_1 = pcb->Arg[NPitch_1].V;
  pitch_2 = pcb->Arg[NPitch_2].V;
  radius = pcb->Arg[NDiam].V*0.5;
  matval = pcb->Arg[NMatVal].V;
  peakval = pcb->Arg[NPeakVal].V;

 /* loop over 1 output image */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  /* loop over the output array  */
  for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
  for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
    W_2 = WORLD( i_2, Off_2[0], Ps_2[0] ); 
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1[0],i_2);
    for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
    for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
      W_1 = WORLD( i_1, Off_1[0], Ps_1[0] ); 
        /* do something with the data */

        itis = isgridpoint( anchor_1, anchor_2,
               pitch_1, pitch_2, radius, W_1, W_2 );
        if (itis==0) Value = matval;
          else if (itis>0) Value = peakval; 
            else { *pstatus = DivByZero; return; } // error

        UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

      pI0Data++;
      for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
      } /* for i_1 ... */
    for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
    } /* for i_2 ... */

 } /* saxs_grid*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,0,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,0,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument : world coordinate 1 of anchor */
  argv_coord(pcb,pcb->RSys.V,ROff_1,RPs_1,pcb->USys.V,UOff_1,UPs_1,
             "Position 1 of grid anchor",&pcb->Arg[NAnch_1], 0.0, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : world coordinate 2 of anchor */
  argv_coord(pcb,pcb->RSys.V,ROff_2,RPs_2,pcb->USys.V,UOff_2,UPs_2,
             "Position 2 of grid anchor",&pcb->Arg[NAnch_2], 0.0, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument : world pitch 1 of grid */
  argv_dist ( pcb, pcb->RSys.V, ROff_1, RPs_1, pcb->USys.V, UOff_1, UPs_1,
              "Pitch 1 of grid", &pcb->Arg[NPitch_1], 50.0, pstatus); 
  if (*pstatus!=Success) return;

  /*--- Argument : world pitch 2 of grid */
  argv_dist ( pcb, pcb->RSys.V, ROff_2, RPs_2, pcb->USys.V, UOff_2, UPs_2,
              "Pitch 2 of grid", &pcb->Arg[NPitch_2], 50.0, pstatus);        
  if (*pstatus!=Success) return;

  /*--- Argument : world pitch 2 of grid */
  argv_dist ( pcb, pcb->RSys.V, ROff_2, RPs_2, pcb->USys.V, UOff_2, UPs_2,
              "Diameter of grid points", &pcb->Arg[NDiam], 5.0, pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
                      printf("reference system   : %s\n",reftostr(pcb->RSys.V));
  if (pcb->Arg[NAnch_1].I)
                       printf("grid anchor 1      : %g (reference system %s)\n",
                       pcb->Arg[NAnch_1].V, reftostr(pcb->RSys.V));
  if (pcb->Arg[NAnch_2].I)
                       printf("grid anchor 2      : %g (reference system %s)\n", 
                       pcb->Arg[NAnch_2].V, reftostr(pcb->RSys.V));
  if (pcb->Arg[NPitch_1].I)
                       printf("pitch  1           : %g (reference system %s)\n",
                       pcb->Arg[NPitch_1].V, reftostr(pcb->RSys.V));
  if (pcb->Arg[NPitch_2].I)
                       printf("pitch  2           : %g (reference system %s)\n",
                       pcb->Arg[NPitch_2].V, reftostr(pcb->RSys.V));
  if (pcb->Arg[NDiam].I)
                       printf("Point diameter     : %g (reference system %s)\n",
                       pcb->Arg[NDiam].V, reftostr(pcb->RSys.V));

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  ib[0].Dim[1].V = 512;
  ib[0].Dim[2].V = 512;

  DefFloatOption ( &cb, SAnch_1,    NAnch_1 );
  DefFloatOption ( &cb, SAnch_2,    NAnch_2 );
  DefFloatOption ( &cb, SPitch_1,   NPitch_1 );
  DefFloatOption ( &cb, SPitch_2,   NPitch_2 );
  DefFloatOption ( &cb, SDiam,      NDiam );
  /* No prompting for the following parameters */
  DefFloatOption ( &cb, SMatVal,    NMatVal );
  cb.Arg[NMatVal].V = 0.0; // default
  DefFloatOption ( &cb, SPeakVal,   NPeakVal );
  cb.Arg[NPeakVal].V = 1.0; // default

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_grid, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

