/*+++
1 saxs_gauss
  Adds gaussian noise to an input image.

2 PURPOSE
  Adds gaussian noise to an input image. A random number generator 
  adds to each input pixel value ny a random number that follows 
  gaussian statistis:

  G(sigm) = 1/sqrt(2*pi) * 1/sigm * exp( (Dx^2/(2*s^2) )

  Arguments:
  saxs_gauss [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <ofac> <ocon> <sigm> <seed> <sinc>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <output const>      : 0.0
  <output factor>     : 1.0
  <sigma>             : 0.0
  <seed>              : 13131
  <sinc>              : 2

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  2000-11-17  PB V3.0 from saxs_mac V3.12
  2000-11-23  PB V3.1 output values rounded to nearest integer
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-20  PB V4.01 Sinc, copy IO, I1, ExpTim
  2001-09-02  PB V4.02 output values are no longer rounded to nearest integer
  2001-09-16  PB V4.03 seed is 13131

---*/
# define Version  "saxs_gauss 4.02 2001-09-02, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"
# include "gauss.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <ofac> <ocon> <sigm> \n\
                <seed> <sinc>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SSigm "sigm" /* float */
# define NSigm 0
# define SSeed "seed" /* int */
# define NSeed 0
# define SSinc "sinc"
# define NSinc 1

/*---------------------------------------------------------------------------
1 saxs_gauss

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void saxs_gauss (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  unsigned int seed = (unsigned int) pcb->Num[NSeed].V;
  double sigm = (double) pcb->Arg[NSigm].V;

  IO_float I0, I1, ExpTim;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 1, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

  if (pcb->TestBit) {
    printf(" seed = %u\n",seed);
    printf(" sigm = %g\n",sigm);
    }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* copy I0 and I1 from input to output */
  if (ReadHeaderFloat ( pcb, 1, ihb, KValueI0, &(I0.V), pstatus )) {
    if (*pstatus!=Success) return;
    I0.I = TRUE;
    } else I0.I = FALSE;
 
  if (ReadHeaderFloat ( pcb, 1, ihb, KValueI1, &(I1.V), pstatus )) {
    if (*pstatus!=Success) return;
    I1.I = TRUE;
    } else I1.I = FALSE;

  if (ReadHeaderFloat ( pcb, 1, ihb, KValueTime, &(ExpTim.V), pstatus )) {
    if (*pstatus!=Success) return;
    ExpTim.I = TRUE;
    } else ExpTim.I = FALSE;
 
  if (I0.I) {
    WriteHeaderFloat ( pcb, 0, ihb, KValueI0, I0.V, pstatus );
    if (*pstatus!=Success) return;
    }
 
  if (I1.I) {
    WriteHeaderFloat ( pcb, 0, ihb, KValueI1, I1.V, pstatus );
    if (*pstatus!=Success) return;
    }

  if (ExpTim.I) {
    WriteHeaderFloat ( pcb, 0, ihb, KValueTime, ExpTim.V, pstatus );
    if (*pstatus!=Success) return;
    }
 
  *pstatus = Success;

 /* loop over 1 input and 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  GaussNoiseSeed( seed );

  /* loop over the output array  */
  for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
  for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1[0],i_2);
    for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
    for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
      if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
        f_1[1], f_2[1], &Value)) {
        /* then do something with the data */

        Value += (float) GaussNoise( sigm );

        UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );

        } /* if (Ipol2d ... */
      pI0Data++;
      for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
      } /* for i_1 ... */
    for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
    } /* for i_2 ... */

   // increment seed
   pcb->Num[NSeed].V += pcb->Num[NSinc].V;

 } /* saxs_gauss*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_float(pcb,"Multiplication factor",&ib[0].Factor,1.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  printf("<output image> = <input image> * Factor + Const\n");
  argv_float(pcb,"Addition constant",&ib[0].Const,0.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sigm */
  printf("<output image> = <input image> + DI \n");
  printf("  with P(DI) = 1/(sqrt(2*pi)/*sigma) * exp(-DI^2/2*sigma^2)\n");
 argv_float(pcb,"Sigma of Gaussian",&pcb->Arg[NSigm],pcb->Arg[NSigm].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : seed */
 argv_long(pcb,"Random number seed",&pcb->Num[NSeed],pcb->Num[NSeed].V,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : sinc */
 argv_long(pcb,"Seed increment",&pcb->Num[NSinc],pcb->Num[NSinc].V,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (ib[0].Factor.I)  printf("factor             : %g\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("constant           : %g\n",ib[0].Const.V);
if (pcb->Arg[NSigm].I) printf("gaussian sigma     : %g\n",pcb->Arg[NSigm].V);
if (pcb->Num[NSeed].I) printf("random number seed : %d\n",pcb->Num[NSeed].V);
if (pcb->Num[NSinc].I) printf("seed increment     : %d\n",pcb->Num[NSinc].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

 /* Define special options */
  DefFloatOption( &cb, SSigm, NSigm);
  cb.Arg[NSigm].V = 0.0;     /* default */
  DefIntOption( &cb, SSeed,  NSeed);
  cb.Num[NSeed].V = 13131l;  /* default */
  DefIntOption( &cb, SSinc,  NSinc);
  cb.Num[NSinc].V =     2l;  /* default */ 

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_gauss, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

