/*+++
1 saxs_col
  Projections of columns.

2 PURPOSE
  Projections of columns in a sequence on a single column.
  The column is written to a line of the output image.

  Arguments:
  saxs_col [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <col1> <col2>"

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>
  <first column>      : start of averaging
  <last column>       : end of averaging

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  08-Oct-1995 PB 1.4
  04-Dec-1995 PB 3.0
  30-Aug-1996 PB 3.12 Option +-vsum (default -vsum)
		      -vsum : switch off sum
                      +vsum : sum of values 
                      Option +-vint (default -vint)
                      -vint : switch off vint
                      +vint : integration of values 
                      Default : averaging
  27-Sep-1996 PB 3.13 Output row number is always calculated with REALREF
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  2000-08-04  PB %f -> %g
  2001-07-08  PB V3.14 FFirst
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_col 4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <col1> <col2>\n\
Purpose : Projections of columns in a sequence, output to a single image\n\
Options : -col1, -col2, -row1, -row2, +vsum, +vint"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Float Options */
enum { ArgCol1=0, ArgCol2, ArgRow1, ArgRow2 };

/* Flags */
enum { FlgSum=0, FlgInt };

/*---------------------------------------------------------------------------
1 saxs_col

2 PURPOSE
  Projections of a column range on a line of the output image.
  output row number : filled line by line, determined by DataCtrl.

  The interesting area in the input image is defined by
  W21 : upper left corner, W22 : upper right corner,
  W11 : lower left corner, W12 : lower right corner.

  W21 = {row2,col1}, WP22 = {row2, col2}.
  W11 = {row1,col1}, WP12 = {row1, col2},

  pcb->Arg[ArgCol1], pcb->Arg[ArgCol2] etc.

  array: averaging of col1 to col2. resulting column to line of output
  image: averaging of col1 to col2. resulting column to line of output
  real:
  saxs:
---------------------------------------------------------------------------*/

void saxs_col (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1[BlockNum], Wmax_1[BlockNum];
  float Wmin_2[BlockNum], Wmax_2[BlockNum];

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  float W1, W2;
  float value, sum, cnt;

  float col1, col2, row1, row2;

  int i_11;

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* Check options */
  if ((pcb->Flg[FlgSum].V) && (pcb->Flg[FlgInt].V)) {
     printf("ERROR: Use of mutually exclusive options\n");
     *pstatus=Failed; return; } 

 /* Get reference system parameters */
   GetReferenceParameters( pcb, ihb, 0, imax-1,
                           Off_1, Off_2, Ps_1,  Ps_2, pstatus );
   if (*pstatus!=Success) return;

 /* Set the output row reference system to real */
   REALREF(Off_2[0],Ps_2[0],ihb[0].Offset[2].V,ihb[0].PixSiz[2].V);
 /* Subtract output shift for calculation */
   if (pcb->Shift[2].I) Off_2[0] = Off_2[0]-pcb->Shift[2].V;

   if (pcb->TestBit) {
     printf("Off_1[%d] = % f, Ps_1[%d] = % f\n", 0,Off_1[0],0,Ps_1[0]);
     printf("Off_2[%d] = % f, Ps_2[%d] = % f\n", 0,Off_2[0],0,Ps_2[0]);
     }

 /* Calculate region of input and output image */
   for (i=0;i<imax;i++) {
     W1 = WORLD(0,Off_1[i],Ps_1[i]);
     W2 = WORLD(ihb[i].Dim[1]-1,Off_1[i],Ps_1[i]);
     Wmin_1[i] = MIN2(W1,W2); Wmax_1[i] = MAX2(W1,W2);
     W1 = WORLD(0,Off_2[i],Ps_2[i]);
     W2 = WORLD(ihb[i].Dim[2]-1,Off_2[i],Ps_2[i]);
     Wmin_2[i] = MIN2(W1,W2); Wmax_2[i] = MAX2(W1,W2);
     }

 if (pcb->TestBit) {
   printf("Standard Regions\n");
   printf("Wmin_1[0] = % f, Wmax_1[0] = % f\n",Wmin_1[0], Wmax_1[0]);
   printf("Wmin_2[0] = % f, Wmax_2[0] = % f\n",Wmin_2[0], Wmax_2[0]);
   printf("Wmin_1[1] = % f, Wmax_1[1] = % f\n",Wmin_1[1], Wmax_1[1]);
   printf("Wmin_2[1] = % f, Wmax_2[1] = % f\n",Wmin_2[1], Wmax_2[1]);
   }

 /* Get region of interest */
 if (pcb->Arg[ArgCol1].I) col1=pcb->Arg[ArgCol1].V; else col1=Wmin_1[1];
 if (pcb->Arg[ArgCol2].I) col2=pcb->Arg[ArgCol2].V; else col2=Wmax_1[1];
 if (pcb->Arg[ArgRow1].I) row1=pcb->Arg[ArgRow1].V; else row1=Wmin_2[1];
 if (pcb->Arg[ArgRow2].I) row2=pcb->Arg[ArgRow2].V; else row2=Wmax_2[1];

 /* Calculate region of interest in input image */
 Wmin_1[1] = MAX2(Wmin_1[1],MIN2(col1,col2));
 Wmax_1[1] = MIN2(Wmax_1[1],MAX2(col1,col2));
 Wmin_2[1] = MAX2(Wmin_2[1],MIN2(row1,row2));
 Wmax_2[1] = MIN2(Wmax_2[1],MAX2(row1,row2));

 /* Calculate the horizontal region of interest in output image */
 W1 = MAX2(Wmin_1[0],Wmin_2[1]);
 W2 = MIN2(Wmax_1[0],Wmax_2[1]);
 Wmin_1[0] = MIN2(W1,W2); Wmax_1[0] = MAX2(W1,W2);

 if (pcb->TestBit) {
   printf("Restricted by col1, col2, row1, row2\n");
   printf("Wmin_1[0] = % f, Wmax_1[0] = % f\n",Wmin_1[0], Wmax_1[0]);
   printf("Wmin_2[0] = % f, Wmax_2[0] = % f\n",Wmin_2[0], Wmax_2[0]);
   printf("Wmin_1[1] = % f, Wmax_1[1] = % f\n",Wmin_1[1], Wmax_1[1]);
   printf("Wmin_2[1] = % f, Wmax_2[1] = % f\n",Wmin_2[1], Wmax_2[1]);
   }

 /* Calculate the horizontal program array indices in the output image */
 fmin_1[0] = INDEX(Wmin_1[0],Off_1[0],Ps_1[0]);
 fmax_1[0] = INDEX(Wmax_1[0],Off_1[0],Ps_1[0]);

 /* Calculate integer indices */
 IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
 IDX(fmax_1[0],Imax_1[0],RImax_1[0]);

 /* recalculate the overlap area from the size of the output array,
    the output array determines the calculation grid,
    thus only discrete array points are taken */
 if (ABS(RImin_1[0]>0)) fmin_1[0]=Imin_1[0]+1; else fmin_1[0]=(float)Imin_1[0];
 fmax_1[0]=(float)Imax_1[0];

 W1  = WORLD(fmin_1[0],Off_1[0],Ps_1[0]);
 W2  = WORLD(fmax_1[0],Off_1[0],Ps_1[0]);
 Wmin_1[0]=MAX2(Wmin_1[0],MIN2(W1,W2));
 Wmax_1[0]=MIN2(Wmax_1[0],MAX2(W1,W2));

 Wmin_2[1]=MAX2(Wmin_2[1],Wmin_1[0]);
 Wmax_2[1]=MIN2(Wmax_2[1],Wmax_1[0]);

 if (pcb->TestBit) {
   printf("Wmin_1[0] = % f, Wmax_1[0] = % f\n",Wmin_1[0], Wmax_1[0]);
   printf("Wmin_1[1] = % f, Wmax_1[1] = % f\n",Wmin_1[1], Wmax_1[1]);
   printf("Wmin_2[1] = % f, Wmax_2[1] = % f\n",Wmin_2[1], Wmax_2[1]);
   }

 /* Recalculate the program array indices */
 fmin_1[0] = INDEX(Wmin_1[0],Off_1[0],Ps_1[0]);
 fmax_1[0] = INDEX(Wmax_1[0],Off_1[0],Ps_1[0]);

 fmin_1[1] = INDEX(Wmin_1[1],Off_1[1],Ps_1[1]);
 fmax_1[1] = INDEX(Wmax_1[1],Off_1[1],Ps_1[1]);
 fmin_2[1] = INDEX(Wmin_2[1],Off_2[1],Ps_2[1]);
 fmax_2[1] = INDEX(Wmax_2[1],Off_2[1],Ps_2[1]);

 /* Print the program array indices */
 if (pcb->TestBit) {
   printf("fmin_1[% d] = % f, fmax_1[% d] = % f\n",0,fmin_1[0],0,fmax_1[0]);
   printf("fmin_1[% d] = % f, fmax_1[% d] = % f\n",1,fmin_1[1],1,fmax_1[1]);
   printf("fmin_2[% d] = % f, fmax_2[% d] = % f\n",1,fmin_2[1],1,fmax_2[1]);
   }

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* loop over 1 input and 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]); IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmin_1[1],Imin_1[1],RImin_1[1]); IDX(fmax_1[1],Imax_1[1],RImax_1[1]);
  IDX(fmin_2[1],Imin_2[1],RImin_2[1]); IDX(fmax_2[1],Imax_2[1],RImax_2[1]);

  if (pcb->TestBit)  {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",0,Imin_1[0],0,Imax_1[0]);
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",1,Imin_1[1],1,Imax_1[1]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",1,Imin_2[1],1,Imax_2[1]);
    } /* if */

  if (pcb->Flg[FlgSum].V) {
    /* Sum */
    if (pcb->TestBit) printf("Summation\n");
    } else {
    if (pcb->Flg[FlgInt].V) {
      /* Integration */
      if (pcb->TestBit) printf("Integration\n");
        } else {
        /* Averaging */
        if (pcb->TestBit) printf("Averaging\n");
        }
      }

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1; /* filling of output row */
  maxloop_2 = Imax_1[1]-Imin_1[1]+1; /* column averaging */

  /* calculate the delta values of the indices */
  if (maxloop_1>1) {
    Df_1[0]=(fmax_1[0]-fmin_1[0])/(maxloop_1-1);
    Df_2[1]=(fmax_2[1]-fmin_2[1])/(maxloop_1-1);
    } else { Df_1[0]=0.0; Df_2[1]=0.0; }
  if (maxloop_2>1)
    Df_1[1]=(fmax_1[1]-fmin_1[1])/(maxloop_2-1); else Df_1[1]=0.0;

 /* determine output row number */
  i_2 = INDEX(ihb[1].ImNum,Off_2[0],Ps_2[0]); /* input image number */
  if (pcb->TestBit)  {
    printf("Output row number = %d (%9.5g)\n",\
      i_2,INDEX(ihb[1].ImNum,Off_2[0],Ps_2[0]));
    }

  if ( (0<=i_2) && (i_2<=I0Dim_2) ) {
  /* integrate along a line */

    /* fill one line of the output array */
    f_1[0]=fmin_1[0]; /* first output column */
    f_2[1]=fmin_2[1]; /* first line to average/sum */

    for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);

      /* average/sum 1 line of the input array */
      sum=0.0;cnt=0.0;
      f_1[1]=fmin_1[1]; /* first column in input line */
      for (i_11=0;i_11<maxloop_2;i_11++) {

        if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
          f_1[1], f_2[1], &value)) {
          sum+=value; cnt+=1.0;
          } /* if (Ipol2d ... */

        f_1[1]+=Df_1[1]; /* next column of input line */
        } /* for i_11 ... */

      if (pcb->Flg[FlgSum].V) {
        /* Sum */
        if (cnt>0) *pI0Data = sum; else *pI0Data = I0Dummy;
        } else {
        if (pcb->Flg[FlgInt].V) {
          /* Integration */
          if (cnt>0) *pI0Data = sum*Ps_1[1]; else *pI0Data = I0Dummy;
            } else {
            /* Average */
            if (cnt>0) *pI0Data = sum/cnt; else *pI0Data = I0Dummy;
            }
          }

/*  printf("i_1=% d,i_2=% d,Imin_1[1]=% d,Imax_1[1]=% d,cnt=% f,result=% f\n",
            i_1, i_2, Imin_1[1],Imax_1[1],cnt,*pI0Data); /* ++++++++++ */

      f_1[0]+=Df_1[0]; /* next output column */
      f_2[1]+=Df_2[1]; /* next input line */
      } /* for i_1 ... */

    } /* if i_2 ... */

 } /* saxs_col */

/*---------------------------------------------------------------------------
ArgvColFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void ArgvColFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[1000];
  float rest;

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib, blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {
    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- File name ib[0].Name of output */
    argv_filename( pcb,"Output sequence",ib, 0,DefaultOutput, pstatus);
    if (*pstatus!=Success) return;

  } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

    /* Defaults for output file */
    if (!ib[0].Dummy.I) ib[0].Dummy.V  = ihb[1].Dummy.V;
    if (!ib[0].Dim[1].I) ib[0].Dim[1].V = ihb[1].Dim[1];
    if (!ib[0].Dim[2].I) ib[0].Dim[2].V = ihb[1].Dim[2];

    /*--- Set output header */
    /* output offset 2 */
    if ((ihb[1].Offset[2].I) && (!ib[0].Offset[1].I)) {
      /* use for output offset 1 */
      ib[0].Offset[1].V = ihb[1].Offset[2].V;
      ib[0].Offset[1].I = TRUE; }

    /* output center 1 */
    if ((ihb[1].Center[2].I) && (!ib[0].Center[1].I)) {
      /* use for output center 1 */
      ib[0].Center[1].V = ihb[1].Center[2].V;
      ib[0].Center[1].I = TRUE; }

    /* output pixel size 1 */
    if ((ihb[1].PixSiz[2].I) && (!ib[0].PixSiz[1].I)) {
      /* use for output pixel size 1 */
      ib[0].PixSiz[1].V = ihb[1].PixSiz[2].V;
      ib[0].PixSiz[1].I = TRUE; }

    /* output offset 2 */
    if (!ib[0].Offset[2].I) {
       ib[0].Offset[2].V = (ib[1].First.V/ib[1].Inc.V) - 1;
       ib[0].Offset[2].I = TRUE; }

    /* output pixel size 2 */
    if (!ib[0].PixSiz[2].I) {
       ib[0].PixSiz[2].V = (float) ib[1].Inc.V;
       ib[0].PixSiz[2].I = TRUE; }

    /* output center 2 */
    if (!ib[0].Center[2].I) {
       ib[0].Center[2].V = INDEX2I(0,ib[0].Offset[2].V);
       ib[0].Center[2].I = TRUE; }

  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;

  } /* all other input files */

  if (block_1st <= 0) {
    /*--- Argument : output dummy */
    argv_float( pcb, "Output dummy", &ib[0].Dummy, ib[0].Dummy.V, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    if (!ib[0].Dim[1].I) {
      ib[0].Dim[1].V = ihb[1].Dim[2];
      ib[0].Dim[1].I = TRUE;
      }
    argv_long(pcb,"Output dimension 1", &ib[0].Dim[1],ib[0].Dim[1].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    if (!ib[0].Dim[2].I) {
      if (ib[0].PixSiz[2].V==0) {
        ib[0].Dim[2].V = (MAX2(ib[1].First.V,ib[1].Last.V) -
                          MIN2(ib[1].First.V,ib[1].Last.V)) + 1;
        } else {
        IDX((ib[1].Last.V-ib[1].First.V)/ib[0].PixSiz[2].V+1.5, \
             ib[0].Dim[2].V, rest);
        }
      ib[0].Dim[2].I = TRUE;
      }

    argv_long(pcb,"Output dimension 2", &ib[0].Dim[2],ib[0].Dim[2].V, pstatus);
    if (*pstatus!=Success) return;

    /* --- into first image */
    if (!ib[0].First.I) {
       ib[0].First.V = 1;
       ib[0].First.I = TRUE; }

    /* --- only 1 output image */
    if (!ib[0].Inc.I) {
       ib[0].Inc.V = 0;
       ib[0].Inc.I = TRUE; }

    }

} /* ArgvColFilenames */
  
/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvColFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument 5 : column 1 */
  argv_coord( pcb, pcb->RSys.V, ROff_1, RPs_1, pcb->USys.V, UOff_1, UPs_1,\
      "boundary (col1)", &pcb->Arg[ArgCol1], \
      WORLD(INDEXSTART,ROff_1,RPs_1), pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument 6 : column 2 */
  argv_coord( pcb, pcb->RSys.V, ROff_1, RPs_1, pcb->USys.V, UOff_1, UPs_1,\
      "boundary (col2)", &pcb->Arg[ArgCol2], \
      WORLD(INDEXSTART+ihb[1].Dim[1]-1,ROff_1,RPs_1), pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (pcb->Arg[ArgCol1].I) 
                       printf("boundary (col1)    : %g\n",pcb->Arg[ArgCol1].V);
  if (pcb->Arg[ArgCol2].I)
                       printf("boundary (col2)    : %g\n",pcb->Arg[ArgCol2].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );
  DefFloatOption(&cb,"col1",ArgCol1);
  DefFloatOption(&cb,"col2",ArgCol2);
  DefFloatOption(&cb,"row1",ArgRow1);
  DefFloatOption(&cb,"row2",ArgRow2);
  DefFlgOption  (&cb,"vsum",FlgSum);
  DefFlgOption  (&cb,"vint",FlgInt);

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, BlockNum );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_col, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

