/*+++
1 saxs_bsl
  Transformation of an edf image into bsl format (daresbury otoko)

2 PURPOSE
  Transformation of an edf image into bsl format (daresbury otoko)

  Arguments:
  saxs_bsl [options] 
                <i1nam> <output header name> <i1fst> <i1lst> <i1inc> 
                <odum> <odim1> <odim2>
  Defaults:
  <input file name>   : input.edf 
  <output header name>: I54000.EDF (default conversion from input filename)
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>

  Specific arguments:
  -pref 'prefix'      : specify the prefix of the ouptput file

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  28-Nov-1995 PB
  21-May-1996 PB option +bsl outputs into a single bsl binary file
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  2000-08-04  PB %f->%g
  2001-07-07  PB IO_DontReadWrite
  2001-07-08  PB V3.11 FFirst
  2001-07-09  PB V3.12 (pcb->ib)[0].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition
  2001-08-30  PB V4.01 ImageIsFirst, ImageIsLast

---*/
# define Version  "saxs_bsl V4.01 2001-08-30, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define SPrefix "hnam"
# define NPrefix 0
# define SSingle "sngl"
# define NSingle 0
# define SBsl    "bsl"
# define NBsl    1
# define NFrameCounter 0

# define Usage "[options] \n\
                <i1nam> <output header name> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2>"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/*---------------------------------------------------------------------------
ToBslName
Conversion of string to a bsl name of the type BNN000.BSL
All characters are converted to capital letters.  The first three characters
are used for BNN. The first three characters after '.' are taken as extension.
char * sout  output string
char * sin   input string
in NNFlg     0: NN taken from second and third character
             1: NN taken from last two characters before '.'
---------------------------------------------------------------------------*/
void ToBslName ( char * sout, char * sin, int NNFlg )
{ register int i, k;
  char extension[InputLineLength];
  char name[InputLineLength];
  char NN[3];

  /* conversion to capital letters */
  CopyLine( sin, sout, InputLineLength, 1);
  /* extraction of extension */
  RightFL(sout,".",extension,InputLineLength);
  for (i=strlen(extension);i<3;i++) extension[i] = 'B';
  extension[3] = (char) 0;
  /* extraction of number NN*/
  LeftFL(sout,".",name,InputLineLength);
  for (i=0;i<2;i++) NN[i]='0';NN[2]=(char) 0;
  for (k=1,i=strlen(name)-1;(k>=0) && (i>=0);k--,i--)
    if ( (i>0) && 
         ( ((sout[i]>=(int)'0') && (sout[i]<=(int)'9'))||(sout[i]==(int)'%') ) 
       )
      NN[k] = sout[i]; else NN[k] = '0'+ (char) (sout[i]%10);
  /* creation of bsl name */
  sout[0] = sout[0];
  if (NNFlg) {
    sout[1] = NN[0];
    sout[2] = NN[1];
    } else {
    if (((sout[1]<(int)'0')||(sout[1]>(int)'9'))&&(sout[1]!=(int)'%'))
      sout[1]='0';
    if (((sout[2]<(int)'0')||(sout[2]>(int)'9'))&&(sout[2]!=(int)'%'))
      sout[2]='0';
    }
  sout[3] = '0';
  sout[4] = '0';
  sout[5] = '0';
  sout[6] = '.';
  sout[7] = extension[0];
  sout[8] = extension[1];
  sout[9] = extension[2];
  sout[10] = (char) 0;

} /* ToBslName */

/*---------------------------------------------------------------------------
BslFileName
Replacement of nnn in the string BNNnnn.BSL
---------------------------------------------------------------------------*/
void BslFileName ( char * sout, const char * sin, int nnn )
{ int ddd;
  char Temp1[IO_len], Temp2[IO_len];

  filename_parameter(Temp1,IO_len,sin,0);
  filename_name(Temp2, IO_len,Temp1);
  CopyLine( Temp2, Temp2, 11, 1);

  nnn = nnn%1000; if (nnn<0) nnn = nnn+1000;

  ddd = nnn%10; nnn = nnn / 10; Temp2[5] = '0'+ (char) ddd;
  ddd = nnn%10; nnn = nnn / 10; Temp2[4] = '0'+ (char) ddd;
  ddd = nnn%10; nnn = nnn / 10; Temp2[3] = '0'+ (char) ddd;

  sprintf(sout,"%s%s", filename_path(Temp1,IO_len,Temp1),Temp2);

} /* BslFileName */

/*---------------------------------------------------------------------------
Saxs2BslFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void Saxs2BslFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long image_1st, image_lst;
  char linebuffer[IO_len];
  char Path[IO_len], Name[IO_len];

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb,linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib, blkno,ib[blkno-1].Name.V,pstatus);
    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile( pcb,ib,blkno,ib[blkno].Name.V,ib[1].FFirst.V,
                   IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    if (blkno==1) {
      /* Search for minimum and maximum image number in 1st input sequence */
      (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
      if (*pstatus!=Success) return;
    }

  } /* for input files */

  if (block_1st<=0) {
    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect | IO_DontReadWrite | IO_DontOpen;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- File name ib[0].Name of output (not opened) */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",DefaultOutput);
      }

    /*--- Argument  : output bsl name B%%nnn.%%%, fixed length 11 characters */
    if (!pcb->Lin[NPrefix].I) {
      /* Determine prefix and extension from input file name */
      filename_name(pcb->Lin[NPrefix].V,IO_len,ib[1].Name.V);
      pcb->Lin[NPrefix].I = TRUE;
      }

    /* extract output filenumbers from BSL file name */
    extract_filenumbers( &ib[0], &(pcb->Lin[NPrefix]), TRUE, pstatus );
    if (*pstatus!=Success) return;

    // filter bsl file name
    filename_parameter ( Name,IO_len,pcb->Lin[NPrefix].V,0);
    filename_path ( Path, IO_len-10, Name ); //bsl name<10 char
    filename_name ( Name, IO_len, Name );
    ToBslName(Name,Name,TRUE);
    if (strcmp(Path,"./"))
      sprintf(pcb->Lin[NPrefix].V,"%s%s",Path,Name);
     else sprintf(pcb->Lin[NPrefix].V,"%s",Name);

    argv_line( pcb,"Output header name",&pcb->Lin[NPrefix],pcb->Lin[NPrefix].V,
               pstatus);
    if (*pstatus!=Success) return;

    /* extract output filenumbers from BSL file name */
    extract_filenumbers( &ib[0], &(pcb->Lin[NPrefix]), TRUE, pstatus );
    if (*pstatus!=Success) return;

    // filter bsl file name
    filename_parameter ( Name,IO_len,pcb->Lin[NPrefix].V,0);
    filename_path ( Path, IO_len-10, Name ); //bsl name<10 char
    filename_name ( Name, IO_len, Name );
    ToBslName(Name,Name,FALSE);
    if (strcmp(Path,"./"))
      sprintf(pcb->Lin[NPrefix].V,"%s%s",Path,Name);
     else sprintf(pcb->Lin[NPrefix].V,"%s",Name);

    /*--- Take prefix as default for output filename */
    if (!ib[0].Name.I) {
      (void) sprintf(ib[0].Name.V,"%s",pcb->Lin[NPrefix].V);
      ib[0].Name.I = TRUE;
      }

  } /* output file */

  /* ---  Image numbers --- */

  if (first_block==1) {

    /*--- Argument : number of first image */
    argv_long( pcb,"First image of input sequence 1", &ib[1].First, image_1st,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last, image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

    /* Defaults for output file */
    if (!ib[0].Dummy.I) ib[0].Dummy.V  = ihb[1].Dummy.V;
    if (!ib[0].Dim[1].I) ib[0].Dim[1].V = ihb[1].Dim[1];
    if (!ib[0].Dim[2].I) ib[0].Dim[2].V = ihb[1].Dim[2];
  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[1].First.V, pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;

  } /* all other input files */

  if (block_1st <= 0) {
    /*--- Argument : output dummy */
    argv_float( pcb, "Output dummy", &ib[0].Dummy, ib[0].Dummy.V, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    argv_long(pcb,"Output dimension 1", &ib[0].Dim[1],ib[0].Dim[1].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    argv_long(pcb,"Output dimension 2", &ib[0].Dim[2],ib[0].Dim[2].V, pstatus);
    if (*pstatus!=Success) return;

    }

} /* Saxs2BslFilenames */

/*---------------------------------------------------------------------------
1 saxs_bsl

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/

void saxs_bsl (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy;
  float *I1Data;
  float *pI1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1=0.0, W_2=0.0;
  float DW_1=0.0, DW_2=0.0;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  char OutputHeaderName[InputLineLength];
  char OutputBinaryName[InputLineLength];
  FILE *OutputHeaderId;
  FILE *OutputBinaryId;
  int  nn;
  char CC;
  char NN[3];
  size_t t_length;
  char fname_full[IO_len], Temp[IO_len];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\nCalculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* loop over 1 input and 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (pcb->TestBit)
    printf("maxloop_1 = % d, maxloop_2 = % d\n",maxloop_1,maxloop_2);

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  if (pcb->TestBit)
    printf("Wmin_1 = % g, DW_1 = % g, Wmin_2 = % g, DW_2 = % g.\n",
            Wmin_1,DW_1,Wmin_2,DW_2);

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  /* loop over the output array */
  W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
  for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
    W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
    for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);

      if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
        f_1[1], f_2[1], &Value)) {
        /* then do something with the data */
        *pI0Data = Value;
        } /* if (Ipol2d ... */

      W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
      } /* for i_2 ... */
    W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */

 if (!pcb->Flg[NSingle].I) pcb->Flg[NSingle].V = FALSE;

 // Update frame counter 
 if  (ImageIsFirst( pcb, 0, ihb )) pcb->Num[NFrameCounter].V=1l;
  else pcb->Num[NFrameCounter].V+=1l;

 // Generate BSL filename
 filename_parameter (fname_full,IO_len,pcb->Lin[NPrefix].V,0);
 filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[0].FileNumber);
 filename_full (fname_full,IO_len,fname_full);

 /* Reset frame counter */
 if  (ImageIsFirst( pcb, 0, ihb )) pcb->Num[NFrameCounter].V=1l;

 if (!pcb->Flg[NSingle].V) {
   if (pcb->TestBit) printf("Output into multi bsl file\n");

   /* Output into multi bsl file (otoko) format */

   BslFileName ( OutputHeaderName, fname_full, 0 );
   BslFileName ( OutputBinaryName, fname_full, ihb[0].ImNum );

   if (pcb->TestBit) printf("OutputHeaderName = %s\n",OutputHeaderName);
   if (pcb->TestBit) printf("OutputBinaryName = %s\n",OutputBinaryName);

   /* write header file */
//   if (ihb[0].ImNum == pcb->ib[0].First.V) { ++++++++++++++}
   if  (ImageIsFirst( pcb, 0, ihb )) {
     if ((OutputHeaderId=fopen(OutputHeaderName,"w")) == NULL) {
       printf("ERROR : Cannot create % s\n",OutputHeaderName); return; }
     if (ihb[0].Title.I) {
       fprintf(OutputHeaderId,"%80s\n",ihb[0].Title.V);}
       else fprintf(OutputHeaderId,"first header\n");
     fprintf(OutputHeaderId,"second header\n");
     } else if ((OutputHeaderId=fopen(OutputHeaderName,"a")) == NULL) {
       printf("ERROR : Cannot open % s\n",OutputHeaderName); return; }

//     if (ihb[0].ImNum != pcb->ib[0].Last.V) ++++++++++++
     if  ( !ImageIsLast( pcb, 0, ihb ) )
       fprintf(OutputHeaderId,"%8d%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
               I0Dim_1,I0Dim_2,1,0,0,0,0,0,0,1);
       else fprintf(OutputHeaderId,"%8d%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
               I0Dim_1,I0Dim_2,1,0,0,0,0,0,0,0);
     fprintf(OutputHeaderId,"% s\n",
             filename_name(Temp,IO_len,OutputBinaryName));

    fclose(OutputHeaderId);

   /* write binary file */

    t_length = sizeof(float)*(I0Dim_1*I0Dim_2);
    if (pcb->TestBit) printf("Length of output binary file = %d\n",t_length);
  
    printf("Writing %s\n",OutputBinaryName);
    if ((OutputBinaryId=fopen(OutputBinaryName,"w")) == NULL) return;

      fwrite(I0Data,t_length,1,OutputBinaryId);

    fclose(OutputBinaryId);

  } else {         
    if (pcb->TestBit) printf("Output into multi bsl file\n");

   /* Output into multi bsl file (otoko) format */

    BslFileName ( OutputHeaderName, fname_full, 0 );
    BslFileName ( OutputBinaryName, fname_full, 1 );

    if (pcb->TestBit) printf("OutputHeaderName = %s\n",OutputHeaderName);
    if (pcb->TestBit) printf("OutputBinaryName = %s\n",OutputBinaryName);

    /* write header file */
    if ((OutputHeaderId=fopen(OutputHeaderName,"w")) == NULL) {
      printf("ERROR : Cannot create % s\n",OutputHeaderName); return; }
    if (ihb[0].Title.I) {
      fprintf(OutputHeaderId,"%80s\n",ihb[0].Title.V);}
      else fprintf(OutputHeaderId,"first header\n");
    fprintf(OutputHeaderId,"second header\n");

    fprintf(OutputHeaderId,"%8d%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
            I0Dim_1,I0Dim_2,pcb->Num[NFrameCounter].V,0,0,0,0,0,0,0);
    fprintf(OutputHeaderId,"% s\n",filename_name(Temp,IO_len,OutputBinaryName));

    fclose(OutputHeaderId);

   /* write binary file */

    t_length = sizeof(float)*(I0Dim_1*I0Dim_2);
    if (pcb->TestBit) printf("Length of output binary file = %d\n",t_length);

//    if (ihb[0].ImNum == pcb->ib[0].First.V) { +++++++++++++++++
    if  (ImageIsFirst( pcb, 0, ihb )) {
      printf("Writing %s\n",OutputBinaryName);

      if ((OutputBinaryId=fopen(OutputBinaryName,"w")) == NULL) return;

        fwrite(I0Data,t_length,1,OutputBinaryId);

      fclose(OutputBinaryId);
      } else {
      printf("Appending data to %s\n",OutputBinaryName);

      if ((OutputBinaryId=fopen(OutputBinaryName,"a")) == NULL) return;

        fwrite(I0Data,t_length,1,OutputBinaryId);

      fclose(OutputBinaryId);

      }

  }

   printf("\n");

   printf("Input file         : % s\n",
           filename_full(fname_full,IO_len,pcb->ib[1].FileName));
   printf("Input image number : % d\n",ihb[1].ImNum);
   printf("Output header file : % s\n",OutputHeaderName);
   printf("Output binary file : % s\n",OutputBinaryName);

 } /* saxs_bsl*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  Saxs2BslFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("input file         : %s\n",ib[1].Name.V);
  if (pcb->Lin[NPrefix].I)  
                       printf("output header      : %s\n",pcb->Lin[NPrefix].V);
  if (ib[1].First.I)   printf("first image        : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  DefLineOption  ( &cb, SPrefix, NPrefix );
  DefFlgOption  ( &cb, SSingle, NSingle );
  DefFlgOption  ( &cb, SBsl, NBsl );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* copy Bsl to Single */
  if (cb.Flg[NBsl].I) {
    printf("Please use option +/-%s instead of +/-%s\n", SSingle, SBsl);
    if (!cb.Flg[NSingle].I) {
      cb.Flg[NSingle].V = cb.Flg[NBsl].V;
      cb.Flg[NSingle].I = TRUE;
      }
    }

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* RESET FRAME COUNTER */
  cb.Num[NFrameCounter].V=0l;

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_bsl, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

