/*+++
1 saxs_add
  Addition of two image sequences

2 PURPOSE
  Addition of two image sequences

  Arguments:
  saxs_add [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> <i2fst> 
             <odum> <odim1> <odim2> <i1con> <i1fac> <i2con> <i2fac> 
             <ocon> <ofac> "

  Defaults:
  <input sequence 1>  : input.edf 
  <input sequence 2>  : <input sequence 1>
  <output sequence>   : output.edf
  <first image>       : <first image number in sequence 1>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <first image>       : <first image number in sequence 2>
  <output dummy>      : <dummy value in first image of input file>
  <output dimension 1>: <horizontal dimension of first image in input file>
  <output dimension 2>: <vertical dimension of first image in input file>
  <i1fac>             : 1.0 input 1 multiplication factor
  <i1con>             : 0.0 input 1 addition constant
  <i2fac>             : 1.0
  <i2con>             : 0.0
  <ofac>              : 1.0
  <ocon>              : 0.0

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  28-Nov-1995 PB
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput
  04-Aug-2000 PB %f->%g
  2000-12-30  PB V3.1 UPDATE
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "saxs_add V4.00 2001-07-09, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define Usage "[options] \n\
             <i1nam> <onam> <i1fst> <i1lst> <i1inc> <i2fst>\n\
             <odum> <odim1> <odim2> <i1con> <i1fac> <i2con> <i2fac>\n\
             <ocon> <ofac> "

# define BlockNum 3       /* 2 input sequences + 1 output sequence */

/*---------------------------------------------------------------------------
1 saxs_add

2 PURPOSE
  Addition of two images
---------------------------------------------------------------------------*/

void saxs_add (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ const int NumberOfImages=3;
  int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy, I0DDummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float I1Value;
  float *I2Data;
  int   I2Dim_1,I2Dim_2;
  float I2Dummy, I2DDummy;
  float I2Value;

  /* Do not change the following definitions,
     they are automatically changed with BlockNum */
  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];
  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];
  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;
  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];
  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];
  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");

  for(i=1;i<imax;i++) {
    if (pcb->TestBit) PrintImageHeaderBlock(i,&ihb[i]);}

 /* Check the number of images */
  if (pcb->ImgBlkLen!=NumberOfImages) {
     printf("%d images found, 2 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) return;

     GetImageOverlap       ( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 2 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];
  I2Data   = ihb[2].Data;
  I2Dummy  = ihb[2].Dummy.V;
  I2DDummy = ihb[2].DDummy.V;
  I2Dim_1  = (int) ihb[2].Dim[1];
  I2Dim_2  = (int) ihb[2].Dim[2];

 /* loop over 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  /* loop over the output array */
  W_1 = Wmin_1; for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
  for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
    W_2 = Wmin_2; for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
    for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {

      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,i_1,i_2);

      if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
            f_1[1], f_2[1], &I1Value ) &&
          Ipol2d (I2Data,I2Dim_1,I2Dim_2,I2Dummy,I2DDummy,
            f_1[2], f_2[2], &I2Value )){

        /* then add both values */
        UPDATE( *pI0Data, I1Value + I2Value, I0Dummy, I0DDummy );

        } /* if (Ipol2d ... */

      W_2+=DW_2; for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
      } /* for i_2 ... */
    W_1+=DW_1; for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
    } /* for i_1 ... */

 } /* saxs_add*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<1st sequence> = <read 1st sequence> * Factor + Const\n");
  argv_float(pcb,"Multiplication factor",&ib[1].Factor,1.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  argv_float(pcb,"Addition constant",&ib[1].Const,0.0,pstatus);
  if (*pstatus!=Success) return;
 
  /*--- Argument  : multiplication factor */
  printf("<2nd sequence> = <read 2nd sequence> * Factor + Const\n");
  argv_float(pcb,"Multiplication factor",&ib[2].Factor,1.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  argv_float(pcb,"Addition constant",&ib[2].Const,0.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : multiplication factor */
  printf("<output sequence> = (<1st sequence>+<2nd sequence>)*Factor+Const\n");
  argv_float(pcb,"Multiplication factor",&ib[0].Factor,1.0,pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : additive constant */
  argv_float(pcb,"Addition constant",&ib[0].Const,0.0,pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)    printf("i/p file 1         : %s\n",ib[1].Name.V);
  if (ib[2].Name.I)    printf("i/p file 2         : %s\n",ib[2].Name.V);
  if (ib[0].Name.I)    printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)   printf("first image in 2   : %d\n",ib[1].First.V);
  if (ib[1].Last.I)    printf("last image         : %d\n",ib[1].Last.V);
  if (ib[1].Inc.I)     printf("increment          : %d\n",ib[1].Inc.V);
  if (ib[2].First.I)   printf("image in 2         : %d\n",ib[2].First.V);
  if (ib[0].Dummy.I)   printf("output dummy       : %g\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)  printf("output dimension 1 : %d\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)  printf("output dimension 2 : %d\n",ib[0].Dim[2].V);
  if (ib[1].Factor.I)  printf("i/p 1 factor       : %g\n",ib[1].Factor.V);
  if (ib[1].Const.I)   printf("i/p 1 constant     : %g\n",ib[1].Const.V);
  if (ib[2].Factor.I)  printf("i/p 2 factor       : %g\n",ib[2].Factor.V);
  if (ib[2].Const.I)   printf("i/p 2 constant     : %g\n",ib[2].Const.V);
  if (ib[0].Factor.I)  printf("output factor      : %g\n",ib[0].Factor.V);
  if (ib[0].Const.I)   printf("output constant    : %g\n",ib[0].Const.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
} /* user_io */

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, saxs_add, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

