# define Version  "edfdump 1.6 2001-06-13, Peter Boesecke"
/*---------------------------------------------------------------------------
NAME
  edfdump - print parameters of an edf file

SYNOPSIS
  edfdump [-h] [-e <keyword>] <files...>

DESCRIPTION
  Edfdump opens the named input <files> as edf files, prints the data 
  block numbers and searches in each header for <keywords>.  

  -h, --help    print help
  -b <block>    block key, e.g. 1 (default: all blocks)
  -c <chain>    chain key, e.g. Image.Psd (default: all chains)
  -e <keyword>  input of a header keyword to look for, by repeting this
                option several keywords can be defined.
  -s            print filename, chain key, block key and keywords in 
                a single line separated by <tab>, if no keywords are 
                given print file name, chain key and all block keys 
                in a single line
  -S            write only keywords separated by <tab>
  -i            ignore non-edf files
  -I            show names of edf files only 

HISTORY
  V1.0 2001-05-19 Peter Boesecke
  V1.2 2001-05-21 PB multiple files, non edf-files are skipped
  V1.4 2001-05-31 PB
  V1.5 2001-06-02 PB
  V1.6 2001-06-13 PB -I -> \r\n
---------------------------------------------------------------------------*/

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "edfio.h"

static const char * OptionStrings[11] =
                     { "Invalid",
                       "-h",  "--help",  "-b",     "-c",     "-e",
                       "-s",  "-S",      "-i",     "-I", 
                       (const char *) NULL };

enum Option          { InValidOption,
                       Help1, Help2,     BlockKey, ChainKey, Keyword, 
                       Short, VeryShort, Ignore,   IgnoreShort,
                       EndOption };

/*---------------------------------------------------------------------------
NAME
  opt2num 

SYNOPSIS

  int opt2num( const char * string )

DESCRIPTION
  Returns the option number according to OptionStrings

RETURN VALUE
  0: not found, otherwise option number
---------------------------------------------------------------------------*/
int opt2num( const char * string )
{
  int NE=1,optno=0;
  while ( (NE && OptionStrings[optno]) )
    NE = strcmp( string , OptionStrings[optno++] );
  if (NE) optno = 0; else optno--;

  return(optno);

} // opt2num

void edfdumphelp( FILE * out, const char * progname, int verbose ) 
{ 
  fprintf(out,"Usage: %s [-h] [options] [-e <keyword>] <files...>\n",progname);
  fprintf(out,"Version: %s\n",Version);
  
  if (!verbose) return;
  fprintf(out,"Data format version: %s\n",edf_dataformat_version());
  fprintf(out,"   -h : this help\n");
  fprintf(out,"   -c <chainkey> -b <blockkey> -e <keyword>\n");
  fprintf(out,"   -s : short format, -S : print keywords only\n");
  fprintf(out,"   -i : ignore non-edf files, -I : show edf file names\n");

} // edfdumphelp

/*---------------------------------------------------------------------------
NAME
  edfdump

SYNOPSIS

  int edfdump( char * fname[], int ignore, int format, 
               char * chainkey[], char * blockkey[], char * keyword[] );

DESCRIPTION
  edfdump to stdout for all files in fname[], fname[] must stop with NULL

ARGUMENTS
  char * fname[]   : table of filenames, terminated with NULL
  int ignore       :  1 : ignore non-edf files
                      0 : normal
  int format       : output format 
  char * keyword[] : table of keywords, terminated with NULL

RETURN VALUE
  0, if OK, otherwise error
---------------------------------------------------------------------------*/
int edfdump( char * fname[], int ignore, int format, 
             char * chainkey[], char * blockkey[], char * keyword[] )
{int status, ErrorValue;
 int stream;

 for (;*fname!=(char*) NULL;fname++) {
   stream = edf_open_data_file ( *fname, "read", &ErrorValue, &status );
   if (status) { 
     if (!ignore) printf("%s: no edf file\r\n", *fname); 
     // release eventually allocated memory
     edf_free_all ( );
    } else {
     if (format==1001) fprintf(stdout,"%s\r\n", *fname);
       else edf_dump ( stdout, stream, format, chainkey, blockkey, keyword );

//     edf_print_filetable ( stdout, 4, 0 ); //++++++++++++++++++++

     edf_close_data_file( stream, &ErrorValue, &status );
     if (status) { 
       printf("%s\n",edf_report_data_error(ErrorValue));return(-1); }
     }
   }

 return(0);

} // edfdump

/*---------------------------------------------------------------------------
NAME
  main 
---------------------------------------------------------------------------*/
main (int argc, char *argv[])
{ char *nul = (char *) NULL;
  char **ckeyv, **bkeyv, **keyv;
  char **pargv, **pckeyv, **pbkeyv, **pkeyv;
  int optno, ckeyc, bkeyc, keyc;
  int format=0, ignore=0;

  // count options or help or help
  pargv = &argv[1];
  ckeyc = 1;
  bkeyc = 1;
  keyc  = 1;
  for (;*pargv!=nul && (**pargv=='-');*pargv++) {
    switch ( opt2num( *pargv ) ) {
       case   Help1  :
       case   Help2  : edfdumphelp( stdout, argv[0], 1 ); exit(0); break; 
       case ChainKey : if (pargv[1]!=nul) { ckeyc++; *(++pargv); } break;
       case BlockKey : if (pargv[1]!=nul) { bkeyc++; *(++pargv); } break;
       case Keyword  : if (pargv[1]!=nul) { keyc++; *(++pargv); } break; 
       default       : break;
      }
    }

  // allocate memory for chainkeys
  if (!(ckeyv = (char **) malloc( sizeof(size_t) * ckeyc ))) {
    perror("malloc( sizeof(size_t) * ckeyc )");
    exit(-1);
    }
  // allocate memory for blockkeys
  if (!(bkeyv = (char **) malloc( sizeof(size_t) * bkeyc ))) {
    perror("malloc( sizeof(size_t) * bkeyc )");
    exit(-1);
    }
  // allocate memory for keywords 
  if (!(keyv = (char **) malloc( sizeof(size_t) * keyc ))) {
    perror("malloc( sizeof(size_t) * keyc )");
    exit(-1);
    }

  // get options
  pargv  = &argv[1];
  pckeyv = ckeyv;
  pbkeyv = bkeyv;
  pkeyv  = keyv;
  for (;*pargv!=nul && (**pargv=='-');*pargv++) {
    optno = opt2num( *pargv );
    switch ( optno ) {
       case Help1     :
       case Help2     : break;
       case ChainKey  : if (pargv[1]!=nul) *pckeyv++ = *(++pargv); break;
       case BlockKey  : if (pargv[1]!=nul) *pbkeyv++ = *(++pargv); break;
       case Keyword   : if (pargv[1]!=nul) *pkeyv++  = *(++pargv); break;
       case Short     : format=1; break; 
       case VeryShort : format=2; break;
       case Ignore    : ignore=1; break;
       case IgnoreShort : ignore=1; format=1001; break;
       default        : break;
      }
    }
  *pckeyv = nul;
  *pbkeyv = nul;
  *pkeyv  = nul;

  if (*pargv) edfdump( pargv, ignore, format, ckeyv, bkeyv, keyv );
    else edfdumphelp( stdout, argv[0], 0 );

  free( ckeyv );
  free( bkeyv );
  free( keyv );

} // main

