/*+++
1 binary2saxs
  Conversion of a binary file into ESRF data format.

2 PURPOSE
  Conversion of a binary file into ESRF data format. 

  Arguments:
  binary2saxs [options] <fnam> <onam> <i1dim1> <i1dim2> <fskp> <ftyp> <odum>"

  Defaults:
  <binary file name>  : datafile 
  <output file name>  : output.edf
  <bytes to skip>     : 0 
  <binary type>       : float
  <input dim1>        : <input dimension 1>
  <input dim2>        : <input dimension 2>
  <dummy>             : <dummy value>

  Wildcard:
  A "=" in the argument list chooses the default.

2 HISTORY
  26-Jan-1996 PB
  27-Jan-1996 PB ReadImageDimension
  28-Jan-1996 PB Binning, Transform
  29-Mar-1996 PB Clip, foff
  22-May-1996 PB Gnaw
  12-Dec-1997 PB byte
  19-Dec-1997 PB Small corrections V3.32
  03-Jan-1998 PB char -> unsigned char (correction) V3.33
  12-Jan-1998 PB binary2saxs: data_size = MAX2(t_size, ...
                 -> data_size = MAX2(t_length, ...  V3.34
  12-Jan-1998 PB All free() calls have been changed to
                 free(xyz);xyz=NULL;
  15-Jan-1998 PB +-fend: endian correction of the input data (byte swap) V3.35
  26-Jun-1999 PB DevSeparator, DirSeparator, ExtSeparator defined in SaxsInput 
  09-Nov-1999 PB V3.36 DUMMYDEFINED updated
  2000-08-04  PB %f -> %g
  2001-07-03  PB V3.37 filename_exists(..), filename_full( )
  2001-07-08  PB V3.38 FFirst
  2001-07-09  PB V3.39 (pcb->ib)[1].FileNumber
  2001-07-09  PB V4.00 new loop, no repetition

---*/
# define Version  "binary2saxs V4.00 2001-07-09, Peter Boesecke"
# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>

# include "SaxsPrograms.h"

# define NFname 0           /* line option */
# define SFname "fnam"
# define NFType 1
# define SFType "ftyp"

# define NFSkip 0           /* long int option */
# define SFSkip "fskp"

/* Special options */
# define NFend 0
# define SFend "fend" 

# define Default_opfn      DefaultOutput
# define Default_inc       1
# define Default_dim_1     512
# define Default_dim_2     512
# define Default_fend      FALSE

# define Usage "[options] \n\
       <fnam> <onam> <i1dim1> <i1dim2> <fskp> <ftyp> <odum>\n\
       The following types are allowed : \n\
       ftyp float, byte, short integer, integer, long integer,\n\
            unsigned short integer, unsigned integer, unsigned long integer"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

const char * STypeFloat    = "float";
const char * STypeShort    = "short integer";
const char * STypeInteger  = "integer";
const char * STypeLong     = "long integer";
const char * STypeUShort   = "unsigned short integer";
const char * STypeUInteger = "unsigned integer";
const char * STypeULong    = "unsigned long integer";
const char * STypeByte     = "byte";

enum BinaryDataType { TypeNone=0, TypeFloat, TypeShort, TypeInteger, TypeLong,
                      TypeUShort, TypeUInteger, TypeULong , TypeByte };

/*---------------------------------------------------------------------------
  NAME

  byteswap - byte swapping inside an item

  SYNOPSIS

  void byteswap ( void * dest, const void * src, size_t item, unsigned long n );

  DESCRIPTION
  Swaps all bytes inside an item of src consisting of item bytes and 
  writes the result to the output buffer dest. The output and input 
  buffers can be identical. The minimum size of both buffers in bytes 
  is item * n.

         byte     byte     byte     byte    byte     byte     byte     byte
  src  :    0   |    1   |    2   |   3   |  ...  | item-3 | item-2 | item-1 |
  dest : item-1 | item-2 | item-3 |  ...  |   3   |    2   |    1   |    0   |

  ARGUMENTS

  void     * dest    ouput buffer<br>
  void     * src     input buffer<br>
  size_t   item      size of a single item in bytes (must be an even number)<br>
  unsigned long n    number of items<br>   
 
  AUTHOR
   Peter Boesecke 14-Jan-1998
  --------------------------------------------------------------------------*/
void byteswap ( void * dest, const void * src, size_t item, unsigned long n )
{ 
  const unsigned char *in; 
  const unsigned char *pi1, *pi2; 
  unsigned char       *out;
  unsigned char       *po1, *po2, temp;
  unsigned long        i, j;
  size_t               step = item;
  unsigned long        jmax = (step + 1u) / 2u;

  pi1 = in  = (const unsigned char *) src; 
  pi2 = in + step;
  po1 = out = (unsigned char *) dest; 
  po2 = out + step;

  for (i=0;i<n;i++) {
    /* swap */
    for (j=0;j<jmax;j++) { temp = *pi1++; *po1++ = *--pi2; *--po2 = temp; } 
    pi1 =  in += step; pi2 =  in + step;
    po1 = out += step; po2 = out + step;
    } /* for */

} /* byteswap */

/*---------------------------------------------------------------------------
1 convert2float 

2 PURPOSE
  Converts Dim_1 * Dim_2 InType data in In to float data in Out.
  In and Out can be the same array. Enough reserve memory must be 
  available.

  return value  : 0 	no error
                  -1    error.
  void * In (i)      : pointer to input data
  long InType        : BinaryDataType of In
  Dim_1, Dim_2 (i)   : dimensions of In
  float * Out (o)    : output float array 
  t_size OutSize (i) : allocated memory for Out (bytes)
---------------------------------------------------------------------------*/
int convert2float ( void * In, long InType, long Dim_1, long Dim_2,
                    float * Out, size_t OutSize )
{ register long i;
  float * pout;
  float * pinfloat;
  short int * pinshort;
  int * pinint;
  long int * pinlong;
  unsigned short int * pinushort;
  unsigned int * pinuint;
  unsigned long int * pinulong;
  unsigned char * pinbyte;

  size_t t_out, t_in;   

  if (OutSize < (sizeof( float ) * (Dim_1*Dim_2)) ) return(-1);

  t_out = sizeof(float); 

  switch (InType) {
    case TypeFloat   : if (In!=Out) {
                          t_in = sizeof(float);
                          pinfloat = (float *) In + Dim_1*Dim_2;
                          pout = Out + Dim_1*Dim_2;
                          for (i=Dim_1*Dim_2;i>0;--i) 
                            *(--pout) = *(--pinfloat);  
                          }
                       break;
    case TypeShort   : t_in = sizeof(short int);
                       if (t_in<t_out) {
                         pinshort = (short int *) In + Dim_1*Dim_2;
                         pout = Out + Dim_1*Dim_2;
                         for (i=Dim_1*Dim_2;i>0;--i) 
                           *(--pout) = (float) *(--pinshort);
                         } else {
                         pinshort = (short int *) In;
                         pout = Out;
                         for (i=0;i<Dim_1*Dim_2;i++)
                           *(pout++) = (float) *(pinshort++);
                         }
                       break;
    case TypeInteger : t_in = sizeof(int );
                       if (t_in<t_out) {
                         pinint = (int *) In + Dim_1*Dim_2;
                         pout = Out + Dim_1*Dim_2;
                         for (i=Dim_1*Dim_2;i>0;--i)
                           *(--pout) = (float) *(--pinint);
                         } else {
                         pinint = (int *) In;
                         pout = Out;
                         for (i=0;i<Dim_1*Dim_2;i++)
                           *(pout++) = (float) *(pinint++);
                         }
                       break;
    case TypeLong    : t_in = sizeof(long int);
                       if (t_in<t_out) {
                         pinlong = (long int *) In + Dim_1*Dim_2;
                         pout = Out + Dim_1*Dim_2;
                         for (i=Dim_1*Dim_2;i>0;--i)
                           *(--pout) = (float) *(--pinlong);
                         } else {
                         pinlong = (long int *) In;
                         pout = Out;
                         for (i=0;i<Dim_1*Dim_2;i++)
                           *(pout++) = (float) *(pinlong++);
                         }
                       break;
    case TypeUShort  : t_in = sizeof(unsigned short int);
                       if (t_in<t_out) {
                         pinushort = (unsigned short int *) In + Dim_1*Dim_2;
                         pout = Out + Dim_1*Dim_2;
                         for (i=Dim_1*Dim_2;i>0;--i)
                           *(--pout) = (float) *(--pinushort);
                         } else {
                         pinushort = (unsigned short int *) In;
                         pout = Out;
                         for (i=0;i<Dim_1*Dim_2;i++)
                           *(pout++) = (float) *(pinushort++);
                         }
                       break;
    case TypeUInteger : t_in = sizeof(unsigned int );
                       if (t_in<t_out) {
                         pinuint = (unsigned int *) In + Dim_1*Dim_2;
                         pout = Out + Dim_1*Dim_2;
                         for (i=Dim_1*Dim_2;i>0;--i)
                           *(--pout) = (float) *(--pinuint);
                         } else {
                         pinuint = (unsigned int *) In;
                         pout = Out;
                         for (i=0;i<Dim_1*Dim_2;i++)
                           *(pout++) = (float) *(pinuint++);
                         }
                       break;
    case TypeULong   : t_in = sizeof(unsigned long int);
                       if (t_in<t_out) {
                         pinulong = (unsigned long int *) In + Dim_1*Dim_2;
                         pout = Out + Dim_1*Dim_2;
                         for (i=Dim_1*Dim_2;i>0;--i)
                           *(--pout) = (float) *(--pinulong);
                         } else {
                         pinulong = (unsigned long int *) In;
                         pout = Out;
                         for (i=0;i<Dim_1*Dim_2;i++)
                           *(pout++) = (float) *(pinulong++);
                         }
                       break;
    case TypeByte    : t_in = sizeof(char);
                       if (t_in<t_out) {
                         pinbyte = (unsigned char *) In + Dim_1*Dim_2;
                         pout = Out + Dim_1*Dim_2;
                         for (i=Dim_1*Dim_2;i>0;--i)
                           *(--pout) = (float) *(--pinbyte);
                         } else {
                         pinbyte = (unsigned char *) In;
                         pout = Out;
                         for (i=0;i<Dim_1*Dim_2;i++)
                           *(pout++) = (float) *(pinbyte++);
                         }
                       break;

    default          : return(-1);
    }

  return( 0 );
} /* convert2float */ 

/*---------------------------------------------------------------------------
1 showdatatypes 

2 PURPOSE
  shows the possible data types
---------------------------------------------------------------------------*/
void showdatatypes ( void )
{ printf("possible data types : \n");
  printf("%s (%d bytes)\n",STypeFloat,    sizeof( float ) ); 
  printf("%s (%d bytes)\n",STypeShort,    sizeof( short int ) );
  printf("%s (%d bytes)\n",STypeInteger,  sizeof( int ) );
  printf("%s (%d bytes)\n",STypeLong,     sizeof( long int ) );
  printf("%s (%d bytes)\n",STypeUShort,   sizeof( unsigned short int ) );
  printf("%s (%d bytes)\n",STypeUInteger, sizeof( unsigned int ) );
  printf("%s (%d bytes)\n",STypeULong,    sizeof( unsigned long int ) );
  printf("%s (%d bytes)\n",STypeByte,     sizeof( unsigned char ) );


  return;
} /* showdatatypes */

/*---------------------------------------------------------------------------
1 string2type 

2 PURPOSE
  Returns the type number that corresponds to the data type string, 
  0 if no match found.

  return value	: number of matches 
  char * typestring (i) : string to be analyzed
  long * datatype (o)   : binary data type number 
---------------------------------------------------------------------------*/
int string2type ( char * typestring, long * datatype )
{ int cnt;
  
  cnt = 0;
  *datatype = TypeNone;

  if (STypeFloat == strstr(STypeFloat,typestring)) {
     cnt++;*datatype = (long) TypeFloat;}
  if (STypeShort == strstr(STypeShort,typestring)) {
     cnt++;*datatype = (long) TypeShort;}
  if (STypeInteger == strstr(STypeInteger,typestring)) {
     cnt++;*datatype = (long) TypeInteger;}
  if (STypeLong == strstr(STypeLong,typestring)) {
     cnt++;*datatype = (long) TypeLong;}
  if (STypeUShort == strstr(STypeUShort,typestring)) {
     cnt++;*datatype = (long) TypeUShort;}
  if (STypeUInteger == strstr(STypeUInteger,typestring)) {
     cnt++;*datatype = (long) TypeUInteger;}
  if (STypeULong == strstr(STypeULong,typestring)) {
     cnt++;*datatype = (long) TypeULong;}
  if (STypeByte == strstr(STypeByte,typestring)) {
     cnt++;*datatype = (long) TypeByte;}

  return(cnt);

} /* string2type */

/*---------------------------------------------------------------------------
1 binary2saxs

2 PURPOSE
  Multiplicates an image with a factor and adds a constant
---------------------------------------------------------------------------*/
void binary2saxs (CmdBlk * pcb, ImgHeadBlk ihb[], int * pstatus)
{ int i,imax;
  int j;

  float *I0Data;
  int   I0Dim_1,I0Dim_2;
  float *pI0Data;
  float I0Dummy;
  float *I1Data;
  int   I1Dim_1,I1Dim_2;
  float I1Dummy, I1DDummy;
  float Value;

  int maxloop_1, maxloop_2;
  int i_1, i_2;
  float W_1, W_2;
  float DW_1, DW_2;
  float f_1[BlockNum], f_2[BlockNum];
  float Df_1[BlockNum], Df_2[BlockNum];

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  float Wmin_1, Wmax_1;
  float Wmin_2, Wmax_2;

  float fmin_1[BlockNum], fmin_2[BlockNum];
  float fmax_1[BlockNum], fmax_2[BlockNum];

  int Imin_1[BlockNum], Imin_2[BlockNum];
  int Imax_1[BlockNum], Imax_2[BlockNum];

  float RImin_1[BlockNum], RImin_2[BlockNum];
  float RImax_1[BlockNum], RImax_2[BlockNum];

  char fname_full[IO_len];

  FILE * InputBinaryID;
  size_t t_length, t_size, data_size;
  long FSkip;
  long FType;
  long int Dim_1, Dim_2;
  long int Bin_1, Bin_2; 
  float MinClip, MaxClip;
  int MinClipI, MaxClipI;
  long GnawCol, GnawRow;
  float Factor, Constant;

  *pstatus = Success;

 /* Check the number of image blocks */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* --- read binary file and convert to float */

     FSkip = pcb->Num[NFSkip].V;

     if (string2type(pcb->Lin[NFType].V,&FType)!=1) {
       printf("Type : %d\n",FType);
       *pstatus = UnknownDataType; return; }

     Bin_1   = pcb->ib[1].Bin[1].V;  Bin_2    = pcb->ib[1].Bin[2].V;
     MinClip = pcb->ib[1].MinClip.V; MinClipI = pcb->ib[1].MinClip.I;
     MaxClip = pcb->ib[1].MaxClip.V; MaxClipI = pcb->ib[1].MaxClip.I;
     Factor  = pcb->ib[1].Factor.V;  Constant = pcb->ib[1].Const.V;
     GnawCol = pcb->ib[1].Gnaw[1].V; GnawRow  = pcb->ib[1].Gnaw[2].V;

     (void) ReadImageDimension ( pcb, pcb->ib, 1, ihb[1].ImNum,
                                 &Dim_1, &Dim_2, pstatus );
     if (*pstatus!=Success) return;

     switch (FType) {
       case TypeFloat   : t_size = sizeof(float); break;
       case TypeShort   : t_size = sizeof(short int); break;
       case TypeInteger : t_size = sizeof(int); break;
       case TypeLong    : t_size = sizeof(long int); break;
       case TypeUShort  : t_size = sizeof(unsigned short int); break;
       case TypeUInteger: t_size = sizeof(unsigned int); break;
       case TypeULong   : t_size = sizeof(unsigned long int); break;
       case TypeByte    : t_size = sizeof(char); break;

       default : *pstatus = UnknownDataType; return;
       } /* switch */

       t_length = t_size*(Dim_1*Dim_2);
       data_size = MAX2(t_length, sizeof(float)*(Dim_1*Dim_2)); 

     if (pcb->TestBit) {
       printf("Bytes to skip (bytes)      = %d\n",FSkip);
       printf("Data type                  = %d\n",FType);
       printf("Data size (bytes)          = %d\n",t_size);
       printf("Binary data length (bytes) = %d\n",t_length);
       printf("Allocated memory (bytes)   = %d\n",data_size);
       printf("Dim_1                      = %d, Dim_2 = %d\n",Dim_1, Dim_2);
       printf("Bin_1                      = %d, Bin_2 = %d\n",Bin_1, Bin_2);
       if (MinClipI) printf("MinClip                    = %g\n",MinClip);
       if (MaxClipI) printf("MaxClip                    = %g\n",MaxClip);
       printf("GnawCol                    = %d, GnawRow = %d\n",
	       GnawCol, GnawRow);
       printf("Factor                     = %g, Const = %g\n",Factor,Constant);
       }

    /* Generate filename of binary file */
    filename_parameter (fname_full,IO_len,pcb->Lin[NFname].V,0);
    filename_pattern (fname_full,IO_len,fname_full,(pcb->ib)[1].FileNumber);
    filename_full (fname_full,IO_len,fname_full);

     printf("Reading %s\n",fname_full);
     if ( (InputBinaryID=fopen(fname_full,"rb")) == NULL ) {
       *pstatus = FileOpeningError; return; }

     if ( fseek(InputBinaryID, FSkip, SEEK_SET) ) {
       *pstatus = FilePositioningError; return; }

     if ( (ihb[1].Data = (float *) malloc(data_size) ) == NULL ) {
       *pstatus = NotEnoughMemoryAvailable; return;}

       (void) fread(ihb[1].Data,t_length,1,InputBinaryID);
       if (ferror(InputBinaryID)) {
         *pstatus = FileReadError; free(ihb[1].Data); ihb[1].Data=NULL;
         ihb[ 1].ImgByteLen = (long) NULL; return;
         }

     fclose(InputBinaryID);

     /* endian correction */
     if (pcb->Flg[NFend].V) {
       if (pcb->TestBit) printf(" Swapping bytes \n");
       byteswap ( ihb[1].Data, ihb[1].Data, t_size, Dim_1*Dim_2 );
       }

     /* conversion to float */
     if (pcb->TestBit) printf(" Converting to float \n");

     if ( convert2float ( ihb[1].Data, FType, Dim_1, Dim_2,
                          ihb[1].Data, data_size ) ) {
        printf("ERROR : Type conversion failed\n"); exit(-1); }

     /* conversion done */

     ihb[1].ImgByteLen = (long) t_length; 
     ihb[1].Dim[1] = Dim_1; ihb[1].Dim[2] = Dim_2;

     ClipFloat2d           ( ihb[1].Data, ihb[1].Dim[1], ihb[1].Dim[2],
                             ihb[1].Dummy.V, ihb[1].DDummy.V,
                             MinClip, MinClipI, MaxClip, MaxClipI );

     GnawFloat2d ( ihb[1].Data, ihb[1].Dim[1], ihb[1].Dim[2], 
		   ihb[1].Dummy.V, ihb[1].DDummy.V,
		   GnawCol, GnawRow, pstatus );
     if (*pstatus!=Success) return;

     RebinFloat2d          ( ihb[1].Data, ihb[1].Data,
                             &ihb[1].Dim[1], &ihb[1].Dim[2],
                             ihb[1].Dummy.V, ihb[1].DDummy.V,
                             Bin_1, Bin_2);

     TransformFloat2d      ( ihb[1].Data, ihb[1].Dim[1], ihb[1].Dim[2],
                             ihb[1].Dummy.V, ihb[1].DDummy.V,
                             Factor, Constant );

     if (pcb->TestBit) PrintImageHeaderBlock   ( 1, &ihb[1] );

  /* binary file is read */

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,% d] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[% d,% d] ", i, ihb[i].ImNum); }

  printf(")\n\n");


     GetReferenceParameters( pcb, ihb, 0, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2, pstatus );
     if (*pstatus!=Success) { 
         free(ihb[1].Data); ihb[1].Data=NULL;
         ihb[ 1].ImgByteLen = (long) NULL; return;
         }

     GetImageOverlap       ( pcb, ihb, 1, imax-1,
                             Off_1, Off_2, Ps_1,  Ps_2,
                             fmin_1, fmax_1, fmin_2, fmax_2,
                             &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  I0Dummy = ihb[0].Dummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

 /* loop over 1 input and 1 output image,
     calculate start integer indices  */
  IDX(fmin_1[0],Imin_1[0],RImin_1[0]);
  IDX(fmin_2[0],Imin_2[0],RImin_2[0]);
  IDX(fmax_1[0],Imax_1[0],RImax_1[0]);
  IDX(fmax_2[0],Imax_2[0],RImax_2[0]);

  if (pcb->TestBit) for (i=0;i<1;i++) {
    printf("Imin_1[% d] = % d, Imax_1[% d] = % d\n",i,Imin_1[i],i,Imax_1[i]);
    printf("Imin_2[% d] = % d, Imax_2[% d] = % d\n",i,Imin_2[i],i,Imax_2[i]);
    } /* for (i ... */

  /* calculate the delta values of the world coordinates */
  maxloop_1 = Imax_1[0]-Imin_1[0]+1;
  maxloop_2 = Imax_2[0]-Imin_2[0]+1;

  if (maxloop_1>1) DW_1 = (Wmax_1-Wmin_1)/(maxloop_1-1); else DW_1=0.0;
  if (maxloop_2>1) DW_2 = (Wmax_2-Wmin_2)/(maxloop_2-1); else DW_2=0.0;

  /* calculate the delta values of the indices */
  for (i=0;i<imax;i++) {
      if (maxloop_1>1)
        Df_1[i]=(fmax_1[i]-fmin_1[i])/(maxloop_1-1); else Df_1[i]=0.0;
      if (maxloop_2>1)
        Df_2[i]=(fmax_2[i]-fmin_2[i])/(maxloop_2-1); else Df_2[i]=0.0;
      } /* for (i= ... */

  /* loop over the output array  */
  if (DUMMYDEFINED(I1Dummy,I1DDummy)) {
    for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
    for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1[0],i_2);
      for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
      for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
        if (Ipol2d (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
          f_1[1], f_2[1], &Value)) {
          /* then do something with the data */
          *pI0Data = Value;
          } /* if (Ipol2d ... */
        pI0Data++;
        for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
        } /* for i_1 ... */
      for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
      } /* for i_2 ... */
    } else {
    for (i=0;i<imax;i++) f_2[i]=fmin_2[i];
    for (i_2=Imin_2[0];i_2<=Imax_2[0];i_2++) {
      pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,Imin_1[0],i_2);
      for (i=0;i<imax;i++) f_1[i]=fmin_1[i];
      for (i_1=Imin_1[0];i_1<=Imax_1[0];i_1++) {
        (void) Ipol2 (I1Data,I1Dim_1,I1Dim_2,f_1[1], f_2[1], &Value);
        *pI0Data++ = Value;
        for (i=0;i<imax;i++) { f_1[i]+=Df_1[i]; }
        } /* for i_1 ... */
      for (i=0;i<imax;i++) { f_2[i]+=Df_2[i]; }
      } /* for i_2 ... */
    } /* if (DUMMYDEFINED( ... */

  free(ihb[1].Data); ihb[1].Data = NULL;
  ihb[ 1].ImgByteLen = (long) NULL;

 } /* binary2saxs*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[InputLineLength];
  ImgHeadBlk ihb[BlockNum];
  char fname_pattern[IO_len];

  long  image_1st, image_lst;
  long  i1dim_1, i1dim_2;

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  long datatype;

 /* Determine program name without directory */
   (void) RightFR((char *) pcb->argv[0],DirSeparator,progfn,InputLineLength);

  /* show usage if no arguments are given */
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,pcb->MainHelp);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,pcb->MainVersion);
  printf("\n");

  /*--- OpenMode of first input file */
  if (!ib[1].OpenMode.I) {
    ib[1].OpenMode.V = IO_Old | IO_FileProtect | IO_DontReadWrite | IO_DontOpen;
    ib[1].OpenMode.I = TRUE;
    }
  /* Under no circumstances read any data */
  ib[1].OpenMode.V = ib[1].OpenMode.V | IO_DontReadData; 

  /*--- Argument  : binary file name */
  if (!(pcb->Lin[NFname].I)) {
    /* Determine input file name without directory and without extension */
    filename_name(pcb->Lin[NFname].V,IO_len,ib[1].Name.V);
    filename_body(pcb->Lin[NFname].V,IO_len,pcb->Lin[NFname].V);
    }

  argv_line(pcb,"Binary file name",&pcb->Lin[NFname],pcb->Lin[NFname].V,
    pstatus);
  /* Exit in all cases after an abort from the first argument */
  if (*pstatus!=Success) return;

  /* extract filenumbers from binary file name */
  extract_filenumbers( &ib[1], &(pcb->Lin[NFname]), TRUE, pstatus );
  if (*pstatus!=Success) return;

  /* Generate filename of first binary file and test, whether it exists */
  filename_parameter (fname_pattern,IO_len,pcb->Lin[NFname].V,0);
  filename_pattern (fname_pattern,IO_len,fname_pattern,ib[1].FFirst.V);
   if (!filename_exists(fname_pattern)) { *pstatus = FileNotFound; return; }

  OpenImageFile( pcb,ib,1,ib[1].Name.V, ib[1].FFirst.V,
      ib[1].OpenMode.V | IO_Old | IO_FileProtect | IO_DontReadData, pstatus);
  if (*pstatus!=Success) return;

  /*--- Endian correction */
    if (!pcb->Flg[NFend].I) pcb->Flg[NFend].V = Default_fend;

  /*--- OpenMode of output file */
  if (!ib[0].OpenMode.I) {
    ib[0].OpenMode.V = IO_ImageProtect;
    ib[0].OpenMode.I = TRUE;
    }

  /*--- Argument  : output file name */
  argv_filename ( pcb, "Output file name", ib, 0, Default_opfn, pstatus);
  if (*pstatus!=Success) return;

  /* Search for minimum and maximum image number */
  (void) SearchMinMaxImage ( pcb, ib, 1, &image_1st, &image_lst, pstatus);
  if (*pstatus!=Success) return;

  /*--- number of first image */
  if (!(ib[1].First.I)) ib[1].First.V = image_1st;

  /*--- number of last image */
  if (!(ib[1].Last.I)) ib[1].Last.V = image_lst;

  /*--- Argument : increment */
  if (!(ib[1].Inc.I)) ib[1].Inc.V = Default_inc;

  (void) ReadImageDimension ( pcb, ib, 1, ib[1].First.V,
                              &i1dim_1, &i1dim_2, pstatus ); 
  if (*pstatus!=Success) return; 

  if (i1dim_1<0) i1dim_1 = Default_dim_1; 
  if (i1dim_2<0) i1dim_2 = Default_dim_2;

  /*--- Argument  : input dimension 1 */
  argv_long( pcb,"Input dimension 1", &ib[1].Dim[1],i1dim_1, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : input dimension 2 */
  argv_long( pcb,"Input dimension 2", &ib[1].Dim[2],i1dim_2, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : bytes to skip */
  if (!pcb->Num[NFSkip].I) pcb->Num[NFSkip].V = 0l;
  argv_long(pcb,"Bytes to skip (bytes)",&pcb->Num[NFSkip],
             pcb->Num[NFSkip].V, pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : input data type */
  showdatatypes( );
  if (!pcb->Lin[NFType].I) strcpy(pcb->Lin[NFType].V,STypeFloat); 
  argv_line(pcb,"Input data type",&pcb->Lin[NFType],pcb->Lin[NFType].V,
    pstatus);
  if (*pstatus!=Success) return;

  /* check type */
  if (string2type(pcb->Lin[NFType].V,&datatype)!=1) {
    *pstatus = UnknownDataType; return; }

  if (pcb->TestBit) 
    printf("TypeString = %s, data type = %d\n",pcb->Lin[NFType].V, datatype);
     
  (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
      if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

  CloseImageFile( pcb, ib, 1, pstatus) ;
  if (*pstatus!=Success) return;

  /*--- Argument  : output dummy */
  argv_float( pcb,"Output dummy", &ib[0].Dummy,ihb[1].Dummy.V, pstatus);
  if (*pstatus!=Success) return;

  /*--- output dimension 1 */
  if (!(ib[0].Dim[1].I)) ib[0].Dim[1].V = ihb[1].Dim[1];

  /*--- output dimension 2 */
  if (!(ib[0].Dim[2].I)) ib[2].Dim[1].V = ihb[1].Dim[2];

  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  printf("\n");
  if (pcb->Lin[NFname].I) 
                      printf("binary file           : %s\n",pcb->Lin[NFname].V);
  if (ib[0].Name.I)   printf("output file           : %s\n",ib[0].Name.V);
  if (ib[1].Dim[1].I) printf("input dimension 1     : %d\n",ib[1].Dim[1].V);
  if (ib[1].Dim[2].I) printf("input dimension 2     : %d\n",ib[1].Dim[2].V);
  if (pcb->Num[NFSkip].I)
                      printf("bytes to skip (bytes) : %d\n",pcb->Num[NFSkip].V);
  if (pcb->Lin[NFType].I)
                      printf("data type             : %s\n",pcb->Lin[NFType].V);
  if (ib[0].Dummy.I)  printf("output dummy          : %g\n",ib[0].Dummy.V);
  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

main (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, TRUE, &cb, ib, BlockNum );

  DefFlgOption( &cb, SFend, NFend);
  DefLineOption( &cb, SFname, NFname);
  DefLineOption( &cb, SFType, NFType);
  DefIntOption ( &cb, SFSkip, NFSkip);

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);
  ReportSaxsStatus( status, &cb.seb, 1 );

 /* Keyboard I/O and sequence calculation */

  /* USER KEYBOARD I/O */
  argv_start ( &cb, 1 );
  user_io( &cb, ib, &status);
  argv_end( &cb ); /* must be called after user_io */

  /* SEQUENCE CALCULATION */
  if (status==Success) ImageLoop( &cb, ib, binary2saxs, TRUE, &status );

  if (status==Abort) ReportSaxsStatus( status, &cb.seb, 0 );
    else ReportSaxsStatus( status, &cb.seb, 1 );

  printf("\nEnd of % s\n",argv[0]);

} /* main */

