/*+++
1 SaxsRoutine

2 DESCRIPTION
  Some general routines for coordinate system determination (GetReference...),
  region of interest determination (GetLinearOverlap, GetImageOverlap), 
  general pixel interpolation (Ipol2) with dummy checks (Ipol2d) and limits 
  checks (Ipol2ld), image data filters (TransformFloat2d, ClipFloat2d, 
  RebinFloat2d, GnawFloat2d). See example routines for details.

2 CALL
     ArgvFilenames()
     GetLinearOverlap()
     GetImageOverlap()
     GetReferenceParameters()
     GetReference()
     Ipol2()
     TransformFloat2d()
     ClipFloat2d()
     RebinFloat2d()
     GnawFloat2d()
     Ipol2ldw()
     Ipol2ld()
     Ipol2d()
     SaxsRoutineVersion()
     RowProjection()
     ColProjection()
     copy_amat()
     invert_amat()
     AffineFloat2d()
     Affine()
     

2 HISTORY
25-Dec-1995 PB test_routines.h
26-Jan-1996 PB ArgvFilenames: test of pcb->ImgBlkLen
28-Jan-1996 PB RebinFloat2d TransformFloat2d
               Transform replaced by TransformFloat2d (long int Dim_1 Dim_2!!!)
29-Mar-1996 PB ClipFloat2d
22-Apr-1996 PB Limit error in Ipol2ld corrected
19-May-1996 PB GnawFloat2d
20-May-1996 PB Error in GnawFloat2d for 0 Gnaws corrected
15-Oct-1996 PB V1.0 test_routines renamed to SaxsRoutine, SaxsRoutineVersion
11-Nov-1999 PB V1.01 DUMMYDEFINED updated
26-Nov-1999 PB V1.02 cc on dec alpha: statements of the type *pd++=*pd+xyz;
                     changed to { *pd=*pd+xyz; pd++; }
21-Dec-1999 PB V1.03 Ipol2, Ipol2d, Ipol2ld: check for NULL pointer
25-Dec-1999 PB V1.04 GetLinearOverlap
27-Dec-1999 PB V1.05 RowProjection and ColProjection
27-Dec-1999 PB V1.06 AffineFloat2d
31-Dec-1999 PB V1.07 GetReference: new reference system BEAM 
                     invert_amat, copy_amat
31-Dec-1999 PB V1.08 Affine
10-Jan-2000 PB V1.09 new reference systems CENTER and NORMAL,
                     NORMAL replaces BEAM
25-Feb-2000 PB V1.10 Error in i_2 loop range in RowProjection and 
                     ColProjection corrected
2000-08-04  PB V1.10 %f changed to %g
2000-11-16  PB V1.11 Ipol2ldw
2000-11-18  PB V1.12 weight in Ipol2ldw is also set without interpolation
2001-01-11  PB V1.13 ihb[0] is created in ArgvFilename
2001-07-07  PB V1.14 GetLinearOverlap (DoNotScale)
2001-07-08  PB V1.15 file name patterns
2001-07-09  PB V1.16 repeat command removed
---*/

/****************************************************************************
* SaxsRoutine                                                               *
****************************************************************************/
# define SR_Version  "SaxsRoutine : V1.16 Peter Boesecke 2001-07-09"

# include "SaxsRoutine.h"
# include "SaxsError.h"

/*----------------------------------------------------------------------------
 Constants
-----------------------------------------------------------------------------*/
const char * SRName = "SaxsRoutine";

/*----------------------------------------------------------------------------
1 SaxsRoutineVersion

2 DESCRIPTION
  Returns a pointer to the version string.
  char * SaxsRoutineVersion ( void )

2 HISTORY
  15-Oct-1996 PB
----------------------------------------------------------------------------*/
char * SaxsRoutineVersion ( void )
{ static char * Version = SR_Version;
  return ( Version );
} /* SaxsRoutineVersion */

/*----------------------------------------------------------------------------
RebinFloat2d
  Two dimensional rebinning of a float array. The result is written into the
  input array and the new dimensions are returned.
Parameters
  float * dataIn  (i)     pointer to input array
  float * dataOut (o)     pointer to output array
  long * dim_1    (i/o)   pointer to input/output dimension 1
  long * dim_2    (i/o)   pointer to input/output dimension 2
  float dummy     (i)     dummy value
  float ddummy    (i)     ddummy value
  long bin_1      (i)     rebin factor 1 (>=1)
  long bin_2      (i)     rebin factor 2 (>=1)
-----------------------------------------------------------------------------*/
void RebinFloat2d ( float * dataIn, float * dataOut,
                      long int * dim_1, long int * dim_2,
                      float dummy, float ddummy,
                      long int bin_1, long int bin_2)
{ register long int j_1, j_2, i_1, i_2;
  float * pin, * pout;
  float value, sum, count;

  if ((bin_1>1) || (bin_2>1)) {
    pout = dataOut;
    for (j_2=0;j_2<=*dim_2-bin_2;j_2+=bin_2)
      for (j_1=0;j_1<=*dim_1-bin_1;j_1+=bin_1) {
        sum = 0.0; count = 0.0;
        for (i_2=j_2;i_2<j_2+bin_2;i_2++) {
          pin = ABSPTR(dataIn,*dim_1,*dim_2,j_1,i_2);
          for (i_1=j_1;i_1<j_1+bin_1;i_1++) {
            value = *(pin++);
            if NODUMMY(value,dummy,ddummy)
              { sum += value; count += 1.0; }
            }
          }
        if (count>0.0) *(pout++) = sum/count; else *(pout++) = dummy;
        }
     *dim_1 /= bin_1; *dim_2 /= bin_2;
     }

} /* RebinFloat2d */

/*---------------------------------------------------------------------------
  ClipFloat2d
  Sets all values of the data field outside minclip and maxclip to dummy.
  ---------------------------------------------------------------------------*/
void ClipFloat2d( float *data, long int dim_1, long int dim_2,
                         float dummy, float ddummy,
                         float minclip, int minclipI,
                         float maxclip, int maxclipI )
{ register long int i;
  long int datanum;
  float *pdata;

  datanum = dim_1 * dim_2;
  pdata = data;
  if (minclipI) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
           if (minclip>*pdata) *pdata = dummy;
           }
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               if (minclip>*pdata) *pdata = dummy;
               pdata +=1;
               }
    } /* if (minclipI) */

  pdata = data;
  if (maxclipI) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) {
           if (maxclip<*pdata) *pdata = dummy;
           }
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               if (maxclip>*pdata) *pdata = dummy;
               pdata +=1;
               }
    } /* if (maxclipI) */

} /* ClipFlat2d */

/*---------------------------------------------------------------------------
  TransformFloat2d
  Multiplies the data field with a factor and adds a constant.
  ---------------------------------------------------------------------------*/
void TransformFloat2d( float *data, long int dim_1, long int dim_2,
                       float dummy, float ddummy,
                       float factor, float constant )
{ register long int i;
  long int datanum;
  const float eps1 = 1e-9, eps2 = 1e-30;
  float *pdata;

  datanum = dim_1 * dim_2;
  pdata = data;
  if ((ABS(factor-1.0)>eps1) || (ABS(constant)>eps2)) {
    if (DUMMYDEFINED(dummy,ddummy)) {
      for (i=0;i<datanum;i++) {
        if NODUMMY(*pdata,dummy,ddummy) *pdata = *pdata * factor + constant;
        pdata +=1;
        }
      } else for (i=0;i<datanum;i++) {
               *pdata = *pdata * factor + constant; pdata++;
               }
    } /* if ((ABS ... */

} /* TransformFloat2d */

/*----------------------------------------------------------------------------
GnawFloat2d
  Sets all pixels inside an ellipse around a dummy to dummy.
  The result is written into the input array. The routine
  allocates a short integer array with the same dimensions as the
  data array for intermediate results. Its memory is released after
  successful termination of GnawFloat2d.
Parameters
  float * data    (i)     pointer to input/output array
  long    dim_1   (i)     input/output/tmp dimension 1
  long    dim_2   (i)     input/output/tmp dimension 2
  float   dummy   (i)     dummy value
  float   ddummy  (i)     ddummy value
  long    gnaw_1  (i)     axis 1 of the ellipse (>=0)
  long    gnaw_2  (i)     axis 2 of the ellipse (>=0)
  int   * pstatus (o)     exit status
-----------------------------------------------------------------------------*/
void GnawFloat2d ( float * data,
                   long int dim_1, long int dim_2,
                   float dummy, float ddummy,
                   long int gnaw_1, long int gnaw_2,
                   int * pstatus )
{ register long int i, j_1, j_2, i_1, i_2;
  long int i_1min;
  float * pdata;
  short int * tmp, * ptmp;
  size_t t_tmp;

  float value;
  long int gnaw2_1, gnaw2_2, gnaw2;
  long int dist2_1, dist2_2;

  *pstatus = Success;

  gnaw_1 = MAX2(0,gnaw_1);
  gnaw_2 = MAX2(0,gnaw_2);

  if ((gnaw_1>0) || (gnaw_2>0)) {

    /* allocate memory for tmp array*/
    t_tmp = sizeof(short int)*dim_1*dim_2;
    tmp = (short int *) malloc (t_tmp);
    if (tmp == (short int *) NULL) {
      *pstatus=NotEnoughMemoryAvailable; return; }

    /* clear tmp array */
    ptmp = tmp;
    for (i=0;i<dim_1*dim_2;i++) *ptmp++=0;

    /* search dummies in data array */
    gnaw2_1 = gnaw_1 * gnaw_1 + 1;
    gnaw2_2 = gnaw_2 * gnaw_2 + 1;
    gnaw2   = gnaw2_1*gnaw2_2;
    pdata = data;
    for (j_2=0;j_2<dim_2;j_2++)
      for (j_1=0;j_1<dim_1;j_1++) {
        value = *(pdata++);
        if DUMMY(value,dummy,ddummy) {
          /* create dummy ellipse around j_1, j_2 in tmp */
          for (i_2=MAX2(0,j_2-gnaw_2);i_2<MIN2(dim_2,j_2+gnaw_2+1);i_2++) {
            dist2_2 = i_2 - j_2; dist2_2 *= dist2_2;
            i_1min = MAX2(0,j_1-gnaw_1);
            ptmp = ABSPTR(tmp,dim_1,dim_2,i_1min,i_2);
            for (i_1=i_1min;i_1<MIN2(dim_1,j_1+gnaw_1+1);i_1++) {
              dist2_1 = i_1 - j_1; dist2_1 *= dist2_1;
              if (dist2_1*gnaw2_2+dist2_2*gnaw2_1<=gnaw2) *ptmp = 1;
              ptmp++;
              }
            }
          }
        }

    /* copy dummies from tmp to data */
    ptmp = tmp;
    pdata = data;
    for (i=0;i<dim_1*dim_2;i++) {
      if (*ptmp++) *pdata=dummy;
      pdata++;
      }

    /* free tmp array */
    free(tmp);

    }


} /* GnawFloat2d */

/*---------------------------------------------------------------------------
1 Ipol2ldw

2 PURPOSE
  Linear two dimensional interpolation including dummies with limit checks
  Ipol2ldw calculates the weighted sum and the sum of the weights of the 
  interpolated pixels. To calculate the interpolated value, sum must be
  devided by weight. The function returns the number of interpolated pixels.
  See also Ipol2ld.

2 METHOD
  The point (f_1, f_2) is linearly interpolated between the 4 closest
  program array elements val1=Data[i_1,i_2], val2=Data[i_1+1,I_2],
  val3=Data[i_1+1,i_2+1] and val4=Data[I_1,i_2+1]. If a float index
  is an integer number 0, 1, 2, 3 etc. no interpolation is done in the
  corresponding direction.
  The program array limits are checked

2 CALL
  int Ipol2d (float *Data, int Dim_1, int Dim_2, float Dummy, float DDummy,
              float *sum, float *weight)

  return value            (o)   : number of pixels contributing to the output
                                  value (0..4), if 0, no valid data point found.
  float Data[Dim_1,Dim_2] (i)   : input array
  int Dim_1, Dim_2        (i)   : dimension of array
  float Dummy, DDummy     (i)   : dummy specification
  float f_1, f_2          (i)   : program array indices (interpolation point)
  float *sum              (o)   : weighted sum of interpolated pixels 
  float *weight           (o)   : weight of sum 
                                  minimum 0.0: no valid pixel found
                                  maximum 1.0: interpolation between 1 to 4 
                                               distance weighted pixels

---------------------------------------------------------------------------*/
int Ipol2ldw (float *Data, int Dim_1, int Dim_2, float Dummy, float DDummy,
              float f_1, float f_2, float *sum, float *weight )
{ const float eps = IPOLEPS;
  int cnt;
  float *pval1, *pval2, *pval3, *pval4;
  float w1, w2, w3, w4;

  int   i_1, i_2;
  float r_1, r_2;

  /* calculate integer indices and rest */
  IDX(f_1,i_1,r_1);
  IDX(f_2,i_2,r_2);

  pval1 = ABSPTR(Data,Dim_1,Dim_2,i_1,i_2);

  *weight=0.0;
  *sum=0.0;
  cnt=0;

  if (!Data) return(cnt); // return, if NULL pointer

  /* General check */
  if ( (i_1>=0) && (i_2>=0) && (i_1<Dim_1-1) && (i_2<Dim_2-1) )
    /* all 4 points inside the array */
    if (r_1<eps) {
      if (r_2<eps) {
        if NODUMMY(*pval1,Dummy,DDummy) {
          cnt++; *weight += 1.0; *sum = *pval1; /* no interpolation */
          } /* if NODUMMY */
        } else {
        if NODUMMY(*pval1,Dummy,DDummy) {
          cnt++;w1=1.0-r_2; *weight+=w1; *sum+=*pval1*w1;
          } /* if NODUMMY */
        pval4 = NEXTROW(pval1,Dim_1,Dim_2);
        if NODUMMY(*pval4,Dummy,DDummy) {
          cnt++;w4=r_2; *weight+=w4; *sum+=*pval4*w4;
          } /* if NODUMMY */
//        if (cnt) *value = *sum/*weight;
        } /* if (r_2<eps) */
      } else {
      if (r_2<eps) {
        if NODUMMY(*pval1,Dummy,DDummy) {
          cnt++; w1=(1.0-r_1); *weight+=w1; *sum = *pval1*w1;
          } /* if NODUMMY */
        pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
        if NODUMMY(*pval2,Dummy,DDummy) {
          cnt++; w2=r_1; *weight+=w2; *sum+=*pval2*w2;
          } /* if NODUMMY */
//        if (cnt) *value = *sum/*weight;
        } else {
        if NODUMMY(*pval1,Dummy,DDummy) {
          cnt++;w1=(1.0-r_1)*(1.0-r_2); *weight+=w1; *sum+=*pval1*w1;
          } /* if NODUMMY */
        pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
        if NODUMMY(*pval2,Dummy,DDummy) {
          cnt++;w2=r_1*(1.0-r_2); *weight+=w2; *sum+=*pval2*w2;
          } /* if NODUMMY */
        pval4 = NEXTROW(pval1,Dim_1,Dim_2);
        if NODUMMY(*pval4,Dummy,DDummy) {
          cnt++;w4=(1.0-r_1)*r_2; *weight+=w4; *sum+=*pval4*w4;
          } /* if NODUMMY */
        pval3 = NEXTCOLROW(pval1,Dim_1,Dim_2);
        if NODUMMY(*pval3,Dummy,DDummy) {
          cnt++;w3=r_1*r_2; *weight+=w3; *sum+=*pval3*w3;
          } /* if NODUMMY */
//        if (cnt) *value = *sum/*weight;
        } /* (r_2<eps) */
      } /* if (r_1 ... */
   else if ( (i_1>=-1) && (i_2>=-1) && (i_1<Dim_1) && (i_2<Dim_2) )
    /* some of the 4 points are inside the array */
    if (r_1<eps) {
      if (r_2<eps) {
        if ( (i_1>=0) && (i_2>=0) ) /* pval1 OK? */
          if NODUMMY(*pval1,Dummy,DDummy) {
            cnt++; *weight += 1.0; *sum = *pval1; /* no interpolation */
            } /* if NODUMMY */

        } else {
        if ( (i_1>=0) && (i_2>=0) ) /* pval1 OK? */
          if NODUMMY(*pval1,Dummy,DDummy) {
            cnt++;w1=1.0-r_2; *weight+=w1; *sum+=*pval1*w1;
            } /* if NODUMMY */

        if ( (i_1>=0) && (i_2<Dim_2-1) ) { 
          pval4 = NEXTROW(pval1,Dim_1,Dim_2);
          if NODUMMY(*pval4,Dummy,DDummy) {
            cnt++;w4=r_2; *weight+=w4; *sum+=*pval4*w4;
            } /* if NODUMMY */
          } /* pval4 */

//        if (cnt) *value = *sum/*weight;
        } /* if (r_2<eps) */
      } else {
      if (r_2<eps) {
        if ( (i_1>=0) && (i_2>=0) ) /* pval1 OK? */
          if NODUMMY(*pval1,Dummy,DDummy) {
            cnt++; w1=(1.0-r_1); *weight+=w1; *sum = *pval1*w1;
            } /* if NODUMMY */

        if ( (i_2>=0) && (i_1<Dim_1-1) ) { 
          pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
          if NODUMMY(*pval2,Dummy,DDummy) {
            cnt++; w2=r_1; *weight+=w2; *sum+=*pval2*w2;
            } /* if NODUMMY */
          } /* pval2 */

//        if (cnt) *value = *sum/*weight;
        } else {
        if ( (i_1>=0) && (i_2>=0) ) /* pval1 OK? */
          if NODUMMY(*pval1,Dummy,DDummy) {
            cnt++;w1=(1.0-r_1)*(1.0-r_2); *weight+=w1; *sum+=*pval1*w1;
            } /* if NODUMMY */

        if ( (i_2>=0) && (i_1<Dim_1-1) ) {
          pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
          if NODUMMY(*pval2,Dummy,DDummy) {
            cnt++;w2=r_1*(1.0-r_2); *weight+=w2; *sum+=*pval2*w2;
            } /* if NODUMMY */
          } /* pval2 */

        if ( (i_1>=0) && (i_2<Dim_2-1) ) { 
          pval4 = NEXTROW(pval1,Dim_1,Dim_2);
          if NODUMMY(*pval4,Dummy,DDummy) {
            cnt++;w4=(1.0-r_1)*r_2; *weight+=w4; *sum+=*pval4*w4;
            } /* if NODUMMY */
          } /* pval4 */

        if ( (i_1<Dim_1-1) && (i_2<Dim_2-1) ) { 
          pval3 = NEXTCOLROW(pval1,Dim_1,Dim_2);
          if NODUMMY(*pval3,Dummy,DDummy) {
            cnt++;w3=r_1*r_2; *weight+=w3; *sum+=*pval3*w3;
            } /* if NODUMMY */
          } /* pval3 */

//        if (cnt) *value = *sum/*weight;
        } /* (r_2<eps) */
      } /* if (r_1 ... */

  return(cnt);

} /* Ipol2ldw */

/*---------------------------------------------------------------------------
1 Ipol2ld

2 PURPOSE
  Linear two dimensional interpolation including dummies with limit checks
  Ipol2ld returns the interpolated value. To calculate the sum and the weight
  of the interpolated pixels use Ipol2ldw.

2 METHOD
  See Ipol2ldw. Ipol2ld calculates only the interpolated value and returns
  the number of interpolated pixels.
---------------------------------------------------------------------------*/
int Ipol2ld (float *Data, int Dim_1, int Dim_2, float Dummy, float DDummy,
             float f_1, float f_2, float *value) 
{ float weight;
  int cnt;

  cnt = Ipol2ldw(Data, Dim_1, Dim_2, Dummy, DDummy,f_1, f_2, value, &weight);
  if (cnt) *value = *value/weight;

  return ( cnt );

} /* Ipol2ld */

/*---------------------------------------------------------------------------
1 Ipol2d

2 PURPOSE
  Linear two dimensional interpolation including dummies
  No limit checks

2 METHOD
  The point (f_1, f_2) is linearly interpolated between the 4 closest
  program array elements val1=Data[i_1,i_2], val2=Data[i_1+1,I_2],
  val3=Data[i_1+1,i_2+1] and val4=Data[I_1,i_2+1]. If a float index
  is an integer number 0, 1, 2, 3 etc. no interpolation is done in the
  corresponding direction.

2 CALL
  int Ipol2d (float *Data, int Dim_1, int Dim_2, float Dummy, float DDummy,
              float *value)

  return value            (o)   : number of pixels contributing to the output
                                  value (0..4), if 0, only dummies found.
  float Data[Dim_1,Dim_2] (i)   : input array
  int Dim_1, Dim_2        (i)   : dimension of array
  float Dummy, DDummy     (i)   : dummy specification
  float f_1, f_2          (i)   : program array indices (interpolation point)
  float *value            (o)   : interpolated value

---------------------------------------------------------------------------*/
int Ipol2d (float *Data, int Dim_1, int Dim_2, float Dummy, float DDummy,
            float f_1, float f_2, float *value)
{ const float eps = IPOLEPS;
  int cnt;
  float *pval1, *pval2, *pval3, *pval4;
  float w1, w2, w3, w4;
  float weight;

  int   i_1, i_2;
  float r_1, r_2;

  /* calculate integer indices and rest */
  IDX(f_1,i_1,r_1);
  IDX(f_2,i_2,r_2);

  pval1 = ABSPTR(Data,Dim_1,Dim_2,i_1,i_2);

  weight=0.0;
  *value=0.0;
  cnt=0;

  if (!Data) return(cnt); // return, if NULL pointer

  if (r_1<eps) {
     if (r_2<eps) {
       if NODUMMY(*pval1,Dummy,DDummy) {
         cnt++; *value = *pval1;  /* no interpolation */
         } /* if NODUMMY */
       } else {
       if NODUMMY(*pval1,Dummy,DDummy) {
         cnt++;w1=1.0-r_2; weight+=w1; *value+=*pval1*w1;
         } /* if NODUMMY */
       pval4 = NEXTROW(pval1,Dim_1,Dim_2);
       if NODUMMY(*pval4,Dummy,DDummy) {
         cnt++;w4=r_2; weight+=w4; *value+=*pval4*w4;
         } /* if NODUMMY */
       if (cnt) *value = *value/weight;
       } /* if (r_2<eps) */
     } else {
     if (r_2<eps) {
       if NODUMMY(*pval1,Dummy,DDummy) {
         cnt++; w1=(1.0-r_1); weight+=w1; *value = *pval1*w1;
         } /* if NODUMMY */
       pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
       if NODUMMY(*pval2,Dummy,DDummy) {
         cnt++; w2=r_1; weight+=w2; *value+=*pval2*w2;
         } /* if NODUMMY */
       if (cnt) *value = *value/weight;
       } else {
       if NODUMMY(*pval1,Dummy,DDummy) {
         cnt++;w1=(1.0-r_1)*(1.0-r_2); weight+=w1; *value+=*pval1*w1;
         } /* if NODUMMY */
       pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
       if NODUMMY(*pval2,Dummy,DDummy) {
         cnt++;w2=r_1*(1.0-r_2); weight+=w2; *value+=*pval2*w2;
         } /* if NODUMMY */
       pval4 = NEXTROW(pval1,Dim_1,Dim_2);
       if NODUMMY(*pval4,Dummy,DDummy) {
         cnt++;w4=(1.0-r_1)*r_2; weight+=w4; *value+=*pval4*w4;
         } /* if NODUMMY */
       pval3 = NEXTCOLROW(pval1,Dim_1,Dim_2);
       if NODUMMY(*pval3,Dummy,DDummy) {
         cnt++;w3=r_1*r_2; weight+=w3; *value+=*pval3*w3;
         } /* if NODUMMY */
       if (cnt) *value = *value/weight;
       } /* (r_2<eps) */
     } /* if (r_1 ... */

  return(cnt);

} /* Ipol2d */

/*---------------------------------------------------------------------------
1 Ipol2

2 PURPOSE
  Linear two dimensional interpolation 
  no limit checks,
  no dummy checks

2 METHOD
  The point (f_1, f_2) is linearly interpolated between the 4 closest
  program array elements val1=Data[i_1,i_2], val2=Data[i_1+1,I_2],
  val3=Data[i_1+1,i_2+1] and val4=Data[I_1,i_2+1]. If a float index
  is an integer number 0, 1, 2, 3 etc. no interpolation is done in the
  corresponding direction.

2 CALL
  void Ipol2 (float *Data, int Dim_1, int Dim_2, 
             float f_1, float f_2, float *value)

  return value            (o)   : number of pixels contributing to the output
                                  value (0..4), if 0, only dummies found.
  float Data[Dim_1,Dim_2] (i)   : input array
  int Dim_1, Dim_2        (i)   : dimension of array
  float f_1, f_2          (i)   : program array indices (interpolation point)
  float *value            (o)   : interpolated value

2 HISTORY
  28-Apr-1995 PB from Ipol2d

---------------------------------------------------------------------------*/
void Ipol2 (float *Data, int Dim_1, int Dim_2,
            float f_1, float f_2, float *value)
{ const float eps = IPOLEPS;
  int cnt;
  float *pval1, *pval2, *pval3, *pval4;
  float w1, w2, w3, w4;
  float weight;

  int   i_1, i_2;
  float r_1, r_2;

  /* calculate integer indices and rest */
  IDX(f_1,i_1,r_1);
  IDX(f_2,i_2,r_2);

  pval1 = ABSPTR(Data,Dim_1,Dim_2,i_1,i_2);

  weight=0.0;
  *value=0.0;
  cnt=0;

  if (!Data) return; // return, if NULL pointer

  if (r_1<eps) {
     if (r_2<eps) {
         cnt++; *value = *pval1;  /* no interpolation */
       } else {
         cnt++;w1=1.0-r_2; weight+=w1; *value+=*pval1*w1;
       pval4 = NEXTROW(pval1,Dim_1,Dim_2);
         cnt++;w4=r_2; weight+=w4; *value+=*pval4*w4;
       if (cnt) *value = *value/weight;
       } /* if (r_2<eps) */
     } else {
     if (r_2<eps) {
         cnt++; w1=(1-r_1); weight+=w1; *value = *pval1*w1;
       pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
         cnt++; w2=r_1; weight+=w2; *value+=*pval2*w2;
       if (cnt) *value = *value/weight;
       } else {
         cnt++;w1=(1.0-r_1)*(1.0-r_2); weight+=w1; *value+=*pval1*w1;
       pval2 = NEXTCOL(pval1,Dim_1,Dim_2);
         cnt++;w2=r_1*(1.0-r_2); weight+=w2; *value+=*pval2*w2;
       pval4 = NEXTROW(pval1,Dim_1,Dim_2);
         cnt++;w4=(1.0-r_1)*r_2; weight+=w4; *value+=*pval4*w4;
       pval3 = NEXTCOLROW(pval1,Dim_1,Dim_2);
         cnt++;w3=r_1*r_2; weight+=w3; *value+=*pval3*w3;
       if (cnt) *value = *value/weight;
       } /* (r_2<eps) */
     } /* if (r_1 ... */

} /* Ipol2 */

/*---------------------------------------------------------------------------
1 GetReference
  Determine the reference system and set Off and Ps

2 PURPOSE
  Determine the reference system and set Off and Ps for a single image.
  Should be used to determine default values from the first image for
  user input. Inside image loops only GetReferenceParameters should be used. 

2 CALL
  void GetReference( CmdBlk * pcb, int blockno, ImgHeadBlk ihb[],
                     float * Off_1, float * Off_2,
                     float * Ps_1,  float * Ps_2,
                     int * pstatus );
  return value            (o)   : void
  CmdBlk * pcb            (i)   : command block
             pcb->RSys    (i)   : reference system
  ImgHeadBlk pihb->Dim   (i)
             pihb->Offset
             pihb->PixSiz
             pihb->Center, ->Wavelength, ->SampleDistance
  float * Off_1, * Off_2  (o)   : offset
  float * Ps_1, * Ps_2    (o)   : pixel size
  int * pstatus           (o)   : SAXS status (Success and Failed)

2 HISTORY
  24-Apr-1995 Peter Boesecke from GetReferenceParameters 
  27-Nov-1995 PB GetReference
-----------------------------------------------------------------------------*/
void GetReference     ( long int RSys, int blockno, ImgHeadBlk ihb[],
                        float * Off_1, float * Off_2,
                        float * Ps_1,  float * Ps_2,
                        int * pstatus )
{ register int i;
  ImgHeadBlk * pihb;

  *pstatus=Success;

  pihb = &ihb[blockno];

      switch (RSys) {
        case IO_Array :
          ARRAYREF((*Off_1),(*Ps_1));
          ARRAYREF((*Off_2),(*Ps_2));

          break;
        case IO_Image :
          if (!((pihb->Offset[1].I) && (pihb->Offset[2].I))) {
            *pstatus = Failed; break; }

          IMAGEREF(*Off_1,*Ps_1,pihb->Offset[1].V);
          IMAGEREF(*Off_2,*Ps_2,pihb->Offset[2].V);

          break;

        case IO_Center :
          if (!( (pihb->Offset[1].I) && (pihb->Center[1].I) &&
                 (pihb->Offset[2].I) && (pihb->Center[2].I) )) {
            *pstatus = Failed; break; }

          CENTERREF(*Off_1,*Ps_1,pihb->Offset[1].V,pihb->Center[1].V);
          CENTERREF(*Off_2,*Ps_2,pihb->Offset[2].V,pihb->Center[2].V);

          break;

        case IO_Real  :
          if (!((pihb->Offset[1].I) && (pihb->PixSiz[1].I) &&
                (pihb->Offset[2].I) && (pihb->PixSiz[2].I))) {
            *pstatus = Failed; break; }

          REALREF(*Off_1,*Ps_1,pihb->Offset[1].V,pihb->PixSiz[1].V);
          REALREF(*Off_2,*Ps_2,pihb->Offset[2].V,pihb->PixSiz[2].V);

          break;

        case IO_Normal :
          if (!( (pihb->Offset[1].I) && (pihb->PixSiz[1].I) &&
                 (pihb->Center[1].I) &&
                 (pihb->Offset[2].I) && (pihb->PixSiz[2].I) &&
                 (pihb->Center[2].I) )) {
            *pstatus = Failed; break; }

          NORMALREF(*Off_1,*Ps_1,pihb->Offset[1].V,
                    pihb->PixSiz[1].V,pihb->Center[1].V);

          NORMALREF(*Off_2,*Ps_2,pihb->Offset[2].V,
                    pihb->PixSiz[2].V,pihb->Center[2].V);

          break;

        case IO_Saxs  :
          if (!( (pihb->Offset[1].I) && (pihb->PixSiz[1].I) &&
                 (pihb->Center[1].I) &&
                 (pihb->SampleDistance.I) && (pihb->WaveLength.I) &&
                 (pihb->Offset[2].I) && (pihb->PixSiz[2].I) &&
                 (pihb->Center[2].I) )) {
            *pstatus = Failed; break; }

          SAXSREF(*Off_1,*Ps_1,pihb->Offset[1].V,
                  pihb->PixSiz[1].V,pihb->Center[1].V,
                  pihb->SampleDistance.V,pihb->WaveLength.V);

          SAXSREF(*Off_2,*Ps_2,pihb->Offset[2].V,
                  pihb->PixSiz[2].V,pihb->Center[2].V,
                  pihb->SampleDistance.V,pihb->WaveLength.V);

          break;

        default       :
          printf("Unknown reference system\n");
          *pstatus = Failed; return;
      } /* switch */

    if (*pstatus!=Success) {
      printf("Missing parameters for the chosen reference system (% s)\n",
              reftostr(RSys)); return; }

  } /* GetReference */

/*---------------------------------------------------------------------------
1 GetReferenceParameters
  Determine the reference system and set Off and Ps

2 PURPOSE
  Determine the reference system and set Off and Ps and shift the output
  image. Only to be used inside image loops.

2 CALL
  void GetReferenceParameters( CmdBlk * pcb, ImgHeadBlk ihb[],
                               int MinImage, int MaxImage,
                               float Off_1[], float Off_2[],
                               float Ps_1[],  float Ps_2[],
                               int * pstatus ); 
  return value            (o)   : void
  CmdBlk * pcb		  (i)   : command block
             pcb->RSys    (i)   : reference system
             pcb->Shift   (i)   : shift of output image (image 0)
  ImgHeadBlk ihb[MaxImage+1](i) : image header block 
             ihb[]->Dim 	
             ihb[]->Offset 
             ihb[]->PixSiz
             ihb[]->Center, ->Wavelength, ->SampleDistance
  int MinImage            (i)   : smallest image number that should be used
  int MaxImage            (i)   : largest images, that should be used
                                  0 : output image, 1 1st input image etc.
                                  MinImage>MaxImage : no image
  float Off_1[], Off_2[]  (o)   : offset 
  float Ps_1[], Ps_[]     (o)   : pixel size
  int *pstatus	  	  (o)   : SAXS status (Success and Failed)

2 HISTORY
  26-Feb-1995 Peter Boesecke creation
---------------------------------------------------------------------------*/
void GetReferenceParameters( CmdBlk * pcb, ImgHeadBlk ihb[],
                             int MinImage, int MaxImage,
                             float Off_1[], float Off_2[],
                             float Ps_1[],  float Ps_2[],
                             int * pstatus )
{ register int i;
  const char * RoutineName = "GetReferenceParameters";

  if (pcb->TestBit) printf("% s : % s\n", RoutineName, reftostr(pcb->RSys.V));
  SetSaxsError( &pcb->seb, SRName, RoutineName, NULL, Success, 0, NULL );

  *pstatus = Success;

  for (i=MinImage;i<=MaxImage;i++) {

    GetReference( pcb->RSys.V, i, ihb, 
                  &Off_1[i], &Off_2[i], &Ps_1[i],  &Ps_2[i],
                  pstatus );
    if (*pstatus!=Success) return; 

    } /* for (i=MinImage;i<=MaxImage ... */

  /* Subtract output shift for calculation */
  if (MinImage==0) {
     if (pcb->Shift[1].I) Off_1[0] = Off_1[0]-pcb->Shift[1].V;
     if (pcb->Shift[2].I) Off_2[0] = Off_2[0]-pcb->Shift[2].V;
     }

  if (pcb->TestBit) for (i=MinImage;i<=MaxImage;i++) {
      printf("Off_1[%d] = % g, Ps_1[%d] = % g\n", i,Off_1[i],i,Ps_1[i]);
      printf("Off_2[%d] = % g, Ps_2[%d] = % g\n", i,Off_2[i],i,Ps_2[i]); }

} /* GetReferenceParameters */

/*---------------------------------------------------------------------------
NAME

   GetLinearOverlap --- linear overlap in world and program array indices 

SYNOPSIS

   void GetLinearOverlap     ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                               int   coordinate, int   OutImage, 
                               int   MinImage,   int   MaxImage,
                               int   WminInI,    float WminIn,
                               int   WmaxInI,    float WmaxIn,
                               float Off[],      float Ps[],
                               float fmin[],     float fmax[],
                               float *Wmin,      float *Wmax );

   Dimension of all arrays must be at least MaxImage+1

DESCRIPTION
  Calculates the overlap region of the output image (BlockNo=OutImage) 
  and all input images (BlockNo=[MinImage..MaxImage]) in world 
  coordinates (*Wmin, *Wmax) and in program array indices (fmin[],fmax[]). 
  If MaxImage<MinImage only the region of block OutImage is returned.

  The values of OutImage, MinImage and MaxImage must be in the range 
  [0..L-1], where L is the allocated length of the arrays Off[], Ps[], fmin[] 
  and fmax[]. The arrays Off[] and Ps[] must have been initialized with
  GetReferenceParameters for all used blocks (usually 0 to L). 

  The block OutImage determines the position of the calculation grid. Its 
  outer border is defined by the centers of the pixels in OutImage that 
  are still inside the overlap region. In this case, fmin and fmax of the
  output image are restricted to integer values.
  If OutImage<0, no output block is chosen and this option is switched off.

  If the flag DoNotScale is set in an input image block the image is 
  skipped.

ARGUMENTS
CmdBlk *pcb
ImgHeadBlk ihb[]
int   coordinate : coordinate number (1 or 2)
int   OutImage : block number of the output image (not used if <0)
int   MinImage : block number of the first used input image (>=0)
int   MaxImage : block number of the last used input image 
int   WminInI    : 0: minimum world coordinate (WminIn) not set, otherwise set
float WminIn     : minimum world coordinate
int   WmaxInI    : 0: maximum world coordinate (WmaxIn) not set, otherwise set 
float WmaxIn     : maximum world coordinate
float Off[],  Ps[]   : program array offsets and pixel sizes
float fmin[], fmax[] (o): range of program array indices corresponding to
                          *Wmin and *Wmax
float *Wmin, *Wmax (o): range of world coordinates

EXAMPLE
  - Overlap of output image with image of input sequence 2:
  GetLinearOverlap      ( pcb, ihb, 1, 0, 2, 2,
                          Off_1, Ps_1, 0, 0, 0, 0, 
                          fmin_1, fmax_1, 
                          &Wmin_1, &Wmax_1 ); // coordinate 1
  GetLinearOverlap      ( pcb, ihb, 2, 0, 2, 2,
                          Off_2, Ps_2, 0, 0, 0, 0,
                          fmin_2, fmax_2,
                          &Wmin_2, &Wmax_2 ); // coordinate 2
  - World region of output image only:
  GetLinearOverlap      ( pcb, ihb, 1, 0,  1, 0,
                          Off_1, Ps_1, ... ); // coordinate 1
  GetLinearOverlap      ( pcb, ihb, 2, 0,  1, 0,
                          Off_2, Ps_2, ... ); // coordinate 2
  - Overlap of all imax images i=0 .. i=imax-1;
  GetLinearOverlap      ( pcb, ihb, 1, 0, 1, imax-1,
                          Off_1, Ps_1, ... );
  GetLinearOverlap      ( pcb, ihb, 2, 0, 1, imax-1,
                          Off_2, Ps_2, ... );
  - Overlap of all imax images i=0 .. i=imax-1 with input region WminIn_1,
    WmaxIn_1, WminIn_2 and WmaxIn_2
  GetLinearOverlap      ( pcb, ihb, 1, 0, 1, imax-1,
                          Off_1, Ps_1, 1, WminIn_1, 1, WmaxIn_1,
                          fmin_1, fmax_1,
                          &Wmin_1, &Wmax_1 ); // coordinate 1
  GetLinearOverlap      ( pcb, ihb, 2, 0, 1, imax-1,
                          Off_2, Ps_2, 1, WminIn_2, 1, WmaxIn_2,
                          fmin_2, fmax_2,
                          &Wmin_2, &Wmax_2 ); // coordinate 1

HISTORY
  1999-12-25 Peter Boesecke from GetImageOverlap (version 27-Apr-1995)
  2001-07-07 DoNotScale
---------------------------------------------------------------------------*/
void GetLinearOverlap     ( CmdBlk *pcb,      ImgHeadBlk ihb[],
                            int   coordinate, int   OutImage,
                            int   MinImage,   int   MaxImage,
                            int   WminInI,    float WminIn,
                            int   WmaxInI,    float WmaxIn,
                            float Off[],      float Ps[],
                            float fmin[],     float fmax[],
                            float *Wmin,      float *Wmax )
{ const char * RoutineName = "GetLinearOverlap";
  const float eps = IPOLEPS/2;
  register int i;
  float W1, W2;
  int Imin, Imax;
  float RImin, RImax;

  if (pcb->TestBit) printf("+++ %s\n",RoutineName);

  /* region of output image OutImage or MinImage*/
  if (OutImage>=0) {
     W1=WORLD(0,Off[OutImage],Ps[OutImage]);
     W2=WORLD(ihb[OutImage].Dim[coordinate]-1,Off[OutImage],Ps[OutImage]);
    } else {
     W1=WORLD(0,Off[MinImage],Ps[MinImage]);
     W2=WORLD(ihb[MinImage].Dim[coordinate]-1,Off[MinImage],Ps[MinImage]);
    }
  *Wmin=MIN2(W1,W2); *Wmax=MAX2(W1,W2);

  /* overlap with input images i */
  for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
    if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
      W1=WORLD(0,Off[i],Ps[i]);
      W2=WORLD(ihb[i].Dim[coordinate]-1,Off[i],Ps[i]);
      *Wmin=MAX2(*Wmin,MIN2(W1,W2));
      *Wmax=MIN2(*Wmax,MAX2(W1,W2));
      } // if (i!=OutImage)
    } /* for (i */

  /* external region */
  if (WminInI) W1=WminIn; else W1 = *Wmin;
  if (WmaxInI) W2=WmaxIn; else W2 = *Wmax;
  *Wmin=MAX2(*Wmin,MIN2(W1,W2));
  *Wmax=MIN2(*Wmax,MAX2(W1,W2));

  /* calculate the program array indices */
  /* output image 0 */
  if (OutImage>=0) {
     fmin[OutImage]=INDEX(*Wmin,Off[OutImage],Ps[OutImage]);
     fmax[OutImage]=INDEX(*Wmax,Off[OutImage],Ps[OutImage]);
    }
  /* input images i */
  for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
    if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
      fmin[i]=INDEX(*Wmin,Off[i],Ps[i]);
      fmax[i]=INDEX(*Wmax,Off[i],Ps[i]);
      } // if (i!=OutImage)
    } /* for (i */

  if (OutImage>=0) {
     /* calculate integer indices from fmin_...[OutImage] and 
        fmax_...[OutImage] for output image OutImage */
     IDX(fmin[OutImage],Imin,RImin); 
     IDX(fmax[OutImage],Imax,RImax); 

     /* recalculate the overlap area from the size of the output array,
        the output image determines the calculation grid,
        thus only discrete array points are allowed (RI<eps)
        PB 19-Jun-1995 changed from RI<0 to RI<eps */
     if (pcb->TestBit) {
      printf(" Begin intermediate results\n");
      printf("  fmin[%d] = %g, Imin[%d] = %d, RImin[%d] = %g (coordinate %d)\n",
               OutImage,fmin[OutImage],OutImage,Imin,
               OutImage,RImin, coordinate);
      printf("  fmax[%d] = %g, Imax[%d] = %d, RImax[%d] = %g (coordinate %d)\n",
               OutImage,fmax[OutImage],OutImage,Imax,
               OutImage,RImax, coordinate);
      printf(" End intermediate results\n");
       }
     if (ABS(RImin)>eps) fmin[OutImage] = Imin+1;
       else fmin[OutImage] = (float) Imin;
     fmax[OutImage] = (float) Imax;

     /* The absolute minimum for fmin[OutImage] is 0.0,
        the absolute maximum for fmax[OutImage] is 
        ihb[OutImage].Dim[coordinate]-1, PB 05-Dec-1995 */
     fmin[OutImage]= MAX2(0.0,fmin[OutImage]);
     fmax[OutImage]=
          MIN2(fmax[OutImage],(float)(ihb[OutImage].Dim[coordinate]-1));
     if (pcb->TestBit) {
       printf(" fmin[%d] = %g, fmax[%d] = %g (coordinate %d)\n",
               OutImage,fmin[OutImage],OutImage,fmax[OutImage],coordinate);
       }

     W1 = WORLD(fmin[OutImage],Off[OutImage],Ps[OutImage]);
     W2 = WORLD(fmax[OutImage],Off[OutImage],Ps[OutImage]);
     *Wmin=MAX2(*Wmin,MIN2(W1,W2));
     *Wmax=MIN2(*Wmax,MAX2(W1,W2));

     /* recalculate the program array indices for the input images i */
     for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
       if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
         fmin[i]=INDEX(*Wmin,Off[i],Ps[i]);
         fmax[i]=INDEX(*Wmax,Off[i],Ps[i]);
         } // if (i!=OutImage)
       } /* for (i */
   } // if (OutImage>=0)

 /* print the program array indices */
  if (pcb->TestBit) {
    printf(" WminIn = ");
    if (WminInI) printf("%g, ", WminIn); else printf("(undefined), ");
    printf("WmaxIn = ");
    if (WmaxInI) printf("%g ", WmaxIn); else printf("(undefined) ");
    printf("(coordinate %d)\n",coordinate);

    printf(" *Wmin = %g, *Wmax = %g (coordinate %d)\n",
            *Wmin,*Wmax, coordinate);
    for (i=MAX2(0,MinImage);i<=MaxImage;i++) {
      if ( (i!=OutImage) && (!ihb[i].DoNotScale) ) {
        printf(" fmin[%d] = %g, fmax[%d] = %g (coordinate %d)\n",
          i,fmin[i],i,fmax[i],coordinate);
        } // if (i!=OutImage)
      } /* for (i */
    } /* if (pcb-> ... */

  if (pcb->TestBit) printf("--- %s\n",RoutineName);

} /* GetLinearOverlap */

/*---------------------------------------------------------------------------
1 GetImageOverlap
  Calculate the overlap region in the world, array boundaries [0..Dim-1]

2 PURPOSE
  Calculates the overlap region of the output image (0) with the input
  images (i>0) in the world, array boundaries [0..Dim-1]
  if imax<imin only the region of the output image is returned.

2 CALL
  void GetImageOverlap       ( CmdBlk * pcb, ImgHeadBlk ihb[],
                               int MinImage, int MaxImage,
                               float Off_1[], float Off_2[],
                               float Ps_1[],  float Ps_2[],
                               float fmin_1[], float fmax_1[],
                               float fmax_1[], float fmax_2[],
                               float *Wmin_1, float *Wmax_1,
                               float *Wmin_2, float *Wmax_2);

  Dimension of all arrays must be at least MaxImage+1.

  return value            (o)   : void
  CmdBlk * pcb            (i)   : command block
           pcb->TestBit   (i)
  ImgHeadBlk ihb[MaxImage+1](i) : image header block
             ihb[]->Dim
  int MinImage            (i)   : smallest input image number that
                                  should be considered > 0
                                  The output image (0) is always
                                  taken into account
  int MaxImage            (i)   : largest input image number, that
                                  should be used
                                   1 1st input image etc.
                                  MinImage>MaxImage : no image
  float Off_1[], Off_2[]  (i)   : offset
  float Ps_1[], Ps_[]     (i)   : pixel size
  float fmin_1[], fmax_1[](o)   : program array indices
  float fmin_2[], fmax_2[](o)   : program array indices
  float *Wmin_1, *Wmax_1  (o)   : world coordinate ranges
  float *Wmin_2, *Wmax_2    (o)   : world coordinate ranges

2 EXAMPLE
  - Overlap of output image with image of input sequence 2:
  GetImageOverlap       ( pcb, ihb, 2,2,
                          Off_1, Off_2, Ps_1,  float Ps_2,
                          fmin_1, fmax_1, fmin_2, fmax_2,
                          &Wmin_1, &Wmax_1, &Wmin_2, &Wmax_2);
  - World region of output image only:
  GetImageOverlap       ( pcb, ihb, 0, 0,
                          ... );
  - Overlap of all imax images i=0 .. i=imax-1;
  GetImageOverlap       ( pcb, ihb, 0, imax-1,
                          ... );

2 HISTORY
  26-Feb-1995 Peter Boesecke creation
  14-Apr-1995 Peter Boesecke Call error of Wmin_1, Wmin_2, Wmax_1, Wmax_1
                             corrected. The addresses of these arguments
                             are now passed.
  27-Apr-1995 Peter Boesecke Checked that the output image is always
                             taken into account.
  25-Dec-1999 Peter Boesecke splitted into two parts with GetLinearOverlap
---------------------------------------------------------------------------*/
void GetImageOverlap       ( CmdBlk * pcb, ImgHeadBlk ihb[],
                             int MinImage, int MaxImage,
                             float Off_1[], float Off_2[],
                             float Ps_1[],  float Ps_2[],
                             float fmin_1[], float fmax_1[],
                             float fmin_2[], float fmax_2[],
                             float * Wmin_1, float * Wmax_1,
                             float * Wmin_2, float * Wmax_2)
{ GetLinearOverlap  ( pcb, ihb, 1, 0, MinImage, MaxImage,
                      0, 0.0, 0, 0.0, Off_1, Ps_1,
                      fmin_1, fmax_1, Wmin_1, Wmax_1 ); // coordinate 1
  GetLinearOverlap  ( pcb, ihb, 2, 0, MinImage, MaxImage,
                      0, 0.0, 0, 0.0, Off_2, Ps_2,
                      fmin_2, fmax_2, Wmin_2, Wmax_2 ); // coordinate 2
} /* GetImageOverlap */

/*---------------------------------------------------------------------------
NAME

   RowProjection --- projection of image rows on a single line

SYNOPSIS

   float RowProjection(float *line, long int dim, float initvalue,
                       float *data, long int dim_1, long int dim_2,
                       float dummy, float ddummy,
                       float f_21, float f_22, 
                       int mode, int testbit);

DESCRIPTION
Averaging of rows-values between f_21 and f_22 in the 2d array data 
(float data[dim_1,dim_2]). The result is written to the 1d array line.
(float line[dim]). dim and dim_1 must be equal. line can be a subarray of
data, e.g. the first line. line is initialized with initvalue. f_21 must 
be smaller or equal to f_22.  

f_21 and f_22 should be calculated from the world coordinates row1 and 
row2 in the following way:

  f_21 = MAX2(-1,INDEX(row1, Off_2, Ps_2));
  f_22 = MIN2(I2Dim_2-1,INDEX(row2, Off_2,Ps_2));

Where Off_2 and Ps_2 are calculated with GetReferenceParameters.

ARGUMENTS
float *line          : 1d output array  float line[dim]
long  dim            : dimension of output array (dim=dim_1!)
float initvalue      : all elements in line that could not be determined
                       are set to initvalue
float *data          : 2d input array  float data[dim_1,dim_2]
long  dim_1, dim_2   : dimensions of input array
float dummy, ddummy  : dummy and ddummy of input array
float f_21, f_22     : indices of integration limits (float indices, 
                       f_21<=f_22)
int   mode           : 0: integration, 1: average
int   testbit        : 1/0: do/dont print debug information

RETURN VALUE
float range          : averaging range

HISTORY
1999-12-26 Peter Boesecke
----------------------------------------------------------------------------*/
float RowProjection(float *line, long int dim, float initvalue,
                    float *data, long int dim_1, long int dim_2,
                    float dummy, float ddummy,
                    float f_21, float f_22, 
                    int mode, int testbit)
{ const char * RoutineName = "RowProjection";
  long  i_21, i_22;  // first and last index
  float r_21, r_22;  // distance to border before i_21/after i_22 
  float *pdata;      // pointer to data point
  float value, sum, weight;
  long  i_1, i_2;    // loop 
  float range;       // averaging range

  if (testbit) printf("+++ %s\n",RoutineName);

  /* row averaging  */
  i_21 = MAX2(0,ceil(f_21));         r_21 = (float) i_21 - f_21; 
  i_22 = MIN2(dim_2-1,ceil(f_22)-1); r_22 = f_22 - (float) i_22;

  if (testbit) { 
    printf(" f_21  = %g\n", f_21);
    printf(" f_22  = %g\n", f_22);
    printf(" i_21  = %d, r_21 = %g\n", i_21, r_21);
    printf(" i_22  = %d, r_22 = %g\n", i_22, r_22);
    printf(" dim_1 = %d, dim_2 = %d\n",dim_1,dim_2);
    printf(" dim   = %d\n",dim);
    if (mode) printf(" mode  = average\n"); 
      else printf(" mode  = integration\n");
    }

  if (dim!=dim_1) {
    printf("Programming error in call to %s: dim!=dim_1\n",RoutineName);
    exit(-1);
    }

  if (f_22>f_21) {
    // projection range
    range = f_22-f_21;
    /* The result of the row averaging is put to line */
    for (i_1=0;i_1<dim_1;i_1++) {
      weight = 0.0;
      sum = 0.0;
      // first interval: value must be interpolated
      if (Ipol2ld (data,dim_1,dim_2,dummy,ddummy,
                   i_1, f_21, &value )) {
          sum += value*r_21;
          weight += r_21;
        }
      // last interval: value must be interpolated
      if (Ipol2ld (data,dim_1,dim_2,dummy,ddummy,
                   i_1, f_22, &value )) {
          sum += value*r_22;
          weight += r_22;
        }

      for (i_2=i_21;i_2<i_22;i_2++) {
        // central intervals: values not interpolated
        pdata = ABSPTR(data,dim_1,dim_2,i_1,i_2);
        value = *pdata;
        if (NODUMMY(value, dummy, ddummy)) {
          sum += value;
          weight += 1.0;
          }
        } /* for i_2 ... */
      // output to line
      if (weight>0.0) 
         if (mode) line[i_1] = sum/weight; // average
            else line[i_1] = (sum/weight)*range; // integration
         else line[i_1] = initvalue;
      } /* for i_1 ... */
    } else {
    // projection range
    range = 0;
    /* fill with initvalue */
    for (i_1=0;i_1<dim;i_1++) line[i_1] = initvalue;
    } /* if (f_22>f_21) */

  if (testbit) printf("--- %s\n",RoutineName);

  return ( range );

} /* RowProjection */

/*---------------------------------------------------------------------------
NAME

   ColProjection --- projection of image columns on a single line

SYNOPSIS

   float ColProjection(float *line, long int dim, float initvalue,
                       float *data, long int dim_1, long int dim_2,
                       float dummy, float ddummy,
                       float f_11, float f_12, 
                       int mode, int testbit);

DESCRIPTION
Averaging of column-values between f_11 and f_12 in the 2d array data
(float data[dim_1,dim_2]). The result is written to the 1d array line.
(float line[dim]). dim and dim_2 must be equal. Line is initialized 
with initvalue. f_11 must be smaller or equal to f_12.

f_11 and f_12 should be calculated from the world coordinates col1 and 
col2 in the following way:

  f_11 = MAX2(-1,INDEX(col1, Off_1, Ps_1));
  f_12 = MIN2(I2Dim_1-1,INDEX(col2, Off_1,Ps_1));

Where Off_1 and Ps_1 are calculated with GetReferenceParameters.

ARGUMENTS
float *line          : 1d output array  float line[dim]
long  dim            : dimension of output array (dim=dim_2!)
float initvalue      : all elements in line that could not be determined
                       are set to initvalue
float *data          : 2d input array  float data[dim_1,dim_2]
long  dim_1, dim_2   : dimensions of input array
float dummy, ddummy  : dummy and ddummy of input array
float f_11, f_12     : indices of integration limits (float indices,
                       f_11<=f_12)
int   mode           : 0: integration, 1: average
int   testbit        : 1/0: do/dont print debug information

RETURN VALUE
float range          : averaging range

HISTORY
1999-12-26 Peter Boesecke
----------------------------------------------------------------------------*/
float ColProjection(float *line, long int dim, float initvalue,
                    float *data, long int dim_1, long int dim_2,
                    float dummy, float ddummy,
                    float f_11, float f_12, 
                    int mode, int testbit)
{ const char * RoutineName = "ColProjection";
  long  i_11, i_12;  // first and last index
  float r_11, r_12;  // distance to border before i_11/after i_12
  float *pdata;      // pointer to data point
  float value, sum, weight;
  long  i_1, i_2;    // loop
  float range;       // averaging range

  if (testbit) printf("+++ %s\n",RoutineName);

  /* column averaging  */
  i_11 = MAX2(0,ceil(f_11));         r_11 = (float) i_11 - f_11;
  i_12 = MIN2(dim_1-1,ceil(f_12)-1); r_12 = f_12 - (float) i_12;

  if (testbit) {
    printf(" f_11  = %g\n", f_11);
    printf(" f_12  = %g\n", f_12);
    printf(" i_11  = %d, r_11 = %g\n", i_11, r_11);
    printf(" i_12  = %d, r_12 = %g\n", i_12, r_12);
    printf(" dim_1 = %d, dim_2 = %d\n",dim_1,dim_2);
    printf(" dim   = %d\n",dim);
    if (mode) printf(" mode  = average\n");
      else printf(" mode  = integration\n");
    }

  if (dim!=dim_2) {
    printf("Programming error in call to %s: dim!=dim_2\n",RoutineName);
    exit(-1);
    }

  if (f_12>f_11) {
    // projection range
    range = f_12-f_11;
    /* The result of the column averaging is put to line */
    for (i_2=0;i_2<dim_2;i_2++) { 
      weight = 0.0;
      sum = 0.0;
      // first interval: value must be interpolated
      if (Ipol2ld (data,dim_1,dim_2,dummy,ddummy,
                   f_11, i_2, &value )) {
          sum += value*r_11;
          weight += r_11;
        }
      // last interval: value must be interpolated
      if (Ipol2ld (data,dim_1,dim_2,dummy,ddummy,
                   f_12, i_2, &value )) {
          sum += value*r_12;
          weight += r_12;
        }

      for (i_1=i_11;i_1<i_12;i_1++) {
        // central intervals: values not interpolated
        pdata = ABSPTR(data,dim_1,dim_2,i_1,i_2);
        value = *pdata;
        if (NODUMMY(value, dummy, ddummy)) {
          sum += value;
          weight += 1.0;
          }
        } /* for i_1 ... */

      // output to line
      if (weight>0.0) 
         if (mode) line[i_2] = sum/weight; // average
            else line[i_2] = (sum/weight)*range; // integration
         else line[i_2] = initvalue;
      } /* for i_2 ... */
    } else {  
    // projection range
    range = 0;
    /* fill with initvalue */
    for (i_2=0;i_2<dim;i_2++) line[i_2] = initvalue;
    } /* if (f_12>f_11) */

  if (testbit) printf("--- %s\n",RoutineName);

  return ( range );

} /* ColProjection */

/*---------------------------------------------------------------------------
NAME

  copy_amat --- copies an affine transformation matrix

SYNOPSIS
  
  int copy_amat(double t[3][2], double p[3][2]);

DESCRIPTION
    Copies the affine transformation matrix t to p.

RETURN VALUE
    0: Success

-----------------------------------------------------------------------------*/
int copy_amat(double t[3][2], double p[3][2])
{ register int i,j;
  for(j=0;j<2;j++) for (i=0;i<3;i++) p[i][j] = t[i][j];
  return(0);
} /* copy_amat */

/*---------------------------------------------------------------------------
NAME

  invert_amat --- inverts an affine transformation matrix

SYNOPSIS

  int invert_amat(double t[3][2], double p[3][2]);

DESCRIPTION

        The affine transformation T: (k,l) -> (i,j) is defined as:

             i = t[0][0]*k+t[1][0]*l + t[2][0];
             j = t[0][1]*k+t[1][1]*l + t[2][1];

        The inverse transformation matrix P: (i,j) -> (k,l) is defined as:

             k = p[0][0]*i+p[1][0]*j + p[2][0];
             l = p[0][1]*i+p[1][1]*j + p[2][1];

        If the determinante of the inverse transformation matrix 
        is smaller than 1e-6, invert_amat returns with error.

ARGUMENTS

   double t[3][2] : transformation matrix T
   double p[3][2] : inverted matrix P (can be identical to T)

RETURN VALUE
    0: Success
   -1: matrix inversion error (determinante too small)

-----------------------------------------------------------------------------*/
int invert_amat(double t[3][2], double p[3][2])
{ const eps = 1e-6;
  double dett;
  double tmp[3][2];

  /* calculation of inverse transformation matrix */

  dett = t[0][0]*t[1][1]-t[1][0]*t[0][1];

  if (fabs(dett)<eps) return(-1); // matrix inversion error

  tmp[0][0] = t[1][1]/dett; tmp[1][0] = -t[1][0]/dett;
    tmp[2][0] = (t[1][0]*t[2][1]-t[2][0]*t[1][1])/dett;
  tmp[0][1] = -t[0][1]/dett; tmp[1][1] = t[0][0]/dett;
    tmp[2][1] = (t[0][1]*t[2][0]-t[0][0]*t[2][1])/dett;

  copy_amat(tmp,p);

  return(0);

} /* invert_amat */

/*----------------------------------------------------------------------------
NAME

   AffineFloat2d --- Affine transformation of a 2d float image

SYNOPSIS

   AffineFloat2d         ( float *data, long int dim_1, long int dim_2,
                           float dummy, float ddummy,
                           float *out, long int odim_1, long int odim_2,
                           float odummy, double t[3][2], int isw, 
                           int *pstatus );

DESCRIPTION

        float t[3][2] : Transformation matrix data[i,j] -- out[k,l]
        
        if (isw > 0) {
             i=t[0][0]*k+t[1][0]*l+t[2][0];
             j=t[0][1]*k+t[1][1]*l+t[2][1];
           } else {
             k=t[0][0]*i+t[1][0]*j+t[2][0];
             l=t[0][1]*i+t[1][1]*j+t[2][1];
           }

        If, in the case of isw<=0, the determinante of the inverse 
        transformation matrix is smaller than 1e-6:
        (fabs(t[0][0]*t[1][1]-t[1][0]*t[0][1]) <  1e-6)
        AffineFloat2d returns without transformation and sets 
        status to FloatingPointError. 

        In case of success AffineFloat2d sets status to Success.

ARGUMENTS

 const float * data      (i) :  input image 
 long int idim_1, idim_2 (i) :  dimensions of input image
 float dummy, ddummy     (i) :  input dummy and ddummy
 float * out             (o) :  output image
 long int odim_1, odim_2 (i) :  dimensions of output image
 float odummy            (i) :  output dummy and ddummy
 float t[3][2]           (i) :  transformation matrix 
 int isw                 (i) :  parameter specifying processing contents
 int * pstatus           (o) :  output status

RETURN VALUE
 void 

HISTORY
 1999-12-27 V1.0 Peter Boesecke
----------------------------------------------------------------------------*/
void AffineFloat2d( float *data, long int dim_1, long int dim_2,
                    float dummy, float ddummy,
                    float *out, long int odim_1, long int odim_2,
                    float odummy, double t[3][2], int isw, 
                    int *pstatus )
{ double p00,p10,p20,p01,p11,p21;

  register long int i_1, i_2;
  float f_1, f_2;
  float value;
  float *pout;

  *pstatus = Failed;

  if (isw<=0) {
    /* invert matrix */
    if (invert_amat(t,t)) { *pstatus = FloatingPointError; return; }
   } // if (isw<=0)

  p00 = t[0][0]; p10 = t[1][0];   p20 = t[2][0];
  p01 = t[0][1]; p11 = t[1][1];   p21 = t[2][1];

  pout = out;
  for (i_2=0;i_2<odim_2;i_2++) {
    for (i_1=0;i_1<odim_1;i_1++) {
      f_1= (float) ( (p00*i_1)+(p10*i_2)+ p20 );
      f_2= (float) ( (p01*i_1)+(p11*i_2)+ p21 );

      if ( Ipol2ld (data, (int) dim_1, (int) dim_2, 
          dummy, ddummy, f_1, f_2, &value) ) {
          *pout = value;
        } else *pout = odummy; // Ipol2ld
      pout++;
      } // i_1
    } // i_2
  
  *pstatus = Success;

} /* AffineFloat2d */

/*-----------------------------------------------------------------------
NAME

   Affine --- Affine transformation of an image block

SYNOPSIS

   void Affine (  CmdBlk * pcb, ImgHeadBlk ihb[],
                  int InputBlock, int OutputBlock,
                  double t[3][2], int * pstatus );

DESCRIPTION
Affine changes the planar orientation of the data using the 
subroutine AffineFloat2d. The transformation is applied to world
coordinates that are defined by the reference system (pcb->RSys.V). 
If required, the values of pixel size and offset of the output 
image are updated after the transformation. The input image block
is block 1 and the output image block is 0. The output shift 
parameters are taken into account.

   Transformation of world coordinates:

      W' = A * W + B;

      W  = WORLD( I , O , P  ) = P  * (O +I );
      W' = WORLD( I', O', P' ) = P' * (O'+I');

      Transformation of indices:

      P' * (O'+I') = A * ( P  * (O +I ) ) + B

            <=> I' = A' * I + B';


           ( i0 )         ( o0 )       ( p0,  0 )
      I  = |    |  ; O  = |    | ; P = |        |
           ( i1 )         ( o1 )       ( 0 , p1 )


           ( i0' )          ( op0 )        ( pp0,  0 )
      I' = |     |  ; O'  = |     | ; P' = |         |
           ( i1' )          ( op1 )        ( 0 , pp1 )


                       ( T[0,0], T[1,0] )       ( T[2,0] )
      T = (A,B) :  A = |                | ; B = |        |
                       ( T[0,1], T[1,1] )       ( T[2,1] )


      Tp = (A',B') :

                   ( Tp[0,0], Tp[1,0] )
             A' =  |                  |
                   ( Tp[0,1], Tp[1,1] )

             A' =  inv(P') * A * P

                   ( 1/pp0 ,   0  )   ( T[0,0], T[1,0] )   ( p0,      0  )
                =  |              | * |                | * |             |
                   (   0  , 1/pp1 )   ( T[0,1], T[1,1] )   (   0    , p1 )

                   ( Tp[2,0] )
             B' =  |         |
                   ( Tp[2,1] )

                =  inv(P') * B + A' * O - O'

                   ( 1/pp0,  0 )   ( T[2,0] )          ( o0 )    ( op0 )
                =  |           | * |        | + A'  *  |    |  - |     |
                   ( 0,  1/pp1 )   ( T[2,1] )          ( o1 )    ( op1 )

ARGUMENTS
  CmdBlk * pcb            (i)   : command block
             pcb->RSys    (i)   : reference system
             pcb->Shift   (i)   : shift of output image (image 0)
  ImgHeadBlk ihb[block]   (i)   : image header block 
                                  (block=OutputBlock, InputBlock)
             ihb[]->Dim
             ihb[]->Offset
             ihb[]->PixSiz
             ihb[]->Center, ->Wavelength, ->SampleDistance
                    ihb[1].Data : input data array
                    ihb[0].Data : output data array
  int InputBlock                : input block number
  int OutputBlock               : output block number
  double t[3][2]          (i)   : world coordinates transformation matrix
  int * pstatus           (o)   : output status

HISTORY
1999-12-31 Peter Boesecke
-----------------------------------------------------------------------*/
void Affine (  CmdBlk * pcb, ImgHeadBlk ihb[],
               int InputBlock, int OutputBlock,
               double t[3][2], int * pstatus )
{ int j,k;
  int imax = pcb->ImgBlkLen;
  ImgHeadBlk *pin  = &ihb[InputBlock], 
             *pout = &ihb[OutputBlock];

  int isw=-1;
  double tp[3][2];
  double p0, p1, o0, o1;
  double pp0, pp1, op0, op1;

  double IndexCenIn_1, IndexCenIn_2, IndexCenOut_1, IndexCenOut_2;
  double pp[3][2];

//++++++++++  float Off_1[BlockNum], Off_2[BlockNum];
//++++++++++  float Ps_1[BlockNum], Ps_2[BlockNum];

  float *Off_1, *Off_2, *Ps_1, *Ps_2;
  Off_1 = (float *) malloc(sizeof(float)*imax);
  if (!Off_1) { *pstatus = NotEnoughMemoryAvailable; return;}
  Off_2 = (float *) malloc(sizeof(float)*imax);
  if (!Off_2) { *pstatus = NotEnoughMemoryAvailable; 
                 free(Off_1); return;}
  Ps_1 = (float *) malloc(sizeof(float)*imax);
  if (!Ps_1) { *pstatus = NotEnoughMemoryAvailable; 
               free(Off_1); free(Off_2); return;}
  Ps_2 = (float *) malloc(sizeof(float)*imax);
  if (!Ps_2) { *pstatus = NotEnoughMemoryAvailable; 
               free(Off_1); free(Off_2); free(Ps_1); return;}

  *pstatus = Success;

  if (imax<2) {
    printf(" ERROR in Affine: Not enough image blocks (%d / 2) found\n",
         imax);
    free(Off_1); free(Off_2); free(Ps_1); free(Ps_2);
    *pstatus=Failed; return; }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) {
     free(Off_1); free(Off_2); free(Ps_1); free(Ps_2); 
     return; }

  p0  = (double) Ps_1[InputBlock];   p1 = (double) Ps_2[InputBlock];
  o0  = (double) Off_1[InputBlock];  o1 = (double) Off_2[InputBlock]; 
  pp0 = (double) Ps_1[OutputBlock];  pp1 = (double) Ps_2[OutputBlock];
  op0 = (double) Off_1[OutputBlock]; op1 = (double) Off_2[OutputBlock];

  free(Off_1); free(Off_2); free(Ps_1); free(Ps_2);

  /* include world coordinates in transformation matrix tp */
  tp[0][0] = t[0][0] * (p0/pp0);
  tp[1][0] = t[1][0] * (p1/pp0);
  tp[2][0] = t[2][0] / pp0 + tp[0][0] * o0 + tp[1][0] * o1 - op0;
  tp[0][1] = t[0][1] * (p0/pp1);
  tp[1][1] = t[1][1] * (p1/pp1);
  tp[2][1] = t[2][1] / pp1 + tp[0][1] * o0 + tp[1][1] * o1 - op1;

  if (pcb->TestBit) {
    for (k=0;k<2;k++) for (j=0;j<3;j++)
        printf("tp[%d][%d] = %g\n",j,k,tp[j][k]);
    }

  AffineFloat2d ( pin->Data, pin->Dim[1], pin->Dim[2],
                  pin->Dummy.V, pin->DDummy.V,
                  pout->Data, pout->Dim[1], pout->Dim[2],
                  pout->Dummy.V, tp, isw,
                  pstatus );
  if (*pstatus!=Success) return; 

  /* transformation of unused image parameters */

     if (isw>0) { copy_amat(tp,pp); } else {
       if (invert_amat(tp,pp)) { *pstatus=FloatingPointError; return; }
       } // if (isw>0)

     if (pcb->TestBit) {
       for (k=0;k<2;k++) for (j=0;j<3;j++)
           printf("pp[%d][%d] = %g\n",j,k,pp[j][k]);
       }

      switch (pcb->RSys.V) {
        case IO_Center: // update pixel size
          if ((pin->PixSiz[1].I)&&(pin->PixSiz[2].I)) {
            pout->PixSiz[1].V =
              pin->PixSiz[1].V/sqrt(pp[0][0]*pp[0][0]+pp[0][1]*pp[0][1]);
            pout->PixSiz[2].V =
              pin->PixSiz[2].V/sqrt(pp[1][0]*pp[1][0]+pp[1][1]*pp[1][1]);
            }
          break;
        case IO_Array: // like IO_Image
        case IO_Image: // update pixel size and position of center like IO_Real
          if ((pin->PixSiz[1].I)&&(pin->PixSiz[2].I)) {
            pout->PixSiz[1].V =
              pin->PixSiz[1].V/sqrt(pp[0][0]*pp[0][0]+pp[0][1]*pp[0][1]);
            pout->PixSiz[2].V =
              pin->PixSiz[2].V/sqrt(pp[1][0]*pp[1][0]+pp[1][1]*pp[1][1]);
            }
        case IO_Real: // update position of center
          if ((pin->Center[1].I)&&(pin->Center[2].I)) {

                IndexCenIn_1 = I2INDEX(pin->Center[1].V,pin->Offset[1].V);
                IndexCenIn_2 = I2INDEX(pin->Center[2].V,pin->Offset[2].V);

            //       W' = A * W + B
            IndexCenOut_1=pp[0][0]*IndexCenIn_1+pp[1][0]*IndexCenIn_2+pp[2][0];
            IndexCenOut_2=pp[0][1]*IndexCenIn_1+pp[1][1]*IndexCenIn_2+pp[2][1];

            pout->Center[1].V = INDEX2I(IndexCenOut_1,pout->Offset[1].V);
            pout->Center[2].V = INDEX2I(IndexCenOut_2,pout->Offset[2].V);
            }

           if (OutputBlock==0) {
             /* Off_1[0], Off_2[0] includes shift. It must be subtracted. */
             if (pcb->Shift[1].I) pout->Center[1].V -= pcb->Shift[1].V;
             if (pcb->Shift[2].I) pout->Center[2].V -= pcb->Shift[2].V;
             }

        } // switch

    return;

} /* Affine */

/*---------------------------------------------------------------------------
ArgvFilenames

Ask for all file names in the following way:
  input file 1 [input.edf] :
  input file 2 [<input file 1>] :
  ...
  output file [output.edf] :

  first image [<minimum image number>] :
  last image [<maximum image number>] :
  increment [1] :

  output dummy [<dummy in first image>] :

If blkno >= 1 the header block of the first image is read into ihb[1].
The routine must be used for all images at once.
---------------------------------------------------------------------------*/
void ArgvFilenames ( CmdBlk * pcb, ImgBlk ib[], ImgHeadBlk ihb[],
                     int block_1st, int block_lst, int * pstatus)
{ int blkno, first_block;
  long int image_1st = 1, image_lst = 1;
  char linebuffer[IO_len];
  const char * RoutineName = "ArgvFilenames";

  SetSaxsError( &pcb->seb, SRName, RoutineName, NULL, Success, 0, NULL );

  first_block = MAX2(1,block_1st);

  /* ---  File names and open modes --- */

  for (blkno = first_block;blkno <= block_lst;blkno++) {

    /*--- Open mode ib[blkno].OpenMode of input files */
    if (!ib[blkno].OpenMode.I) {
      ib[blkno].OpenMode.V = IO_Old | IO_FileProtect;
      ib[blkno].OpenMode.I = TRUE;
      }

    /*--- File name ib[blkno].Name of input files */
    (void) sprintf(linebuffer,"Input sequence %d",blkno);
    if (blkno==1)
      argv_filename( pcb, linebuffer, ib, 1, DefaultInput, pstatus);
    else argv_filename( pcb,linebuffer,ib,blkno,ib[blkno-1].Name.V,pstatus );

    /* Exit in all cases after an abort from the first argument */
    if (*pstatus!=Success) return;

    OpenImageFile(pcb,ib,blkno,ib[blkno].Name.V,ib[blkno].FFirst.V,
                  IO_Old|IO_FileProtect,pstatus);
    if (*pstatus!=Success) return;

    /* Search for minimum and maximum image number */
    (void) SearchMinMaxImage (pcb,ib,blkno,&image_1st,&image_lst,pstatus);
    if (*pstatus!=Success) return;
    if (!ib[blkno].First.I) ib[blkno].First.V = image_1st;
    if (!ib[blkno].Last.I)  ib[blkno].Last.V  = image_lst;

  } /* for input files */

  if (block_1st<=0) {
    /*--- OpenMode of output file */
    if (!ib[0].OpenMode.I) {
      ib[0].OpenMode.V = IO_ImageProtect;
      ib[0].OpenMode.I = TRUE;
      }

    /*--- File name ib[0].Name of output */
    argv_filename( pcb, "Output sequence", ib, 0, DefaultOutput, pstatus);
    if (*pstatus!=Success) return;

  } /* output file */

  /* ---  Image numbers --- */

  if ((first_block==1) && (pcb->ImgBlkLen>1)) {

    /*--- Argument : number of first image */
    argv_long(pcb,"First image of input sequence 1",&ib[1].First,ib[1].First.V,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : number of last image */
    if (pcb->argc==*pcb->parg_no+1) image_lst = ib[1].First.V;
      else image_lst = ib[1].Last.V;
    argv_long( pcb,"Last image of input sequence 1", &ib[1].Last,image_lst,
               pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : increment (default = 1) */
    argv_long( pcb,"Increment of input sequence 1", &ib[1].Inc, 1 , pstatus);
    if (*pstatus!=Success) return;

    (void) ReadImageHeader( pcb, ib, 1, ib[1].First.V, ihb, pstatus);
    if (*pstatus!=Success) return;
    if (pcb->TestBit) PrintImageHeaderBlock(1,&ihb[1]);

    CloseImageFile( pcb, ib, 1, pstatus) ;
    if (*pstatus!=Success) return;

    /* Defaults for output file */
    if (!ib[0].Dummy.I) ib[0].Dummy.V  = ihb[1].Dummy.V;
    if (!ib[0].Dim[1].I) ib[0].Dim[1].V = ihb[1].Dim[1];
    if (!ib[0].Dim[2].I) ib[0].Dim[2].V = ihb[1].Dim[2];
  } /* first input file */

  for (blkno = MAX2(2,first_block);blkno<=block_lst;blkno++) {
    /*--- image numbers of all other images */
    (void) sprintf(linebuffer,"First image of input sequence %d",blkno);
    argv_long( pcb,linebuffer, &ib[blkno].First, ib[blkno].First.V, pstatus);
    if (*pstatus!=Success) return;

    CloseImageFile( pcb, ib, blkno, pstatus) ;
    if (*pstatus!=Success) return;

  } /* all other input files */

  if (block_1st <= 0) {
    if ( (pcb->ImgBlkLen>1) && (ihb[1].I) ) {
      NewImageHeader(pcb, ib, 0, image_1st, ihb, &(ihb[1]), pstatus);
      if (*pstatus!=Success) return;
      } else {
      NewImageHeader(pcb, ib, 0, image_1st, ihb, (ImgHeadBlk *)NULL, pstatus);
      if (*pstatus!=Success) return;
      }

    /*--- Argument : output dummy */
    argv_float( pcb, "Output dummy", &ib[0].Dummy, ib[0].Dummy.V, pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 1 */
    argv_long(pcb,"Output dimension 1", &ib[0].Dim[1],ib[0].Dim[1].V,pstatus);
    if (*pstatus!=Success) return;

    /*--- Argument : output dimension 2 */
    argv_long(pcb,"Output dimension 2", &ib[0].Dim[2],ib[0].Dim[2].V, pstatus);
    if (*pstatus!=Success) return;

    }

} /* ArgvFilenames */

/*****************************************************************************/
