/*+++
1 SaxsOption

2 PURPOSE
Reading of options from the program parameter list and creation of command, variable and image blocks for image analysis programs.

2 HISTORY
04-Jan-1995 PB creation
06-Jan-1995 PB IO_Version 1.2
07-Jan-1995 PB IO_Version 1.4 rjctnum, version numbers in ShowHelp
10-Jan-1995 PB manually realigned for RED
11-Jan-1995 PB some output formatting changed
12-Jan-1995 PB arrays for Dim, Shift, Center etc. enlarged from Dim[2] to [3].
12-Jan-1995 PB Shift (float)
16-Jan-1995 PB V2.3 file opening modi in imageoption.h
28-Jan-1995 PB V2.4 Title, SampleDistance [m], WaveLength [m] added,
	       GetOption changed
29-Jan-1995 PB V2.42 CommandOption and ImageOption dynamically generated and
	       pointer in command block, after first call of ReadOption the
	       option descriptors are freed. ReadOption can only be used once!
	       DefineLineOption added, -rsys option added
29-Jan-1995 PB V2.5 final version
30-Jan-1995 PB V2.51 USys parameter for user reference system added
07-Feb-1995 PB V2.6  Open modes changed
11-Feb-1995 PB V2.7  image block with Stream
12-Feb-1995 PB V2.8  image block with FullName
26-Feb-1995 PB V2.9  reftostr added
05-Mar-1995 PB V2.91 IO_DontScale added
08-Mar-1995 PB V2.92 Help suppressed for ImgBlkLen <2 and <1.
08-Apr-1995 PB V3.0  pointer ib to image block in CmdBlk added.
08-Apr-1995 PB V3.1  IO_FlgOption added
04-Jun-1995 PB V4.0  Changed to 4.0 at the same time where the definition of
                     saxsref was changed in saxs_definition. No change in
                     imageoption. Programs with imageoption Version>4.0
                     are using the new saxsref definition.
19-Jun-1995 PB V4.1  for a corrected version of saxs_definitions
09-Sep-1995 PB V5    a) wild card for options: a wild card does neither
                        change the value (.V) nor the the i-flag (.I).
                        The option is IGNORED for the parameter, e.g.
                        " -opix = 0.4 +opix = = " sets only PixSiz[2]:
                        PixSiz[2].I = -1; PixSiz[2].V = 0.4.
                     b) options can be repeated, the last option is valid
                     c) It can be tested whether a "+" or "-"-option has
                        been used:
                        xxx.I > 0 if set by "+"-option
                        xxx.I == 0 if not set
                        xxx.I < 0 if set by "-"-option
                        The test of an option must be: if (xxx.I)  ...
                     d) pcb->Message pointer added for display of errors etc.
                     e) saxs_definitions.h included for display of SD_Version
                     f) floatexpr and longexpr used in ScanFloat and ScanLong
15-Sep-1995 PB V5.1  a) prompt routines
                     b) default for pcb->Prompt.V changed to TRUE
14-Nov-1995 PB V5.2  a) new prompt routines
                     b) default for pcb->USys.V is always pcb->RSys.V
                     c) new rebin option i<n>bin <bin_1> <bin_2>
                     d) pcb->ErrorValue added
24-Nov-1995 PB V5.3  a) In command block Message, ErrorValue and
			DataErrorValue replaced by SaxsErrBlk seb.
03-Dec-1995 PB V5.4  StreamOpenMode
25-Dec-1995 PB V5.41 int mode --> long mode
26-Dec-1995 PB V5.42 long --> long int, mhlp, mver --> const char *
29-Mar-1996 PB V5.6  clip function -i<N>min and -i<N>max
18-May-1996 PB V5.61 number of float options increased to 23
19-May-1996 PB V5.7  gnaw function -i<N>gna <d> <d>
06-Oct-1996 PB V5.71 constants of reftostr declared outside
10-Oct-1996 PB V5.72 FLT_MIN, FLT_MAX calculated from FLT_MIN_EXP etc.
                     if not defined
10-Oct-1996 PB V5.8  imageoption renamed to SaxsOption
14-Oct-1996 PB V5.9  SaxsOptionVersion
14-Nov-1996 PB V5.91 hidden options
17-Mar-1998 PB V6.00 GeneralBlock
01-Jun-1998 PB V6.01 BslInputByteOrder
24-Jun-1999 PB V6.1  edf_history (include edfio.h)
26-Jun-1999 PB V6.11 Time keyword 
21-Dec-1999 PB V6.12 IO_DontScale replaced with IO_Dummy
31-Dec-1999 PB V6.13 new reference system BEAM 
10-Jan-2000 PB V6.14 new reference systems CENTER and NORMAL,
                     NORMAL replaces BEAM
19-Mar-2000 PB V6.15 memory added, mem=0 excluded
31-Mar-2000 PB V6.16 MaxFloatOpt increased to 25 (from 23)
07-Apr-2000 PB V6.17 option -mlw <long> added (max_line_width)
01-Mai-2000 PB V6.18 ShowDataTypes()
30-Jul-2000 PB V6.19 Orientation
31-Aug-2000 PB V6.20 MaxFloatOptions increased from 25 to 30
15-Sep-2000 PB V6.21 MaxLineOptions increased from 4 to 5 
16-Nov-2000 PB V6.22 Image block: IO_long Dim[4] instead of Dim[3]
04-Dec-2000 PB V6.23 SaxsExpression.h
06-Jul-2001 PB V6.24 Stream default is -1
08-Jul-2001 PB V6.25 FFirst, FLast, FInc, argv_filename, FullName->FileName
09-Jul-2001 PB V6.26 ib[].FileNumber and ib[].FileName are set by OpenImageFile
09-Jul-2001 PB V6.27 repeat command +r -r removed
11-Jul-2001 PB V6.28 new command option: -add (number of output images to add) 
02-Sep-2001 PB V6.29 OutputDataType, OutputDataValueOffset
13-Sep-2001 PB V6.30 SaxsHistoryKey
16-Sep-2001 PB V6.31 PassHeader

2 OPTION KEYWORDS
output file <N> = 0
input file  <N> > 0
ib<N>OpenMode                         -i<N>mod  <InputOpenMode>M
ib<N>Name                             -i<N>nam  <Name>s
ib<N>Memory                           -i<N>mem  <MemNum>l
ib<N>First 	                      -i<N>fst  <First>l
ib<N>Last	                      -i<N>lst  <Last>l
ib<N>Inc	                      -i<N>inc  <Increment>l
ib<N>Const                            -i<N>con  <Const>f
ib<N>Factor                           -i<N>fac  <Factor>f
ib<N>Bin[1]      ib<N>Bin[2]          -i<N>bin  <Bin1>l <Bin2>l
ib<N>MinClip                          -i<N>min  <MinClip>
ib<N>MaxClip                          -i<N>max  <MaxClip>
ib<N>Gnaw[1]     ib<N>Gnaw[2]         -i<N>gna  <Gnaw1> <Gnaw2>

ib<N>Title                            -i<N>tit  <Title>s
ib<N>Time                             -i<N>time <Time>s
ib<N>Dim[1]      ib<N>Dim[2]          -i<N>dim  <Dim1>l <Dim2>l
ib<N>Dummy                            -i<N>dum  <Dummy>f
ib<N>DDummy                           -i<N>ddum <DDummy>f
ib<N>Offset[1]   ib<N>Offset[2]       -i<N>off  <Offset1>f <Offset2>f
ib<N>Center[1]   ib<N>Center[2]       -i<N>cen  <Center1>f <Center2>f
ib<N>PixSiz[1]   ib<N>PixSiz[2]       -i<N>pix  <PixSiz1>f <PixSiz2>f
ib<N>SampleDistance                   -i<N>dis  <SampleDistance>f
ib<N>WaveLength                       -i<N>wvl  <WaveLength>f

output file
ib<0>OpenMode                         -omod  <OutOpenMode>M
ib<0>Name                             -onam  <OutName>s
ib<0>Memory                           -omem  <MemNum>l
ib<0>First                            -ofst  <OutFirst>l
ib<0>Last                             -olst  <OutLast>l
ib<0>Inc                              -oinc  <OutInc>l
ib<0>Const                            -ocon  <Const>f
ib<0>Factor                           -ofac  <Factor>f
ib<0>Bin[1]        ib<0>Bin[2]        -obin  <Bin1>l <Bin2>l
ib<0>MinClip                          -omin  <MinClip>
ib<0>MaxClip                          -omax  <MaxClip>
ib<0>Gnaw[1]       ib<0>Gnaw[2]       -ogna  <Gnaw1> <Gnaw2>

ib<0>Title                            -otit  <Title>s
ib<0>Time                             -otime <Time>s
ib<0>Dim[1]        ib<0>Dim[2]        -odim  <OutDim1>l <OutDim2>l
ib<0>Dummy                            -odum  <OutDummy>f
ib<0>DDummy                           -oddum <OutDDummy>f
ib<0>Offset[1]     ib<0>Offset[2]     -ooff  <OutOffset1> <OutOffset2>
ib<0>Center[1]     ib<0>Center[2]     -ocen  <Center1>f <Center2>f
ib<0>PixSiz[1]     ib<0>PixSiz[2]     -opix  <OutPixSiz1>f <OutPixSiz2>f
ib<0>SampleDistance                   -odis  <SampleDistance>f
ib<0>WaveLength                       -owvl  <WaveLength>f

Commands
Help      extended      +h
	  simple        -h
Promptf   on            +p
          off           -p
Test      on            +test
	  off           -test
cb.Shift[1]        cb.Shift[2]        -cshft <Shift1>f <Shift2>f
cb.Arg[0].V			      -<nameXXX> <value>f
...
cb.Arg[IO_MaxFloatOpt-1].V	      -<nameZZZ> <value>f
cb.Num[0].V			      -<nameAAA> <value>l
...
cb.Num[IO_MaxIntOpt-1].V	      -<nameCCC> <value>l
cb.Lin[0].V			      -<nameDDD> <value>s
...
cb.Lin[IO_MaxLinOpt-1].V	      -<nameFFF> <value>s

OpenMode
n+fp+ip new file created,
	old file write protected, all images write protected
n-fp+ip new file created,
	any old file deleted, all images write protected
n+fp-ip new file created,
	old file write protected, images can be deleted
n-fp-ip new file created, any old file deleted, images can be deleted
o+fp+ip old file opened, all images write protected
o+fp-ip old file opened, images can be deleted
a+ip     open either old or new, all images write protected
a-ip     open either old or new, images can be deleted

2	EXAMPLE
#	include <errno.h>
#	include <stdio.h>
#	include <fcntl.h>
#	include <stddef.h>
#	include "SaxsInput.h"
#	include "SaxsOption.h"
#	define ImageNum 2
main ( int argc, char * argv[] )
{
	const char * MyHelp    = "HELP: This is an example program";
	const char * MyVersion = "This is my private version";
	int status;
	int arg_no;
	const int RejectNum = 1;
	CmdBlk cb;
	ImgBlk ib[ImageNum];
	InitOptions( MyHelp, MyVersion, RejectNum, &cb, ib, ImageNum );
	arg_no = 0;
	ReadOptions(argv, &arg_no, &cb, ib, &status);
	if (status!=Success) ReportSaxsError( &cb.seb, 1 );
	PrintBlocks ( &cb, ib, ImageNum );
	MainProgram(&cb,ib);
}
2	CHANGE OF OPTIONS AND BLOCKS
	To change image and command blocks the following manual procedure
	is necessary:
- Change of block definition
a) Change of structure definition (CmdBlk or ImgBlk).
b) Update of initialisation procedure (InitCommandBlock or InitImageBlock).
   The .I values have to be set to FALSE.
c) Update of print block procedure (PrintCommandBlock or PrintImageBlock).
- Add options
a) Add option string to preprocessor definitions of command options or
   image option strings, proceeded with capital S, e.g. # define SMySize 10.
b) Add a new enum constant preceeded with capital N to CmdOptions or
   ImgOptions, e.g. enum { ... , NMySize, ..., IO_CmdOptLen }. Use the same
   name as in a.
c) NO MANUAL CHANGE NECESSARY V2.4 PB 28-Jan-1995.
   Last constant in enum must be IO_CmdOptLen or IO_ImgOptLen respectively.
   The option arrays are dynamically generated in InitOptions and defined as:
   OptDsc CommandOption[IO_CmdOptLen]; OptDsc ImageOption[IO_ImgOptLen].
   The pointers to the option arrays are stored in the command block.
   After leaving ReadOption the corresponding memory is released and the
   pointers are not valid any more.
d) Update the initialisation procedure InitCommandOptions or InitImageOptions.
e) Add the corresponding option number in DoCmdOptions or for image options
   in DoImgOptions.
f) After compilation the new option should appear by calling the program with
   the option +test, e.g.  TestImageOption +test -<option string> ...
---*/

/****************************************************************************/
/* SaxsOption.c
/****************************************************************************/
# define SO_Version "SaxsOption V6.31 Peter Boesecke 2001-09-16"

#	include <errno.h>
#	include <stdio.h>
#	include <fcntl.h>
#	include <stddef.h>
#	include <float.h>

#       include "SaxsOption.h"
#       include "SaxsFilename.h"

# ifndef FLT_MAX
#   define FLT_MAX pow(10,FLT_MAX_EXP)
# endif
# ifndef FLT_MIN
#   define FLT_MIN pow(10,FLT_MIN_EXP)
# endif
#ifndef ROUND
# define ROUND( x ) floor( ( x ) + 0.5 )
#endif

# define DefaultFilename "datafile"DefaultExtension

/* Fixed command option strings, small characters only! */
# define STestBit	"test"
# define SHelp		"h"
# define SPrompt	"p"
# define SShift		"shft"
# define SRSys		"rsys"
# define SUSys		"usys"
# define SGeneralBlock  "gblk"
# define SPassHeader    "pass"
# define SOutputDataType "type"
# define SOutputDataValueOffset "dvo"
# define SBslInputByteOrder "bibo"
# define SAdd           "add"
# define SMaxLineWidth  "mlw"
/* Fixed image option strings, small characters only! */
# define OutChar	'o'
# define InChar		'i'
# define SOpenMode	"mod"
# define SName		"nam"
# define SMemory        "mem"
# define SFirst		"fst"
# define SLast		"lst"
# define SInc		"inc"
# define SMinClip       "min"
# define SMaxClip       "max"
# define SConst		"con"
# define SFactor	"fac"
# define SGnaw          "gna"
# define SBin           "bin"
# define STitle		"tit"
# define STime          "time"
# define SDim		"dim"
# define SDummy		"dum"
# define SDDummy	"ddum"
# define SOffset	"off"
# define SCen   	"cen"
# define SPixSiz	"pix"
# define SSampleDistance "dis"
# define SWaveLength	"wvl"
# define SOrientation   "ori"

enum CmdOptions
{	NEmpty=0,	  NTestBit,       NHelp,          NPrompt,
	NShift,           NRSys,          NUSys,          NGeneralBlock,
        NPassHeader,      NOutputDataType,  NOutputDataValueOffset,         
        NBslInputByteOrder,               NAdd,           NMaxLineWidth,
        NArgFst,          NArgLst=NArgFst+IO_MaxFloatOpt-1,
	NNumFst,	  NNumLst=NNumFst+IO_MaxIntOpt-1,
	NLinFst,	  NLinLst=NLinFst+IO_MaxLineOpt-1,
	NFlgFst,	  NFlgLst=NFlgFst+IO_MaxFlagOpt-1,
	NCmdLast,	  IO_CmdOptLen };

enum ImgOptions
{    /* NEmpty=0, */	NOpenMode=1,    NName,          NMemory,
        NFirst,         NLast,		NInc,           NMinClip,
	NMaxClip,       NConst,		NFactor,	NBin,
        NGnaw,          NTitle,		NTime,		NDim,
        NDummy,         NDDummy,	NOffset,	NCenter,
        NPixSiz,        NSampleDistance,NWaveLength,	NOrientation,
        NImgLast,       IO_ImgOptLen };

/* Parameter type */
enum {
	IO_tpctrl=0,    IO_tpline,      IO_tplong,      IO_tpfloat,
	IO_tpmode,	IO_tprsys };

/* Reference system code */
#	define SArray  "array"
#	define SImage  "image"
#       define SCenter "center"
#	define SReal   "real"
#       define SNormal "normal"
#	define SSaxs   "saxs"

/* File opening code */
#	define SNewIP     "n+ip"
  /* = new file created, images write protected */
#	define SNewFP     "n+fp"
  /* = new file created, file write protected */
#	define SNew       "n"
  /* = new file created,  no protection */
#	define SOldIP     "o+ip"
  /* = old file opened, all images write protected */
#	define SOldFP     "o+fp"
  /* = old file opened, file write protected */
#       define SOld       "o"
  /* = old file opened, no protection */
#	define SAnyIP     "a+ip"
  /* = open either old or new, images write protected */
#       define SAnyFP     "a+fp"
  /* = open either old or new, , file write protected */
#	define SAny       "a"
  /* = open either old or new, no protection */

/*----------------------------------------------------------------------------
 String constants
----------------------------------------------------------------------------*/
const char * SOName                = "SaxsOption";
const char * ErrorOptionDuplicated = "ERROR: Option duplicated or reserved: ";
const char * ErrorOptionDefnFailed = "ERROR: Option definition failed: ";
const char * ErrorCmdArrayTooShort = "ERROR: Command option array too short";
const char * ErrorImgArrayTooShort = "ERROR: Image option array too short";
const char * ErrorCannotAllcMemory = "ERROR: Cannot allocate memory";

const char * ReferenceArray        = "Array";
const char * ReferenceImage        = "Image";
const char * ReferenceCenter       = "Center";
const char * ReferenceReal         = "Real";
const char * ReferenceNormal       = "Normal";
const char * ReferenceSaxs         = "Saxs";
const char * ReferenceUnknown      = "<unknown>";

/*----------------------------------------------------------------------------
1 SaxsOptionVersion

2 DESCRIPTION
  Returns a pointer to the version string.
  char * SaxsOptionVersion ( void )

2 HISTORY
  14-Oct-1996 PB
----------------------------------------------------------------------------*/
char * SaxsOptionVersion ( void )
{ static char * Version = SO_Version;
  return ( Version );
} /* SaxsOptionVersion */

/*----------------------------------------------------------------------------
 Private include files
----------------------------------------------------------------------------*/
# include "edfio.h"        /* included for edf_history and edf_showdatatypes */
# include "SaxsPrompt.inc" /* included because file otherwise too long */

/*--------------------------------------------------------------------------
NAME
  scan_for_filenumber --- extract and convert a parameter 

SYNOPSIS
  int scan_for_filenumber( IO_long *pNumber, const char * fname, int parno,
                           int opt_onoff, int * pstatus );

DESCRIPTION
  Extract parameter parno from fname, scan it as float expression and
  convert it to IO_long *pNumber. If the parameter was found and could be
  converted the .I-flag is set to opt_onoff. The function returns the number of
  scanned parameters (0 or 1). In case of success the function *pstatus
  is set to success otherwise to the corresponding error status.

HISTORY
  2001-07-08 PB
--------------------------------------------------------------------------*/
int scan_for_filenumber( IO_long *pNumber, const char * fname, int parno,
                         int opt_onoff, int * pstatus )
{ char Number[IO_len];
  int scanned=0;

  *pstatus = Success;

  if ( filename_parameter ( Number, IO_len, fname, parno ) ) {
    scanned=1;
    if (strlen(Number)>0) {
      pNumber->I = opt_onoff;
      pNumber->V = (long) ROUND( doubleexpr( Number, pstatus) );
      }
  }

  return(scanned);

} // scan_for_filenumber

/*--------------------------------------------------------------------------
NAME
   extract_filenumbers --- extract FFirst, FLast, FInc

SYNOPSIS
   int extract_filenumbers( ImgBlk *pib, IO_line *pName,
                            int opt_onoff, int * pstatus );

DESCRIPTION
   Scans p->Name.V for pib->FFirst.V, pib->FLast.V and 
   pib->FInc.V. The return value is the number of found parameters.

HISTORY
  2001-07-08 PB
--------------------------------------------------------------------------*/
int extract_filenumbers( ImgBlk *pib, IO_line *pName,
                         int opt_onoff, int * pstatus )
{ int found=0;

  if (!pName->I) return(found);

  /* scan Name for file numbers */
  found+=scan_for_filenumber( &(pib->FFirst), pName->V, 1,
                             opt_onoff, pstatus );
  if (*pstatus!=Success) return(found);
  found+=scan_for_filenumber( &(pib->FLast), pName->V, 2,
                             opt_onoff, pstatus );
  if (*pstatus!=Success) return(found);
  found+=scan_for_filenumber( &(pib->FInc), pName->V, 3,
                              opt_onoff, pstatus );
  if (*pstatus!=Success) return(found);

  return(found);

} // extract_filenumbers

/*--------------------------------------------------------------------------
NAME
   argv_filename --- prompt for filename and filenumbers of ib[blkno]

SYNOPSIS
  void argv_filename ( CmdBlk * pcb,
                       const char * promptline, ImgBlk ib[], int blkno,  
                       const char * defval, int * pstatus);
DESCRIPTION
  Reads a filename with parameters from the argument list and from the 
input file using the functions argv_line and extract_filenumbers. 
If parameters are found it modifies ib[blkno].Name, ib[blkno].FFirst, 
ib[blkno].FLast and ib[blkno].FInc. 

HISTORY
  2001-07-08 PB
--------------------------------------------------------------------------*/
void argv_filename ( CmdBlk * pcb,
                     char * promptline, ImgBlk ib[], int blkno, 
                     char * defval, int * pstatus)
{ 
  argv_line( pcb,promptline,&ib[blkno].Name,defval,pstatus);
  if (*pstatus!=Success) return;

  extract_filenumbers( &ib[blkno], &(ib[blkno].Name), TRUE, pstatus );
  if (*pstatus!=Success) return;

} // argv_filename

/*---------------------------------------------------------------------------
1 reftostr

2 PURPOSE
  Returns a pointer to a string with the name of the reference system

2 CALL
  char *reftostr(long int refsys);

2 HISTORY
  Feb-1995 Peter Boesecke creation
---------------------------------------------------------------------------*/
char *reftostr(long int refsys)
{
  char *pcref;

  switch ((int) refsys) {
    case IO_Array  : pcref = (char *) ReferenceArray; break;
    case IO_Image  : pcref = (char *) ReferenceImage; break;
    case IO_Center : pcref = (char *) ReferenceCenter; break;
    case IO_Real   : pcref = (char *) ReferenceReal; break;
    case IO_Normal : pcref = (char *) ReferenceNormal; break;
    case IO_Saxs   : pcref = (char *) ReferenceSaxs; break;
    default        : pcref = (char *) ReferenceUnknown;
    } /* switch */

  return(pcref);
} /* reftostr */

/*--------------------------------------------------------------------------*/
/* ShowSys : Print reference systems
/*--------------------------------------------------------------------------*/
void ShowSys ( void )
{
printf("\n");
printf("Use -%s and -%s with one of the following reference systems : \n",\
       SRSys, SUSys);
printf(" % 8s, % 8s, % 8s, % 8s, % 8s, % 8s\n",
       SArray, SImage, SCenter, SReal, SNormal, SSaxs);
} /* ShowSys */

/*--------------------------------------------------------------------------*/
/* ShowModes : Print File Opening Modes
/*--------------------------------------------------------------------------*/
void ShowModes ( void )
{
printf("Use -o%s with one of the following modes : \n",SOpenMode);
printf(" % 8s, % 8s, % 8s\n % 8s, % 8s, % 8s\n % 8s, % 8s, % 8s\n",
SNewFP, SNewIP, SNew, SOldFP, SOldIP, SOld, SAnyFP, SAnyIP, SAny );
}

/*--------------------------------------------------------------------------*/
/* ShowOptions : Show defined options of dsc[]  ( for Help )
/*--------------------------------------------------------------------------*/
void ShowOptions ( char * c, OptDsc dsc[] )
{
	char line[IO_len];
	const int NperLine = 4;
	register int k;
	int cnt;
	char tp[2];

	cnt=0;
	tp[1] = (char) NULL;
	for (k=1;(k<dsc[0].npar);k++)
	  if ( (dsc[k].code[0]) && (dsc[k].code[0]!='#') ) {
	    if ((cnt % NperLine)==0) printf(" \n ");
	    switch (dsc[k].ptyp) {
	      case IO_tpctrl: tp[0]='+'; break;
	      case IO_tplong: tp[0]='d'; break;
	      case IO_tpfloat: tp[0]='f'; break;
	      case IO_tpline: tp[0]='s'; break;
	      case IO_tpmode: tp[0]='M'; break;
	      case IO_tprsys: tp[0]='R'; break;
	      default : tp[0]='?'; break;
	      }
            printf("% 10s [% 1d, %s]", \
                   strcat(strcpy(line,c),dsc[k].code),dsc[k].npar,tp);
            cnt += 1;
            }
	printf("\n");
	
} /* ShowOptions */

/*--------------------------------------------------------------------------*/
/* ShowDataTypes : Print Conversion Table of Supported Data Types 
/*--------------------------------------------------------------------------*/
void ShowDataTypes ( void )
{
  printf("\n");
  printf("Conversion of file data types to internal data types\n");
  edf_showdatatypes( False );
}

/*--------------------------------------------------------------------------*/
/* PrintOptions : Print Option Descriptors
/*--------------------------------------------------------------------------*/
void PrintOptions ( char * dname, OptDsc dsc[], int dscl )
{
	register int i;
	printf("% 20s		    code      ptyp	   npar      poff\n",
                dname);
	for (i=0;(i<dscl-1);i++)
          printf("% 20s[% 4d] % 9s % 9d % 9d\n",
                 dname,i, dsc[i].code,dsc[i].ptyp,dsc[i].npar);
} /* PrintOptions */

/*--------------------------------------------------------------------------*/
/* PrintImageBlock
/*--------------------------------------------------------------------------*/
void PrintImageBlock ( int i, ImgBlk * ib )
{
char * VName = "ImageBlock";
printf("Image Block (blkno = %d)\n",i);
if (ib->Name.I)	     printf("% s[% d].Name.V      = % s\n",
  VName,i,ib->Name.V);
if (ib->FFirst.I)    printf("% s[% d].FFirst.V    = % d\n",
  VName,i,ib->FFirst.V);
if (ib->FLast.I)     printf("% s[% d].FLast.V     = % d\n",
  VName,i,ib->FLast.V);
if (ib->FInc.I)      printf("% s[% d].FInc.V      = % d\n",
  VName,i,ib->FInc.V);
if (ib->OpenMode.I)  printf("% s[% d].OpenMode.V  = %#x\n",
  VName,i,ib->OpenMode.V);
if (ib->Memory.I)    printf("% s[% d].Memory.V    = % d\n",
  VName,i,ib->Memory.V);
if (ib->First.I)     printf("% s[% d].First.V     = % d\n",
  VName,i,ib->First.V);
if (ib->Last.I)	     printf("% s[% d].Last.V      = % d\n",
  VName,i,ib->Last.V);
if (ib->Inc.I)	     printf("% s[% d].Inc.V       = % d\n",
  VName,i,ib->Inc.V);
if (ib->MinClip.I)   printf("% s[% d].MinClip.V   = % e\n",
  VName,i,ib->MinClip.V);
if (ib->MaxClip.I)   printf("% s[% d].MaxClip.V   = % e\n",
  VName,i,ib->MaxClip.V);
if (ib->Const.I)     printf("% s[% d].Const.V     = % e\n",
  VName,i,ib->Const.V);
if (ib->Factor.I)    printf("% s[% d] Factor.V    = % e\n",
  VName,i,ib->Factor.V);
if (ib->Bin[1].I)    printf("% s[% d].Bin[1].V    = % d\n",
  VName,i,ib->Bin[1].V);
if (ib->Bin[2].I)    printf("% s[% d].Bin[2].V    = % d\n",
  VName,i,ib->Bin[2].V);
if (ib->Gnaw[1].I)   printf("% s[% d].Gnaw[1].V   = % d\n",
  VName,i,ib->Gnaw[1].V);
if (ib->Gnaw[2].I)   printf("% s[% d].Gnaw[2].V   = % d\n",
  VName,i,ib->Gnaw[2].V);
if (ib->Title.I)     printf("% s[% d].Title.V     = % s\n",
  VName,i,ib->Title.V);
if (ib->Time.I)      printf("% s[% d].Time.V      = % s\n",
  VName,i,ib->Time.V);
if (ib->Dim[1].I)    printf("% s[% d].Dim[1].V    = % d\n",
  VName,i,ib->Dim[1].V);
if (ib->Dim[2].I)    printf("% s[% d].Dim[2].V    = % d\n",
  VName,i,ib->Dim[2].V);
if (ib->Dummy.I)     printf("% s[% d].Dummy.V     = % e\n",
  VName,i,ib->Dummy.V);
if (ib->DDummy.I)    printf("% s[% d].DDummy.V    = % e\n",
  VName,i,ib->DDummy.V);
if (ib->Offset[1].I) printf("% s[% d].Offset[1].V = % e\n",
  VName,i,ib->Offset[1].V);
if (ib->Offset[2].I) printf("% s[% d].Offset[2].V = % e\n",
  VName,i,ib->Offset[2].V);
if (ib->Center[1].I) printf("% s[% d].Center[1].V = % e\n",
  VName,i,ib->Center[1].V);
if (ib->Center[2].I) printf("% s[% d].Center[2].V = % e\n",
  VName,i,ib->Center[2].V);
if (ib->PixSiz[1].I) printf("% s[% d].PixSiz[1].V = % e\n",
  VName,i,ib->PixSiz[1].V);
if (ib->PixSiz[2].I) printf("% s[% d].PixSiz[2].V = % e\n",
  VName,i,ib->PixSiz[2].V);
if (ib->SampleDistance.I) printf("% s[% d].SampleDistance.V = % e\n",
  VName,i,ib->SampleDistance.V);
if (ib->WaveLength.I) printf("% s[% d].WaveLength.V = % e\n",
  VName,i,ib->WaveLength.V);
if (ib->Orientation.I)    printf("% s[% d].Orientation.V    = % d\n",
  VName,i,ib->Orientation.V);

} /* PrintImageBlock */

/*-------------------------------------------------------------------------*/
/* PrintCommandBlock
/*-------------------------------------------------------------------------*/
void PrintCommandBlock ( CmdBlk * cb )
{
int i;
char * VName = "CommandBlock";
if (cb->Prompt.I)    printf("% s.Prompt.V                = % d\n",
  VName,cb->Prompt.V);
if (cb->Help.I)	     printf("% s.Help.V                  = % d\n",
  VName,cb->Help.V);
if (cb->Shift[1].I)  printf("% s.Shift[1].V              = % e\n",
  VName,cb->Shift[1].V);
if (cb->Shift[2].I)  printf("% s.Shift[2].V              = % e\n",
  VName,cb->Shift[2].V);
if (cb->RSys.I)      printf("% s.RSys.V                  = % d\n",
  VName,cb->RSys.V);
if (cb->USys.I)      printf("% s.USys.V                  = % d\n",
  VName,cb->USys.V);
if (cb->GeneralBlock.I)
                     printf("% s.GeneralBlock.V          = % d\n",
  VName,cb->GeneralBlock.V);
if (cb->PassHeader.I)
                     printf("% s.PassHeader.V            = % d\n",
  VName,cb->PassHeader.V);
if (cb->OutputDataType.I)
                     printf("% s.OutputDataType.V        = % s\n",
  VName,edf_datatype2string(cb->OutputDataType.V));
if (cb->OutputDataValueOffset.I)
                     printf("% s.OutputDataValueOffset.V = % d\n",
  VName,cb->OutputDataValueOffset.V);
if (cb->BslInputByteOrder.I)
                     printf("% s.BslInputByteOrder.V     = % d\n",
  VName,cb->BslInputByteOrder.V);
if (cb->MaxLineWidth.I)
                     printf("% s.MaxLineWidth.V          = % d\n",
  VName,cb->MaxLineWidth.V);
if (cb->Add.I)
                     printf("% s.Add.V                   = % d\n",
  VName,cb->Add.V);


for (i=0;i<IO_MaxFloatOpt;i++)
  if (cb->Arg[i].I)  printf("% s.Arg[% d].V	         = % e\n",
    VName,i,cb->Arg[i].V);
for (i=0;i<IO_MaxIntOpt;i++)
  if (cb->Num[i].I)  printf("% s.Num[% d].V	         = % d\n",
    VName,i,cb->Num[i].V);
for (i=0;i<IO_MaxLineOpt;i++)
  if (cb->Lin[i].I)  printf("% s.Lin[% d].V	         = % s\n",
    VName,i,cb->Lin[i].V);
for (i=0;i<IO_MaxFlagOpt;i++)
  if (cb->Flg[i].I)  printf("% s.Flg[% d].V              = % d\n",
    VName,i,cb->Flg[i].V);
} /* PrintCommandBlock */

/*--------------------------------------------------------------------------*/
/* PrintBlocks
/*--------------------------------------------------------------------------*/
void PrintBlocks ( CmdBlk * pcb, ImgBlk ib[] )
{
	int ib_len;
	register int i;
	ib_len = pcb->ImgBlkLen;
	PrintCommandBlock( pcb );
	for (i=0;i<ib_len;i++) PrintImageBlock(i,&ib[i]);
} /* PrintBlocks */

/*-------------------------------------------------------------------------*/
/* ShowHelp
/*-------------------------------------------------------------------------*/
/* The next modules are only included for display of version strings */
# include "SaxsImage.h"
# include "SaxsRoutine.h"
# include "edfio.h"
void ShowHelp ( CmdBlk * pcb  )
{
	char c[8];
	printf("\nMain    : % s\n",pcb->MainVersion);
	printf(  "Usage   : % s % s\n",
				pcb->MainName, pcb->MainHelp);
	if (pcb->Help.V) {
        printf("\n% s\n",SD_Version);
        printf(  "% s\n",SaxsErrorVersion());
        printf(  "% s\n",SaxsInputVersion());
	printf(  "% s\n",SaxsOptionVersion());
        printf(  "% s\n",SaxsImageVersion());
        printf(  "% s\n",SaxsRoutineVersion());
        printf(  "% s\n",edfio_version());
        printf(  "E-mail : boesecke@esrf.fr\n");

        ShowDataTypes();

	printf("\nList of options [<num of parameters>,<type>]\n");
	ShowOptions ( "-", pcb->co ); /* CommandOption */
        if (pcb->ImgBlkLen>0) {
 	  c[0] = '-'; c[1] = OutChar; c[2] = (int) NULL;
	  ShowOptions ( c, pcb->io ); /* ImageOption */
	  }
	if (pcb->ImgBlkLen>1) {
	  c[1] = InChar; strcat(c,"<n>");
	  ShowOptions ( c, pcb->io );  /* ImageOption */
	  }
	printf("\nNumber of image blocks            : % d\n",pcb->ImgBlkLen);
	if (pcb->ImgBlkLen>1)
	 printf  ("Maximum number <n> of input blocks: % d\n",pcb->ImgBlkLen-1);
        if (pcb->ImgBlkLen>0) {
         printf  ("Dummy file name                   : % s\n",DUMMYFILENAME);
         }

        printf("\n");

	if (pcb->ImgBlkLen>0) ShowModes();
	ShowSys(); 
        }
}
/*--------------------------------------------------------------------------*/
/* InitCommandOptions : Initialisation of command options
/*
/* co[] (o) : command option array with length cl
/* cl   (i) : lenght of array
/*--------------------------------------------------------------------------*/
void InitCommandOptions( OptDsc co[], int cl )
{
	int nct;
	if (cl<IO_CmdOptLen)
	{printf("% s\n",ErrorCmdArrayTooShort); exit(0);}
	nct = NTestBit;
	CopyLine(STestBit,co[nct].code,IO_CodeLen,-1);
	co[nct].ptyp = IO_tpctrl;
	co[nct].npar = 0;

	nct = NHelp;
	CopyLine(SHelp,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tpctrl;
	co[nct].npar = 0;

	nct = NPrompt;
	CopyLine(SPrompt,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tpctrl;
	co[nct].npar = 0;

	nct = NShift;
	CopyLine(SShift,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tpfloat;
	co[nct].npar = 2;

	nct = NRSys;
	CopyLine(SRSys,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tprsys;
	co[nct].npar = 1;

	nct = NUSys;
	CopyLine(SUSys,co[nct].code,IO_CodeLen,-1); co[nct].ptyp = IO_tprsys;
	co[nct].npar = 1;

        nct = NGeneralBlock;
        CopyLine(SGeneralBlock,co[nct].code,IO_CodeLen,-1); 
                                                    co[nct].ptyp = IO_tpctrl;
        co[nct].npar = 1;

        nct = NPassHeader;
        CopyLine(SPassHeader,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tpctrl;
        co[nct].npar = 1;

        nct = NOutputDataType;
        CopyLine(SOutputDataType,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1; 

        nct = NOutputDataValueOffset;
        CopyLine(SOutputDataValueOffset,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NBslInputByteOrder;
        CopyLine(SBslInputByteOrder,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NAdd;
        CopyLine(SAdd,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;

        nct = NMaxLineWidth;
        CopyLine(SMaxLineWidth,co[nct].code,IO_CodeLen,-1);
                                                    co[nct].ptyp = IO_tplong;
        co[nct].npar = 1;


	for (nct=NArgFst;nct<=NArgLst;nct++) {	
	  co[nct].code[0] = (char) NULL; /*must be defined with DefFloatOpt*/
	  co[nct].ptyp    = IO_tpfloat;
	  co[nct].npar    = 1;
	  }

	for (nct=NNumFst;nct<=NNumLst;nct++) {	
	  co[nct].code[0] = (char) NULL; /*must be defined with DefIntOpt*/
	  co[nct].ptyp    = IO_tplong;
	  co[nct].npar    = 1;
	  }

	for (nct=NLinFst;nct<=NLinLst;nct++) {	
	  co[nct].code[0] = (char) NULL; /*must be defined with DefLineOpt*/
	  co[nct].ptyp    = IO_tpline;
	  co[nct].npar    = 1;
	  }

        for (nct=NFlgFst;nct<=NFlgLst;nct++) {
          co[nct].code[0] = (char) NULL; /*must be defined with DefFlgOpt*/
          co[nct].ptyp    = IO_tpctrl;
          co[nct].npar    = 0;
          }

 /* stop with NULL in last field */
	nct = NCmdLast;
	co[nct].code[0] = (char) NULL;
 /* write length into field 0 */
	CopyLine("Cmd",co[0].code,IO_CodeLen,0); co[0].ptyp = IO_tpctrl;
	co[0].npar = nct;

} /* InitCommandOptions */
/*--------------------------------------------------------------------------*/
/* InitImageOptions : Initialisation of image options
/*-------------------------------------------------------------------------*/
void InitImageOptions( OptDsc io[], int il )
{
	int nct;
	if (il<IO_ImgOptLen)
{printf("% s\n",ErrorImgArrayTooShort); exit(0);}

	nct = NName;
	CopyLine(SName,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpline;
	io[nct].npar = 1;

        nct = NMemory;
        CopyLine(SMemory,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
        io[nct].npar = 1;

	nct = NFirst;
	CopyLine(SFirst,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 1;

	nct = NLast;
	CopyLine(SLast,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 1;

	nct = NInc;
	CopyLine(SInc,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 1;

	nct = NTitle;
	CopyLine(STitle,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpline;
	io[nct].npar = 1;

        nct = NTime;
        CopyLine(STime,io[nct].code,IO_CodeLen,-1);
        io[nct].ptyp = IO_tpline;
        io[nct].npar = 1;

	nct = NDim;
	CopyLine(SDim,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 2;

	nct = NDummy;
	CopyLine(SDummy,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 1;

	nct = NDDummy;
	CopyLine(SDDummy,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 1;

	nct = NOffset;
	CopyLine(SOffset,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 2;

	nct = NCenter;
	CopyLine(SCen,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 2;

	nct = NPixSiz;
	CopyLine(SPixSiz,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 2;

	nct = NSampleDistance;
	CopyLine(SSampleDistance,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 1;

	nct = NWaveLength;
	CopyLine(SWaveLength,io[nct].code,IO_CodeLen,-1);
	io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 1;

        nct = NOrientation;
        CopyLine(SOrientation,io[nct].code,IO_CodeLen,-1); 
        io[nct].ptyp = IO_tplong;
        io[nct].npar = 1;

	nct = NOpenMode;
	CopyLine(SOpenMode,io[nct].code,IO_CodeLen,-1);io[nct].ptyp = IO_tpmode;
	io[nct].npar = 1;

        nct = NMinClip;
        CopyLine(SMinClip,io[nct].code,IO_CodeLen,-1);io[nct].ptyp = IO_tpfloat;
        io[nct].npar = 1;

        nct = NMaxClip;
        CopyLine(SMaxClip,io[nct].code,IO_CodeLen,-1);io[nct].ptyp = IO_tpfloat;
        io[nct].npar = 1;

	nct = NConst;
	CopyLine(SConst,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 1;

	nct = NFactor;
	CopyLine(SFactor,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tpfloat;
	io[nct].npar = 1;

	nct = NBin;
	CopyLine(SBin,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
	io[nct].npar = 2;

        nct = NGnaw;
        CopyLine(SGnaw,io[nct].code,IO_CodeLen,-1); io[nct].ptyp = IO_tplong;
        io[nct].npar = 2;

 /* stop with NULL in last field */
	nct = NImgLast;
	io[nct].code[0] = (char) NULL;

/* write length into field 0 */
	CopyLine("Cmd",io[0].code,IO_CodeLen,0); io[0].ptyp = IO_tpctrl;
	io[0].npar = nct;
} /* InitImageOptions */
/*---------------------------------------------------------------------------*/
/* InitCommandBlock : Initialisation of command block
/*	   Sets init values of *pcb to FALSE
/*         And initializes default values
/*---------------------------------------------------------------------------*/
void InitCommandBlock ( const char * mhlp, const char * mvers,
			int rjctnum, ImgBlk ib[], int ib_len, CmdBlk * pcb )
{
	int i;

	pcb->MainName		= NULL;    /* set by ReadOptions, if NULL */
	pcb->MainVersion	= mvers;
	pcb->MainHelp		= mhlp;
	pcb->argc		= 0;       /* set by ReadOptions, if 0    */
	pcb->argv		= NULL;    /* set by ReadOptions, if NULL */
	pcb->parg_no		= NULL;    /* set by ReadOptions, if NULL */
	pcb->infile		= stdin;
	pcb->RejectNum		= rjctnum;
	pcb->ib			= ib;
	pcb->ImgBlkLen		= ib_len;

        InitSaxsError(&pcb->seb,NULL); /*pstatus set by ReadOptions, if NULL*/

	pcb->co			= (OptDsc *) NULL;
	pcb->io			= (OptDsc *) NULL;
	pcb->TestBit		= FALSE;
	pcb->Prompt.I		= FALSE;
	pcb->Help.I		= FALSE;
	pcb->Shift[0].I         = FALSE;
	pcb->Shift[1].I 	= FALSE;
	pcb->Shift[2].I 	= FALSE;
	pcb->RSys.I		= FALSE;
	pcb->USys.I		= FALSE;
        pcb->GeneralBlock.I     = FALSE; 
        pcb->PassHeader.I       = FALSE;
        pcb->OutputDataType.I   = FALSE;
        pcb->OutputDataValueOffset.I = FALSE;
        pcb->BslInputByteOrder.I= FALSE;
        pcb->MaxLineWidth.I     = FALSE;
        pcb->Add.I              = FALSE;
	for (i=0;i<IO_MaxFloatOpt;i++) pcb->Arg[i].I = FALSE;
	for (i=0;i<IO_MaxIntOpt;i++)   pcb->Num[i].I = FALSE;
	for (i=0;i<IO_MaxLineOpt;i++)  pcb->Lin[i].I = FALSE;
	for (i=0;i<IO_MaxFlagOpt;i++)  pcb->Flg[i].I = FALSE;

/* Initialization of default values */

	pcb->Prompt.V		= TRUE;
	pcb->Help.V		= FALSE;
        pcb->Shift[0].V         = 2.0;
        pcb->Shift[1].V         = 0.0;
        pcb->Shift[2].V         = 0.0;
	pcb->RSys.V		= IO_Image;
	pcb->USys.V		= pcb->RSys.V;
        pcb->GeneralBlock.V     = 0l;
        pcb->PassHeader.V       = 0l;
        pcb->OutputDataType.V   = 0l;
        pcb->OutputDataValueOffset.V = 0l;
        pcb->BslInputByteOrder.V= 0l;
        pcb->MaxLineWidth.V     = 0l;
        pcb->Add.V              = 1l;
	for (i=0;i<IO_MaxFloatOpt;i++) pcb->Arg[i].V = 0.0;
	for (i=0;i<IO_MaxIntOpt;i++)   pcb->Num[i].V = 0l;
	for (i=0;i<IO_MaxLineOpt;i++)  pcb->Lin[i].V[0] = (char) NULL;
	for (i=0;i<IO_MaxFlagOpt;i++)  pcb->Flg[i].V = 0l;

} /* InitCommandBlock */
/*--------------------------------------------------------------------------*/
/* InitImageBlock: Initialisation of image block
/* Sets init values of *pib to FALSE
/*--------------------------------------------------------------------------*/
void InitImageBlock ( ImgBlk * pib )
{
	pib->Name.I		= FALSE;
        pib->FFirst.I           = FALSE;
        pib->FLast.I            = FALSE;
        pib->FInc.I             = FALSE;
	pib->OpenMode.I		= FALSE;
        pib->Memory.I           = FALSE;
	pib->First.I		= FALSE;
	pib->Last.I		= FALSE;
	pib->Inc.I		= FALSE;
        pib->MinClip.I          = FALSE;
        pib->MaxClip.I          = FALSE;
	pib->Const.I		= FALSE;
	pib->Factor.I		= FALSE;
	pib->Bin[0].I		= FALSE;
	pib->Bin[1].I		= FALSE;
	pib->Bin[2].I		= FALSE;
	pib->Gnaw[0].I		= FALSE;
        pib->Gnaw[1].I          = FALSE;
        pib->Gnaw[2].I          = FALSE;
	pib->Title.I		= FALSE;
        pib->Time.I             = FALSE;
	pib->Dim[0].I           = FALSE;
	pib->Dim[1].I		= FALSE;
	pib->Dim[2].I		= FALSE;
	pib->Dummy.I		= FALSE;
	pib->DDummy.I		= FALSE;
	pib->Offset[0].I        = FALSE;
	pib->Offset[1].I 	= FALSE;
	pib->Offset[2].I 	= FALSE;
	pib->Center[0].I        = FALSE;
	pib->Center[1].I 	= FALSE;
	pib->Center[2].I 	= FALSE;
	pib->PixSiz[0].I        = FALSE;
	pib->PixSiz[1].I 	= FALSE;
	pib->PixSiz[2].I 	= FALSE;
	pib->SampleDistance.I   = FALSE;
	pib->WaveLength.I	= FALSE;
        pib->Orientation.I      = FALSE;

/* Initialization of default values */

        /* --- file control variables */
	pib->Stream		= -1; // default must be less than zero
	pib->StreamOpenMode	= 0l;
        pib->FileNumber         = 1l;
	CopyLine("",pib->FileName,IO_len,0);
        /* --- image option variables */
        CopyLine(DefaultFilename,pib->Name.V,IO_len,0);
        pib->FFirst.V           = 1l;
        pib->FLast.V            = 1l;
        pib->FInc.V             = 1l;
        pib->OpenMode.V         = IO_ImageProtect;
        pib->Memory.V           = 1l;
        pib->First.V            = 1l;
        pib->Last.V             = 1l;
        pib->Inc.V              = 1l;
        pib->MinClip.V          = -FLT_MAX;
        pib->MaxClip.V          = FLT_MAX;
        pib->Const.V            = 0.0;
        pib->Factor.V           = 1.0;
	pib->Bin[0].V		= 2l;
	pib->Bin[1].V		= 1l;
	pib->Bin[2].V		= 1l;
        pib->Gnaw[0].V          = 2l;
        pib->Gnaw[1].V          = 0l;
        pib->Gnaw[2].V          = 0l;
        CopyLine("",pib->Title.V,IO_len,0);
        CopyLine("",pib->Time.V,IO_len,0);
        pib->Dim[0].V           = 2l;
        pib->Dim[1].V           = 1l;
        pib->Dim[2].V           = 1l;
        pib->Dummy.V            = 0.0;
        pib->DDummy.V           = 0.5;
        pib->Offset[0].V        = 2.0;
        pib->Offset[1].V        = 0.0;
        pib->Offset[2].V        = 0.0;
        pib->Center[0].V        = 2.0;
        pib->Center[1].V        = 0.0;
        pib->Center[2].V        = 0.0;
        pib->PixSiz[0].V        = 2.0;
        pib->PixSiz[1].V        = 1.0;
        pib->PixSiz[2].V        = 1.0;
	pib->SampleDistance.V	= 1.0;
	pib->WaveLength.V	= 1.0e-10; /* 0.1 nm */
        pib->Orientation.V      = 1l;

} /* InitImageBlock */
/*--------------------------------------------------------------------------*/
/* InitOptions : Initialisation of options,
/* initialisation of command block and all image blocks
/* If ib_len<0 no image block is initialized
/* return value  :	void
/*   *mainhelp	:	help string for main program
/*			activated with -h, +h with listing of options
/*   *pcb	:	command block
/*   rjctnum	:	TRUE  : If the first letter after an
/*			        option marker ('-' or '+') is a number
/*			        it is assumed that the argument is a
/*			        number and not an option.
/*			FALSE : all signs after '-' or '+' are allowed
/*   ib[]	:	image block array
/*   ib_len	:	length of image block array
/*			(ib[0] is always input)
/*--------------------------------------------------------------------------*/
void InitOptions( const char * mainhelp, const char * mainvers, int rjctnum,
                  CmdBlk * pcb, ImgBlk ib[], int ib_len )
{
	register int i;
	
	InitCommandBlock( mainhelp, mainvers,
                          rjctnum, ib, ib_len, pcb );
	for (i=0;i<ib_len;i++) InitImageBlock( &ib[i] );

        pcb->co = (OptDsc *) malloc(IO_CmdOptLen*sizeof(OptDsc));
	if (pcb->co == NULL)
	   {printf("%s for command options\n",
             ErrorCannotAllcMemory); exit(-1);}
        pcb->io = (OptDsc *) malloc(IO_ImgOptLen*sizeof(OptDsc));
	if (pcb->io == NULL)
	   {printf("%s for image options\n",
             ErrorCannotAllcMemory); exit(-1);}

	InitCommandOptions ( pcb->co, IO_CmdOptLen );
	InitImageOptions ( pcb->io, IO_ImgOptLen );

} /* InitOptions */
/*--------------------------------------------------------------------------*/
/* CheckOptionDuplication : Test, whether an option is duplicated
/* return value		:	TRUE  : duplicated
/*			 	FALSE : not duplicated
/*   CmdBlk *pcb	:	command option block
/*   char *name	(i)	:	new command option name
/* For options beginning with with 'OutChar' or 'InChar' the return value is
/* TRUE.
/* The IO_CodeLen-1 first characters of name and pcb.co[k].code are compared.
/* If they are identical the function returns TRUE, if not FALSE.
/*--------------------------------------------------------------------------*/
int CheckOptionDuplication( CmdBlk * pcb, char * name )
{ int k;
  if ((name[0]==OutChar) || (name[0]==InChar) ) return(TRUE);
  for (k=1;k<(pcb->co[0]).npar;k++)
      if (strncmp(pcb->co[k].code, name,IO_CodeLen-1)==0) return(TRUE);
  return(FALSE);
} /* CheckOptionDuplication */
/*--------------------------------------------------------------------------*/
/* DefFloatOption : Definition of float command option number argopt_no
/* return value		:	void
/*   CmdBlk *pcb	:	command option block
/*   char *name	(i)	:	new command option name
/*   int argopt_no (i)	:	number of float argument [0..IO_MaxFloatOpt-1]
/*--------------------------------------------------------------------------*/
void DefFloatOption ( CmdBlk * pcb, char * name, int argopt_no )
{
	int nct;

        if( CheckOptionDuplication( pcb, name ) )
	  {printf("% s-% s\n",ErrorOptionDuplicated,name);
           exit(-1);}

	nct = NArgFst + argopt_no;
	if ((nct<NArgFst) || (nct>NArgLst))
	  {printf("% s% s, argopt_no = % d \n", ErrorOptionDefnFailed,
                    name, argopt_no); exit(-1);}
	CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
	(pcb->co[nct]).ptyp = IO_tpfloat; (pcb->co[nct]).npar = 1;
} /* DefFloatOption */
/*--------------------------------------------------------------------------*/
/* DefIntOption : Definition of integer command option number numopt_no
/* return value		:	void
/*   CmdBlk *pcb	:	command option block
/*   char *name	(i)	:	new command option name
/*   int numopt_no (i)	:	number of integer argument [0..IO_MaxIntOpt-1]
/*--------------------------------------------------------------------------*/
void DefIntOption   ( CmdBlk * pcb, char * name, int numopt_no )
{
	int nct;

        if( CheckOptionDuplication( pcb, name ) )
	  {printf("% s-% s\n",ErrorOptionDuplicated,name);
	   exit(-1);}

	nct = NNumFst + numopt_no;
	if ((nct<NNumFst) || (nct>NNumLst))
	  {printf("% s% s, numopt_no = % d \n", ErrorOptionDefnFailed,
		   name, numopt_no); exit(-1);}
	CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
	(pcb->co[nct]).ptyp = IO_tplong; (pcb->co)[nct].npar = 1;
} /* DefIntOption */
/*--------------------------------------------------------------------------*/
/* DefLineOption : Definition of line command option number linopt_no
/* return value		:	void
/*   CmdBlk *pcb	:	command option block
/*   char *name	(i)	:	new command option name
/*   int linopt_no (i)	:	number of line argument [0..IO_MaxIntOpt-1]
/*--------------------------------------------------------------------------*/
void DefLineOption   ( CmdBlk * pcb, char * name, int linopt_no )
{
	int nct;

        if ( CheckOptionDuplication( pcb, name ) )
	  {printf("% s-% s\n",ErrorOptionDuplicated,name);
	   exit(-1);}

	nct = NLinFst + linopt_no;
	if ((nct<NLinFst) || (nct>NLinLst))
	  {printf("% s% s, linopt_no = % d \n", ErrorOptionDefnFailed,
		   name, linopt_no); exit(-1);}
	CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
	(pcb->co[nct]).ptyp = IO_tpline; (pcb->co)[nct].npar = 1;
} /* DefLineOption */
/*--------------------------------------------------------------------------*/
/* DefFlgOption : Definition of flag command option number flgopt_no
/* return value         :       void
/*   CmdBlk *pcb        :       command option block
/*   char *name (i)     :       new command option name
/*   int flgopt_no (i)  :       number of flag argument [0..IO_MaxFlagOpt-1]
/*--------------------------------------------------------------------------*/
void DefFlgOption   ( CmdBlk * pcb, char * name, int flgopt_no )
{
        int nct;

        if( CheckOptionDuplication( pcb, name ) )
          {printf("% s-% s\n",ErrorOptionDuplicated,name);
           exit(-1);}

        nct = NFlgFst + flgopt_no;
        if ((nct<NFlgFst) || (nct>NFlgLst))
          {printf("% s% s, flgopt_no = % d \n", ErrorOptionDefnFailed,
                   name, flgopt_no); exit(-1);}
        CopyLine(name,(pcb->co[nct]).code,IO_CodeLen,-1);
        (pcb->co[nct]).ptyp = IO_tpctrl; (pcb->co)[nct].npar = 0;
} /* DefFlgOption */
/*--------------------------------------------------------------------------*/
/* GetOption : Get option number from the string
/* History : 28-Jan-1995 PB use of dsc[0].npar for loop
/*
/*	return value	(o) : void
/*		   *ps	(i) : string with option
/*		   dsc	(i) : option descriptor with length in dsc[0].npar
/*	    *popt_no	(o) : option number
/*				0 if not found
/*	    *pstatus	(o) : Success or NoOption
/*--------------------------------------------------------------------------*/
void GetOption ( char * ps, OptDsc dsc[], int * popt_no, int * pstatus )
{	int nct;
	*pstatus = NoOption;
	*popt_no = 0;
	nct = 1;
	while ( (nct < dsc[0].npar) && strcmp(ps,dsc[nct].code) ) nct++;
	if (dsc[nct].code[0]) { *popt_no = nct; *pstatus = Success; }
} /* GetOption */
/*-------------------------------------------------------------------------*/
/* FindOption   : Read next argument and test whether it is an option.
/*                If yes, specify it.
/* It has no effect on *parg_no if argument string does not contain
/* an option, i.e. if it does not contain a "-" or "+". In this case the
/* return value is FALSE.
/*
/*	return value	(o) : TRUE if s contains an option
/*				(known or unknown!)
/*				FALSE if s does not contain an option
/*	argv		(i) : pointer to argument string array
/*      *parg_no      (i/o) : i: field number of previous argument
/*			      o: incremented, when return value is TRUE
/*      pcb->
/*	RejectNum	(i) : TRUE  : reject numbers as options
/*		              and return with FALSE and
/*		              *pstatus = Success
/*		      	      FALSE : accept numbers as options
/*      *pblk_no	(o) : block number of corresponding block,
/*				-1: CommandBlock
/*				 0: ImageBlock[0]
/*				 n: ImageBlock[n]
/*      *popt_no	(o) : option number,
/*				0 if option not found
/*      *popt_onoff	(o) : >0  if "+" option
/*			      <0  if "-" option
/*      *pstatus	(o) : status : Success, ScanError, NoOption
/*-------------------------------------------------------------------------*/
int FindOption(char * argv[], int * parg_no, CmdBlk * pcb,
		int * pblk_no, int * popt_no, int * popt_onoff, int * pstatus)
{
	int	retval;
	char *  p1;
	char	c1[IO_len];
	int	i1;

	*pstatus    = Success;
	*popt_no    = 0;
	*popt_onoff = 0;
	*pblk_no    = -2;

	/* convert to small characters */
	CopyLine(argv[*parg_no+1],c1,IO_len,-1);
	/* Is argument an option? */
	if (c1[0] == '-') *popt_onoff = -1; else {
	if (c1[0] == '+') *popt_onoff = +1; else
	   /* no option */ return(FALSE); }
	if ( (pcb->RejectNum) && (0<=(c1[1]-'0')) && ((c1[1]-'0')<=9) )
	   /* no option */ return(FALSE);
	*parg_no += 1;   /* argument is an option */
	p1=&c1[1];   /* skip first character in string */
	/* What type of option? */
	*pstatus = ScanError;
	switch (*p1) {
		case 'o':
			/* output image option */
			p1++;
			*pblk_no = 0;
                        if (pcb->ImgBlkLen<1)
                           return(TRUE); /* error, no output block */
			break;
		case 'i':
			/* input image option */
			p1++;
			/* read number */
			i1 = (int) ((*p1++) - '0');
			if ((i1<=0) || (i1>9))
			   return(TRUE); /* error, no number */
			*pblk_no = i1;
			break;
		default:
			/* command option */
			*pblk_no = -1;
			break;
		}
	if (!(*p1)) return(TRUE); /* ScanError when rest string empty */
	*pstatus = NoOption;
	if (*pblk_no==-1) /* command option */
		GetOption ( p1, pcb->co, popt_no, pstatus ); else
	if (*pblk_no>=0)   /* image option */
		GetOption ( p1, pcb->io, popt_no, pstatus );
	return(TRUE);
} /* FindOption */

/*-------------------------------------------------------------------------*/
/* ScanSys :  Read reference system
/*	return value		(o) : 0 wild card, no rsys
/*				      1 rsys found and returned
/*	*s1			(i) : input string
/*	*prsys			(o) : rsys
/*	*pstatus		(o) : status : Success, ScanError
/* The return value is always 0 in the case of an error.
/*--------------------------------------------------------------------------*/
int ScanSys( char * s1, long int * prsys, int * pstatus )
{
        *pstatus = Success;
        if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
	CopyLine(s1,s1,IO_len,-1);   /* convert to small characters */
	*prsys  = (long int) IO_NoRSys;
	if (strcmp( s1, SArray )  == 0) {
	  *prsys = (long int) IO_Array; } else
	if (strcmp( s1, SImage )  == 0) {
	  *prsys = (long int) IO_Image; } else
        if (strcmp( s1, SCenter )  == 0) {
          *prsys = (long int) IO_Center; } else
	if (strcmp( s1, SReal  )  == 0) {
	  *prsys = (long int) IO_Real; } else
        if (strcmp( s1, SNormal  )  == 0) {
          *prsys = (long int) IO_Normal; } else
	if (strcmp( s1, SSaxs  )  == 0) {
	  *prsys = (long int) IO_Saxs; } else
	  { /* unknown reference system*/
	    *pstatus = ScanError; ShowSys(); return(0); }
	return(1);
} /* ScanSys */
/*-------------------------------------------------------------------------*/
/* ScanMode :  Read file opening modus
/*	return value		(o) : 0 : wild card, no modus
/*				      1 : modus found and returned
/*	*ps			(i) : input string
/*	*pmode			(o) : output mode
/*	*pstatus		(o) : status : Success, ScanError
/* The return value is always 0 in the case of an error.
/*--------------------------------------------------------------------------*/
int ScanMode( char * s1, long int * pmode, int * pstatus )
{
	long int mode;

	*pstatus = Success;
        if (strcmp(s1,WildCard) == 0) return(0); /* wild card, don't change value */
	CopyLine(s1,s1,IO_len,-1);   /* convert to small characters */
	*pmode   = 0l;
	mode = 0x0l;
	if (strcmp( s1, SNewFP )  == 0) {
	  mode = mode | IO_New | IO_FileProtect; }                   else
	if (strcmp( s1, SNewIP )  == 0) {
	  mode = mode | IO_New | IO_ImageProtect; }                  else
	if (strcmp( s1, SNew )    == 0) {
	  mode = mode | IO_New; }                                    else
	if (strcmp( s1, SOldFP )  == 0) {
	  mode = mode | IO_Old | IO_FileProtect; }                   else
	if (strcmp( s1, SOldIP )  == 0) {
	  mode = mode | IO_Old | IO_ImageProtect; }                  else
	if (strcmp( s1, SOld )    == 0) {
	  mode = mode | IO_Old; }                                    else
	if (strcmp( s1, SAnyIP )  == 0) {
	  mode = mode | IO_ImageProtect; }                           else
        if (strcmp( s1, SAnyFP )  == 0) {
          mode = mode | IO_FileProtect; }                            else
	if (strcmp( s1, SAny	) == 0) {
	  mode = mode; }                                             else
	  { /* no possible mode */
	    *pstatus = ScanError; ShowModes(); return(0); }
	*pmode = (long) mode;
	return(1);
} /* ScanMode */
/*-------------------------------------------------------------------------*/
/* ScanLong :  Read long integer
/*	return value		(o) : 0 wild card, no value
/*				      1 value found and returned
/*	*s1			(i) : input string
/*	*pvalue			(o) : long value
/*	*pstatus		(o) : status : Success, ScanError
/* The return value is always 0 in the case of an error.
/*--------------------------------------------------------------------------*/
int ScanLong( char * s1, long int * pvalue, int * pstatus )
{
        *pstatus = Success;
        if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
        *pvalue = longexpr(s1,pstatus);
        if (*pstatus!=Success) return(0);
	return(1);

} /* ScanLong */
/*-------------------------------------------------------------------------*/
/* ScanFloat :  Read float
/*	return value		(o) : 0 wild card, no value
/*				      1 value found and returned
/*	*s1			(i) : input string
/*	*pvalue			(o) : float value
/*	*pstatus		(o) : status : Success, ScanError
/* The return value is always 0 in the case of an error.
/*--------------------------------------------------------------------------*/
int ScanFloat( char * s1, float * pvalue, int * pstatus )
{
        *pstatus = Success;
        if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
        *pvalue = floatexpr(s1,pstatus);
        if (*pstatus!=Success) return(0);
	return(1);

} /* ScanFloat */
/*-------------------------------------------------------------------------*/
/* ScanLine :  Read line
/*	return value		(o) : 0 wild card, no value
/*				      1 value found and returned
/*	*s1			(i) : input string
/*	*line			(o) : line
/*	*pstatus		(o) : status : Success, ScanError
/* The return value is always 0 in the case of an error.
/* The wild card string can be read if s1 starts with a blank, e.g.
/* " =" for WildCard == "=". Leeding blanks are ignored.
/*--------------------------------------------------------------------------*/
int ScanLine( char * s1, char * line, int * pstatus )
{
        *pstatus = Success;
        if (strcmp(s1,WildCard) == 0) return(0); /* wild card */
	CopyLine(s1,line,IO_len,0);
	if (strlen(line)==0) {
	  *pstatus = ScanError; return(0); }
	return(1);
} /* ScanLine */
/*-------------------------------------------------------------------------*/
/* DoCmdOptions :  Write option values into command block
/*	return value     (o) : void
/*	ps	         (i) : pointer to argument string array
/*	*pc	    (i/o) : i: field number of previous argument
/*                          o: incremened
/*	opt_no	         (i) : option number > 0
/*	opt_onoff        (i) : >0  if "+" option
/*		               >0  if "-" option
/*	*pcb	       (i/o) : command block
/*	*pstatus           (o) : status : Success, ScanError, NoOption
/*-------------------------------------------------------------------------*/
void DoCmdOption( char * ps[], int * pc, int opt_no, int opt_onoff,
	 	  CmdBlk * pcb, int * pstatus )
{
	int	 argopt_no;
	int	 numopt_no;
	int	 linopt_no;
	int	 flgopt_no;
	long int i1;
	float	 f1;
	char	 c1[IO_len];

	*pstatus = ScanError;

	switch (opt_no)
	{ case NTestBit:
		/* TestBit */
		if (opt_onoff>=0) pcb->TestBit += TRUE;
                  else pcb->TestBit = FALSE;
		break;
	  case NHelp:
		/* Help */
		if (opt_onoff>=0) pcb->Help.V = (long int) TRUE;
                  else pcb->Help.V = (long int) FALSE;
		pcb->Help.I = opt_onoff;
		ShowHelp(pcb);
		exit(0);   /* Help only */
	  case NPrompt:
		/* Prompt */
		if (opt_onoff>=0) pcb->Prompt.V =(long int) TRUE;
                  else pcb->Prompt.V =(long int) FALSE;
                pcb->Prompt.I = opt_onoff;
		break;
	  case NShift:
		/* Shift1 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pcb->Shift[1].V = f1;
		  pcb->Shift[1].I = opt_onoff;
		  } else if (*pstatus!=Success) return;
		/* Shift2 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pcb->Shift[2].V = f1;
		  pcb->Shift[2].I = opt_onoff;
		} else if (*pstatus!=Success) return;
                break;
	  case NRSys:
		/* reference system for calculation */
		if (ScanSys(ps[++*pc],&i1,pstatus)) {
		  pcb->RSys.V = i1;
		  pcb->RSys.I = opt_onoff;
                  /* RSys.V is default for USys.V */
                  if (!pcb->USys.I) pcb->USys.V = pcb->RSys.V;
		} else if (*pstatus!=Success) return;
                break;
	  case NUSys:
		/* reference system for user input */
		if (ScanSys(ps[++*pc],&i1,pstatus)) {
		  pcb->USys.V = i1;
		  pcb->USys.I = opt_onoff;
		} else if (*pstatus!=Success) return;
                break;
          case NGeneralBlock:
                /* GeneralBlock */
                if (opt_onoff>=0) pcb->GeneralBlock.V =(long int) TRUE;
                  else pcb->GeneralBlock.V =(long int) FALSE;
                pcb->GeneralBlock.I = opt_onoff;
                break;
          case NPassHeader:
                /* PassHeader */
                if (opt_onoff>=0) pcb->PassHeader.V =(long int) TRUE;
                  else pcb->PassHeader.V =(long int) FALSE;
                pcb->PassHeader.I = opt_onoff;
                break;
          case NOutputDataType:
                /* reference system for calculation */
                if (i1 = (long int) edf_string2datatype( ps[++*pc] )) {
                  pcb->OutputDataType.V = i1;
                  pcb->OutputDataType.I = opt_onoff;
                } else return;
                break;
          case NOutputDataValueOffset:
                /* number of output images to add */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->OutputDataValueOffset.V = i1;
                  pcb->OutputDataValueOffset.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
          case NBslInputByteOrder:
                /* First */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->BslInputByteOrder.V = i1;
                  pcb->BslInputByteOrder.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
          case NAdd:
                /* number of output images to add */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->Add.V = i1;
                  pcb->Add.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
          case NMaxLineWidth:
                /* First */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pcb->MaxLineWidth.V = i1;
                  pcb->MaxLineWidth.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
	  default:
		if ( (opt_no>=NArgFst) && (opt_no<=NArgLst) ) {
		  /* Float Option */
		  argopt_no = opt_no-NArgFst;
		  if (ScanFloat(ps[++*pc], &f1, pstatus)) {
	 	    pcb->Arg[argopt_no].V = f1;
		    pcb->Arg[argopt_no].I = opt_onoff;
		    } else if (*pstatus!=Success) return;
		  break;
		  } /* Float */
		if ( (opt_no>=NNumFst) && (opt_no<=NNumLst) ) {
		  /* Integer Option */
		  numopt_no = opt_no-NNumFst;
		  if (ScanLong(ps[++*pc], &i1, pstatus)) {
	 	    pcb->Num[numopt_no].V = i1;
		    pcb->Num[numopt_no].I = opt_onoff;
		    } else if (*pstatus!=Success) return;
 		  break;
		  } /* Integer */
		if ( (opt_no>=NLinFst) && (opt_no<=NLinLst) ) {
		  /* Line Option */
		  linopt_no = opt_no-NLinFst;
		  if (ScanLine(ps[++*pc], c1, pstatus)) {
		    CopyLine(c1,pcb->Lin[linopt_no].V,IO_len,0);
		    pcb->Lin[linopt_no].I = opt_onoff;
		    } else if (*pstatus!=Success) return;
 		  break;
		  } /* Line */
                if ( (opt_no>=NFlgFst) && (opt_no<=NFlgLst) ) {
                  /* Flag Option */
                  flgopt_no = opt_no-NFlgFst;
                  if (opt_onoff>=0) pcb->Flg[flgopt_no].V = (long int) TRUE;
                    else pcb->Flg[flgopt_no].V = (long int) FALSE;
                  pcb->Flg[flgopt_no].I = opt_onoff;
                  break;
                  } /* Flag */
		*pstatus = NoOption;
		return;
		}
	*pstatus = Success;
} /* DoCmdOptions */
/*--------------------------------------------------------------------------*/
/* DoImgOptions :  Write option values into image block
/*	return value	(o) : void
/*	ps		(i) : pointer to argument string array
/*	*pc	       (i/o): i: field number of previous argument
/*			 o: incremented
/*	opt_no		(i) : option number > 0
/*	opt_onoff	(i) : >0 if "+" option
/*			      <0 if "-" option
/*	*pib	       (i/o): image block
/*	*pstatus	(o) : status : Success, ScanError, NoOption
/*--------------------------------------------------------------------------*/
void DoImgOption( char * ps[], int * pc, int opt_no, int opt_onoff,
                  ImgBlk * pib, int * pstatus )
{
	long int	i1;
	float		f1;
	char		c1[IO_len];

	*pstatus = ScanError;
	switch (opt_no)
	{ case NName:
		/* Name */
	  	if (ScanLine(ps[++*pc], c1, pstatus)) {
		  CopyLine(c1,pib->Name.V,IO_len,0);
		  pib->Name.I = opt_onoff;
                  /* scan Name for file numbers */
                  extract_filenumbers( pib, &(pib->Name), opt_onoff, pstatus );
                  if (*pstatus!=Success) return;
		} else if (*pstatus!=Success) return;
 		break;
	  case NOpenMode:
		/* OpenMode */
		if (ScanMode(ps[++*pc],&i1,pstatus)) {
		  pib->OpenMode.V = i1;
		  pib->OpenMode.I = opt_onoff;
		} else if (*pstatus!=Success) return;
                break;
          case NMemory:
                /* Memory */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pib->Memory.V = i1;
                  pib->Memory.I = opt_onoff;
                  if (i1==0) { *pstatus=OutOfRange; return; } //+++++++++++++
                }  else if (*pstatus!=Success) return;
                break;
	  case NFirst:
		/* First */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->First.V = i1;
		  pib->First.I = opt_onoff;
		}  else if (*pstatus!=Success) return;
 		break;
	  case NLast:
		/* Last */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Last.V = i1;
		  pib->Last.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	  case NInc:
		/* Inc */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Inc.V = i1;
		  pib->Inc.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
        case NMinClip:
                /* MinClip */
                if (ScanFloat(ps[++*pc], &f1, pstatus)) {
                  pib->MinClip.V = f1;
                  pib->MinClip.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
        case NMaxClip:
                /* MaxClip */
                if (ScanFloat(ps[++*pc], &f1, pstatus)) {
                  pib->MaxClip.V = f1;
                  pib->MaxClip.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	case NConst:
		/* Const */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->Const.V = f1;
		  pib->Const.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
       case NFactor:
		/* Factor */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->Factor.V = f1;
		  pib->Factor.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	   case NBin:
		/* Bin1 */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Bin[1].V = i1;
		  pib->Bin[1].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		/* Dim2 */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Bin[2].V = i1;
		  pib->Bin[2].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
           case NGnaw:
                /* Gnaw1 */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pib->Gnaw[1].V = i1;
                  pib->Gnaw[1].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                /* Dim2 */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pib->Gnaw[2].V = i1;
                  pib->Gnaw[2].I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	  case NTitle:
		/* Title */
		if (ScanLine(ps[++*pc], c1, pstatus)) {
		  CopyLine(c1,pib->Title.V,IO_len,0);
		  pib->Title.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
          case NTime:
                /* Time */
                if (ScanLine(ps[++*pc], c1, pstatus)) {
                  CopyLine(c1,pib->Time.V,IO_len,0);
                  pib->Time.I = opt_onoff;
                } else if (*pstatus!=Success) return;
                break;
	   case NDim:
		/* Dim1 */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Dim[1].V = i1;
		  pib->Dim[1].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		/* Dim2 */
		if (ScanLong(ps[++*pc], &i1, pstatus)) {
		  pib->Dim[2].V = i1;
		  pib->Dim[2].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	 case NDummy:
		/* Dummy */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->Dummy.V = f1;
		  pib->Dummy.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	case NDDummy:
		/* DDummy */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->DDummy.V = f1;
		  pib->DDummy.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	case NOffset:
		/* Offset1 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->Offset[1].V = f1;
		  pib->Offset[1].I = opt_onoff;
		  } else if (*pstatus!=Success) return;
 		/* Offset2 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->Offset[2].V = f1;
		  pib->Offset[2].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	case NCenter:
		/* Center1 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->Center[1].V = f1;
		  pib->Center[1].I = opt_onoff;
		  } else if (*pstatus!=Success) return;
 		/* Center2 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->Center[2].V = f1;
		  pib->Center[2].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	case NPixSiz:
		/* PixSiz1 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->PixSiz[1].V = f1;
		  pib->PixSiz[1].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		/* PixSiz2 */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->PixSiz[2].V = f1;
		  pib->PixSiz[2].I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	case NSampleDistance:
		/* SampleDistance */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->SampleDistance.V = f1;
		  pib->SampleDistance.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
	case NWaveLength:
		/* WaveLength */
		if (ScanFloat(ps[++*pc], &f1, pstatus)) {
		  pib->WaveLength.V = f1;
		  pib->WaveLength.I = opt_onoff;
		} else if (*pstatus!=Success) return;
 		break;
        case NOrientation:
                /* Orientation */
                if (ScanLong(ps[++*pc], &i1, pstatus)) {
                  pib->Orientation.V = i1;
                  pib->Orientation.I = opt_onoff;
                }  else if (*pstatus!=Success) return;
                break;
	default:
		*pstatus = NoOption;
		return;
	}
	*pstatus = Success;
} /* DoImgOptions */
/*--------------------------------------------------------------------------*/
/* DoOption :  Write option values into command and image blocks
/*	return value	(o) : void
/*	argv		(i) : pointer to argument string array
/*	*parg_no      (i/o) : i: field number of previous argument
/*			      o: incremented
/*	blk_no		(i) : block number of corresponding block,
/*			     -1  : CommandBlock
/*			      0  : ImageBlock[0]
/*			      n>0: ImageBlock[n]
/*	opt_no		(i) : option number > 0
/*	opt_onoff	(i) : >0 if "+" option
/*			      <0 if "-" option
/*	*pcb          (i/o) : command block
/*	ib[]          (i/o) : image block array ib[ib_len]
/*      ib[0] : output image block
/*      ib[n] : input image n block
/*	ib_len	(i) : dimension of ib
/*	*pstatus	(o) : status : Success, ScanError, NoOption
/*--------------------------------------------------------------------------*/
void DoOption ( char * argv[], int * parg_no,
		int blk_no, int opt_no, int opt_onoff,
		CmdBlk * pcb, ImgBlk ib[], int ib_len , int * pstatus )
{
	if (pcb->TestBit)
	  printf("blk_no = %d, opt_no = %d, opt_onoff = %d\n",
		  blk_no,opt_no,opt_onoff);

	if ((blk_no>=0) && (blk_no<ib_len))  /* image option */
	  DoImgOption(argv, parg_no, opt_no, opt_onoff,
		       &ib[blk_no], pstatus); else

	if (blk_no==-1)   /* command option */
	  DoCmdOption(argv, parg_no, opt_no, opt_onoff, pcb, pstatus); else
	
	*pstatus = NoOption; /* unknown option block */

} /* DoOptions */
/*-------------------------------------------------------------------------*/
/* ReadOptions : Read all options from the argument list
/*-------------------------------------------------------------------------*/
void ReadOptions( char * argv[], int * parg_no,
CmdBlk * pcb, ImgBlk ib[], int * pstatus )
{
        const char * filename;
	char ** pa;
	int ib_len;
	int blk_no;
	int opt_no;
	int opt_onoff;
        const char * RoutineName = "ReadOptions";
        int argno;

        if (pcb->TestBit) printf(" %s \n",RoutineName);

        /* Initialization */
	if (pcb->MainName == NULL) pcb->MainName = argv[0];
        if (pcb->argc == 0)
           while (argv[pcb->argc]) pcb->argc++;
        if (pcb->argv == NULL) pcb->argv = argv;
        if (pcb->parg_no == NULL) pcb->parg_no = parg_no;
        if (pcb->seb.pstatus == NULL) pcb->seb.pstatus = pstatus;

        SetSaxsError( &pcb->seb, SOName, RoutineName, NULL, Success, 0, NULL );

        /* init history lists */
        edf_history_new( SaxsHistoryKey );

        ib_len  = pcb->ImgBlkLen;
	while (FindOption( argv, parg_no, pcb,
	       &blk_no, &opt_no, &opt_onoff, pstatus )) {
	 DoOption( argv, parg_no, blk_no, opt_no, opt_onoff,
                   pcb, ib, ib_len , pstatus );
	 if (*pstatus!=Success) {
           /* Show the last two arguments */
           pcb->seb.Message=(char *) malloc( 11 +
             strlen(argv[*parg_no-1])+strlen(argv[*parg_no]) );
	   if ( (pcb->seb.Message) != (char *) NULL )
             sprintf(pcb->seb.Message,"    %s %s ... ",
                     argv[*parg_no-1],argv[*parg_no]);
           free(pcb->co);free(pcb->io);return;
           } /* if */
         } /* while */	

        /* look for start of filename */
        if ( (filename = strrchr(argv[0],DirSeparator[0])) ) filename++; 
          else filename = argv[0];

        /* add all arguments from 0 to *parg_no to history */
        edf_history_argv( SaxsHistoryKey, filename );
        for (argno=1;argno<=*parg_no;argno++)
           edf_history_argv( SaxsHistoryKey, argv[argno] );
	
	if (pcb->TestBit) {
	/* write parameter list */
	  pa = argv;
	  printf("List :");
	  while (*++pa!=NULL) printf(" %s", *pa);
	  printf("\n");
	  PrintOptions("CommandOption",pcb->co,IO_CmdOptLen);
	  PrintOptions("ImageOption",pcb->io,IO_ImgOptLen);
	  PrintBlocks ( pcb, ib );
	}
        /* Release memory for option definitions */
	free(pcb->co);free(pcb->io);

} /* ReadOptions */

/*****************************************************************************/
